<?php

/**
 * اختبار شامل للتوافق مع ZATCA Phase 2
 * Test comprehensive ZATCA Phase 2 compliance
 */

require_once 'vendor/autoload.php';

use App\Models\Invoice;
use App\Models\Company;
use App\Models\Customer;
use App\Models\ZatcaSettings;
use App\Services\ZatcaService;
use Carbon\Carbon;

class ZatcaPhase2ComplianceTest
{
    private $zatcaService;
    private $testResults = [];
    
    public function __construct()
    {
        echo "🚀 بدء اختبار التوافق مع ZATCA Phase 2...\n\n";
    }
    
    /**
     * تشغيل جميع الاختبارات
     */
    public function runAllTests()
    {
        try {
            $this->testQRCodeGeneration();
            $this->testXMLGeneration();
            $this->testDigitalSignature();
            $this->testInvoiceHashValidation();
            $this->testCertificateManagement();
            $this->testPhase2Compliance();
            
            $this->displayResults();
            
        } catch (Exception $e) {
            echo "❌ خطأ في الاختبار: " . $e->getMessage() . "\n";
            echo "التفاصيل: " . $e->getTraceAsString() . "\n";
        }
    }
    
    /**
     * اختبار توليد QR Code للمرحلة الثانية
     */
    private function testQRCodeGeneration()
    {
        echo "📱 اختبار توليد QR Code للمرحلة الثانية...\n";
        
        try {
            $invoice = $this->createTestInvoice();
            $zatcaService = $this->getZatcaService();
            
            // توليد QR Code محسن
            $qrCode = $zatcaService->generateQRCode($invoice);
            
            // التحقق من صحة QR Code
            if (empty($qrCode)) {
                $this->addResult('QR Code Generation', false, 'QR Code فارغ');
                return;
            }
            
            // فك تشفير وتحليل TLV
            $tlvData = base64_decode($qrCode, true);
            if ($tlvData === false) {
                $this->addResult('QR Code Generation', false, 'QR Code غير صحيح Base64');
                return;
            }
            
            // تحليل TLV للتأكد من وجود جميع العلامات المطلوبة
            $tags = $this->parseTLV($tlvData);
            
            $requiredTags = [1, 2, 3, 4, 5]; // العلامات الأساسية
            $phase2Tags = [6, 7, 8, 9]; // علامات المرحلة الثانية للفواتير المبسطة
            
            $missingTags = [];
            foreach ($requiredTags as $tag) {
                if (!isset($tags[$tag])) {
                    $missingTags[] = $tag;
                }
            }
            
            if (!empty($missingTags)) {
                $this->addResult('QR Code Generation', false, 'العلامات المفقودة: ' . implode(', ', $missingTags));
                return;
            }
            
            // التحقق من علامات المرحلة الثانية للفواتير المبسطة
            if ($invoice->type === 'simplified') {
                $missingPhase2Tags = [];
                foreach ($phase2Tags as $tag) {
                    if (!isset($tags[$tag])) {
                        $missingPhase2Tags[] = $tag;
                    }
                }
                
                if (!empty($missingPhase2Tags)) {
                    echo "⚠️  تحذير: علامات المرحلة الثانية المفقودة: " . implode(', ', $missingPhase2Tags) . "\n";
                }
            }
            
            // التحقق من طول QR Code
            if (strlen($qrCode) > 1000) {
                $this->addResult('QR Code Generation', false, 'QR Code يتجاوز الحد الأقصى (1000 حرف): ' . strlen($qrCode));
                return;
            }
            
            $this->addResult('QR Code Generation', true, 'QR Code صحيح مع ' . count($tags) . ' علامات');
            echo "✅ QR Code تم توليده بنجاح - الطول: " . strlen($qrCode) . " أحرف\n";
            echo "   العلامات الموجودة: " . implode(', ', array_keys($tags)) . "\n";
            
        } catch (Exception $e) {
            $this->addResult('QR Code Generation', false, $e->getMessage());
            echo "❌ فشل اختبار QR Code: " . $e->getMessage() . "\n";
        }
    }
    
    /**
     * اختبار توليد XML للمرحلة الثانية
     */
    private function testXMLGeneration()
    {
        echo "\n📄 اختبار توليد XML للمرحلة الثانية...\n";
        
        try {
            $invoice = $this->createTestInvoice();
            $zatcaService = $this->getZatcaService();
            
            // توليد XML محسن
            $xml = $zatcaService->generateInvoiceXml($invoice);
            
            if (empty($xml)) {
                $this->addResult('XML Generation', false, 'XML فارغ');
                return;
            }
            
            // تحليل XML للتحقق من العناصر المطلوبة
            $dom = new DOMDocument();
            $dom->loadXML($xml);
            
            $xpath = new DOMXPath($dom);
            $xpath->registerNamespace('cbc', 'urn:oasis:names:specification:ubl:schema:xsd:CommonBasicComponents-2');
            $xpath->registerNamespace('ext', 'urn:oasis:names:specification:ubl:schema:xsd:CommonExtensionComponents-2');
            $xpath->registerNamespace('ds', 'http://www.w3.org/2000/09/xmldsig#');
            
            // العناصر المطلوبة للمرحلة الثانية
            $requiredElements = [
                '//cbc:UUID' => 'معرف الفاتورة الفريد',
                '//cbc:IssueDate' => 'تاريخ الإصدار',
                '//cbc:InvoiceTypeCode' => 'نوع الفاتورة',
                '//ext:UBLExtensions' => 'امتدادات UBL',
            ];
            
            $missingElements = [];
            foreach ($requiredElements as $xpath_query => $description) {
                $nodes = $xpath->query($xpath_query);
                if ($nodes->length === 0) {
                    $missingElements[] = $description;
                }
            }
            
            if (!empty($missingElements)) {
                $this->addResult('XML Generation', false, 'عناصر مفقودة: ' . implode(', ', $missingElements));
                return;
            }
            
            // التحقق من التوقيع الرقمي
            $signatures = $xpath->query('//ds:Signature');
            if ($signatures->length === 0) {
                echo "⚠️  تحذير: لا يوجد توقيع رقمي في XML\n";
            } else {
                echo "✅ التوقيع الرقمي موجود في XML\n";
            }
            
            $this->addResult('XML Generation', true, 'XML صحيح مع جميع العناصر المطلوبة');
            echo "✅ XML تم توليده بنجاح - الحجم: " . strlen($xml) . " بايت\n";
            
        } catch (Exception $e) {
            $this->addResult('XML Generation', false, $e->getMessage());
            echo "❌ فشل اختبار XML: " . $e->getMessage() . "\n";
        }
    }
    
    /**
     * اختبار التوقيع الرقمي المحسن
     */
    private function testDigitalSignature()
    {
        echo "\n🔐 اختبار التوقيع الرقمي المحسن...\n";
        
        try {
            $invoice = $this->createTestInvoice();
            $zatcaService = $this->getZatcaService();
            
            // توليد XML مع التوقيع
            $signedXml = $zatcaService->generateInvoiceXml($invoice);
            
            $dom = new DOMDocument();
            $dom->loadXML($signedXml);
            
            $xpath = new DOMXPath($dom);
            $xpath->registerNamespace('ds', 'http://www.w3.org/2000/09/xmldsig#');
            $xpath->registerNamespace('xades', 'http://uri.etsi.org/01903/v1.3.2#');
            
            // التحقق من عناصر التوقيع
            $signatureTests = [
                '//ds:Signature' => 'عنصر التوقيع الرئيسي',
                '//ds:SignedInfo' => 'معلومات التوقيع',
                '//ds:SignatureValue' => 'قيمة التوقيع',
                '//ds:KeyInfo' => 'معلومات المفتاح',
                '//ds:X509Certificate' => 'الشهادة الرقمية',
            ];
            
            $missingSignatureElements = [];
            foreach ($signatureTests as $xpath_query => $description) {
                $nodes = $xpath->query($xpath_query);
                if ($nodes->length === 0) {
                    $missingSignatureElements[] = $description;
                }
            }
            
            if (!empty($missingSignatureElements)) {
                $this->addResult('Digital Signature', false, 'عناصر توقيع مفقودة: ' . implode(', ', $missingSignatureElements));
            } else {
                $this->addResult('Digital Signature', true, 'جميع عناصر التوقيع موجودة');
                echo "✅ التوقيع الرقمي صحيح ومتكامل\n";
            }
            
            // التحقق من XAdES
            $xadesNodes = $xpath->query('//xades:QualifyingProperties');
            if ($xadesNodes->length > 0) {
                echo "✅ XAdES موجود للتوقيع المتقدم\n";
            } else {
                echo "⚠️  XAdES غير موجود (اختياري)\n";
            }
            
        } catch (Exception $e) {
            $this->addResult('Digital Signature', false, $e->getMessage());
            echo "❌ فشل اختبار التوقيع: " . $e->getMessage() . "\n";
        }
    }
    
    /**
     * اختبار التحقق من صحة Hash الفاتورة
     */
    private function testInvoiceHashValidation()
    {
        echo "\n🔍 اختبار التحقق من صحة Hash الفاتورة...\n";
        
        try {
            $invoice = $this->createTestInvoice();
            $zatcaService = $this->getZatcaService();
            
            // توليد XML وحساب Hash
            $xml = $zatcaService->generateInvoiceXml($invoice);
            
            // التحقق من وجود Hash في الفاتورة
            if (empty($invoice->invoice_hash)) {
                $this->addResult('Invoice Hash', false, 'Hash الفاتورة مفقود');
                return;
            }
            
            // التحقق من صحة تنسيق Hash
            $decodedHash = base64_decode($invoice->invoice_hash, true);
            if ($decodedHash === false) {
                $this->addResult('Invoice Hash', false, 'Hash غير صحيح Base64');
                return;
            }
            
            // التحقق من طول Hash (SHA-256 = 32 بايت)
            if (strlen($decodedHash) !== 32) {
                $this->addResult('Invoice Hash', false, 'طول Hash غير صحيح: ' . strlen($decodedHash));
                return;
            }
            
            $this->addResult('Invoice Hash', true, 'Hash صحيح ومطابق لمعايير SHA-256');
            echo "✅ Hash الفاتورة صحيح: " . substr($invoice->invoice_hash, 0, 20) . "...\n";
            
        } catch (Exception $e) {
            $this->addResult('Invoice Hash', false, $e->getMessage());
            echo "❌ فشل اختبار Hash: " . $e->getMessage() . "\n";
        }
    }
    
    /**
     * اختبار إدارة الشهادات الشاملة
     */
    private function testCertificateManagement()
    {
        echo "\n📜 اختبار إدارة الشهادات الشاملة...\n";
        
        try {
            $zatcaService = $this->getZatcaService();
            $invoice = $this->createTestInvoice();
            
            // الحصول على شهادة صالحة
            $certificate = $zatcaService->getValidCertificate($invoice);
            
            if (empty($certificate)) {
                $this->addResult('Certificate Management', false, 'لا توجد شهادة صالحة');
                return;
            }
            
            // التحقق من صحة تنسيق الشهادة
            $decodedCert = base64_decode($certificate, true);
            if ($decodedCert === false) {
                $this->addResult('Certificate Management', false, 'الشهادة ليست Base64 صحيح');
                return;
            }
            
            // التحقق من صحة الشهادة
            $validation = $zatcaService->validateCertificate($certificate);
            if (!$validation['valid']) {
                $this->addResult('Certificate Management', false, 'الشهادة غير صالحة: ' . $validation['error']);
                return;
            }
            
            $this->addResult('Certificate Management', true, 'الشهادة صالحة ومدارة بشكل صحيح');
            echo "✅ إدارة الشهادات تعمل بشكل صحيح\n";
            echo "   معلومات الشهادة: " . (isset($validation['info']['subject']['CN']) ? $validation['info']['subject']['CN'] : 'غير معروف') . "\n";
            
        } catch (Exception $e) {
            $this->addResult('Certificate Management', false, $e->getMessage());
            echo "❌ فشل اختبار إدارة الشهادات: " . $e->getMessage() . "\n";
        }
    }
    
    /**
     * اختبار التوافق الشامل مع المرحلة الثانية
     */
    private function testPhase2Compliance()
    {
        echo "\n🎯 اختبار التوافق الشامل مع المرحلة الثانية...\n";
        
        try {
            $invoice = $this->createTestInvoice();
            $zatcaService = $this->getZatcaService();
            
            // إجراء فحص شامل للتوافق
            $compliance = $zatcaService->validatePhase2Compliance($invoice);
            
            if ($compliance['compliant']) {
                $this->addResult('Phase 2 Compliance', true, 'الفاتورة متوافقة مع المرحلة الثانية');
                echo "✅ الفاتورة متوافقة بالكامل مع ZATCA Phase 2\n";
            } else {
                $this->addResult('Phase 2 Compliance', false, 'مشاكل في التوافق: ' . implode(', ', $compliance['issues']));
                echo "❌ مشاكل في التوافق مع المرحلة الثانية:\n";
                foreach ($compliance['issues'] as $issue) {
                    echo "   - $issue\n";
                }
            }
            
            if (!empty($compliance['recommendations'])) {
                echo "💡 توصيات للتحسين:\n";
                foreach ($compliance['recommendations'] as $recommendation) {
                    echo "   - $recommendation\n";
                }
            }
            
        } catch (Exception $e) {
            $this->addResult('Phase 2 Compliance', false, $e->getMessage());
            echo "❌ فشل اختبار التوافق: " . $e->getMessage() . "\n";
        }
    }
    
    /**
     * عرض نتائج الاختبار
     */
    private function displayResults()
    {
        echo "\n" . str_repeat("=", 60) . "\n";
        echo "📊 نتائج اختبار ZATCA Phase 2 Compliance\n";
        echo str_repeat("=", 60) . "\n\n";
        
        $passed = 0;
        $total = count($this->testResults);
        
        foreach ($this->testResults as $test) {
            $status = $test['passed'] ? '✅ نجح' : '❌ فشل';
            echo sprintf("%-30s %s\n", $test['name'], $status);
            if (!$test['passed']) {
                echo sprintf("%-30s   السبب: %s\n", '', $test['message']);
            }
            if ($test['passed']) $passed++;
        }
        
        echo "\n" . str_repeat("-", 60) . "\n";
        echo sprintf("النتيجة النهائية: %d/%d اختبارات نجحت (%.1f%%)\n", 
                    $passed, $total, ($passed / $total) * 100);
        
        if ($passed === $total) {
            echo "\n🎉 تهانينا! النظام متوافق بالكامل مع ZATCA Phase 2\n";
        } else {
            echo "\n⚠️  يحتاج النظام إلى تحسينات للتوافق الكامل مع ZATCA Phase 2\n";
        }
        
        echo str_repeat("=", 60) . "\n";
    }
    
    /**
     * إضافة نتيجة اختبار
     */
    private function addResult(string $testName, bool $passed, string $message = '')
    {
        $this->testResults[] = [
            'name' => $testName,
            'passed' => $passed,
            'message' => $message
        ];
    }
    
    /**
     * تحليل بيانات TLV
     */
    private function parseTLV(string $tlvData): array
    {
        $tags = [];
        $offset = 0;
        $length = strlen($tlvData);
        
        while ($offset < $length) {
            if ($offset + 2 > $length) break;
            
            $tag = ord($tlvData[$offset]);
            $valueLength = ord($tlvData[$offset + 1]);
            
            if ($offset + 2 + $valueLength > $length) break;
            
            $value = substr($tlvData, $offset + 2, $valueLength);
            $tags[$tag] = $value;
            
            $offset += 2 + $valueLength;
        }
        
        return $tags;
    }
    
    /**
     * إنشاء فاتورة اختبار
     */
    private function createTestInvoice(): Invoice
    {
        $invoice = new Invoice();
        $invoice->id = 1;
        $invoice->invoice_number = 'SIM-202507-000001';
        $invoice->type = 'simplified';
        $invoice->issue_date = Carbon::now();
        $invoice->subtotal = 100.00;
        $invoice->tax_amount = 15.00;
        $invoice->total_amount = 115.00;
        $invoice->currency = 'SAR';
        $invoice->uuid = '550e8400-e29b-41d4-a716-446655440000';
        
        // إنشاء شركة اختبار
        $company = new stdClass();
        $company->id = 1;
        $company->name = 'شركة الاختبار المحدودة';
        $company->vat_number = '399999999900003';
        $company->commercial_registration = '1234567890';
        $company->address = 'شارع الملك عبدالعزيز';
        $company->building_number = '1234';
        $company->district = 'الملز';
        $company->city = 'الرياض';
        $company->postal_code = '11564';
        
        $invoice->company = $company;
        
        // إنشاء عميل اختبار
        $customer = new stdClass();
        $customer->id = 1;
        $customer->name = 'عميل الاختبار';
        $customer->vat_number = '300111222333003';
        
        $invoice->customer = $customer;
        
        // إنشاء عناصر اختبار
        $item = new stdClass();
        $item->id = 1;
        $item->name = 'منتج اختبار';
        $item->quantity = 1;
        $item->unit_price = 100.00;
        $item->line_total = 100.00;
        $item->tax_rate = 15;
        $item->tax_amount = 15.00;
        $item->is_taxable = true;
        
        $invoice->items = collect([$item]);
        
        return $invoice;
    }
    
    /**
     * الحصول على خدمة ZATCA
     */
    private function getZatcaService(): ZatcaService
    {
        if (!$this->zatcaService) {
            $settings = new ZatcaSettings();
            $settings->environment = 'sandbox';
            $settings->username = 'test_user';
            $settings->password = 'test_password';
            $settings->certificate = 'MIIDXTCCAkWgAwIBAgIJAKC1X2GjHnmqMA0GCSqGSIb3DQEBCwUAMEUxCzAJBgNVBAYTAlNBMQ0wCwYDVQQIDARSaXlhZDEUMBIGA1UECgwLQnJpbWEgVGVjaCBMdGQxEzARBgNVBAMMCmJyaW1hLWxvY2FsMB4XDTE5MDkxMDEyMzQ1NFoXDTI5MDkwNzEyMzQ1NFowRTELMAkGA1UEBhMCU0ExDTALBgNVBAgMBFJpeWQxFDASBgNVBAoMC0JyaW1hIFRlY2ggTHRkMRMwEQYDVQQDDApicmltYS1sb2NhbDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAK9pEvGtVXZ+YbgOhICqOeYMpBivSpLd7BiHPL2QkG4F1FPFn2Mj8GcxVuTDN97tB8NvDfVWX5kEZnmJf9h+EwA13VRzOSWElP8O8UnSQKMRDCHskjv/M6dbCVBCQfgMSL9I8Q4e2Cn1iXv4m1PLnCGF2GkKiowvdZQ6Fes+uTct0zY7gvU7NGzRS6K2U8iA2MRieobdrvo2gLeQGB6QIO2Gg6fd3zOpA+XhcmddBE3xL7NltgRCeB2I/vK6k/UzKBaI4+dB7ZKKRw5q1QvUkgHDIM2xYDwKzQPLnV51Z84KkEdzYVV6jA8jAl4Azs0eq5ixAjdvJGGv/Es5CAyljeUCAwEAAaNQME4wHQYDVR0OBBYEFL+HHcTqP3VvVv4IBSmHMQmSkGs2MB8GA1UdIwQYMBaAFL+HHcTqP3VvVv4IBSmHMQmSkGs2MAwGA1UdEwQFMAMBAf8wDQYJKoZIhvcNAQELBQADggEBAEQ+YHRLMzP5/0S3ldfa2M1dE/9KdilKnOAfNHK/2HE4Bkblt+4Lj8TzHzmQOGxB6C2G95ZL0Y1zWSj8ShdDG0bpEl8CRjOx6VPpD2cLkFzQJY9VY8I3ZPC6S9WjlyNihLO+oQFEMtckQaKoxWqIbvYpFqevZ4cAMvbq0YUMv0U3+EKTRf2OwT5T4TYEXk7n+cvOtH+ZGLuA2L6UAcKKDsnN5HLFLLyoUgXbWcSRI3KL44n0XjMydBYvBLmVYy1jj1hkF+OVJsmKoLuHxJdBRcpz+jKKfF9KzO9Uu/2L6UAlzFmpzXkUe5H4pZ9d2wZ1s4Ix97OIoOulx6v0apUpkzM=';
            
            $this->zatcaService = new ZatcaService($settings);
        }
        
        return $this->zatcaService;
    }
}

// تشغيل الاختبار
if (php_sapi_name() === 'cli') {
    $test = new ZatcaPhase2ComplianceTest();
    $test->runAllTests();
}