<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Services\ZatcaService;
use App\Models\Invoice;
use App\Models\Company;
use App\Models\ZatcaSettings;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\RefreshDatabase;

class ZatcaFixesTest extends TestCase
{
    use RefreshDatabase;

    private ZatcaService $zatcaService;
    private ZatcaSettings $settings;
    private Company $company;

    protected function setUp(): void
    {
        parent::setUp();
        
        // إنشاء إعدادات ZATCA للاختبار
        $this->settings = ZatcaSettings::create([
            'environment' => 'sandbox',
            'username' => 'test_user',
            'password' => 'test_password',
            'certificate' => 'test_certificate_base64',
            'private_key' => 'test_private_key',
            'company_id' => 1
        ]);

        $this->zatcaService = new ZatcaService($this->settings);

        // إنشاء شركة للاختبار
        $this->company = Company::create([
            'name' => 'شركة الاختبار المحدودة',
            'vat_number' => '399999999900003',
            'commercial_registration' => '1234567890',
            'address' => 'شارع الملك عبدالعزيز',
            'building_number' => '1234',
            'district' => 'حي الملك فهد',
            'city' => 'الرياض',
            'postal_code' => '11564',
            'phone' => '+966501234567',
            'email' => 'test@company.com'
        ]);
    }

    /**
     * اختبار إصلاح SIGNATURE_ERROR
     */
    public function test_signature_error_fix()
    {
        $invoice = $this->createTestInvoice();
        
        // اختبار الحصول على شهادة صالحة
        $certificate = $this->zatcaService->getValidZatcaCertificate();
        $this->assertNotNull($certificate, 'يجب أن تعيد الشهادة قيمة صالحة');
        
        // اختبار حماية NullPointerException
        $cleanCertificate = $this->zatcaService->cleanAndValidateCertificate($certificate);
        $this->assertNotNull($cleanCertificate, 'يجب أن تعيد شهادة منظفة وصالحة');
        
        // اختبار توليد XML مع توقيع رقمي
        $xmlContent = $this->zatcaService->generateInvoiceXml($invoice);
        $this->assertNotEmpty($xmlContent, 'يجب أن يحتوي XML على محتوى');
        
        // فحص وجود عنصر التوقيع
        $this->assertStringContains('<ds:Signature', $xmlContent, 'يجب أن يحتوي XML على التوقيع الرقمي');
        $this->assertStringContains('</ds:Signature>', $xmlContent, 'يجب أن يحتوي XML على نهاية التوقيع الرقمي');
    }

    /**
     * اختبار إصلاح QR_CODE_ERROR
     */
    public function test_qrcode_invalid_fix()
    {
        $invoice = $this->createTestInvoice();
        
        // اختبار توليد QR Code مع مواصفات ZATCA Phase 2
        $qrCode = $this->zatcaService->generateZatcaExactQRCode($invoice);
        $this->assertNotEmpty($qrCode, 'يجب أن يعيد QR Code صالح');
        
        // فحص تنسيق Base64
        $decoded = base64_decode($qrCode, true);
        $this->assertNotFalse($decoded, 'يجب أن يكون QR Code في تنسيق Base64 صحيح');
        
        // فحص بنية TLV
        $tlvValidation = $this->zatcaService->validateZatcaQrStrict($qrCode);
        $this->assertTrue($tlvValidation['is_valid'], 'يجب أن يجتاز QR Code فحص TLV الصارم');
        
        // فحص وجود Tags مطلوبة (1-5 للفواتير المبسطة)
        $this->assertArrayHasKey('tags', $tlvValidation);
        $tags = $tlvValidation['tags'];
        
        $this->assertArrayHasKey('01', $tags, 'يجب أن يحتوي على Tag 01 (اسم البائع)');
        $this->assertArrayHasKey('02', $tags, 'يجب أن يحتوي على Tag 02 (رقم ضريبي)');
        $this->assertArrayHasKey('03', $tags, 'يجب أن يحتوي على Tag 03 (تاريخ ووقت)');
        $this->assertArrayHasKey('04', $tags, 'يجب أن يحتوي على Tag 04 (إجمالي المبلغ)');
        $this->assertArrayHasKey('05', $tags, 'يجب أن يحتوي على Tag 05 (مبلغ الضريبة)');
    }

    /**
     * اختبار إصلاح XSD_SCHEMA_ERROR
     */
    public function test_xsd_schema_error_fix()
    {
        $invoice = $this->createTestInvoice();
        
        // توليد XML
        $xmlContent = $this->zatcaService->generateInvoiceXml($invoice);
        $this->assertNotEmpty($xmlContent, 'يجب أن يحتوي XML على محتوى');
        
        // فحص بنية UBL 2.1 صحيحة
        $this->assertStringContains('<Invoice xmlns="urn:oasis:names:specification:ubl:schema:xsd:Invoice-2"', $xmlContent, 'يجب أن يحتوي على namespace صحيح');
        
        // فحص موضع ds:Signature داخل UBLExtensions
        $this->assertRegExp('/<cac:UBLExtensions>.*<ds:Signature.*<\/cac:UBLExtensions>/s', $xmlContent, 'يجب أن يكون ds:Signature داخل UBLExtensions');
        
        // فحص عدم وجود ds:Signature خارج UBLExtensions
        $xmlWithoutExtensions = preg_replace('/<cac:UBLExtensions>.*<\/cac:UBLExtensions>/s', '', $xmlContent);
        $this->assertStringNotContains('<ds:Signature', $xmlWithoutExtensions, 'يجب ألا يوجد ds:Signature خارج UBLExtensions');
    }

    /**
     * اختبار إصلاح BR-KSA-17
     */
    public function test_br_ksa_17_fix()
    {
        // إنشاء فاتورة إضافية (credit note)
        $invoice = $this->createTestInvoice('credit');
        
        // توليد XML
        $xmlContent = $this->zatcaService->generateInvoiceXml($invoice);
        $this->assertNotEmpty($xmlContent, 'يجب أن يحتوي XML على محتوى');
        
        // فحص وجود BillingReference لفواتير الائتمان/الخصم
        if (in_array($invoice->type, ['credit', 'debit'])) {
            $this->assertStringContains('<cac:BillingReference>', $xmlContent, 'يجب أن تحتوي فاتورة الائتمان على BillingReference');
            $this->assertStringContains('<cac:DiscrepancyResponse>', $xmlContent, 'يجب أن تحتوي على DiscrepancyResponse');
            $this->assertStringContains('<cbc:ResponseCode>', $xmlContent, 'يجب أن تحتوي على ResponseCode');
            $this->assertStringContains('<cbc:Description>', $xmlContent, 'يجب أن تحتوي على Description');
        }
    }

    /**
     * اختبار حماية NullPointerException
     */
    public function test_null_pointer_exception_protection()
    {
        // اختبار مع بيانات فارغة
        $nullCertificate = $this->zatcaService->getValidZatcaCertificate();
        $this->assertNotNull($nullCertificate, 'يجب أن تعيد قيمة افتراضية حتى لو كانت البيانات فارغة');
        
        // اختبار تنظيف شهادة فارغة
        $cleanedNull = $this->zatcaService->cleanAndValidateCertificate(null);
        $this->assertNotNull($cleanedNull, 'يجب أن تتعامل مع القيم الفارغة بأمان');
        
        // اختبار توليد QR مع بيانات ناقصة
        $incompleteInvoice = new Invoice([
            'invoice_number' => 'TEST-NULL',
            'total_amount' => 0,
            'currency' => 'SAR'
        ]);
        
        $qrCode = $this->zatcaService->generateZatcaExactQRCode($incompleteInvoice);
        $this->assertNotEmpty($qrCode, 'يجب أن تتعامل مع البيانات الناقصة بأمان');
    }

    /**
     * اختبار شامل لجميع الإصلاحات
     */
    public function test_comprehensive_zatca_fixes()
    {
        $invoice = $this->createTestInvoice();
        
        // تشغيل الاختبار الشامل
        $results = $this->zatcaService->testAllZatcaFixesComprehensive($invoice);
        
        // فحص النتائج العامة
        $this->assertIsArray($results, 'يجب أن تعيد مصفوفة من النتائج');
        $this->assertArrayHasKey('overall_status', $results, 'يجب أن تحتوي على حالة عامة');
        
        // فحص حالة الإصلاحات
        if (isset($results['fixes_validation'])) {
            $fixes = $results['fixes_validation'];
            
            $this->assertTrue($fixes['signature_error_fixed'] ?? false, 'يجب أن يكون إصلاح SIGNATURE_ERROR نجح');
            $this->assertTrue($fixes['qrcode_invalid_fixed'] ?? false, 'يجب أن يكون إصلاح QRCODE_INVALID نجح');
            $this->assertTrue($fixes['xsd_schema_error_fixed'] ?? false, 'يجب أن يكون إصلاح XSD_SCHEMA_ERROR نجح');
            $this->assertTrue($fixes['br_ksa_17_fixed'] ?? false, 'يجب أن يكون إصلاح BR-KSA-17 نجح');
            $this->assertTrue($fixes['null_pointer_exception_fixed'] ?? false, 'يجب أن يكون إصلاح NullPointerException نجح');
        }
        
        // فحص الاختبارات الفردية
        if (isset($results['tests'])) {
            foreach ($results['tests'] as $testName => $testResult) {
                $this->assertTrue($testResult['passed'], "يجب أن ينجح اختبار: {$testName}");
            }
        }
    }

    /**
     * إنشاء فاتورة اختبار
     */
    private function createTestInvoice(string $type = 'simplified'): Invoice
    {
        $invoice = new Invoice([
            'invoice_number' => 'TEST-' . time() . '-' . rand(1000, 9999),
            'issue_date' => Carbon::now(),
            'type' => $type,
            'subtotal' => 100.00,
            'tax_amount' => 15.00,
            'total_amount' => 115.00,
            'currency' => 'SAR',
            'notes' => 'فاتورة اختبار لفحص إصلاحات ZATCA',
            'company_id' => $this->company->id,
            'user_id' => 1,
            'uuid' => \Illuminate\Support\Str::uuid(),
            'invoice_counter' => rand(1, 1000),
            'supply_date' => Carbon::now()
        ]);

        // ربط الشركة
        $invoice->company = $this->company;

        // إنشاء عناصر وهمية
        $invoice->items = collect([
            (object)[
                'name' => 'منتج اختبار',
                'quantity' => 1,
                'unit_price' => 100.00,
                'line_total' => 100.00,
                'tax_amount' => 15.00,
                'tax_rate' => 15,
                'is_taxable' => true,
                'unit' => 'PCE',
                'zatca_category_code' => 'S'
            ]
        ]);

        return $invoice;
    }

    /**
     * اختبار الأداء
     */
    public function test_performance()
    {
        $invoice = $this->createTestInvoice();
        
        $startTime = microtime(true);
        
        // توليد XML
        $xmlContent = $this->zatcaService->generateInvoiceXml($invoice);
        
        // توليد QR Code
        $qrCode = $this->zatcaService->generateQRCode($invoice);
        
        $endTime = microtime(true);
        $executionTime = $endTime - $startTime;
        
        // فحص أن الوقت معقول (أقل من 5 ثواني)
        $this->assertLessThan(5.0, $executionTime, 'يجب أن يكون توليد XML و QR Code سريع (أقل من 5 ثواني)');
        
        // فحص أن النتائج صحيحة
        $this->assertNotEmpty($xmlContent, 'يجب أن يحتوي XML على محتوى');
        $this->assertNotEmpty($qrCode, 'يجب أن يحتوي QR Code على محتوى');
    }

    /**
     * اختبار الموثوقية
     */
    public function test_reliability()
    {
        $invoice = $this->createTestInvoice();
        
        // تشغيل العملية عدة مرات
        $results = [];
        for ($i = 0; $i < 5; $i++) {
            $xmlContent = $this->zatcaService->generateInvoiceXml($invoice);
            $qrCode = $this->zatcaService->generateQRCode($invoice);
            
            $results[] = [
                'xml_length' => strlen($xmlContent),
                'qr_length' => strlen($qrCode),
                'xml_valid' => !empty($xmlContent),
                'qr_valid' => !empty($qrCode)
            ];
        }
        
        // فحص الاستقرار
        $xmlLengths = array_column($results, 'xml_length');
        $qrLengths = array_column($results, 'qr_length');
        
        // جميع النتائج يجب أن تكون متشابهة
        $this->assertEquals(1, count(array_unique($xmlLengths)), 'يجب أن يكون طول XML ثابت في جميع المحاولات');
        $this->assertEquals(1, count(array_unique($qrLengths)), 'يجب أن يكون طول QR Code ثابت في جميع المحاولات');
        
        // جميع النتائج يجب أن تكون صالحة
        foreach ($results as $result) {
            $this->assertTrue($result['xml_valid'], 'جميع محاولات XML يجب أن تكون صالحة');
            $this->assertTrue($result['qr_valid'], 'جميع محاولات QR Code يجب أن تكون صالحة');
        }
    }
}