<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;
use App\Models\Account;
use App\Models\Product;
use App\Models\InvoiceItem;
use App\Models\Invoice;
use App\Models\Company;
use App\Models\Customer;
use App\Models\User;

class TaxCalculationTest extends TestCase
{
    use RefreshDatabase;

    protected $user;
    protected $company;
    protected $customer;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->user = User::factory()->create();
        $this->company = Company::factory()->create();
        $this->customer = Customer::factory()->create();
    }

    /** @test */
    public function it_can_create_tax_exclusive_account()
    {
        $account = Account::create([
            'code' => '4100',
            'name' => 'مبيعات - غير شامل الضريبة',
            'type' => 'revenue',
            'subtype' => 'sales',
            'debit_credit' => 'credit',
            'tax_calculation_method' => 'exclusive',
            'applies_tax' => true,
            'default_tax_rate' => 15.00,
            'is_active' => true
        ]);

        $this->assertTrue($account->isTaxExclusive());
        $this->assertFalse($account->isTaxInclusive());
        $this->assertEquals(15.00, $account->getDefaultTaxRate());
    }

    /** @test */
    public function it_can_create_tax_inclusive_account()
    {
        $account = Account::create([
            'code' => '4101',
            'name' => 'مبيعات - شامل الضريبة',
            'type' => 'revenue',
            'subtype' => 'sales',
            'debit_credit' => 'credit',
            'tax_calculation_method' => 'inclusive',
            'applies_tax' => true,
            'default_tax_rate' => 15.00,
            'is_active' => true
        ]);

        $this->assertTrue($account->isTaxInclusive());
        $this->assertFalse($account->isTaxExclusive());
        $this->assertEquals(15.00, $account->getDefaultTaxRate());
    }

    /** @test */
    public function it_calculates_tax_exclusive_correctly()
    {
        $account = Account::create([
            'code' => '4100',
            'name' => 'مبيعات - غير شامل الضريبة',
            'type' => 'revenue',
            'subtype' => 'sales',
            'debit_credit' => 'credit',
            'tax_calculation_method' => 'exclusive',
            'applies_tax' => true,
            'default_tax_rate' => 15.00,
            'is_active' => true
        ]);

        $amount = 100.00;
        $calculation = $account->calculateTaxAmount($amount);

        $this->assertEquals(100.00, $calculation['tax_exclusive_amount']);
        $this->assertEquals(15.00, $calculation['tax_amount']);
        $this->assertEquals(115.00, $calculation['tax_inclusive_amount']);
        $this->assertEquals('exclusive', $calculation['calculation_method']);
    }

    /** @test */
    public function it_calculates_tax_inclusive_correctly()
    {
        $account = Account::create([
            'code' => '4101',
            'name' => 'مبيعات - شامل الضريبة',
            'type' => 'revenue',
            'subtype' => 'sales',
            'debit_credit' => 'credit',
            'tax_calculation_method' => 'inclusive',
            'applies_tax' => true,
            'default_tax_rate' => 15.00,
            'is_active' => true
        ]);

        $amount = 115.00; // 100 + 15 tax
        $calculation = $account->calculateTaxAmount($amount);

        $this->assertEquals(100.00, round($calculation['tax_exclusive_amount'], 2));
        $this->assertEquals(15.00, round($calculation['tax_amount'], 2));
        $this->assertEquals(115.00, $calculation['tax_inclusive_amount']);
        $this->assertEquals('inclusive', $calculation['calculation_method']);
    }

    /** @test */
    public function it_handles_non_taxable_account()
    {
        $account = Account::create([
            'code' => '4300',
            'name' => 'مبيعات معفاة من الضريبة',
            'type' => 'revenue',
            'subtype' => 'sales',
            'debit_credit' => 'credit',
            'tax_calculation_method' => 'exclusive',
            'applies_tax' => false,
            'default_tax_rate' => 0.00,
            'is_active' => true
        ]);

        $amount = 100.00;
        $calculation = $account->calculateTaxAmount($amount);

        $this->assertEquals(100.00, $calculation['tax_exclusive_amount']);
        $this->assertEquals(0.00, $calculation['tax_amount']);
        $this->assertEquals(100.00, $calculation['tax_inclusive_amount']);
    }

    /** @test */
    public function invoice_item_calculates_with_tax_exclusive_account()
    {
        $account = Account::create([
            'code' => '4100',
            'name' => 'مبيعات - غير شامل الضريبة',
            'type' => 'revenue',
            'subtype' => 'sales',
            'debit_credit' => 'credit',
            'tax_calculation_method' => 'exclusive',
            'applies_tax' => true,
            'default_tax_rate' => 15.00,
            'is_active' => true
        ]);

        $invoice = Invoice::create([
            'invoice_number' => 'TEST-001',
            'type' => 'standard',
            'company_id' => $this->company->id,
            'customer_id' => $this->customer->id,
            'user_id' => $this->user->id,
            'issue_date' => now(),
            'currency' => 'SAR',
            'subtotal' => 0,
            'tax_amount' => 0,
            'total_amount' => 0
        ]);

        $item = InvoiceItem::create([
            'invoice_id' => $invoice->id,
            'account_id' => $account->id,
            'name' => 'منتج تجريبي',
            'quantity' => 2,
            'unit_price' => 50.00, // 2 × 50 = 100
            'discount_amount' => 0,
            'tax_rate' => 15,
            'is_taxable' => true,
            'line_total' => 0,
            'tax_amount' => 0,
            'total_with_tax' => 0
        ]);

        // يجب أن يحسب التحديثات تلقائياً
        $item->updateCalculations();
        $item->save();

        $this->assertEquals(100.00, $item->line_total); // 2 × 50
        $this->assertEquals(15.00, $item->tax_amount);  // 100 × 15%
        $this->assertEquals(115.00, $item->total_with_tax); // 100 + 15
    }

    /** @test */
    public function invoice_item_calculates_with_tax_inclusive_account()
    {
        $account = Account::create([
            'code' => '4101',
            'name' => 'مبيعات - شامل الضريبة',
            'type' => 'revenue',
            'subtype' => 'sales',
            'debit_credit' => 'credit',
            'tax_calculation_method' => 'inclusive',
            'applies_tax' => true,
            'default_tax_rate' => 15.00,
            'is_active' => true
        ]);

        $invoice = Invoice::create([
            'invoice_number' => 'TEST-002',
            'type' => 'standard',
            'company_id' => $this->company->id,
            'customer_id' => $this->customer->id,
            'user_id' => $this->user->id,
            'issue_date' => now(),
            'currency' => 'SAR',
            'subtotal' => 0,
            'tax_amount' => 0,
            'total_amount' => 0
        ]);

        $item = InvoiceItem::create([
            'invoice_id' => $invoice->id,
            'account_id' => $account->id,
            'name' => 'منتج تجريبي شامل الضريبة',
            'quantity' => 1,
            'unit_price' => 115.00, // سعر شامل الضريبة
            'discount_amount' => 0,
            'tax_rate' => 15,
            'is_taxable' => true,
            'line_total' => 0,
            'tax_amount' => 0,
            'total_with_tax' => 0
        ]);

        $item->updateCalculations();
        $item->save();

        $this->assertEquals(115.00, $item->line_total); // السعر الشامل
        $this->assertEquals(15.00, round($item->tax_amount, 2)); // 15% من 100
        $this->assertEquals(115.00, $item->total_with_tax); // نفس المبلغ لأنه شامل
    }

    /** @test */
    public function it_can_convert_between_inclusive_and_exclusive()
    {
        $account = Account::create([
            'code' => '4101',
            'name' => 'مبيعات - شامل الضريبة',
            'type' => 'revenue',
            'subtype' => 'sales',
            'debit_credit' => 'credit',
            'tax_calculation_method' => 'inclusive',
            'applies_tax' => true,
            'default_tax_rate' => 15.00,
            'is_active' => true
        ]);

        // تحويل من شامل إلى غير شامل
        $inclusiveAmount = 115.00;
        $exclusiveAmount = $account->convertInclusiveToExclusive($inclusiveAmount);
        $this->assertEquals(100.00, $exclusiveAmount);

        // تحويل من غير شامل إلى شامل
        $exclusiveAmount = 100.00;
        $inclusiveAmount = $account->convertExclusiveToInclusive($exclusiveAmount);
        $this->assertEquals(115.00, $inclusiveAmount);
    }

    /** @test */
    public function invoice_item_gets_tax_calculation_details()
    {
        $account = Account::create([
            'code' => '4100',
            'name' => 'مبيعات - غير شامل الضريبة',
            'type' => 'revenue',
            'subtype' => 'sales',
            'debit_credit' => 'credit',
            'tax_calculation_method' => 'exclusive',
            'applies_tax' => true,
            'default_tax_rate' => 15.00,
            'is_active' => true
        ]);

        $invoice = Invoice::create([
            'invoice_number' => 'TEST-003',
            'type' => 'standard',
            'company_id' => $this->company->id,
            'customer_id' => $this->customer->id,
            'user_id' => $this->user->id,
            'issue_date' => now(),
            'currency' => 'SAR',
            'subtotal' => 0,
            'tax_amount' => 0,
            'total_amount' => 0
        ]);

        $item = InvoiceItem::create([
            'invoice_id' => $invoice->id,
            'account_id' => $account->id,
            'name' => 'منتج تجريبي',
            'quantity' => 1,
            'unit_price' => 100.00,
            'discount_amount' => 0,
            'tax_rate' => 15,
            'is_taxable' => true,
            'line_total' => 100.00,
            'tax_amount' => 15.00,
            'total_with_tax' => 115.00
        ]);

        $details = $item->getTaxCalculationDetails();

        $this->assertEquals('exclusive', $details['method']);
        $this->assertEquals(15, $details['rate']);
        $this->assertEquals(100.00, $details['base_amount']);
        $this->assertEquals(15.00, $details['tax_amount']);
        $this->assertEquals(115.00, $details['total_amount']);
        $this->assertEquals($account->name, $details['account_name']);
    }
}