<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use App\Models\Product;
use App\Exports\ProductsExportProblematic;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;

class ExcelExportProblemsTest extends TestCase
{
    use RefreshDatabase;

    private $user;

    protected function setUp(): void
    {
        parent::setUp();
        
        // إنشاء مستخدم للاختبار
        $this->user = User::factory()->create([
            'name' => 'Test User',
            'email' => 'test@example.com',
        ]);
        
        // إنشاء بعض المنتجات للاختبار
        Product::factory()->count(10)->create([
            'is_active' => true,
            'is_available' => true,
        ]);
    }

    /** @test */
    public function it_can_access_export_problems_page()
    {
        $response = $this->actingAs($this->user)
            ->get(route('products.export-problems'));

        $response->assertStatus(200);
        $response->assertSee('مشاكل تصدير Excel - تدريب عملي');
        $response->assertSee('24 مشكلة مختلفة');
    }

    /** @test */
    public function it_shows_problematic_export_button()
    {
        $response = $this->actingAs($this->user)
            ->get(route('products.index'));

        $response->assertStatus(200);
        $response->assertSee('مشاكل التصدير (تدريب)');
    }

    /** @test */
    public function problematic_export_class_has_memory_issues()
    {
        $export = new ProductsExportProblematic();
        
        // التحقق من أن الكلاس يحتوي على مشاكل الذاكرة
        $reflection = new \ReflectionClass($export);
        $collectionMethod = $reflection->getMethod('collection');
        
        // قراءة الكود للتحقق من وجود get() بدلاً من chunk()
        $filename = $reflection->getFileName();
        $content = file_get_contents($filename);
        
        $this->assertStringContainsString('->get()', $content);
        $this->assertStringContainsString('N+1 Query Problem', $content);
    }

    /** @test */
    public function problematic_export_has_encoding_issues()
    {
        $export = new ProductsExportProblematic();
        
        $reflection = new \ReflectionClass($export);
        $filename = $reflection->getFileName();
        $content = file_get_contents($filename);
        
        // التحقق من وجود مشكلة الترميز
        $this->assertStringContainsString('iconv', $content);
        $this->assertStringContainsString('ISO-8859-1', $content);
    }

    /** @test */
    public function problematic_export_has_performance_issues()
    {
        $export = new ProductsExportProblematic();
        
        $reflection = new \ReflectionClass($export);
        $filename = $reflection->getFileName();
        $content = file_get_contents($filename);
        
        // التحقق من وجود حسابات معقدة
        $this->assertStringContainsString('sin($i) * cos($i) * tan($i)', $content);
        $this->assertStringContainsString('for ($i = 0; $i < 10000', $content);
    }

    /** @test */
    public function problematic_controller_has_security_issues()
    {
        $controllerFile = app_path('Http/Controllers/ProductExportProblematicController.php');
        $content = file_get_contents($controllerFile);
        
        // التحقق من وجود eval (مشكلة أمنية خطيرة)
        $this->assertStringContainsString('eval(', $content);
        $this->assertStringContainsString('$_SERVER', $content);
    }

    /** @test */
    public function problematic_controller_has_unnecessary_queries()
    {
        $controllerFile = app_path('Http/Controllers/ProductExportProblematicController.php');
        $content = file_get_contents($controllerFile);
        
        // التحقق من وجود استعلامات غير ضرورية
        $this->assertStringContainsString('DB::table(\'users\')', $content);
        $this->assertStringContainsString('->join(\'roles\'', $content);
    }

    /** @test */
    public function normal_export_works_correctly()
    {
        Storage::fake('local');

        $response = $this->actingAs($this->user)
            ->post(route('products.export'), [
                'item_type' => 'food'
            ]);

        // التحقق من أن التصدير العادي يعمل بشكل صحيح
        $this->assertTrue($response->getStatusCode() === 200 || $response->getStatusCode() === 302);
    }

    /** @test */
    public function export_problems_documentation_exists()
    {
        $readmeFile = base_path('EXCEL_EXPORT_PROBLEMS.md');
        
        $this->assertFileExists($readmeFile);
        
        $content = file_get_contents($readmeFile);
        $this->assertStringContainsString('24 مشكلة مختلفة', $content);
        $this->assertStringContainsString('المشاكل الحرجة', $content);
        $this->assertStringContainsString('ProductsExportProblematic', $content);
    }

    /** @test */
    public function problematic_export_has_chunk_size_issues()
    {
        $export = new ProductsExportProblematic();
        
        // التحقق من أن chunk size صغير جداً
        $this->assertEquals(10, $export->chunkSize());
    }

    /** @test */
    public function problematic_export_creates_unnecessary_columns()
    {
        $export = new ProductsExportProblematic();
        $headings = $export->headings();
        
        // التحقق من وجود أعمدة إضافية غير ضرورية
        $this->assertContains('معلومات إضافية 1', $headings);
        $this->assertContains('معلومات إضافية 2', $headings);
        $this->assertContains('حسابات معقدة', $headings);
    }

    /** @test */
    public function routes_are_properly_configured()
    {
        // التحقق من وجود المسارات
        $this->assertTrue(route('products.export-problems') !== null);
        $this->assertTrue(route('products.export-problematic') !== null);
    }

    /** @test */
    public function problematic_files_contain_arabic_comments()
    {
        $exportFile = app_path('Exports/ProductsExportProblematic.php');
        $controllerFile = app_path('Http/Controllers/ProductExportProblematicController.php');
        
        $exportContent = file_get_contents($exportFile);
        $controllerContent = file_get_contents($controllerFile);
        
        // التحقق من وجود تعليقات عربية توضح المشاكل
        $this->assertStringContainsString('مشكلة', $exportContent);
        $this->assertStringContainsString('مشكلة', $controllerContent);
        $this->assertStringContainsString('للتدريب', $exportContent);
    }

    protected function tearDown(): void
    {
        parent::tearDown();
    }
}