<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Api\ZatcaSyncController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

Route::middleware('auth:sanctum')->get('/user', function (Request $request) {
    return $request->user();
});

// مسارات API لمزامنة ZATCA
Route::middleware(['auth:sanctum'])->prefix('zatca')->group(function () {
    // الحصول على إحصائيات المزامنة
    Route::get('/sync/stats', [ZatcaSyncController::class, 'getStats']);
    
    // مزامنة فاتورة واحدة
    Route::post('/sync/invoice/{invoice}', [ZatcaSyncController::class, 'syncSingle']);
    
    // مزامنة شاملة
    Route::post('/sync/all', [ZatcaSyncController::class, 'syncAll']);
    
    // سجل العمليات
    Route::get('/sync/logs', [ZatcaSyncController::class, 'getOperationLog']);
    
    // فحص حالة ZATCA
    Route::get('/status', [ZatcaSyncController::class, 'checkStatus']);
});

// مسارات عامة (بدون مصادقة) لأغراض المراقبة
Route::prefix('public/zatca')->group(function () {
    // فحص حالة النظام العامة
    Route::get('/health', function () {
        try {
            $activeSettings = \App\Models\ZatcaSettings::where('is_active', true)->count();
            $totalInvoices = \App\Models\Invoice::whereNotNull('sent_to_zatca_at')->count();
            
            return response()->json([
                'status' => 'healthy',
                'timestamp' => now()->toISOString(),
                'active_companies' => $activeSettings,
                'total_invoices_sent' => $totalInvoices,
                'system_version' => config('app.version', '1.0.0')
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'System health check failed',
                'timestamp' => now()->toISOString()
            ], 500);
        }
    });
    
    // إحصائيات عامة (بدون تفاصيل حساسة)
    Route::get('/public-stats', function () {
        try {
            $stats = [
                'total_invoices_sent' => \App\Models\Invoice::whereNotNull('sent_to_zatca_at')->count(),
                'cleared_invoices' => \App\Models\Invoice::where('status', 'cleared')->count(),
                'reported_invoices' => \App\Models\Invoice::where('status', 'reported')->count(),
                'active_companies' => \App\Models\ZatcaSettings::where('is_active', true)->count(),
                'last_24h_activity' => \App\Models\Invoice::where('sent_to_zatca_at', '>', now()->subDay())->count(),
                'timestamp' => now()->toISOString()
            ];
            
            return response()->json([
                'success' => true,
                'data' => $stats
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch public stats'
            ], 500);
        }
    });
});

// Webhook لاستقبال إشعارات من ZATCA (إذا كان متوفراً)
Route::post('/zatca/webhook', function (Request $request) {
    try {
        // التحقق من صحة الطلب (توقيع، IP، إلخ)
        $signature = $request->header('X-ZATCA-Signature');
        $payload = $request->getContent();
        
        // التحقق من التوقيع
        $expectedSignature = hash_hmac('sha256', $payload, config('zatca.webhook_secret'));
        
        if (!hash_equals($signature, $expectedSignature)) {
            return response()->json(['error' => 'Invalid signature'], 401);
        }
        
        $data = $request->json()->all();
        
        // معالجة البيانات الواردة من ZATCA
        if (isset($data['invoice_uuid']) && isset($data['status'])) {
            $invoice = \App\Models\Invoice::where('uuid', $data['invoice_uuid'])->first();
            
            if ($invoice) {
                $invoice->status = $data['status'];
                
                if (isset($data['irn'])) {
                    $invoice->irn = $data['irn'];
                }
                
                if (isset($data['zatca_response'])) {
                    $invoice->zatca_response = array_merge(
                        $invoice->zatca_response ?? [],
                        $data['zatca_response']
                    );
                }
                
                $invoice->save();
                
                // تسجيل العملية
                $zatcaSettings = \App\Models\ZatcaSettings::where('company_id', $invoice->company_id)->first();
                if ($zatcaSettings) {
                    $zatcaSettings->logOperation('webhook_update', [
                        'invoice_id' => $invoice->id,
                        'invoice_number' => $invoice->invoice_number,
                        'new_status' => $data['status'],
                        'webhook_data' => $data
                    ]);
                }
                
                \Log::info('ZATCA webhook processed successfully', [
                    'invoice_id' => $invoice->id,
                    'invoice_number' => $invoice->invoice_number,
                    'new_status' => $data['status']
                ]);
            }
        }
        
        return response()->json(['success' => true]);
        
    } catch (\Exception $e) {
        \Log::error('ZATCA webhook failed', [
            'error' => $e->getMessage(),
            'payload' => $request->getContent()
        ]);
        
        return response()->json(['error' => 'Webhook processing failed'], 500);
    }
})->name('zatca.webhook');