<?php

use App\Http\Controllers\AccountController;
use App\Http\Controllers\JournalEntryController;
use App\Http\Controllers\FinancialReportController;
use App\Http\Controllers\CashTransactionController;
use App\Http\Controllers\FiscalYearController;
use App\Http\Controllers\BudgetController;
use App\Http\Controllers\AuditLogController;
use App\Http\Controllers\AccountingSettingController;
use App\Http\Controllers\SavedReportController;
use Illuminate\Support\Facades\Route;

/*
|--------------------------------------------------------------------------
| Accounting Routes
|--------------------------------------------------------------------------
|
| مسارات النظام المحاسبي الشامل
|
*/

Route::middleware(['web', 'auth', 'accounting.permissions'])->group(function () {
    
    // مسارات الحسابات
    Route::prefix('accounts')->name('accounts.')->group(function () {
        Route::get('/', [AccountController::class, 'index'])->name('index');
        Route::get('/create', [AccountController::class, 'create'])->name('create');
        Route::post('/', [AccountController::class, 'store'])->name('store');
        Route::get('/{account}', [AccountController::class, 'show'])->name('show');
        Route::get('/{account}/edit', [AccountController::class, 'edit'])->name('edit');
        Route::put('/{account}', [AccountController::class, 'update'])->name('update');
        Route::delete('/{account}', [AccountController::class, 'destroy'])->name('destroy');
        
        // مسارات إضافية للحسابات
        Route::get('/tree/view', [AccountController::class, 'treeView'])->name('tree');
        Route::post('/import', [AccountController::class, 'import'])->name('import');
        Route::get('/export/{format}', [AccountController::class, 'export'])->name('export');
        Route::post('/{account}/activate', [AccountController::class, 'activate'])->name('activate');
        Route::post('/{account}/deactivate', [AccountController::class, 'deactivate'])->name('deactivate');
        Route::get('/{account}/ledger', [AccountController::class, 'ledger'])->name('ledger');
        Route::get('/{account}/statement', [AccountController::class, 'statement'])->name('statement');
        Route::get('/{account}/balance', [AccountController::class, 'balance'])->name('balance');
        
        // API routes للبحث والتحديد
        Route::get('/api/search', [AccountController::class, 'apiSearch'])->name('api.search');
        Route::get('/api/children/{account}', [AccountController::class, 'apiChildren'])->name('api.children');
    });

    // مسارات القيود المحاسبية
    Route::prefix('journal-entries')->name('journal-entries.')->group(function () {
        Route::get('/', [JournalEntryController::class, 'index'])->name('index');
        Route::get('/create', [JournalEntryController::class, 'create'])->name('create');
        Route::post('/', [JournalEntryController::class, 'store'])->name('store');
        Route::get('/{journalEntry}', [JournalEntryController::class, 'show'])->name('show');
        Route::get('/{journalEntry}/edit', [JournalEntryController::class, 'edit'])->name('edit');
        Route::put('/{journalEntry}', [JournalEntryController::class, 'update'])->name('update');
        Route::delete('/{journalEntry}', [JournalEntryController::class, 'destroy'])->name('destroy');
        
        // عمليات القيود المحاسبية
        Route::post('/{journalEntry}/approve', [JournalEntryController::class, 'approve'])->name('approve');
        Route::post('/{journalEntry}/reject', [JournalEntryController::class, 'reject'])->name('reject');
        Route::post('/{journalEntry}/reverse', [JournalEntryController::class, 'reverse'])->name('reverse');
        Route::get('/{journalEntry}/duplicate', [JournalEntryController::class, 'duplicate'])->name('duplicate');
        Route::get('/{journalEntry}/print', [JournalEntryController::class, 'print'])->name('print');
        
        // تقارير القيود
        Route::get('/reports/general-ledger', [JournalEntryController::class, 'generalLedger'])->name('reports.general-ledger');
        Route::get('/reports/trial-balance', [JournalEntryController::class, 'trialBalance'])->name('reports.trial-balance');
        
        // عمليات متعددة
        Route::post('/bulk/approve', [JournalEntryController::class, 'bulkApprove'])->name('bulk.approve');
        Route::post('/bulk/delete', [JournalEntryController::class, 'bulkDelete'])->name('bulk.delete');
    });

    // مسارات المعاملات النقدية
    Route::prefix('cash-transactions')->name('cash-transactions.')->group(function () {
        Route::get('/', [CashTransactionController::class, 'index'])->name('index');
        Route::get('/create', [CashTransactionController::class, 'create'])->name('create');
        Route::post('/', [CashTransactionController::class, 'store'])->name('store');
        Route::get('/{cashTransaction}', [CashTransactionController::class, 'show'])->name('show');
        Route::get('/{cashTransaction}/edit', [CashTransactionController::class, 'edit'])->name('edit');
        Route::put('/{cashTransaction}', [CashTransactionController::class, 'update'])->name('update');
        Route::delete('/{cashTransaction}', [CashTransactionController::class, 'destroy'])->name('destroy');
        
        // عمليات إضافية
        Route::post('/{cashTransaction}/approve', [CashTransactionController::class, 'approve'])->name('approve');
        Route::get('/{cashTransaction}/print', [CashTransactionController::class, 'print'])->name('print');
        Route::get('/reports/summary', [CashTransactionController::class, 'summary'])->name('reports.summary');
        
        // تصدير البيانات
        Route::get('/export/excel', [CashTransactionController::class, 'exportExcel'])->name('export.excel');
        Route::get('/export/pdf', [CashTransactionController::class, 'exportPdf'])->name('export.pdf');
    });

    // مسارات السنوات المالية
    Route::prefix('fiscal-years')->name('fiscal-years.')->group(function () {
        Route::get('/', [FiscalYearController::class, 'index'])->name('index');
        Route::get('/create', [FiscalYearController::class, 'create'])->name('create');
        Route::post('/', [FiscalYearController::class, 'store'])->name('store');
        Route::get('/{fiscalYear}', [FiscalYearController::class, 'show'])->name('show');
        Route::get('/{fiscalYear}/edit', [FiscalYearController::class, 'edit'])->name('edit');
        Route::put('/{fiscalYear}', [FiscalYearController::class, 'update'])->name('update');
        Route::delete('/{fiscalYear}', [FiscalYearController::class, 'destroy'])->name('destroy');
        
        // عمليات إدارة السنة المالية
        Route::post('/{fiscalYear}/close', [FiscalYearController::class, 'close'])->name('close');
        Route::post('/{fiscalYear}/reopen', [FiscalYearController::class, 'reopen'])->name('reopen');
        Route::post('/{fiscalYear}/set-current', [FiscalYearController::class, 'setCurrent'])->name('set-current');
        Route::get('/{fiscalYear}/closing-report', [FiscalYearController::class, 'closingReport'])->name('closing-report');
    });

    // مسارات الميزانيات
    Route::prefix('budgets')->name('budgets.')->group(function () {
        Route::get('/', [BudgetController::class, 'index'])->name('index');
        Route::get('/create', [BudgetController::class, 'create'])->name('create');
        Route::post('/', [BudgetController::class, 'store'])->name('store');
        Route::get('/{budget}', [BudgetController::class, 'show'])->name('show');
        Route::get('/{budget}/edit', [BudgetController::class, 'edit'])->name('edit');
        Route::put('/{budget}', [BudgetController::class, 'update'])->name('update');
        Route::delete('/{budget}', [BudgetController::class, 'destroy'])->name('destroy');
        
        // عمليات الميزانيات
        Route::post('/{budget}/approve', [BudgetController::class, 'approve'])->name('approve');
        Route::post('/{budget}/activate', [BudgetController::class, 'activate'])->name('activate');
        Route::post('/{budget}/close', [BudgetController::class, 'close'])->name('close');
        
        // تقارير ولوحات التحكم
        Route::get('/reports/comparison', [BudgetController::class, 'comparison'])->name('comparison');
        Route::get('/alerts/dashboard', [BudgetController::class, 'alerts'])->name('alerts');
        
        // عمليات متعددة
        Route::post('/calculate-actual', [BudgetController::class, 'calculateActual'])->name('calculate-actual');
    });

    // مسارات سجل المراجعة
    Route::prefix('audit-logs')->name('audit-logs.')->group(function () {
        Route::get('/', [AuditLogController::class, 'index'])->name('index');
        Route::get('/{auditLog}', [AuditLogController::class, 'show'])->name('show');
        
        // لوحات التحكم والتقارير
        Route::get('/dashboard/overview', [AuditLogController::class, 'dashboard'])->name('dashboard');
        Route::get('/trail/model', [AuditLogController::class, 'trail'])->name('trail');
        Route::get('/security/incidents', [AuditLogController::class, 'security'])->name('security');
        Route::get('/compliance/report', [AuditLogController::class, 'compliance'])->name('compliance');
        
        // عمليات الإدارة
        Route::post('/export', [AuditLogController::class, 'export'])->name('export');
        Route::post('/cleanup', [AuditLogController::class, 'cleanup'])->name('cleanup');
    });

    // مسارات إعدادات النظام المحاسبي
    Route::prefix('settings')->name('accounting-settings.')->group(function () {
        Route::get('/', [AccountingSettingController::class, 'index'])->name('index');
        Route::get('/create', [AccountingSettingController::class, 'create'])->name('create');
        Route::post('/', [AccountingSettingController::class, 'store'])->name('store');
        Route::get('/{accountingSetting}', [AccountingSettingController::class, 'show'])->name('show');
        Route::get('/{accountingSetting}/edit', [AccountingSettingController::class, 'edit'])->name('edit');
        Route::put('/{accountingSetting}', [AccountingSettingController::class, 'update'])->name('update');
        Route::delete('/{accountingSetting}', [AccountingSettingController::class, 'destroy'])->name('destroy');
        
        // عمليات الإعدادات
        Route::patch('/{accountingSetting}/update-value', [AccountingSettingController::class, 'updateValue'])->name('update-value');
        Route::post('/bulk-update', [AccountingSettingController::class, 'bulkUpdate'])->name('bulk-update');
        Route::post('/{accountingSetting}/reset', [AccountingSettingController::class, 'reset'])->name('reset');
        Route::post('/load-defaults', [AccountingSettingController::class, 'loadDefaults'])->name('load-defaults');
        Route::post('/clear-cache', [AccountingSettingController::class, 'clearCache'])->name('clear-cache');
        
        // استيراد وتصدير
        Route::post('/export', [AccountingSettingController::class, 'export'])->name('export');
        Route::post('/import', [AccountingSettingController::class, 'import'])->name('import');
    });

    // مسارات التقارير المحفوظة
    Route::prefix('saved-reports')->name('saved-reports.')->group(function () {
        Route::get('/', [SavedReportController::class, 'index'])->name('index');
        Route::get('/create', [SavedReportController::class, 'create'])->name('create');
        Route::post('/', [SavedReportController::class, 'store'])->name('store');
        Route::get('/{savedReport}', [SavedReportController::class, 'show'])->name('show');
        Route::get('/{savedReport}/edit', [SavedReportController::class, 'edit'])->name('edit');
        Route::put('/{savedReport}', [SavedReportController::class, 'update'])->name('update');
        Route::delete('/{savedReport}', [SavedReportController::class, 'destroy'])->name('destroy');
        
        // عمليات التقارير
        Route::post('/{savedReport}/generate', [SavedReportController::class, 'generate'])->name('generate');
        Route::get('/{savedReport}/download', [SavedReportController::class, 'download'])->name('download');
        Route::post('/{savedReport}/toggle-active', [SavedReportController::class, 'toggleActive'])->name('toggle-active');
        Route::post('/{savedReport}/duplicate', [SavedReportController::class, 'duplicate'])->name('duplicate');
        
        // التقارير المجدولة
        Route::get('/scheduled/dashboard', [SavedReportController::class, 'scheduled'])->name('scheduled');
        Route::post('/generate-due', [SavedReportController::class, 'generateDue'])->name('generate-due');
        Route::post('/cleanup', [SavedReportController::class, 'cleanup'])->name('cleanup');
    });

    // مسارات التقارير المالية
    Route::prefix('reports')->name('reports.')->group(function () {
        Route::get('/', [FinancialReportController::class, 'index'])->name('index');
        
        // التقارير الأساسية
        Route::get('/trial-balance', [FinancialReportController::class, 'trialBalance'])->name('trial-balance');
        Route::get('/balance-sheet', [FinancialReportController::class, 'balanceSheet'])->name('balance-sheet');
        Route::get('/income-statement', [FinancialReportController::class, 'incomeStatement'])->name('income-statement');
        Route::get('/cash-flow', [FinancialReportController::class, 'cashFlow'])->name('cash-flow');
        Route::get('/general-ledger', [FinancialReportController::class, 'generalLedger'])->name('general-ledger');
        
        // تقارير متقدمة
        Route::get('/aging-report', [FinancialReportController::class, 'agingReport'])->name('aging-report');
        Route::get('/financial-ratios', [FinancialReportController::class, 'financialRatios'])->name('financial-ratios');
        Route::get('/cost-center-analysis', [FinancialReportController::class, 'costCenterAnalysis'])->name('cost-center-analysis');
        Route::get('/budget-variance', [FinancialReportController::class, 'budgetVariance'])->name('budget-variance');
        
        // تصدير التقارير
        Route::post('/export/{reportType}', [FinancialReportController::class, 'export'])->name('export');
        Route::post('/email/{reportType}', [FinancialReportController::class, 'email'])->name('email');
    });

    // مسارات API للنظام المحاسبي
    Route::prefix('api')->name('api.')->group(function () {
        // API البحث والتحديد
        Route::get('/accounts/search', function () {
            $query = request('q');
            $accounts = \App\Models\Account::where('is_active', true)
                ->where(function ($q) use ($query) {
                    $q->where('name', 'like', "%{$query}%")
                      ->orWhere('code', 'like', "%{$query}%");
                })
                ->limit(20)
                ->get(['id', 'code', 'name', 'type']);
            
            return response()->json($accounts);
        })->name('accounts.search');

        Route::get('/cost-centers/search', function () {
            $query = request('q');
            $costCenters = \App\Models\CostCenter::where('is_active', true)
                ->where(function ($q) use ($query) {
                    $q->where('name', 'like', "%{$query}%")
                      ->orWhere('code', 'like', "%{$query}%");
                })
                ->limit(20)
                ->get(['id', 'code', 'name']);
            
            return response()->json($costCenters);
        })->name('cost-centers.search');

        // API المعلومات المالية
        Route::get('/account/{account}/balance', function (\App\Models\Account $account) {
            return response()->json([
                'current_balance' => $account->current_balance,
                'formatted_balance' => number_format($account->current_balance, 2),
                'balance_nature' => $account->balance_nature,
            ]);
        })->name('account.balance');

        Route::get('/fiscal-year/current', function () {
            $fiscalYear = \App\Models\FiscalYear::current()->first();
            return response()->json($fiscalYear);
        })->name('fiscal-year.current');

        // إحصائيات سريعة
        Route::get('/stats/dashboard', function () {
            return response()->json([
                'total_accounts' => \App\Models\Account::count(),
                'active_accounts' => \App\Models\Account::where('is_active', true)->count(),
                'total_journal_entries' => \App\Models\JournalEntry::count(),
                'pending_approval' => \App\Models\JournalEntry::where('is_approved', false)->count(),
                'total_budgets' => \App\Models\Budget::count(),
                'over_budget' => \App\Models\Budget::overWarningThreshold()->count(),
            ]);
        })->name('stats.dashboard');
    });

    // مسارات إضافية للعمليات المتقدمة
    Route::prefix('advanced')->name('advanced.')->group(function () {
        // إدارة الأرصدة
        Route::get('/balances', function () {
            return view('accounting.advanced.balances');
        })->name('balances');

        Route::post('/balances/calculate', function () {
            $fiscalYearId = request('fiscal_year_id');
            $calculated = \App\Models\AccountBalance::calculateAllBalances($fiscalYearId);
            return back()->with('success', "تم حساب {$calculated} رصيد");
        })->name('balances.calculate');

        // صيانة النظام
        Route::get('/maintenance', function () {
            return view('accounting.advanced.maintenance');
        })->name('maintenance');

        Route::post('/maintenance/run', function () {
            \Artisan::call('accounting:maintenance', ['operation' => 'all']);
            return back()->with('success', 'تم تشغيل عمليات الصيانة بنجاح');
        })->name('maintenance.run');

        // النسخ الاحتياطي
        Route::post('/backup', function () {
            \Artisan::call('accounting:backup');
            return back()->with('success', 'تم إنشاء نسخة احتياطية');
        })->name('backup');

        // تحديث الأرصدة
        Route::post('/update-balances', function () {
            \Artisan::call('accounting:update-balances');
            return back()->with('success', 'تم تحديث جميع الأرصدة');
        })->name('update-balances');
    });
});

// مسارات عامة للنظام المحاسبي (بدون authentication للتقارير العامة)
Route::prefix('accounting')->name('accounting.')->group(function () {
    // صفحة رئيسية للنظام المحاسبي
    Route::get('/', function () {
        return view('accounting.dashboard');
    })->name('dashboard')->middleware('auth');

    // تقارير عامة (إذا كانت مسموحة)
    Route::get('/public-reports/{savedReport}/view', function (\App\Models\SavedReport $savedReport) {
        if (!$savedReport->is_public || !$savedReport->is_active) {
            abort(404);
        }
        
        return view('accounting.public-report', compact('savedReport'));
    })->name('public-report');
});