@extends('layouts.app')

@section('title', 'المعاملات الضريبية')
@section('page-title', 'المعاملات الضريبية')

@section('breadcrumb')
    <span>تقارير النظام</span>
    <i class="fas fa-chevron-left breadcrumb-separator"></i>
    <span>الإقرارات الضريبية</span>
    <i class="fas fa-chevron-left breadcrumb-separator"></i>
    <span>المعاملات الضريبية</span>
@endsection

@section('content')
<div class="container-fluid">
    <!-- عنوان الصفحة والمرشحات -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h4 class="mb-0">
            <i class="fas fa-list-alt me-2 text-primary"></i>
            سجل المعاملات الضريبية
        </h4>
        <div class="d-flex gap-2">
            <a href="{{ route('system-reports.tax.index') }}" class="btn btn-outline-secondary btn-sm">
                <i class="fas fa-arrow-right me-2"></i>العودة للتقارير
            </a>
            <button class="btn btn-success btn-sm" onclick="exportReport()">
                <i class="fas fa-download me-2"></i>تصدير التقرير
            </button>
        </div>
    </div>

    <!-- فلاتر التاريخ والنوع -->
    <div class="card border-0 shadow-sm mb-4">
        <div class="card-body">
            <form method="GET" class="row g-3">
                <div class="col-md-3">
                    <label for="start_date" class="form-label">من تاريخ</label>
                    <input type="date" class="form-control" id="start_date" name="start_date" 
                           value="{{ request('start_date', $startDate ?? now()->startOfMonth()->format('Y-m-d')) }}">
                </div>
                <div class="col-md-3">
                    <label for="end_date" class="form-label">إلى تاريخ</label>
                    <input type="date" class="form-control" id="end_date" name="end_date" 
                           value="{{ request('end_date', $endDate ?? now()->endOfMonth()->format('Y-m-d')) }}">
                </div>
                <div class="col-md-3">
                    <label for="type" class="form-label">نوع المعاملة</label>
                    <select class="form-control" id="type" name="type">
                        <option value="all" {{ request('type', $type ?? 'all') == 'all' ? 'selected' : '' }}>جميع المعاملات</option>
                        <option value="sales" {{ request('type', $type ?? 'all') == 'sales' ? 'selected' : '' }}>فواتير المبيعات فقط</option>
                        <option value="purchases" {{ request('type', $type ?? 'all') == 'purchases' ? 'selected' : '' }}>فواتير المشتريات فقط</option>
                    </select>
                </div>
                <div class="col-md-3 d-flex align-items-end">
                    <button type="submit" class="btn btn-primary me-2">
                        <i class="fas fa-filter me-2"></i>تطبيق المرشح
                    </button>
                    <a href="{{ route('system-reports.tax.transactions') }}" class="btn btn-outline-secondary">
                        إعادة تعيين
                    </a>
                </div>
            </form>
        </div>
    </div>

    <!-- إحصائيات المعاملات الضريبية -->
    @php
        $salesTransactions = $transactions->where('type', 'مبيعات');
        $purchaseTransactions = $transactions->where('type', 'مشتريات');
        $totalTransactions = $transactions->count();
        $totalVAT = $transactions->sum('vat_amount');
        $totalSubtotal = $transactions->sum('subtotal');
        $totalAmount = $transactions->sum('total');
    @endphp

    <div class="row mb-4">
        <div class="col-lg-3 col-md-6 mb-3">
            <div class="card border-0 shadow-sm stats-card">
                <div class="card-body text-center">
                    <div class="stats-icon bg-primary text-white mb-3">
                        <i class="fas fa-receipt"></i>
                    </div>
                    <h3 class="mb-2">{{ number_format($totalTransactions) }}</h3>
                    <p class="text-muted mb-0">إجمالي المعاملات</p>
                </div>
            </div>
        </div>

        <div class="col-lg-3 col-md-6 mb-3">
            <div class="card border-0 shadow-sm stats-card">
                <div class="card-body text-center">
                    <div class="stats-icon bg-success text-white mb-3">
                        <i class="fas fa-arrow-up"></i>
                    </div>
                    <h3 class="mb-2">{{ number_format($salesTransactions->count()) }}</h3>
                    <p class="text-muted mb-0">فواتير مبيعات</p>
                </div>
            </div>
        </div>

        <div class="col-lg-3 col-md-6 mb-3">
            <div class="card border-0 shadow-sm stats-card">
                <div class="card-body text-center">
                    <div class="stats-icon bg-info text-white mb-3">
                        <i class="fas fa-arrow-down"></i>
                    </div>
                    <h3 class="mb-2">{{ number_format($purchaseTransactions->count()) }}</h3>
                    <p class="text-muted mb-0">فواتير مشتريات</p>
                </div>
            </div>
        </div>

        <div class="col-lg-3 col-md-6 mb-3">
            <div class="card border-0 shadow-sm stats-card">
                <div class="card-body text-center">
                    <div class="stats-icon bg-warning text-white mb-3">
                        <i class="fas fa-percentage"></i>
                    </div>
                    <h3 class="mb-2">{{ number_format($totalVAT, 2) }}</h3>
                    <p class="text-muted mb-0">إجمالي الضرائب (ريال)</p>
                </div>
            </div>
        </div>
    </div>

    <!-- الرسوم البيانية -->
    <div class="row mb-4">
        <div class="col-md-6">
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-light">
                    <h6 class="card-title mb-0">
                        <i class="fas fa-chart-line me-2 text-info"></i>
                        تطور المعاملات الضريبية
                    </h6>
                </div>
                <div class="card-body">
                    <canvas id="transactionTrendChart" height="200"></canvas>
                </div>
            </div>
        </div>
        
        <div class="col-md-6">
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-light">
                    <h6 class="card-title mb-0">
                        <i class="fas fa-chart-pie me-2 text-success"></i>
                        توزيع المعاملات حسب النوع
                    </h6>
                </div>
                <div class="card-body">
                    <canvas id="transactionTypeChart" height="200"></canvas>
                </div>
            </div>
        </div>
    </div>

    <!-- جدول تفصيل المعاملات الضريبية -->
    <div class="card border-0 shadow-sm">
        <div class="card-header bg-white border-bottom-0">
            <div class="d-flex justify-content-between align-items-center">
                <h5 class="card-title mb-0">
                    <i class="fas fa-table me-2 text-primary"></i>
                    تفصيل المعاملات الضريبية
                </h5>
                <div class="d-flex align-items-center gap-2">
                    <small class="text-muted">
                        عرض {{ number_format($transactions->count()) }} معاملة
                    </small>
                </div>
            </div>
        </div>
        <div class="card-body p-0">
            @if($transactions && $transactions->count() > 0)
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead class="table-light">
                            <tr>
                                <th class="px-4 py-3">رقم الفاتورة</th>
                                <th class="px-4 py-3 text-center">النوع</th>
                                <th class="px-4 py-3 text-center">الطرف</th>
                                <th class="px-4 py-3 text-center">التاريخ</th>
                                <th class="px-4 py-3 text-center">المبلغ قبل الضريبة</th>
                                <th class="px-4 py-3 text-center">نسبة الضريبة</th>
                                <th class="px-4 py-3 text-center">ضريبة القيمة المضافة</th>
                                <th class="px-4 py-3 text-center">المبلغ الإجمالي</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($transactions->take(100) as $transaction)
                                <tr>
                                    <td class="px-4 py-3">
                                        <div class="d-flex align-items-center">
                                            <div class="transaction-icon {{ $transaction['type'] === 'مبيعات' ? 'bg-success' : 'bg-info' }} text-white me-3">
                                                <i class="fas {{ $transaction['type'] === 'مبيعات' ? 'fa-arrow-up' : 'fa-arrow-down' }}"></i>
                                            </div>
                                            <div>
                                                <div class="fw-semibold">{{ $transaction['number'] ?: 'غير محدد' }}</div>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        <span class="badge {{ $transaction['type'] === 'مبيعات' ? 'bg-success' : 'bg-info' }}">
                                            {{ $transaction['type'] }}
                                        </span>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        <span class="text-muted">{{ $transaction['party'] ?: 'غير محدد' }}</span>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        <span class="text-muted">{{ \Carbon\Carbon::parse($transaction['date'])->format('d/m/Y') }}</span>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        <strong class="text-primary">{{ number_format($transaction['subtotal'], 2) }} ريال</strong>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        <span class="badge bg-light text-dark">{{ $transaction['vat_rate'] }}%</span>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        <strong class="text-warning">{{ number_format($transaction['vat_amount'], 2) }} ريال</strong>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        <strong class="text-success">{{ number_format($transaction['total'], 2) }} ريال</strong>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
                
                @if($transactions->count() > 100)
                    <div class="alert alert-info m-3" role="alert">
                        <i class="fas fa-info-circle me-2"></i>
                        <strong>ملاحظة:</strong> يتم عرض أول 100 معاملة فقط. استخدم المرشحات لتضييق النتائج أو قم بتصدير التقرير للحصول على جميع المعاملات.
                    </div>
                @endif
            @else
                <div class="text-center py-5">
                    <div class="mb-3">
                        <i class="fas fa-list-alt fa-3x text-muted"></i>
                    </div>
                    <h5 class="text-muted">لا توجد معاملات ضريبية</h5>
                    <p class="text-muted">لم يتم العثور على أي معاملات ضريبية في هذه الفترة.</p>
                </div>
            @endif
        </div>
    </div>

    <!-- ملخص ضريبي -->
    @if($transactions && $transactions->count() > 0)
        <div class="row mt-4">
            <!-- ملخص المبيعات -->
            <div class="col-md-4">
                <div class="card border-0 shadow-sm">
                    <div class="card-header bg-success text-white">
                        <h6 class="card-title mb-0">
                            <i class="fas fa-arrow-up me-2"></i>
                            ملخص فواتير المبيعات
                        </h6>
                    </div>
                    <div class="card-body">
                        <div class="row text-center">
                            <div class="col-12 mb-3">
                                <div class="border-bottom pb-2">
                                    <strong>{{ number_format($salesTransactions->count()) }}</strong>
                                    <br><small class="text-muted">عدد الفواتير</small>
                                </div>
                            </div>
                            <div class="col-12 mb-3">
                                <div class="border-bottom pb-2">
                                    <strong>{{ number_format($salesTransactions->sum('vat_amount'), 2) }} ريال</strong>
                                    <br><small class="text-muted">ضريبة مخرجات</small>
                                </div>
                            </div>
                            <div class="col-12">
                                <strong>{{ number_format($salesTransactions->sum('total'), 2) }} ريال</strong>
                                <br><small class="text-muted">إجمالي المبيعات</small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- ملخص المشتريات -->
            <div class="col-md-4">
                <div class="card border-0 shadow-sm">
                    <div class="card-header bg-info text-white">
                        <h6 class="card-title mb-0">
                            <i class="fas fa-arrow-down me-2"></i>
                            ملخص فواتير المشتريات
                        </h6>
                    </div>
                    <div class="card-body">
                        <div class="row text-center">
                            <div class="col-12 mb-3">
                                <div class="border-bottom pb-2">
                                    <strong>{{ number_format($purchaseTransactions->count()) }}</strong>
                                    <br><small class="text-muted">عدد الفواتير</small>
                                </div>
                            </div>
                            <div class="col-12 mb-3">
                                <div class="border-bottom pb-2">
                                    <strong>{{ number_format($purchaseTransactions->sum('vat_amount'), 2) }} ريال</strong>
                                    <br><small class="text-muted">ضريبة مدخلات</small>
                                </div>
                            </div>
                            <div class="col-12">
                                <strong>{{ number_format($purchaseTransactions->sum('total'), 2) }} ريال</strong>
                                <br><small class="text-muted">إجمالي المشتريات</small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- صافي الضريبة -->
            <div class="col-md-4">
                <div class="card border-0 shadow-sm">
                    <div class="card-header bg-warning text-dark">
                        <h6 class="card-title mb-0">
                            <i class="fas fa-balance-scale me-2"></i>
                            صافي الضريبة
                        </h6>
                    </div>
                    <div class="card-body">
                        @php
                            $outputVAT = $salesTransactions->sum('vat_amount');
                            $inputVAT = $purchaseTransactions->sum('vat_amount');
                            $netVAT = $outputVAT - $inputVAT;
                        @endphp
                        
                        <div class="row text-center">
                            <div class="col-12 mb-3">
                                <div class="border-bottom pb-2">
                                    <strong class="text-success">{{ number_format($outputVAT, 2) }} ريال</strong>
                                    <br><small class="text-muted">ضريبة مخرجات</small>
                                </div>
                            </div>
                            <div class="col-12 mb-3">
                                <div class="border-bottom pb-2">
                                    <strong class="text-info">{{ number_format($inputVAT, 2) }} ريال</strong>
                                    <br><small class="text-muted">ضريبة مدخلات</small>
                                </div>
                            </div>
                            <div class="col-12">
                                <strong class="{{ $netVAT >= 0 ? 'text-warning' : 'text-primary' }}">
                                    {{ number_format($netVAT, 2) }} ريال
                                </strong>
                                <br><small class="text-muted">
                                    {{ $netVAT >= 0 ? 'مستحقة للدولة' : 'مستردة من الدولة' }}
                                </small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    @endif

    <!-- ملاحظة هامة -->
    <div class="alert alert-info mt-4" role="alert">
        <div class="d-flex align-items-start">
            <i class="fas fa-info-circle me-2 mt-1"></i>
            <div>
                <strong>ملاحظة حول المعاملات الضريبية:</strong>
                <ul class="mb-0 mt-2">
                    <li><strong>ضريبة المخرجات:</strong> الضرائب المحصلة من فواتير المبيعات</li>
                    <li><strong>ضريبة المدخلات:</strong> الضرائب المدفوعة في فواتير المشتريات</li>
                    <li><strong>صافي الضريبة:</strong> الفرق بين المخرجات والمدخلات</li>
                    <li><strong>الفترة:</strong> يشمل التقرير جميع المعاملات في الفترة المحددة</li>
                </ul>
            </div>
        </div>
    </div>
</div>

<style>
.stats-card {
    transition: transform 0.3s ease, box-shadow 0.3s ease;
    border-radius: 12px;
}

.stats-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 8px 25px rgba(0,0,0,0.15) !important;
}

.stats-icon {
    width: 60px;
    height: 60px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto;
    font-size: 1.5rem;
}

.transaction-icon {
    width: 35px;
    height: 35px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 0.9rem;
}

.card {
    border-radius: 12px;
    animation: fadeInUp 0.6s ease forwards;
}

@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.table th {
    border-top: none;
    font-weight: 600;
    color: #495057;
    background-color: #f8f9fa !important;
}

.table tbody tr:hover {
    background-color: #f8f9fa;
}

.badge {
    font-size: 0.75rem;
    padding: 0.4rem 0.8rem;
}

.border-bottom {
    border-bottom: 1px solid #dee2e6 !important;
}
</style>

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
function exportReport() {
    // يمكن تطوير هذه الوظيفة لاحقاً لتصدير التقرير
    alert('سيتم تطوير وظيفة التصدير قريباً');
}

document.addEventListener('DOMContentLoaded', function() {
    const transactions = @json($transactions->values());
    
    // رسم خطي لتطور المعاملات الضريبية
    const trendCtx = document.getElementById('transactionTrendChart').getContext('2d');
    
    // تجميع البيانات حسب التاريخ
    const dailyData = {};
    transactions.forEach(transaction => {
        const date = new Date(transaction.date).toLocaleDateString('en-CA'); // YYYY-MM-DD format
        if (!dailyData[date]) {
            dailyData[date] = { sales: 0, purchases: 0, salesVAT: 0, purchasesVAT: 0 };
        }
        
        if (transaction.type === 'مبيعات') {
            dailyData[date].sales++;
            dailyData[date].salesVAT += transaction.vat_amount || 0;
        } else {
            dailyData[date].purchases++;
            dailyData[date].purchasesVAT += transaction.vat_amount || 0;
        }
    });
    
    const sortedDates = Object.keys(dailyData).sort();
    const salesData = sortedDates.map(date => dailyData[date].salesVAT);
    const purchasesData = sortedDates.map(date => dailyData[date].purchasesVAT);
    
    new Chart(trendCtx, {
        type: 'line',
        data: {
            labels: sortedDates.map(date => new Date(date).toLocaleDateString('ar-SA')),
            datasets: [
                {
                    label: 'ضريبة المبيعات',
                    data: salesData,
                    borderColor: 'rgba(40, 167, 69, 1)',
                    backgroundColor: 'rgba(40, 167, 69, 0.1)',
                    borderWidth: 2,
                    fill: true,
                    tension: 0.4
                },
                {
                    label: 'ضريبة المشتريات',
                    data: purchasesData,
                    borderColor: 'rgba(23, 162, 184, 1)',
                    backgroundColor: 'rgba(23, 162, 184, 0.1)',
                    borderWidth: 2,
                    fill: true,
                    tension: 0.4
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'top'
                },
                tooltip: {
                    backgroundColor: 'rgba(0, 0, 0, 0.8)',
                    titleColor: '#fff',
                    bodyColor: '#fff',
                    borderColor: '#28a745',
                    borderWidth: 1,
                    cornerRadius: 6,
                    callbacks: {
                        label: function(context) {
                            return context.dataset.label + ': ' + context.parsed.y.toLocaleString('ar-SA', {minimumFractionDigits: 2}) + ' ريال';
                        }
                    }
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        callback: function(value) {
                            return value.toLocaleString('ar-SA') + ' ريال';
                        }
                    }
                }
            }
        }
    });

    // رسم دائري لتوزيع المعاملات حسب النوع
    const typeCtx = document.getElementById('transactionTypeChart').getContext('2d');
    
    const salesCount = transactions.filter(t => t.type === 'مبيعات').length;
    const purchasesCount = transactions.filter(t => t.type === 'مشتريات').length;
    
    new Chart(typeCtx, {
        type: 'doughnut',
        data: {
            labels: ['فواتير المبيعات', 'فواتير المشتريات'],
            datasets: [{
                data: [salesCount, purchasesCount],
                backgroundColor: [
                    'rgba(40, 167, 69, 0.8)',
                    'rgba(23, 162, 184, 0.8)'
                ],
                borderColor: [
                    'rgba(40, 167, 69, 1)',
                    'rgba(23, 162, 184, 1)'
                ],
                borderWidth: 2
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            cutout: '50%',
            plugins: {
                legend: {
                    position: 'bottom',
                    labels: {
                        padding: 20,
                        usePointStyle: true
                    }
                },
                tooltip: {
                    backgroundColor: 'rgba(0, 0, 0, 0.8)',
                    titleColor: '#fff',
                    bodyColor: '#fff',
                    borderColor: '#28a745',
                    borderWidth: 1,
                    cornerRadius: 6,
                    callbacks: {
                        label: function(context) {
                            const label = context.label || '';
                            const value = context.parsed;
                            const total = salesCount + purchasesCount;
                            const percentage = total > 0 ? ((value / total) * 100).toFixed(1) : 0;
                            return label + ': ' + value + ' فاتورة (' + percentage + '%)';
                        }
                    }
                }
            }
        }
    });

    // تحديث الصفحة عند تغيير التاريخ
    const startDate = document.getElementById('start_date');
    const endDate = document.getElementById('end_date');
    
    if (startDate && endDate) {
        startDate.addEventListener('change', function() {
            if (endDate.value && this.value > endDate.value) {
                endDate.value = this.value;
            }
        });
        
        endDate.addEventListener('change', function() {
            if (startDate.value && this.value < startDate.value) {
                startDate.value = this.value;
            }
        });
    }
});
</script>
@endpush
@endsection