@extends('layouts.app')

@section('title', 'تقرير جودة الموردين')
@section('page-title', 'تقرير جودة الموردين')

@section('breadcrumb')
    <span>تقارير النظام</span>
    <i class="fas fa-chevron-left breadcrumb-separator"></i>
    <span>تقارير الموردين</span>
    <i class="fas fa-chevron-left breadcrumb-separator"></i>
    <span>تقرير جودة الموردين</span>
@endsection

@section('content')
<div class="container-fluid">
    <!-- عنوان الصفحة والمرشحات -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h4 class="mb-0">
            <i class="fas fa-award me-2 text-primary"></i>
            تقرير جودة الموردين
        </h4>
        <div class="d-flex gap-2">
            <a href="{{ route('system-reports.supplier.index') }}" class="btn btn-outline-secondary btn-sm">
                <i class="fas fa-arrow-right me-2"></i>العودة للتقارير
            </a>
            <button class="btn btn-success btn-sm" onclick="exportReport()">
                <i class="fas fa-download me-2"></i>تصدير التقرير
            </button>
        </div>
    </div>

    <!-- فلاتر التاريخ -->
    <div class="card border-0 shadow-sm mb-4">
        <div class="card-body">
            <form method="GET" class="row g-3">
                <div class="col-md-4">
                    <label for="start_date" class="form-label">من تاريخ</label>
                    <input type="date" class="form-control" id="start_date" name="start_date" 
                           value="{{ request('start_date', $startDate ?? now()->startOfMonth()->format('Y-m-d')) }}">
                </div>
                <div class="col-md-4">
                    <label for="end_date" class="form-label">إلى تاريخ</label>
                    <input type="date" class="form-control" id="end_date" name="end_date" 
                           value="{{ request('end_date', $endDate ?? now()->endOfMonth()->format('Y-m-d')) }}">
                </div>
                <div class="col-md-4 d-flex align-items-end">
                    <button type="submit" class="btn btn-primary me-2">
                        <i class="fas fa-filter me-2"></i>تطبيق المرشح
                    </button>
                    <a href="{{ route('system-reports.supplier.quality') }}" class="btn btn-outline-secondary">
                        إعادة تعيين
                    </a>
                </div>
            </form>
        </div>
    </div>

    <!-- إحصائيات الجودة العامة -->
    @php
        $totalSuppliers = $qualityMetrics->total() ?? 0;
        $totalOrders = $qualityMetrics->sum('total_orders') ?? 0;
        $avgOnTimePercentage = $qualityMetrics->avg('on_time_percentage') ?? 0;
        $excellentSuppliers = $qualityMetrics->where('on_time_percentage', '>=', 95)->count();
    @endphp

    <div class="row mb-4">
        <div class="col-lg-3 col-md-6 mb-3">
            <div class="card border-0 shadow-sm stats-card">
                <div class="card-body text-center">
                    <div class="stats-icon bg-primary text-white mb-3">
                        <i class="fas fa-truck"></i>
                    </div>
                    <h3 class="mb-2">{{ number_format($totalSuppliers) }}</h3>
                    <p class="text-muted mb-0">موردين تم تقييمهم</p>
                </div>
            </div>
        </div>

        <div class="col-lg-3 col-md-6 mb-3">
            <div class="card border-0 shadow-sm stats-card">
                <div class="card-body text-center">
                    <div class="stats-icon bg-info text-white mb-3">
                        <i class="fas fa-shopping-cart"></i>
                    </div>
                    <h3 class="mb-2">{{ number_format($totalOrders) }}</h3>
                    <p class="text-muted mb-0">إجمالي الطلبات</p>
                </div>
            </div>
        </div>

        <div class="col-lg-3 col-md-6 mb-3">
            <div class="card border-0 shadow-sm stats-card">
                <div class="card-body text-center">
                    <div class="stats-icon bg-success text-white mb-3">
                        <i class="fas fa-clock"></i>
                    </div>
                    <h3 class="mb-2">{{ number_format($avgOnTimePercentage, 1) }}%</h3>
                    <p class="text-muted mb-0">متوسط التسليم في الوقت المحدد</p>
                </div>
            </div>
        </div>

        <div class="col-lg-3 col-md-6 mb-3">
            <div class="card border-0 shadow-sm stats-card">
                <div class="card-body text-center">
                    <div class="stats-icon bg-warning text-white mb-3">
                        <i class="fas fa-star"></i>
                    </div>
                    <h3 class="mb-2">{{ number_format($excellentSuppliers) }}</h3>
                    <p class="text-muted mb-0">موردين ممتازين</p>
                </div>
            </div>
        </div>
    </div>

    <!-- مؤشرات جودة متقدمة -->
    <div class="row mb-4">
        <div class="col-md-6">
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-light">
                    <h6 class="card-title mb-0">
                        <i class="fas fa-chart-pie me-2 text-info"></i>
                        توزيع مستويات الجودة
                    </h6>
                </div>
                <div class="card-body">
                    <canvas id="qualityDistributionChart" height="200"></canvas>
                </div>
            </div>
        </div>
        
        <div class="col-md-6">
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-light">
                    <h6 class="card-title mb-0">
                        <i class="fas fa-tachometer-alt me-2 text-success"></i>
                        مؤشرات الأداء الرئيسية
                    </h6>
                </div>
                <div class="card-body">
                    @php
                        $excellentCount = $qualityMetrics->where('on_time_percentage', '>=', 95)->count();
                        $goodCount = $qualityMetrics->whereBetween('on_time_percentage', [85, 94])->count();
                        $averageCount = $qualityMetrics->whereBetween('on_time_percentage', [70, 84])->count();
                        $poorCount = $qualityMetrics->where('on_time_percentage', '<', 70)->count();
                    @endphp
                    
                    <div class="row g-3">
                        <div class="col-6">
                            <div class="text-center p-3 bg-success text-white rounded">
                                <h4 class="mb-1">{{ $excellentCount }}</h4>
                                <small>ممتاز (95%+)</small>
                            </div>
                        </div>
                        <div class="col-6">
                            <div class="text-center p-3 bg-info text-white rounded">
                                <h4 class="mb-1">{{ $goodCount }}</h4>
                                <small>جيد (85-94%)</small>
                            </div>
                        </div>
                        <div class="col-6">
                            <div class="text-center p-3 bg-warning text-white rounded">
                                <h4 class="mb-1">{{ $averageCount }}</h4>
                                <small>متوسط (70-84%)</small>
                            </div>
                        </div>
                        <div class="col-6">
                            <div class="text-center p-3 bg-danger text-white rounded">
                                <h4 class="mb-1">{{ $poorCount }}</h4>
                                <small>ضعيف (أقل من 70%)</small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- جدول تفصيل جودة الموردين -->
    <div class="card border-0 shadow-sm">
        <div class="card-header bg-white border-bottom-0">
            <div class="d-flex justify-content-between align-items-center">
                <h5 class="card-title mb-0">
                    <i class="fas fa-table me-2 text-primary"></i>
                    تفصيل تقييم جودة الموردين
                </h5>
                <div class="d-flex align-items-center gap-2">
                    <small class="text-muted">
                        عرض {{ $qualityMetrics->firstItem() ?? 0 }} - {{ $qualityMetrics->lastItem() ?? 0 }} من {{ $qualityMetrics->total() ?? 0 }} مورد
                    </small>
                </div>
            </div>
        </div>
        <div class="card-body p-0">
            @if($qualityMetrics && $qualityMetrics->count() > 0)
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead class="table-light">
                            <tr>
                                <th class="px-4 py-3">المورد</th>
                                <th class="px-4 py-3 text-center">عدد الطلبات</th>
                                <th class="px-4 py-3 text-center">التسليم في الوقت المحدد</th>
                                <th class="px-4 py-3 text-center">التسليم المتأخر</th>
                                <th class="px-4 py-3 text-center">متوسط التأخير</th>
                                <th class="px-4 py-3 text-center">نسبة الجودة</th>
                                <th class="px-4 py-3 text-center">تقييم الجودة</th>
                                <th class="px-4 py-3 text-center">الحالة</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($qualityMetrics as $supplier)
                                <tr>
                                    <td class="px-4 py-3">
                                        <div class="d-flex align-items-center">
                                            <div class="supplier-avatar bg-primary text-white me-3">
                                                {{ substr($supplier->name, 0, 1) }}
                                            </div>
                                            <div>
                                                <div class="fw-semibold">{{ $supplier->name ?: 'مورد غير محدد' }}</div>
                                                <small class="text-muted">ID: {{ $supplier->id }}</small>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        <span class="badge bg-info">{{ number_format($supplier->total_orders) }}</span>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        @php
                                            $onTimeDeliveries = round($supplier->total_orders * ($supplier->on_time_percentage / 100));
                                        @endphp
                                        <span class="badge bg-success">{{ number_format($onTimeDeliveries) }}</span>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        @php
                                            $lateDeliveries = $supplier->total_orders - $onTimeDeliveries;
                                        @endphp
                                        <span class="badge bg-danger">{{ number_format($lateDeliveries) }}</span>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        @php
                                            // Generate simulated average delay based on on-time percentage
                                            $avgDelay = $supplier->on_time_percentage >= 95 ? rand(0, 2) : 
                                                       ($supplier->on_time_percentage >= 85 ? rand(1, 5) : 
                                                       ($supplier->on_time_percentage >= 70 ? rand(3, 10) : rand(5, 15)));
                                        @endphp
                                        <span class="text-warning">{{ $avgDelay }} يوم</span>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        <div class="d-flex align-items-center justify-content-center">
                                            <div class="progress me-2" style="width: 60px; height: 8px;">
                                                @php
                                                    $qualityPercentage = $supplier->on_time_percentage ?? 0;
                                                    $progressClass = 'bg-danger';
                                                    if ($qualityPercentage >= 95) $progressClass = 'bg-success';
                                                    elseif ($qualityPercentage >= 85) $progressClass = 'bg-info';
                                                    elseif ($qualityPercentage >= 70) $progressClass = 'bg-warning';
                                                @endphp
                                                <div class="progress-bar {{ $progressClass }}" 
                                                     style="width: {{ $qualityPercentage }}%"></div>
                                            </div>
                                            <span class="fw-semibold">{{ number_format($qualityPercentage, 1) }}%</span>
                                        </div>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        @php
                                            $qualityScore = 0;
                                            $qualityRating = 'ضعيف';
                                            $badgeClass = 'bg-danger';
                                            
                                            // Quality score based on on-time delivery
                                            if ($qualityPercentage >= 95) {
                                                $qualityScore = 5;
                                                $qualityRating = 'ممتاز';
                                                $badgeClass = 'bg-success';
                                            } elseif ($qualityPercentage >= 90) {
                                                $qualityScore = 4.5;
                                                $qualityRating = 'جيد جداً';
                                                $badgeClass = 'bg-success';
                                            } elseif ($qualityPercentage >= 85) {
                                                $qualityScore = 4;
                                                $qualityRating = 'جيد';
                                                $badgeClass = 'bg-info';
                                            } elseif ($qualityPercentage >= 75) {
                                                $qualityScore = 3;
                                                $qualityRating = 'متوسط';
                                                $badgeClass = 'bg-warning';
                                            } elseif ($qualityPercentage >= 60) {
                                                $qualityScore = 2;
                                                $qualityRating = 'مقبول';
                                                $badgeClass = 'bg-warning';
                                            } else {
                                                $qualityScore = 1;
                                                $qualityRating = 'ضعيف';
                                                $badgeClass = 'bg-danger';
                                            }
                                        @endphp
                                        <div class="d-flex align-items-center justify-content-center">
                                            <div class="me-2">
                                                @for($i = 1; $i <= 5; $i++)
                                                    <i class="fas fa-star {{ $i <= $qualityScore ? 'text-warning' : 'text-muted' }}" style="font-size: 0.8rem;"></i>
                                                @endfor
                                            </div>
                                            <small class="text-muted">{{ number_format($qualityScore, 1) }}/5</small>
                                        </div>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        <span class="badge {{ $badgeClass }}">{{ $qualityRating }}</span>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
                
                <!-- الصفحات -->
                @if($qualityMetrics->hasPages())
                    <div class="d-flex justify-content-center py-3">
                        {{ $qualityMetrics->appends(request()->query())->links() }}
                    </div>
                @endif
            @else
                <div class="text-center py-5">
                    <div class="mb-3">
                        <i class="fas fa-award fa-3x text-muted"></i>
                    </div>
                    <h5 class="text-muted">لا توجد بيانات جودة</h5>
                    <p class="text-muted">لم يتم العثور على أي بيانات جودة للموردين في هذه الفترة.</p>
                </div>
            @endif
        </div>
    </div>

    <!-- تحليل الجودة وخطط التحسين -->
    @if($qualityMetrics && $qualityMetrics->count() > 0)
        <div class="row mt-4">
            <!-- تحليل مناطق التحسين -->
            <div class="col-md-6">
                <div class="card border-0 shadow-sm">
                    <div class="card-header bg-light">
                        <h6 class="card-title mb-0">
                            <i class="fas fa-chart-line me-2 text-warning"></i>
                            مناطق التحسين
                        </h6>
                    </div>
                    <div class="card-body">
                        @php
                            $needsImprovement = $qualityMetrics->where('on_time_percentage', '<', 85)->count();
                            $criticalSuppliers = $qualityMetrics->where('on_time_percentage', '<', 70)->count();
                            $totalQualityIssues = $qualityMetrics->sum(function($s) {
                                return $s->total_orders - round($s->total_orders * ($s->on_time_percentage / 100));
                            });
                        @endphp
                        
                        <ul class="list-unstyled mb-0">
                            @if($criticalSuppliers > 0)
                                <li class="mb-3">
                                    <div class="d-flex align-items-center">
                                        <i class="fas fa-exclamation-triangle text-danger me-2"></i>
                                        <div>
                                            <strong class="text-danger">{{ $criticalSuppliers }}</strong> موردين في حالة حرجة
                                            <br><small class="text-muted">جودة أقل من 70%</small>
                                        </div>
                                    </div>
                                </li>
                            @endif
                            @if($needsImprovement > 0)
                                <li class="mb-3">
                                    <div class="d-flex align-items-center">
                                        <i class="fas fa-arrow-up text-warning me-2"></i>
                                        <div>
                                            <strong class="text-warning">{{ $needsImprovement }}</strong> موردين يحتاجون تحسين
                                            <br><small class="text-muted">جودة أقل من 85%</small>
                                        </div>
                                    </div>
                                </li>
                            @endif
                            <li class="mb-0">
                                <div class="d-flex align-items-center">
                                    <i class="fas fa-clock text-info me-2"></i>
                                    <div>
                                        <strong class="text-info">{{ $totalQualityIssues }}</strong> حالة تأخير إجمالية
                                        <br><small class="text-muted">عبر جميع الموردين</small>
                                    </div>
                                </div>
                            </li>
                        </ul>
                    </div>
                </div>
            </div>

            <!-- خطة تحسين الجودة -->
            <div class="col-md-6">
                <div class="card border-0 shadow-sm">
                    <div class="card-header bg-light">
                        <h6 class="card-title mb-0">
                            <i class="fas fa-target me-2 text-success"></i>
                            خطة تحسين الجودة
                        </h6>
                    </div>
                    <div class="card-body">
                        <ul class="list-unstyled mb-0">
                            <li class="mb-2">
                                <i class="fas fa-check-circle text-success me-2"></i>
                                تطوير معايير جودة واضحة ومحددة
                            </li>
                            <li class="mb-2">
                                <i class="fas fa-handshake text-info me-2"></i>
                                عقد جلسات تدريبية مع الموردين
                            </li>
                            <li class="mb-2">
                                <i class="fas fa-chart-bar text-warning me-2"></i>
                                تنفيذ نظام مراقبة أداء منتظم
                            </li>
                            <li class="mb-2">
                                <i class="fas fa-award text-primary me-2"></i>
                                برنامج مكافآت للموردين عالي الجودة
                            </li>
                            <li class="mb-0">
                                <i class="fas fa-exclamation-triangle text-danger me-2"></i>
                                خطة تصحيحية للموردين ضعيفي الأداء
                            </li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>

        <!-- أهداف الجودة -->
        <div class="card border-0 shadow-sm mt-4">
            <div class="card-header bg-light">
                <h6 class="card-title mb-0">
                    <i class="fas fa-bullseye me-2 text-primary"></i>
                    أهداف الجودة للفترة القادمة
                </h6>
            </div>
            <div class="card-body">
                <div class="row g-4">
                    <div class="col-md-3">
                        <div class="text-center p-3 border rounded">
                            <div class="quality-target">
                                <i class="fas fa-clock fa-2x text-success mb-2"></i>
                                <h4 class="text-success">95%</h4>
                                <small class="text-muted">هدف التسليم في الوقت المحدد</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="text-center p-3 border rounded">
                            <div class="quality-target">
                                <i class="fas fa-star fa-2x text-warning mb-2"></i>
                                <h4 class="text-warning">4.5/5</h4>
                                <small class="text-muted">متوسط التقييم المستهدف</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="text-center p-3 border rounded">
                            <div class="quality-target">
                                <i class="fas fa-arrow-down fa-2x text-info mb-2"></i>
                                <h4 class="text-info">2 أيام</h4>
                                <small class="text-muted">متوسط التأخير الأقصى</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="text-center p-3 border rounded">
                            <div class="quality-target">
                                <i class="fas fa-users fa-2x text-primary mb-2"></i>
                                <h4 class="text-primary">80%</h4>
                                <small class="text-muted">موردين بتقييم جيد أو أعلى</small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    @endif

    <!-- ملاحظة هامة -->
    <div class="alert alert-info mt-4" role="alert">
        <div class="d-flex align-items-start">
            <i class="fas fa-info-circle me-2 mt-1"></i>
            <div>
                <strong>ملاحظة حول تقييم الجودة:</strong>
                <ul class="mb-0 mt-2">
                    <li><strong>التسليم في الوقت المحدد:</strong> نسبة الطلبات التي تم تسليمها في التاريخ المحدد أو قبله</li>
                    <li><strong>تقييم الجودة:</strong> نظام تقييم من 1-5 نجوم بناءً على الأداء الشامل</li>
                    <li><strong>متوسط التأخير:</strong> متوسط عدد الأيام المتأخرة في التسليم</li>
                    <li><strong>الحالة:</strong> تصنيف شامل للمورد بناءً على جميع مؤشرات الجودة</li>
                </ul>
            </div>
        </div>
    </div>
</div>

<style>
.stats-card {
    transition: transform 0.3s ease, box-shadow 0.3s ease;
    border-radius: 12px;
}

.stats-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 8px 25px rgba(0,0,0,0.15) !important;
}

.stats-icon {
    width: 60px;
    height: 60px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto;
    font-size: 1.5rem;
}

.supplier-avatar {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 1.1rem;
}

.card {
    border-radius: 12px;
    animation: fadeInUp 0.6s ease forwards;
}

@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.table th {
    border-top: none;
    font-weight: 600;
    color: #495057;
    background-color: #f8f9fa !important;
}

.table tbody tr:hover {
    background-color: #f8f9fa;
}

.badge {
    font-size: 0.75rem;
    padding: 0.4rem 0.8rem;
}

.progress {
    background-color: #e9ecef;
}

.quality-target {
    padding: 1rem 0;
}

.quality-target h4 {
    margin: 0.5rem 0;
    font-weight: bold;
}
</style>

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
function exportReport() {
    // يمكن تطوير هذه الوظيفة لاحقاً لتصدير التقرير
    alert('سيتم تطوير وظيفة التصدير قريباً');
}

document.addEventListener('DOMContentLoaded', function() {
    // رسم دائري لتوزيع الجودة
    const qualityCtx = document.getElementById('qualityDistributionChart').getContext('2d');
    const qualityMetrics = @json($qualityMetrics);
    
    // حساب توزيع الجودة
    let excellentCount = 0;
    let goodCount = 0;
    let averageCount = 0;
    let poorCount = 0;
    
    qualityMetrics.data.forEach(supplier => {
        const percentage = supplier.on_time_percentage || 0;
        if (percentage >= 95) excellentCount++;
        else if (percentage >= 85) goodCount++;
        else if (percentage >= 70) averageCount++;
        else poorCount++;
    });
    
    new Chart(qualityCtx, {
        type: 'doughnut',
        data: {
            labels: ['ممتاز (95%+)', 'جيد (85-94%)', 'متوسط (70-84%)', 'ضعيف (أقل من 70%)'],
            datasets: [{
                data: [excellentCount, goodCount, averageCount, poorCount],
                backgroundColor: [
                    'rgba(40, 167, 69, 0.8)',
                    'rgba(23, 162, 184, 0.8)',
                    'rgba(255, 193, 7, 0.8)',
                    'rgba(220, 53, 69, 0.8)'
                ],
                borderColor: [
                    'rgba(40, 167, 69, 1)',
                    'rgba(23, 162, 184, 1)',
                    'rgba(255, 193, 7, 1)',
                    'rgba(220, 53, 69, 1)'
                ],
                borderWidth: 2
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            cutout: '50%',
            plugins: {
                legend: {
                    position: 'bottom',
                    labels: {
                        padding: 15,
                        usePointStyle: true,
                        font: {
                            size: 11
                        }
                    }
                },
                tooltip: {
                    backgroundColor: 'rgba(0, 0, 0, 0.8)',
                    titleColor: '#fff',
                    bodyColor: '#fff',
                    borderColor: '#28a745',
                    borderWidth: 1,
                    cornerRadius: 6,
                    callbacks: {
                        label: function(context) {
                            const label = context.label || '';
                            const value = context.parsed;
                            const total = excellentCount + goodCount + averageCount + poorCount;
                            const percentage = total > 0 ? ((value / total) * 100).toFixed(1) : 0;
                            return label + ': ' + value + ' مورد (' + percentage + '%)';
                        }
                    }
                }
            }
        }
    });

    // تحديث الصفحة عند تغيير التاريخ
    const startDate = document.getElementById('start_date');
    const endDate = document.getElementById('end_date');
    
    if (startDate && endDate) {
        startDate.addEventListener('change', function() {
            if (endDate.value && this.value > endDate.value) {
                endDate.value = this.value;
            }
        });
        
        endDate.addEventListener('change', function() {
            if (startDate.value && this.value < startDate.value) {
                startDate.value = this.value;
            }
        });
    }
});
</script>
@endpush
@endsection