@extends('layouts.app')

@section('title', 'تحليل المدفوعات للموردين')
@section('page-title', 'تحليل المدفوعات للموردين')

@section('breadcrumb')
    <span>تقارير النظام</span>
    <i class="fas fa-chevron-left breadcrumb-separator"></i>
    <span>تقارير الموردين</span>
    <i class="fas fa-chevron-left breadcrumb-separator"></i>
    <span>تحليل المدفوعات للموردين</span>
@endsection

@section('content')
<div class="container-fluid">
    <!-- عنوان الصفحة والمرشحات -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h4 class="mb-0">
            <i class="fas fa-credit-card me-2 text-primary"></i>
            تحليل المدفوعات للموردين
        </h4>
        <div class="d-flex gap-2">
            <a href="{{ route('system-reports.supplier.index') }}" class="btn btn-outline-secondary btn-sm">
                <i class="fas fa-arrow-right me-2"></i>العودة للتقارير
            </a>
            <button class="btn btn-success btn-sm" onclick="exportReport()">
                <i class="fas fa-download me-2"></i>تصدير التقرير
            </button>
        </div>
    </div>

    <!-- فلاتر التاريخ -->
    <div class="card border-0 shadow-sm mb-4">
        <div class="card-body">
            <form method="GET" class="row g-3">
                <div class="col-md-4">
                    <label for="start_date" class="form-label">من تاريخ</label>
                    <input type="date" class="form-control" id="start_date" name="start_date" 
                           value="{{ request('start_date', $startDate ?? now()->startOfMonth()->format('Y-m-d')) }}">
                </div>
                <div class="col-md-4">
                    <label for="end_date" class="form-label">إلى تاريخ</label>
                    <input type="date" class="form-control" id="end_date" name="end_date" 
                           value="{{ request('end_date', $endDate ?? now()->endOfMonth()->format('Y-m-d')) }}">
                </div>
                <div class="col-md-4 d-flex align-items-end">
                    <button type="submit" class="btn btn-primary me-2">
                        <i class="fas fa-filter me-2"></i>تطبيق المرشح
                    </button>
                    <a href="{{ route('system-reports.supplier.payment-analysis') }}" class="btn btn-outline-secondary">
                        إعادة تعيين
                    </a>
                </div>
            </form>
        </div>
    </div>

    <!-- إحصائيات المدفوعات العامة -->
    <div class="row mb-4">
        <div class="col-lg-3 col-md-6 mb-3">
            <div class="card border-0 shadow-sm stats-card">
                <div class="card-body text-center">
                    <div class="stats-icon bg-primary text-white mb-3">
                        <i class="fas fa-receipt"></i>
                    </div>
                    <h3 class="mb-2">{{ number_format($paymentStats->total_purchases ?? 0, 2) }}</h3>
                    <p class="text-muted mb-0">إجمالي المشتريات (ريال)</p>
                </div>
            </div>
        </div>

        <div class="col-lg-3 col-md-6 mb-3">
            <div class="card border-0 shadow-sm stats-card">
                <div class="card-body text-center">
                    <div class="stats-icon bg-success text-white mb-3">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <h3 class="mb-2">{{ number_format($paymentStats->total_paid ?? 0, 2) }}</h3>
                    <p class="text-muted mb-0">إجمالي المدفوع (ريال)</p>
                </div>
            </div>
        </div>

        <div class="col-lg-3 col-md-6 mb-3">
            <div class="card border-0 shadow-sm stats-card">
                <div class="card-body text-center">
                    <div class="stats-icon bg-warning text-white mb-3">
                        <i class="fas fa-clock"></i>
                    </div>
                    <h3 class="mb-2">{{ number_format($paymentStats->total_outstanding ?? 0, 2) }}</h3>
                    <p class="text-muted mb-0">المبالغ المستحقة (ريال)</p>
                </div>
            </div>
        </div>

        <div class="col-lg-3 col-md-6 mb-3">
            <div class="card border-0 shadow-sm stats-card">
                <div class="card-body text-center">
                    <div class="stats-icon bg-info text-white mb-3">
                        <i class="fas fa-percentage"></i>
                    </div>
                    @php
                        $paymentRate = ($paymentStats->total_purchases ?? 0) > 0 
                            ? (($paymentStats->total_paid ?? 0) / ($paymentStats->total_purchases ?? 0)) * 100 
                            : 0;
                    @endphp
                    <h3 class="mb-2">{{ number_format($paymentRate, 1) }}%</h3>
                    <p class="text-muted mb-0">معدل السداد</p>
                </div>
            </div>
        </div>
    </div>

    <!-- تحليل حالة المدفوعات -->
    <div class="row mb-4">
        <div class="col-md-6">
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-light">
                    <h6 class="card-title mb-0">
                        <i class="fas fa-chart-pie me-2 text-info"></i>
                        توزيع المدفوعات
                    </h6>
                </div>
                <div class="card-body">
                    <canvas id="paymentDistributionChart" height="200"></canvas>
                </div>
            </div>
        </div>
        
        <div class="col-md-6">
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-light">
                    <h6 class="card-title mb-0">
                        <i class="fas fa-chart-bar me-2 text-success"></i>
                        ملخص حالة المدفوعات
                    </h6>
                </div>
                <div class="card-body">
                    @php
                        $totalPaid = $paymentAnalysis->sum('total_paid') ?? 0;
                        $totalOutstanding = $paymentAnalysis->sum('outstanding_balance') ?? 0;
                        $paidInvoices = $paymentAnalysis->sum('paid_invoices') ?? 0;
                        $overdueInvoices = $paymentAnalysis->sum('overdue_invoices') ?? 0;
                    @endphp
                    
                    <div class="row g-3">
                        <div class="col-6">
                            <div class="text-center p-3 bg-success text-white rounded">
                                <h4 class="mb-1">{{ number_format($paidInvoices) }}</h4>
                                <small>فواتير مدفوعة</small>
                            </div>
                        </div>
                        <div class="col-6">
                            <div class="text-center p-3 bg-danger text-white rounded">
                                <h4 class="mb-1">{{ number_format($overdueInvoices) }}</h4>
                                <small>فواتير متأخرة</small>
                            </div>
                        </div>
                        <div class="col-12">
                            <div class="d-flex justify-content-between align-items-center p-3 bg-light rounded">
                                <span class="text-muted">نسبة السداد في الوقت المحدد:</span>
                                @php
                                    $totalInvoices = $paidInvoices + $overdueInvoices;
                                    $onTimeRate = $totalInvoices > 0 ? ($paidInvoices / $totalInvoices) * 100 : 0;
                                @endphp
                                <strong class="{{ $onTimeRate >= 80 ? 'text-success' : ($onTimeRate >= 60 ? 'text-warning' : 'text-danger') }}">
                                    {{ number_format($onTimeRate, 1) }}%
                                </strong>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- جدول تفصيل تحليل المدفوعات -->
    <div class="card border-0 shadow-sm">
        <div class="card-header bg-white border-bottom-0">
            <div class="d-flex justify-content-between align-items-center">
                <h5 class="card-title mb-0">
                    <i class="fas fa-table me-2 text-primary"></i>
                    تفصيل تحليل المدفوعات للموردين
                </h5>
                <div class="d-flex align-items-center gap-2">
                    <small class="text-muted">
                        عرض {{ $paymentAnalysis->firstItem() ?? 0 }} - {{ $paymentAnalysis->lastItem() ?? 0 }} من {{ $paymentAnalysis->total() ?? 0 }} مورد
                    </small>
                </div>
            </div>
        </div>
        <div class="card-body p-0">
            @if($paymentAnalysis && $paymentAnalysis->count() > 0)
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead class="table-light">
                            <tr>
                                <th class="px-4 py-3">المورد</th>
                                <th class="px-4 py-3 text-center">شروط الدفع</th>
                                <th class="px-4 py-3 text-center">عدد الفواتير</th>
                                <th class="px-4 py-3 text-center">إجمالي المبلغ</th>
                                <th class="px-4 py-3 text-center">المدفوع</th>
                                <th class="px-4 py-3 text-center">المتبقي</th>
                                <th class="px-4 py-3 text-center">نسبة السداد</th>
                                <th class="px-4 py-3 text-center">فواتير متأخرة</th>
                                <th class="px-4 py-3 text-center">التقييم</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($paymentAnalysis as $supplier)
                                <tr>
                                    <td class="px-4 py-3">
                                        <div class="d-flex align-items-center">
                                            <div class="supplier-avatar bg-primary text-white me-3">
                                                {{ substr($supplier->name, 0, 1) }}
                                            </div>
                                            <div>
                                                <div class="fw-semibold">{{ $supplier->name ?: 'مورد غير محدد' }}</div>
                                                <small class="text-muted">ID: {{ $supplier->id }}</small>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        <span class="badge bg-light text-dark">
                                            {{ $supplier->payment_terms ?: 'غير محدد' }}
                                        </span>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        <span class="badge bg-info">{{ number_format($supplier->total_invoices) }}</span>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        <strong class="text-primary">{{ number_format($supplier->total_amount, 2) }} ريال</strong>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        <strong class="text-success">{{ number_format($supplier->total_paid, 2) }} ريال</strong>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        <strong class="text-warning">{{ number_format($supplier->outstanding_balance, 2) }} ريال</strong>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        @php
                                            $paymentPercentage = $supplier->total_amount > 0 
                                                ? ($supplier->total_paid / $supplier->total_amount) * 100 
                                                : 0;
                                        @endphp
                                        <div class="d-flex align-items-center justify-content-center">
                                            <div class="progress me-2" style="width: 60px; height: 8px;">
                                                @php
                                                    $progressClass = 'bg-danger';
                                                    if ($paymentPercentage >= 80) $progressClass = 'bg-success';
                                                    elseif ($paymentPercentage >= 60) $progressClass = 'bg-warning';
                                                    elseif ($paymentPercentage >= 40) $progressClass = 'bg-info';
                                                @endphp
                                                <div class="progress-bar {{ $progressClass }}" 
                                                     style="width: {{ $paymentPercentage }}%"></div>
                                            </div>
                                            <span class="fw-semibold">{{ number_format($paymentPercentage, 1) }}%</span>
                                        </div>
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        @if($supplier->overdue_invoices > 0)
                                            <span class="badge bg-danger">{{ number_format($supplier->overdue_invoices) }}</span>
                                        @else
                                            <span class="badge bg-success">0</span>
                                        @endif
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        @php
                                            $rating = 'ضعيف';
                                            $badgeClass = 'bg-danger';
                                            
                                            if ($paymentPercentage >= 95 && $supplier->overdue_invoices == 0) {
                                                $rating = 'ممتاز';
                                                $badgeClass = 'bg-success';
                                            } elseif ($paymentPercentage >= 80 && $supplier->overdue_invoices <= 2) {
                                                $rating = 'جيد';
                                                $badgeClass = 'bg-success';
                                            } elseif ($paymentPercentage >= 60) {
                                                $rating = 'مقبول';
                                                $badgeClass = 'bg-warning';
                                            } elseif ($paymentPercentage >= 40) {
                                                $rating = 'متأخر';
                                                $badgeClass = 'bg-warning';
                                            } else {
                                                $rating = 'مشكوك';
                                                $badgeClass = 'bg-danger';
                                            }
                                        @endphp
                                        <span class="badge {{ $badgeClass }}">{{ $rating }}</span>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
                
                <!-- الصفحات -->
                @if($paymentAnalysis->hasPages())
                    <div class="d-flex justify-content-center py-3">
                        {{ $paymentAnalysis->appends(request()->query())->links() }}
                    </div>
                @endif
            @else
                <div class="text-center py-5">
                    <div class="mb-3">
                        <i class="fas fa-credit-card fa-3x text-muted"></i>
                    </div>
                    <h5 class="text-muted">لا توجد بيانات مدفوعات</h5>
                    <p class="text-muted">لم يتم العثور على أي بيانات مدفوعات للموردين في هذه الفترة.</p>
                </div>
            @endif
        </div>
    </div>

    <!-- تحليل المخاطر والتوصيات -->
    @if($paymentAnalysis && $paymentAnalysis->count() > 0)
        <div class="row mt-4">
            <!-- تحليل المخاطر -->
            <div class="col-md-6">
                <div class="card border-0 shadow-sm">
                    <div class="card-header bg-light">
                        <h6 class="card-title mb-0">
                            <i class="fas fa-exclamation-triangle me-2 text-warning"></i>
                            تحليل مخاطر المدفوعات
                        </h6>
                    </div>
                    <div class="card-body">
                        @php
                            $highRiskSuppliers = $paymentAnalysis->filter(function($s) {
                                $paymentRate = $s->total_amount > 0 ? ($s->total_paid / $s->total_amount) * 100 : 0;
                                return $paymentRate < 60 || $s->overdue_invoices > 3;
                            })->count();
                            
                            $mediumRiskSuppliers = $paymentAnalysis->filter(function($s) {
                                $paymentRate = $s->total_amount > 0 ? ($s->total_paid / $s->total_amount) * 100 : 0;
                                return ($paymentRate >= 60 && $paymentRate < 80) || ($s->overdue_invoices >= 1 && $s->overdue_invoices <= 3);
                            })->count();
                            
                            $lowRiskSuppliers = $paymentAnalysis->filter(function($s) {
                                $paymentRate = $s->total_amount > 0 ? ($s->total_paid / $s->total_amount) * 100 : 0;
                                return $paymentRate >= 80 && $s->overdue_invoices == 0;
                            })->count();
                        @endphp
                        
                        <div class="row g-3">
                            <div class="col-12">
                                <div class="d-flex justify-content-between align-items-center p-3 bg-danger text-white rounded">
                                    <span>مخاطر عالية:</span>
                                    <strong>{{ $highRiskSuppliers }} مورد</strong>
                                </div>
                            </div>
                            <div class="col-12">
                                <div class="d-flex justify-content-between align-items-center p-3 bg-warning text-white rounded">
                                    <span>مخاطر متوسطة:</span>
                                    <strong>{{ $mediumRiskSuppliers }} مورد</strong>
                                </div>
                            </div>
                            <div class="col-12">
                                <div class="d-flex justify-content-between align-items-center p-3 bg-success text-white rounded">
                                    <span>مخاطر منخفضة:</span>
                                    <strong>{{ $lowRiskSuppliers }} مورد</strong>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- التوصيات -->
            <div class="col-md-6">
                <div class="card border-0 shadow-sm">
                    <div class="card-header bg-light">
                        <h6 class="card-title mb-0">
                            <i class="fas fa-lightbulb me-2 text-info"></i>
                            توصيات إدارة المدفوعات
                        </h6>
                    </div>
                    <div class="card-body">
                        <ul class="list-unstyled mb-0">
                            @if($highRiskSuppliers > 0)
                                <li class="mb-2">
                                    <i class="fas fa-exclamation-circle text-danger me-2"></i>
                                    مراجعة عاجلة للموردين عالي المخاطر ({{ $highRiskSuppliers }})
                                </li>
                            @endif
                            @if($paymentRate < 80)
                                <li class="mb-2">
                                    <i class="fas fa-chart-line text-warning me-2"></i>
                                    تحسين معدل السداد العام إلى 85%+
                                </li>
                            @endif
                            <li class="mb-2">
                                <i class="fas fa-clock text-info me-2"></i>
                                وضع نظام تذكير للفواتير المستحقة
                            </li>
                            <li class="mb-2">
                                <i class="fas fa-handshake text-success me-2"></i>
                                مراجعة شروط الدفع مع الموردين المتأخرين
                            </li>
                            <li class="mb-0">
                                <i class="fas fa-shield-alt text-primary me-2"></i>
                                تطوير سياسات ائتمان أكثر صرامة
                            </li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    @endif

    <!-- ملاحظة هامة -->
    <div class="alert alert-info mt-4" role="alert">
        <div class="d-flex align-items-start">
            <i class="fas fa-info-circle me-2 mt-1"></i>
            <div>
                <strong>ملاحظة حول تحليل المدفوعات:</strong>
                <ul class="mb-0 mt-2">
                    <li><strong>نسبة السداد:</strong> المبلغ المدفوع مقسوماً على إجمالي المبلغ المستحق</li>
                    <li><strong>التقييم الائتماني:</strong> يأخذ في الاعتبار نسبة السداد وعدد الفواتير المتأخرة</li>
                    <li><strong>المخاطر العالية:</strong> نسبة سداد أقل من 60% أو أكثر من 3 فواتير متأخرة</li>
                    <li><strong>شروط الدفع:</strong> الفترة المتفق عليها لسداد الفواتير</li>
                </ul>
            </div>
        </div>
    </div>
</div>

<style>
.stats-card {
    transition: transform 0.3s ease, box-shadow 0.3s ease;
    border-radius: 12px;
}

.stats-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 8px 25px rgba(0,0,0,0.15) !important;
}

.stats-icon {
    width: 60px;
    height: 60px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto;
    font-size: 1.5rem;
}

.supplier-avatar {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 1.1rem;
}

.card {
    border-radius: 12px;
    animation: fadeInUp 0.6s ease forwards;
}

@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.table th {
    border-top: none;
    font-weight: 600;
    color: #495057;
    background-color: #f8f9fa !important;
}

.table tbody tr:hover {
    background-color: #f8f9fa;
}

.badge {
    font-size: 0.75rem;
    padding: 0.4rem 0.8rem;
}

.progress {
    background-color: #e9ecef;
}
</style>

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
function exportReport() {
    // يمكن تطوير هذه الوظيفة لاحقاً لتصدير التقرير
    alert('سيتم تطوير وظيفة التصدير قريباً');
}

document.addEventListener('DOMContentLoaded', function() {
    // رسم دائري لتوزيع المدفوعات
    const paymentCtx = document.getElementById('paymentDistributionChart').getContext('2d');
    const paymentStats = @json($paymentStats);
    
    const totalPaid = paymentStats.total_paid || 0;
    const totalOutstanding = paymentStats.total_outstanding || 0;
    
    new Chart(paymentCtx, {
        type: 'doughnut',
        data: {
            labels: ['مدفوع', 'مستحق'],
            datasets: [{
                data: [totalPaid, totalOutstanding],
                backgroundColor: [
                    'rgba(40, 167, 69, 0.8)',
                    'rgba(255, 193, 7, 0.8)'
                ],
                borderColor: [
                    'rgba(40, 167, 69, 1)',
                    'rgba(255, 193, 7, 1)'
                ],
                borderWidth: 2
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            cutout: '50%',
            plugins: {
                legend: {
                    position: 'bottom',
                    labels: {
                        padding: 20,
                        usePointStyle: true,
                        font: {
                            size: 12
                        }
                    }
                },
                tooltip: {
                    backgroundColor: 'rgba(0, 0, 0, 0.8)',
                    titleColor: '#fff',
                    bodyColor: '#fff',
                    borderColor: '#28a745',
                    borderWidth: 1,
                    cornerRadius: 6,
                    callbacks: {
                        label: function(context) {
                            const label = context.label || '';
                            const value = context.parsed;
                            const total = totalPaid + totalOutstanding;
                            const percentage = total > 0 ? ((value / total) * 100).toFixed(1) : 0;
                            return label + ': ' + value.toLocaleString('ar-SA', {minimumFractionDigits: 2}) + ' ريال (' + percentage + '%)';
                        }
                    }
                }
            }
        }
    });

    // تحديث الصفحة عند تغيير التاريخ
    const startDate = document.getElementById('start_date');
    const endDate = document.getElementById('end_date');
    
    if (startDate && endDate) {
        startDate.addEventListener('change', function() {
            if (endDate.value && this.value > endDate.value) {
                endDate.value = this.value;
            }
        });
        
        endDate.addEventListener('change', function() {
            if (startDate.value && this.value < startDate.value) {
                startDate.value = this.value;
            }
        });
    }
});
</script>
@endpush
@endsection