@extends('layouts.app')

@section('title', 'ملخص تقارير العملاء')
@section('page-title', 'ملخص تقارير العملاء')

@section('breadcrumb')
    <span>تقارير النظام</span>
    <i class="fas fa-chevron-left breadcrumb-separator"></i>
    <span>تقارير العملاء</span>
    <i class="fas fa-chevron-left breadcrumb-separator"></i>
    <span>الملخص الشامل</span>
@endsection

@section('page-actions')
    <div class="d-flex gap-2">
        <button type="button" class="btn btn-success rounded-pill px-4" onclick="exportReport()">
            <i class="fas fa-download me-2"></i>
            تصدير التقرير
        </button>
        <button type="button" class="btn btn-info rounded-pill px-4" onclick="refreshReport()">
            <i class="fas fa-sync-alt me-2"></i>
            تحديث البيانات
        </button>
        <button type="button" class="btn btn-outline-primary rounded-pill px-4" data-bs-toggle="modal" data-bs-target="#filterModal">
            <i class="fas fa-filter me-2"></i>
            تخصيص التقرير
        </button>
    </div>
@endsection

@push('styles')
<style>
    :root {
        --gradient-primary: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        --gradient-success: linear-gradient(135deg, #11998e 0%, #38ef7d 100%);
        --gradient-warning: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
        --gradient-info: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
        --gradient-danger: linear-gradient(135deg, #fa709a 0%, #fee140 100%);
        --shadow-modern: 0 10px 30px rgba(0,0,0,0.1);
        --shadow-hover: 0 20px 40px rgba(0,0,0,0.15);
        --border-radius: 20px;
    }

    .dashboard-header {
        background: var(--gradient-primary);
        border-radius: var(--border-radius);
        padding: 2.5rem;
        margin-bottom: 2rem;
        color: white;
        position: relative;
        overflow: hidden;
    }

    .dashboard-header::before {
        content: '';
        position: absolute;
        top: 0;
        right: 0;
        width: 100%;
        height: 100%;
        background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="pattern" width="20" height="20" patternUnits="userSpaceOnUse"><circle cx="10" cy="10" r="1" fill="white" fill-opacity="0.1"/></pattern></defs><rect width="100" height="100" fill="url(%23pattern)"/></svg>');
        pointer-events: none;
    }

    .metric-card {
        background: white;
        border-radius: var(--border-radius);
        padding: 2rem;
        box-shadow: var(--shadow-modern);
        transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
        position: relative;
        overflow: hidden;
        border: none;
    }

    .metric-card::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 4px;
        transition: all 0.3s ease;
    }

    .metric-card.primary::before {
        background: var(--gradient-primary);
    }

    .metric-card.success::before {
        background: var(--gradient-success);
    }

    .metric-card.warning::before {
        background: var(--gradient-warning);
    }

    .metric-card.info::before {
        background: var(--gradient-info);
    }

    .metric-card:hover {
        transform: translateY(-10px) scale(1.02);
        box-shadow: var(--shadow-hover);
    }

    .metric-card:hover::before {
        height: 6px;
    }

    .metric-icon {
        width: 70px;
        height: 70px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 2rem;
        color: white;
        margin-bottom: 1.5rem;
        position: relative;
    }

    .metric-icon.primary {
        background: var(--gradient-primary);
    }

    .metric-icon.success {
        background: var(--gradient-success);
    }

    .metric-icon.warning {
        background: var(--gradient-warning);
    }

    .metric-icon.info {
        background: var(--gradient-info);
    }

    .metric-number {
        font-size: 2.5rem;
        font-weight: 900;
        line-height: 1;
        margin-bottom: 0.5rem;
        background: linear-gradient(45deg, #2c3e50, #3498db);
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
        background-clip: text;
    }

    .metric-label {
        font-size: 1rem;
        color: #6c757d;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 1px;
    }

    .metric-trend {
        position: absolute;
        top: 1rem;
        right: 1rem;
        padding: 0.5rem 1rem;
        border-radius: 50px;
        font-size: 0.8rem;
        font-weight: 700;
    }

    .trend-up {
        background: rgba(34, 197, 94, 0.1);
        color: #22c55e;
    }

    .trend-down {
        background: rgba(239, 68, 68, 0.1);
        color: #ef4444;
    }

    .analytics-card {
        background: white;
        border-radius: var(--border-radius);
        box-shadow: var(--shadow-modern);
        border: none;
        overflow: hidden;
        transition: all 0.3s ease;
    }

    .analytics-card:hover {
        transform: translateY(-5px);
        box-shadow: var(--shadow-hover);
    }

    .analytics-header {
        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
        padding: 1.5rem 2rem;
        border-bottom: 1px solid #dee2e6;
        position: relative;
    }

    .analytics-header::before {
        content: '';
        position: absolute;
        bottom: 0;
        left: 50%;
        transform: translateX(-50%);
        width: 60px;
        height: 3px;
        background: var(--gradient-primary);
        border-radius: 3px;
    }

    .chart-container {
        padding: 2rem;
        min-height: 400px;
        display: flex;
        align-items: center;
        justify-content: center;
        position: relative;
    }

    .data-table {
        border-radius: var(--border-radius);
        overflow: hidden;
        box-shadow: var(--shadow-modern);
        background: white;
    }

    .data-table thead {
        background: linear-gradient(135deg, #2c3e50 0%, #34495e 100%);
        color: white;
    }

    .data-table thead th {
        border: none;
        padding: 1.5rem;
        font-weight: 700;
        text-transform: uppercase;
        letter-spacing: 1px;
        font-size: 0.9rem;
    }

    .data-table tbody tr {
        transition: all 0.3s ease;
        border-bottom: 1px solid #f8f9fa;
    }

    .data-table tbody tr:hover {
        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 50%, #f8f9fa 100%);
        transform: scale(1.01);
        box-shadow: 0 5px 15px rgba(0,0,0,0.1);
    }

    .data-table tbody td {
        padding: 1.5rem;
        border: none;
        vertical-align: middle;
    }

    .customer-avatar {
        width: 50px;
        height: 50px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-weight: bold;
        font-size: 1.2rem;
        margin-left: 15px;
        flex-shrink: 0;
        text-transform: uppercase;
    }

    .progress-modern {
        height: 8px;
        border-radius: 10px;
        background: #e9ecef;
        overflow: hidden;
        position: relative;
    }

    .progress-bar-animated {
        background: var(--gradient-success);
        transition: width 0.6s ease;
        position: relative;
        overflow: hidden;
    }

    .progress-bar-animated::after {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        bottom: 0;
        right: 0;
        background-image: linear-gradient(
            45deg,
            rgba(255, 255, 255, 0.2) 25%,
            transparent 25%,
            transparent 50%,
            rgba(255, 255, 255, 0.2) 50%,
            rgba(255, 255, 255, 0.2) 75%,
            transparent 75%,
            transparent
        );
        background-size: 1rem 1rem;
        animation: progress-bar-stripes 1s linear infinite;
    }

    @keyframes progress-bar-stripes {
        0% {
            background-position: 1rem 0;
        }
        100% {
            background-position: 0 0;
        }
    }

    .filter-chip {
        display: inline-flex;
        align-items: center;
        padding: 0.5rem 1rem;
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        border-radius: 25px;
        font-size: 0.8rem;
        font-weight: 600;
        margin: 0.25rem;
        transition: all 0.3s ease;
    }

    .filter-chip:hover {
        transform: translateY(-2px);
        box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
    }

    .filter-chip .btn-close {
        margin-right: 0.5rem;
        margin-left: 0;
        font-size: 0.7rem;
        opacity: 0.8;
    }

    .empty-state {
        text-align: center;
        padding: 4rem 2rem;
        color: #6c757d;
    }

    .empty-state-icon {
        font-size: 5rem;
        margin-bottom: 1.5rem;
        opacity: 0.3;
        background: var(--gradient-primary);
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
        background-clip: text;
    }

    .loading-spinner {
        display: inline-block;
        width: 2rem;
        height: 2rem;
        border: 3px solid rgba(0, 0, 0, 0.1);
        border-radius: 50%;
        border-top-color: #3498db;
        animation: spin 1s ease-in-out infinite;
    }

    @keyframes spin {
        to {
            transform: rotate(360deg);
        }
    }

    @keyframes fadeInUp {
        from {
            opacity: 0;
            transform: translateY(30px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    @keyframes slideInRight {
        from {
            opacity: 0;
            transform: translateX(30px);
        }
        to {
            opacity: 1;
            transform: translateX(0);
        }
    }

    .fade-in-up {
        animation: fadeInUp 0.6s ease-out;
    }

    .slide-in-right {
        animation: slideInRight 0.6s ease-out;
    }

    .metric-card:nth-child(1) { animation-delay: 0.1s; }
    .metric-card:nth-child(2) { animation-delay: 0.2s; }
    .metric-card:nth-child(3) { animation-delay: 0.3s; }
    .metric-card:nth-child(4) { animation-delay: 0.4s; }

    @media (max-width: 768px) {
        .dashboard-header {
            padding: 1.5rem;
            text-align: center;
        }
        
        .metric-card {
            padding: 1.5rem;
            margin-bottom: 1rem;
        }
        
        .metric-number {
            font-size: 2rem;
        }
        
        .chart-container {
            padding: 1rem;
            min-height: 300px;
        }
        
        .data-table thead th,
        .data-table tbody td {
            padding: 1rem;
        }
    }
</style>
@endpush

@section('content')
<!-- رأس لوحة التحكم -->
<div class="dashboard-header fade-in-up">
    <div class="row align-items-center">
        <div class="col-lg-8">
            <h1 class="display-5 fw-bold mb-2">
                <i class="fas fa-chart-line me-3"></i>
                ملخص تقارير العملاء
            </h1>
            <p class="lead mb-0 opacity-90">
                تحليل شامل لأداء العملاء وسلوكياتهم الشرائية مع رؤى تفصيلية للنمو
            </p>
        </div>
        <div class="col-lg-4 text-lg-end mt-3 mt-lg-0">
            <div class="d-flex justify-content-lg-end gap-2">
                <div class="bg-white bg-opacity-20 rounded-pill px-3 py-2">
                    <small class="d-block opacity-75">آخر تحديث</small>
                    <strong>{{ now()->format('Y-m-d H:i') }}</strong>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- البطاقات الإحصائية الرئيسية -->
<div class="row g-4 mb-4">
    <div class="col-lg-3 col-md-6">
        <div class="metric-card primary fade-in-up">
            <div class="d-flex align-items-start justify-content-between">
                <div>
                    <div class="metric-icon primary">
                        <i class="fas fa-users"></i>
                    </div>
                    <div class="metric-number" id="totalCustomers">1,247</div>
                    <div class="metric-label">إجمالي العملاء</div>
                </div>
                <div class="metric-trend trend-up">
                    <i class="fas fa-arrow-up me-1"></i>
                    12.5%
                </div>
            </div>
        </div>
    </div>

    <div class="col-lg-3 col-md-6">
        <div class="metric-card success fade-in-up">
            <div class="d-flex align-items-start justify-content-between">
                <div>
                    <div class="metric-icon success">
                        <i class="fas fa-user-check"></i>
                    </div>
                    <div class="metric-number" id="activeCustomers">1,089</div>
                    <div class="metric-label">عملاء نشطين</div>
                </div>
                <div class="metric-trend trend-up">
                    <i class="fas fa-arrow-up me-1"></i>
                    8.3%
                </div>
            </div>
        </div>
    </div>

    <div class="col-lg-3 col-md-6">
        <div class="metric-card warning fade-in-up">
            <div class="d-flex align-items-start justify-content-between">
                <div>
                    <div class="metric-icon warning">
                        <i class="fas fa-crown"></i>
                    </div>
                    <div class="metric-number" id="vipCustomers">158</div>
                    <div class="metric-label">عملاء مميزين</div>
                </div>
                <div class="metric-trend trend-up">
                    <i class="fas fa-arrow-up me-1"></i>
                    15.7%
                </div>
            </div>
        </div>
    </div>

    <div class="col-lg-3 col-md-6">
        <div class="metric-card info fade-in-up">
            <div class="d-flex align-items-start justify-content-between">
                <div>
                    <div class="metric-icon info">
                        <i class="fas fa-chart-bar"></i>
                    </div>
                    <div class="metric-number" id="avgOrderValue">2,450</div>
                    <div class="metric-label">متوسط قيمة الطلب</div>
                </div>
                <div class="metric-trend trend-down">
                    <i class="fas fa-arrow-down me-1"></i>
                    3.2%
                </div>
            </div>
        </div>
    </div>
</div>

<!-- الفلاتر النشطة -->
<div class="mb-4 fade-in-up" id="activeFilters" style="display: none;">
    <div class="d-flex align-items-center flex-wrap">
        <span class="me-2 fw-bold text-muted">الفلاتر النشطة:</span>
        <div id="filterChips"></div>
        <button type="button" class="btn btn-sm btn-outline-danger ms-2" onclick="clearAllFilters()">
            <i class="fas fa-times me-1"></i>
            إلغاء جميع الفلاتر
        </button>
    </div>
</div>

<!-- التحليلات والرسوم البيانية -->
<div class="row g-4 mb-4">
    <!-- رسم بياني للنمو -->
    <div class="col-lg-8">
        <div class="analytics-card slide-in-right">
            <div class="analytics-header">
                <h5 class="mb-0 fw-bold">
                    <i class="fas fa-chart-area me-2 text-primary"></i>
                    نمو قاعدة العملاء
                </h5>
                <small class="text-muted">نمو العملاء خلال الـ 12 شهر الماضية</small>
            </div>
            <div class="chart-container">
                <canvas id="customerGrowthChart"></canvas>
            </div>
        </div>
    </div>

    <!-- توزيع أنواع العملاء -->
    <div class="col-lg-4">
        <div class="analytics-card slide-in-right">
            <div class="analytics-header">
                <h5 class="mb-0 fw-bold">
                    <i class="fas fa-chart-pie me-2 text-success"></i>
                    توزيع العملاء
                </h5>
                <small class="text-muted">حسب النوع والحالة</small>
            </div>
            <div class="chart-container">
                <canvas id="customerDistributionChart"></canvas>
            </div>
        </div>
    </div>
</div>

<!-- جدول أفضل العملاء -->
<div class="analytics-card fade-in-up">
    <div class="analytics-header">
        <div class="d-flex justify-content-between align-items-center">
            <div>
                <h5 class="mb-0 fw-bold">
                    <i class="fas fa-star me-2 text-warning"></i>
                    أفضل 10 عملاء
                </h5>
                <small class="text-muted">مرتبين حسب قيمة المشتريات</small>
            </div>
            <div class="d-flex gap-2">
                <button type="button" class="btn btn-sm btn-outline-primary" onclick="refreshTopCustomers()">
                    <i class="fas fa-sync-alt me-1"></i>
                    تحديث
                </button>
                <button type="button" class="btn btn-sm btn-primary" onclick="viewAllCustomers()">
                    <i class="fas fa-eye me-1"></i>
                    عرض الكل
                </button>
            </div>
        </div>
    </div>
    
    <div class="table-responsive">
        <table class="table data-table">
            <thead>
                <tr>
                    <th>العميل</th>
                    <th>النوع</th>
                    <th>إجمالي المشتريات</th>
                    <th>عدد الطلبات</th>
                    <th>متوسط الطلب</th>
                    <th>النشاط</th>
                    <th>التقييم</th>
                </tr>
            </thead>
            <tbody id="topCustomersTable">
                <tr>
                    <td>
                        <div class="d-flex align-items-center">
                            <div class="customer-avatar" style="background: linear-gradient(135deg, #667eea, #764ba2);">
                                م
                            </div>
                            <div>
                                <div class="fw-bold">محمد أحمد السعيد</div>
                                <small class="text-muted">الرياض، المملكة العربية السعودية</small>
                            </div>
                        </div>
                    </td>
                    <td>
                        <span class="badge bg-primary rounded-pill">
                            <i class="fas fa-building me-1"></i>
                            شركة
                        </span>
                    </td>
                    <td>
                        <div class="fw-bold text-success">245,000 ريال</div>
                        <small class="text-muted">منذ 2023</small>
                    </td>
                    <td>
                        <div class="fw-bold">47</div>
                        <small class="text-muted">طلب</small>
                    </td>
                    <td>
                        <div class="fw-bold">5,213 ريال</div>
                    </td>
                    <td>
                        <div class="progress-modern">
                            <div class="progress-bar-animated" style="width: 95%"></div>
                        </div>
                        <small class="text-muted">95% نشاط</small>
                    </td>
                    <td>
                        <div class="text-warning">
                            <i class="fas fa-star"></i>
                            <i class="fas fa-star"></i>
                            <i class="fas fa-star"></i>
                            <i class="fas fa-star"></i>
                            <i class="fas fa-star"></i>
                            <small class="text-muted ms-1">5.0</small>
                        </div>
                    </td>
                </tr>
                <tr>
                    <td>
                        <div class="d-flex align-items-center">
                            <div class="customer-avatar" style="background: linear-gradient(135deg, #11998e, #38ef7d);">
                                ف
                            </div>
                            <div>
                                <div class="fw-bold">فاطمة محمد علي</div>
                                <small class="text-muted">جدة، المملكة العربية السعودية</small>
                            </div>
                        </div>
                    </td>
                    <td>
                        <span class="badge bg-info rounded-pill">
                            <i class="fas fa-user me-1"></i>
                            فرد
                        </span>
                    </td>
                    <td>
                        <div class="fw-bold text-success">189,500 ريال</div>
                        <small class="text-muted">منذ 2022</small>
                    </td>
                    <td>
                        <div class="fw-bold">62</div>
                        <small class="text-muted">طلب</small>
                    </td>
                    <td>
                        <div class="fw-bold">3,056 ريال</div>
                    </td>
                    <td>
                        <div class="progress-modern">
                            <div class="progress-bar-animated" style="width: 88%"></div>
                        </div>
                        <small class="text-muted">88% نشاط</small>
                    </td>
                    <td>
                        <div class="text-warning">
                            <i class="fas fa-star"></i>
                            <i class="fas fa-star"></i>
                            <i class="fas fa-star"></i>
                            <i class="fas fa-star"></i>
                            <i class="far fa-star"></i>
                            <small class="text-muted ms-1">4.8</small>
                        </div>
                    </td>
                </tr>
                <tr>
                    <td>
                        <div class="d-flex align-items-center">
                            <div class="customer-avatar" style="background: linear-gradient(135deg, #f093fb, #f5576c);">
                                ع
                            </div>
                            <div>
                                <div class="fw-bold">عبدالله خالد المطيري</div>
                                <small class="text-muted">الدمام، المملكة العربية السعودية</small>
                            </div>
                        </div>
                    </td>
                    <td>
                        <span class="badge bg-primary rounded-pill">
                            <i class="fas fa-building me-1"></i>
                            شركة
                        </span>
                    </td>
                    <td>
                        <div class="fw-bold text-success">156,700 ريال</div>
                        <small class="text-muted">منذ 2023</small>
                    </td>
                    <td>
                        <div class="fw-bold">33</div>
                        <small class="text-muted">طلب</small>
                    </td>
                    <td>
                        <div class="fw-bold">4,749 ريال</div>
                    </td>
                    <td>
                        <div class="progress-modern">
                            <div class="progress-bar-animated" style="width: 92%"></div>
                        </div>
                        <small class="text-muted">92% نشاط</small>
                    </td>
                    <td>
                        <div class="text-warning">
                            <i class="fas fa-star"></i>
                            <i class="fas fa-star"></i>
                            <i class="fas fa-star"></i>
                            <i class="fas fa-star"></i>
                            <i class="fas fa-star-half-alt"></i>
                            <small class="text-muted ms-1">4.5</small>
                        </div>
                    </td>
                </tr>
            </tbody>
        </table>
    </div>
</div>
@endsection

<!-- مودال تخصيص التقرير -->
<div class="modal fade" id="filterModal" tabindex="-1" aria-labelledby="filterModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg modal-dialog-centered">
        <div class="modal-content" style="border-radius: var(--border-radius);">
            <div class="modal-header" style="background: var(--gradient-primary); color: white; border-radius: var(--border-radius) var(--border-radius) 0 0;">
                <h5 class="modal-title" id="filterModalLabel">
                    <i class="fas fa-sliders-h me-2"></i>
                    تخصيص التقرير
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="إغلاق"></button>
            </div>
            <div class="modal-body p-4">
                <form id="reportFiltersForm">
                    <div class="row g-4">
                        <div class="col-md-6">
                            <label class="form-label fw-bold">
                                <i class="fas fa-calendar me-2 text-primary"></i>
                                الفترة الزمنية
                            </label>
                            <select class="form-select" name="period" id="periodFilter">
                                <option value="last_30_days">آخر 30 يوم</option>
                                <option value="last_3_months" selected>آخر 3 أشهر</option>
                                <option value="last_6_months">آخر 6 أشهر</option>
                                <option value="last_year">آخر سنة</option>
                                <option value="custom">فترة مخصصة</option>
                            </select>
                        </div>

                        <div class="col-md-6">
                            <label class="form-label fw-bold">
                                <i class="fas fa-users me-2 text-success"></i>
                                نوع العميل
                            </label>
                            <select class="form-select" name="customer_type" id="customerTypeFilter">
                                <option value="all" selected>جميع الأنواع</option>
                                <option value="individual">أفراد</option>
                                <option value="business">شركات</option>
                                <option value="vip">عملاء مميزين</option>
                            </select>
                        </div>

                        <div class="col-md-6">
                            <label class="form-label fw-bold">
                                <i class="fas fa-map-marker-alt me-2 text-info"></i>
                                المنطقة
                            </label>
                            <select class="form-select" name="region" id="regionFilter">
                                <option value="all" selected>جميع المناطق</option>
                                <option value="riyadh">الرياض</option>
                                <option value="jeddah">جدة</option>
                                <option value="dammam">الدمام</option>
                                <option value="makkah">مكة المكرمة</option>
                                <option value="madinah">المدينة المنورة</option>
                            </select>
                        </div>

                        <div class="col-md-6">
                            <label class="form-label fw-bold">
                                <i class="fas fa-toggle-on me-2 text-warning"></i>
                                الحالة
                            </label>
                            <select class="form-select" name="status" id="statusFilter">
                                <option value="all" selected>جميع الحالات</option>
                                <option value="active">نشط</option>
                                <option value="inactive">غير نشط</option>
                                <option value="suspended">معلق</option>
                            </select>
                        </div>

                        <div class="col-12" id="customDateRange" style="display: none;">
                            <div class="row g-3">
                                <div class="col-md-6">
                                    <label class="form-label fw-bold">تاريخ البداية</label>
                                    <input type="date" class="form-control" name="start_date" id="startDate">
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label fw-bold">تاريخ النهاية</label>
                                    <input type="date" class="form-control" name="end_date" id="endDate">
                                </div>
                            </div>
                        </div>

                        <div class="col-12">
                            <label class="form-label fw-bold">
                                <i class="fas fa-chart-bar me-2 text-primary"></i>
                                المقاييس المطلوبة
                            </label>
                            <div class="row g-2">
                                <div class="col-md-4">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" name="metrics[]" value="total_sales" id="totalSales" checked>
                                        <label class="form-check-label" for="totalSales">
                                            إجمالي المبيعات
                                        </label>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" name="metrics[]" value="order_count" id="orderCount" checked>
                                        <label class="form-check-label" for="orderCount">
                                            عدد الطلبات
                                        </label>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" name="metrics[]" value="avg_order_value" id="avgOrderValue" checked>
                                        <label class="form-check-label" for="avgOrderValue">
                                            متوسط قيمة الطلب
                                        </label>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" name="metrics[]" value="customer_lifetime_value" id="customerLifetimeValue">
                                        <label class="form-check-label" for="customerLifetimeValue">
                                            قيمة العميل مدى الحياة
                                        </label>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" name="metrics[]" value="retention_rate" id="retentionRate">
                                        <label class="form-check-label" for="retentionRate">
                                            معدل الاحتفاظ
                                        </label>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" name="metrics[]" value="churn_rate" id="churnRate">
                                        <label class="form-check-label" for="churnRate">
                                            معدل التسرب
                                        </label>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">
                    <i class="fas fa-times me-1"></i>
                    إلغاء
                </button>
                <button type="button" class="btn btn-primary" onclick="applyFilters()">
                    <i class="fas fa-filter me-1"></i>
                    تطبيق الفلاتر
                </button>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    initializeCharts();
    setupEventListeners();
    loadInitialData();
});

function initializeCharts() {
    // رسم بياني لنمو العملاء
    const growthCtx = document.getElementById('customerGrowthChart').getContext('2d');
    new Chart(growthCtx, {
        type: 'line',
        data: {
            labels: ['يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو', 'يوليو', 'أغسطس', 'سبتمبر', 'أكتوبر', 'نوفمبر', 'ديسمبر'],
            datasets: [{
                label: 'عملاء جدد',
                data: [65, 78, 92, 85, 98, 112, 125, 135, 142, 158, 175, 189],
                borderColor: 'rgb(102, 126, 234)',
                backgroundColor: 'rgba(102, 126, 234, 0.1)',
                borderWidth: 3,
                fill: true,
                tension: 0.4,
                pointBackgroundColor: 'rgb(102, 126, 234)',
                pointBorderColor: '#fff',
                pointBorderWidth: 2,
                pointRadius: 6
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    grid: {
                        color: 'rgba(0,0,0,0.05)'
                    }
                },
                x: {
                    grid: {
                        display: false
                    }
                }
            }
        }
    });

    // رسم بياني لتوزيع العملاء
    const distributionCtx = document.getElementById('customerDistributionChart').getContext('2d');
    new Chart(distributionCtx, {
        type: 'doughnut',
        data: {
            labels: ['أفراد نشطين', 'شركات نشطة', 'أفراد غير نشطين', 'شركات غير نشطة'],
            datasets: [{
                data: [45, 35, 12, 8],
                backgroundColor: [
                    'rgba(102, 126, 234, 0.8)',
                    'rgba(56, 239, 125, 0.8)',
                    'rgba(255, 206, 84, 0.8)',
                    'rgba(255, 99, 132, 0.8)'
                ],
                borderColor: [
                    'rgb(102, 126, 234)',
                    'rgb(56, 239, 125)',
                    'rgb(255, 206, 84)',
                    'rgb(255, 99, 132)'
                ],
                borderWidth: 2
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'bottom',
                    labels: {
                        padding: 20,
                        usePointStyle: true,
                        font: {
                            size: 12
                        }
                    }
                }
            }
        }
    });
}

function setupEventListeners() {
    // مستمع لتغيير الفترة الزمنية
    document.getElementById('periodFilter').addEventListener('change', function() {
        const customDateRange = document.getElementById('customDateRange');
        if (this.value === 'custom') {
            customDateRange.style.display = 'block';
        } else {
            customDateRange.style.display = 'none';
        }
    });
}

function loadInitialData() {
    // محاكاة تحميل البيانات
    animateNumbers();
}

function animateNumbers() {
    animateValue('totalCustomers', 0, 1247, 2000);
    animateValue('activeCustomers', 0, 1089, 2000);
    animateValue('vipCustomers', 0, 158, 2000);
    animateValue('avgOrderValue', 0, 2450, 2000);
}

function animateValue(id, start, end, duration) {
    const element = document.getElementById(id);
    const startTimestamp = performance.now();
    
    function step(timestamp) {
        const progress = Math.min((timestamp - startTimestamp) / duration, 1);
        const current = Math.floor(progress * (end - start) + start);
        element.textContent = current.toLocaleString('ar-SA');
        
        if (progress < 1) {
            requestAnimationFrame(step);
        }
    }
    
    requestAnimationFrame(step);
}

function applyFilters() {
    const form = document.getElementById('reportFiltersForm');
    const formData = new FormData(form);
    
    // إظهار مؤشر التحميل
    showLoadingState();
    
    // محاكاة تحميل البيانات
    setTimeout(() => {
        hideLoadingState();
        updateFilterChips(formData);
        updateReportData(formData);
        
        // إغلاق المودال
        const modal = bootstrap.Modal.getInstance(document.getElementById('filterModal'));
        modal.hide();
        
        // إظهار رسالة نجاح
        showSuccessMessage('تم تطبيق الفلاتر بنجاح');
    }, 1500);
}

function updateFilterChips(formData) {
    const filterChips = document.getElementById('filterChips');
    const activeFilters = document.getElementById('activeFilters');
    
    filterChips.innerHTML = '';
    
    // إضافة رقائق الفلاتر
    for (const [key, value] of formData.entries()) {
        if (value && value !== 'all') {
            const chip = createFilterChip(key, value);
            filterChips.appendChild(chip);
        }
    }
    
    // إظهار أو إخفاء قسم الفلاتر
    if (filterChips.children.length > 0) {
        activeFilters.style.display = 'block';
    } else {
        activeFilters.style.display = 'none';
    }
}

function createFilterChip(key, value) {
    const chip = document.createElement('span');
    chip.className = 'filter-chip';
    
    const labelMap = {
        'period': 'الفترة الزمنية',
        'customer_type': 'نوع العميل',
        'region': 'المنطقة',
        'status': 'الحالة'
    };
    
    const valueMap = {
        'last_30_days': 'آخر 30 يوم',
        'last_3_months': 'آخر 3 أشهر',
        'last_6_months': 'آخر 6 أشهر',
        'last_year': 'آخر سنة',
        'individual': 'أفراد',
        'business': 'شركات',
        'vip': 'عملاء مميزين',
        'active': 'نشط',
        'inactive': 'غير نشط',
        'suspended': 'معلق'
    };
    
    const displayValue = valueMap[value] || value;
    chip.innerHTML = `${labelMap[key]}: ${displayValue} <button type="button" class="btn-close" onclick="removeFilter('${key}')"></button>`;
    
    return chip;
}

function removeFilter(key) {
    // إزالة الفلتر وتحديث التقرير
    const element = document.querySelector(`[name="${key}"]`);
    if (element) {
        element.value = element.tagName === 'SELECT' ? 'all' : '';
    }
    
    applyFilters();
}

function clearAllFilters() {
    const form = document.getElementById('reportFiltersForm');
    const selects = form.querySelectorAll('select');
    const inputs = form.querySelectorAll('input');
    
    selects.forEach(select => select.value = 'all');
    inputs.forEach(input => {
        if (input.type === 'checkbox') {
            input.checked = ['total_sales', 'order_count', 'avg_order_value'].includes(input.value);
        } else {
            input.value = '';
        }
    });
    
    applyFilters();
}

function updateReportData(formData) {
    // محاكاة تحديث البيانات
    const multiplier = Math.random() * 0.3 + 0.85; // تغيير عشوائي في البيانات
    
    animateValue('totalCustomers', 0, Math.floor(1247 * multiplier), 1500);
    animateValue('activeCustomers', 0, Math.floor(1089 * multiplier), 1500);
    animateValue('vipCustomers', 0, Math.floor(158 * multiplier), 1500);
    animateValue('avgOrderValue', 0, Math.floor(2450 * multiplier), 1500);
    
    // تحديث الجدول
    updateTopCustomersTable();
}

function updateTopCustomersTable() {
    const tableBody = document.getElementById('topCustomersTable');
    
    // إضافة تأثير تحميل
    tableBody.style.opacity = '0.5';
    
    setTimeout(() => {
        tableBody.style.opacity = '1';
        // هنا يمكن تحديث بيانات الجدول الفعلية
    }, 1000);
}

function exportReport() {
    // إظهار مودال خيارات التصدير
    const exportModal = `
        <div class="modal fade" id="exportReportModal" tabindex="-1">
            <div class="modal-dialog modal-dialog-centered">
                <div class="modal-content" style="border-radius: var(--border-radius);">
                    <div class="modal-header" style="background: var(--gradient-success); color: white; border-radius: var(--border-radius) var(--border-radius) 0 0;">
                        <h5 class="modal-title">
                            <i class="fas fa-download me-2"></i>
                            تصدير التقرير
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body text-center py-4">
                        <i class="fas fa-file-export fa-4x text-success mb-3"></i>
                        <h6 class="mb-4">اختر صيغة التصدير المرغوبة:</h6>
                        <div class="d-grid gap-3">
                            <button class="btn btn-outline-success btn-lg" onclick="exportToPDF()">
                                <i class="fas fa-file-pdf me-2"></i>
                                تصدير PDF
                            </button>
                            <button class="btn btn-outline-primary btn-lg" onclick="exportToExcel()">
                                <i class="fas fa-file-excel me-2"></i>
                                تصدير Excel
                            </button>
                            <button class="btn btn-outline-info btn-lg" onclick="exportToCSV()">
                                <i class="fas fa-file-csv me-2"></i>
                                تصدير CSV
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    if (!document.getElementById('exportReportModal')) {
        document.body.insertAdjacentHTML('beforeend', exportModal);
    }
    
    const modal = new bootstrap.Modal(document.getElementById('exportReportModal'));
    modal.show();
}

function exportToPDF() {
    showLoadingMessage('جاري تصدير التقرير بصيغة PDF...');
    setTimeout(() => {
        hideLoadingMessage();
        showSuccessMessage('تم تصدير التقرير بنجاح');
        bootstrap.Modal.getInstance(document.getElementById('exportReportModal')).hide();
    }, 2000);
}

function exportToExcel() {
    showLoadingMessage('جاري تصدير التقرير بصيغة Excel...');
    setTimeout(() => {
        hideLoadingMessage();
        showSuccessMessage('تم تصدير التقرير بنجاح');
        bootstrap.Modal.getInstance(document.getElementById('exportReportModal')).hide();
    }, 2000);
}

function exportToCSV() {
    showLoadingMessage('جاري تصدير التقرير بصيغة CSV...');
    setTimeout(() => {
        hideLoadingMessage();
        showSuccessMessage('تم تصدير التقرير بنجاح');
        bootstrap.Modal.getInstance(document.getElementById('exportReportModal')).hide();
    }, 2000);
}

function refreshReport() {
    showLoadingState();
    
    setTimeout(() => {
        hideLoadingState();
        loadInitialData();
        showSuccessMessage('تم تحديث التقرير بنجاح');
    }, 1500);
}

function refreshTopCustomers() {
    const refreshBtn = event.target;
    const originalContent = refreshBtn.innerHTML;
    refreshBtn.innerHTML = '<div class="loading-spinner me-1"></div>تحديث...';
    refreshBtn.disabled = true;
    
    setTimeout(() => {
        refreshBtn.innerHTML = originalContent;
        refreshBtn.disabled = false;
        updateTopCustomersTable();
        showSuccessMessage('تم تحديث قائمة أفضل العملاء');
    }, 1500);
}

function viewAllCustomers() {
    window.location.href = '{{ route("customers.index") }}';
}

function showLoadingState() {
    const cards = document.querySelectorAll('.metric-card');
    cards.forEach(card => {
        card.style.opacity = '0.6';
        card.style.pointerEvents = 'none';
    });
}

function hideLoadingState() {
    const cards = document.querySelectorAll('.metric-card');
    cards.forEach(card => {
        card.style.opacity = '1';
        card.style.pointerEvents = 'auto';
    });
}

function showLoadingMessage(message) {
    const toast = `
        <div class="toast-container position-fixed top-0 start-50 translate-middle-x p-3" style="z-index: 11000;">
            <div id="loadingToast" class="toast show align-items-center text-white bg-info border-0" role="alert" aria-live="assertive" aria-atomic="true">
                <div class="d-flex">
                    <div class="toast-body">
                        <div class="loading-spinner me-2"></div>
                        ${message}
                    </div>
                </div>
            </div>
        </div>
    `;
    
    document.body.insertAdjacentHTML('beforeend', toast);
}

function hideLoadingMessage() {
    const loadingToast = document.getElementById('loadingToast');
    if (loadingToast) {
        loadingToast.remove();
    }
}

function showSuccessMessage(message) {
    const toast = `
        <div class="toast-container position-fixed top-0 start-50 translate-middle-x p-3" style="z-index: 11000;">
            <div id="successToast" class="toast show align-items-center text-white bg-success border-0" role="alert" aria-live="assertive" aria-atomic="true">
                <div class="d-flex">
                    <div class="toast-body">
                        <i class="fas fa-check-circle me-2"></i>
                        ${message}
                    </div>
                    <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="إغلاق"></button>
                </div>
            </div>
        </div>
    `;
    
    document.body.insertAdjacentHTML('beforeend', toast);
    
    setTimeout(() => {
        const successToast = document.getElementById('successToast');
        if (successToast) {
            successToast.remove();
        }
    }, 3000);
}

// تحسين الأداء وتجربة المستخدم
window.addEventListener('scroll', function() {
    const cards = document.querySelectorAll('.metric-card, .analytics-card');
    cards.forEach(card => {
        const rect = card.getBoundingClientRect();
        if (rect.top < window.innerHeight && rect.bottom > 0) {
            card.classList.add('fade-in-up');
        }
    });
});

// إضافة مؤثرات تفاعلية للبطاقات
document.querySelectorAll('.metric-card').forEach(card => {
    card.addEventListener('mouseenter', function() {
        this.style.transform = 'translateY(-10px) scale(1.02)';
        this.style.zIndex = '10';
    });
    
    card.addEventListener('mouseleave', function() {
        this.style.transform = 'translateY(0) scale(1)';
        this.style.zIndex = '1';
    });
});
</script>
@endpush