@extends('layouts.app')

@section('title', 'دفعة جديدة للمورد')
@section('page-title', 'دفعة جديدة للمورد')

@section('page-actions')
    <a href="{{ route('suppliers.payments.index') }}" class="btn btn-secondary">
        <i class="fas fa-arrow-right me-2"></i>
        العودة لقائمة المدفوعات
    </a>
@endsection

@section('content')
<form id="paymentForm" method="POST" action="{{ route('suppliers.payments.store') }}">
    @csrf
    
    <div class="row">
        <!-- معلومات الدفعة -->
        <div class="col-lg-8">
            <div class="card mb-4">
                <div class="card-header">
                    <h6 class="card-title mb-0">
                        <i class="fas fa-info-circle me-2"></i>
                        معلومات الدفعة
                    </h6>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="supplier_id" class="form-label">المورد <span class="text-danger">*</span></label>
                                <select class="form-control @error('supplier_id') is-invalid @enderror" 
                                        id="supplier_id" name="supplier_id" required>
                                    <option value="">اختر المورد</option>
                                    @foreach($suppliers as $supplier)
                                        <option value="{{ $supplier->id }}" 
                                                {{ (request('supplier_id') == $supplier->id || old('supplier_id') == $supplier->id) ? 'selected' : '' }}
                                                data-balance="{{ $supplier->current_balance }}"
                                                data-credit-limit="{{ $supplier->credit_limit }}"
                                                data-payment-terms="{{ $supplier->payment_terms }}">
                                            {{ $supplier->name }} 
                                            @if($supplier->current_balance > 0)
                                                ({{ number_format($supplier->current_balance, 2) }} ر.س)
                                            @endif
                                        </option>
                                    @endforeach
                                </select>
                                @error('supplier_id')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="payment_date" class="form-label">تاريخ الدفعة <span class="text-danger">*</span></label>
                                <input type="date" class="form-control @error('payment_date') is-invalid @enderror" 
                                       id="payment_date" name="payment_date" 
                                       value="{{ old('payment_date', date('Y-m-d')) }}" 
                                       max="{{ date('Y-m-d') }}" required>
                                @error('payment_date')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="payment_method" class="form-label">طريقة الدفع <span class="text-danger">*</span></label>
                                <select class="form-control @error('payment_method') is-invalid @enderror" 
                                        id="payment_method" name="payment_method" required>
                                    <option value="">اختر طريقة الدفع</option>
                                    <option value="cash" {{ old('payment_method') == 'cash' ? 'selected' : '' }}>نقداً</option>
                                    <option value="bank_transfer" {{ old('payment_method') == 'bank_transfer' ? 'selected' : '' }}>تحويل بنكي</option>
                                    <option value="check" {{ old('payment_method') == 'check' ? 'selected' : '' }}>شيك</option>
                                    <option value="credit_card" {{ old('payment_method') == 'credit_card' ? 'selected' : '' }}>بطاقة ائتمان</option>
                                </select>
                                @error('payment_method')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-6" id="bank_account_field" style="display: none;">
                            <div class="mb-3">
                                <label for="bank_account_id" class="form-label">الحساب البنكي</label>
                                <select class="form-control" id="bank_account_id" name="bank_account_id">
                                    <option value="">اختر الحساب البنكي</option>
                                    @foreach($bankAccounts as $account)
                                        <option value="{{ $account->id }}" {{ old('bank_account_id') == $account->id ? 'selected' : '' }}>
                                            {{ $account->bank_name }} - {{ $account->account_number }}
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6" id="check_fields" style="display: none;">
                            <div class="mb-3">
                                <label for="check_number" class="form-label">رقم الشيك</label>
                                <input type="text" class="form-control" id="check_number" name="check_number" 
                                       value="{{ old('check_number') }}">
                            </div>
                        </div>
                        <div class="col-md-6" id="check_date_field" style="display: none;">
                            <div class="mb-3">
                                <label for="check_date" class="form-label">تاريخ الشيك</label>
                                <input type="date" class="form-control" id="check_date" name="check_date" 
                                       value="{{ old('check_date') }}">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="reference" class="form-label">المرجع</label>
                                <input type="text" class="form-control @error('reference') is-invalid @enderror" 
                                       id="reference" name="reference" 
                                       value="{{ old('reference') }}" placeholder="رقم المرجع أو الإشارة">
                                @error('reference')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="currency" class="form-label">العملة</label>
                                <select class="form-control" id="currency" name="currency">
                                    <option value="SAR" {{ old('currency', 'SAR') == 'SAR' ? 'selected' : '' }}>ريال سعودي</option>
                                    <option value="USD" {{ old('currency') == 'USD' ? 'selected' : '' }}>دولار أمريكي</option>
                                    <option value="EUR" {{ old('currency') == 'EUR' ? 'selected' : '' }}>يورو</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="exchange_rate" class="form-label">سعر الصرف</label>
                                <input type="number" class="form-control" id="exchange_rate" name="exchange_rate" 
                                       value="{{ old('exchange_rate', 1) }}" step="0.0001" min="0.0001">
                            </div>
                        </div>
                        <div class="col-12">
                            <div class="mb-3">
                                <label for="description" class="form-label">الوصف <span class="text-danger">*</span></label>
                                <input type="text" class="form-control @error('description') is-invalid @enderror" 
                                       id="description" name="description" 
                                       value="{{ old('description') }}" 
                                       placeholder="وصف موجز للدفعة" required>
                                @error('description')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-12">
                            <div class="mb-3">
                                <label for="notes" class="form-label">ملاحظات</label>
                                <textarea class="form-control @error('notes') is-invalid @enderror" 
                                          id="notes" name="notes" rows="3" 
                                          placeholder="ملاحظات إضافية حول الدفعة">{{ old('notes') }}</textarea>
                                @error('notes')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- توزيع الدفعة على الفواتير -->
            <div class="card" id="invoicesCard" style="display: none;">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h6 class="card-title mb-0">
                        <i class="fas fa-file-invoice-dollar me-2"></i>
                        توزيع الدفعة على الفواتير
                    </h6>
                    <div class="form-check form-switch">
                        <input class="form-check-input" type="radio" name="payment_allocation" 
                               value="automatic" id="automatic_allocation" checked>
                        <label class="form-check-label me-3" for="automatic_allocation">
                            تلقائي (الأقدم أولاً)
                        </label>
                        <input class="form-check-input" type="radio" name="payment_allocation" 
                               value="manual" id="manual_allocation">
                        <label class="form-check-label" for="manual_allocation">
                            يدوي
                        </label>
                    </div>
                </div>
                <div class="card-body">
                    <div id="automatic_info" class="alert alert-info">
                        <i class="fas fa-info-circle me-2"></i>
                        سيتم توزيع المبلغ تلقائياً على الفواتير المستحقة بدءاً من الأقدم.
                    </div>
                    
                    <div id="manual_invoices" style="display: none;">
                        <div class="table-responsive">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>رقم الفاتورة</th>
                                        <th>تاريخ الاستحقاق</th>
                                        <th class="text-center">المبلغ المتبقي</th>
                                        <th class="text-center">المبلغ المدفوع</th>
                                    </tr>
                                </thead>
                                <tbody id="outstandingInvoices">
                                    <!-- سيتم ملء الجدول ديناميكياً -->
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- ملخص المعلومات -->
        <div class="col-lg-4">
            <!-- معلومات المورد -->
            <div class="card mb-4" id="supplierInfo" style="display: none;">
                <div class="card-header">
                    <h6 class="card-title mb-0">
                        <i class="fas fa-user-tie me-2"></i>
                        معلومات المورد
                    </h6>
                </div>
                <div class="card-body">
                    <div id="supplierDetails">
                        <!-- سيتم ملء التفاصيل ديناميكياً -->
                    </div>
                </div>
            </div>
            
            <!-- معلومات الدفعة -->
            <div class="card mb-4">
                <div class="card-header">
                    <h6 class="card-title mb-0">
                        <i class="fas fa-money-bill-wave me-2"></i>
                        معلومات الدفعة
                    </h6>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <label for="total_amount" class="form-label">مبلغ الدفعة (ر.س) <span class="text-danger">*</span></label>
                        <input type="number" class="form-control @error('total_amount') is-invalid @enderror" 
                               id="total_amount" name="total_amount" 
                               value="{{ old('total_amount') }}" 
                               step="0.01" min="0.01" required>
                        @error('total_amount')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                        <small class="text-muted" id="maxAmount"></small>
                    </div>
                    
                    <div class="alert alert-warning" id="creditWarning" style="display: none;">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <small>تنبيه: المبلغ يتجاوز المديونية المستحقة</small>
                    </div>
                    
                    <div class="d-grid gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i>
                            حفظ الدفعة
                        </button>
                        <a href="{{ route('suppliers.payments.index') }}" class="btn btn-outline-secondary">
                            <i class="fas fa-times me-2"></i>
                            إلغاء
                        </a>
                    </div>
                </div>
            </div>
            
            <!-- إرشادات -->
            <div class="card">
                <div class="card-header">
                    <h6 class="card-title mb-0">
                        <i class="fas fa-lightbulb me-2"></i>
                        إرشادات
                    </h6>
                </div>
                <div class="card-body">
                    <ul class="list-unstyled mb-0">
                        <li class="mb-2">
                            <i class="fas fa-check text-success me-2"></i>
                            <small>تأكد من صحة المبلغ وطريقة الدفع</small>
                        </li>
                        <li class="mb-2">
                            <i class="fas fa-check text-success me-2"></i>
                            <small>اختر التوزيع التلقائي للسرعة</small>
                        </li>
                        <li class="mb-2">
                            <i class="fas fa-check text-success me-2"></i>
                            <small>أضف ملاحظات للتوضيح</small>
                        </li>
                        <li class="mb-0">
                            <i class="fas fa-info-circle text-info me-2"></i>
                            <small>يمكن حذف الدفعة خلال 24 ساعة فقط</small>
                        </li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</form>
@endsection

@push('scripts')
<script>
    document.addEventListener('DOMContentLoaded', function() {
        const supplierSelect = document.getElementById('supplier_id');
        const paymentMethodSelect = document.getElementById('payment_method');
        const totalAmountInput = document.getElementById('total_amount');
        const allocationRadios = document.querySelectorAll('input[name="payment_allocation"]');
        
        // معالج تغيير المورد
        supplierSelect.addEventListener('change', function() {
            const selectedOption = this.options[this.selectedIndex];
            
            if (this.value) {
                const balance = parseFloat(selectedOption.dataset.balance);
                const creditLimit = parseFloat(selectedOption.dataset.creditLimit);
                const paymentTerms = selectedOption.dataset.paymentTerms;
                
                // تحديث معلومات المورد
                updateSupplierInfo(selectedOption.text, balance, creditLimit, paymentTerms);
                
                // تحديث الحد الأقصى للمبلغ
                totalAmountInput.max = balance;
                document.getElementById('maxAmount').textContent = `الحد الأقصى: ${balance.toFixed(2)} ر.س`;
                
                // جلب الفواتير المستحقة
                loadOutstandingInvoices(this.value);
                
                // إظهار كارت معلومات المورد
                document.getElementById('supplierInfo').style.display = 'block';
                document.getElementById('invoicesCard').style.display = 'block';
            } else {
                document.getElementById('supplierInfo').style.display = 'none';
                document.getElementById('invoicesCard').style.display = 'none';
                document.getElementById('maxAmount').textContent = '';
                totalAmountInput.max = '';
            }
        });
        
        // معالج تغيير طريقة الدفع
        paymentMethodSelect.addEventListener('change', function() {
            const bankAccountField = document.getElementById('bank_account_field');
            const checkFields = document.getElementById('check_fields');
            const checkDateField = document.getElementById('check_date_field');
            
            // إخفاء جميع الحقول أولاً
            bankAccountField.style.display = 'none';
            checkFields.style.display = 'none';
            checkDateField.style.display = 'none';
            
            // إظهار الحقول المطلوبة حسب طريقة الدفع
            if (this.value === 'bank_transfer') {
                bankAccountField.style.display = 'block';
            } else if (this.value === 'check') {
                checkFields.style.display = 'block';
                checkDateField.style.display = 'block';
            }
        });
        
        // معالج تغيير المبلغ
        totalAmountInput.addEventListener('input', function() {
            const supplierSelect = document.getElementById('supplier_id');
            const selectedOption = supplierSelect.options[supplierSelect.selectedIndex];
            
            if (selectedOption && selectedOption.value) {
                const balance = parseFloat(selectedOption.dataset.balance);
                const amount = parseFloat(this.value);
                const creditWarning = document.getElementById('creditWarning');
                
                if (amount > balance) {
                    creditWarning.style.display = 'block';
                } else {
                    creditWarning.style.display = 'none';
                }
            }
        });
        
        // معالج تغيير طريقة التوزيع
        allocationRadios.forEach(radio => {
            radio.addEventListener('change', function() {
                const automaticInfo = document.getElementById('automatic_info');
                const manualInvoices = document.getElementById('manual_invoices');
                
                if (this.value === 'automatic') {
                    automaticInfo.style.display = 'block';
                    manualInvoices.style.display = 'none';
                } else {
                    automaticInfo.style.display = 'none';
                    manualInvoices.style.display = 'block';
                }
            });
        });
        
        // تحميل بيانات المورد المحدد مسبقاً إذا وجد
        if (supplierSelect.value) {
            supplierSelect.dispatchEvent(new Event('change'));
        }
        
        // تحديث طريقة الدفع إذا كانت محددة مسبقاً
        if (paymentMethodSelect.value) {
            paymentMethodSelect.dispatchEvent(new Event('change'));
        }
    });
    
    function updateSupplierInfo(name, balance, creditLimit, paymentTerms) {
        const supplierDetails = document.getElementById('supplierDetails');
        
        supplierDetails.innerHTML = `
            <div class="text-center mb-3">
                <h6>${name}</h6>
            </div>
            <table class="table table-borderless table-sm">
                <tr>
                    <td><strong>الرصيد الحالي:</strong></td>
                    <td class="text-end text-${balance > 0 ? 'danger' : 'success'}">${balance.toFixed(2)} ر.س</td>
                </tr>
                <tr>
                    <td><strong>حد الائتمان:</strong></td>
                    <td class="text-end">${creditLimit.toFixed(2)} ر.س</td>
                </tr>
                ${paymentTerms ? `
                <tr>
                    <td><strong>شروط الدفع:</strong></td>
                    <td class="text-end">${paymentTerms}</td>
                </tr>
                ` : ''}
            </table>
        `;
    }
    
    async function loadOutstandingInvoices(supplierId) {
        try {
            const response = await fetch(`/suppliers/payments/api/supplier/${supplierId}/outstanding-invoices`);
            const invoices = await response.json();
            
            const tbody = document.getElementById('outstandingInvoices');
            tbody.innerHTML = '';
            
            invoices.forEach((invoice, index) => {
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td>
                        <strong>${invoice.invoice_number}</strong>
                        ${invoice.is_overdue ? '<br><span class="badge bg-danger badge-sm">متأخر</span>' : ''}
                    </td>
                    <td>${invoice.due_date}</td>
                    <td class="text-center">
                        <strong class="text-danger">${parseFloat(invoice.remaining_amount).toFixed(2)} ر.س</strong>
                    </td>
                    <td class="text-center">
                        <input type="hidden" name="invoices[${index}][invoice_id]" value="${invoice.id}">
                        <input type="number" class="form-control form-control-sm" 
                               name="invoices[${index}][amount]" 
                               value="0" min="0" max="${invoice.remaining_amount}" step="0.01"
                               onchange="updateManualTotal()">
                    </td>
                `;
                tbody.appendChild(row);
            });
        } catch (error) {
            console.error('Error loading invoices:', error);
        }
    }
    
    function updateManualTotal() {
        const inputs = document.querySelectorAll('input[name^="invoices["][name$="[amount]"]');
        let total = 0;
        
        inputs.forEach(input => {
            total += parseFloat(input.value) || 0;
        });
        
        document.getElementById('total_amount').value = total.toFixed(2);
    }
</script>
@endpush

@push('styles')
<style>
    .badge-sm {
        font-size: 0.7rem;
        padding: 0.25rem 0.4rem;
    }
    
    .form-check-input[type=radio] {
        margin-left: 0.5rem;
    }
    
    .alert-warning {
        border-left: 4px solid #ffc107;
    }
</style>
@endpush