@extends('layouts.app')

@section('title', 'إنشاء جدولة سداد جديدة')

@section('page-actions')
    <a href="{{ route('suppliers.debt.dashboard') }}" class="btn btn-secondary">
        <i class="fas fa-arrow-right me-2"></i>
        العودة للوحة التحكم
    </a>
@endsection

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="card shadow">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="fas fa-calendar-plus me-2"></i>
                        إنشاء جدولة سداد جديدة
                    </h6>
                </div>
                <div class="card-body">
                    <form action="{{ route('suppliers.debt.store-schedule') }}" method="POST" id="scheduleForm">
                        @csrf
                        
                        <div class="row">
                            <div class="col-lg-8">
                                <!-- اختيار المورد -->
                                <div class="card mb-4">
                                    <div class="card-header">
                                        <h6 class="card-title mb-0">
                                            <i class="fas fa-user me-2"></i>
                                            بيانات المورد
                                        </h6>
                                    </div>
                                    <div class="card-body">
                                        <div class="mb-3">
                                            <label for="supplier_id" class="form-label">المورد *</label>
                                            <select name="supplier_id" id="supplier_id" 
                                                    class="form-select @error('supplier_id') is-invalid @enderror" 
                                                    onchange="loadSupplierInvoices()">
                                                <option value="">اختر المورد...</option>
                                                @foreach($suppliers as $supplierOption)
                                                    <option value="{{ $supplierOption->id }}" 
                                                            {{ (request('supplier_id') == $supplierOption->id || old('supplier_id') == $supplierOption->id) ? 'selected' : '' }}
                                                            data-balance="{{ $supplierOption->current_balance }}"
                                                            data-credit-limit="{{ $supplierOption->credit_limit }}"
                                                            data-penalty-rate="{{ $supplierOption->penalty_rate }}">
                                                        {{ $supplierOption->name }} - 
                                                        مديونية: {{ number_format($supplierOption->current_balance, 2) }} ر.س
                                                    </option>
                                                @endforeach
                                            </select>
                                            @error('supplier_id')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                        
                                        <!-- معلومات المورد -->
                                        <div id="supplierInfo" style="display: none;">
                                            <div class="alert alert-info">
                                                <div class="row">
                                                    <div class="col-md-6">
                                                        <strong>الرصيد الحالي:</strong> 
                                                        <span id="currentBalance" class="text-danger">0</span> ر.س
                                                    </div>
                                                    <div class="col-md-6">
                                                        <strong>حد الائتمان:</strong> 
                                                        <span id="creditLimit" class="text-info">0</span> ر.س
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- الفاتورة -->
                                <div class="card mb-4" id="invoiceCard" style="display: none;">
                                    <div class="card-header">
                                        <h6 class="card-title mb-0">
                                            <i class="fas fa-file-invoice me-2"></i>
                                            اختيار الفاتورة
                                        </h6>
                                    </div>
                                    <div class="card-body">
                                        <div class="mb-3">
                                            <label for="purchase_invoice_id" class="form-label">الفاتورة *</label>
                                            <select name="purchase_invoice_id" id="purchase_invoice_id" 
                                                    class="form-select @error('purchase_invoice_id') is-invalid @enderror"
                                                    onchange="updateInvoiceDetails()">
                                                <option value="">اختر الفاتورة...</option>
                                            </select>
                                            @error('purchase_invoice_id')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>

                                        <!-- تفاصيل الفاتورة -->
                                        <div id="invoiceDetails" style="display: none;">
                                            <div class="table-responsive">
                                                <table class="table table-sm table-bordered">
                                                    <tr>
                                                        <td><strong>رقم الفاتورة:</strong></td>
                                                        <td id="invoiceNumber">-</td>
                                                    </tr>
                                                    <tr>
                                                        <td><strong>تاريخ الفاتورة:</strong></td>
                                                        <td id="invoiceDate">-</td>
                                                    </tr>
                                                    <tr>
                                                        <td><strong>تاريخ الاستحقاق:</strong></td>
                                                        <td id="dueDate">-</td>
                                                    </tr>
                                                    <tr>
                                                        <td><strong>المبلغ المتبقي:</strong></td>
                                                        <td id="remainingAmount" class="text-danger">0 ر.س</td>
                                                    </tr>
                                                </table>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- تفاصيل الجدولة -->
                                <div class="card mb-4" id="scheduleCard" style="display: none;">
                                    <div class="card-header">
                                        <h6 class="card-title mb-0">
                                            <i class="fas fa-calendar-alt me-2"></i>
                                            تفاصيل الجدولة
                                        </h6>
                                    </div>
                                    <div class="card-body">
                                        <div class="row">
                                            <div class="col-md-6">
                                                <div class="mb-3">
                                                    <label for="schedule_date" class="form-label">تاريخ السداد المجدول *</label>
                                                    <input type="date" name="schedule_date" id="schedule_date" 
                                                           class="form-control @error('schedule_date') is-invalid @enderror"
                                                           value="{{ old('schedule_date') }}" min="{{ date('Y-m-d', strtotime('+1 day')) }}">
                                                    @error('schedule_date')
                                                        <div class="invalid-feedback">{{ $message }}</div>
                                                    @enderror
                                                </div>
                                            </div>
                                            <div class="col-md-6">
                                                <div class="mb-3">
                                                    <label for="payment_deadline" class="form-label">الموعد النهائي للدفع</label>
                                                    <input type="date" name="payment_deadline" id="payment_deadline" 
                                                           class="form-control @error('payment_deadline') is-invalid @enderror"
                                                           value="{{ old('payment_deadline') }}">
                                                    <small class="text-muted">سيتم تطبيق غرامة التأخير بعد هذا التاريخ</small>
                                                    @error('payment_deadline')
                                                        <div class="invalid-feedback">{{ $message }}</div>
                                                    @enderror
                                                </div>
                                            </div>
                                        </div>

                                        <div class="mb-3">
                                            <label for="scheduled_amount" class="form-label">المبلغ المجدول *</label>
                                            <div class="input-group">
                                                <input type="number" name="scheduled_amount" id="scheduled_amount" 
                                                       class="form-control @error('scheduled_amount') is-invalid @enderror"
                                                       value="{{ old('scheduled_amount') }}" step="0.01" min="0.01">
                                                <span class="input-group-text">ر.س</span>
                                            </div>
                                            <div class="form-check mt-2">
                                                <input class="form-check-input" type="checkbox" id="useFullAmount" 
                                                       onchange="setFullAmount()">
                                                <label class="form-check-label" for="useFullAmount">
                                                    استخدام كامل المبلغ المتبقي
                                                </label>
                                            </div>
                                            @error('scheduled_amount')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>

                                        <div class="mb-3">
                                            <label for="notes" class="form-label">ملاحظات</label>
                                            <textarea name="notes" id="notes" class="form-control" rows="3" 
                                                      placeholder="أي ملاحظات إضافية...">{{ old('notes') }}</textarea>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- الشريط الجانبي -->
                            <div class="col-lg-4">
                                <div class="card bg-light">
                                    <div class="card-header">
                                        <h6 class="card-title mb-0">
                                            <i class="fas fa-info-circle me-2"></i>
                                            معلومات هامة
                                        </h6>
                                    </div>
                                    <div class="card-body">
                                        <div class="alert alert-info">
                                            <h6><i class="fas fa-lightbulb me-2"></i>نصائح:</h6>
                                            <ul class="mb-0">
                                                <li>يمكن إنشاء عدة جدولات للفاتورة الواحدة</li>
                                                <li>سيتم إرسال تذكير قبل 3 أيام من موعد السداد</li>
                                                <li>ستطبق غرامة التأخير حسب معدل المورد</li>
                                                <li>يمكن تعديل الجدولة قبل تاريخ الاستحقاق</li>
                                            </ul>
                                        </div>

                                        <div class="alert alert-warning" id="penaltyInfo" style="display: none;">
                                            <h6><i class="fas fa-exclamation-triangle me-2"></i>غرامة التأخير:</h6>
                                            <p class="mb-0">معدل الغرامة: <strong id="penaltyRate">0%</strong> من المبلغ المستحق</p>
                                        </div>

                                        <div class="d-grid gap-2">
                                            <button type="submit" class="btn btn-primary" id="submitBtn" disabled>
                                                <i class="fas fa-save me-2"></i>
                                                حفظ الجدولة
                                            </button>
                                            <a href="{{ route('suppliers.debt.dashboard') }}" class="btn btn-outline-secondary">
                                                <i class="fas fa-times me-2"></i>
                                                إلغاء
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
let supplierInvoices = [];

// تحميل فواتير المورد
function loadSupplierInvoices() {
    const supplierId = document.getElementById('supplier_id').value;
    const supplierInfo = document.getElementById('supplierInfo');
    const invoiceCard = document.getElementById('invoiceCard');
    const scheduleCard = document.getElementById('scheduleCard');
    const invoiceSelect = document.getElementById('purchase_invoice_id');
    
    if (!supplierId) {
        supplierInfo.style.display = 'none';
        invoiceCard.style.display = 'none';
        scheduleCard.style.display = 'none';
        document.getElementById('submitBtn').disabled = true;
        return;
    }
    
    // عرض معلومات المورد
    const selectedOption = document.querySelector(`#supplier_id option[value="${supplierId}"]`);
    const balance = parseFloat(selectedOption.dataset.balance);
    const creditLimit = parseFloat(selectedOption.dataset.creditLimit);
    const penaltyRate = parseFloat(selectedOption.dataset.penaltyRate);
    
    document.getElementById('currentBalance').textContent = balance.toFixed(2);
    document.getElementById('creditLimit').textContent = creditLimit.toFixed(2);
    document.getElementById('penaltyRate').textContent = (penaltyRate * 100).toFixed(2) + '%';
    
    supplierInfo.style.display = 'block';
    document.getElementById('penaltyInfo').style.display = 'block';
    
    // تحميل الفواتير
    fetch(`/api/suppliers/${supplierId}/outstanding-invoices`)
        .then(response => response.json())
        .then(data => {
            supplierInvoices = data;
            
            // مسح الخيارات السابقة
            invoiceSelect.innerHTML = '<option value="">اختر الفاتورة...</option>';
            
            // إضافة الفواتير الجديدة
            data.forEach(invoice => {
                const option = document.createElement('option');
                option.value = invoice.id;
                option.textContent = `${invoice.invoice_number} - ${invoice.remaining_amount} ر.س`;
                option.dataset.invoiceNumber = invoice.invoice_number;
                option.dataset.invoiceDate = invoice.invoice_date;
                option.dataset.dueDate = invoice.due_date;
                option.dataset.remainingAmount = invoice.remaining_amount;
                invoiceSelect.appendChild(option);
            });
            
            invoiceCard.style.display = 'block';
        })
        .catch(error => {
            console.error('Error loading invoices:', error);
            alert('حدث خطأ في تحميل الفواتير');
        });
}

// تحديث تفاصيل الفاتورة
function updateInvoiceDetails() {
    const invoiceId = document.getElementById('purchase_invoice_id').value;
    const invoiceDetails = document.getElementById('invoiceDetails');
    const scheduleCard = document.getElementById('scheduleCard');
    
    if (!invoiceId) {
        invoiceDetails.style.display = 'none';
        scheduleCard.style.display = 'none';
        document.getElementById('submitBtn').disabled = true;
        return;
    }
    
    const selectedOption = document.querySelector(`#purchase_invoice_id option[value="${invoiceId}"]`);
    const invoiceNumber = selectedOption.dataset.invoiceNumber;
    const invoiceDate = selectedOption.dataset.invoiceDate;
    const dueDate = selectedOption.dataset.dueDate;
    const remainingAmount = parseFloat(selectedOption.dataset.remainingAmount);
    
    document.getElementById('invoiceNumber').textContent = invoiceNumber;
    document.getElementById('invoiceDate').textContent = new Date(invoiceDate).toLocaleDateString('ar-SA');
    document.getElementById('dueDate').textContent = new Date(dueDate).toLocaleDateString('ar-SA');
    document.getElementById('remainingAmount').textContent = remainingAmount.toFixed(2) + ' ر.س';
    
    // تعيين القيم الافتراضية
    document.getElementById('scheduled_amount').value = remainingAmount.toFixed(2);
    document.getElementById('scheduled_amount').setAttribute('max', remainingAmount);
    
    // تعيين تاريخ افتراضي للسداد (بعد أسبوع)
    const nextWeek = new Date();
    nextWeek.setDate(nextWeek.getDate() + 7);
    document.getElementById('schedule_date').value = nextWeek.toISOString().split('T')[0];
    
    // تعيين الموعد النهائي (بعد 10 أيام)
    const deadline = new Date();
    deadline.setDate(deadline.getDate() + 10);
    document.getElementById('payment_deadline').value = deadline.toISOString().split('T')[0];
    
    invoiceDetails.style.display = 'block';
    scheduleCard.style.display = 'block';
    document.getElementById('submitBtn').disabled = false;
}

// استخدام كامل المبلغ
function setFullAmount() {
    const checkbox = document.getElementById('useFullAmount');
    const amountInput = document.getElementById('scheduled_amount');
    const invoiceId = document.getElementById('purchase_invoice_id').value;
    
    if (checkbox.checked && invoiceId) {
        const selectedOption = document.querySelector(`#purchase_invoice_id option[value="${invoiceId}"]`);
        const remainingAmount = parseFloat(selectedOption.dataset.remainingAmount);
        amountInput.value = remainingAmount.toFixed(2);
        amountInput.readOnly = true;
    } else {
        amountInput.readOnly = false;
    }
}

// التحقق من صحة النموذج قبل الإرسال
document.getElementById('scheduleForm').addEventListener('submit', function(e) {
    const supplierId = document.getElementById('supplier_id').value;
    const invoiceId = document.getElementById('purchase_invoice_id').value;
    const scheduleDate = document.getElementById('schedule_date').value;
    const scheduledAmount = parseFloat(document.getElementById('scheduled_amount').value);
    
    if (!supplierId || !invoiceId || !scheduleDate || !scheduledAmount) {
        e.preventDefault();
        alert('يرجى ملء جميع الحقول المطلوبة');
        return;
    }
    
    const selectedOption = document.querySelector(`#purchase_invoice_id option[value="${invoiceId}"]`);
    const remainingAmount = parseFloat(selectedOption.dataset.remainingAmount);
    
    if (scheduledAmount > remainingAmount) {
        e.preventDefault();
        alert('المبلغ المجدول لا يمكن أن يكون أكبر من المبلغ المتبقي');
        return;
    }
    
    if (scheduledAmount <= 0) {
        e.preventDefault();
        alert('المبلغ المجدول يجب أن يكون أكبر من الصفر');
        return;
    }
});

// تحديث الموعد النهائي تلقائياً عند تغيير تاريخ السداد
document.getElementById('schedule_date').addEventListener('change', function() {
    const scheduleDate = new Date(this.value);
    if (scheduleDate) {
        const deadline = new Date(scheduleDate);
        deadline.setDate(deadline.getDate() + 3); // إضافة 3 أيام مهلة
        document.getElementById('payment_deadline').value = deadline.toISOString().split('T')[0];
    }
});

// تحميل بيانات المورد إذا كان محدد مسبقاً
@if(request('supplier_id'))
    document.addEventListener('DOMContentLoaded', function() {
        loadSupplierInvoices();
    });
@endif
</script>
@endpush