@extends('layouts.app')

@section('title', 'تقرير أداء الموردين المتقدم')

@section('content')
<div class="container-fluid">
    <!-- Header -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h1 class="h3 mb-0 text-gray-800">
                        <i class="fas fa-chart-bar me-2"></i>
                        تقرير أداء الموردين المتقدم
                    </h1>
                    <p class="text-muted mb-0">تحليل شامل لأداء الموردين وجودة الخدمة</p>
                </div>
                <div>
                    <button type="button" class="btn btn-success" onclick="exportReport()">
                        <i class="fas fa-file-excel me-1"></i>
                        تصدير Excel
                    </button>
                    <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#filterModal">
                        <i class="fas fa-filter me-1"></i>
                        فلترة
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- فلترة التواريخ -->
    <div class="card shadow mb-4">
        <div class="card-body">
            <form method="GET" class="row g-3">
                <div class="col-md-3">
                    <label class="form-label">من تاريخ</label>
                    <input type="date" name="date_from" class="form-control" value="{{ $dateFrom }}">
                </div>
                <div class="col-md-3">
                    <label class="form-label">إلى تاريخ</label>
                    <input type="date" name="date_to" class="form-control" value="{{ $dateTo }}">
                </div>
                <div class="col-md-3">
                    <label class="form-label">ترتيب حسب</label>
                    <select name="sort_by" class="form-select">
                        <option value="total_purchases">إجمالي المشتريات</option>
                        <option value="on_time_rate">معدل التسليم في الوقت</option>
                        <option value="quality_rate">معدل الجودة</option>
                        <option value="credit_utilization">استخدام الائتمان</option>
                    </select>
                </div>
                <div class="col-md-3 d-flex align-items-end">
                    <button type="submit" class="btn btn-primary w-100">
                        <i class="fas fa-search me-1"></i>
                        تطبيق الفلترة
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- جدول أداء الموردين -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">أداء الموردين - الفترة من {{ $dateFrom }} إلى {{ $dateTo }}</h6>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-hover" id="suppliersTable">
                    <thead class="table-light">
                        <tr>
                            <th>المورد</th>
                            <th>إجمالي المشتريات</th>
                            <th>عدد الأوامر</th>
                            <th>متوسط قيمة الأمر</th>
                            <th>معدل التسليم في الوقت</th>
                            <th>معدل الجودة</th>
                            <th>متوسط أيام التسليم</th>
                            <th>الرصيد الحالي</th>
                            <th>استخدام الائتمان</th>
                            <th>التقييم العام</th>
                            <th>الإجراءات</th>
                        </tr>
                    </thead>
                    <tbody>
                        @forelse($suppliers as $supplierData)
                        @php
                            $supplier = $supplierData['supplier'];
                            $overallScore = $this->calculateOverallScore($supplierData);
                            $ratingClass = $overallScore >= 85 ? 'success' : ($overallScore >= 70 ? 'warning' : 'danger');
                        @endphp
                        <tr>
                            <td>
                                <div class="d-flex align-items-center">
                                    <div class="me-3">
                                        <div class="rounded-circle bg-{{ $supplier->is_active ? 'success' : 'secondary' }} 
                                                    d-flex align-items-center justify-content-center" 
                                             style="width: 40px; height: 40px;">
                                            <i class="fas fa-building text-white"></i>
                                        </div>
                                    </div>
                                    <div>
                                        <div class="font-weight-bold">{{ $supplier->name }}</div>
                                        <small class="text-muted">{{ $supplier->code }}</small>
                                        @if($supplier->exceedsCreditLimit())
                                            <br><span class="badge badge-danger">تجاوز حد الائتمان</span>
                                        @endif
                                    </div>
                                </div>
                            </td>
                            <td>
                                <strong class="text-primary">{{ number_format($supplierData['total_purchases'], 2) }}</strong> ر.س
                            </td>
                            <td>
                                <div class="d-flex flex-column">
                                    <strong>{{ $supplierData['total_orders'] }}</strong>
                                    <small class="text-muted">مكتمل: {{ $supplierData['completed_orders'] }}</small>
                                </div>
                            </td>
                            <td>
                                <strong>{{ number_format($supplierData['average_invoice_value'], 2) }}</strong> ر.س
                            </td>
                            <td>
                                <div class="d-flex align-items-center">
                                    <div class="progress me-2" style="width: 60px; height: 6px;">
                                        <div class="progress-bar bg-{{ $supplierData['on_time_rate'] >= 90 ? 'success' : ($supplierData['on_time_rate'] >= 70 ? 'warning' : 'danger') }}" 
                                             style="width: {{ $supplierData['on_time_rate'] }}%"></div>
                                    </div>
                                    <span class="badge badge-{{ $supplierData['on_time_rate'] >= 90 ? 'success' : ($supplierData['on_time_rate'] >= 70 ? 'warning' : 'danger') }}">
                                        {{ $supplierData['on_time_rate'] }}%
                                    </span>
                                </div>
                            </td>
                            <td>
                                <div class="d-flex align-items-center">
                                    <div class="progress me-2" style="width: 60px; height: 6px;">
                                        <div class="progress-bar bg-{{ $supplierData['quality_rate'] >= 95 ? 'success' : ($supplierData['quality_rate'] >= 85 ? 'warning' : 'danger') }}" 
                                             style="width: {{ $supplierData['quality_rate'] }}%"></div>
                                    </div>
                                    <span class="badge badge-{{ $supplierData['quality_rate'] >= 95 ? 'success' : ($supplierData['quality_rate'] >= 85 ? 'warning' : 'danger') }}">
                                        {{ $supplierData['quality_rate'] }}%
                                    </span>
                                </div>
                            </td>
                            <td>
                                <span class="badge badge-{{ $supplierData['avg_delivery_days'] <= 7 ? 'success' : ($supplierData['avg_delivery_days'] <= 14 ? 'warning' : 'danger') }}">
                                    {{ $supplierData['avg_delivery_days'] }} يوم
                                </span>
                            </td>
                            <td>
                                <strong class="{{ $supplier->current_balance > 0 ? 'text-danger' : 'text-success' }}">
                                    {{ number_format($supplier->current_balance, 2) }} ر.س
                                </strong>
                            </td>
                            <td>
                                @if($supplier->credit_limit > 0)
                                    <div class="d-flex align-items-center">
                                        <div class="progress me-2" style="width: 60px; height: 6px;">
                                            <div class="progress-bar bg-{{ $supplier->credit_utilization <= 50 ? 'success' : ($supplier->credit_utilization <= 80 ? 'warning' : 'danger') }}" 
                                                 style="width: {{ min($supplier->credit_utilization, 100) }}%"></div>
                                        </div>
                                        <span class="badge badge-{{ $supplier->credit_utilization <= 50 ? 'success' : ($supplier->credit_utilization <= 80 ? 'warning' : 'danger') }}">
                                            {{ round($supplier->credit_utilization, 1) }}%
                                        </span>
                                    </div>
                                @else
                                    <span class="text-muted">لا يوجد حد</span>
                                @endif
                            </td>
                            <td>
                                <div class="text-center">
                                    <div class="circular-progress" data-score="{{ $overallScore }}">
                                        <svg class="progress-ring" width="50" height="50">
                                            <circle class="progress-ring__circle" stroke="{{ $ratingClass === 'success' ? '#1cc88a' : ($ratingClass === 'warning' ? '#f6c23e' : '#e74a3b') }}" 
                                                    stroke-width="3" fill="transparent" r="20" cx="25" cy="25"/>
                                        </svg>
                                        <div class="progress-text">{{ $overallScore }}%</div>
                                    </div>
                                    <small class="d-block mt-1 text-{{ $ratingClass }}">
                                        {{ $overallScore >= 85 ? 'ممتاز' : ($overallScore >= 70 ? 'جيد' : 'ضعيف') }}
                                    </small>
                                </div>
                            </td>
                            <td>
                                <div class="dropdown">
                                    <button class="btn btn-sm btn-outline-primary dropdown-toggle" type="button" 
                                            data-bs-toggle="dropdown">
                                        إجراءات
                                    </button>
                                    <ul class="dropdown-menu">
                                        <li>
                                            <a class="dropdown-item" href="{{ route('suppliers.analytics.statement', $supplier) }}">
                                                <i class="fas fa-file-alt me-2"></i>
                                                كشف الحساب
                                            </a>
                                        </li>
                                        <li>
                                            <a class="dropdown-item" href="{{ route('suppliers.show', $supplier) }}">
                                                <i class="fas fa-eye me-2"></i>
                                                عرض التفاصيل
                                            </a>
                                        </li>
                                        <li>
                                            <a class="dropdown-item" href="{{ route('suppliers.edit', $supplier) }}">
                                                <i class="fas fa-edit me-2"></i>
                                                تعديل
                                            </a>
                                        </li>
                                        <li><hr class="dropdown-divider"></li>
                                        <li>
                                            <a class="dropdown-item" href="#" onclick="generateSupplierReport({{ $supplier->id }})">
                                                <i class="fas fa-chart-line me-2"></i>
                                                تقرير مفصل
                                            </a>
                                        </li>
                                    </ul>
                                </div>
                            </td>
                        </tr>
                        @empty
                        <tr>
                            <td colspan="11" class="text-center text-muted py-5">
                                <i class="fas fa-inbox fa-3x mb-3"></i>
                                <div>لا توجد بيانات موردين في هذه الفترة</div>
                            </td>
                        </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- إحصائيات إضافية -->
    <div class="row">
        <div class="col-md-6 mb-4">
            <div class="card shadow">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">توزيع الموردين حسب الأداء</h6>
                </div>
                <div class="card-body">
                    <canvas id="performanceDistChart"></canvas>
                </div>
            </div>
        </div>

        <div class="col-md-6 mb-4">
            <div class="card shadow">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">مقارنة المؤشرات الرئيسية</h6>
                </div>
                <div class="card-body">
                    <canvas id="kpiComparisonChart"></canvas>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Modal للفلترة المتقدمة -->
<div class="modal fade" id="filterModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">فلترة متقدمة</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="advancedFilterForm">
                    <div class="mb-3">
                        <label class="form-label">الحد الأدنى لقيمة المشتريات</label>
                        <input type="number" name="min_purchases" class="form-control" step="0.01">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">معدل التسليم في الوقت (أكبر من)</label>
                        <input type="number" name="min_on_time_rate" class="form-control" min="0" max="100">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">حالة المورد</label>
                        <select name="supplier_status" class="form-select">
                            <option value="">الكل</option>
                            <option value="active">نشط</option>
                            <option value="inactive">غير نشط</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">مستوى المخاطر</label>
                        <select name="risk_level" class="form-select">
                            <option value="">الكل</option>
                            <option value="low">منخفض</option>
                            <option value="medium">متوسط</option>
                            <option value="high">عالي</option>
                            <option value="critical">حرج</option>
                        </select>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إغلاق</button>
                <button type="button" class="btn btn-primary" onclick="applyAdvancedFilter()">تطبيق</button>
            </div>
        </div>
    </div>
</div>
@endsection

@push('styles')
<style>
.circular-progress {
    position: relative;
    display: inline-block;
}

.progress-ring {
    transform: rotate(-90deg);
}

.progress-ring__circle {
    transition: stroke-dashoffset 0.35s;
    transform: rotate(-90deg);
    transform-origin: 50% 50%;
}

.progress-text {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    font-size: 11px;
    font-weight: bold;
}
</style>
@endpush

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
// تهيئة الدوائر الدائرية للتقييم
document.querySelectorAll('.circular-progress').forEach(element => {
    const score = element.dataset.score;
    const circle = element.querySelector('.progress-ring__circle');
    const radius = circle.r.baseVal.value;
    const circumference = radius * 2 * Math.PI;
    
    circle.style.strokeDasharray = `${circumference} ${circumference}`;
    circle.style.strokeDashoffset = circumference;
    
    const offset = circumference - score / 100 * circumference;
    circle.style.strokeDashoffset = offset;
});

// رسم توزيع الأداء
const performanceData = @json($suppliers->groupBy(function($supplier) {
    $score = calculateOverallScore($supplier);
    if ($score >= 85) return 'ممتاز';
    if ($score >= 70) return 'جيد';
    return 'ضعيف';
})->map->count());

const performanceDistCtx = document.getElementById('performanceDistChart').getContext('2d');
new Chart(performanceDistCtx, {
    type: 'pie',
    data: {
        labels: Object.keys(performanceData),
        datasets: [{
            data: Object.values(performanceData),
            backgroundColor: ['#1cc88a', '#f6c23e', '#e74a3b']
        }]
    },
    options: {
        responsive: true,
        plugins: {
            legend: {
                position: 'bottom'
            }
        }
    }
});

// رسم مقارنة المؤشرات
const kpiCtx = document.getElementById('kpiComparisonChart').getContext('2d');
new Chart(kpiCtx, {
    type: 'radar',
    data: {
        labels: ['التسليم في الوقت', 'الجودة', 'السعر التنافسي', 'سرعة الاستجابة', 'الموثوقية'],
        datasets: [{
            label: 'متوسط الصناعة',
            data: [85, 90, 75, 80, 88],
            borderColor: '#e74a3b',
            backgroundColor: 'rgba(231, 74, 59, 0.1)'
        }, {
            label: 'موردينا',
            data: [
                {{ $suppliers->avg('on_time_rate') ?? 0 }},
                {{ $suppliers->avg('quality_rate') ?? 0 }},
                78, 82, 85
            ],
            borderColor: '#1cc88a',
            backgroundColor: 'rgba(28, 200, 138, 0.1)'
        }]
    },
    options: {
        responsive: true,
        scales: {
            r: {
                beginAtZero: true,
                max: 100
            }
        }
    }
});

// تطبيق الفلترة المتقدمة
function applyAdvancedFilter() {
    const formData = new FormData(document.getElementById('advancedFilterForm'));
    const params = new URLSearchParams(window.location.search);
    
    for (let [key, value] of formData) {
        if (value) params.set(key, value);
    }
    
    window.location.search = params.toString();
}

// تصدير التقرير
function exportReport() {
    const params = new URLSearchParams(window.location.search);
    params.set('export', 'excel');
    window.open(`{{ route('suppliers.analytics.export') }}?${params.toString()}`);
}

// إنتاج تقرير مفصل للمورد
function generateSupplierReport(supplierId) {
    window.open(`/suppliers/analytics/${supplierId}/statement?detailed=1`);
}

// حساب التقييم العام (للاستخدام في JavaScript)
function calculateOverallScore(supplierData) {
    const onTimeWeight = 0.3;
    const qualityWeight = 0.3;
    const reliabilityWeight = 0.2;
    const financialWeight = 0.2;
    
    const onTimeScore = supplierData.on_time_rate;
    const qualityScore = supplierData.quality_rate;
    const reliabilityScore = 85; // افتراضي
    const financialScore = supplierData.payment_terms_compliance;
    
    return Math.round(
        (onTimeScore * onTimeWeight) +
        (qualityScore * qualityWeight) +
        (reliabilityScore * reliabilityWeight) +
        (financialScore * financialWeight)
    );
}
</script>
@endpush