@extends('layouts.app')

@section('title', 'مراقبة حدود الائتمان')
@section('page-title', 'مراقبة حدود الائتمان')

@section('page-actions')
    <div class="btn-group" role="group">
        <a href="{{ route('suppliers.analytics.dashboard') }}" class="btn btn-secondary">
            <i class="fas fa-arrow-right me-2"></i>
            العودة للوحة التحكم
        </a>
        <button type="button" class="btn btn-warning" onclick="refreshData()">
            <i class="fas fa-sync-alt me-2"></i>
            تحديث البيانات
        </button>
        <div class="btn-group" role="group">
            <button type="button" class="btn btn-success dropdown-toggle" data-bs-toggle="dropdown">
                <i class="fas fa-download me-2"></i>
                تصدير
            </button>
            <ul class="dropdown-menu">
                <li><a class="dropdown-item" href="#" onclick="exportReport('credit-monitoring', 'excel')">
                    <i class="fas fa-file-excel me-2"></i>Excel
                </a></li>
                <li><a class="dropdown-item" href="#" onclick="exportReport('credit-monitoring', 'pdf')">
                    <i class="fas fa-file-pdf me-2"></i>PDF
                </a></li>
            </ul>
        </div>
    </div>
@endsection

@section('content')
<!-- إحصائيات سريعة -->
<div class="row mb-4">
    @php
        $totalSuppliers = $suppliers->count();
        $criticalSuppliers = $suppliers->where('risk_level', 'critical')->count();
        $highRiskSuppliers = $suppliers->where('risk_level', 'high')->count();
        $exceedingLimitSuppliers = $suppliers->where('exceeds_limit', true)->count();
        $averageUtilization = $suppliers->avg('credit_utilization');
    @endphp
    
    <div class="col-lg-3 col-md-6 mb-3">
        <div class="card text-white bg-primary">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="flex-shrink-0">
                        <i class="fas fa-credit-card fa-2x"></i>
                    </div>
                    <div class="flex-grow-1 ms-3">
                        <div class="card-title h5 mb-1">{{ $totalSuppliers }}</div>
                        <div class="card-text small">إجمالي الموردين</div>
                    </div>
                </div>
                <div class="progress mt-2" style="height: 4px;">
                    <div class="progress-bar bg-white" style="width: {{ $averageUtilization }}%"></div>
                </div>
                <small class="text-white-50">متوسط الاستخدام {{ number_format($averageUtilization, 1) }}%</small>
            </div>
        </div>
    </div>
    
    <div class="col-lg-3 col-md-6 mb-3">
        <div class="card text-white bg-danger">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="flex-shrink-0">
                        <i class="fas fa-exclamation-triangle fa-2x"></i>
                    </div>
                    <div class="flex-grow-1 ms-3">
                        <div class="card-title h5 mb-1">{{ $criticalSuppliers }}</div>
                        <div class="card-text small">حالة حرجة</div>
                    </div>
                </div>
                <div class="progress mt-2" style="height: 4px;">
                    <div class="progress-bar bg-white" style="width: {{ $totalSuppliers > 0 ? ($criticalSuppliers / $totalSuppliers) * 100 : 0 }}%"></div>
                </div>
                <small class="text-white-50">استخدام 90%+</small>
            </div>
        </div>
    </div>
    
    <div class="col-lg-3 col-md-6 mb-3">
        <div class="card text-white bg-warning">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="flex-shrink-0">
                        <i class="fas fa-exclamation fa-2x"></i>
                    </div>
                    <div class="flex-grow-1 ms-3">
                        <div class="card-title h5 mb-1">{{ $highRiskSuppliers }}</div>
                        <div class="card-text small">مخاطر عالية</div>
                    </div>
                </div>
                <div class="progress mt-2" style="height: 4px;">
                    <div class="progress-bar bg-white" style="width: {{ $totalSuppliers > 0 ? ($highRiskSuppliers / $totalSuppliers) * 100 : 0 }}%"></div>
                </div>
                <small class="text-white-50">استخدام 75-90%</small>
            </div>
        </div>
    </div>
    
    <div class="col-lg-3 col-md-6 mb-3">
        <div class="card text-white bg-info">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="flex-shrink-0">
                        <i class="fas fa-ban fa-2x"></i>
                    </div>
                    <div class="flex-grow-1 ms-3">
                        <div class="card-title h5 mb-1">{{ $exceedingLimitSuppliers }}</div>
                        <div class="card-text small">تجاوز الحد</div>
                    </div>
                </div>
                <div class="progress mt-2" style="height: 4px;">
                    <div class="progress-bar bg-white" style="width: {{ $totalSuppliers > 0 ? ($exceedingLimitSuppliers / $totalSuppliers) * 100 : 0 }}%"></div>
                </div>
                <small class="text-white-50">استخدام 100%+</small>
            </div>
        </div>
    </div>
</div>

<!-- فلاتر سريعة -->
<div class="row mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-body">
                <div class="row align-items-center">
                    <div class="col-md-8">
                        <div class="btn-group" role="group">
                            <input type="radio" class="btn-check" name="filter" id="all" autocomplete="off" checked>
                            <label class="btn btn-outline-primary" for="all" onclick="filterSuppliers('all')">
                                الكل ({{ $totalSuppliers }})
                            </label>
                            
                            <input type="radio" class="btn-check" name="filter" id="critical" autocomplete="off">
                            <label class="btn btn-outline-danger" for="critical" onclick="filterSuppliers('critical')">
                                حرج ({{ $criticalSuppliers }})
                            </label>
                            
                            <input type="radio" class="btn-check" name="filter" id="high" autocomplete="off">
                            <label class="btn btn-outline-warning" for="high" onclick="filterSuppliers('high')">
                                مرتفع ({{ $highRiskSuppliers }})
                            </label>
                            
                            <input type="radio" class="btn-check" name="filter" id="exceeding" autocomplete="off">
                            <label class="btn btn-outline-info" for="exceeding" onclick="filterSuppliers('exceeding')">
                                تجاوز الحد ({{ $exceedingLimitSuppliers }})
                            </label>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="input-group">
                            <input type="text" class="form-control" placeholder="البحث في الموردين..." id="searchInput">
                            <button class="btn btn-outline-secondary" type="button">
                                <i class="fas fa-search"></i>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- جدول مراقبة الائتمان -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h6 class="card-title mb-0">
                    <i class="fas fa-table me-2"></i>
                    تفاصيل حدود الائتمان
                </h6>
                <div class="card-tools">
                    <span class="badge bg-secondary">{{ $totalSuppliers }} مورد</span>
                </div>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0" id="creditTable">
                        <thead class="table-light">
                            <tr>
                                <th>المورد</th>
                                <th class="text-center">حد الائتمان</th>
                                <th class="text-center">الرصيد الحالي</th>
                                <th class="text-center">الائتمان المتاح</th>
                                <th class="text-center">نسبة الاستخدام</th>
                                <th class="text-center">مستوى المخاطر</th>
                                <th class="text-center">الإجراءات</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($suppliers as $data)
                            <tr class="supplier-row" data-risk="{{ $data['risk_level'] }}" data-exceeding="{{ $data['exceeds_limit'] ? '1' : '0' }}">
                                <td>
                                    <div class="d-flex align-items-center">
                                        <div class="flex-shrink-0 me-3">
                                            @switch($data['risk_level'])
                                                @case('critical')
                                                    <i class="fas fa-exclamation-triangle text-danger fa-lg"></i>
                                                    @break
                                                @case('high')
                                                    <i class="fas fa-exclamation text-warning fa-lg"></i>
                                                    @break
                                                @case('medium')
                                                    <i class="fas fa-minus-circle text-info fa-lg"></i>
                                                    @break
                                                @default
                                                    <i class="fas fa-check-circle text-success fa-lg"></i>
                                            @endswitch
                                        </div>
                                        <div class="flex-grow-1">
                                            <strong>{{ $data['supplier']->name }}</strong>
                                            @if($data['supplier']->name_en)
                                                <br><small class="text-muted">{{ $data['supplier']->name_en }}</small>
                                            @endif
                                            @if($data['exceeds_limit'])
                                                <br><span class="badge bg-danger badge-sm">تجاوز الحد</span>
                                            @endif
                                        </div>
                                    </div>
                                </td>
                                <td class="text-center">
                                    <strong class="text-primary">{{ number_format($data['supplier']->credit_limit, 0) }} ر.س</strong>
                                </td>
                                <td class="text-center">
                                    <span class="text-{{ $data['exceeds_limit'] ? 'danger' : 'dark' }} fw-bold">
                                        {{ number_format($data['supplier']->current_balance, 0) }} ر.س
                                    </span>
                                </td>
                                <td class="text-center">
                                    <span class="text-{{ $data['available_credit'] > 0 ? 'success' : 'danger' }} fw-bold">
                                        {{ number_format($data['available_credit'], 0) }} ر.س
                                    </span>
                                </td>
                                <td class="text-center">
                                    <div class="d-flex align-items-center justify-content-center">
                                        <div class="progress me-2" style="width: 60px; height: 8px;">
                                            @php
                                                $progressClass = 'bg-success';
                                                if ($data['credit_utilization'] >= 90) $progressClass = 'bg-danger';
                                                elseif ($data['credit_utilization'] >= 75) $progressClass = 'bg-warning';
                                                elseif ($data['credit_utilization'] >= 50) $progressClass = 'bg-info';
                                            @endphp
                                            <div class="progress-bar {{ $progressClass }}" 
                                                 style="width: {{ min(100, $data['credit_utilization']) }}%"></div>
                                        </div>
                                        <span class="fw-bold text-{{ str_replace('bg-', '', $progressClass) }}">
                                            {{ number_format($data['credit_utilization'], 1) }}%
                                        </span>
                                    </div>
                                </td>
                                <td class="text-center">
                                    @switch($data['risk_level'])
                                        @case('critical')
                                            <span class="badge bg-danger">حرج</span>
                                            @break
                                        @case('high')
                                            <span class="badge bg-warning">مرتفع</span>
                                            @break
                                        @case('medium')
                                            <span class="badge bg-info">متوسط</span>
                                            @break
                                        @default
                                            <span class="badge bg-success">منخفض</span>
                                    @endswitch
                                </td>
                                <td class="text-center">
                                    <div class="btn-group btn-group-sm" role="group">
                                        <a href="{{ route('suppliers.show', $data['supplier']) }}" 
                                           class="btn btn-outline-info" title="عرض المورد">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                        <a href="{{ route('suppliers.analytics.statement', $data['supplier']) }}" 
                                           class="btn btn-outline-primary" title="كشف الحساب">
                                            <i class="fas fa-file-invoice"></i>
                                        </a>
                                        @if($data['supplier']->current_balance > 0)
                                        <a href="{{ route('suppliers.payments.create', ['supplier_id' => $data['supplier']->id]) }}" 
                                           class="btn btn-outline-success" title="إضافة دفعة">
                                            <i class="fas fa-money-bill-wave"></i>
                                        </a>
                                        @endif
                                        <button type="button" class="btn btn-outline-warning" 
                                                onclick="updateCreditLimit({{ $data['supplier']->id }}, {{ $data['supplier']->credit_limit }})"
                                                title="تعديل حد الائتمان">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- الرسوم البيانية -->
<div class="row mt-4">
    <div class="col-md-6">
        <div class="card">
            <div class="card-header">
                <h6 class="card-title mb-0">
                    <i class="fas fa-chart-pie me-2"></i>
                    توزيع مستويات المخاطر
                </h6>
            </div>
            <div class="card-body">
                <canvas id="riskDistributionChart" style="height: 300px;"></canvas>
            </div>
        </div>
    </div>
    <div class="col-md-6">
        <div class="card">
            <div class="card-header">
                <h6 class="card-title mb-0">
                    <i class="fas fa-chart-bar me-2"></i>
                    استخدام حدود الائتمان
                </h6>
            </div>
            <div class="card-body">
                <canvas id="utilizationChart" style="height: 300px;"></canvas>
            </div>
        </div>
    </div>
</div>
@endsection

<!-- مودال تعديل حد الائتمان -->
<div class="modal fade" id="creditLimitModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form id="creditLimitForm">
                <div class="modal-header">
                    <h5 class="modal-title">تعديل حد الائتمان</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="supplierName" class="form-label">المورد</label>
                        <input type="text" class="form-control" id="supplierName" readonly>
                        <input type="hidden" id="supplierId">
                    </div>
                    <div class="mb-3">
                        <label for="currentLimit" class="form-label">الحد الحالي (ر.س)</label>
                        <input type="number" class="form-control" id="currentLimit" readonly>
                    </div>
                    <div class="mb-3">
                        <label for="newLimit" class="form-label">الحد الجديد (ر.س)</label>
                        <input type="number" class="form-control" id="newLimit" min="0" step="0.01" required>
                    </div>
                    <div class="mb-3">
                        <label for="reason" class="form-label">سبب التعديل</label>
                        <textarea class="form-control" id="reason" rows="3" required></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                    <button type="submit" class="btn btn-warning">حفظ التعديل</button>
                </div>
            </form>
        </div>
    </div>
</div>

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
    // بيانات الرسوم البيانية
    const riskData = {
        low: {{ $suppliers->where('risk_level', 'low')->count() }},
        medium: {{ $suppliers->where('risk_level', 'medium')->count() }},
        high: {{ $suppliers->where('risk_level', 'high')->count() }},
        critical: {{ $suppliers->where('risk_level', 'critical')->count() }}
    };
    
    // رسم بياني لتوزيع المخاطر
    const riskCtx = document.getElementById('riskDistributionChart').getContext('2d');
    new Chart(riskCtx, {
        type: 'doughnut',
        data: {
            labels: ['منخفض', 'متوسط', 'مرتفع', 'حرج'],
            datasets: [{
                data: [riskData.low, riskData.medium, riskData.high, riskData.critical],
                backgroundColor: [
                    'rgb(40, 167, 69)',
                    'rgb(23, 162, 184)',
                    'rgb(255, 193, 7)',
                    'rgb(220, 53, 69)'
                ]
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'bottom'
                }
            }
        }
    });
    
    // رسم بياني لاستخدام الائتمان
    const utilizationCtx = document.getElementById('utilizationChart').getContext('2d');
    const utilizationRanges = {
        '0-25%': {{ $suppliers->where('credit_utilization', '<', 25)->count() }},
        '25-50%': {{ $suppliers->whereBetween('credit_utilization', [25, 50])->count() }},
        '50-75%': {{ $suppliers->whereBetween('credit_utilization', [50, 75])->count() }},
        '75-90%': {{ $suppliers->whereBetween('credit_utilization', [75, 90])->count() }},
        '90%+': {{ $suppliers->where('credit_utilization', '>=', 90)->count() }}
    };
    
    new Chart(utilizationCtx, {
        type: 'bar',
        data: {
            labels: ['0-25%', '25-50%', '50-75%', '75-90%', '90%+'],
            datasets: [{
                label: 'عدد الموردين',
                data: Object.values(utilizationRanges),
                backgroundColor: [
                    'rgba(40, 167, 69, 0.8)',
                    'rgba(23, 162, 184, 0.8)',
                    'rgba(255, 193, 7, 0.8)',
                    'rgba(253, 126, 20, 0.8)',
                    'rgba(220, 53, 69, 0.8)'
                ]
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                y: {
                    beginAtZero: true
                }
            },
            plugins: {
                legend: {
                    display: false
                }
            }
        }
    });
    
    // فلترة الموردين
    function filterSuppliers(filter) {
        const rows = document.querySelectorAll('.supplier-row');
        
        rows.forEach(row => {
            let show = false;
            
            switch(filter) {
                case 'all':
                    show = true;
                    break;
                case 'critical':
                    show = row.dataset.risk === 'critical';
                    break;
                case 'high':
                    show = row.dataset.risk === 'high';
                    break;
                case 'exceeding':
                    show = row.dataset.exceeding === '1';
                    break;
            }
            
            row.style.display = show ? '' : 'none';
        });
    }
    
    // البحث في الموردين
    document.getElementById('searchInput').addEventListener('input', function(e) {
        const searchTerm = e.target.value.toLowerCase();
        const rows = document.querySelectorAll('.supplier-row');
        
        rows.forEach(row => {
            const supplierName = row.querySelector('strong').textContent.toLowerCase();
            const show = supplierName.includes(searchTerm);
            row.style.display = show ? '' : 'none';
        });
    });
    
    // تعديل حد الائتمان
    function updateCreditLimit(supplierId, currentLimit) {
        // الحصول على اسم المورد من الجدول
        const supplierRow = document.querySelector(`[data-supplier-id="${supplierId}"]`);
        const supplierName = supplierRow ? supplierRow.querySelector('strong').textContent : 'غير محدد';
        
        document.getElementById('supplierId').value = supplierId;
        document.getElementById('supplierName').value = supplierName;
        document.getElementById('currentLimit').value = currentLimit;
        document.getElementById('newLimit').value = currentLimit;
        
        const modal = new bootstrap.Modal(document.getElementById('creditLimitModal'));
        modal.show();
    }
    
    // معالج نموذج تعديل حد الائتمان
    document.getElementById('creditLimitForm').addEventListener('submit', function(e) {
        e.preventDefault();
        
        const formData = {
            supplier_id: document.getElementById('supplierId').value,
            new_limit: document.getElementById('newLimit').value,
            reason: document.getElementById('reason').value
        };
        
        // إرسال الطلب إلى الخادم
        fetch('/suppliers/update-credit-limit', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify(formData)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                location.reload(); // إعادة تحميل الصفحة لعرض البيانات المحدثة
            } else {
                alert('حدث خطأ في تحديث حد الائتمان');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('حدث خطأ في تحديث حد الائتمان');
        });
    });
    
    // تحديث البيانات
    function refreshData() {
        location.reload();
    }
    
    // دالة التصدير
    function exportReport(type, format) {
        const url = "{{ route('suppliers.analytics.export') }}";
        const params = new URLSearchParams({
            type: type,
            format: format
        });
        
        window.open(`${url}?${params}`, '_blank');
    }
</script>
@endpush

@push('styles')
<style>
    .badge-sm {
        font-size: 0.75rem;
        padding: 0.25rem 0.5rem;
    }
    
    .progress {
        background-color: #e9ecef;
    }
    
    .table td {
        vertical-align: middle;
    }
    
    .btn-group-sm .btn {
        font-size: 0.75rem;
        padding: 0.25rem 0.5rem;
    }
    
    .card-title {
        font-weight: 600;
    }
    
    @media (max-width: 768px) {
        .table-responsive {
            font-size: 0.875rem;
        }
        
        .btn-group .btn {
            font-size: 0.75rem;
            padding: 0.25rem 0.5rem;
        }
    }
</style>
@endpush