@extends('layouts.app')

@section('title', 'مقارنة الموردين')
@section('page-title', 'مقارنة أداء الموردين')

@section('page-actions')
    <div class="btn-group" role="group">
        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#supplierSelectionModal">
            <i class="fas fa-users me-2"></i>
            اختيار الموردين
        </button>
        <div class="btn-group" role="group">
            <button type="button" class="btn btn-outline-secondary dropdown-toggle" data-bs-toggle="dropdown">
                <i class="fas fa-download me-2"></i>
                تصدير المقارنة
            </button>
            <ul class="dropdown-menu">
                <li><a class="dropdown-item" href="#" onclick="exportComparison('excel')">
                    <i class="fas fa-file-excel me-2"></i>Excel
                </a></li>
                <li><a class="dropdown-item" href="#" onclick="exportComparison('pdf')">
                    <i class="fas fa-file-pdf me-2"></i>PDF
                </a></li>
            </ul>
        </div>
    </div>
@endsection

@section('content')
<!-- معلومات الفترة الزمنية -->
<div class="row mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-body">
                <form method="GET" action="{{ route('suppliers.analytics.comparison') }}" class="row align-items-end">
                    <div class="col-md-3">
                        <label for="date_from" class="form-label">من تاريخ</label>
                        <input type="date" name="date_from" id="date_from" class="form-control" value="{{ $dateFrom }}">
                    </div>
                    <div class="col-md-3">
                        <label for="date_to" class="form-label">إلى تاريخ</label>
                        <input type="date" name="date_to" id="date_to" class="form-control" value="{{ $dateTo }}">
                    </div>
                    <div class="col-md-4">
                        <label for="supplier_ids" class="form-label">الموردين المحددين</label>
                        <select name="supplier_ids[]" id="supplier_ids" class="form-select" multiple>
                            @foreach($allSuppliers as $supplier)
                                <option value="{{ $supplier->id }}" 
                                        {{ in_array($supplier->id, $supplierIds) ? 'selected' : '' }}>
                                    {{ $supplier->name }}
                                </option>
                            @endforeach
                        </select>
                    </div>
                    <div class="col-md-2">
                        <button type="submit" class="btn btn-primary w-100">
                            <i class="fas fa-search me-1"></i>
                            تطبيق
                        </button>
                    </div>
                    @foreach($supplierIds as $id)
                        <input type="hidden" name="supplier_ids[]" value="{{ $id }}">
                    @endforeach
                </form>
            </div>
        </div>
    </div>
</div>

<!-- مقارنة الأداء العام -->
<div class="row mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h6 class="card-title mb-0">
                    <i class="fas fa-chart-bar me-2"></i>
                    مقارنة الأداء العام
                </h6>
                <div class="card-tools">
                    <span class="badge bg-secondary">{{ count($comparison) }} مورد</span>
                </div>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-8">
                        <canvas id="performanceComparisonChart" height="300"></canvas>
                    </div>
                    <div class="col-md-4">
                        <div class="table-responsive">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>المؤشر</th>
                                        <th class="text-center">المتوسط</th>
                                        <th class="text-center">الأفضل</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <tr>
                                        <td>إجمالي المشتريات</td>
                                        <td class="text-center">{{ number_format($comparison->avg('total_amount'), 0) }}</td>
                                        <td class="text-center text-success">{{ number_format($comparison->max('total_amount'), 0) }}</td>
                                    </tr>
                                    <tr>
                                        <td>متوسط قيمة الفاتورة</td>
                                        <td class="text-center">{{ number_format($comparison->avg('average_amount'), 0) }}</td>
                                        <td class="text-center text-success">{{ number_format($comparison->max('average_amount'), 0) }}</td>
                                    </tr>
                                    <tr>
                                        <td>معدل الدفع</td>
                                        <td class="text-center">{{ number_format($comparison->avg('payment_rate'), 1) }}%</td>
                                        <td class="text-center text-success">{{ number_format($comparison->max('payment_rate'), 1) }}%</td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- جدول المقارنة التفصيلية -->
<div class="row mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h6 class="card-title mb-0">
                    <i class="fas fa-table me-2"></i>
                    مقارنة تفصيلية للموردين
                </h6>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead class="table-light">
                            <tr>
                                <th>المورد</th>
                                <th class="text-center">إجمالي المشتريات</th>
                                <th class="text-center">عدد الفواتير</th>
                                <th class="text-center">متوسط قيمة الفاتورة</th>
                                <th class="text-center">معدل الدفع</th>
                                <th class="text-center">الرصيد الحالي</th>
                                <th class="text-center">التقييم النسبي</th>
                            </tr>
                        </thead>
                        <tbody>
                            @forelse($comparison as $data)
                            @php
                                $supplier = $data['supplier'];
                                $maxPurchases = $comparison->max('total_amount');
                                $relativeScore = $maxPurchases > 0 ? ($data['total_amount'] / $maxPurchases) * 100 : 0;
                                $scoreClass = $relativeScore >= 80 ? 'success' : ($relativeScore >= 60 ? 'warning' : 'danger');
                            @endphp
                            <tr>
                                <td>
                                    <div class="d-flex align-items-center">
                                        <div class="flex-shrink-0 me-3">
                                            <div class="rounded-circle bg-{{ $supplier->is_active ? 'success' : 'secondary' }} 
                                                        d-flex align-items-center justify-content-center" 
                                                 style="width: 40px; height: 40px;">
                                                <i class="fas fa-building text-white"></i>
                                            </div>
                                        </div>
                                        <div>
                                            <strong>{{ $supplier->name }}</strong>
                                            @if($supplier->email)
                                                <br><small class="text-muted">{{ $supplier->email }}</small>
                                            @endif
                                            @if($supplier->phone)
                                                <br><small class="text-info">{{ $supplier->phone }}</small>
                                            @endif
                                        </div>
                                    </div>
                                </td>
                                <td class="text-center">
                                    <div class="d-flex flex-column align-items-center">
                                        <strong class="text-primary">{{ number_format($data['total_amount'], 2) }}</strong>
                                        <small class="text-muted">ر.س</small>
                                        <div class="progress mt-1" style="width: 60px; height: 4px;">
                                            <div class="progress-bar bg-primary" 
                                                 style="width: {{ $maxPurchases > 0 ? ($data['total_amount'] / $maxPurchases) * 100 : 0 }}%"></div>
                                        </div>
                                    </div>
                                </td>
                                <td class="text-center">
                                    <strong>{{ $data['invoice_count'] }}</strong>
                                    <br><small class="text-muted">فاتورة</small>
                                </td>
                                <td class="text-center">
                                    <strong>{{ number_format($data['average_amount'], 2) }}</strong>
                                    <br><small class="text-muted">ر.س</small>
                                </td>
                                <td class="text-center">
                                    <div class="d-flex flex-column align-items-center">
                                        <strong class="text-{{ $data['payment_rate'] >= 90 ? 'success' : ($data['payment_rate'] >= 70 ? 'warning' : 'danger') }}">
                                            {{ number_format($data['payment_rate'], 1) }}%
                                        </strong>
                                        <div class="progress mt-1" style="width: 50px; height: 4px;">
                                            <div class="progress-bar bg-{{ $data['payment_rate'] >= 90 ? 'success' : ($data['payment_rate'] >= 70 ? 'warning' : 'danger') }}" 
                                                 style="width: {{ $data['payment_rate'] }}%"></div>
                                        </div>
                                    </div>
                                </td>
                                <td class="text-center">
                                    <strong class="text-{{ $data['current_balance'] > 0 ? 'danger' : 'success' }}">
                                        {{ number_format($data['current_balance'], 2) }}
                                    </strong>
                                    <br><small class="text-muted">ر.س</small>
                                </td>
                                <td class="text-center">
                                    <div class="d-flex flex-column align-items-center">
                                        <div class="circular-progress-small" data-score="{{ $relativeScore }}">
                                            <svg class="progress-ring-small" width="40" height="40">
                                                <circle class="progress-ring__circle-small" 
                                                        stroke="{{ $scoreClass === 'success' ? '#1cc88a' : ($scoreClass === 'warning' ? '#f6c23e' : '#e74a3b') }}" 
                                                        stroke-width="3" fill="transparent" r="16" cx="20" cy="20"/>
                                            </svg>
                                            <div class="progress-text-small">{{ number_format($relativeScore, 0) }}%</div>
                                        </div>
                                        <small class="mt-1 text-{{ $scoreClass }}">
                                            {{ $relativeScore >= 80 ? 'ممتاز' : ($relativeScore >= 60 ? 'جيد' : 'ضعيف') }}
                                        </small>
                                    </div>
                                </td>
                            </tr>
                            @empty
                            <tr>
                                <td colspan="7" class="text-center py-5">
                                    <div class="text-muted">
                                        <i class="fas fa-users fa-3x mb-3"></i>
                                        <h5>لا توجد موردين للمقارنة</h5>
                                        <p>يرجى اختيار موردين للمقارنة بينهم</p>
                                        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#supplierSelectionModal">
                                            <i class="fas fa-plus me-2"></i>
                                            اختيار الموردين
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- تحليل إضافي -->
@if(count($comparison) > 0)
<div class="row">
    <div class="col-md-6 mb-4">
        <div class="card">
            <div class="card-header">
                <h6 class="card-title mb-0">
                    <i class="fas fa-chart-pie me-2"></i>
                    توزيع المشتريات
                </h6>
            </div>
            <div class="card-body">
                <canvas id="purchaseDistributionChart"></canvas>
            </div>
        </div>
    </div>
    
    <div class="col-md-6 mb-4">
        <div class="card">
            <div class="card-header">
                <h6 class="card-title mb-0">
                    <i class="fas fa-chart-line me-2"></i>
                    مقارنة معدلات الدفع
                </h6>
            </div>
            <div class="card-body">
                <canvas id="paymentRatesChart"></canvas>
            </div>
        </div>
    </div>
</div>

<!-- توصيات -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h6 class="card-title mb-0">
                    <i class="fas fa-lightbulb me-2"></i>
                    توصيات التحسين
                </h6>
            </div>
            <div class="card-body">
                <div class="row">
                    @php
                        $bestPerformer = $comparison->sortByDesc('total_amount')->first();
                        $worstPaymentRate = $comparison->sortBy('payment_rate')->first();
                        $highestBalance = $comparison->sortByDesc('current_balance')->first();
                    @endphp
                    
                    @if($bestPerformer)
                    <div class="col-md-4 mb-3">
                        <div class="alert alert-success">
                            <h6><i class="fas fa-star me-2"></i>أفضل مورد</h6>
                            <p class="mb-1"><strong>{{ $bestPerformer['supplier']->name }}</strong> يحقق أعلى قيمة مشتريات</p>
                            <small>يُنصح بتعزيز العلاقة وزيادة حجم التعاملات</small>
                        </div>
                    </div>
                    @endif
                    
                    @if($worstPaymentRate && $worstPaymentRate['payment_rate'] < 80)
                    <div class="col-md-4 mb-3">
                        <div class="alert alert-warning">
                            <h6><i class="fas fa-exclamation-triangle me-2"></i>تحسين الدفع</h6>
                            <p class="mb-1"><strong>{{ $worstPaymentRate['supplier']->name }}</strong> لديه معدل دفع منخفض</p>
                            <small>يُنصح بمراجعة شروط الدفع والمتابعة</small>
                        </div>
                    </div>
                    @endif
                    
                    @if($highestBalance && $highestBalance['current_balance'] > 10000)
                    <div class="col-md-4 mb-3">
                        <div class="alert alert-info">
                            <h6><i class="fas fa-money-bill-wave me-2"></i>إدارة الديون</h6>
                            <p class="mb-1"><strong>{{ $highestBalance['supplier']->name }}</strong> لديه أعلى رصيد مستحق</p>
                            <small>يُنصح بوضع خطة سداد أو مراجعة حد الائتمان</small>
                        </div>
                    </div>
                    @endif
                </div>
            </div>
        </div>
    </div>
</div>
@endif

<!-- مودال اختيار الموردين -->
<div class="modal fade" id="supplierSelectionModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <form method="GET" action="{{ route('suppliers.analytics.comparison') }}">
                <div class="modal-header">
                    <h5 class="modal-title">اختيار الموردين للمقارنة</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <input type="hidden" name="date_from" value="{{ $dateFrom }}">
                    <input type="hidden" name="date_to" value="{{ $dateTo }}">
                    
                    <div class="mb-3">
                        <label class="form-label">اختر الموردين (يمكن اختيار حتى 10 موردين)</label>
                        <div class="row">
                            @foreach($allSuppliers as $supplier)
                            <div class="col-md-6 mb-2">
                                <div class="form-check">
                                    <input class="form-check-input supplier-checkbox" type="checkbox" 
                                           name="supplier_ids[]" value="{{ $supplier->id }}" 
                                           id="supplier_{{ $supplier->id }}"
                                           {{ in_array($supplier->id, $supplierIds) ? 'checked' : '' }}>
                                    <label class="form-check-label" for="supplier_{{ $supplier->id }}">
                                        {{ $supplier->name }}
                                        @if($supplier->current_balance > 0)
                                            <small class="text-muted">(رصيد: {{ number_format($supplier->current_balance, 0) }})</small>
                                        @endif
                                    </label>
                                </div>
                            </div>
                            @endforeach
                        </div>
                    </div>
                    
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle me-2"></i>
                        يُنصح باختيار 3-5 موردين للحصول على أفضل مقارنة
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                    <button type="submit" class="btn btn-primary">تطبيق المقارنة</button>
                </div>
            </form>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
// تهيئة الدوائر الدائرية للتقييم
document.querySelectorAll('.circular-progress-small').forEach(element => {
    const score = element.dataset.score;
    const circle = element.querySelector('.progress-ring__circle-small');
    const radius = circle.r.baseVal.value;
    const circumference = radius * 2 * Math.PI;
    
    circle.style.strokeDasharray = `${circumference} ${circumference}`;
    circle.style.strokeDashoffset = circumference;
    
    const offset = circumference - score / 100 * circumference;
    circle.style.strokeDashoffset = offset;
});

@if(count($comparison) > 0)
// رسم مقارنة الأداء
const comparisonData = @json($comparison->values());
const supplierNames = comparisonData.map(item => item.supplier.name);
const purchaseAmounts = comparisonData.map(item => item.total_amount);
const paymentRates = comparisonData.map(item => item.payment_rate);

// رسم مقارنة المشتريات
const performanceCtx = document.getElementById('performanceComparisonChart').getContext('2d');
new Chart(performanceCtx, {
    type: 'bar',
    data: {
        labels: supplierNames,
        datasets: [{
            label: 'إجمالي المشتريات (ر.س)',
            data: purchaseAmounts,
            backgroundColor: '#007bff',
            borderColor: '#0056b3',
            borderWidth: 1
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        scales: {
            y: {
                beginAtZero: true,
                ticks: {
                    callback: function(value) {
                        return new Intl.NumberFormat('ar-SA').format(value);
                    }
                }
            }
        },
        plugins: {
            legend: {
                display: false
            }
        }
    }
});

// رسم توزيع المشتريات
const distributionCtx = document.getElementById('purchaseDistributionChart').getContext('2d');
new Chart(distributionCtx, {
    type: 'doughnut',
    data: {
        labels: supplierNames,
        datasets: [{
            data: purchaseAmounts,
            backgroundColor: [
                '#007bff', '#28a745', '#ffc107', '#dc3545', '#6f42c1',
                '#fd7e14', '#20c997', '#e83e8c', '#6c757d', '#17a2b8'
            ]
        }]
    },
    options: {
        responsive: true,
        plugins: {
            legend: {
                position: 'bottom'
            }
        }
    }
});

// رسم معدلات الدفع
const paymentRatesCtx = document.getElementById('paymentRatesChart').getContext('2d');
new Chart(paymentRatesCtx, {
    type: 'line',
    data: {
        labels: supplierNames,
        datasets: [{
            label: 'معدل الدفع (%)',
            data: paymentRates,
            borderColor: '#28a745',
            backgroundColor: 'rgba(40, 167, 69, 0.1)',
            tension: 0.4
        }]
    },
    options: {
        responsive: true,
        scales: {
            y: {
                beginAtZero: true,
                max: 100
            }
        }
    }
});
@endif

// تحديد عدد الموردين المحددين
document.addEventListener('change', function(e) {
    if (e.target.classList.contains('supplier-checkbox')) {
        const checkedBoxes = document.querySelectorAll('.supplier-checkbox:checked');
        if (checkedBoxes.length > 10) {
            e.target.checked = false;
            alert('يمكن اختيار حد أقصى 10 موردين للمقارنة');
        }
    }
});

function exportComparison(format) {
    const params = new URLSearchParams(window.location.search);
    params.set('format', format);
    
    window.open(`{{ route('suppliers.analytics.export') }}?type=comparison&${params}`, '_blank');
}
</script>
@endpush

@push('styles')
<style>
.circular-progress-small {
    position: relative;
    display: inline-block;
}

.progress-ring-small {
    transform: rotate(-90deg);
}

.progress-ring__circle-small {
    transition: stroke-dashoffset 0.35s;
    transform: rotate(-90deg);
    transform-origin: 50% 50%;
}

.progress-text-small {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    font-size: 10px;
    font-weight: bold;
}

.progress {
    background-color: #e9ecef;
}

.table td {
    vertical-align: middle;
}

.badge {
    font-size: 0.75rem;
}

.card-tools {
    display: flex;
    align-items: center;
}

@media (max-width: 768px) {
    .table-responsive {
        font-size: 0.875rem;
    }
}
</style>
@endpush