@extends('layouts.app')

@section('title', 'تقارير المطعم')
@section('page-title', 'تقارير المطعم')

@section('page-actions')
<div class="d-flex gap-2">
    <button type="button" class="btn btn-success" onclick="exportReport()">
        <i class="fas fa-download me-2"></i>
        تصدير التقرير
    </button>
    <button type="button" class="btn btn-info" onclick="printReport()">
        <i class="fas fa-print me-2"></i>
        طباعة
    </button>
    <a href="{{ route('invoices.create') }}" class="btn btn-primary">
        <i class="fas fa-cash-register me-2"></i>
        نقطة البيع
    </a>
</div>
@endsection

@section('content')
<div class="row mb-4">
    <div class="col-12">
        <div class="card filter-card">
            <div class="card-body">
                <form method="GET" action="{{ route('restaurant.reports.index') }}" class="row align-items-end">
                    <div class="col-md-3">
                        <label class="form-label">من تاريخ</label>
                        <input type="date" name="date_from" class="form-control" value="{{ $dateFrom }}">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">إلى تاريخ</label>
                        <input type="date" name="date_to" class="form-control" value="{{ $dateTo }}">
                    </div>
                    <div class="col-md-3">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-search me-2"></i>
                            عرض التقرير
                        </button>
                    </div>
                    <div class="col-md-3">
                        <div class="quick-dates">
                            <button type="button" class="btn btn-outline-secondary btn-sm me-1" onclick="setDateRange('today')">اليوم</button>
                            <button type="button" class="btn btn-outline-secondary btn-sm me-1" onclick="setDateRange('week')">الأسبوع</button>
                            <button type="button" class="btn btn-outline-secondary btn-sm" onclick="setDateRange('month')">الشهر</button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- البطاقات الإحصائية -->
<div class="row mb-4">
    <div class="col-lg-3 col-md-6">
        <div class="stats-card sales">
            <div class="stats-icon">
                <i class="fas fa-money-bill-wave"></i>
            </div>
            <div class="stats-content">
                <h3>{{ number_format($dailyStats->sum('total_sales'), 2) }} ر.س</h3>
                <p>إجمالي المبيعات</p>
                <span class="stats-change positive">
                    <i class="fas fa-arrow-up"></i>
                    +12.5%
                </span>
            </div>
        </div>
    </div>
    <div class="col-lg-3 col-md-6">
        <div class="stats-card orders">
            <div class="stats-icon">
                <i class="fas fa-shopping-cart"></i>
            </div>
            <div class="stats-content">
                <h3>{{ $dailyStats->sum('total_orders') }}</h3>
                <p>إجمالي الطلبات</p>
                <span class="stats-change positive">
                    <i class="fas fa-arrow-up"></i>
                    +8.3%
                </span>
            </div>
        </div>
    </div>
    <div class="col-lg-3 col-md-6">
        <div class="stats-card average">
            <div class="stats-icon">
                <i class="fas fa-chart-line"></i>
            </div>
            <div class="stats-content">
                <h3>{{ number_format($dailyStats->avg('average_order'), 2) }} ر.س</h3>
                <p>متوسط الطلب</p>
                <span class="stats-change negative">
                    <i class="fas fa-arrow-down"></i>
                    -2.1%
                </span>
            </div>
        </div>
    </div>
    <div class="col-lg-3 col-md-6">
        <div class="stats-card items">
            <div class="stats-icon">
                <i class="fas fa-utensils"></i>
            </div>
            <div class="stats-content">
                <h3>{{ $topProducts->sum('total_quantity') }}</h3>
                <p>العناصر المباعة</p>
                <span class="stats-change positive">
                    <i class="fas fa-arrow-up"></i>
                    +15.7%
                </span>
            </div>
        </div>
    </div>
</div>

<div class="row">
    <!-- أكثر المنتجات مبيعاً -->
    <div class="col-lg-6">
        <div class="card chart-card">
            <div class="card-header">
                <h5 class="card-title">
                    <i class="fas fa-trophy me-2"></i>
                    أكثر المنتجات مبيعاً
                </h5>
            </div>
            <div class="card-body">
                <div class="products-list">
                    @foreach($topProducts->take(10) as $index => $product)
                    <div class="product-item">
                        <div class="product-rank">
                            <span class="rank-number">#{{ $index + 1 }}</span>
                        </div>
                        <div class="product-info">
                            <h6>{{ $product->name }}</h6>
                            <span class="category-badge category-{{ $product->category }}">
                                @switch($product->category)
                                    @case('meals')
                                        الوجبات الرئيسية
                                        @break
                                    @case('appetizers')
                                        المقبلات
                                        @break
                                    @case('beverages')
                                        المشروبات
                                        @break
                                    @case('desserts')
                                        الحلويات
                                        @break
                                    @default
                                        {{ $product->category }}
                                @endswitch
                            </span>
                        </div>
                        <div class="product-stats">
                            <div class="stat">
                                <span class="value">{{ $product->total_quantity }}</span>
                                <span class="label">كمية</span>
                            </div>
                            <div class="stat">
                                <span class="value">{{ number_format($product->total_revenue, 0) }}</span>
                                <span class="label">ر.س</span>
                            </div>
                        </div>
                    </div>
                    @endforeach
                </div>
            </div>
        </div>
    </div>

    <!-- إحصائيات الفئات -->
    <div class="col-lg-6">
        <div class="card chart-card">
            <div class="card-header">
                <h5 class="card-title">
                    <i class="fas fa-chart-pie me-2"></i>
                    مبيعات حسب الفئة
                </h5>
            </div>
            <div class="card-body">
                <div class="category-stats">
                    @foreach($categoryStats as $category)
                    <div class="category-item">
                        <div class="category-header">
                            <span class="category-name">{{ $category->category_name }}</span>
                            <span class="category-revenue">{{ number_format($category->total_revenue, 0) }} ر.س</span>
                        </div>
                        <div class="category-progress">
                            <div class="progress">
                                @php
                                    $colorClass = match($category->category) {
                                        'meals' => 'bg-danger',
                                        'appetizers' => 'bg-success',
                                        'beverages' => 'bg-primary',
                                        'desserts' => 'bg-warning',
                                        default => 'bg-secondary'
                                    };
                                @endphp
                                <div class="progress-bar {{ $colorClass }}"
                                     style="width: {{ ($category->total_revenue / $categoryStats->max('total_revenue')) * 100 }}%">
                                </div>
                            </div>
                        </div>
                        <div class="category-details">
                            <small>{{ $category->total_quantity }} عنصر مباع</small>
                            <small>{{ $category->item_count }} طلب</small>
                        </div>
                    </div>
                    @endforeach
                </div>
            </div>
        </div>
    </div>
</div>

<!-- المبيعات اليومية -->
<div class="row mt-4">
    <div class="col-12">
        <div class="card chart-card">
            <div class="card-header">
                <h5 class="card-title">
                    <i class="fas fa-chart-area me-2"></i>
                    المبيعات اليومية
                </h5>
            </div>
            <div class="card-body">
                <canvas id="dailySalesChart" height="100"></canvas>
            </div>
        </div>
    </div>
</div>

<!-- إحصائيات الساعات -->
<div class="row mt-4">
    <div class="col-lg-8">
        <div class="card chart-card">
            <div class="card-header">
                <h5 class="card-title">
                    <i class="fas fa-clock me-2"></i>
                    توزيع الطلبات حسب الساعة
                </h5>
            </div>
            <div class="card-body">
                <canvas id="hourlyChart" height="80"></canvas>
            </div>
        </div>
    </div>

    <!-- أنواع الطلبات -->
    <div class="col-lg-4">
        <div class="card chart-card">
            <div class="card-header">
                <h5 class="card-title">
                    <i class="fas fa-list me-2"></i>
                    أنواع الطلبات
                </h5>
            </div>
            <div class="card-body">
                <div class="order-types">
                    @foreach($orderTypeStats as $orderType)
                    <div class="order-type-item">
                        <div class="order-type-icon">
                            @php
                                $iconClass = match($orderType['type']) {
                                    'dine_in' => 'utensils',
                                    'takeaway' => 'shopping-bag',
                                    'delivery' => 'truck',
                                    default => 'shopping-cart'
                                };
                            @endphp
                            <i class="fas fa-{{ $iconClass }}"></i>
                        </div>
                        <div class="order-type-info">
                            <h6>{{ $orderType['name'] }}</h6>
                            <p>{{ number_format($orderType['count']) }} طلب</p>
                            <small>{{ number_format($orderType['revenue'], 0) }} ر.س</small>
                        </div>
                        <div class="order-type-percentage">
                            @php
                                $totalRevenue = $orderTypeStats->sum('revenue');
                                $percentage = $totalRevenue > 0 ? ($orderType['revenue'] / $totalRevenue) * 100 : 0;
                            @endphp
                            {{ number_format($percentage, 1) }}%
                        </div>
                    </div>
                    @endforeach
                </div>
            </div>
        </div>
    </div>
</div>

@endsection

@push('styles')
<style>
/* تصميم التقارير */
.filter-card {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 15px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
}

.filter-card .form-label {
    color: white;
    font-weight: 600;
}

.filter-card .form-control {
    border-radius: 8px;
    border: 2px solid rgba(255,255,255,0.2);
    background: rgba(255,255,255,0.9);
}

.stats-card {
    background: white;
    border-radius: 15px;
    padding: 20px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.08);
    border-left: 4px solid;
    margin-bottom: 20px;
    transition: transform 0.3s ease;
}

.stats-card:hover {
    transform: translateY(-2px);
}

.stats-card.sales { border-left-color: #00b894; }
.stats-card.orders { border-left-color: #0984e3; }
.stats-card.average { border-left-color: #fdcb6e; }
.stats-card.items { border-left-color: #e17055; }

.stats-card .stats-icon {
    width: 60px;
    height: 60px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    float: left;
    margin-right: 15px;
    font-size: 1.5rem;
    color: white;
}

.stats-card.sales .stats-icon { background: #00b894; }
.stats-card.orders .stats-icon { background: #0984e3; }
.stats-card.average .stats-icon { background: #fdcb6e; }
.stats-card.items .stats-icon { background: #e17055; }

.stats-content h3 {
    margin: 0;
    font-size: 1.8rem;
    font-weight: 700;
    color: #2d3436;
}

.stats-content p {
    margin: 5px 0;
    color: #636e72;
    font-weight: 500;
}

.stats-change {
    font-size: 0.9rem;
    font-weight: 600;
}

.stats-change.positive { color: #00b894; }
.stats-change.negative { color: #d63031; }

.chart-card {
    border-radius: 15px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.08);
    border: none;
}

.chart-card .card-header {
    background: white;
    border-bottom: 1px solid #eee;
    border-radius: 15px 15px 0 0;
}

.products-list {
    max-height: 400px;
    overflow-y: auto;
}

.product-item {
    display: flex;
    align-items: center;
    padding: 15px 0;
    border-bottom: 1px solid #eee;
}

.product-item:last-child {
    border-bottom: none;
}

.product-rank .rank-number {
    display: inline-block;
    width: 30px;
    height: 30px;
    line-height: 30px;
    text-align: center;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 50%;
    font-weight: 700;
    font-size: 0.9rem;
}

.product-info {
    flex: 1;
    margin-left: 15px;
}

.product-info h6 {
    margin: 0;
    font-weight: 600;
    color: #2d3436;
}

.category-badge {
    font-size: 0.75rem;
    padding: 2px 8px;
    border-radius: 12px;
    color: white;
    font-weight: 500;
}

.category-badge.category-meals { background: #e17055; }
.category-badge.category-appetizers { background: #00b894; }
.category-badge.category-beverages { background: #0984e3; }
.category-badge.category-desserts { background: #fd79a8; }

.product-stats {
    display: flex;
    gap: 15px;
}

.product-stats .stat {
    text-align: center;
}

.product-stats .value {
    display: block;
    font-weight: 700;
    color: #2d3436;
}

.product-stats .label {
    font-size: 0.8rem;
    color: #636e72;
}

.category-stats .category-item {
    margin-bottom: 20px;
}

.category-header {
    display: flex;
    justify-content: space-between;
    margin-bottom: 8px;
}

.category-name {
    font-weight: 600;
    color: #2d3436;
}

.category-revenue {
    font-weight: 700;
    color: #00b894;
}

.category-progress .progress {
    height: 8px;
    border-radius: 4px;
}

.category-details {
    display: flex;
    justify-content: space-between;
    margin-top: 5px;
}

.category-details small {
    color: #636e72;
}

.order-types .order-type-item {
    display: flex;
    align-items: center;
    padding: 15px 0;
    border-bottom: 1px solid #eee;
}

.order-type-icon {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-right: 15px;
}

.order-type-info {
    flex: 1;
}

.order-type-info h6 {
    margin: 0;
    font-weight: 600;
    color: #2d3436;
}

.order-type-info p {
    margin: 2px 0;
    color: #636e72;
}

.order-type-percentage {
    font-weight: 700;
    color: #00b894;
    font-size: 1.1rem;
}

.quick-dates .btn {
    padding: 4px 8px;
    font-size: 0.8rem;
}

/* Scrollbar styling */
.products-list::-webkit-scrollbar {
    width: 6px;
}

.products-list::-webkit-scrollbar-track {
    background: #f1f1f1;
    border-radius: 10px;
}

.products-list::-webkit-scrollbar-thumb {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border-radius: 10px;
}
</style>
@endpush

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
// بيانات الرسوم البيانية
const dailyData = @json($dailyStats);
const hourlyData = @json($hourlyStats);

// رسم المبيعات اليومية
const dailySalesCtx = document.getElementById('dailySalesChart').getContext('2d');
new Chart(dailySalesCtx, {
    type: 'line',
    data: {
        labels: dailyData.map(d => new Date(d.date).toLocaleDateString('ar-SA')),
        datasets: [{
            label: 'المبيعات (ر.س)',
            data: dailyData.map(d => d.total_sales),
            borderColor: '#667eea',
            backgroundColor: 'rgba(102, 126, 234, 0.1)',
            fill: true,
            tension: 0.4
        }, {
            label: 'عدد الطلبات',
            data: dailyData.map(d => d.total_orders),
            borderColor: '#00b894',
            backgroundColor: 'rgba(0, 184, 148, 0.1)',
            fill: false,
            yAxisID: 'y1'
        }]
    },
    options: {
        responsive: true,
        plugins: {
            legend: {
                position: 'top',
            }
        },
        scales: {
            y: {
                type: 'linear',
                display: true,
                position: 'left',
            },
            y1: {
                type: 'linear',
                display: true,
                position: 'right',
                grid: {
                    drawOnChartArea: false,
                },
            }
        }
    }
});

// رسم توزيع الساعات
const hourlyCtx = document.getElementById('hourlyChart').getContext('2d');
new Chart(hourlyCtx, {
    type: 'bar',
    data: {
        labels: hourlyData.map(d => d.hour_display),
        datasets: [{
            label: 'عدد الطلبات',
            data: hourlyData.map(d => d.order_count),
            backgroundColor: 'rgba(102, 126, 234, 0.8)',
            borderColor: '#667eea',
            borderWidth: 1
        }]
    },
    options: {
        responsive: true,
        plugins: {
            legend: {
                display: false
            }
        }
    }
});

// دوال التواريخ السريعة
function setDateRange(range) {
    const today = new Date();
    let dateFrom, dateTo;
    
    switch(range) {
        case 'today':
            dateFrom = dateTo = today.toISOString().split('T')[0];
            break;
        case 'week':
            dateFrom = new Date(today.setDate(today.getDate() - 6)).toISOString().split('T')[0];
            dateTo = new Date().toISOString().split('T')[0];
            break;
        case 'month':
            dateFrom = new Date(today.getFullYear(), today.getMonth(), 1).toISOString().split('T')[0];
            dateTo = new Date().toISOString().split('T')[0];
            break;
    }
    
    document.querySelector('input[name="date_from"]').value = dateFrom;
    document.querySelector('input[name="date_to"]').value = dateTo;
}

// تصدير التقرير
function exportReport() {
    const dateFrom = document.querySelector('input[name="date_from"]').value;
    const dateTo = document.querySelector('input[name="date_to"]').value;
    
    window.open(`{{ route('restaurant.reports.export') }}?date_from=${dateFrom}&date_to=${dateTo}`, '_blank');
}

// طباعة التقرير
function printReport() {
    window.print();
}
</script>
@endpush