@extends('layouts.app')

@section('title', 'إنشاء أمر شراء جديد')
@section('page-title', 'إنشاء أمر شراء جديد')

@section('page-actions')
    <a href="{{ route('purchase-orders.index') }}" class="btn btn-secondary">
        <i class="fas fa-arrow-right me-2"></i>
        العودة للقائمة
    </a>
@endsection

@push('styles')
<style>
    .product-row {
        border: 1px solid #dee2e6;
        border-radius: 8px;
        padding: 15px;
        margin-bottom: 15px;
        background-color: #f8f9fa;
    }
    .remove-item {
        position: absolute;
        top: 10px;
        left: 10px;
    }
</style>
@endpush

@section('content')
<form method="POST" action="{{ route('purchase-orders.store') }}" id="purchaseOrderForm">
    @csrf
    
    <div class="row">
        <!-- بيانات أمر الشراء -->
        <div class="col-md-8">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">
                        <i class="fas fa-info-circle me-2"></i>
                        بيانات أمر الشراء
                    </h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="supplier_id" class="form-label">المورد <span class="text-danger">*</span></label>
                                <select class="form-control @error('supplier_id') is-invalid @enderror" 
                                        id="supplier_id" name="supplier_id" required>
                                    <option value="">اختر المورد</option>
                                    @foreach($suppliers as $supplier)
                                        <option value="{{ $supplier->id }}" {{ old('supplier_id') == $supplier->id ? 'selected' : '' }}>
                                            {{ $supplier->name }}
                                        </option>
                                    @endforeach
                                </select>
                                @error('supplier_id')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="order_date" class="form-label">تاريخ الأمر <span class="text-danger">*</span></label>
                                <input type="date" class="form-control @error('order_date') is-invalid @enderror" 
                                       id="order_date" name="order_date" value="{{ old('order_date', date('Y-m-d')) }}" required>
                                @error('order_date')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="expected_delivery_date" class="form-label">تاريخ التسليم المتوقع</label>
                                <input type="date" class="form-control @error('expected_delivery_date') is-invalid @enderror" 
                                       id="expected_delivery_date" name="expected_delivery_date" value="{{ old('expected_delivery_date') }}">
                                @error('expected_delivery_date')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="notes" class="form-label">ملاحظات</label>
                        <textarea class="form-control @error('notes') is-invalid @enderror" 
                                  id="notes" name="notes" rows="3">{{ old('notes') }}</textarea>
                        @error('notes')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                </div>
            </div>
            
            <!-- عناصر أمر الشراء -->
            <div class="card mt-4">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">
                        <i class="fas fa-list me-2"></i>
                        عناصر أمر الشراء
                    </h5>
                    <button type="button" class="btn btn-primary btn-sm" id="addItem">
                        <i class="fas fa-plus me-2"></i>
                        إضافة عنصر
                    </button>
                </div>
                <div class="card-body">
                    <div id="itemsContainer">
                        <!-- سيتم إضافة العناصر هنا -->
                    </div>
                    
                    @error('items')
                        <div class="alert alert-danger">{{ $message }}</div>
                    @enderror
                </div>
            </div>
        </div>
        
        <!-- ملخص أمر الشراء -->
        <div class="col-md-4">
            <div class="card sticky-top">
                <div class="card-header">
                    <h5 class="mb-0">
                        <i class="fas fa-calculator me-2"></i>
                        ملخص أمر الشراء
                    </h5>
                </div>
                <div class="card-body">
                    <table class="table table-borderless">
                        <tr>
                            <td>المجموع الفرعي:</td>
                            <td class="text-end" id="subtotal">0.00 ر.س</td>
                        </tr>
                        <tr>
                            <td>الضريبة:</td>
                            <td class="text-end" id="tax">0.00 ر.س</td>
                        </tr>
                        <tr class="table-primary">
                            <td><strong>المجموع الكلي:</strong></td>
                            <td class="text-end"><strong id="total">0.00 ر.س</strong></td>
                        </tr>
                    </table>
                    
                    <hr>
                    
                    <div class="d-grid gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i>
                            حفظ أمر الشراء
                        </button>
                        
                        <a href="{{ route('purchase-orders.index') }}" class="btn btn-secondary">
                            <i class="fas fa-times me-2"></i>
                            إلغاء
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</form>

<!-- قالب عنصر المنتج -->
<template id="itemTemplate">
    <div class="product-row position-relative" data-index="__INDEX__">
        <button type="button" class="btn btn-danger btn-sm remove-item">
            <i class="fas fa-times"></i>
        </button>
        
        <div class="row">
            <div class="col-md-6">
                <div class="mb-3">
                    <label class="form-label">المنتج <span class="text-danger">*</span></label>
                    <select class="form-control product-select" name="items[__INDEX__][product_id]" required>
                        <option value="">اختر المنتج</option>
                        @foreach($products as $product)
                            <option value="{{ $product->id }}" data-price="{{ $product->price }}" data-tax="{{ $product->tax_rate }}">
                                {{ $product->name }} ({{ $product->sku }})
                            </option>
                        @endforeach
                    </select>
                </div>
            </div>
            
            <div class="col-md-3">
                <div class="mb-3">
                    <label class="form-label">الكمية <span class="text-danger">*</span></label>
                    <input type="number" class="form-control quantity-input" 
                           name="items[__INDEX__][quantity]" min="1" value="1" required>
                </div>
            </div>
            
            <div class="col-md-3">
                <div class="mb-3">
                    <label class="form-label">سعر الوحدة <span class="text-danger">*</span></label>
                    <input type="number" step="0.01" class="form-control price-input" 
                           name="items[__INDEX__][unit_price]" min="0" required>
                </div>
            </div>
        </div>
        
        <div class="row">
            <div class="col-md-6">
                <div class="mb-3">
                    <label class="form-label">معدل الضريبة (%)</label>
                    <input type="number" step="0.01" class="form-control tax-input" 
                           name="items[__INDEX__][tax_rate]" min="0" max="100" value="15">
                </div>
            </div>
            
            <div class="col-md-6">
                <div class="mb-3">
                    <label class="form-label">المجموع</label>
                    <input type="text" class="form-control item-total" readonly>
                </div>
            </div>
        </div>
    </div>
</template>
@endsection

@push('scripts')
<script>
let itemIndex = 0;

$(document).ready(function() {
    // إضافة عنصر جديد
    $('#addItem').click(function() {
        addNewItem();
    });
    
    // إضافة أول عنصر
    addNewItem();
    
    // حذف عنصر
    $(document).on('click', '.remove-item', function() {
        $(this).closest('.product-row').remove();
        calculateTotals();
    });
    
    // تحديث الأسعار عند تغيير المنتج
    $(document).on('change', '.product-select', function() {
        const option = $(this).find('option:selected');
        const row = $(this).closest('.product-row');
        
        row.find('.price-input').val(option.data('price') || 0);
        row.find('.tax-input').val(option.data('tax') || 15);
        
        calculateItemTotal(row);
    });
    
    // تحديث المجاميع عند تغيير الكمية أو السعر
    $(document).on('input', '.quantity-input, .price-input, .tax-input', function() {
        const row = $(this).closest('.product-row');
        calculateItemTotal(row);
    });
});

function addNewItem() {
    const template = $('#itemTemplate').html();
    const newItem = template.replace(/__INDEX__/g, itemIndex);
    $('#itemsContainer').append(newItem);
    itemIndex++;
}

function calculateItemTotal(row) {
    const quantity = parseFloat(row.find('.quantity-input').val()) || 0;
    const price = parseFloat(row.find('.price-input').val()) || 0;
    const taxRate = parseFloat(row.find('.tax-input').val()) || 0;
    
    const subtotal = quantity * price;
    const tax = subtotal * (taxRate / 100);
    const total = subtotal + tax;
    
    row.find('.item-total').val(total.toFixed(2) + ' ر.س');
    
    calculateTotals();
}

function calculateTotals() {
    let subtotal = 0;
    let tax = 0;
    
    $('.product-row').each(function() {
        const quantity = parseFloat($(this).find('.quantity-input').val()) || 0;
        const price = parseFloat($(this).find('.price-input').val()) || 0;
        const taxRate = parseFloat($(this).find('.tax-input').val()) || 0;
        
        const itemSubtotal = quantity * price;
        const itemTax = itemSubtotal * (taxRate / 100);
        
        subtotal += itemSubtotal;
        tax += itemTax;
    });
    
    const total = subtotal + tax;
    
    $('#subtotal').text(subtotal.toFixed(2) + ' ر.س');
    $('#tax').text(tax.toFixed(2) + ' ر.س');
    $('#total').text(total.toFixed(2) + ' ر.س');
}
</script>
@endpush