@extends('layouts.app')

@section('title', 'تعديل المنتج: ' . $product->name)
@section('page-title', 'تعديل المنتج')

@push('styles')
<style>
    .simple-form {
        background: #f8f9fa;
        min-height: 100vh;
        padding: 20px 0;
    }
    
    .form-container {
        background: white;
        border-radius: 12px;
        box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        margin: 0 auto;
        max-width: 1000px;
        overflow: hidden;
    }
    
    .form-header {
        background: #2563eb;
        color: white;
        padding: 20px 30px;
        text-align: center;
    }
    
    .form-header h1 {
        font-size: 1.8rem;
        font-weight: 600;
        margin: 0;
    }
    
    .form-body {
        padding: 30px;
    }
    
    .simple-card {
        background: white;
        border: 1px solid #e5e7eb;
        border-radius: 8px;
        margin-bottom: 20px;
        overflow: hidden;
    }
    
    .card-header {
        background: #f9fafb;
        border-bottom: 1px solid #e5e7eb;
        padding: 15px 20px;
    }
    
    .card-title {
        font-size: 1.1rem;
        font-weight: 600;
        color: #374151;
        margin: 0;
    }
    
    .card-body {
        padding: 20px;
    }
    
    .form-group {
        margin-bottom: 20px;
    }
    
    .form-label {
        display: block;
        font-weight: 500;
        color: #374151;
        margin-bottom: 5px;
        font-size: 0.9rem;
    }
    
    .form-control, .form-select {
        width: 100%;
        padding: 8px 12px;
        border: 1px solid #d1d5db;
        border-radius: 6px;
        font-size: 0.9rem;
        transition: border-color 0.2s;
    }
    
    .form-control:focus, .form-select:focus {
        outline: none;
        border-color: #2563eb;
        box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
    }
    
    .input-group {
        display: flex;
    }
    
    .input-group .form-control {
        border-top-left-radius: 0;
        border-bottom-left-radius: 0;
        border-right: none;
    }
    
    .input-group-text {
        background: #f3f4f6;
        border: 1px solid #d1d5db;
        border-top-right-radius: 6px;
        border-bottom-right-radius: 6px;
        padding: 8px 12px;
        font-size: 0.9rem;
        color: #6b7280;
    }
    
    .btn-primary {
        background: #2563eb;
        border: 1px solid #2563eb;
        color: white;
        padding: 10px 20px;
        border-radius: 6px;
        font-weight: 500;
        cursor: pointer;
        transition: background-color 0.2s;
    }
    
    .btn-primary:hover {
        background: #1d4ed8;
        border-color: #1d4ed8;
    }
    
    .btn-secondary {
        background: #6b7280;
        border: 1px solid #6b7280;
        color: white;
        padding: 8px 16px;
        border-radius: 6px;
        text-decoration: none;
        display: inline-block;
        font-weight: 500;
        transition: background-color 0.2s;
    }
    
    .btn-secondary:hover {
        background: #4b5563;
        color: white;
        text-decoration: none;
    }
    
    .checkbox-group {
        display: flex;
        align-items: center;
        padding: 10px 0;
    }
    
    .checkbox-group input[type="checkbox"] {
        margin-left: 8px;
        width: 16px;
        height: 16px;
    }
    
    .checkbox-group label {
        margin: 0;
        cursor: pointer;
    }
    
    .alert {
        padding: 12px 16px;
        border-radius: 6px;
        margin-bottom: 20px;
    }
    
    .alert-info {
        background: #eff6ff;
        border: 1px solid #bfdbfe;
        color: #1e40af;
    }
    
    .alert-warning {
        background: #fffbeb;
        border: 1px solid #fed7aa;
        color: #92400e;
    }
    
    .text-danger {
        color: #dc2626;
    }
    
    .text-muted {
        color: #6b7280;
        font-size: 0.85rem;
    }
    
    .invalid-feedback {
        color: #dc2626;
        font-size: 0.8rem;
        margin-top: 4px;
    }
    
    .is-invalid {
        border-color: #dc2626;
    }
    
    .row {
        display: flex;
        flex-wrap: wrap;
        margin: -10px;
    }
    
    .col-md-6 {
        flex: 0 0 50%;
        padding: 10px;
    }
    
    .col-md-4 {
        flex: 0 0 33.333333%;
        padding: 10px;
    }
    
    .col-md-12 {
        flex: 0 0 100%;
        padding: 10px;
    }
    
    @media (max-width: 768px) {
        .col-md-6, .col-md-4 {
            flex: 0 0 100%;
        }
        
        .form-container {
            margin: 10px;
        }
        
        .form-body {
            padding: 20px;
        }
    }
</style>
@endpush

@section('page-actions')
<div style="display: flex; gap: 10px;">
    <a href="{{ route('products.show', $product) }}" class="btn-secondary">
        <i class="fas fa-eye"></i> عرض المنتج
    </a>
    <a href="{{ route('products.index') }}" class="btn-secondary">
        <i class="fas fa-arrow-left"></i> العودة للقائمة
    </a>
</div>
@endsection

@section('content')
<div class="simple-form">
    <div class="container-fluid">
        <div class="form-container">
            <div class="form-header">
                <h1>تعديل المنتج</h1>
            </div>
            
            <div class="form-body">
                <form method="POST" action="{{ route('products.update', $product) }}" id="product-form">
                    @csrf
                    @method('PUT')
                    
                    <!-- المعلومات الأساسية -->
                    <div class="simple-card">
                        <div class="card-header">
                            <h5 class="card-title">المعلومات الأساسية</h5>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-12">
                                    <div class="form-group">
                                        <label class="form-label">اسم المنتج <span class="text-danger">*</span></label>
                                        <input type="text" name="name" class="form-control @error('name') is-invalid @enderror" 
                                               value="{{ old('name', $product->name) }}" required>
                                        @error('name')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label class="form-label">الفئة <span class="text-danger">*</span></label>
                                        <select name="category_id" class="form-select @error('category_id') is-invalid @enderror">
                                            <option value="">اختر الفئة</option>
                                            @php
                                                try {
                                                    $availableCategories = \App\Models\Category::all();
                                                } catch (\Exception $e) {
                                                    $availableCategories = collect();
                                                }
                                            @endphp
                                            @foreach($availableCategories as $category)
                                                <option value="{{ $category->id }}" 
                                                        {{ old('category_id', $product->category_id ?? '') == $category->id ? 'selected' : '' }}>
                                                    {{ $category->name }}
                                                </option>
                                            @endforeach
                                        </select>
                                        @error('category_id')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label class="form-label">النوع <span class="text-danger">*</span></label>
                                        <select name="type" class="form-select @error('type') is-invalid @enderror" 
                                                required onchange="toggleStockFields()">
                                            <option value="">اختر النوع</option>
                                            <option value="product" {{ old('type', $product->type) == 'product' ? 'selected' : '' }}>منتج</option>
                                            <option value="service" {{ old('type', $product->type) == 'service' ? 'selected' : '' }}>خدمة</option>
                                        </select>
                                        @error('type')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label class="form-label">رمز المنتج (SKU) <span class="text-danger">*</span></label>
                                        <input type="text" name="sku" class="form-control @error('sku') is-invalid @enderror" 
                                               value="{{ old('sku', $product->sku) }}" required>
                                        @error('sku')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label class="form-label">الباركود</label>
                                        <input type="text" name="barcode" class="form-control @error('barcode') is-invalid @enderror" 
                                               value="{{ old('barcode', $product->barcode) }}">
                                        @error('barcode')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- الأسعار -->
                    <div class="simple-card">
                        <div class="card-header">
                            <h5 class="card-title">الأسعار والضرائب</h5>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-4">
                                    <div class="form-group">
                                        <label class="form-label">السعر <span class="text-danger">*</span></label>
                                        <div class="input-group">
                                            <input type="number" name="price" class="form-control @error('price') is-invalid @enderror" 
                                                   value="{{ old('price', $product->price) }}" step="0.01" min="0" required 
                                                   onchange="calculatePriceWithTax()">
                                            <span class="input-group-text">ر.س</span>
                                        </div>
                                        @error('price')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                                
                                <div class="col-md-4">
                                    <div class="form-group">
                                        <label class="form-label">معدل الضريبة <span class="text-danger">*</span></label>
                                        <div class="input-group">
                                            <input type="number" name="tax_rate" class="form-control @error('tax_rate') is-invalid @enderror" 
                                                   value="{{ old('tax_rate', $product->tax_rate) }}" step="0.01" min="0" max="100" 
                                                   required onchange="calculatePriceWithTax()">
                                            <span class="input-group-text">%</span>
                                        </div>
                                        @error('tax_rate')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                                
                                <div class="col-md-4">
                                    <div class="form-group">
                                        <label class="form-label">السعر مع الضريبة</label>
                                        <div class="input-group">
                                            <input type="text" id="price-with-tax-display" class="form-control" readonly 
                                                   value="{{ number_format($product->price_with_tax, 2) }}">
                                            <span class="input-group-text">ر.س</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="checkbox-group">
                                        <input type="checkbox" name="is_taxable" id="is_taxable" value="1" 
                                               {{ old('is_taxable', $product->is_taxable) ? 'checked' : '' }} 
                                               onchange="toggleTaxFields()">
                                        <label for="is_taxable">خاضع للضريبة</label>
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label class="form-label">رمز تصنيف ZATCA</label>
                                        <input type="text" name="zatca_category_code" class="form-control @error('zatca_category_code') is-invalid @enderror" 
                                               value="{{ old('zatca_category_code', $product->zatca_category_code) }}">
                                        @error('zatca_category_code')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- إدارة المخزون -->
                    <div class="simple-card" id="stock-section">
                        <div class="card-header">
                            <h5 class="card-title">إدارة المخزون</h5>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-4">
                                    <div class="form-group">
                                        <label class="form-label">الوحدة <span class="text-danger">*</span></label>
                                        <input type="text" name="unit" class="form-control @error('unit') is-invalid @enderror" 
                                               value="{{ old('unit', $product->unit) }}" required>
                                        @error('unit')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                                
                                <div class="col-md-4" id="stock-quantity-field">
                                    <div class="form-group">
                                        <label class="form-label">الكمية الحالية <span class="text-danger">*</span></label>
                                        <input type="number" name="stock_quantity" class="form-control @error('stock_quantity') is-invalid @enderror" 
                                               value="{{ old('stock_quantity', $product->stock_quantity) }}" min="0" required>
                                        @error('stock_quantity')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                                
                                <div class="col-md-4" id="min-stock-field">
                                    <div class="form-group">
                                        <label class="form-label">الحد الأدنى للمخزون <span class="text-danger">*</span></label>
                                        <input type="number" name="min_stock" class="form-control @error('min_stock') is-invalid @enderror" 
                                               value="{{ old('min_stock', $product->min_stock) }}" min="0" required>
                                        @error('min_stock')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- الحالة -->
                    <div class="simple-card">
                        <div class="card-header">
                            <h5 class="card-title">حالة المنتج</h5>
                        </div>
                        <div class="card-body">
                            <div class="checkbox-group">
                                <input type="checkbox" name="is_active" id="is_active" value="1" 
                                       {{ old('is_active', $product->is_active) ? 'checked' : '' }}>
                                <label for="is_active">منتج نشط</label>
                            </div>
                        </div>
                    </div>
                    
                    <!-- تحذير -->
                    @if($product->invoiceItems->count() > 0)
                    <div class="alert alert-warning">
                        <strong>تنبيه:</strong> هذا المنتج مستخدم في {{ $product->invoiceItems->count() }} فاتورة. 
                        تعديل السعر أو النوع قد يؤثر على التقارير.
                    </div>
                    @endif
                    
                    <!-- أزرار الحفظ -->
                    <div style="text-align: center; padding: 20px 0;">
                        <button type="submit" class="btn-primary">
                            <i class="fas fa-save"></i> حفظ التعديلات
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
function toggleStockFields() {
    const type = document.querySelector('select[name="type"]').value;
    const stockSection = document.getElementById('stock-section');
    const stockQuantityField = document.getElementById('stock-quantity-field');
    const minStockField = document.getElementById('min-stock-field');
    const stockQuantityInput = document.querySelector('input[name="stock_quantity"]');
    const minStockInput = document.querySelector('input[name="min_stock"]');
    
    if (type === 'service') {
        stockQuantityField.style.display = 'none';
        minStockField.style.display = 'none';
        stockQuantityInput.required = false;
        minStockInput.required = false;
        stockQuantityInput.value = 0;
        minStockInput.value = 0;
    } else if (type === 'product') {
        stockQuantityField.style.display = 'block';
        minStockField.style.display = 'block';
        stockQuantityInput.required = true;
        minStockInput.required = true;
    }
}

function toggleTaxFields() {
    const isTaxable = document.querySelector('input[name="is_taxable"]').checked;
    const taxRateInput = document.querySelector('input[name="tax_rate"]');
    
    if (isTaxable) {
        taxRateInput.disabled = false;
        taxRateInput.required = true;
    } else {
        taxRateInput.disabled = true;
        taxRateInput.required = false;
        taxRateInput.value = 0;
    }
    
    calculatePriceWithTax();
}

function calculatePriceWithTax() {
    const price = parseFloat(document.querySelector('input[name="price"]').value) || 0;
    const taxRate = parseFloat(document.querySelector('input[name="tax_rate"]').value) || 0;
    const isTaxable = document.querySelector('input[name="is_taxable"]').checked;
    
    let priceWithTax = price;
    if (isTaxable) {
        priceWithTax = price * (1 + (taxRate / 100));
    }
    
    document.getElementById('price-with-tax-display').value = priceWithTax.toFixed(2);
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    toggleStockFields();
    toggleTaxFields();
    calculatePriceWithTax();
});
</script>
@endpush