@extends('layouts.app')

@section('title', 'مزامنة فواتير ZATCA')
@section('page-title', 'مزامنة الفواتير مع هيئة الزكاة والضريبة والجمارك')

@section('page-actions')
<div class="d-flex gap-3 flex-wrap">
    <button type="button" class="btn btn-primary btn-lg cairo-btn" onclick="syncAllInvoices()">
        <i class="bi bi-arrow-clockwise me-2"></i>
        مزامنة جميع الفواتير
    </button>
    <a href="{{ route('invoices.sync-export') }}" class="btn btn-success btn-lg cairo-btn">
        <i class="bi bi-download me-2"></i>
        تصدير التقرير
    </a>
    <a href="{{ route('invoices.index') }}" class="btn btn-outline-secondary btn-lg cairo-btn">
        <i class="bi bi-arrow-right me-2"></i>
        العودة للفواتير
    </a>
</div>
@endsection

@push('styles')
<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Cairo:wght@200;300;400;500;600;700;800;900&display=swap" rel="stylesheet">
<style>
:root {
    /* Professional Color Palette */
    --primary-color: #0F4C75;
    --primary-light: #E8F4FD;
    --primary-dark: #0A3D61;
    --secondary-color: #3282B8;
    --secondary-light: #EBF5FF;
    
    --success-color: #059669;
    --success-light: #ECFDF5;
    --success-dark: #047857;
    
    --warning-color: #D97706;
    --warning-light: #FFFBEB;
    --warning-dark: #B45309;
    
    --danger-color: #DC2626;
    --danger-light: #FEF2F2;
    --danger-dark: #B91C1C;
    
    --info-color: #0891B2;
    --info-light: #F0F9FF;
    --info-dark: #0E7490;
    
    /* Neutral Colors */
    --white: #FFFFFF;
    --gray-50: #F8FAFC;
    --gray-100: #F1F5F9;
    --gray-200: #E2E8F0;
    --gray-300: #CBD5E1;
    --gray-400: #94A3B8;
    --gray-500: #64748B;
    --gray-600: #475569;
    --gray-700: #334155;
    --gray-800: #1E293B;
    --gray-900: #0F172A;
    
    /* Professional Design Tokens */
    --border-radius: 8px;
    --border-radius-lg: 12px;
    --border-radius-xl: 16px;
    
    /* Professional Shadows */
    --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
    --shadow: 0 1px 3px 0 rgb(0 0 0 / 0.1), 0 1px 2px -1px rgb(0 0 0 / 0.1);
    --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
    --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
    --shadow-xl: 0 20px 25px -5px rgb(0 0 0 / 0.1), 0 10px 10px -5px rgb(0 0 0 / 0.04);
    
    /* Professional Gradients */
    --gradient-primary: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
    --gradient-success: linear-gradient(135deg, var(--success-color) 0%, var(--success-dark) 100%);
    --gradient-warning: linear-gradient(135deg, var(--warning-color) 0%, var(--warning-dark) 100%);
    --gradient-danger: linear-gradient(135deg, var(--danger-color) 0%, var(--danger-dark) 100%);
    
    /* Professional Animations */
    --transition: all 0.2s ease-in-out;
    --transition-fast: all 0.15s ease-in-out;
    --transition-slow: all 0.3s ease-in-out;
    
    /* Typography */
    --font-family: 'Cairo', -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
    --font-weight-light: 300;
    --font-weight-normal: 400;
    --font-weight-medium: 500;
    --font-weight-semibold: 600;
    --font-weight-bold: 700;
    --font-weight-black: 800;
}

/* Global Professional Styles */
* {
    box-sizing: border-box;
}

body {
    background: linear-gradient(135deg, var(--gray-50) 0%, var(--gray-100) 100%);
    font-family: var(--font-family);
    font-weight: var(--font-weight-normal);
    color: var(--gray-900);
    line-height: 1.6;
    direction: rtl;
    text-align: right;
}

/* Professional Button Styles */
.cairo-btn {
    font-family: var(--font-family);
    font-weight: var(--font-weight-semibold);
    border-radius: var(--border-radius-lg);
    transition: var(--transition);
    box-shadow: var(--shadow-sm);
}

.cairo-btn:hover {
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
}

/* Professional Layout */
.sync-container {
    padding: 2rem 0;
    min-height: calc(100vh - 120px);
}

/* Professional Alert Messages */
.alert-modern {
    background: var(--white);
    border: 1px solid var(--gray-200);
    border-radius: var(--border-radius-lg);
    box-shadow: var(--shadow-md);
    padding: 1.5rem;
    margin-bottom: 2rem;
    position: relative;
    border-right: 4px solid;
    font-family: var(--font-family);
}

.alert-modern.alert-success {
    border-right-color: var(--success-color);
    background: linear-gradient(90deg, var(--success-light) 0%, var(--white) 15%);
}

.alert-modern.alert-warning {
    border-right-color: var(--warning-color);
    background: linear-gradient(90deg, var(--warning-light) 0%, var(--white) 15%);
}

.alert-modern.alert-danger {
    border-right-color: var(--danger-color);
    background: linear-gradient(90deg, var(--danger-light) 0%, var(--white) 15%);
}

.alert-modern .fw-semibold {
    color: var(--gray-900);
    font-weight: var(--font-weight-semibold);
    font-size: 1.1rem;
    font-family: var(--font-family);
}

.alert-modern .small {
    color: var(--gray-600);
    font-size: 0.9rem;
    font-family: var(--font-family);
}

/* Professional Section Headers */
.section-header {
    margin-bottom: 2rem;
    text-align: center;
}

.section-title {
    font-size: 2rem;
    font-weight: var(--font-weight-bold);
    color: var(--primary-color);
    margin: 0 0 0.5rem 0;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.75rem;
    font-family: var(--font-family);
}

.section-title i {
    color: var(--secondary-color);
    font-size: 1.75rem;
}

.section-subtitle {
    color: var(--gray-600);
    font-size: 1.1rem;
    font-weight: var(--font-weight-medium);
    margin: 0;
    font-family: var(--font-family);
}

/* Professional Statistics Cards */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 1.5rem;
    margin-bottom: 3rem;
}

.stat-card {
    background: var(--white);
    border: 1px solid var(--gray-200);
    border-radius: var(--border-radius-xl);
    padding: 2rem;
    box-shadow: var(--shadow-md);
    transition: var(--transition);
    position: relative;
    overflow: hidden;
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    right: 0;
    width: 100%;
    height: 4px;
    border-radius: var(--border-radius-xl) var(--border-radius-xl) 0 0;
}

.stat-card.primary::before {
    background: var(--gradient-primary);
}

.stat-card.success::before {
    background: var(--gradient-success);
}

.stat-card.warning::before {
    background: var(--gradient-warning);
}

.stat-card.danger::before {
    background: var(--gradient-danger);
}

.stat-card:hover {
    transform: translateY(-4px);
    box-shadow: var(--shadow-xl);
}

.stat-content {
    display: flex;
    align-items: center;
    gap: 1.5rem;
}

.stat-icon {
    width: 60px;
    height: 60px;
    border-radius: var(--border-radius-lg);
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    color: var(--white);
    flex-shrink: 0;
    box-shadow: var(--shadow-md);
}

.stat-icon.primary {
    background: var(--gradient-primary);
}

.stat-icon.success {
    background: var(--gradient-success);
}

.stat-icon.warning {
    background: var(--gradient-warning);
}

.stat-icon.danger {
    background: var(--gradient-danger);
}

.stat-info {
    flex: 1;
}

.stat-number {
    font-size: 2.5rem;
    font-weight: var(--font-weight-black);
    color: var(--gray-900);
    line-height: 1;
    margin-bottom: 0.5rem;
    font-family: var(--font-family);
}

.stat-label {
    color: var(--gray-600);
    font-size: 1.1rem;
    font-weight: var(--font-weight-medium);
    font-family: var(--font-family);
}

/* Professional Results Section */
.results-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1.5rem;
    margin-bottom: 3rem;
}

.result-card {
    background: var(--white);
    border: 1px solid var(--gray-200);
    border-radius: var(--border-radius-lg);
    padding: 1.5rem;
    text-align: center;
    box-shadow: var(--shadow-md);
    transition: var(--transition);
}

.result-card:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow-lg);
}

.result-number {
    font-size: 2rem;
    font-weight: var(--font-weight-black);
    margin-bottom: 0.5rem;
    font-family: var(--font-family);
}

.result-number.success { color: var(--success-color); }
.result-number.danger { color: var(--danger-color); }
.result-number.warning { color: var(--warning-color); }
.result-number.info { color: var(--info-color); }

.result-label {
    color: var(--gray-600);
    font-size: 1rem;
    font-weight: var(--font-weight-medium);
    font-family: var(--font-family);
}

/* Professional Companies Grid */
.companies-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
    gap: 1.5rem;
    margin-bottom: 3rem;
}

.company-card {
    background: var(--white);
    border: 1px solid var(--gray-200);
    border-radius: var(--border-radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-md);
    transition: var(--transition);
}

.company-card:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow-lg);
}

.company-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
    padding-bottom: 1rem;
    border-bottom: 1px solid var(--gray-200);
}

.company-name {
    font-weight: var(--font-weight-semibold);
    color: var(--gray-900);
    margin: 0;
    font-family: var(--font-family);
}

.company-badge {
    background: var(--primary-light);
    color: var(--primary-color);
    padding: 0.25rem 0.75rem;
    border-radius: var(--border-radius);
    font-size: 0.9rem;
    font-weight: var(--font-weight-semibold);
    font-family: var(--font-family);
}

.company-stats {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
}

.status-badge {
    padding: 0.25rem 0.75rem;
    border-radius: var(--border-radius);
    font-size: 0.85rem;
    font-weight: var(--font-weight-medium);
    font-family: var(--font-family);
}

.badge-success {
    background: var(--success-light);
    color: var(--success-color);
}

.badge-danger {
    background: var(--danger-light);
    color: var(--danger-color);
}

.badge-warning {
    background: var(--warning-light);
    color: var(--warning-color);
}

.badge-secondary {
    background: var(--gray-100);
    color: var(--gray-600);
}

/* Professional Main Content Card */
.content-card {
    background: var(--white);
    border: 1px solid var(--gray-200);
    border-radius: var(--border-radius-xl);
    box-shadow: var(--shadow-lg);
    overflow: hidden;
}

.content-header {
    padding: 2rem;
    border-bottom: 1px solid var(--gray-200);
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: linear-gradient(135deg, var(--gray-50) 0%, var(--white) 100%);
}

.content-header .section-title {
    font-size: 1.5rem;
    text-align: right;
    margin-bottom: 0.25rem;
}

.content-header .section-subtitle {
    font-size: 0.95rem;
    text-align: right;
}

.header-actions {
    display: flex;
    gap: 0.75rem;
}

.header-actions .btn {
    font-family: var(--font-family);
    font-weight: var(--font-weight-medium);
    border-radius: var(--border-radius);
    padding: 0.5rem 1rem;
}

/* Professional Filters */
.filters-bar {
    padding: 1.5rem 2rem;
    background: var(--gray-50);
    border-bottom: 1px solid var(--gray-200);
    display: flex;
    gap: 1rem;
    align-items: center;
    flex-wrap: wrap;
}

.search-box {
    flex: 1;
    min-width: 250px;
    position: relative;
}

.search-input {
    width: 100%;
    padding: 0.75rem 1rem 0.75rem 2.5rem;
    border: 1px solid var(--gray-300);
    border-radius: var(--border-radius);
    font-size: 0.95rem;
    font-family: var(--font-family);
    transition: var(--transition);
}

.search-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(15, 76, 117, 0.1);
}

.search-icon {
    position: absolute;
    right: 0.75rem;
    top: 50%;
    transform: translateY(-50%);
    color: var(--gray-400);
}

.filter-select {
    padding: 0.75rem 1rem;
    border: 1px solid var(--gray-300);
    border-radius: var(--border-radius);
    font-size: 0.95rem;
    font-family: var(--font-family);
    min-width: 140px;
    transition: var(--transition);
}

.filter-select:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(15, 76, 117, 0.1);
}

.filters-bar .btn {
    font-family: var(--font-family);
    font-weight: var(--font-weight-medium);
    border-radius: var(--border-radius);
    padding: 0.75rem 1rem;
}

/* Professional Table */
.table-container {
    overflow-x: auto;
}

.invoice-table {
    width: 100%;
    border-collapse: collapse;
    margin: 0;
    font-family: var(--font-family);
}

.invoice-table thead {
    background: var(--gray-50);
}

.invoice-table th {
    padding: 1rem;
    text-align: right;
    font-weight: var(--font-weight-semibold);
    color: var(--gray-700);
    border-bottom: 2px solid var(--gray-200);
    white-space: nowrap;
    font-size: 0.95rem;
    font-family: var(--font-family);
}

.invoice-table tbody tr {
    transition: var(--transition);
    border-bottom: 1px solid var(--gray-100);
}

.invoice-table tbody tr:hover {
    background-color: var(--gray-50);
}

.invoice-table td {
    padding: 1rem;
    vertical-align: middle;
    font-size: 0.95rem;
    font-family: var(--font-family);
}

/* Professional Table Content */
.cell-content {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

.primary-text {
    font-weight: var(--font-weight-semibold);
    color: var(--gray-900);
    font-family: var(--font-family);
}

.secondary-text {
    font-size: 0.85rem;
    color: var(--gray-500);
    font-family: var(--font-family);
}

.amount-text {
    font-weight: var(--font-weight-bold);
    color: var(--success-color);
    font-size: 1rem;
    font-family: var(--font-family);
}

/* Professional Status Badges */
.status-indicator {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    border-radius: var(--border-radius);
    font-size: 0.85rem;
    font-weight: var(--font-weight-medium);
    white-space: nowrap;
    font-family: var(--font-family);
}

.status-cleared {
    background: var(--success-light);
    color: var(--success-color);
}

.status-reported {
    background: var(--info-light);
    color: var(--info-color);
}

.status-rejected {
    background: var(--danger-light);
    color: var(--danger-color);
}

.status-pending {
    background: var(--warning-light);
    color: var(--warning-color);
}

/* Professional IRN Display */
.irn-display {
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.irn-code {
    background: var(--gray-100);
    padding: 0.25rem 0.5rem;
    border-radius: var(--border-radius);
    font-family: 'Courier New', monospace;
    font-size: 0.8rem;
    color: var(--gray-700);
    border: 1px solid var(--gray-200);
}

.copy-btn {
    background: none;
    border: 1px solid var(--gray-300);
    color: var(--gray-500);
    cursor: pointer;
    padding: 0.25rem;
    border-radius: var(--border-radius);
    transition: var(--transition);
}

.copy-btn:hover {
    background: var(--gray-100);
    color: var(--gray-700);
    border-color: var(--gray-400);
}

/* Professional Action Buttons */
.action-group {
    display: flex;
    gap: 0.5rem;
    justify-content: center;
}

.action-btn {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 36px;
    height: 36px;
    border: none;
    border-radius: var(--border-radius);
    cursor: pointer;
    transition: var(--transition);
    text-decoration: none;
    font-size: 0.9rem;
    box-shadow: var(--shadow-sm);
}

.action-btn:hover {
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
}

.btn-sync {
    background: var(--gradient-primary);
    color: var(--white);
}

.btn-view {
    background: var(--gray-200);
    color: var(--gray-600);
}

.btn-view:hover {
    background: var(--gray-300);
    color: var(--gray-700);
}

.btn-info {
    background: var(--gradient-warning);
    color: var(--white);
}

.btn-xml {
    background: var(--gradient-success);
    color: var(--white);
}

/* Professional Empty State */
.empty-state {
    text-align: center;
    padding: 4rem 2rem;
    color: var(--gray-500);
    font-family: var(--font-family);
}

.empty-icon {
    font-size: 3rem;
    margin-bottom: 1rem;
    color: var(--gray-300);
}

.empty-title {
    font-size: 1.25rem;
    font-weight: var(--font-weight-semibold);
    color: var(--gray-700);
    margin-bottom: 0.5rem;
    font-family: var(--font-family);
}

.empty-description {
    margin-bottom: 2rem;
    font-family: var(--font-family);
}

/* Professional Pagination */
.pagination-container {
    padding: 1.5rem 2rem;
    background: var(--gray-50);
    border-top: 1px solid var(--gray-200);
}

/* Professional Modal Improvements */
.modal-content {
    border: none;
    border-radius: var(--border-radius-lg);
    box-shadow: var(--shadow-xl);
    font-family: var(--font-family);
}

.modal-header {
    background: var(--gradient-primary);
    color: var(--white);
    border-radius: var(--border-radius-lg) var(--border-radius-lg) 0 0;
    border: none;
    padding: 1.5rem;
}

.modal-header .modal-title {
    font-family: var(--font-family);
    font-weight: var(--font-weight-semibold);
}

.modal-header .btn-close {
    filter: brightness(0) invert(1);
}

.modal-body {
    padding: 1.5rem;
    font-family: var(--font-family);
}

.json-display {
    background: var(--gray-50);
    border-radius: var(--border-radius);
    padding: 1rem;
    max-height: 400px;
    overflow-y: auto;
    font-family: 'Courier New', monospace;
    font-size: 0.85rem;
    border: 1px solid var(--gray-200);
}

/* Professional Loading State */
.loading-spinner {
    animation: spin 1s linear infinite;
}

@keyframes spin {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}

/* Professional Responsive Design */
@media (max-width: 768px) {
    .sync-container {
        padding: 1rem 0;
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .results-grid {
        grid-template-columns: repeat(2, 1fr);
    }
    
    .companies-grid {
        grid-template-columns: 1fr;
    }
    
    .filters-bar {
        flex-direction: column;
        align-items: stretch;
        gap: 1rem;
    }
    
    .search-box {
        min-width: auto;
    }
    
    .filter-select {
        min-width: auto;
    }
    
    .content-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 1rem;
    }
    
    .header-actions {
        width: 100%;
        justify-content: flex-start;
    }
    
    .invoice-table th,
    .invoice-table td {
        padding: 0.75rem 0.5rem;
        font-size: 0.85rem;
    }
    
    .action-group {
        flex-direction: column;
        gap: 0.25rem;
    }
    
    .section-title {
        font-size: 1.5rem;
    }
    
    .stat-number {
        font-size: 2rem;
    }
}

@media (max-width: 576px) {
    .results-grid {
        grid-template-columns: 1fr;
    }
    
    .stat-content {
        flex-direction: column;
        text-align: center;
        gap: 1rem;
    }
    
    .stat-number {
        font-size: 1.75rem;
    }
    
    .section-title {
        font-size: 1.25rem;
    }
}

/* Professional Print Styles */
@media print {
    .sync-container {
        background: var(--white);
    }
    
    .action-group,
    .header-actions,
    .filters-bar {
        display: none !important;
    }
    
    .content-card {
        box-shadow: none;
        border: 1px solid var(--gray-300);
    }
    
    .stat-card,
    .result-card,
    .company-card {
        box-shadow: none;
        border: 1px solid var(--gray-200);
    }
}
</style>
@endpush

@section('content')
<div class="sync-container">
    <div class="container-fluid">
        <!-- Status Messages -->
        @if(session('success') || session('warning') || session('error'))
        <div class="alert-modern alert-{{ session('success') ? 'success' : (session('warning') ? 'warning' : 'danger') }}">
            <div class="d-flex align-items-start gap-3">
                <i class="bi bi-{{ session('success') ? 'check-circle-fill' : (session('warning') ? 'exclamation-triangle-fill' : 'x-circle-fill') }} fs-5"></i>
                <div class="flex-fill">
                    <div class="fw-semibold">{{ session('success') ?? session('warning') ?? session('error') }}</div>
                    @if(session('sync_action'))
                        <div class="mt-1 small">{!! session('sync_action') !!}</div>
                    @endif
                </div>
                <button type="button" class="btn-close" onclick="this.parentElement.parentElement.remove()"></button>
            </div>
            
            @if(session('sync_details'))
            <div class="mt-3">
                <button class="btn btn-sm btn-outline-secondary" type="button" data-bs-toggle="collapse" data-bs-target="#syncDetails">
                    <i class="bi bi-chevron-down me-1"></i>
                    عرض التفاصيل
                </button>
                <div class="collapse mt-2" id="syncDetails">
                    <div class="json-display">
                        <pre>{{ json_encode(session('sync_details'), JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE) }}</pre>
                    </div>
                </div>
            </div>
            @endif
        </div>
        @endif

        <!-- Statistics Section -->
        <div class="section-header">
            <h2 class="section-title">
                <i class="bi bi-bar-chart-fill"></i>
                إحصائيات المزامنة
            </h2>
            <p class="section-subtitle">نظرة عامة على حالة الفواتير المرسلة إلى ZATCA</p>
        </div>

        <div class="stats-grid">
            <div class="stat-card primary">
                <div class="stat-content">
                    <div class="stat-icon primary">
                        <i class="bi bi-send-fill"></i>
                    </div>
                    <div class="stat-info">
                        <div class="stat-number">{{ number_format($stats['total_sent']) }}</div>
                        <div class="stat-label">إجمالي المرسل</div>
                    </div>
                </div>
            </div>
            
            <div class="stat-card success">
                <div class="stat-content">
                    <div class="stat-icon success">
                        <i class="bi bi-check-circle-fill"></i>
                    </div>
                    <div class="stat-info">
                        <div class="stat-number">{{ number_format($stats['cleared'] + $stats['reported']) }}</div>
                        <div class="stat-label">مؤكدة ومبلغة</div>
                    </div>
                </div>
            </div>
            
            <div class="stat-card danger">
                <div class="stat-content">
                    <div class="stat-icon danger">
                        <i class="bi bi-x-circle-fill"></i>
                    </div>
                    <div class="stat-info">
                        <div class="stat-number">{{ number_format($stats['rejected']) }}</div>
                        <div class="stat-label">مرفوضة</div>
                    </div>
                </div>
            </div>
            
            <div class="stat-card warning">
                <div class="stat-content">
                    <div class="stat-icon warning">
                        <i class="bi bi-clock-fill"></i>
                    </div>
                    <div class="stat-info">
                        <div class="stat-number">{{ number_format($stats['pending_sync']) }}</div>
                        <div class="stat-label">قيد الانتظار</div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Sync Results -->
        @if(session('sync_results'))
        <div class="section-header">
            <h3 class="section-title">
                <i class="bi bi-list-check"></i>
                نتائج المزامنة الأخيرة
            </h3>
            <p class="section-subtitle">تفاصيل آخر عملية مزامنة تم تنفيذها</p>
        </div>
        
        @php $results = session('sync_results') @endphp
        <div class="results-grid">
            <div class="result-card">
                <div class="result-number success">{{ $results['success'] }}</div>
                <div class="result-label">نجحت</div>
            </div>
            <div class="result-card">
                <div class="result-number danger">{{ $results['failed'] }}</div>
                <div class="result-label">فشلت</div>
            </div>
            @if(isset($results['not_found']) && $results['not_found'] > 0)
            <div class="result-card">
                <div class="result-number warning">{{ $results['not_found'] }}</div>
                <div class="result-label">غير موجودة</div>
            </div>
            @endif
            @if(isset($results['not_sent']) && $results['not_sent'] > 0)
            <div class="result-card">
                <div class="result-number info">{{ $results['not_sent'] }}</div>
                <div class="result-label">لم يتم إرسالها</div>
            </div>
            @endif
        </div>
        
        @if(isset($results['details']))
        <div class="companies-grid">
            @foreach($results['details'] as $companyName => $details)
            <div class="company-card">
                <div class="company-header">
                    <h5 class="company-name">{{ $companyName }}</h5>
                    <span class="company-badge">{{ $details['total_processed'] ?? 0 }}</span>
                </div>
                <div class="company-stats">
                    <span class="status-badge badge-success">نجح: {{ $details['success'] ?? 0 }}</span>
                    <span class="status-badge badge-danger">فشل: {{ $details['failed'] ?? 0 }}</span>
                    @if(isset($details['not_found']) && $details['not_found'] > 0)
                        <span class="status-badge badge-warning">غير موجود: {{ $details['not_found'] }}</span>
                    @endif
                    @if(isset($details['not_sent']) && $details['not_sent'] > 0)
                        <span class="status-badge badge-secondary">لم يتم إرساله: {{ $details['not_sent'] }}</span>
                    @endif
                </div>
                @if(isset($details['error']))
                <div class="mt-3 p-2 bg-danger-subtle text-danger rounded">
                    <i class="bi bi-exclamation-triangle-fill me-1"></i>
                    {{ $details['error'] }}
                </div>
                @endif
            </div>
            @endforeach
        </div>
        @endif
        @endif

        <!-- Main Content -->
        <div class="content-card">
            <div class="content-header">
                <div>
                    <h3 class="section-title">
                        <i class="bi bi-receipt"></i>
                        الفواتير المرسلة إلى ZATCA
                    </h3>
                    <p class="section-subtitle">مراقبة ومزامنة حالة الفواتير مع هيئة الزكاة والضريبة</p>
                </div>
                <div class="header-actions">
                    <button class="btn btn-sm btn-outline-primary" onclick="refreshSync()">
                        <i class="bi bi-arrow-clockwise me-1"></i>
                        تحديث
                    </button>
                    <button class="btn btn-sm btn-outline-secondary" data-bs-toggle="modal" data-bs-target="#syncSettingsModal">
                        <i class="bi bi-gear me-1"></i>
                        الإعدادات
                    </button>
                </div>
            </div>
            
            <!-- Filters -->
            <div class="filters-bar">
                <div class="search-box">
                    <i class="bi bi-search search-icon"></i>
                    <input type="text" id="searchInput" class="search-input" placeholder="البحث في الفواتير..." onkeyup="searchInvoices(this.value)">
                </div>
                <select class="filter-select" id="statusFilter" onchange="filterByStatus(this.value)">
                    <option value="all">جميع الحالات</option>
                    <option value="cleared">مؤكدة</option>
                    <option value="reported">مبلغة</option>
                    <option value="rejected">مرفوضة</option>
                    <option value="pending">قيد المراجعة</option>
                </select>
                <select class="filter-select" id="dateFilter" onchange="filterByDate(this.value)">
                    <option value="all">جميع التواريخ</option>
                    <option value="today">اليوم</option>
                    <option value="week">هذا الأسبوع</option>
                    <option value="month">هذا الشهر</option>
                </select>
                <button class="btn btn-outline-success btn-sm" onclick="exportVisibleResults()">
                    <i class="bi bi-download me-1"></i>
                    تصدير
                </button>
                <button class="btn btn-outline-secondary btn-sm" onclick="resetFilters()">
                    <i class="bi bi-arrow-clockwise me-1"></i>
                    إعادة تعيين
                </button>
            </div>
            
            <!-- Table -->
            @if($sentInvoices->count() > 0)
                <div class="table-container">
                    <table class="invoice-table">
                        <thead>
                            <tr>
                                <th>رقم الفاتورة</th>
                                <th>الشركة</th>
                                <th>العميل</th>
                                <th>تاريخ الإرسال</th>
                                <th>المبلغ</th>
                                <th>الحالة</th>
                                <th>IRN</th>
                                <th>الإجراءات</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($sentInvoices as $invoice)
                            <tr class="invoice-row" data-invoice-id="{{ $invoice->id }}" data-status="{{ $invoice->status }}" data-date="{{ $invoice->sent_to_zatca_at->format('Y-m-d') }}">
                                <td>
                                    <div class="cell-content">
                                        <div class="primary-text">{{ $invoice->invoice_number }}</div>
                                        <div class="secondary-text">{{ Str::limit($invoice->uuid, 8) }}...</div>
                                    </div>
                                </td>
                                <td>
                                    <div class="cell-content">
                                        <div class="primary-text">{{ $invoice->company->name }}</div>
                                        <div class="secondary-text">{{ $invoice->company->vat_number }}</div>
                                    </div>
                                </td>
                                <td>
                                    <div class="cell-content">
                                        @if($invoice->customer)
                                            <div class="primary-text">{{ $invoice->customer->name }}</div>
                                            @if($invoice->customer->customer_type == 'business' && $invoice->customer->vat_number)
                                                <div class="secondary-text">{{ $invoice->customer->vat_number }}</div>
                                            @endif
                                        @else
                                            <span class="text-muted">غير محدد</span>
                                        @endif
                                    </div>
                                </td>
                                <td>
                                    <div class="cell-content">
                                        <div class="primary-text">{{ $invoice->sent_to_zatca_at->format('Y-m-d') }}</div>
                                        <div class="secondary-text">{{ $invoice->sent_to_zatca_at->format('H:i') }}</div>
                                    </div>
                                </td>
                                <td>
                                    <div class="amount-text">{{ number_format($invoice->total_amount, 2) }} ر.س</div>
                                </td>
                                <td>
                                    @switch($invoice->status)
                                        @case('cleared')
                                            <span class="status-indicator status-cleared">
                                                <i class="bi bi-check-circle-fill"></i>
                                                مؤكدة
                                            </span>
                                            @break
                                        @case('reported')
                                            <span class="status-indicator status-reported">
                                                <i class="bi bi-flag-fill"></i>
                                                مبلغة
                                            </span>
                                            @break
                                        @case('rejected')
                                            <span class="status-indicator status-rejected">
                                                <i class="bi bi-x-circle-fill"></i>
                                                مرفوضة
                                            </span>
                                            @break
                                        @default
                                            <span class="status-indicator status-pending">
                                                <i class="bi bi-clock-fill"></i>
                                                قيد المراجعة
                                            </span>
                                    @endswitch
                                </td>
                                <td>
                                    @if($invoice->irn)
                                        <div class="irn-display">
                                            <code class="irn-code">{{ Str::limit($invoice->irn, 12) }}...</code>
                                            <button class="copy-btn" onclick="copyToClipboard('{{ $invoice->irn }}')" title="نسخ IRN">
                                                <i class="bi bi-clipboard"></i>
                                            </button>
                                        </div>
                                    @else
                                        <span class="text-muted">غير متوفر</span>
                                    @endif
                                </td>
                                <td>
                                    <div class="action-group">
                                        <form method="POST" action="{{ route('invoices.sync-single', $invoice) }}" style="display: inline;">
                                            @csrf
                                            <button type="submit" class="action-btn btn-sync" title="مزامنة الفاتورة">
                                                <i class="bi bi-arrow-clockwise"></i>
                                            </button>
                                        </form>
                                        
                                        <a href="{{ route('invoices.show', $invoice) }}" class="action-btn btn-view" title="عرض الفاتورة">
                                            <i class="bi bi-eye"></i>
                                        </a>
                                        
                                        @if($invoice->zatca_response)
                                            <button type="button" class="action-btn btn-info" onclick="showZatcaDetails({{ $invoice->id }})" title="تفاصيل ZATCA">
                                                <i class="bi bi-info-circle"></i>
                                            </button>
                                        @endif
                                        
                                        <a href="{{ route('invoices.ubl-xml', $invoice) }}" class="action-btn btn-xml" title="تحميل XML">
                                            <i class="bi bi-file-earmark-code"></i>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
                
                <!-- Pagination -->
                <div class="pagination-container">
                    {{ $sentInvoices->links() }}
                </div>
            @else
                <div class="empty-state">
                    <div class="empty-icon">
                        <i class="bi bi-inbox"></i>
                    </div>
                    <h4 class="empty-title">لا توجد فواتير مرسلة</h4>
                    <p class="empty-description">لم يتم إرسال أي فاتورة إلى ZATCA بعد</p>
                    <a href="{{ route('invoices.index') }}" class="btn btn-primary cairo-btn">
                        <i class="bi bi-arrow-right me-2"></i>
                        الذهاب للفواتير
                    </a>
                </div>
            @endif
        </div>
    </div>
</div>

<!-- ZATCA Details Modal -->
<div class="modal fade" id="zatcaDetailsModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="bi bi-info-circle me-2"></i>
                    تفاصيل استجابة ZATCA
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <pre id="zatcaDetailsContent" class="json-display"></pre>
            </div>
        </div>
    </div>
</div>

<!-- Sync Settings Modal -->
<div class="modal fade" id="syncSettingsModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="bi bi-gear me-2"></i>
                    إعدادات المزامنة
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="mb-3">
                    <label class="form-label fw-semibold">تكرار المزامنة التلقائية</label>
                    <select class="form-select" id="syncInterval">
                        <option value="0">يدوياً فقط</option>
                        <option value="300">كل 5 دقائق</option>
                        <option value="900">كل 15 دقيقة</option>
                        <option value="1800">كل 30 دقيقة</option>
                        <option value="3600">كل ساعة</option>
                    </select>
                </div>
                <div class="mb-3">
                    <div class="form-check form-switch">
                        <input class="form-check-input" type="checkbox" id="autoRefresh">
                        <label class="form-check-label fw-semibold" for="autoRefresh">
                            تحديث تلقائي للصفحة
                        </label>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                <button type="button" class="btn btn-primary cairo-btn" onclick="saveSyncSettings()">
                    <i class="bi bi-check-lg me-2"></i>
                    حفظ
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
// Global Variables
let syncInterval = null;
let autoRefresh = false;
let syncInProgress = false;

// ZATCA Data for each invoice
const zatcaData = {
    @foreach($sentInvoices as $invoice)
        @if($invoice->zatca_response)
        {{ $invoice->id }}: @json($invoice->zatca_response),
        @endif
    @endforeach
};

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    initializePage();
});

function initializePage() {
    loadSyncSettings();
    setupAutoRefresh();
}

// Load sync settings from localStorage
function loadSyncSettings() {
    const interval = localStorage.getItem('sync_interval') || '0';
    const autoRefreshEnabled = localStorage.getItem('auto_refresh') === 'true';
    
    const syncIntervalSelect = document.getElementById('syncInterval');
    const autoRefreshCheckbox = document.getElementById('autoRefresh');
    
    if (syncIntervalSelect) {
        syncIntervalSelect.value = interval;
    }
    if (autoRefreshCheckbox) {
        autoRefreshCheckbox.checked = autoRefreshEnabled;
    }
    
    applySyncSettings(interval, autoRefreshEnabled);
}

// Apply sync settings
function applySyncSettings(interval, autoRefreshEnabled) {
    if (syncInterval) {
        clearInterval(syncInterval);
        syncInterval = null;
    }
    
    autoRefresh = autoRefreshEnabled;
    
    if (interval > 0) {
        syncInterval = setInterval(() => {
            if (autoRefresh) {
                refreshSync();
            } else {
                syncAllInvoices();
            }
        }, interval * 1000);
    }
}

// Save sync settings
function saveSyncSettings() {
    const interval = document.getElementById('syncInterval').value;
    const autoRefreshEnabled = document.getElementById('autoRefresh').checked;
    
    localStorage.setItem('sync_interval', interval);
    localStorage.setItem('auto_refresh', autoRefreshEnabled);
    
    applySyncSettings(interval, autoRefreshEnabled);
    
    const modal = bootstrap.Modal.getInstance(document.getElementById('syncSettingsModal'));
    if (modal) modal.hide();
    
    showNotification('success', 'تم حفظ إعدادات المزامنة بنجاح');
}

// Sync all invoices
function syncAllInvoices() {
    if (syncInProgress) {
        showNotification('warning', 'المزامنة قيد التنفيذ بالفعل');
        return;
    }
    
    if (!confirm('هل تريد مزامنة جميع الفواتير مع ZATCA؟')) {
        return;
    }
    
    const btn = event.target.closest('button');
    const originalHTML = btn.innerHTML;
    
    syncInProgress = true;
    btn.innerHTML = '<i class="bi bi-arrow-clockwise loading-spinner me-2"></i> جاري المزامنة...';
    btn.disabled = true;
    
    fetch('{{ route("invoices.sync-all") }}', {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
            'Content-Type': 'application/json',
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('success', 'تمت المزامنة بنجاح');
            setTimeout(() => window.location.reload(), 2000);
        } else {
            showNotification('error', data.message || 'حدث خطأ أثناء المزامنة');
        }
    })
    .catch(error => {
        console.error('Sync Error:', error);
        showNotification('error', 'حدث خطأ أثناء المزامنة');
    })
    .finally(() => {
        syncInProgress = false;
        btn.innerHTML = originalHTML;
        btn.disabled = false;
    });
}

// Refresh page
function refreshSync() {
    showNotification('info', 'جاري تحديث البيانات...');
    setTimeout(() => window.location.reload(), 1000);
}

// Show ZATCA details
function showZatcaDetails(invoiceId) {
    const details = zatcaData[invoiceId];
    if (details) {
        document.getElementById('zatcaDetailsContent').textContent = JSON.stringify(details, null, 2);
        new bootstrap.Modal(document.getElementById('zatcaDetailsModal')).show();
    } else {
        showNotification('warning', 'لا توجد تفاصيل ZATCA لهذه الفاتورة');
    }
}

// Copy to clipboard
function copyToClipboard(text) {
    navigator.clipboard.writeText(text).then(() => {
        showNotification('success', 'تم نسخ IRN إلى الحافظة');
    }).catch(() => {
        // Fallback for older browsers
        const textArea = document.createElement('textarea');
        textArea.value = text;
        document.body.appendChild(textArea);
        textArea.select();
        document.execCommand('copy');
        document.body.removeChild(textArea);
        showNotification('success', 'تم نسخ IRN إلى الحافظة');
    });
}

// Search invoices
function searchInvoices(searchTerm) {
    const rows = document.querySelectorAll('.invoice-row');
    const term = searchTerm.toLowerCase();
    
    rows.forEach(row => {
        const text = row.textContent.toLowerCase();
        const shouldShow = text.includes(term);
        row.style.display = shouldShow ? '' : 'none';
    });
}

// Filter by status
function filterByStatus(status) {
    const rows = document.querySelectorAll('.invoice-row');
    
    rows.forEach(row => {
        if (status === 'all') {
            row.style.display = '';
        } else {
            const hasStatus = row.querySelector(`.status-${status}`) !== null;
            row.style.display = hasStatus ? '' : 'none';
        }
    });
}

// Filter by date
function filterByDate(period) {
    const rows = document.querySelectorAll('.invoice-row');
    const today = new Date();
    
    rows.forEach(row => {
        if (period === 'all') {
            row.style.display = '';
            return;
        }
        
        const dateStr = row.dataset.date;
        const rowDate = new Date(dateStr);
        let show = false;
        
        switch(period) {
            case 'today':
                show = rowDate.toDateString() === today.toDateString();
                break;
            case 'week':
                const weekAgo = new Date(today.getTime() - 7 * 24 * 60 * 60 * 1000);
                show = rowDate >= weekAgo;
                break;
            case 'month':
                const monthAgo = new Date(today.getTime() - 30 * 24 * 60 * 60 * 1000);
                show = rowDate >= monthAgo;
                break;
        }
        
        row.style.display = show ? '' : 'none';
    });
}

// Export visible results
function exportVisibleResults() {
    showNotification('info', 'جاري تصدير البيانات...');
    // Add export logic here
}

// Reset filters
function resetFilters() {
    document.getElementById('searchInput').value = '';
    document.getElementById('statusFilter').value = 'all';
    document.getElementById('dateFilter').value = 'all';
    document.querySelectorAll('.invoice-row').forEach(row => {
        row.style.display = '';
    });
    showNotification('success', 'تم إعادة تعيين الفلاتر');
}

// Show notification
function showNotification(type, message) {
    const alertTypes = {
        success: { icon: 'check-circle-fill', class: 'alert-success' },
        warning: { icon: 'exclamation-triangle-fill', class: 'alert-warning' },
        error: { icon: 'x-circle-fill', class: 'alert-danger' },
        info: { icon: 'info-circle-fill', class: 'alert-info' }
    };
    
    const config = alertTypes[type] || alertTypes.info;
    
    const notification = document.createElement('div');
    notification.className = `alert ${config.class} position-fixed`;
    notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; max-width: 400px; box-shadow: var(--shadow-lg); border-radius: var(--border-radius); border: none; font-family: var(--font-family);';
    
    notification.innerHTML = `
        <div class="d-flex align-items-center">
            <i class="bi bi-${config.icon} me-2" style="font-size: 1.1rem;"></i>
            <span style="flex: 1;">${message}</span>
            <button type="button" class="btn-close ms-2" onclick="this.parentElement.parentElement.remove()"></button>
        </div>
    `;
    
    document.body.appendChild(notification);
    
    // Animate in
    notification.style.transform = 'translateX(100%)';
    notification.style.opacity = '0';
    setTimeout(() => {
        notification.style.transition = 'all 0.3s ease';
        notification.style.transform = 'translateX(0)';
        notification.style.opacity = '1';
    }, 10);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        if (notification.parentElement) {
            notification.style.transform = 'translateX(100%)';
            notification.style.opacity = '0';
            setTimeout(() => notification.remove(), 300);
        }
    }, 5000);
}

// Setup auto refresh
function setupAutoRefresh() {
    // Update stats every 30 seconds if needed
    setInterval(() => {
        // Add stats update logic here if needed
    }, 30000);
}

// Cleanup on page unload
window.addEventListener('beforeunload', function() {
    if (syncInterval) {
        clearInterval(syncInterval);
    }
});
</script>
@endpush