@extends('layouts.app')

@section('title', 'فاتورة رقم ' . $invoice->invoice_number)
@section('page-title', 'تفاصيل الفاتورة')

@push('styles')
<style>
/* ========== Professional Invoice Variables ========== */
:root {
    --invoice-primary: #2563eb;
    --invoice-secondary: #64748b;
    --invoice-accent: #06b6d4;
    --invoice-success: #10b981;
    --invoice-warning: #f59e0b;
    --invoice-danger: #ef4444;
    --invoice-dark: #1e293b;
    --invoice-light: #f8fafc;
    --invoice-border: #e2e8f0;
    --invoice-text: #0f172a;
    --invoice-muted: #64748b;
    
    /* Professional shadows */
    --shadow-sm: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
    --shadow-md: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
    --shadow-xl: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
    
    /* Modern gradients */
    --gradient-primary: linear-gradient(135deg, #2563eb 0%, #3b82f6 100%);
    --gradient-accent: linear-gradient(135deg, #06b6d4 0%, #0891b2 100%);
    --gradient-success: linear-gradient(135deg, #10b981 0%, #059669 100%);
    --gradient-surface: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
}

/* ========== Page Layout ========== */
.invoice-page {
    min-height: 100vh;
    background: linear-gradient(135deg, #f1f5f9 0%, #e2e8f0 100%);
    padding: 2rem 0;
}

.invoice-container {
    max-width: 100%;
    margin: 0 auto;
    padding: 0 1rem;
}

/* ========== Modern Actions Bar ========== */
.invoice-actions {
    background: white;
    border-radius: 24px;
    padding: 2rem;
    margin-bottom: 2rem;
    box-shadow: var(--shadow-lg);
    border: 1px solid var(--invoice-border);
}

.actions-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
    align-items: start;
}

.action-group {
    display: flex;
    flex-direction: column;
    gap: 0.75rem;
}

.action-title {
    font-size: 0.875rem;
    font-weight: 600;
    color: var(--invoice-muted);
    text-transform: uppercase;
    letter-spacing: 0.05em;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.btn-modern {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
    padding: 0.875rem 1.5rem;
    background: white;
    border: 2px solid var(--invoice-border);
    border-radius: 12px;
    font-weight: 600;
    font-size: 0.875rem;
    text-decoration: none;
    transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
    overflow: hidden;
}

.btn-modern::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
    transition: left 0.6s;
}

.btn-modern:hover::before {
    left: 100%;
}

.btn-modern:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow-lg);
}

.btn-modern.primary {
    background: var(--gradient-primary);
    color: white;
    border-color: #2563eb;
}

.btn-modern.success {
    background: var(--gradient-success);
    color: white;
    border-color: #10b981;
}

.btn-modern.warning {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
    color: white;
    border-color: #f59e0b;
}

.btn-modern.danger {
    background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
    color: white;
    border-color: #ef4444;
}

.btn-modern.info {
    background: var(--gradient-accent);
    color: white;
    border-color: #06b6d4;
}

.btn-modern.dark {
    background: linear-gradient(135deg, #1e293b 0%, #0f172a 100%);
    color: white;
    border-color: #1e293b;
}

/* ========== Professional Invoice Card ========== */
.invoice-card {
    background: white;
    border-radius: 24px;
    overflow: hidden;
    box-shadow: var(--shadow-xl);
    border: 1px solid var(--invoice-border);
    margin-bottom: 2rem;
}

/* ========== Invoice Header Enhancement ========== */
.invoice-header {
    background: var(--gradient-primary);
    color: white;
    padding: 3rem 2rem;
    position: relative;
    overflow: hidden;
}

.invoice-header::before {
    content: '';
    position: absolute;
    top: 0;
    right: 0;
    width: 100%;
    height: 100%;
    background-image: radial-gradient(circle at 25% 25%, rgba(255,255,255,0.1) 0%, transparent 50%),
                      radial-gradient(circle at 75% 75%, rgba(255,255,255,0.05) 0%, transparent 50%);
    pointer-events: none;
}

.header-content {
    position: relative;
    z-index: 1;
    display: grid;
    grid-template-columns: 1fr auto auto;
    gap: 2rem;
    align-items: center;
}

.company-section h1 {
    font-size: 2.5rem;
    font-weight: 800;
    margin-bottom: 1rem;
    text-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.company-details {
    display: grid;
    gap: 0.5rem;
    font-size: 1rem;
    opacity: 0.95;
}

.invoice-meta {
    text-align: center;
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 1rem;
}

.invoice-type-badge {
    background: rgba(255,255,255,0.15);
    backdrop-filter: blur(10px);
    padding: 0.75rem 1.5rem;
    border-radius: 50px;
    font-weight: 600;
    border: 1px solid rgba(255,255,255,0.2);
    white-space: nowrap;
}

.invoice-number {
    font-size: 2rem;
    font-weight: 800;
    text-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.qr-container {
    background: rgba(255,255,255,0.1);
    padding: 1rem;
    border-radius: 16px;
    border: 1px solid rgba(255,255,255,0.2);
}

.qr-code {
    border-radius: 12px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
}

/* ========== Professional Invoice Variables Table ========== */
.invoice-variables {
    padding: 2rem;
    background: linear-gradient(135deg, #f8fafc 0%, #ffffff 100%);
}

.variables-header {
    display: flex;
    align-items: center;
    gap: 1rem;
    margin-bottom: 2rem;
    padding-bottom: 1rem;
    border-bottom: 3px solid var(--invoice-primary);
}

.variables-header h2 {
    font-size: 1.5rem;
    font-weight: 700;
    color: var(--invoice-dark);
    margin: 0;
}

.variables-header i {
    font-size: 1.75rem;
    color: var(--invoice-primary);
}

.variables-table {
    background: white;
    border-radius: 16px;
    overflow: hidden;
    box-shadow: var(--shadow-lg);
    border: 1px solid var(--invoice-border);
}

.variables-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 0;
}

.variable-section {
    border-right: 1px solid var(--invoice-border);
}

.variable-section:last-child {
    border-right: none;
}

.section-header {
    background: var(--gradient-primary);
    color: white;
    padding: 1rem 1.5rem;
    font-weight: 600;
    font-size: 1rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.section-body {
    padding: 0;
}

.variable-item {
    display: flex;
    align-items: center;
    padding: 1rem 1.5rem;
    border-bottom: 1px solid rgba(226, 232, 240, 0.5);
    transition: background-color 0.2s;
}

.variable-item:hover {
    background: rgba(37, 99, 235, 0.02);
}

.variable-item:last-child {
    border-bottom: none;
}

.variable-label {
    flex: 1;
    font-weight: 600;
    color: var(--invoice-muted);
    font-size: 0.875rem;
}

.variable-value {
    flex: 1.5;
    font-weight: 500;
    color: var(--invoice-text);
    text-align: left;
    word-break: break-word;
}

.variable-value.highlight {
    font-weight: 700;
    color: var(--invoice-primary);
}

.variable-value.amount {
    font-family: 'Courier New', monospace;
    font-weight: 700;
    font-size: 1rem;
}

/* ========== Invoice Items Table ========== */
.items-section {
    padding: 2rem;
}

.items-header {
    display: flex;
    align-items: center;
    gap: 1rem;
    margin-bottom: 2rem;
    padding-bottom: 1rem;
    border-bottom: 3px solid var(--invoice-accent);
}

.items-header h2 {
    font-size: 1.5rem;
    font-weight: 700;
    color: var(--invoice-dark);
    margin: 0;
}

.items-header i {
    font-size: 1.75rem;
    color: var(--invoice-accent);
}

.items-table {
    background: white;
    border-radius: 16px;
    overflow: hidden;
    box-shadow: var(--shadow-lg);
    border: 1px solid var(--invoice-border);
}

.items-table table {
    width: 100%;
    border-collapse: separate;
    border-spacing: 0;
}

.items-table thead {
    background: var(--gradient-accent);
    color: white;
}

.items-table th {
    padding: 1.25rem 1rem;
    font-weight: 600;
    font-size: 0.875rem;
    text-align: center;
    border: none;
}

.items-table tbody tr {
    border-bottom: 1px solid var(--invoice-border);
    transition: all 0.2s;
}

.items-table tbody tr:hover {
    background: rgba(6, 182, 212, 0.02);
    transform: translateX(2px);
}

.items-table tbody tr:last-child {
    border-bottom: none;
}

.items-table td {
    padding: 1.25rem 1rem;
    vertical-align: middle;
    border: none;
}

.item-details {
    text-align: right;
}

.item-name {
    font-weight: 600;
    color: var(--invoice-dark);
    margin-bottom: 0.25rem;
}

.item-description {
    color: var(--invoice-muted);
    font-size: 0.875rem;
    margin-bottom: 0.25rem;
}

.item-sku {
    font-size: 0.75rem;
    color: var(--invoice-muted);
    background: rgba(100, 116, 139, 0.1);
    padding: 0.125rem 0.5rem;
    border-radius: 4px;
    display: inline-block;
}

/* ========== Totals Section ========== */
.totals-section {
    padding: 2rem;
    background: linear-gradient(135deg, #f1f5f9 0%, #f8fafc 100%);
    display: grid;
    grid-template-columns: 1fr 400px;
    gap: 2rem;
    align-items: start;
}

.notes-card {
    background: white;
    border-radius: 16px;
    padding: 1.5rem;
    box-shadow: var(--shadow-md);
    border: 1px solid var(--invoice-border);
}

.totals-card {
    background: white;
    border-radius: 16px;
    overflow: hidden;
    box-shadow: var(--shadow-lg);
    border: 1px solid var(--invoice-border);
}

.totals-table {
    width: 100%;
    border-collapse: collapse;
}

.totals-table tr {
    border-bottom: 1px solid var(--invoice-border);
}

.totals-table tr:last-child {
    background: var(--gradient-primary);
    color: white;
    font-weight: 700;
}

.totals-table tr:last-child td {
    border-bottom: none;
}

.totals-table td {
    padding: 1rem 1.5rem;
    font-size: 1rem;
}

.totals-table .label {
    font-weight: 600;
}

.totals-table .amount {
    text-align: left;
    font-family: 'Courier New', monospace;
    font-weight: 600;
}

/* ========== Status Badges ========== */
.status-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    border-radius: 50px;
    font-weight: 600;
    font-size: 0.75rem;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.status-badge::before {
    content: '';
    width: 8px;
    height: 8px;
    border-radius: 50%;
}

.status-badge.draft {
    background: linear-gradient(135deg, #f3f4f6 0%, #e5e7eb 100%);
    color: #374151;
}

.status-badge.draft::before {
    background: #9ca3af;
}

.status-badge.sent {
    background: linear-gradient(135deg, #dbeafe 0%, #bfdbfe 100%);
    color: #1e40af;
}

.status-badge.sent::before {
    background: #3b82f6;
}

.status-badge.cleared {
    background: linear-gradient(135deg, #d1fae5 0%, #a7f3d0 100%);
    color: #065f46;
}

.status-badge.cleared::before {
    background: #10b981;
}

.status-badge.rejected {
    background: linear-gradient(135deg, #fee2e2 0%, #fecaca 100%);
    color: #991b1b;
}

.status-badge.rejected::before {
    background: #ef4444;
}

/* ========== Responsive Design ========== */
@media (max-width: 1200px) {
    .header-content {
        grid-template-columns: 1fr;
        gap: 2rem;
        text-align: center;
    }
    
    .totals-section {
        grid-template-columns: 1fr;
    }
}

@media (max-width: 768px) {
    .invoice-page {
        padding: 1rem 0;
    }
    
    .invoice-actions {
        padding: 1.5rem;
    }
    
    .actions-grid {
        grid-template-columns: 1fr;
    }
    
    .variables-grid {
        grid-template-columns: 1fr;
    }
    
    .variable-section {
        border-right: none;
        border-bottom: 1px solid var(--invoice-border);
    }
    
    .variable-section:last-child {
        border-bottom: none;
    }
    
    .variable-item {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.5rem;
    }
    
    .variable-value {
        text-align: right;
        width: 100%;
    }
    
    .invoice-header {
        padding: 2rem 1rem;
    }
    
    .company-section h1 {
        font-size: 1.75rem;
    }
    
    .invoice-number {
        font-size: 1.5rem;
    }
    
    .items-table {
        font-size: 0.875rem;
    }
    
    .items-table th,
    .items-table td {
        padding: 0.75rem 0.5rem;
    }
}

/* ========== Print Styles ========== */
@media print {
    .invoice-actions {
        display: none !important;
    }
    
    .invoice-page {
        background: white;
        padding: 0;
    }
    
    .invoice-card {
        box-shadow: none;
        border: 1px solid #000;
    }
    
    .invoice-header {
        background: #2563eb !important;
        color: white !important;
        -webkit-print-color-adjust: exact;
        print-color-adjust: exact;
    }
}

/* ========== Animations ========== */
.fade-in {
    animation: fadeIn 0.6s ease-out forwards;
}

.slide-up {
    animation: slideUp 0.6s ease-out forwards;
}

@keyframes fadeIn {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

@keyframes slideUp {
    from {
        opacity: 0;
        transform: translateY(40px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

/* ========== Professional ZATCA Modal ========== */
.zatca-modal {
    z-index: 1055;
}

.zatca-modal .modal-dialog {
    max-width: 700px;
    margin: 3vh auto;
}

.zatca-modal .modal-content {
    border: none;
    border-radius: 28px;
    box-shadow:
        0 32px 64px rgba(0, 0, 0, 0.12),
        0 16px 32px rgba(0, 0, 0, 0.08),
        0 8px 16px rgba(0, 0, 0, 0.04);
    overflow: hidden;
    backdrop-filter: blur(20px);
    background: rgba(255, 255, 255, 0.98);
}

.zatca-modal .modal-header {
    background: linear-gradient(135deg, #1e40af 0%, #3b82f6 50%, #06b6d4 100%);
    color: white;
    padding: 3rem 3rem 2rem;
    border-bottom: none;
    position: relative;
    overflow: hidden;
}

.zatca-modal .modal-header::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background:
        radial-gradient(circle at 20% 20%, rgba(255,255,255,0.15) 0%, transparent 40%),
        radial-gradient(circle at 80% 80%, rgba(255,255,255,0.1) 0%, transparent 40%),
        linear-gradient(45deg, transparent 30%, rgba(255,255,255,0.05) 50%, transparent 70%);
    animation: shimmerFlow 4s ease-in-out infinite;
    pointer-events: none;
}

.zatca-modal .modal-title {
    position: relative;
    z-index: 1;
    font-size: 1.75rem;
    font-weight: 800;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 1rem;
    text-align: center;
    line-height: 1.2;
    text-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.zatca-modal .modal-title i {
    font-size: 2.5rem;
    animation: iconPulse 3s ease-in-out infinite;
    filter: drop-shadow(0 2px 4px rgba(0,0,0,0.2));
}

.zatca-modal .modal-body {
    padding: 3rem;
    background: white;
    position: relative;
}

.zatca-modal .modal-footer {
    padding: 2rem 3rem 3rem;
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    border-top: 1px solid rgba(226, 232, 240, 0.6);
    display: flex;
    justify-content: center;
    gap: 1.5rem;
}

.zatca-modal .btn-close {
    position: absolute;
    top: 1.5rem;
    left: 1.5rem;
    background: rgba(255,255,255,0.25);
    border-radius: 50%;
    padding: 0.875rem;
    opacity: 1;
    backdrop-filter: blur(15px);
    border: 2px solid rgba(255,255,255,0.3);
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    z-index: 10;
}

.zatca-modal .btn-close:hover {
    background: rgba(255,255,255,0.4);
    transform: rotate(90deg) scale(1.1);
    border-color: rgba(255,255,255,0.5);
}

/* Enhanced Warning Section */
.warning-section {
    background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
    border: 2px solid #f59e0b;
    border-radius: 20px;
    padding: 2rem;
    margin-bottom: 2rem;
    position: relative;
    overflow: hidden;
}

.warning-section::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: linear-gradient(90deg, #f59e0b, #d97706, #f59e0b);
    background-size: 200% 100%;
    animation: warningGlow 2s linear infinite;
}

.warning-content {
    display: flex;
    align-items: flex-start;
    gap: 1.5rem;
}

.warning-icon-container {
    background: rgba(217, 119, 6, 0.15);
    border-radius: 50%;
    padding: 1rem;
    flex-shrink: 0;
}

.warning-icon {
    color: #d97706;
    font-size: 2rem;
    animation: warningPulse 2s ease-in-out infinite;
}

.warning-text h6 {
    color: #92400e;
    font-weight: 700;
    font-size: 1.125rem;
    margin-bottom: 0.75rem;
}

.warning-text p {
    color: #78350f;
    font-size: 0.95rem;
    line-height: 1.6;
    margin: 0;
}

/* Enhanced Invoice Summary */
.invoice-summary-card {
    background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
    border: 2px solid #e2e8f0;
    border-radius: 24px;
    padding: 2.5rem;
    margin: 2rem 0;
    position: relative;
    overflow: hidden;
}

.invoice-summary-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 6px;
    background: linear-gradient(90deg, #2563eb, #06b6d4, #2563eb);
    background-size: 200% 100%;
    animation: summaryGlow 3s linear infinite;
}

.summary-header {
    text-align: center;
    margin-bottom: 2rem;
    padding-bottom: 1.5rem;
    border-bottom: 2px solid #e2e8f0;
}

.summary-header h6 {
    font-size: 1.25rem;
    font-weight: 700;
    color: #1e293b;
    margin: 0;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.75rem;
}

.summary-header .summary-icon {
    color: #2563eb;
    font-size: 1.5rem;
}

.summary-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
}

.summary-item {
    background: rgba(248, 250, 252, 0.8);
    border: 1px solid #e2e8f0;
    border-radius: 16px;
    padding: 1.25rem;
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.summary-item:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(0,0,0,0.1);
    border-color: #2563eb;
}

.summary-item::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 3px;
    background: linear-gradient(90deg, #2563eb, #06b6d4);
    transform: scaleX(0);
    transition: transform 0.3s ease;
}

.summary-item:hover::before {
    transform: scaleX(1);
}

.summary-label {
    font-size: 0.875rem;
    color: #64748b;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    margin-bottom: 0.5rem;
    display: block;
}

.summary-value {
    font-size: 1.125rem;
    color: #1e293b;
    font-weight: 700;
    line-height: 1.3;
    display: block;
}

.summary-value.highlight {
    color: #2563eb;
    font-size: 1.25rem;
}

/* Confirmation Section */
.confirmation-section {
    text-align: center;
    padding: 2rem 1rem;
    background: linear-gradient(135deg, rgba(37, 99, 235, 0.05) 0%, rgba(6, 182, 212, 0.05) 100%);
    border-radius: 20px;
    margin-top: 2rem;
}

.confirmation-section h6 {
    font-size: 1.125rem;
    font-weight: 700;
    color: #1e293b;
    margin-bottom: 0.75rem;
    line-height: 1.4;
}

.confirmation-section p {
    color: #64748b;
    font-size: 0.95rem;
    margin: 0;

/* Enhanced Modal Buttons */
.modal-btn {
    padding: 1rem 2.5rem;
    border-radius: 16px;
    font-weight: 700;
    font-size: 1rem;
    border: 2px solid;
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
    overflow: hidden;
    min-width: 160px;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 0.75rem;
    text-transform: none;
    letter-spacing: 0.02em;
}

.modal-btn::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
    transition: left 0.8s ease;
}

.modal-btn:hover::before {
    left: 100%;
}

.modal-btn:hover {
    transform: translateY(-3px) scale(1.02);
    box-shadow: 0 12px 32px rgba(0,0,0,0.15);
}

.modal-btn:active {
    transform: translateY(-1px) scale(0.98);
}

.modal-btn.btn-cancel {
    background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
    color: #64748b;
    border-color: #cbd5e1;
}

.modal-btn.btn-cancel:hover {
    background: linear-gradient(135deg, #f1f5f9 0%, #e2e8f0 100%);
    color: #1e293b;
    border-color: #94a3b8;
}

.modal-btn.btn-confirm {
    background: linear-gradient(135deg, #10b981 0%, #059669 50%, #047857 100%);
    color: white;
    border-color: #059669;
    box-shadow: 0 4px 16px rgba(16, 185, 129, 0.3);
}

.modal-btn.btn-confirm:hover {
    background: linear-gradient(135deg, #059669 0%, #047857 50%, #065f46 100%);
    border-color: #047857;
    box-shadow: 0 8px 32px rgba(16, 185, 129, 0.4);
}

/* Enhanced Animations */
@keyframes shimmerFlow {
    0%, 100% { 
        transform: translateX(-100%) rotate(45deg); 
        opacity: 0.6;
    }
    50% { 
        transform: translateX(100%) rotate(45deg); 
        opacity: 1;
    }
}

@keyframes iconPulse {
    0%, 100% {
        transform: scale(1);
        opacity: 1;
    }
    50% {
        transform: scale(1.1);
        opacity: 0.9;
    }
}

@keyframes warningPulse {
    0%, 100% {
        transform: scale(1) rotate(0deg);
        opacity: 1;
    }
    25% {
        transform: scale(1.1) rotate(-2deg);
        opacity: 0.9;
    }
    75% {
        transform: scale(1.1) rotate(2deg);
        opacity: 0.9;
    }
}

@keyframes warningGlow {
    0%, 100% { 
        background-position: 0% 50%; 
    }
    50% { 
        background-position: 100% 50%; 
    }
}

@keyframes summaryGlow {
    0%, 100% { 
        background-position: 0% 50%; 
    }
    50% { 
        background-position: 100% 50%; 
    }
}

/* Responsive Modal Enhancements */
@media (max-width: 768px) {
    .zatca-modal .modal-dialog {
        margin: 1rem;
        max-width: calc(100vw - 2rem);
    }
    
    .zatca-modal .modal-header {
        padding: 2rem 1.5rem 1.5rem;
    }
    
    .zatca-modal .modal-title {
        font-size: 1.5rem;
        flex-direction: column;
        gap: 0.75rem;
    }
    
    .zatca-modal .modal-body {
        padding: 2rem 1.5rem;
    }
    
    .zatca-modal .modal-footer {
        padding: 1.5rem;
        flex-direction: column;
        gap: 1rem;

/* Simplified Beautiful Modal Styles */
.simple-summary {
    display: flex;
    flex-direction: column;
    gap: 1rem;
    margin-top: 1.5rem;
}

.summary-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 1rem 1.5rem;
    background: rgba(248, 250, 252, 0.7);
    border-radius: 12px;
    border: 1px solid #e2e8f0;
    transition: all 0.3s ease;
}

.summary-row:hover {
    background: rgba(37, 99, 235, 0.05);
    border-color: #3b82f6;
    transform: translateX(3px);
}

.summary-row .label {
    font-weight: 600;
    color: #64748b;
    font-size: 0.95rem;
}

.summary-row .value {
    font-weight: 700;
    color: #1e293b;
    font-size: 1rem;
}

.summary-row .value.highlight {
    color: #2563eb;
    font-size: 1.1rem;
}

/* Simplified Warning Section */
.warning-section {
    background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
    border: 2px solid #f59e0b;
    border-radius: 16px;
    padding: 1.5rem;
    margin-bottom: 2rem;
    text-align: center;
}

.warning-content {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 1rem;
}

.warning-icon-container {
    background: rgba(217, 119, 6, 0.2);
    border-radius: 50%;
    padding: 0.75rem;
    flex-shrink: 0;
}

.warning-icon {
    color: #d97706;
    font-size: 1.5rem;
    animation: gentlePulse 2s ease-in-out infinite;
}

.warning-text h6 {
    color: #92400e;
    font-weight: 700;
    font-size: 1rem;
    margin-bottom: 0.5rem;
}

.warning-text p {
    color: #78350f;
    font-size: 0.9rem;
    margin: 0;
}

/* Simplified Invoice Summary Card */
.invoice-summary-card {
    background: white;
    border: 2px solid #e2e8f0;
    border-radius: 20px;
    padding: 2rem;
    margin: 2rem 0;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
}

.summary-header {
    text-align: center;
    margin-bottom: 1.5rem;
    padding-bottom: 1rem;
    border-bottom: 2px solid #f1f5f9;
}

.summary-header h6 {
    font-size: 1.2rem;
    font-weight: 700;
    color: #1e293b;
    margin: 0;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.75rem;
}

.summary-header .summary-icon {
    color: #2563eb;
    font-size: 1.3rem;
}

/* Simplified Confirmation Section */
.confirmation-section {
    text-align: center;
    padding: 2rem 1rem;
    background: linear-gradient(135deg, rgba(37, 99, 235, 0.08) 0%, rgba(6, 182, 212, 0.08) 100%);
    border-radius: 16px;
    margin-top: 2rem;
    border: 2px solid rgba(37, 99, 235, 0.1);
}

.confirmation-section h6 {
    font-size: 1.1rem;
    font-weight: 700;
    color: #1e293b;
    margin-bottom: 0.75rem;
    line-height: 1.4;
}

.confirmation-section p {
    color: #64748b;
    font-size: 0.95rem;
    margin: 0;
    line-height: 1.5;
}

/* Gentle Animations */
@keyframes gentlePulse {
    0%, 100% {
        transform: scale(1);
        opacity: 1;
    }
    50% {
        transform: scale(1.05);
        opacity: 0.9;
    }
}

/* Mobile Responsive for Simple Design */
@media (max-width: 768px) {
    .warning-content {
        flex-direction: column;
        text-align: center;
        gap: 1rem;
    }
    
    .summary-row {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.5rem;
        text-align: center;
    }
    
    .summary-row .value {
        width: 100%;
        text-align: center;
        font-size: 1.1rem;
    }
    
    .invoice-summary-card {
        padding: 1.5rem;
    }
    
    .confirmation-section {
        padding: 1.5rem 1rem;
    }
    
    /* Keep buttons in single row on mobile */
    .zatca-modal .modal-footer {
        flex-direction: row !important;
        justify-content: center;
        gap: 1rem;
    }
    
    .zatca-modal .modal-btn {
        flex: 1;
        max-width: 150px;
        padding: 1rem 1.25rem;
        font-size: 0.9rem;
    }
}
    }
    
    .modal-btn {
        width: 100%;
        padding: 1.25rem 1.5rem;
    }
    
    .summary-grid {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .warning-content {
        flex-direction: column;
        text-align: center;
        gap: 1rem;
    }
    
    .invoice-summary-card {
        padding: 1.5rem;
    }
}
    line-height: 1.5;
}

@keyframes pulse {
    0%, 100% {
        opacity: 1;
        transform: scale(1);
    }
    50% {
        opacity: 0.8;
        transform: scale(1.05);
    }
}

.modal-btn {
    padding: 0.875rem 2rem;
    border-radius: 12px;
    font-weight: 600;
    font-size: 0.875rem;
    border: 2px solid;
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.modal-btn::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
    transition: left 0.6s;
}

.modal-btn:hover::before {
    left: 100%;
}

.modal-btn:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow-lg);
}

.modal-btn.btn-cancel {
    background: white;
    color: var(--invoice-muted);
    border-color: var(--invoice-border);
}

.modal-btn.btn-cancel:hover {
    background: #f8fafc;
    color: var(--invoice-dark);
}

.modal-btn.btn-confirm {
    background: var(--gradient-success);
    color: white;
    border-color: #10b981;
}

.modal-btn.btn-confirm:hover {
    background: linear-gradient(135deg, #059669 0%, #047857 100%);
}
</style>
@endpush

@section('content')
<div class="invoice-page">
    <div class="invoice-container">
        
        <!-- Customer Alert -->
        @if($invoice->customer)
            <div class="alert alert-info fade-in mb-4" style="animation-delay: 0.1s;">
                <div class="d-flex align-items-center">
                    <i class="fas fa-user-circle fa-2x text-primary me-3"></i>
                    <div>
                        <h6 class="mb-1 fw-bold">{{ $invoice->customer->name }}</h6>
                        <div class="d-flex flex-wrap gap-3">
                            @if($invoice->customer->phone)
                                <small><i class="fas fa-phone me-1"></i>{{ $invoice->customer->phone }}</small>
                            @endif
                            @if($invoice->customer->email)
                                <small><i class="fas fa-envelope me-1"></i>{{ $invoice->customer->email }}</small>
                            @endif
                        </div>
                    </div>
                </div>
            </div>
        @else
            <div class="alert alert-warning fade-in mb-4" style="animation-delay: 0.1s;">
                <i class="fas fa-exclamation-triangle me-2"></i>
                لا يوجد عميل مرتبط بهذه الفاتورة
            </div>
        @endif

        <!-- Actions Bar -->
        <div class="invoice-actions fade-in" style="animation-delay: 0.2s;">
            <div class="">
                <!-- Navigation -->
               
                    <a href="{{ route('invoices.index') }}" style="width:185px;" class="btn-modern primary">
                        <i class="fas fa-arrow-right"></i>
                        الرئيسية 
                    </a>
                    @if($invoice->status == 'draft')
                        <a href="{{ route('invoices.edit', $invoice) }}" style="width:185px;" class="btn-modern warning">
                            <i class="fas fa-edit"></i>
                            تعديل 
                        </a>
                    @endif

         
                    <a href="{{ route('invoices.print', $invoice) }}" style="width:189px;" class="btn-modern dark" target="_blank">
                        <i class="fas fa-print"></i>
                        طباعة
                    </a>
                    <a href="{{ route('invoices.pdf', $invoice) }}" style="width:200px;" class="btn-modern danger" target="_blank">
                        <i class="fas fa-file-pdf"></i>
                         PDF
                    </a>
           
                    @if($invoice->customer && $invoice->customer->phone)
                        <a href="{{ route('invoices.whatsapp', $invoice) }}" style="width:200px;" class="btn-modern success" target="_blank">
                            <i class="fab fa-whatsapp"></i>
                             واتساب
                        </a>
                    @endif
            
        
                    <a href="{{ route('invoices.xml', $invoice) }}" style="width:200px;" class="btn-modern info">
                        <i class="fas fa-file-code"></i>
                        XML
                    </a>
                    <a href="{{ route('invoices.ubl-xml', $invoice) }}" style="width:200px;" class="btn-modern primary">
                        <i class="fas fa-code"></i>
                        UBL XML
                    </a>
             

                <!-- ZATCA Actions -->
                @if($invoice->status == 'draft' && !$invoice->sent_to_zatca_at)
           
                    <form method="POST" action="{{ route('invoices.send-to-zatca', $invoice) }}" class="d-inline" id="zatcaForm">
                        @csrf
                        <button type="button" style="width:200px;" class="btn-modern success" data-bs-toggle="modal" data-bs-target="#zatcaConfirmModal">
                            <i class="fas fa-paper-plane"></i>
                            إرسال إلى ZATCA
                        </button>
                    </form>
                </div>
                @endif
            </div>
      

        <!-- Invoice Card -->
        <div class="invoice-card slide-up" style="animation-delay: 0.3s;">
            
            <!-- Invoice Header -->
            <div class="invoice-header">
                <div class="header-content">
                    <!-- Company Info -->
                    <div class="company-section">
                        <h1>{{ $invoice->company->name }}</h1>
                        <div class="company-details">
                            <div><strong>الرقم الضريبي:</strong> {{ $invoice->company->vat_number }}</div>
                            <div><strong>رقم السجل التجاري:</strong> {{ $invoice->company->cr_number }}</div>
                            <div>{{ $invoice->company->getFullAddressAttribute() }}</div>
                            @if($invoice->company->email)
                                <div>{{ $invoice->company->email }}</div>
                            @endif
                            @if($invoice->company->phone)
                                <div>{{ $invoice->company->phone }}</div>
                            @endif
                        </div>
                    </div>

                    <!-- Invoice Meta -->
                    <div class="invoice-meta">
                        <div class="invoice-type-badge">
                            @if($invoice->type == 'standard')
                                فاتورة ضريبية معيارية
                            @else
                                فاتورة ضريبية مبسطة
                            @endif
                        </div>
                        <div class="invoice-number">{{ $invoice->invoice_number }}</div>
                        <div class="mt-2">
                            @switch($invoice->status)
                                @case('draft')
                                    <span class="status-badge draft">مسودة</span>
                                    @break
                                @case('sent')
                                    <span class="status-badge sent">مرسلة</span>
                                    @break
                                @case('cleared')
                                    <span class="status-badge cleared">مقبولة</span>
                                    @break
                                @case('rejected')
                                    <span class="status-badge rejected">مرفوضة</span>
                                    @break
                            @endswitch
                        </div>
                    </div>

                    <!-- QR Code -->
                    @if($invoice->qr_code)
                        <div class="qr-container">
                            <img src="{{ $invoice->getQrCodeDataUrl() }}" alt="QR Code" class="qr-code" 
                                 style="width: 150px; height: 150px;">
                        </div>
                    @endif
                </div>
            </div>

            <!-- Invoice Variables Section -->
            <div class="invoice-variables">
                <div class="variables-header">
                    <i class="fas fa-info-circle"></i>
                    <h2>متغيرات الفاتورة</h2>
                </div>
                
                <div class="variables-table">
                    <div class="variables-grid">
                        
                        <!-- Basic Information -->
                        <div class="variable-section">
                            <div class="section-header">
                                <i class="fas fa-file-invoice"></i>
                                المعلومات الأساسية
                            </div>
                            <div class="section-body">
                                <div class="variable-item">
                                    <span class="variable-label">رقم الفاتورة</span>
                                    <span class="variable-value highlight">{{ $invoice->invoice_number }}</span>
                                </div>
                                <div class="variable-item">
                                    <span class="variable-label">نوع الفاتورة</span>
                                    <span class="variable-value">{{ $invoice->type == 'standard' ? 'معيارية' : 'مبسطة' }}</span>
                                </div>
                                <div class="variable-item">
                                    <span class="variable-label">حالة الفاتورة</span>
                                    <span class="variable-value">
                                        @switch($invoice->status)
                                            @case('draft') مسودة @break
                                            @case('sent') مرسلة @break
                                            @case('cleared') مقبولة @break
                                            @case('rejected') مرفوضة @break
                                        @endswitch
                                    </span>
                                </div>
                                <div class="variable-item">
                                    <span class="variable-label">العملة</span>
                                    <span class="variable-value">{{ $invoice->currency }}</span>
                                </div>
                                @if($invoice->uuid)
                                <div class="variable-item">
                                    <span class="variable-label">UUID</span>
                                    <span class="variable-value" style="font-size: 0.75rem;">{{ $invoice->uuid }}</span>
                                </div>
                                @endif
                            </div>
                        </div>

                        <!-- Dates -->
                        <div class="variable-section">
                            <div class="section-header">
                                <i class="fas fa-calendar"></i>
                                التواريخ
                            </div>
                            <div class="section-body">
                                <div class="variable-item">
                                    <span class="variable-label">تاريخ الإصدار</span>
                                    <span class="variable-value">{{ $invoice->issue_date->format('Y-m-d') }}</span>
                                </div>
                                <div class="variable-item">
                                    <span class="variable-label">تاريخ التوريد</span>
                                    <span class="variable-value">{{ $invoice->supply_date->format('Y-m-d') }}</span>
                                </div>
                                @if($invoice->due_date)
                                <div class="variable-item">
                                    <span class="variable-label">تاريخ الاستحقاق</span>
                                    <span class="variable-value">{{ $invoice->due_date->format('Y-m-d') }}</span>
                                </div>
                                @endif
                                <div class="variable-item">
                                    <span class="variable-label">تاريخ الإنشاء</span>
                                    <span class="variable-value">{{ $invoice->created_at->format('Y-m-d H:i') }}</span>
                                </div>
                                @if($invoice->sent_to_zatca_at)
                                <div class="variable-item">
                                    <span class="variable-label">تاريخ إرسال ZATCA</span>
                                    <span class="variable-value">{{ $invoice->sent_to_zatca_at->format('Y-m-d H:i') }}</span>
                                </div>
                                @endif
                            </div>
                        </div>

                        <!-- Financial Amounts -->
                        <div class="variable-section">
                            <div class="section-header">
                                <i class="fas fa-calculator"></i>
                                المبالغ المالية
                            </div>
                            <div class="section-body">
                                <div class="variable-item">
                                    <span class="variable-label">المجموع الفرعي</span>
                                    <span class="variable-value amount">{{ number_format($invoice->subtotal, 2) }} ر.س</span>
                                </div>
                                @if($invoice->discount_amount > 0)
                                <div class="variable-item">
                                    <span class="variable-label">الخصم</span>
                                    <span class="variable-value amount" style="color: var(--invoice-danger);">{{ number_format($invoice->discount_amount, 2) }} ر.س</span>
                                </div>
                                @endif
                                <div class="variable-item">
                                    <span class="variable-label">مبلغ الضريبة</span>
                                    <span class="variable-value amount">{{ number_format($invoice->tax_amount, 2) }} ر.س</span>
                                </div>
                                <div class="variable-item">
                                    <span class="variable-label">المبلغ الإجمالي</span>
                                    <span class="variable-value amount highlight">{{ number_format($invoice->total_amount, 2) }} ر.س</span>
                                </div>
                                @if($invoice->paid_amount > 0)
                                <div class="variable-item">
                                    <span class="variable-label">المبلغ المدفوع</span>
                                    <span class="variable-value amount" style="color: var(--invoice-success);">{{ number_format($invoice->paid_amount, 2) }} ر.س</span>
                                </div>
                                <div class="variable-item">
                                    <span class="variable-label">المبلغ المتبقي</span>
                                    <span class="variable-value amount" style="color: var(--invoice-warning);">{{ number_format($invoice->remaining_amount, 2) }} ر.س</span>
                                </div>
                                @endif
                            </div>
                        </div>

                        <!-- Customer Information -->
                        @if($invoice->customer)
                        <div class="variable-section">
                            <div class="section-header">
                                <i class="fas fa-user"></i>
                                معلومات العميل
                            </div>
                            <div class="section-body">
                                <div class="variable-item">
                                    <span class="variable-label">اسم العميل</span>
                                    <span class="variable-value">{{ $invoice->customer->name }}</span>
                                </div>
                                <div class="variable-item">
                                    <span class="variable-label">نوع العميل</span>
                                    <span class="variable-value">{{ $invoice->customer->customer_type == 'business' ? 'أعمال' : 'فرد' }}</span>
                                </div>
                                @if($invoice->customer->vat_number)
                                <div class="variable-item">
                                    <span class="variable-label">الرقم الضريبي</span>
                                    <span class="variable-value">{{ $invoice->customer->vat_number }}</span>
                                </div>
                                @endif
                                @if($invoice->customer->cr_number)
                                <div class="variable-item">
                                    <span class="variable-label">السجل التجاري</span>
                                    <span class="variable-value">{{ $invoice->customer->cr_number }}</span>
                                </div>
                                @endif
                                @if($invoice->customer->national_id)
                                <div class="variable-item">
                                    <span class="variable-label">الهوية الوطنية</span>
                                    <span class="variable-value">{{ $invoice->customer->national_id }}</span>
                                </div>
                                @endif
                                @if($invoice->customer->phone)
                                <div class="variable-item">
                                    <span class="variable-label">رقم الهاتف</span>
                                    <span class="variable-value">{{ $invoice->customer->phone }}</span>
                                </div>
                                @endif
                                @if($invoice->customer->email)
                                <div class="variable-item">
                                    <span class="variable-label">البريد الإلكتروني</span>
                                    <span class="variable-value">{{ $invoice->customer->email }}</span>
                                </div>
                                @endif
                            </div>
                        </div>
                        @endif

                        <!-- ZATCA Information -->
                        @if($invoice->invoice_hash || $invoice->invoice_counter || $invoice->irn)
                        <div class="variable-section">
                            <div class="section-header">
                                <i class="fas fa-shield-alt"></i>
                                معلومات ZATCA
                            </div>
                            <div class="section-body">
                                @if($invoice->invoice_counter)
                                <div class="variable-item">
                                    <span class="variable-label">عداد الفاتورة</span>
                                    <span class="variable-value">{{ $invoice->invoice_counter }}</span>
                                </div>
                                @endif
                                @if($invoice->irn)
                                <div class="variable-item">
                                    <span class="variable-label">IRN</span>
                                    <span class="variable-value" style="font-size: 0.75rem;">{{ $invoice->irn }}</span>
                                </div>
                                @endif
                                @if($invoice->invoice_hash)
                                <div class="variable-item">
                                    <span class="variable-label">Hash الفاتورة</span>
                                    <span class="variable-value" style="font-size: 0.75rem;">{{ Str::limit($invoice->invoice_hash, 30) }}...</span>
                                </div>
                                @endif
                                @if($invoice->previous_invoice_hash)
                                <div class="variable-item">
                                    <span class="variable-label">Hash السابقة</span>
                                    <span class="variable-value" style="font-size: 0.75rem;">{{ Str::limit($invoice->previous_invoice_hash, 30) }}...</span>
                                </div>
                                @endif
                            </div>
                        </div>
                        @endif

                        <!-- Additional Information -->
                        <div class="variable-section">
                            <div class="section-header">
                                <i class="fas fa-info"></i>
                                معلومات إضافية
                            </div>
                            <div class="section-body">
                                <div class="variable-item">
                                    <span class="variable-label">المنشئ</span>
                                    <span class="variable-value">{{ $invoice->user->name }}</span>
                                </div>
                                @if($invoice->payment_method)
                                <div class="variable-item">
                                    <span class="variable-label">طريقة السداد</span>
                                    <span class="variable-value">{{ $invoice->getPaymentMethodText() }}</span>
                                </div>
                                @endif
                                <div class="variable-item">
                                    <span class="variable-label">حالة السداد</span>
                                    <span class="variable-value">{{ $invoice->getPaymentStatusText() }}</span>
                                </div>
                                <div class="variable-item">
                                    <span class="variable-label">عدد العناصر</span>
                                    <span class="variable-value">{{ $invoice->items->count() }} عنصر</span>
                                </div>
                                @if($invoice->notes)
                                <div class="variable-item">
                                    <span class="variable-label">ملاحظات</span>
                                    <span class="variable-value">{{ Str::limit($invoice->notes, 50) }}</span>
                                </div>
                                @endif
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Invoice Items -->
            <div class="items-section">
                <div class="items-header">
                    <i class="fas fa-list"></i>
                    <h2>عناصر الفاتورة</h2>
                </div>
                
                <div class="items-table">
                    <table>
                        <thead>
                            <tr>
                                <th width="4%">#</th>
                                <th width="25%">الوصف</th>
                                <th width="8%">الكمية</th>
                                <th width="8%">الوحدة</th>
                                <th width="10%">سعر الوحدة</th>
                                <th width="10%">الخصم</th>
                                <th width="10%">الإجمالي</th>
                                <th width="8%">معدل الضريبة</th>
                                <th width="10%">مبلغ الضريبة</th>
                                <th width="12%">الإجمالي مع الضريبة</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($invoice->items as $index => $item)
                            <tr>
                                <td class="text-center">{{ $index + 1 }}</td>
                                <td class="item-details">
                                    <div class="item-name">{{ $item->name }}</div>
                                    @if($item->description)
                                        <div class="item-description">{{ $item->description }}</div>
                                    @endif
                                    @if($item->sku)
                                        <div class="item-sku">رمز: {{ $item->sku }}</div>
                                    @endif
                                </td>
                                <td class="text-center">{{ number_format($item->quantity, 3) }}</td>
                                <td class="text-center">{{ $item->unit }}</td>
                                <td class="text-end">{{ number_format($item->unit_price, 2) }} ر.س</td>
                                <td class="text-end">{{ number_format($item->discount_amount, 2) }} ر.س</td>
                                <td class="text-end">{{ number_format($item->line_total, 2) }} ر.س</td>
                                <td class="text-center">{{ number_format($item->tax_rate, 2) }}%</td>
                                <td class="text-end">{{ number_format($item->tax_amount, 2) }} ر.س</td>
                                <td class="text-end fw-bold">{{ number_format($item->total_with_tax, 2) }} ر.س</td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Totals Section -->
            <div class="totals-section">
                <!-- Notes -->
                @if($invoice->notes)
                <div class="notes-card">
                    <h3 class="mb-3">
                        <i class="fas fa-sticky-note text-primary me-2"></i>
                        ملاحظات
                    </h3>
                    <p class="mb-0">{{ $invoice->notes }}</p>
                </div>
                @else
                <div></div>
                @endif
                
                <!-- Totals -->
                <div class="totals-card">
                    <table class="totals-table">
                        <tr>
                            <td class="label">المجموع الفرعي:</td>
                            <td class="amount">{{ number_format($invoice->subtotal, 2) }} ر.س</td>
                        </tr>
                        @if($invoice->discount_amount > 0)
                        <tr>
                            <td class="label">الخصم:</td>
                            <td class="amount text-danger">-{{ number_format($invoice->discount_amount, 2) }} ر.س</td>
                        </tr>
                        @endif
                        <tr>
                            <td class="label">إجمالي الضريبة (15%):</td>
                            <td class="amount">{{ number_format($invoice->tax_amount, 2) }} ر.س</td>
                        </tr>
                        <tr>
                            <td class="label">المبلغ الإجمالي:</td>
                            <td class="amount">{{ number_format($invoice->total_amount, 2) }} ر.س</td>
                        </tr>
                        @if($invoice->paid_amount > 0)
                        <tr style="background: rgba(16, 185, 129, 0.1); color: var(--invoice-success);">
                            <td class="label">المبلغ المدفوع:</td>
                            <td class="amount">{{ number_format($invoice->paid_amount, 2) }} ر.س</td>
                        </tr>
                        <tr style="background: rgba(245, 158, 11, 0.1); color: var(--invoice-warning);">
                            <td class="label">المبلغ المتبقي:</td>
                            <td class="amount">{{ number_format($invoice->remaining_amount, 2) }} ر.س</td>
                        </tr>
                        @endif
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
// تأكيد إرسال إلى ZATCA
function confirmSendToZatca() {
    const invoiceNumber = '{{ $invoice->invoice_number }}';
    const invoiceType = '{{ $invoice->type == "standard" ? "معيارية" : "مبسطة" }}';
    const issueDate = '{{ $invoice->issue_date->format("Y-m-d") }}';
    const totalAmount = '{{ number_format($invoice->total_amount, 2) }}';
    
    const message = `هل تريد إرسال هذه الفاتورة إلى هيئة الزكاة والضريبة والجمارك (ZATCA)؟

تفاصيل الفاتورة:
• رقم الفاتورة: ${invoiceNumber}
• نوع الفاتورة: ${invoiceType}
• تاريخ الإصدار: ${issueDate}
• المبلغ الإجمالي: ${totalAmount} ر.س

⚠️ تنبيه مهم: بعد الإرسال لن يمكن تعديل الفاتورة

اضغط "موافق" للمتابعة أو "إلغاء" للتراجع`;

    return confirm(message);
}

// تأثيرات الحركة
document.addEventListener('DOMContentLoaded', function() {
    // تأثير الحركة للعناصر
    const elements = document.querySelectorAll('.fade-in, .slide-up');
    elements.forEach((element, index) => {
        element.style.animationDelay = (index * 0.1) + 's';
    });

    // تحسين الأزرار
    const buttons = document.querySelectorAll('.btn-modern');
    buttons.forEach(button => {
        button.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-2px)';
        });
        
        button.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
        });
    });
});
</script>
@endpush


<!-- ZATCA Confirmation Modal -->
<div class="modal fade zatca-modal" id="zatcaConfirmModal" tabindex="-1" aria-labelledby="zatcaConfirmModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="zatcaConfirmModalLabel">
                    <i class="fas fa-shield-alt"></i>
                    تأكيد إرسال الفاتورة إلى ZATCA
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="إغلاق"></button>
            </div>
            <div class="modal-body">
                <!-- Beautiful Warning Section -->
                <div class="warning-section">
                    <div class="warning-content">
                        <div class="warning-icon-container">
                            <i class="fas fa-exclamation-triangle warning-icon"></i>
                        </div>
                        <div class="warning-text">
                            <h6>⚠️ تنبيه مهم</h6>
                            <p>لن يمكن تعديل الفاتورة بعد الإرسال إلى زاتكا</p>
                        </div>
                    </div>
                </div>

                <!-- Beautiful Invoice Summary -->
                <div class="invoice-summary-card">
                    <div class="summary-header">
                        <h6>
                            <i class="fas fa-file-invoice summary-icon"></i>
                            بيانات الفاتورة
                        </h6>
                    </div>
                    
                    <div class="simple-summary">
                        <div class="summary-row">
                            <span class="label">📄 رقم الفاتورة:</span>
                            <span class="value highlight">{{ $invoice->invoice_number }}</span>
                        </div>
                        
                        <div class="summary-row">
                            <span class="label">📅 التاريخ:</span>
                            <span class="value">{{ $invoice->issue_date->format('Y-m-d') }}</span>
                        </div>
                        
                        @if($invoice->customer)
                        <div class="summary-row">
                            <span class="label">👤 العميل:</span>
                            <span class="value">{{ $invoice->customer->name }}</span>
                        </div>
                        @endif
                        
                        <div class="summary-row">
                            <span class="label">💰 المبلغ الإجمالي:</span>
                            <span class="value highlight">{{ number_format($invoice->total_amount, 2) }} ر.س</span>
                        </div>
                    </div>
                </div>

                <!-- Simple Confirmation -->
                <div class="confirmation-section">
                    <h6>🔐 هل تريد إرسال هذه الفاتورة إلى زاتكا؟</h6>
                    <p>سيتم الإرسال فوراً وتحديث حالة الفاتورة</p>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="modal-btn btn-cancel" data-bs-dismiss="modal">
                    <i class="fas fa-times me-2"></i>
                    إلغاء
                </button>
                <button type="button" class="modal-btn btn-confirm" onclick="submitZatcaForm()">
                    <i class="fas fa-paper-plane me-2"></i>
                    تأكيد الإرسال
                </button>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
// إرسال النموذج إلى ZATCA
function submitZatcaForm() {
    // إغلاق المودال
    const modal = bootstrap.Modal.getInstance(document.getElementById('zatcaConfirmModal'));
    modal.hide();
    
    // إرسال النموذج
    document.getElementById('zatcaForm').submit();
}

// تأثيرات الحركة
document.addEventListener('DOMContentLoaded', function() {
    // تأثير الحركة للعناصر
    const elements = document.querySelectorAll('.fade-in, .slide-up');
    elements.forEach((element, index) => {
        element.style.animationDelay = (index * 0.1) + 's';
    });

    // تحسين الأزرار
    const buttons = document.querySelectorAll('.btn-modern');
    buttons.forEach(button => {
        button.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-2px)';
        });
        
        button.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
        });
    });

    // تحسين أزرار المودال
    const modalButtons = document.querySelectorAll('.modal-btn');
    modalButtons.forEach(button => {
        button.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-2px)';
        });
        
        button.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
        });
    });

    // تأثيرات المودال
    const zatcaModal = document.getElementById('zatcaConfirmModal');
    if (zatcaModal) {
        zatcaModal.addEventListener('shown.bs.modal', function() {
            // تركيز على زر التأكيد
            const confirmBtn = this.querySelector('.btn-confirm');
            if (confirmBtn) {
                setTimeout(() => {
                    confirmBtn.focus();
                }, 100);
            }
        });
    }
});
</script>
@endpush
