@extends('layouts.app')

@section('title', 'تقرير المدفوعات')
@section('page-title', 'تقرير المدفوعات')

@push('styles')
<style>
/* Modern CSS Variables */
:root {
    --primary-color: #2563eb;
    --primary-gradient: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%);
    --success-color: #059669;
    --success-gradient: linear-gradient(135deg, #059669 0%, #047857 100%);
    --warning-color: #d97706;
    --warning-gradient: linear-gradient(135deg, #d97706 0%, #b45309 100%);
    --danger-color: #dc2626;
    --danger-gradient: linear-gradient(135deg, #dc2626 0%, #b91c1c 100%);
    --info-color: #0891b2;
    --info-gradient: linear-gradient(135deg, #0891b2 0%, #0e7490 100%);
    --dark-color: #1f2937;
    --light-color: #f8fafc;
    --border-color: #e2e8f0;
    --text-primary: #1e293b;
    --text-secondary: #64748b;
    --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
    --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
    --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
    --shadow-xl: 0 20px 25px -5px rgb(0 0 0 / 0.1), 0 8px 10px -6px rgb(0 0 0 / 0.1);
    --border-radius: 12px;
    --border-radius-lg: 16px;
    --transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    
    /* Enhanced Gradients from Analytics */
    --gradient-blue: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    --gradient-green: linear-gradient(135deg, #11998e 0%, #38ef7d 100%);
    --gradient-orange: linear-gradient(135deg, #ff7e5f 0%, #feb47b 100%);
    --gradient-purple: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    --gradient-pink: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
    --gradient-teal: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
    --gradient-sunset: linear-gradient(135deg, #fa709a 0%, #fee140 100%);
    --gradient-ocean: linear-gradient(135deg, #2196F3 0%, #21CBF3 100%);
    
    /* Enhanced Shadows */
    --shadow-card: 0 4px 20px rgba(0, 0, 0, 0.1);
    --shadow-card-hover: 0 8px 30px rgba(0, 0, 0, 0.15);
    --shadow-deep: 0 10px 40px rgba(0, 0, 0, 0.2);
}

/* ========== Enhanced Stat Cards from Analytics ========== */
.stat-card {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 20px;
    padding: 2rem 1.5rem;
    position: relative;
    overflow: hidden;
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    box-shadow: var(--shadow-card);
}

.stat-card::before {
    content: '';
    position: absolute;
    top: -50%;
    right: -50%;
    width: 100%;
    height: 200%;
    background: linear-gradient(45deg, transparent 30%, rgba(255,255,255,0.1) 50%, transparent 70%);
    transform: rotate(45deg);
    transition: all 0.6s ease;
    opacity: 0;
}

.stat-card:hover::before {
    animation: shimmer 1.5s ease-in-out;
}

@keyframes shimmer {
    0% { transform: translateX(-100%) rotate(45deg); opacity: 0; }
    50% { opacity: 1; }
    100% { transform: translateX(100%) rotate(45deg); opacity: 0; }
}

.stat-card .stat-value {
    font-size: 2.5rem;
    font-weight: 800;
    line-height: 1;
    margin-bottom: 0.5rem;
    text-shadow: 0 2px 10px rgba(0,0,0,0.2);
}

.stat-card .stat-label {
    font-size: 0.9rem;
    opacity: 0.9;
    font-weight: 500;
}

.stat-card .stat-icon {
    position: absolute;
    top: 1.5rem;
    left: 1.5rem;
    font-size: 2.5rem;
    opacity: 0.3;
    transition: all 0.3s ease;
}

.stat-card:hover .stat-icon {
    opacity: 0.6;
    transform: scale(1.1);
}

/* Individual card color schemes */
.stat-card.primary { background: var(--gradient-blue); }
.stat-card.success { background: var(--gradient-green); }
.stat-card.warning { background: var(--gradient-orange); }
.stat-card.info { background: var(--gradient-teal); }
.stat-card.purple { background: var(--gradient-purple); }
.stat-card.pink { background: var(--gradient-pink); }
.stat-card.danger { background: linear-gradient(135deg, #ff416c 0%, #ff4b2b 100%); }
.stat-card.ocean { background: var(--gradient-ocean); }

/* ========== Hover Effects ========== */
.hover-lift {
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
}

.hover-lift:hover {
    transform: translateY(-8px);
    box-shadow: var(--shadow-card-hover);
}

.hover-scale {
    transition: transform 0.3s ease;
}

.hover-scale:hover {
    transform: scale(1.02);
}

/* Global Styles */
body {
    background: linear-gradient(135deg, #f1f5f9 0%, #e2e8f0 100%);
    font-family: 'Cairo', 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    direction: rtl;
}

/* Page Container */
.payments-report-container {
    padding: 2rem; 
    max-width: 1400px;
    margin: 0 auto;
}

/* Professional Page Header */
.report-page-header {
    background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
    border-radius: var(--border-radius-lg);
    padding: 2rem;
    margin-bottom: 2rem;
    box-shadow: var(--shadow-lg);
    border: 1px solid var(--border-color);
}

.header-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 1.5rem;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 1.5rem;
}

.page-icon {
    width: 64px;
    height: 64px;
    background: var(--primary-gradient);
    border-radius: var(--border-radius);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.5rem;
    box-shadow: var(--shadow-md);
}

.page-title-section h1 {
    font-size: 2rem;
    font-weight: 700;
    color: var(--text-primary);
    margin: 0 0 0.5rem 0;
    background: var(--primary-gradient);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.page-subtitle {
    color: var(--text-secondary);
    font-size: 1rem;
    margin: 0;
}

.header-actions {
    display: flex;
    gap: 0.75rem;
    flex-wrap: wrap;
}

/* Action Buttons */
.action-btn {
    padding: 0.75rem 1.5rem;
    border-radius: var(--border-radius);
    border: none;
    font-weight: 600;
    font-size: 0.875rem;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    transition: var(--transition);
    cursor: pointer;
    white-space: nowrap;
    box-shadow: var(--shadow-sm);
}

.action-btn:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.btn-primary {
    background: var(--primary-gradient);
    color: white;
}

.btn-success {
    background: var(--success-gradient);
    color: white;
}

.btn-info {
    background: var(--info-gradient);
    color: white;
}

.btn-outline {
    background: white;
    color: var(--text-primary);
    border: 2px solid var(--border-color);
}

.btn-outline:hover {
    border-color: var(--primary-color);
    color: var(--primary-color);
}

/* Statistics Grid */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.modern-stat-card {
    background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
    border-radius: var(--border-radius-lg);
    padding: 2rem;
    box-shadow: var(--shadow-lg);
    border: 1px solid var(--border-color);
    position: relative;
    overflow: hidden;
    transition: var(--transition);
}

.modern-stat-card:hover {
    transform: translateY(-6px) scale(1.02);
    box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.15);
}

.modern-stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 6px;
    background: var(--primary-gradient);
}

.modern-stat-card::after {
    content: '';
    position: absolute;
    top: -50%;
    right: -50%;
    width: 100%;
    height: 100%;
    background: radial-gradient(circle, rgba(255,255,255,0.1) 0%, transparent 70%);
    opacity: 0;
    transition: var(--transition);
}

.modern-stat-card:hover::after {
    opacity: 1;
}

.stat-primary::before { background: var(--primary-gradient); }
.stat-success::before { background: var(--success-gradient); }
.stat-warning::before { background: var(--warning-gradient); }
.stat-danger::before { background: var(--danger-gradient); }

.stat-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 1.5rem;
}

.stat-icon-wrapper {
    width: 64px;
    height: 64px;
    border-radius: 20px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.75rem;
    color: white;
    box-shadow: 0 8px 16px rgba(0,0,0,0.15);
    position: relative;
}

.stat-icon-wrapper::before {
    content: '';
    position: absolute;
    inset: 0;
    border-radius: 20px;
    padding: 2px;
    background: linear-gradient(135deg, rgba(255,255,255,0.3), transparent);
    mask: linear-gradient(#fff 0 0) content-box, linear-gradient(#fff 0 0);
    mask-composite: xor;
}

.stat-primary .stat-icon-wrapper { background: var(--primary-gradient); }
.stat-success .stat-icon-wrapper { background: var(--success-gradient); }
.stat-warning .stat-icon-wrapper { background: var(--warning-gradient); }
.stat-danger .stat-icon-wrapper { background: var(--danger-gradient); }

.stat-badge {
    background: rgba(255,255,255,0.9);
    color: var(--text-primary);
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.75rem;
    font-weight: 600;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.stat-body {
    text-align: center;
}

.stat-body h3 {
    font-size: 3rem;
    font-weight: 900;
    background: var(--primary-gradient);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    margin: 0 0 0.5rem 0;
    text-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.stat-success .stat-body h3 {
    background: var(--success-gradient);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.stat-warning .stat-body h3 {
    background: var(--warning-gradient);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.stat-danger .stat-body h3 {
    background: var(--danger-gradient);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.stat-body p {
    color: var(--text-secondary);
    font-size: 1.125rem;
    margin: 0;
    font-weight: 600;
    letter-spacing: 0.5px;
}

.stat-trend {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
    margin-top: 1rem;
    font-size: 0.875rem;
    font-weight: 600;
}

.trend-up {
    color: var(--success-color);
}

.trend-down {
    color: var(--danger-color);
}

/* Financial Summary */
.financial-summary-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(320px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.modern-summary-card {
    background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
    border-radius: var(--border-radius-lg);
    padding: 2.5rem;
    box-shadow: var(--shadow-lg);
    border: 1px solid var(--border-color);
    transition: var(--transition);
    position: relative;
    overflow: hidden;
}

.modern-summary-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(135deg, transparent 0%, rgba(255,255,255,0.1) 100%);
    opacity: 0;
    transition: var(--transition);
}

.modern-summary-card:hover {
    transform: translateY(-8px) scale(1.03);
    box-shadow: 0 32px 64px -12px rgba(0, 0, 0, 0.18);
}

.modern-summary-card:hover::before {
    opacity: 1;
}

.summary-card-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 2rem;
}

.summary-icon {
    width: 72px;
    height: 72px;
    border-radius: 24px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 2rem;
    color: white;
    box-shadow: 0 12px 24px rgba(0,0,0,0.15);
    position: relative;
}

.summary-icon::before {
    content: '';
    position: absolute;
    inset: 0;
    border-radius: 24px;
    padding: 2px;
    background: linear-gradient(135deg, rgba(255,255,255,0.4), transparent);
    mask: linear-gradient(#fff 0 0) content-box, linear-gradient(#fff 0 0);
    mask-composite: xor;
}

.summary-card.total .summary-icon { background: var(--primary-gradient); }
.summary-card.collected .summary-icon { background: var(--success-gradient); }
.summary-card.pending .summary-icon { background: var(--warning-gradient); }

.summary-title {
    font-size: 1.25rem;
    font-weight: 700;
    color: var(--text-primary);
    letter-spacing: 0.5px;
}

.summary-amount {
    font-size: 2.5rem;
    font-weight: 900;
    margin: 0;
    text-align: center;
    text-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.summary-amount.total {
    background: var(--primary-gradient);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.summary-amount.success {
    background: var(--success-gradient);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.summary-amount.warning {
    background: var(--warning-gradient);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.summary-percentage {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
    margin-top: 1rem;
    font-size: 0.875rem;
    font-weight: 600;
}

.percentage-up {
    color: var(--success-color);
}

.percentage-down {
    color: var(--danger-color);
}

/* Modern Filter Section */
.modern-filter-section {
    background: white;
    border-radius: var(--border-radius-lg);
    margin-bottom: 2rem;
    box-shadow: var(--shadow-lg);
    border: 1px solid var(--border-color);
    overflow: hidden;
}

.filter-header {
    padding: 1.5rem 2rem;
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    border-bottom: 1px solid var(--border-color);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.filter-title {
    font-size: 1.25rem;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0;
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.filter-toggle {
    background: none;
    border: none;
    color: var(--text-secondary);
    font-size: 1.25rem;
    cursor: pointer;
    transition: var(--transition);
    padding: 0.5rem;
    border-radius: var(--border-radius);
}

.filter-toggle:hover {
    background: var(--border-color);
    color: var(--text-primary);
}

.filter-content {
    max-height: 0;
    overflow: hidden;
    transition: max-height 0.3s ease-out;
}

.filter-content.open {
    max-height: 400px;
    transition: max-height 0.3s ease-in;
}

.filter-form {
    padding: 2rem;
}

.filter-form-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
    align-items: end;
}

.modern-form-group {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.modern-form-group label {
    font-weight: 600;
    color: var(--text-primary);
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-size: 0.875rem;
}

.modern-form-input,
.modern-form-select {
    padding: 0.75rem 1rem;
    border: 2px solid var(--border-color);
    border-radius: var(--border-radius);
    font-size: 0.875rem;
    transition: var(--transition);
    background: white;
    color: var(--text-primary);
    font-weight: 500;
}

.modern-form-input::placeholder {
    color: #999;
}

.modern-form-input:focus,
.modern-form-select:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgb(37 99 235 / 0.1);
    background: white;
}

.modern-form-select option {
    background: white;
    color: var(--text-primary);
}

.filter-actions {
    display: flex;
    gap: 1rem;
    align-items: center;
}

/* Modern Table Section */
.modern-table-section {
    background: white;
    border-radius: var(--border-radius-lg);
    box-shadow: var(--shadow-lg);
    border: 1px solid var(--border-color);
    overflow: hidden;
}

.table-header {
    padding: 1.5rem 2rem;
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    border-bottom: 1px solid var(--border-color);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.table-title {
    font-size: 1.25rem;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0;
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.table-action-btn {
    background: var(--primary-gradient);
    border: none;
    color: white;
    padding: 0.5rem;
    border-radius: var(--border-radius);
    cursor: pointer;
    transition: var(--transition);
    width: 40px;
    height: 40px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.table-action-btn:hover {
    transform: scale(1.1);
}

.modern-table-container {
    overflow-x: auto;
}

.modern-data-table {
    width: 100%;
    border-collapse: collapse;
}

.modern-data-table th {
    background: linear-gradient(135deg, #1e293b 0%, #334155 100%);
    color: white;
    padding: 1rem;
    text-align: right;
    font-weight: 600;
    font-size: 0.875rem;
    white-space: nowrap;
}

.modern-data-table td {
    padding: 1rem;
    border-bottom: 1px solid var(--border-color);
    vertical-align: middle;
}

.modern-data-table tbody tr {
    transition: var(--transition);
}

.modern-data-table tbody tr:hover {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
}

/* Table Cell Styles */
.invoice-cell {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

.invoice-link {
    font-weight: 600;
    color: var(--primary-color);
    text-decoration: none;
    transition: var(--transition);
}

.invoice-link:hover {
    color: var(--primary-color);
    text-decoration: underline;
}

.modern-badge {
    padding: 0.25rem 0.5rem;
    border-radius: 6px;
    font-size: 0.75rem;
    font-weight: 600;
    text-align: center;
    white-space: nowrap;
}

.badge-primary {
    background: var(--primary-gradient);
    color: white;
}

.badge-secondary {
    background: var(--dark-color);
    color: white;
}

.customer-name {
    font-weight: 500;
    color: var(--text-primary);
}

.date-cell {
    color: var(--text-secondary);
    font-size: 0.875rem;
}

.overdue-indicator {
    color: var(--danger-color);
    font-size: 0.75rem;
    font-weight: 600;
    margin-top: 0.25rem;
    display: block;
}

.amount-cell {
    font-weight: 600;
    text-align: left;
    font-family: 'Courier New', monospace;
}

.amount-cell.success { color: var(--success-color); }
.amount-cell.warning { color: var(--warning-color); }

/* Progress Bar */
.progress-cell {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
    min-width: 120px;
}

.modern-progress-bar {
    width: 100%;
    height: 8px;
    background: var(--border-color);
    border-radius: 4px;
    overflow: hidden;
}

.progress-fill {
    height: 100%;
    border-radius: 4px;
    transition: width 0.3s ease;
}

.progress-fill.success { background: var(--success-gradient); }
.progress-fill.warning { background: var(--warning-gradient); }
.progress-fill.danger { background: var(--danger-gradient); }

.progress-text {
    font-size: 0.75rem;
    font-weight: 600;
    color: var(--text-secondary);
    text-align: center;
}

/* Status Badge */
.modern-status-badge {
    padding: 0.375rem 0.75rem;
    border-radius: 20px;
    font-size: 0.75rem;
    font-weight: 600;
    text-align: center;
    white-space: nowrap;
}

.status-paid {
    background: var(--success-gradient);
    color: white;
}

.status-unpaid {
    background: var(--danger-gradient);
    color: white;
}

.status-partial {
    background: var(--warning-gradient);
    color: white;
}

/* Action Buttons */
.modern-action-buttons {
    display: flex;
    gap: 0.5rem;
}

.modern-btn-icon {
    width: 36px;
    height: 36px;
    border-radius: var(--border-radius);
    border: none;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: var(--transition);
    font-size: 0.875rem;
}

.modern-btn-icon.view {
    background: var(--info-gradient);
    color: white;
}

.modern-btn-icon.add {
    background: var(--success-gradient);
    color: white;
}

.modern-btn-icon:hover {
    transform: scale(1.1);
    box-shadow: var(--shadow-md);
}

/* Empty State */
.modern-empty-state {
    text-align: center;
    padding: 3rem 2rem;
    color: var(--text-secondary);
}

.modern-empty-state i {
    font-size: 3rem;
    margin-bottom: 1rem;
    color: var(--border-color);
}

.modern-empty-state h4 {
    font-size: 1.25rem;
    font-weight: 600;
    margin: 0 0 0.5rem 0;
    color: var(--text-primary);
}

/* Pagination */
.modern-pagination-wrapper {
    padding: 1.5rem 2rem;
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    border-top: 1px solid var(--border-color);
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 1rem;
}

.pagination-info {
    color: var(--text-secondary);
    font-size: 0.875rem;
}

/* Modal Styles */
.modern-modal .modal-content {
    border: none;
    border-radius: var(--border-radius-lg);
    box-shadow: var(--shadow-xl);
}

.modern-modal .modal-header {
    background: var(--primary-gradient);
    color: white;
    border-radius: var(--border-radius-lg) var(--border-radius-lg) 0 0;
    padding: 1.5rem 2rem;
    border-bottom: none;
}

.modern-modal .modal-title {
    font-weight: 600;
    font-size: 1.25rem;
}

.modern-modal .modal-body {
    padding: 2rem;
}

.modern-modal .modal-footer {
    padding: 1.5rem 2rem;
    background: var(--light-color);
    border-top: 1px solid var(--border-color);
    border-radius: 0 0 var(--border-radius-lg) var(--border-radius-lg);
}

.modern-alert {
    padding: 1rem 1.5rem;
    border-radius: var(--border-radius);
    margin-bottom: 1.5rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.alert-info {
    background: linear-gradient(135deg, #dbeafe 0%, #bfdbfe 100%);
    color: #1e40af;
    border: 1px solid #93c5fd;
}

.modern-textarea {
    padding: 0.75rem 1rem;
    border: 2px solid var(--border-color);
    border-radius: var(--border-radius);
    font-size: 0.875rem;
    transition: var(--transition);
    background: white;
    resize: vertical;
    min-height: 80px;
    font-family: inherit;
}

.modern-textarea:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgb(37 99 235 / 0.1);
}

.input-group {
    display: flex;
    align-items: center;
}

.input-suffix {
    background: var(--border-color);
    padding: 0.75rem 1rem;
    border: 2px solid var(--border-color);
    border-right: none;
    border-radius: 0 var(--border-radius) var(--border-radius) 0;
    color: var(--text-secondary);
    font-weight: 600;
}

.input-group .modern-form-input {
    border-radius: var(--border-radius) 0 0 var(--border-radius);
    border-left: none;
}

/* Animations */
@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

@keyframes slideInRight {
    from {
        opacity: 0;
        transform: translateX(30px);
    }
    to {
        opacity: 1;
        transform: translateX(0);
    }
}

.fade-in-up {
    animation: fadeInUp 0.6s ease-out forwards;
}

.slide-in-right {
    animation: slideInRight 0.6s ease-out forwards;
}

/* Responsive Design */
@media (max-width: 768px) {
    .payments-report-container {
        padding: 1rem;
    }
    
    .header-content {
        flex-direction: column;
        align-items: stretch;
    }
    
    .header-actions {
        justify-content: center;
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
    }
    
    .financial-summary-grid {
        grid-template-columns: 1fr;
    }
    
    .filter-form-grid {
        grid-template-columns: 1fr;
    }
    
    .modern-table-container {
        font-size: 0.75rem;
    }
    
    .modern-data-table th,
    .modern-data-table td {
        padding: 0.5rem;
    }
    
    .modern-pagination-wrapper {
        flex-direction: column;
        text-align: center;
    }
}

@media (max-width: 480px) {
    .action-btn {
        padding: 0.5rem 1rem;
        font-size: 0.75rem;
    }
    
    .page-title-section h1 {
        font-size: 1.5rem;
    }
    
    .stat-body h3 {
        font-size: 1.75rem;
    }
    
    .summary-amount {
        font-size: 1.5rem;
    }
}

/* Enhanced Print Styles */
@media print {
    /* Hide interactive elements */
    .header-actions,
    .modern-filter-section,
    .modern-pagination-wrapper,
    .modern-action-buttons,
    .filter-toggle,
    .table-action-btn {
        display: none !important;
    }
    
    /* Reset body and container */
    body {
        background: white !important;
        color: #000 !important;
        font-size: 11pt !important;
        line-height: 1.3 !important;
        margin: 0 !important;
        padding: 0 !important;
        -webkit-print-color-adjust: exact !important;
        print-color-adjust: exact !important;
    }
    
    .payments-report-container {
        background: white !important;
        padding: 0 !important;
        max-width: none !important;
        margin: 0 !important;
    }
    
    /* Professional header styling */
    .report-page-header {
        background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%) !important;
        border: 2px solid #2563eb !important;
        border-radius: 8px !important;
        margin-bottom: 20pt !important;
        padding: 15pt !important;
        page-break-inside: avoid !important;
    }
    
    .page-title-section h1 {
        font-size: 18pt !important;
        color: #1e293b !important;
        margin-bottom: 5pt !important;
        text-align: center !important;
    }
    
    .page-subtitle {
        font-size: 12pt !important;
        color: #64748b !important;
        text-align: center !important;
        margin-bottom: 0 !important;
    }
    
    /* Statistics cards styling */
    .stats-grid {
        display: grid !important;
        grid-template-columns: repeat(4, 1fr) !important;
        gap: 10pt !important;
        margin-bottom: 15pt !important;
        page-break-inside: avoid !important;
    }
    
    .stat-card {
        background: #f1f5f9 !important;
        border: 1.5px solid #cbd5e1 !important;
        border-radius: 6px !important;
        padding: 10pt !important;
        text-align: center !important;
        page-break-inside: avoid !important;
        box-shadow: none !important;
    }
    
    .stat-card .stat-value {
        font-size: 16pt !important;
        font-weight: bold !important;
        color: #1e293b !important;
        margin-bottom: 3pt !important;
    }
    
    .stat-card .stat-label {
        font-size: 9pt !important;
        color: #475569 !important;
        font-weight: 600 !important;
    }
    
    .stat-card .stat-icon {
        display: none !important;
    }
    
    /* Financial summary styling */
    .financial-summary-grid {
        display: grid !important;
        grid-template-columns: repeat(3, 1fr) !important;
        gap: 10pt !important;
        margin-bottom: 15pt !important;
        page-break-inside: avoid !important;
    }
    
    .modern-summary-card {
        background: white !important;
        border: 2px solid #e2e8f0 !important;
        border-radius: 6px !important;
        padding: 12pt !important;
        text-align: center !important;
        page-break-inside: avoid !important;
        box-shadow: none !important;
    }
    
    .summary-amount {
        font-size: 14pt !important;
        font-weight: bold !important;
        color: #1e293b !important;
        margin-bottom: 3pt !important;
    }
    
    .summary-title {
        font-size: 10pt !important;
        color: #475569 !important;
        font-weight: 600 !important;
    }
    
    .summary-icon {
        display: none !important;
    }
    
    /* Table section styling */
    .modern-table-section {
        background: white !important;
        border: 1.5px solid #cbd5e1 !important;
        border-radius: 6px !important;
        overflow: visible !important;
        box-shadow: none !important;
        page-break-inside: auto !important;
    }
    
    .table-header {
        background: #1e293b !important;
        color: white !important;
        padding: 8pt 12pt !important;
        border-bottom: none !important;
        border-radius: 4px 4px 0 0 !important;
        page-break-after: avoid !important;
    }
    
    .table-title {
        font-size: 12pt !important;
        font-weight: bold !important;
        text-align: center !important;
        margin: 0 !important;
        color: white !important;
    }
    
    /* Data table styling */
    .modern-data-table {
        width: 100% !important;
        border-collapse: collapse !important;
        font-size: 8pt !important;
        page-break-inside: auto !important;
    }
    
    .modern-data-table th {
        background: #374151 !important;
        color: white !important;
        padding: 6pt 4pt !important;
        text-align: center !important;
        font-weight: bold !important;
        font-size: 8pt !important;
        border: 1px solid #000 !important;
        page-break-after: avoid !important;
    }
    
    .modern-data-table td {
        padding: 5pt 3pt !important;
        border: 1px solid #cbd5e1 !important;
        text-align: center !important;
        font-size: 7pt !important;
        vertical-align: middle !important;
        page-break-inside: avoid !important;
    }
    
    .modern-data-table tbody tr:nth-child(even) {
        background: #f8fafc !important;
    }
    
    .modern-data-table tbody tr:nth-child(odd) {
        background: white !important;
    }
    
    /* Status badges */
    .modern-status-badge {
        padding: 2pt 4pt !important;
        border-radius: 3px !important;
        font-size: 7pt !important;
        font-weight: bold !important;
        border: 1px solid !important;
    }
    
    .status-paid {
        background: #dcfce7 !important;
        color: #166534 !important;
        border-color: #166534 !important;
    }
    
    .status-unpaid {
        background: #fee2e2 !important;
        color: #991b1b !important;
        border-color: #991b1b !important;
    }
    
    .status-partial {
        background: #fef3c7 !important;
        color: #92400e !important;
        border-color: #92400e !important;
    }
    
    /* Progress bars */
    .modern-progress-bar {
        background: #e2e8f0 !important;
        border: 1px solid #cbd5e1 !important;
        height: 6pt !important;
        border-radius: 2px !important;
    }
    
    .progress-fill {
        height: 100% !important;
        border-radius: 1px !important;
    }
    
    .progress-fill.success {
        background: #22c55e !important;
    }
    
    .progress-fill.warning {
        background: #f59e0b !important;
    }
    
    .progress-fill.danger {
        background: #ef4444 !important;
    }
    
    .progress-text {
        font-size: 6pt !important;
        font-weight: bold !important;
        color: #374151 !important;
        margin-top: 1pt !important;
    }
    
    /* Amount cells */
    .amount-cell {
        font-family: 'Courier New', monospace !important;
        font-weight: bold !important;
        font-size: 7pt !important;
    }
    
    .amount-cell.success {
        color: #166534 !important;
    }
    
    .amount-cell.warning {
        color: #92400e !important;
    }
    
    /* Links */
    .invoice-link {
        color: #1e293b !important;
        text-decoration: none !important;
        font-weight: bold !important;
    }
    
    /* Badges */
    .modern-badge {
        padding: 1pt 3pt !important;
        border-radius: 2px !important;
        font-size: 6pt !important;
        font-weight: bold !important;
        border: 1px solid !important;
    }
    
    .badge-primary {
        background: #dbeafe !important;
        color: #1e40af !important;
        border-color: #1e40af !important;
    }
    
    .badge-secondary {
        background: #f1f5f9 !important;
        color: #475569 !important;
        border-color: #475569 !important;
    }
    
    /* Date cells */
    .date-cell {
        font-size: 7pt !important;
        color: #374151 !important;
    }
    
    .overdue-indicator {
        color: #dc2626 !important;
        font-size: 6pt !important;
        font-weight: bold !important;
        display: block !important;
        margin-top: 1pt !important;
    }
    
    /* Customer names */
    .customer-name {
        font-weight: bold !important;
        color: #1e293b !important;
        font-size: 7pt !important;
    }
    
    /* Empty state */
    .modern-empty-state {
        padding: 20pt !important;
        text-align: center !important;
        color: #64748b !important;
        font-size: 10pt !important;
    }
    
    /* Page breaks */
    .report-page-header {
        page-break-after: avoid !important;
    }
    
    .stats-grid,
    .financial-summary-grid {
        page-break-after: avoid !important;
        page-break-inside: avoid !important;
    }
    
    .modern-table-section {
        page-break-before: auto !important;
    }
    
    /* Hide unnecessary elements */
    .hover-lift,
    .fade-in-up,
    .slide-in-right,
    .modern-action-buttons,
    .table-action-btn,
    .filter-toggle {
        display: none !important;
    }
    
    /* Ensure proper margins and page settings */
    @page {
        margin: 1.5cm 1cm 1cm 1cm !important;
        size: A4 portrait !important;
    }
    
    /* Force black text for better printing */
    * {
        color-adjust: exact !important;
        -webkit-print-color-adjust: exact !important;
    }
    
    /* Additional typography optimizations for print */
    .modern-data-table {
        font-family: 'Cairo', 'Segoe UI', Arial, sans-serif !important;
    }
    
    .modern-data-table th,
    .modern-data-table td {
        word-break: break-word !important;
        hyphens: auto !important;
    }
    
    /* Ensure better line spacing for Arabic text */
    .customer-name,
    .invoice-link,
    .date-cell {
        line-height: 1.2 !important;
    }
    
    /* Prevent orphaned table rows */
    .modern-data-table tbody tr {
        page-break-inside: avoid !important;
    }
    
    /* Better spacing for status badges */
    .modern-status-badge {
        white-space: nowrap !important;
        overflow: hidden !important;
        text-overflow: ellipsis !important;
        max-width: 60pt !important;
    }
    
    /* Enhanced readability for amounts */
    .amount-cell {
        white-space: nowrap !important;
        overflow: hidden !important;
        text-overflow: ellipsis !important;
    }
}

/* RTL Specific Styles */
[dir="rtl"] .header-left {
    flex-direction: row-reverse;
}

[dir="rtl"] .filter-title {
    flex-direction: row-reverse;
}

[dir="rtl"] .table-title {
    flex-direction: row-reverse;
}

[dir="rtl"] .amount-cell {
    text-align: right;
}
</style>
@endpush

@section('content')
<div class="">
    <div class="">
        <!-- Professional Page Header -->
        <div class="report-page-header fade-in-up">
            <div class="header-content">
                <div class="header-left">
                    <div class="page-icon">
                        <i class="fas fa-chart-line"></i>
                    </div>
                    <div class="page-title-section">
                        <h1>تقرير المدفوعات</h1>
                        <p class="page-subtitle">تحليل شامل لحالة المدفوعات والتحصيلات</p>
                    </div>
                </div>
                
                <div class="header-actions">
                    <a href="{{ route('invoices.index') }}" class="action-btn btn-outline">
                        <i class="fas fa-arrow-right"></i>
                        العودة للفواتير
                    </a>
                    
                    <button type="button" class="action-btn btn-success" onclick="exportReport()">
                        <i class="fas fa-download"></i>
                        تصدير Excel
                    </button>
                    
                    <button type="button" class="action-btn btn-primary" onclick="printReport()">
                        <i class="fas fa-print"></i>
                        طباعة
                    </button>
                    
                    <a href="{{ route('invoices.payment-analytics') }}" class="action-btn btn-info">
                        <i class="fas fa-chart-bar"></i>
                        التحليلات
                    </a>
                </div>
            </div>
        </div>

        <!-- Enhanced Statistics Overview -->
        <div class="row mb-5">
            <div class="col-lg-3 col-md-6 mb-4">
                <div class="stat-card primary hover-lift fade-in-up">
                    <i class="fas fa-file-invoice-dollar stat-icon"></i>
                    <div class="text-right">
                        <div class="stat-value">{{ number_format($stats['total_invoices']) }}</div>
                        <div class="stat-label">إجمالي الفواتير</div>
                    </div>
                </div>
            </div>
            
            <div class="col-lg-3 col-md-6 mb-4">
                <div class="stat-card success hover-lift fade-in-up" style="animation-delay: 0.1s">
                    <i class="fas fa-check-circle stat-icon"></i>
                    <div class="text-right">
                        <div class="stat-value">{{ number_format($stats['paid_invoices']) }}</div>
                        <div class="stat-label">فواتير مدفوعة</div>
                    </div>
                </div>
            </div>
            
            <div class="col-lg-3 col-md-6 mb-4">
                <div class="stat-card warning hover-lift fade-in-up" style="animation-delay: 0.2s">
                    <i class="fas fa-clock stat-icon"></i>
                    <div class="text-right">
                        <div class="stat-value">{{ number_format($stats['unpaid_invoices'] + $stats['partially_paid']) }}</div>
                        <div class="stat-label">فواتير معلقة</div>
                    </div>
                </div>
            </div>
            
            <div class="col-lg-3 col-md-6 mb-4">
                <div class="stat-card danger hover-lift fade-in-up" style="animation-delay: 0.3s">
                    <i class="fas fa-exclamation-triangle stat-icon"></i>
                    <div class="text-right">
                        <div class="stat-value">{{ number_format($stats['overdue_invoices']) }}</div>
                        <div class="stat-label">فواتير متأخرة</div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Enhanced Financial Summary -->
        <div class="row mb-5">
            <div class="col-lg-4 col-md-6 mb-4">
                <div class="stat-card primary hover-lift fade-in-up" style="animation-delay: 0.4s">
                    <i class="fas fa-coins stat-icon"></i>
                    <div class="text-right">
                        <div class="stat-value">{{ number_format($stats['total_amount'], 0) }}</div>
                        <div class="stat-label">إجمالي المبالغ (ر.س)</div>
                    </div>
                </div>
            </div>
            
            <div class="col-lg-4 col-md-6 mb-4">
                <div class="stat-card success hover-lift fade-in-up" style="animation-delay: 0.5s">
                    <i class="fas fa-hand-holding-usd stat-icon"></i>
                    <div class="text-right">
                        <div class="stat-value">{{ number_format($stats['paid_amount'], 0) }}</div>
                        <div class="stat-label">المبالغ المحصلة (ر.س)</div>
                    </div>
                </div>
            </div>
            
            <div class="col-lg-4 col-md-6 mb-4">
                <div class="stat-card warning hover-lift fade-in-up" style="animation-delay: 0.6s">
                    <i class="fas fa-hourglass-half stat-icon"></i>
                    <div class="text-right">
                        <div class="stat-value">{{ number_format($stats['remaining_amount'], 0) }}</div>
                        <div class="stat-label">المبالغ المعلقة (ر.س)</div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Modern Filter Section -->
        <div class="modern-filter-section fade-in-up">
            <div class="filter-header">
                <h3 class="filter-title">
                    <i class="fas fa-filter"></i>
                    فلاتر البحث
                </h3>
                <button type="button" class="filter-toggle" onclick="toggleFilters()">
                    <i class="fas fa-chevron-down"></i>
                </button>
            </div>
            
            <div class="filter-content" id="filterContent">
                <form method="GET" action="{{ route('invoices.payments-report') }}" class="filter-form">
                    <div class="filter-form-grid">
                        <div class="modern-form-group">
                            <label>
                                <i class="fas fa-tags"></i>
                                حالة السداد
                            </label>
                            <select name="payment_status" class="modern-form-select">
                                <option value="">جميع الحالات</option>
                                <option value="unpaid" {{ request('payment_status') == 'unpaid' ? 'selected' : '' }}>غير مدفوعة</option>
                                <option value="partial" {{ request('payment_status') == 'partial' ? 'selected' : '' }}>مدفوعة جزئياً</option>
                                <option value="paid" {{ request('payment_status') == 'paid' ? 'selected' : '' }}>مدفوعة بالكامل</option>
                                <option value="overdue" {{ request('payment_status') == 'overdue' ? 'selected' : '' }}>متأخرة</option>
                            </select>
                        </div>
                        
                        <div class="modern-form-group">
                            <label>
                                <i class="fas fa-calendar-alt"></i>
                                من تاريخ
                            </label>
                            <input type="date" name="date_from" class="modern-form-input" value="{{ request('date_from') }}">
                        </div>
                        
                        <div class="modern-form-group">
                            <label>
                                <i class="fas fa-calendar-check"></i>
                                إلى تاريخ
                            </label>
                            <input type="date" name="date_to" class="modern-form-input" value="{{ request('date_to') }}">
                        </div>
                        
                        <div class="filter-actions">
                            <button type="submit" class="action-btn btn-primary">
                                <i class="fas fa-search"></i>
                                بحث
                            </button>
                            <button type="button" class="action-btn btn-outline" onclick="clearFilters()">
                                <i class="fas fa-times"></i>
                                مسح
                            </button>
                        </div>
                    </div>
                </form>
            </div>
        </div>

        <!-- Professional Data Table -->
        <div class="modern-table-section slide-in-right">
            <div class="table-header">
                <h3 class="table-title">
                    <i class="fas fa-table"></i>
                    تفاصيل المدفوعات
                </h3>
                <button class="table-action-btn" title="تحديث" onclick="location.reload()">
                    <i class="fas fa-sync-alt"></i>
                </button>
            </div>
            
            <div class="modern-table-container">
                <table class="modern-data-table">
                    <thead>
                        <tr>
                            <th>رقم الفاتورة</th>
                            <th>العميل</th>
                            <th>تاريخ الإصدار</th>
                            <th>تاريخ الاستحقاق</th>
                            <th>المبلغ الإجمالي</th>
                            <th>المبلغ المدفوع</th>
                            <th>المبلغ المتبقي</th>
                            <th>نسبة السداد</th>
                            <th>حالة السداد</th>
                            <th>إجراءات</th>
                        </tr>
                    </thead>
                    <tbody>
                        @forelse($invoices as $invoice)
                        <tr>
                            <td>
                                <div class="invoice-cell">
                                    <a href="{{ route('invoices.show', $invoice) }}" class="invoice-link">
                                        {{ $invoice->invoice_number }}
                                    </a>
                                    @if($invoice->type == 'standard')
                                        <span class="modern-badge badge-primary">معيارية</span>
                                    @else
                                        <span class="modern-badge badge-secondary">مبسطة</span>
                                    @endif
                                </div>
                            </td>
                            <td>
                                @if($invoice->customer)
                                    <span class="customer-name">{{ $invoice->customer->name }}</span>
                                @else
                                    <span class="text-muted">غير محدد</span>
                                @endif
                            </td>
                            <td class="date-cell">{{ $invoice->issue_date->format('Y-m-d') }}</td>
                            <td class="date-cell">
                                @if($invoice->due_date)
                                    <span class="date">{{ $invoice->due_date->format('Y-m-d') }}</span>
                                    @if($invoice->due_date->isPast() && $invoice->payment_status != 'paid')
                                        <span class="overdue-indicator">متأخرة</span>
                                    @endif
                                @else
                                    <span class="text-muted">-</span>
                                @endif
                            </td>
                            <td class="amount-cell">{{ number_format($invoice->total_amount, 2) }} ر.س</td>
                            <td class="amount-cell success">{{ number_format($invoice->paid_amount, 2) }} ر.س</td>
                            <td class="amount-cell warning">{{ number_format($invoice->remaining_amount, 2) }} ر.س</td>
                            <td>
                                <div class="progress-cell">
                                    <div class="modern-progress-bar">
                                        <div class="progress-fill 
                                            @if($invoice->getPaymentPercentage() >= 100) success
                                            @elseif($invoice->getPaymentPercentage() > 0) warning
                                            @else danger @endif"
                                             style="width: {{ $invoice->getPaymentPercentage() }}%">
                                        </div>
                                    </div>
                                    <span class="progress-text">{{ number_format($invoice->getPaymentPercentage(), 1) }}%</span>
                                </div>
                            </td>
                            <td>
                                <span class="modern-status-badge status-{{ $invoice->payment_status }}">
                                    {{ $invoice->getPaymentStatusText() }}
                                </span>
                            </td>
                            <td>
                                <div class="modern-action-buttons">
                                    <a href="{{ route('invoices.show', $invoice) }}" class="modern-btn-icon view" title="عرض">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                    @if($invoice->remaining_amount > 0)
                                        <button type="button" class="modern-btn-icon add" 
                                                onclick="addPayment({{ $invoice->id }}, '{{ $invoice->invoice_number }}', {{ $invoice->remaining_amount }})"
                                                title="إضافة دفعة">
                                            <i class="fas fa-plus"></i>
                                        </button>
                                    @endif
                                </div>
                            </td>
                        </tr>
                        @empty
                        <tr>
                            <td colspan="10">
                                <div class="modern-empty-state">
                                    <i class="fas fa-inbox"></i>
                                    <h4>لا توجد فواتير للعرض</h4>
                                    <p>جرب تغيير معايير البحث</p>
                                </div>
                            </td>
                        </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
            
            @if($invoices->hasPages())
                <div class="modern-pagination-wrapper">
                    <div class="pagination-info">
                        <span class="pagination-text">
                            عرض {{ $invoices->firstItem() ?? 0 }} إلى {{ $invoices->lastItem() ?? 0 }}
                            من {{ $invoices->total() }} نتيجة
                        </span>
                    </div>
                    <div class="modern-pagination">
                        {{ $invoices->appends(request()->query())->links('components.modern-pagination') }}
                    </div>
                </div>
            @endif
        </div>
    </div>
</div>

<!-- Modern Quick Payment Modal -->
<div class="modal fade modern-modal" id="quickPaymentModal" tabindex="-1" role="dialog" aria-labelledby="modal-title" aria-describedby="modal-description">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 id="modal-title" class="modal-title">إضافة دفعة سريعة</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="إغلاق النافذة"></button>
            </div>
            <form id="quickPaymentForm" novalidate aria-describedby="modal-description">
                @csrf
                <div class="modal-body">
                    <div id="modal-description" class="modern-alert alert-info" role="status">
                        <i class="fas fa-info-circle" aria-hidden="true"></i>
                        <div>
                            <strong>فاتورة رقم:</strong> <span id="modal-invoice-number"></span>
                        </div>
                    </div>
                    
                    <div class="modern-form-group">
                        <label for="modal-amount">المبلغ <span style="color: var(--danger-color);" aria-label="حقل مطلوب">*</span></label>
                        <div class="input-group">
                            <input type="number" id="modal-amount" name="amount" class="form-control modern-form-input"
                                   step="0.01" min="0.01" required aria-describedby="amount-help">
                            <span class="input-suffix">ر.س</span>
                        </div>
                        <small id="amount-help" style="color: var(--text-secondary);">الحد الأقصى: <span id="modal-max-amount"></span> ر.س</small>
                    </div>
                    
                    <div class="modern-form-group">
                        <label for="payment-method">طريقة السداد</label>
                        <select id="payment-method" name="method" class="modern-form-select">
                            <option value="">اختر طريقة السداد</option>
                            <option value="cash">نقداً</option>
                            <option value="bank_transfer">تحويل بنكي</option>
                            <option value="check">شيك</option>
                            <option value="credit_card">بطاقة ائتمان</option>
                            <option value="installment">تقسيط</option>
                        </select>
                    </div>
                    
                    <div class="modern-form-group">
                        <label for="payment-notes">ملاحظات</label>
                        <textarea id="payment-notes" name="notes" class="modern-textarea" rows="3" placeholder="ملاحظات حول عملية السداد" style="color: #000000 !important;"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="action-btn btn-outline" data-bs-dismiss="modal">إلغاء</button>
                    <button type="submit" class="action-btn btn-primary">
                        <i class="fas fa-save" aria-hidden="true"></i>
                        <span>إضافة الدفعة</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Hidden accessibility styles -->
<style>
.visually-hidden {
    position: absolute !important;
    width: 1px !important;
    height: 1px !important;
    padding: 0 !important;
    margin: -1px !important;
    overflow: hidden !important;
    clip: rect(0, 0, 0, 0) !important;
    white-space: nowrap !important;
    border: 0 !important;
}
</style>
@endsection

@push('scripts')
<script>
let currentInvoiceId = null;

document.addEventListener('DOMContentLoaded', function() {
    initializeAnimations();
    setupEventListeners();
});

function initializeAnimations() {
    // Add smooth animations for table rows
    const rows = document.querySelectorAll('.modern-data-table tbody tr');
    rows.forEach((row, index) => {
        row.style.animationDelay = `${index * 50}ms`;
        row.classList.add('fade-in-up');
    });
}

function setupEventListeners() {
    // Enhanced hover effects for action buttons
    document.querySelectorAll('.modern-btn-icon').forEach(btn => {
        btn.addEventListener('mouseenter', function() {
            this.style.transform = 'scale(1.1)';
        });
        
        btn.addEventListener('mouseleave', function() {
            this.style.transform = 'scale(1)';
        });
    });
}

function toggleFilters() {
    const filterContent = document.getElementById('filterContent');
    const toggleBtn = document.querySelector('.filter-toggle i');
    
    if (filterContent.classList.contains('open')) {
        filterContent.classList.remove('open');
        toggleBtn.style.transform = 'rotate(0deg)';
    } else {
        filterContent.classList.add('open');
        toggleBtn.style.transform = 'rotate(180deg)';
    }
}

function clearFilters() {
    const form = document.querySelector('.filter-form');
    const inputs = form.querySelectorAll('input, select');
    
    inputs.forEach(input => {
        if (input.type === 'date' || input.tagName === 'SELECT') {
            input.value = '';
        }
    });
    
    // Add visual feedback
    const clearBtn = form.querySelector('.btn-outline');
    const originalText = clearBtn.innerHTML;
    clearBtn.innerHTML = '<i class="fas fa-check"></i> تم المسح';
    clearBtn.style.background = 'var(--gradient-success)';
    clearBtn.style.color = 'white';
    
    setTimeout(() => {
        clearBtn.innerHTML = originalText;
        clearBtn.style.background = '';
        clearBtn.style.color = '';
    }, 1000);
}

function addPayment(invoiceId, invoiceNumber, remainingAmount) {
    currentInvoiceId = invoiceId;
    document.getElementById('modal-invoice-number').textContent = invoiceNumber;
    document.getElementById('modal-amount').setAttribute('max', remainingAmount);
    document.getElementById('modal-max-amount').textContent = parseFloat(remainingAmount).toFixed(2);
    
    const modal = new bootstrap.Modal(document.getElementById('quickPaymentModal'));
    modal.show();
    
    // Focus on amount input
    setTimeout(() => {
        document.getElementById('modal-amount').focus();
    }, 300);
}

document.getElementById('quickPaymentForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    if (!currentInvoiceId) return;
    
    const submitBtn = this.querySelector('button[type="submit"]');
    const originalText = submitBtn.innerHTML;
    
    // Show loading state
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري الحفظ...';
    submitBtn.disabled = true;
    
    const formData = new FormData(this);
    
    fetch(`{{ url('/invoices') }}/${currentInvoiceId}/add-payment`, {
        method: 'POST',
        body: formData,
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json'
        }
    })
    .then(response => {
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            throw new Error('الاستجابة ليست من نوع JSON صالح');
        }
        return response.json();
    })
    .then(data => {
        if (data.success) {
            submitBtn.innerHTML = '<i class="fas fa-check"></i> تم الحفظ بنجاح!';
            submitBtn.style.background = 'var(--gradient-success)';
            
            // Close modal after short delay
            setTimeout(() => {
                const modal = bootstrap.Modal.getInstance(document.getElementById('quickPaymentModal'));
                if (modal) {
                    modal.hide();
                }
                location.reload();
            }, 1500);
        } else {
            throw new Error(data.message || 'خطأ غير معروف');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        submitBtn.innerHTML = '<i class="fas fa-exclamation-triangle"></i> حدث خطأ';
        submitBtn.style.background = 'var(--gradient-danger)';
        
        setTimeout(() => {
            submitBtn.innerHTML = originalText;
            submitBtn.style.background = '';
            submitBtn.disabled = false;
        }, 3000);
        
        // Show more user-friendly error message
        if (error.message.includes('JSON')) {
            alert('حدث خطأ في الاتصال بالخادم. يرجى المحاولة مرة أخرى.');
        } else {
            alert('حدث خطأ: ' + error.message);
        }
    });
});

function exportReport() {
    const exportBtn = document.querySelector('.btn-success');
    const originalContent = exportBtn.innerHTML;
    
    exportBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري التصدير...';
    exportBtn.disabled = true;
    
    // تجميع بيانات التقرير
    const reportData = {
        title: 'تقرير المدفوعات',
        date: new Date().toLocaleDateString('ar-SA'),
        filters: getActiveFilters(),
        stats: getStatsData(),
        invoices: getInvoicesData()
    };
    
    // إنشاء ملف Excel
    const params = new URLSearchParams(window.location.search);
    params.set('export', 'excel');
    params.set('format', 'detailed');
    
    fetch(window.location.pathname + '?' + params.toString(), {
        method: 'GET',
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
        }
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('فشل في تصدير الملف');
        }
        return response.blob();
    })
    .then(blob => {
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.style.display = 'none';
        a.href = url;
        a.download = `تقرير_المدفوعات_${new Date().toISOString().split('T')[0]}.xlsx`;
        document.body.appendChild(a);
        a.click();
        window.URL.revokeObjectURL(url);
        document.body.removeChild(a);
        
        exportBtn.innerHTML = '<i class="fas fa-check"></i> تم التصدير بنجاح!';
        exportBtn.style.background = 'var(--success-gradient)';
        
        setTimeout(() => {
            exportBtn.innerHTML = originalContent;
            exportBtn.style.background = '';
            exportBtn.disabled = false;
        }, 3000);
    })
    .catch(error => {
        console.error('Error:', error);
        exportBtn.innerHTML = '<i class="fas fa-exclamation-triangle"></i> فشل التصدير';
        exportBtn.style.background = 'var(--danger-gradient)';
        
        setTimeout(() => {
            exportBtn.innerHTML = originalContent;
            exportBtn.style.background = '';
            exportBtn.disabled = false;
        }, 3000);
        
        alert('حدث خطأ أثناء تصدير الملف: ' + error.message);
    });
}

function getActiveFilters() {
    const filters = {};
    const form = document.querySelector('.filter-form');
    if (form) {
        const inputs = form.querySelectorAll('input, select');
        inputs.forEach(input => {
            if (input.value) {
                filters[input.name] = input.value;
            }
        });
    }
    return filters;
}

function getStatsData() {
    return {
        total_invoices: document.querySelector('.stat-primary h3')?.textContent || '0',
        paid_invoices: document.querySelector('.stat-success h3')?.textContent || '0',
        pending_invoices: document.querySelector('.stat-warning h3')?.textContent || '0',
        overdue_invoices: document.querySelector('.stat-danger h3')?.textContent || '0',
        total_amount: document.querySelector('.summary-card.total .summary-amount')?.textContent || '0 ر.س',
        collected_amount: document.querySelector('.summary-card.collected .summary-amount')?.textContent || '0 ر.س',
        pending_amount: document.querySelector('.summary-card.pending .summary-amount')?.textContent || '0 ر.س'
    };
}

function getInvoicesData() {
    const rows = document.querySelectorAll('.modern-data-table tbody tr');
    const invoices = [];
    
    rows.forEach(row => {
        const cells = row.querySelectorAll('td');
        if (cells.length >= 9) {
            invoices.push({
                invoice_number: cells[0]?.textContent?.trim() || '',
                customer: cells[1]?.textContent?.trim() || '',
                issue_date: cells[2]?.textContent?.trim() || '',
                due_date: cells[3]?.textContent?.trim() || '',
                total_amount: cells[4]?.textContent?.trim() || '',
                paid_amount: cells[5]?.textContent?.trim() || '',
                remaining_amount: cells[6]?.textContent?.trim() || '',
                payment_percentage: cells[7]?.textContent?.trim() || '',
                payment_status: cells[8]?.textContent?.trim() || ''
            });
        }
    });
    
    return invoices;
}

function printReport() {
    const printBtn = document.querySelector('.action-btn.btn-primary');
    const originalContent = printBtn.innerHTML;
    
    printBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري التحضير...';
    printBtn.disabled = true;
    
    // تحسين النافذة المخصصة للطباعة الاحترافية
    const printWindow = window.open('', '_blank');
    const currentDate = new Date().toLocaleDateString('ar-SA', {
        year: 'numeric',
        month: 'long',
        day: 'numeric',
        weekday: 'long'
    });
    
    const currentTime = new Date().toLocaleTimeString('ar-SA', {
        hour: '2-digit',
        minute: '2-digit'
    });
    
    // جمع بيانات الفلاتر النشطة
    const activeFilters = getActiveFilters();
    const filtersText = Object.keys(activeFilters).length > 0
        ? Object.entries(activeFilters).map(([key, value]) => `${getFilterLabel(key)}: ${value}`).join(' | ')
        : 'جميع الفواتير';
    
    const printContent = `
        <!DOCTYPE html>
        <html dir="rtl" lang="ar">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>تقرير المدفوعات التفصيلي - ${currentDate}</title>
            <style>
                @import url('https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;600;700;800&display=swap');
                
                * {
                    margin: 0;
                    padding: 0;
                    box-sizing: border-box;
                }
                
                @page {
                    margin: 1.5cm 1cm;
                    size: A4 portrait;
                }
                
                body {
                    font-family: 'Cairo', 'Segoe UI', Arial, sans-serif;
                    font-size: 10pt;
                    line-height: 1.3;
                    color: #1a1a1a;
                    background: white;
                    direction: rtl;
                    -webkit-print-color-adjust: exact;
                    print-color-adjust: exact;
                }
                
                .print-container {
                    width: 100%;
                    max-width: 210mm;
                    margin: 0 auto;
                }
                
                /* رأس التقرير المحسن والمبسط */
                .print-header {
                    text-align: center;
                    margin-bottom: 25pt;
                    padding: 15pt;
                    border: 2pt solid #2563eb;
                    border-radius: 8pt;
                    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
                    page-break-after: avoid;
                }
                
                .company-section {
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    margin-bottom: 12pt;
                    gap: 15pt;
                }
                
                .company-logo {
                    width: 50pt;
                    height: 50pt;
                    background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%);
                    border-radius: 50%;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    color: white;
                    font-size: 18pt;
                    font-weight: 800;
                    box-shadow: 0 3pt 6pt rgba(37, 99, 235, 0.3);
                }
                
                .company-info {
                    text-align: right;
                }
                
                .company-name {
                    font-size: 16pt;
                    font-weight: 800;
                    color: #1e293b;
                    margin-bottom: 3pt;
                    line-height: 1.2;
                }
                
                .company-subtitle {
                    font-size: 9pt;
                    color: #64748b;
                    font-weight: 500;
                }
                
                .report-title {
                    font-size: 14pt;
                    font-weight: 700;
                    color: #2563eb;
                    margin: 8pt 0 5pt 0;
                    text-decoration: underline;
                    text-decoration-color: #2563eb;
                    text-underline-offset: 3pt;
                }
                
                .report-meta {
                    display: grid;
                    grid-template-columns: 1fr 1fr 1fr;
                    gap: 8pt;
                    margin-top: 8pt;
                    font-size: 8pt;
                }
                
                .meta-item {
                    background: white;
                    padding: 4pt;
                    border-radius: 4pt;
                    border: 1pt solid #cbd5e1;
                    text-align: center;
                }
                
                .meta-label {
                    font-weight: 600;
                    color: #374151;
                    display: block;
                    margin-bottom: 1pt;
                }
                
                .meta-value {
                    color: #1e293b;
                    font-weight: 500;
                }
                
                /* قسم الفلاتر المبسط */
                .filters-section {
                    background: #f1f5f9;
                    padding: 6pt 10pt;
                    border-radius: 6pt;
                    margin-bottom: 15pt;
                    border: 1pt solid #cbd5e1;
                    page-break-inside: avoid;
                }
                
                .filters-title {
                    font-size: 9pt;
                    font-weight: 700;
                    color: #374151;
                    margin-bottom: 3pt;
                }
                
                .filters-content {
                    font-size: 8pt;
                    color: #1e293b;
                    font-weight: 500;
                }
                
                /* إحصائيات مبسطة */
                .stats-section {
                    display: grid;
                    grid-template-columns: repeat(4, 1fr);
                    gap: 6pt;
                    margin-bottom: 12pt;
                    page-break-inside: avoid;
                }
                
                .stat-card {
                    border: 1.5pt solid #e2e8f0;
                    border-radius: 6pt;
                    padding: 8pt;
                    text-align: center;
                    background: #fafafa;
                    position: relative;
                }
                
                .stat-card::before {
                    content: '';
                    position: absolute;
                    top: 0;
                    left: 0;
                    right: 0;
                    height: 2pt;
                    background: #2563eb;
                }
                
                .stat-card.success::before { background: #059669; }
                .stat-card.warning::before { background: #d97706; }
                .stat-card.danger::before { background: #dc2626; }
                
                .stat-value {
                    font-size: 12pt;
                    font-weight: 800;
                    color: #1e293b;
                    margin-bottom: 2pt;
                    line-height: 1;
                }
                
                .stat-label {
                    font-size: 7pt;
                    color: #64748b;
                    font-weight: 600;
                    line-height: 1.2;
                }
                
                /* القسم المالي المبسط */
                .financial-section {
                    display: grid;
                    grid-template-columns: repeat(3, 1fr);
                    gap: 8pt;
                    margin-bottom: 15pt;
                    page-break-inside: avoid;
                }
                
                .financial-card {
                    border: 2pt solid;
                    border-radius: 6pt;
                    padding: 10pt;
                    text-align: center;
                    position: relative;
                }
                
                .financial-card.total {
                    border-color: #2563eb;
                    background: linear-gradient(135deg, #eff6ff 0%, #dbeafe 100%);
                }
                .financial-card.collected {
                    border-color: #059669;
                    background: linear-gradient(135deg, #ecfdf5 0%, #d1fae5 100%);
                }
                .financial-card.pending {
                    border-color: #d97706;
                    background: linear-gradient(135deg, #fffbeb 0%, #fef3c7 100%);
                }
                
                .financial-amount {
                    font-size: 11pt;
                    font-weight: 800;
                    margin-bottom: 3pt;
                    line-height: 1;
                }
                
                .financial-card.total .financial-amount { color: #2563eb; }
                .financial-card.collected .financial-amount { color: #059669; }
                .financial-card.pending .financial-amount { color: #d97706; }
                
                .financial-label {
                    font-size: 8pt;
                    color: #374151;
                    font-weight: 600;
                }
                
                /* قسم الجدول المحسن */
                .table-section {
                    margin-top: 12pt;
                    page-break-inside: auto;
                }
                
                .table-header {
                    background: linear-gradient(135deg, #1e293b 0%, #374151 100%);
                    color: white;
                    padding: 6pt 10pt;
                    border-radius: 6pt 6pt 0 0;
                    text-align: center;
                    page-break-after: avoid;
                }
                
                .table-title {
                    font-size: 10pt;
                    font-weight: 700;
                    margin: 0;
                }
                
                .data-table {
                    width: 100%;
                    border-collapse: collapse;
                    font-size: 6pt;
                    border: 1pt solid #374151;
                    page-break-inside: auto;
                }
                
                .data-table th {
                    background: linear-gradient(135deg, #374151 0%, #4b5563 100%);
                    color: white;
                    padding: 4pt 2pt;
                    text-align: center;
                    font-weight: 700;
                    border: 1pt solid #1f2937;
                    page-break-after: avoid;
                    font-size: 6pt;
                    line-height: 1.1;
                }
                
                .data-table td {
                    padding: 3pt 1pt;
                    border: 1pt solid #d1d5db;
                    text-align: center;
                    vertical-align: middle;
                    page-break-inside: avoid;
                    font-size: 5.5pt;
                    line-height: 1.1;
                }
                
                .data-table tbody tr:nth-child(even) {
                    background: #f9fafb;
                }
                
                .data-table tbody tr:nth-child(odd) {
                    background: white;
                }
                
                /* حالات السداد مبسطة */
                .status-badge {
                    padding: 1pt 3pt;
                    border-radius: 2pt;
                    font-size: 5pt;
                    font-weight: 700;
                    border: 1pt solid;
                    display: inline-block;
                }
                
                .status-paid {
                    background: #dcfce7;
                    color: #166534;
                    border-color: #16a34a;
                }
                
                .status-unpaid {
                    background: #fee2e2;
                    color: #991b1b;
                    border-color: #dc2626;
                }
                
                .status-partial {
                    background: #fef3c7;
                    color: #92400e;
                    border-color: #f59e0b;
                }
                
                /* شريط التقدم مبسط */
                .progress-container {
                    display: flex;
                    flex-direction: column;
                    align-items: center;
                    gap: 1pt;
                }
                
                .progress-bar {
                    width: 25pt;
                    height: 3pt;
                    background: #e5e7eb;
                    border-radius: 1pt;
                    border: 0.5pt solid #d1d5db;
                    overflow: hidden;
                }
                
                .progress-fill {
                    height: 100%;
                    border-radius: 1pt;
                }
                
                .progress-fill.success { background: #22c55e; }
                .progress-fill.warning { background: #f59e0b; }
                .progress-fill.danger { background: #ef4444; }
                
                .progress-text {
                    font-size: 5pt;
                    font-weight: 700;
                    color: #374151;
                }
                
                /* المبالغ مبسطة */
                .amount-cell {
                    font-family: 'Courier New', monospace;
                    font-weight: 700;
                    font-size: 5.5pt;
                    text-align: left;
                }
                
                .amount-cell.total { color: #1e293b; }
                .amount-cell.success { color: #166534; }
                .amount-cell.warning { color: #92400e; }
                
                /* الروابط والشارات */
                .invoice-link {
                    color: #1e293b;
                    text-decoration: none;
                    font-weight: 700;
                    font-size: 5.5pt;
                }
                
                .badge {
                    padding: 0.5pt 2pt;
                    border-radius: 1pt;
                    font-size: 4.5pt;
                    font-weight: 700;
                    border: 0.5pt solid;
                    display: inline-block;
                    margin-top: 0.5pt;
                }
                
                .badge-primary {
                    background: #dbeafe;
                    color: #1e40af;
                    border-color: #3b82f6;
                }
                
                .badge-secondary {
                    background: #f1f5f9;
                    color: #475569;
                    border-color: #64748b;
                }
                
                /* التواريخ */
                .date-text {
                    font-size: 5.5pt;
                    color: #374151;
                    line-height: 1.1;
                }
                
                .overdue-text {
                    color: #dc2626;
                    font-size: 4.5pt;
                    font-weight: 700;
                    display: block;
                    margin-top: 0.5pt;
                }
                
                /* أسماء العملاء */
                .customer-name {
                    font-weight: 700;
                    color: #1e293b;
                    font-size: 5.5pt;
                }
                
                /* ذيل التقرير المحسن */
                .print-footer {
                    margin-top: 15pt;
                    padding-top: 8pt;
                    border-top: 2pt solid #e5e7eb;
                    text-align: center;
                    font-size: 7pt;
                    color: #64748b;
                    page-break-inside: avoid;
                }
                
                .footer-grid {
                    display: grid;
                    grid-template-columns: repeat(3, 1fr);
                    gap: 8pt;
                    margin-bottom: 6pt;
                }
                
                .footer-item {
                    text-align: center;
                    padding: 3pt;
                    background: #f8fafc;
                    border-radius: 3pt;
                    border: 1pt solid #e2e8f0;
                }
                
                .footer-label {
                    font-weight: 700;
                    color: #374151;
                    margin-bottom: 1pt;
                    font-size: 6pt;
                }
                
                .footer-value {
                    color: #1e293b;
                    font-size: 6pt;
                }
                
                /* حالة فارغة */
                .empty-state {
                    text-align: center;
                    padding: 20pt;
                    color: #6b7280;
                    font-size: 9pt;
                }
                
                .empty-icon {
                    font-size: 18pt;
                    margin-bottom: 6pt;
                    color: #d1d5db;
                }
                
                /* تحسينات الطباعة */
                @media print {
                    body { -webkit-print-color-adjust: exact; print-color-adjust: exact; }
                    * { color-adjust: exact; -webkit-print-color-adjust: exact; }
                }
            </style>
        </head>
        <body>
            <div class="print-container">
                <!-- رأس التقرير المحسن -->
                <div class="print-header">
                    <div class="company-section">
                        <div class="company-logo">${getCompanyLogo()}</div>
                        <div class="company-info">
                            <div class="company-name">${getCompanyName()}</div>
                            <div class="company-subtitle">${getCompanySubtitle()}</div>
                        </div>
                    </div>
                    <div class="report-title">تقرير المدفوعات التفصيلي</div>
                    <div class="report-meta">
                        <div class="meta-item">
                            <div class="meta-label">التاريخ</div>
                            <div class="meta-value">${currentDate}</div>
                        </div>
                        <div class="meta-item">
                            <div class="meta-label">الوقت</div>
                            <div class="meta-value">${currentTime}</div>
                        </div>
                        <div class="meta-item">
                            <div class="meta-label">الصفحة</div>
                            <div class="meta-value">1 من 1</div>
                        </div>
                    </div>
                </div>
                
                <!-- فلاتر البحث -->
                <div class="filters-section">
                    <div class="filters-title">🔍 معايير التقرير:</div>
                    <div class="filters-content">${filtersText}</div>
                </div>
                
                <!-- الإحصائيات المبسطة -->
                <div class="stats-section">
                    <div class="stat-card">
                        <div class="stat-value">${getStatValue('.stat-card.primary .stat-value')}</div>
                        <div class="stat-label">إجمالي الفواتير</div>
                    </div>
                    <div class="stat-card success">
                        <div class="stat-value">${getStatValue('.stat-card.success .stat-value')}</div>
                        <div class="stat-label">فواتير مدفوعة</div>
                    </div>
                    <div class="stat-card warning">
                        <div class="stat-value">${getStatValue('.stat-card.warning .stat-value')}</div>
                        <div class="stat-label">فواتير معلقة</div>
                    </div>
                    <div class="stat-card danger">
                        <div class="stat-value">${getStatValue('.stat-card.danger .stat-value')}</div>
                        <div class="stat-label">فواتير متأخرة</div>
                    </div>
                </div>
                
                <!-- الملخص المالي المبسط -->
                <div class="financial-section">
                    <div class="financial-card total">
                        <div class="financial-amount">${getFinancialValue('total')}</div>
                        <div class="financial-label">💰 إجمالي المبالغ</div>
                    </div>
                    <div class="financial-card collected">
                        <div class="financial-amount">${getFinancialValue('collected')}</div>
                        <div class="financial-label">✅ المبالغ المحصلة</div>
                    </div>
                    <div class="financial-card pending">
                        <div class="financial-amount">${getFinancialValue('pending')}</div>
                        <div class="financial-label">⏳ المبالغ المعلقة</div>
                    </div>
                </div>
                
                <!-- جدول البيانات المحسن -->
                <div class="table-section">
                    <div class="table-header">
                        <div class="table-title">📊 تفاصيل الفواتير والمدفوعات</div>
                    </div>
                    ${getEnhancedTableForPrint()}
                </div>
                
                <!-- ذيل التقرير المحسن -->
                <div class="print-footer">
                    <div class="footer-grid">
                        <div class="footer-item">
                            <div class="footer-label">عدد السجلات</div>
                            <div class="footer-value">${document.querySelectorAll('.modern-data-table tbody tr').length || 0}</div>
                        </div>
                        <div class="footer-item">
                            <div class="footer-label">حالة التقرير</div>
                            <div class="footer-value">مكتمل ✓</div>
                        </div>
                        <div class="footer-item">
                            <div class="footer-label">معلومات الاتصال</div>
                            <div class="footer-value">${getCompanyContact()}</div>
                        </div>
                    </div>
                    <p style="margin: 4pt 0; font-weight: 600;">📄 تم إنشاء هذا التقرير بواسطة ${getCompanyName()}</p>
                    <p style="margin: 2pt 0; font-size: 6pt;">📍 ${getCompanyAddress()}</p>
                    <p style="margin: 0; font-size: 6pt;">© ${new Date().getFullYear()} - جميع الحقوق محفوظة - طُبع في ${new Date().toLocaleString('ar-SA')}</p>
                </div>
            </div>
        </body>
        </html>
    `;
    
    printWindow.document.write(printContent);
    printWindow.document.close();
    
    printWindow.onload = function() {
        setTimeout(() => {
            printWindow.focus();
            printWindow.print();
            
            printBtn.innerHTML = '<i class="fas fa-check"></i> تم التحضير للطباعة!';
            printBtn.style.background = 'var(--success-gradient)';
            
            setTimeout(() => {
                printWindow.close();
                printBtn.innerHTML = originalContent;
                printBtn.style.background = '';
                printBtn.disabled = false;
            }, 1500);
        }, 500);
    };
}

// دالة مساعدة لجلب قيم الإحصائيات
function getStatValue(selector) {
    const element = document.querySelector(selector);
    return element ? element.textContent.trim() : '0';
}

// دالة مساعدة لجلب القيم المالية
function getFinancialValue(type) {
    let value = '0';
    switch(type) {
        case 'total':
            const totalEl = document.querySelector('.stat-card.primary .stat-value');
            value = totalEl ? totalEl.textContent.trim() : '0';
            break;
        case 'collected':
            const collectedEl = document.querySelector('.stat-card.success .stat-value');
            value = collectedEl ? collectedEl.textContent.trim() : '0';
            break;
        case 'pending':
            const pendingEl = document.querySelector('.stat-card.warning .stat-value');
            value = pendingEl ? pendingEl.textContent.trim() : '0';
            break;
    }
    return value + ' ر.س';
}

// دالة مساعدة لجلب تسمية الفلتر
function getFilterLabel(key) {
    const labels = {
        'payment_status': 'حالة السداد',
        'date_from': 'من تاريخ',
        'date_to': 'إلى تاريخ',
        'customer_id': 'العميل'
    };
    return labels[key] || key;
}

// دوال مساعدة لمعلومات الشركة
function getCompanyName() {
    @if(isset($company) && $company)
        return '{{ $company->name }}';
    @else
        return 'شركة نظم المحاسبة المتقدمة';
    @endif
}

function getCompanySubtitle() {
    @if(isset($company) && $company)
        return 'الرقم الضريبي: {{ $company->vat_number ?? "غير محدد" }} | السجل التجاري: {{ $company->cr_number ?? "غير محدد" }}';
    @else
        return 'نظام إدارة الفواتير والمدفوعات';
    @endif
}

function getCompanyLogo() {
    @if(isset($company) && $company && $company->hasLogo())
        return '🏢';
    @else
        return '💰';
    @endif
}

function getCompanyAddress() {
    @if(isset($company) && $company)
        return '{{ $company->full_address ?? "العنوان غير محدد" }}';
    @else
        return 'الرياض، المملكة العربية السعودية';
    @endif
}

function getCompanyContact() {
    @if(isset($company) && $company)
        let contact = '';
        @if($company->phone)
            contact += 'الهاتف: {{ $company->phone }}';
        @endif
        @if($company->email)
            if(contact) contact += ' | ';
            contact += 'البريد: {{ $company->email }}';
        @endif
        @if($company->website)
            if(contact) contact += ' | ';
            contact += 'الموقع: {{ $company->website }}';
        @endif
        return contact || 'معلومات الاتصال غير متوفرة';
    @else
        return 'info@company.com | www.company.com';
    @endif
}

// دالة محسنة لإنشاء الجدول للطباعة
function getEnhancedTableForPrint() {
    const table = document.querySelector('.modern-data-table');
    if (!table) {
        return `
            <div class="empty-state">
                <div class="empty-icon">📄</div>
                <h4>لا توجد بيانات للعرض</h4>
                <p>لم يتم العثور على فواتير تطابق معايير البحث المحددة</p>
            </div>
        `;
    }
    
    const rows = table.querySelectorAll('tbody tr');
    if (rows.length === 0) {
        return getEmptyTableForPrint();
    }
    
    let tableHTML = `
        <table class="data-table">
            <thead>
                <tr>
                    <th style="width: 12%;">رقم الفاتورة</th>
                    <th style="width: 15%;">العميل</th>
                    <th style="width: 10%;">تاريخ الإصدار</th>
                    <th style="width: 10%;">تاريخ الاستحقاق</th>
                    <th style="width: 12%;">المبلغ الإجمالي</th>
                    <th style="width: 12%;">المبلغ المدفوع</th>
                    <th style="width: 12%;">المبلغ المتبقي</th>
                    <th style="width: 8%;">نسبة السداد</th>
                    <th style="width: 9%;">حالة السداد</th>
                </tr>
            </thead>
            <tbody>
    `;
    
    rows.forEach((row, index) => {
        const cells = row.querySelectorAll('td');
        if (cells.length >= 9) {
            const invoiceNumber = extractInvoiceNumber(cells[0]);
            const customerName = extractCustomerName(cells[1]);
            const issueDate = cells[2]?.textContent?.trim() || '';
            const dueDate = extractDueDate(cells[3]);
            const totalAmount = cells[4]?.textContent?.trim() || '';
            const paidAmount = cells[5]?.textContent?.trim() || '';
            const remainingAmount = cells[6]?.textContent?.trim() || '';
            const progressInfo = extractProgressInfo(cells[7]);
            const statusInfo = extractStatusInfo(cells[8]);
            
            tableHTML += `
                <tr>
                    <td>
                        <div style="font-weight: 700; line-height: 1.1;">${invoiceNumber.number}</div>
                        ${invoiceNumber.badge}
                    </td>
                    <td class="customer-name">${customerName}</td>
                    <td class="date-text">${issueDate}</td>
                    <td class="date-text">
                        ${dueDate.date}
                        ${dueDate.isOverdue ? '<div class="overdue-text">متأخرة ⚠️</div>' : ''}
                    </td>
                    <td class="amount-cell total">${totalAmount}</td>
                    <td class="amount-cell success">${paidAmount}</td>
                    <td class="amount-cell warning">${remainingAmount}</td>
                    <td>
                        <div class="progress-container">
                            <div class="progress-bar">
                                <div class="progress-fill ${progressInfo.class}" style="width: ${progressInfo.percentage}%"></div>
                            </div>
                            <div class="progress-text">${progressInfo.text}</div>
                        </div>
                    </td>
                    <td>
                        <span class="status-badge ${statusInfo.class}">${statusInfo.text}</span>
                    </td>
                </tr>
            `;
        }
    });
    
    tableHTML += '</tbody></table>';
    return tableHTML;
}

// دوال مساعدة لاستخراج البيانات بشكل محسن
function extractInvoiceNumber(cell) {
    const link = cell.querySelector('.invoice-link');
    const badge = cell.querySelector('.modern-badge');
    
    return {
        number: link ? link.textContent.trim() : cell.textContent.trim(),
        badge: badge ? `<div class="badge ${badge.classList.contains('badge-primary') ? 'badge-primary' : 'badge-secondary'}">${badge.textContent.trim()}</div>` : ''
    };
}

function extractCustomerName(cell) {
    const nameElement = cell.querySelector('.customer-name');
    return nameElement ? nameElement.textContent.trim() : cell.textContent.trim();
}

function extractDueDate(cell) {
    const dateElement = cell.querySelector('.date');
    const overdueElement = cell.querySelector('.overdue-indicator');
    
    return {
        date: dateElement ? dateElement.textContent.trim() : cell.textContent.trim(),
        isOverdue: !!overdueElement
    };
}

function extractProgressInfo(cell) {
    const progressText = cell.querySelector('.progress-text');
    const progressFill = cell.querySelector('.progress-fill');
    
    let progressClass = 'danger';
    let percentage = '0';
    
    if (progressFill) {
        if (progressFill.classList.contains('success')) progressClass = 'success';
        else if (progressFill.classList.contains('warning')) progressClass = 'warning';
        
        const style = progressFill.style.width;
        percentage = style.replace('%', '') || '0';
    }
    
    return {
        class: progressClass,
        percentage: percentage,
        text: progressText ? progressText.textContent.trim() : '0%'
    };
}

function extractStatusInfo(cell) {
    const statusElement = cell.querySelector('.modern-status-badge');
    let statusClass = 'status-unpaid';
    let statusText = '';
    
    if (statusElement) {
        statusText = statusElement.textContent.trim();
        if (statusElement.classList.contains('status-paid')) statusClass = 'status-paid';
        else if (statusElement.classList.contains('status-partial')) statusClass = 'status-partial';
    }
    
    return {
        class: statusClass,
        text: statusText || 'غير محدد'
    };
}

function getTableForPrint() {
    const table = document.querySelector('.modern-data-table');
    if (!table) return '<p>لا توجد بيانات للعرض</p>';
    
    const rows = table.querySelectorAll('tbody tr');
    let tableHTML = `
        <table class="data-table">
            <thead>
                <tr>
                    <th>رقم الفاتورة</th>
                    <th>العميل</th>
                    <th>تاريخ الإصدار</th>
                    <th>تاريخ الاستحقاق</th>
                    <th>المبلغ الإجمالي</th>
                    <th>المبلغ المدفوع</th>
                    <th>المبلغ المتبقي</th>
                    <th>نسبة السداد</th>
                    <th>حالة السداد</th>
                </tr>
            </thead>
            <tbody>
    `;
    
    rows.forEach(row => {
        const cells = row.querySelectorAll('td');
        if (cells.length >= 9) {
            const statusText = cells[8]?.textContent?.trim() || '';
            let statusClass = 'status-unpaid';
            if (statusText.includes('مدفوعة بالكامل') || statusText.includes('مدفوعة')) {
                statusClass = 'status-paid';
            } else if (statusText.includes('جزئياً')) {
                statusClass = 'status-partial';
            }
            
            tableHTML += `
                <tr>
                    <td>${cells[0]?.textContent?.trim() || ''}</td>
                    <td>${cells[1]?.textContent?.trim() || ''}</td>
                    <td>${cells[2]?.textContent?.trim() || ''}</td>
                    <td>${cells[3]?.textContent?.trim() || ''}</td>
                    <td>${cells[4]?.textContent?.trim() || ''}</td>
                    <td>${cells[5]?.textContent?.trim() || ''}</td>
                    <td>${cells[6]?.textContent?.trim() || ''}</td>
                    <td>${cells[7]?.textContent?.trim() || ''}</td>
                    <td><span class="${statusClass}">${statusText}</span></td>
                </tr>
            `;
        }
    });
    
    tableHTML += '</tbody></table>';
    return tableHTML;
}
</script>
@endpush
