@extends('layouts.app')

@section('title', 'الفواتير')
@section('page-title', 'إدارة الفواتير')

@push('styles')
<style>
/* ========== Modern Simplified Design ========== */
:root {
    --primary: #4f46e5;
    --primary-light: #818cf8;
    --success: #10b981;
    --warning: #f59e0b;
    --danger: #ef4444;
    --info: #06b6d4;
    --gray-50: #f9fafb;
    --gray-100: #f3f4f6;
    --gray-200: #e5e7eb;
    --gray-300: #d1d5db;
    --gray-400: #9ca3af;
    --gray-500: #6b7280;
    --gray-600: #4b5563;
    --gray-700: #374151;
    --gray-800: #1f2937;
    --gray-900: #111827;
    --white: #ffffff;
    --border-radius: 12px;
    --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
    --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
    --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
}

.page-container {
    padding: 2rem;
    max-width: 1400px;
    margin: 0 auto;
}

/* ========== Header Section ========== */
.page-header {
    background: linear-gradient(135deg, var(--primary) 0%, var(--primary-light) 100%);
    border-radius: var(--border-radius);
    padding: 2.5rem;
    margin-bottom: 2rem;
    color: white;
    position: relative;
    overflow: hidden;
}

.page-header::before {
    content: '';
    position: absolute;
    top: 0;
    right: 0;
    width: 200px;
    height: 200px;
    background: rgba(255, 255, 255, 0.1);
    border-radius: 50%;
    transform: translate(50px, -50px);
}

.header-content {
    position: relative;
    z-index: 1;
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 2rem;
}

.header-title {
    flex: 1;
}

.header-title h1 {
    font-size: 2.5rem;
    font-weight: 800;
    margin: 0 0 0.5rem 0;
    text-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.header-title p {
    font-size: 1.1rem;
    opacity: 0.9;
    margin: 0;
    font-weight: 300;
}

.header-actions {
    display: flex;
    gap: 1rem;
    flex-wrap: wrap;
}

.btn-primary {
    background: rgba(255, 255, 255, 0.2);
    border: 1px solid rgba(255, 255, 255, 0.3);
    color: white;
    padding: 0.75rem 1.5rem;
    border-radius: var(--border-radius);
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
    backdrop-filter: blur(10px);
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.btn-primary:hover {
    background: rgba(255, 255, 255, 0.3);
    transform: translateY(-2px);
    color: white;
    box-shadow: var(--shadow-lg);
}

.btn-primary.solid {
    background: white;
    color: var(--primary);
    border-color: white;
}

.btn-primary.solid:hover {
    background: var(--gray-100);
    color: var(--primary);
}

/* ========== Enhanced Professional Stats Cards ========== */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(5, 1fr);
    gap: 2rem;
    margin-bottom: 2.5rem;
}

.stat-card {
    background: var(--white);
    border-radius: 16px;
    padding: 0;
    border: 1px solid var(--gray-200);
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
    overflow: hidden;
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
}

.stat-card:hover {
    transform: translateY(-8px) scale(1.02);
    box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
    border-color: var(--gray-300);
}

/* Professional Gradient Backgrounds */
.stat-card.primary-gradient::before {
    background: linear-gradient(135deg, var(--primary) 0%, #6366f1 100%);
}

.stat-card.info-gradient::before {
    background: linear-gradient(135deg, var(--info) 0%, #0891b2 100%);
}

.stat-card.success-gradient::before {
    background: linear-gradient(135deg, var(--success) 0%, #059669 100%);
}

.stat-card.warning-gradient::before {
    background: linear-gradient(135deg, var(--warning) 0%, #d97706 100%);
}

.stat-card.danger-gradient::before {
    background: linear-gradient(135deg, var(--danger) 0%, #dc2626 100%);
}

.stat-card.total-gradient::before {
    background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%);
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 6px;
    background: var(--primary);
}

/* Card Header */
.stat-card-header {
    padding: 1.5rem 1.5rem 0 1.5rem;
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 1rem;
}

.stat-icon-wrapper {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.stat-icon {
    width: 56px;
    height: 56px;
    border-radius: 14px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.75rem;
    color: white;
    background: var(--primary);
    box-shadow: 0 8px 16px rgba(0, 0, 0, 0.15);
    position: relative;
}

.stat-icon.primary { background: linear-gradient(135deg, var(--primary), #6366f1); }
.stat-icon.info { background: linear-gradient(135deg, var(--info), #0891b2); }
.stat-icon.success { background: linear-gradient(135deg, var(--success), #059669); }
.stat-icon.warning { background: linear-gradient(135deg, var(--warning), #d97706); }
.stat-icon.danger { background: linear-gradient(135deg, var(--danger), #dc2626); }
.stat-icon.total { background: linear-gradient(135deg, #8b5cf6, #7c3aed); }

.stat-icon::after {
    content: '';
    position: absolute;
    inset: 0;
    border-radius: inherit;
    background: rgba(255, 255, 255, 0.2);
    opacity: 0;
    transition: opacity 0.3s ease;
}

.stat-card:hover .stat-icon::after {
    opacity: 1;
}

/* Trend Indicators */
.stat-trend {
    display: flex;
    align-items: center;
    gap: 0.25rem;
    padding: 0.375rem 0.75rem;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 700;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.stat-trend.positive {
    background: rgba(16, 185, 129, 0.1);
    color: var(--success);
    border: 1px solid rgba(16, 185, 129, 0.2);
}

.stat-trend.neutral {
    background: rgba(107, 114, 128, 0.1);
    color: var(--gray-600);
    border: 1px solid rgba(107, 114, 128, 0.2);
}

.stat-trend.negative {
    background: rgba(239, 68, 68, 0.1);
    color: var(--danger);
    border: 1px solid rgba(239, 68, 68, 0.2);
}

/* Card Badge */
.stat-badge {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    background: rgba(255, 255, 255, 0.9);
    border-radius: 20px;
    font-size: 0.75rem;
    font-weight: 600;
    color: var(--gray-700);
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
    backdrop-filter: blur(10px);
}

/* Card Content */
.stat-content {
    padding: 0 1.5rem 1.5rem 1.5rem;
}

.stat-value {
    font-size: 3rem;
    font-weight: 900;
    color: var(--gray-900);
    margin-bottom: 0.5rem;
    line-height: 1;
    background: linear-gradient(135deg, var(--gray-900), var(--gray-700));
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.stat-label {
    font-size: 1.1rem;
    font-weight: 700;
    color: var(--gray-800);
    margin-bottom: 0.5rem;
    line-height: 1.2;
}

.stat-description {
    font-size: 0.9rem;
    color: var(--gray-600);
    line-height: 1.4;
    margin-bottom: 1rem;
}

.stat-amount {
    font-size: 1.1rem;
    font-weight: 700;
    color: var(--primary);
    background: rgba(79, 70, 229, 0.1);
    padding: 0.5rem 1rem;
    border-radius: 8px;
    text-align: center;
    border: 1px solid rgba(79, 70, 229, 0.2);
}

.stat-amount.highlight {
    background: linear-gradient(135deg, #8b5cf6, #7c3aed);
    color: white;
    border: none;
    box-shadow: 0 4px 12px rgba(139, 92, 246, 0.3);
}

/* Professional Hover Effects */
.stat-card::after {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.4), transparent);
    transition: left 0.6s ease;
}

.stat-card:hover::after {
    left: 100%;
}

/* Enhanced Animations */
@keyframes cardFloat {
    0%, 100% { transform: translateY(0px); }
    50% { transform: translateY(-5px); }
}

.stat-card {
    animation: cardFloat 6s ease-in-out infinite;
}

.stat-card:nth-child(2) { animation-delay: -1s; }
.stat-card:nth-child(3) { animation-delay: -2s; }
.stat-card:nth-child(4) { animation-delay: -3s; }
.stat-card:nth-child(5) { animation-delay: -4s; }
.stat-card:nth-child(6) { animation-delay: -5s; }

/* ========== Main Content Card ========== */
.content-card {
    background: var(--white);
    border-radius: var(--border-radius);
    border: 1px solid var(--gray-200);
    overflow: hidden;
    box-shadow: var(--shadow-sm);
}

.content-header {
    background: var(--gray-50);
    padding: 2rem;
    border-bottom: 1px solid var(--gray-200);
}

.content-header-main {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1.5rem;
    flex-wrap: wrap;
    gap: 1rem;
}

.content-title h2 {
    font-size: 1.5rem;
    font-weight: 700;
    color: var(--gray-900);
    margin: 0 0 0.25rem 0;
}

.content-title p {
    color: var(--gray-600);
    margin: 0;
}

.content-controls {
    display: flex;
    gap: 1rem;
    align-items: center;
    flex-wrap: wrap;
}

.btn-secondary {
    background: var(--white);
    border: 1px solid var(--gray-300);
    color: var(--gray-700);
    padding: 0.5rem 1rem;
    border-radius: 8px;
    font-size: 0.875rem;
    font-weight: 500;
    transition: all 0.2s ease;
    text-decoration: none;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    cursor: pointer;
}

.btn-secondary:hover {
    background: var(--gray-50);
    border-color: var(--gray-400);
    color: var(--gray-800);
}

.btn-secondary.active {
    background: var(--primary);
    color: white;
    border-color: var(--primary);
}

/* ========== Filters ========== */
.filters-section {
    background: var(--gray-50);
    padding: 2rem;
    border-bottom: 1px solid var(--gray-200);
    display: none;
}

.filters-section.show { display: block; }

.filters-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
    margin-bottom: 1.5rem;
}

.filter-group label {
    display: block;
    font-size: 0.875rem;
    font-weight: 600;
    color: var(--gray-700);
    margin-bottom: 0.5rem;
}

.filter-input {
    width: 100%;
    padding: 0.75rem;
    border: 1px solid var(--gray-300);
    border-radius: 8px;
    font-size: 0.875rem;
    transition: all 0.2s ease;
    background: var(--white);
}

.filter-input:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
}

.search-wrapper {
    position: relative;
}

.search-input {
    padding-left: 2.5rem;
}

.search-icon {
    position: absolute;
    right: 0.75rem;
    top: 50%;
    transform: translateY(-50%);
    color: var(--gray-400);
}

.filter-actions {
    display: flex;
    gap: 1rem;
    padding-top: 1.5rem;
    border-top: 1px solid var(--gray-200);
}

/* ========== Table ========== */
.table-container {
    overflow-x: auto;
}

.invoice-table {
    width: 100%;
    border-collapse: collapse;
}

.invoice-table th {
    background: var(--gray-50);
    padding: 1rem 1.5rem;
    text-align: right;
    font-weight: 600;
    font-size: 0.875rem;
    color: var(--gray-700);
    border-bottom: 1px solid var(--gray-200);
    white-space: nowrap;
}

.invoice-table td {
    padding: 1.25rem 1.5rem;
    border-bottom: 1px solid var(--gray-100);
    vertical-align: middle;
}

.invoice-table tbody tr {
    transition: background-color 0.2s ease;
}

.invoice-table tbody tr:hover {
    background-color: var(--gray-50);
}

/* ========== Table Content Styles ========== */
.invoice-number {
    font-weight: 600;
    color: var(--gray-900);
    margin-bottom: 0.25rem;
}

.invoice-uuid {
    font-size: 0.75rem;
    color: var(--gray-500);
    font-family: 'Courier New', monospace;
}

.customer-name {
    font-weight: 500;
    color: var(--gray-800);
    margin-bottom: 0.25rem;
}

.customer-vat {
    font-size: 0.75rem;
    color: var(--gray-500);
    font-family: 'Courier New', monospace;
}

.amount-main {
    font-weight: 600;
    color: var(--gray-900);
    margin-bottom: 0.25rem;
}

.amount-tax {
    font-size: 0.75rem;
    color: var(--gray-500);
}

/* ========== Status Badges ========== */
.status-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.375rem;
    padding: 0.5rem 0.75rem;
    border-radius: 6px;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.025em;
}

.status-badge.primary {
    background: rgba(79, 70, 229, 0.1);
    color: var(--primary);
}

.status-badge.success {
    background: rgba(16, 185, 129, 0.1);
    color: var(--success);
}

.status-badge.warning {
    background: rgba(245, 158, 11, 0.1);
    color: var(--warning);
}

.status-badge.danger {
    background: rgba(239, 68, 68, 0.1);
    color: var(--danger);
}

.status-badge.info {
    background: rgba(6, 182, 212, 0.1);
    color: var(--info);
}

.status-badge.secondary {
    background: rgba(107, 114, 128, 0.1);
    color: var(--gray-600);
}

/* ========== Action Buttons ========== */
.action-menu {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    justify-content: center;
    flex-wrap: wrap;
}

.action-btn {
    width: 36px;
    height: 36px;
    border-radius: 8px;
    border: 1px solid var(--gray-300);
    background: var(--white);
    color: var(--gray-500);
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 0.3s ease;
    text-decoration: none;
    cursor: pointer;
    font-size: 0.875rem;
    position: relative;
    overflow: hidden;
}

.action-btn:hover {
    background: var(--gray-50);
    border-color: var(--gray-400);
    color: var(--gray-700);
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
}

.action-btn:active {
    transform: translateY(0);
}

/* Action Button Solid Background Colors */
.action-btn.view {
    background: #3b82f6;
    color: white;
    border-color: #2563eb;
}

.action-btn.view:hover {
    background: #2563eb;
    color: white;
    border-color: #1d4ed8;
}

.action-btn.edit {
    background: #f59e0b;
    color: white;
    border-color: #d97706;
}

.action-btn.edit:hover {
    background: #d97706;
    color: white;
    border-color: #b45309;
}

.action-btn.pdf {
    background: #ef4444;
    color: white;
    border-color: #dc2626;
}

.action-btn.pdf:hover {
    background: #dc2626;
    color: white;
    border-color: #b91c1c;
}

.action-btn.send {
    background: #10b981;
    color: white;
    border-color: #059669;
}

.action-btn.send:hover {
    background: #059669;
    color: white;
    border-color: #047857;
}

.action-btn.whatsapp {
    background: #25d366;
    color: white;
    border-color: #128c7e;
}

.action-btn.whatsapp:hover {
    background: #128c7e;
    color: white;
    border-color: #075e54;
}

.action-btn.email {
    background: #ea4335;
    color: white;
    border-color: #d93025;
}

.action-btn.email:hover {
    background: #d93025;
    color: white;
    border-color: #b52d20;
}

.action-btn.disabled {
    opacity: 0.5;
    cursor: not-allowed;
    pointer-events: none;
}

/* Action button tooltips */
.action-btn::before {
    content: attr(title);
    position: absolute;
    bottom: 100%;
    left: 50%;
    transform: translateX(-50%);
    background: var(--gray-900);
    color: white;
    padding: 0.5rem 0.75rem;
    border-radius: 6px;
    font-size: 0.75rem;
    white-space: nowrap;
    opacity: 0;
    visibility: hidden;
    transition: all 0.3s ease;
    z-index: 1000;
    margin-bottom: 5px;
}

.action-btn::after {
    content: '';
    position: absolute;
    bottom: 100%;
    left: 50%;
    transform: translateX(-50%);
    border: 5px solid transparent;
    border-top-color: var(--gray-900);
    opacity: 0;
    visibility: hidden;
    transition: all 0.3s ease;
}

.action-btn:hover::before,
.action-btn:hover::after {
    opacity: 1;
    visibility: visible;
}

/* Additional professional enhancements */
.action-btn.loading {
    pointer-events: none;
    opacity: 0.7;
}

.action-btn.loading i {
    animation: spin 1s linear infinite;
}

@keyframes spin {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}

/* Professional hover effects for buttons */
.btn-primary,
.btn-secondary {
    position: relative;
    overflow: hidden;
}

.btn-primary::before,
.btn-secondary::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.2), transparent);
    transition: left 0.5s;
}

.btn-primary:hover::before,
.btn-secondary:hover::before {
    left: 100%;
}

/* Enhanced table row animations */
.invoice-table tbody tr {
    position: relative;
}

.invoice-table tbody tr::before {
    content: '';
    position: absolute;
    left: 0;
    top: 0;
    bottom: 0;
    width: 3px;
    background: var(--primary);
    transform: scaleY(0);
    transition: transform 0.3s ease;
}

.invoice-table tbody tr:hover::before {
    transform: scaleY(1);
}

/* ========== Empty State ========== */
.empty-state {
    text-align: center;
    padding: 4rem 2rem;
    color: var(--gray-500);
}

.empty-icon {
    width: 80px;
    height: 80px;
    margin: 0 auto 1.5rem auto;
    background: var(--gray-100);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 2rem;
    color: var(--gray-400);
}

.empty-state h3 {
    font-size: 1.25rem;
    font-weight: 600;
    color: var(--gray-700);
    margin: 0 0 0.5rem 0;
}

.empty-state p {
    margin: 0 0 2rem 0;
}

/* ========== Simple Pagination Design ========== */
.pagination-wrapper {
    padding: 2rem;
    background: var(--gray-50);
    border-top: 1px solid var(--gray-200);
    display: flex;
    justify-content: center;
    align-items: center;
}

.simple-pagination {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    direction: ltr; /* Keep pagination LTR for number consistency */
}

.pagination-item {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 44px;
    height: 44px;
    border-radius: 10px;
    border: 1px solid var(--gray-300);
    background: var(--white);
    color: var(--gray-700);
    text-decoration: none;
    font-weight: 500;
    font-size: 0.875rem;
    transition: all 0.3s ease;
    cursor: pointer;
    position: relative;
    overflow: hidden;
}

.pagination-item:hover {
    background: var(--primary);
    color: white;
    border-color: var(--primary);
    transform: translateY(-2px);
    box-shadow: 0 8px 25px -8px rgba(79, 70, 229, 0.4);
}

.pagination-item.active {
    background: var(--primary);
    color: white;
    border-color: var(--primary);
    box-shadow: 0 4px 15px -3px rgba(79, 70, 229, 0.3);
}

.pagination-item.disabled {
    opacity: 0.5;
    cursor: not-allowed;
    pointer-events: none;
}

.pagination-item.prev,
.pagination-item.next {
    width: 48px;
    font-size: 1rem;
}

.pagination-item.prev:hover,
.pagination-item.next:hover {
    background: var(--info);
    border-color: var(--info);
    box-shadow: 0 8px 25px -8px rgba(6, 182, 212, 0.4);
}

.pagination-dots {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 44px;
    height: 44px;
    color: var(--gray-400);
    font-weight: 600;
    cursor: default;
}

.pagination-info {
    display: flex;
    align-items: center;
    margin-left: 1.5rem;
    padding: 0.75rem 1.25rem;
    background: white;
    border-radius: 10px;
    border: 1px solid var(--gray-200);
    font-size: 0.875rem;
    color: var(--gray-600);
    direction: rtl;
}

.pagination-info i {
    margin-left: 0.5rem;
    color: var(--primary);
}

/* Enhanced hover effects */
.pagination-item::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(45deg, transparent 30%, rgba(255,255,255,0.3) 50%, transparent 70%);
    transform: translateX(-100%);
    transition: transform 0.6s ease;
}

.pagination-item:hover::before {
    transform: translateX(100%);
}

/* Responsive pagination */
@media (max-width: 768px) {
    .pagination-wrapper {
        padding: 1.5rem 1rem;
        flex-direction: column;
        gap: 1rem;
    }
    
    .pagination-info {
        margin-left: 0;
        order: -1;
    }
    
    .simple-pagination {
        gap: 0.25rem;
    }
    
    .pagination-item {
        width: 36px;
        height: 36px;
        font-size: 0.8rem;
    }
    
    .pagination-item.prev,
    .pagination-item.next {
        width: 40px;
    }
}

@media (max-width: 480px) {
    .simple-pagination {
        flex-wrap: wrap;
        justify-content: center;
    }
    
    .pagination-item {
        width: 32px;
        height: 32px;
        font-size: 0.75rem;
    }
    
    .pagination-item.prev,
    .pagination-item.next {
        width: 36px;
    }
}

/* ========== Enhanced Responsive Design ========== */
@media (max-width: 1400px) {
    .stats-grid {
        grid-template-columns: repeat(3, 1fr);
        gap: 1.5rem;
    }
}

@media (max-width: 1024px) {
    .stats-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 1.5rem;
    }
    
    .stat-card-header {
        flex-direction: column;
        gap: 1rem;
    }
    
    .stat-icon-wrapper {
        justify-content: space-between;
    }
}

@media (max-width: 768px) {
    .page-container {
        padding: 1rem;
    }
    
    .page-header {
        padding: 1.5rem;
    }
    
    .header-content {
        flex-direction: column;
        align-items: stretch;
        text-align: center;
    }
    
    .header-title h1 {
        font-size: 2rem;
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
        gap: 1.25rem;
    }
    
    .stat-card {
        transform: none;
        animation: none;
    }
    
    .stat-card:hover {
        transform: translateY(-4px);
        scale: 1;
    }
    
    .stat-card-header {
        padding: 1.25rem 1.25rem 0 1.25rem;
        flex-direction: column;
        gap: 1rem;
        align-items: stretch;
    }
    
    .stat-icon-wrapper {
        justify-content: space-between;
    }
    
    .stat-icon {
        width: 48px;
        height: 48px;
        font-size: 1.5rem;
    }
    
    .stat-content {
        padding: 0 1.25rem 1.25rem 1.25rem;
        text-align: center;
    }
    
    .stat-value {
        font-size: 2.5rem;
    }
    
    .stat-label {
        font-size: 1rem;
    }
    
    .stat-description {
        font-size: 0.875rem;
    }
    
    .stat-amount {
        font-size: 1rem;
    }
    
    .content-header {
        padding: 1.5rem;
    }
    
    .content-header-main {
        flex-direction: column;
        align-items: stretch;
    }
    
    .filters-grid {
        grid-template-columns: 1fr;
    }
    
    .invoice-table th,
    .invoice-table td {
        padding: 0.75rem 0.5rem;
        font-size: 0.875rem;
    }
}

@media (max-width: 480px) {
    .page-container {
        padding: 0.75rem;
    }
    
    .page-header {
        padding: 1.25rem;
        margin-bottom: 1.5rem;
    }
    
    .header-title h1 {
        font-size: 1.75rem;
    }
    
    .header-title p {
        font-size: 1rem;
    }
    
    .stats-grid {
        margin-bottom: 2rem;
    }
    
    .stat-card-header {
        padding: 1rem 1rem 0 1rem;
    }
    
    .stat-content {
        padding: 0 1rem 1rem 1rem;
    }
    
    .stat-value {
        font-size: 2.25rem;
    }
    
    .stat-badge {
        padding: 0.375rem 0.75rem;
        font-size: 0.7rem;
    }
    
    .stat-trend {
        padding: 0.25rem 0.5rem;
        font-size: 0.7rem;
    }
    
    .header-actions {
        flex-direction: column;
        width: 100%;
    }
    
    .btn-primary {
        justify-content: center;
        width: 100%;
    }
}

/* ========== Animations ========== */
.fade-in {
    animation: fadeIn 0.6s ease-out;
}

@keyframes fadeIn {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}
</style>
@endpush

@section('content')
<div class="">
    <!-- Page Header -->
    <div class="page-header">
        <div class="header-content">
            <div class="header-title">
                <h1>إدارة الفواتير</h1>
                <p>نظام متقدم لإدارة الفواتير الإلكترونية وفقاً لمعايير هيئة الزكاة والضريبة والجمارك</p>
            </div>
            <div class="header-actions">
                <a href="{{ route('invoices.create') }}" class="btn-primary solid">
                    <i class="fas fa-plus"></i>
                    فاتورة جديدة
                </a>
                <a href="{{ route('invoices.cash-flow-report') }}" class="btn-primary">
                    <i class="fas fa-chart-line"></i>
                    حركة الصندوق
                </a>
                <a href="{{ route('invoices.advanced-search') }}" class="btn-primary">
                    <i class="fas fa-search-plus"></i>
                    البحث المتقدم
                </a>
                <a href="{{ route('invoices.sync') }}" class="btn-primary">
                    <i class="fas fa-sync-alt"></i>
                    مزامنة ZATCA
                </a>
                <button type="button" class="btn-primary" onclick="exportInvoices()">
                    <i class="fas fa-download"></i>
                    تصدير
                </button>
            </div>
        </div>
    </div>

    <!-- Enhanced Statistics Cards -->
    <div class="stats-grid">
        <!-- الفواتير الضريبية (Tax Invoices - Standard) -->
        <div class="stat-card primary-gradient">
            <div class="stat-card-header">
                <div class="stat-icon-wrapper">
                    <div class="stat-icon primary">
                        <i class="fas fa-file-invoice-dollar"></i>
                    </div>
                    <div class="stat-trend positive">
                        <i class="fas fa-arrow-up"></i>
                        +12%
                    </div>
                </div>
                <div class="stat-badge">
                    <i class="fas fa-certificate"></i>
                    معيارية
                </div>
            </div>
            <div class="stat-content">
                <div class="stat-value">{{ $invoices->where('type', 'standard')->count() }}</div>
                <div class="stat-label">الفواتير الضريبية</div>
                <div class="stat-description">الفواتير الضريبية وفقاً لمعايير الزكاة</div>
                <div class="stat-amount">
                    {{ number_format($invoices->where('type', 'standard')->sum('total_amount'), 2) }} ر.س
                </div>
            </div>
        </div>

        <!-- الفواتير الضريبة المبسطة (Simplified Tax Invoices) -->
        <div class="stat-card info-gradient">
            <div class="stat-card-header">
                <div class="stat-icon-wrapper">
                    <div class="stat-icon info">
                        <i class="fas fa-receipt"></i>
                    </div>
                    <div class="stat-trend positive">
                        <i class="fas fa-arrow-up"></i>
                        +8%
                    </div>
                </div>
                <div class="stat-badge">
                    <i class="fas fa-bolt"></i>
                    مبسطة
                </div>
            </div>
            <div class="stat-content">
                <div class="stat-value">{{ $invoices->where('type', 'simplified')->count() }}</div>
                <div class="stat-label">الفواتير الضريبة المبسطة</div>
                <div class="stat-description">فواتير التجزئة والخدمات السريعة</div>
                <div class="stat-amount">
                    {{ number_format($invoices->where('type', 'simplified')->sum('total_amount'), 2) }} ر.س
                </div>
            </div>
        </div>

        <!-- الفواتير المرسلة (Sent Invoices) -->
        <div class="stat-card warning-gradient">
            <div class="stat-card-header">
                <div class="stat-icon-wrapper">
                    <div class="stat-icon warning">
                        <i class="fas fa-paper-plane"></i>
                    </div>
                    <div class="stat-trend positive">
                        <i class="fas fa-arrow-up"></i>
                        +18%
                    </div>
                </div>
                <div class="stat-badge">
                    <i class="fas fa-clock"></i>
                    قيد المراجعة
                </div>
            </div>
            <div class="stat-content">
                <div class="stat-value">{{ $invoices->where('status', 'reported')->count() }}</div>
                <div class="stat-label">الفواتير المرسلة</div>
                <div class="stat-description">إرسلت للهيئة الزكاة وفي انتظار المراجعة</div>
                <div class="stat-amount">
                    {{ number_format($invoices->where('status', 'reported')->sum('total_amount'), 2) }} ر.س
                </div>
            </div>
        </div>

        <!-- الفواتير المؤكدة (Confirmed Invoices) -->
        <div class="stat-card success-gradient">
            <div class="stat-card-header">
                <div class="stat-icon-wrapper">
                    <div class="stat-icon success">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <div class="stat-trend positive">
                        <i class="fas fa-arrow-up"></i>
                        +5%
                    </div>
                </div>
                <div class="stat-badge">
                    <i class="fas fa-shield-check"></i>
                    مؤكدة
                </div>
            </div>
            <div class="stat-content">
                <div class="stat-value">{{ $invoices->where('status', 'cleared')->count() }}</div>
                <div class="stat-label">الفواتير المؤكدة</div>
                <div class="stat-description">تم تأكيدها من هيئة الزكاة بنجاح</div>
                <div class="stat-amount">
                    {{ number_format($invoices->where('status', 'cleared')->sum('total_amount'), 2) }} ر.س
                </div>
            </div>
        </div>

        <!-- الفواتير المرفوضة (Imposed/Reported Invoices) -->
        <div class="stat-card danger-gradient">
            <div class="stat-card-header">
                <div class="stat-icon-wrapper">
                    <div class="stat-icon danger">
                        <i class="fas fa-flag"></i>
                    </div>
                    <div class="stat-trend neutral">
                        <i class="fas fa-minus"></i>
                        0%
                    </div>
                </div>
                <div class="stat-badge">
                    <i class="fas fa-exclamation-triangle"></i>
                    مفروضة
                </div>
            </div>
            <div class="stat-content">
                <div class="stat-value">{{ $invoices->where('status', 'rejected')->count() }}</div>
                <div class="stat-label">الفواتير المرفوضة</div>
                <div class="stat-description">تم الإبلاغ عنها إلى الجهات المختصة</div>
                <div class="stat-amount">
                    {{ number_format($invoices->where('status', 'rejected')->sum('total_amount'), 2) }} ر.س
                </div>
            </div>
        </div>

    </div>

    <!-- Main Content -->
    <div class="content-card">
        <!-- Header -->
        <div class="content-header">
            <div class="content-header-main">
                <div class="content-title">
                    <h2>قائمة الفواتير</h2>
                    <p>إدارة ومتابعة جميع الفواتير الإلكترونية</p>
                </div>
                <div class="content-controls">
                    <button type="button" class="btn-secondary" onclick="toggleFilters()" id="filterBtn">
                        <i class="fas fa-filter"></i>
                        فلترة
                    </button>
                    <button type="button" class="btn-secondary" onclick="refreshPage()">
                        <i class="fas fa-sync-alt"></i>
                        تحديث
                    </button>
                </div>
            </div>
        </div>

        <!-- Filters -->
        <div class="filters-section" id="filters-section">
            <form method="GET" action="{{ route('invoices.index') }}">
                <div class="filters-grid">
                    <div class="filter-group">
                        <label>نوع الفاتورة</label>
                        <select name="type" class="filter-input">
                            <option value="">جميع الأنواع</option>
                            <option value="standard" {{ request('type') == 'standard' ? 'selected' : '' }}>معيارية</option>
                            <option value="simplified" {{ request('type') == 'simplified' ? 'selected' : '' }}>مبسطة</option>
                        </select>
                    </div>

                    <div class="filter-group">
                        <label>حالة الفاتورة</label>
                        <select name="status" class="filter-input">
                            <option value="">جميع الحالات</option>
                            <option value="draft" {{ request('status') == 'draft' ? 'selected' : '' }}>مسودة</option>
                            <option value="sent" {{ request('status') == 'sent' ? 'selected' : '' }}>مرسلة</option>
                            <option value="cleared" {{ request('status') == 'cleared' ? 'selected' : '' }}>مؤكدة</option>
                            <option value="reported" {{ request('status') == 'reported' ? 'selected' : '' }}>مبلغة</option>
                            <option value="rejected" {{ request('status') == 'rejected' ? 'selected' : '' }}>مرفوضة</option>
                        </select>
                    </div>

                    <div class="filter-group">
                        <label>من تاريخ</label>
                        <input type="date" name="date_from" class="filter-input" value="{{ request('date_from') }}">
                    </div>

                    <div class="filter-group">
                        <label>إلى تاريخ</label>
                        <input type="date" name="date_to" class="filter-input" value="{{ request('date_to') }}">
                    </div>

                    <div class="filter-group">
                        <label>البحث</label>
                        <div class="search-wrapper">
                            <input type="text" name="search" class="filter-input search-input" 
                                   placeholder="رقم الفاتورة، اسم العميل..." value="{{ request('search') }}">
                            <i class="fas fa-search search-icon"></i>
                        </div>
                    </div>
                </div>

                <div class="filter-actions">
                    <button type="submit" class="btn btn-success">
                        <i class="fas fa-check"></i>
                        تطبيق الفلاتر
                    </button>
                    <a href="{{ route('invoices.index') }}" class="btn btn-warning">
                        <i class="fas fa-undo"></i>
                        إعادة تعيين
                    </a>
                </div>
            </form>
        </div>

        <!-- Table -->
        @if($invoices->count() > 0)
            <div class="table-container">
                <table class="invoice-table">
                        <tr>
                            <th>رقم الفاتورة</th>
                            <th>النوع</th>
                            <th>العميل</th>
                            <th>التاريخ</th>
                            <th>المبلغ</th>
                            <th>الحالة</th>
                            <th>ZATCA</th>
                            <th style="text-align: center;">الإجراءات</th>
                        </tr>
                    <tbody>
                        @foreach($invoices as $invoice)
                        <tr>
                            <td>
                                <div class="invoice-number">{{ $invoice->invoice_number }}</div>
                                @if($invoice->uuid)
                                    <div class="invoice-uuid">{{ Str::limit($invoice->uuid, 8) }}...</div>
                                @endif
                            </td>
                            <td>
                                @if($invoice->type == 'standard')
                                    <span class="status-badge primary">
                                        <i class="fas fa-file-invoice-dollar"></i>
                                        معيارية
                                    </span>
                                @else
                                    <span class="status-badge info">
                                        <i class="fas fa-receipt"></i>
                                        مبسطة
                                    </span>
                                @endif
                            </td>
                            <td>
                                @if($invoice->customer)
                                    <div class="customer-name">{{ $invoice->customer->name }}</div>
                                    @if($invoice->customer->customer_type == 'business' && $invoice->customer->vat_number)
                                        <div class="customer-vat">{{ $invoice->customer->vat_number }}</div>
                                    @endif
                                @else
                                    <span style="color: var(--gray-400);">غير محدد</span>
                                @endif
                            </td>
                            <td>
                                <div>{{ $invoice->issue_date->format('Y-m-d') }}</div>
                                <div style="font-size: 0.75rem; color: var(--gray-500);">{{ $invoice->issue_date->format('l') }}</div>
                            </td>
                            <td>
                                <div class="amount-main">{{ number_format($invoice->total_amount, 2) }} ر.س</div>
                                <div class="amount-tax">ضريبة: {{ number_format($invoice->tax_amount, 2) }}</div>
                            </td>
                            <td>
                                @switch($invoice->status)
                                    @case('draft')
                                        <span class="status-badge secondary">
                                            <i class="fas fa-edit"></i>
                                            مسودة
                                        </span>
                                        @break
                                    @case('sent')
                                        <span class="status-badge info">
                                            <i class="fas fa-paper-plane"></i>
                                            مرسلة
                                        </span>
                                        @break
                                    @case('cleared')
                                        <span class="status-badge success">
                                            <i class="fas fa-check-circle"></i>
                                            مؤكدة
                                        </span>
                                        @break
                                    @case('reported')
                                        <span class="status-badge primary">
                                            <i class="fas fa-flag"></i>
                                            مبلغة
                                        </span>
                                        @break
                                    @case('rejected')
                                        <span class="status-badge danger">
                                            <i class="fas fa-times-circle"></i>
                                            مرفوضة
                                        </span>
                                        @break
                                    @default
                                        <span class="status-badge warning">
                                            <i class="fas fa-exclamation-triangle"></i>
                                            غير محدد
                                        </span>
                                @endswitch
                            </td>
                            <td style="text-align: center;">
                                @if($invoice->sent_to_zatca_at)
                                    <span class="status-badge success">
                                        <i class="fas fa-check"></i>
                                        تم الإرسال
                                    </span>
                                @else
                                    <span class="status-badge warning">
                                        <i class="fas fa-clock"></i>
                                        في الانتظار
                                    </span>
                                @endif
                            </td>
                            <td>
                                <div class="action-menu">
                                    <a href="{{ route('invoices.show', $invoice) }}" class="action-btn view" title="عرض الفاتورة">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                    @if($invoice->status == 'draft')
                                        <a href="{{ route('invoices.edit', $invoice) }}" class="action-btn edit" title="تعديل الفاتورة">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                    @endif
                                    <a href="{{ route('invoices.print', $invoice) }}" target="_blank" class="action-btn pdf" title="طباعة PDF">
                                        <i class="fas fa-print"></i>
                                    </a>
                                    
                                    <!-- إرسال واتساب -->
                                    <button type="button" class="action-btn whatsapp" title="إرسال عبر واتساب"
                                            onclick="sendWhatsApp({{ $invoice->id }}, '{{ $invoice->invoice_number }}', {{ $invoice->total_amount }})">
                                        <i class="fab fa-whatsapp"></i>
                                    </button>
                                    
                                    <!-- إرسال بريد إلكتروني -->
                                    <button type="button" class="action-btn email" title="إرسال بريد إلكتروني"
                                            onclick="sendEmail({{ $invoice->id }}, '{{ $invoice->invoice_number }}')">
                                        <i class="fas fa-envelope"></i>
                                    </button>
                                    
                                    <!-- إرسال لهيئة الزكاة - يخفى عندما تكون مرسلة -->
                                    @if(!$invoice->sent_to_zatca_at && $invoice->status !== 'sent')
                                        <button type="button" class="action-btn send" title="إرسال لهيئة الزكاة"
                                                onclick="sendToZakat({{ $invoice->id }})">
                                            <i class="fas fa-paper-plane"></i>
                                        </button>
                                    @endif
                                </div>
                            </td>
                        </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>

            <!-- Simple Custom Pagination -->
            <div class="pagination-wrapper">
                @if($invoices->hasPages())
                    <div class="simple-pagination">
                        {{-- Previous Page Link --}}
                        @if($invoices->onFirstPage())
                            <span class="pagination-item disabled">
                                <i class="fas fa-chevron-right"></i>
                            </span>
                        @else
                            <a href="{{ $invoices->previousPageUrl() }}" class="pagination-item prev">
                                <i class="fas fa-chevron-right"></i>
                            </a>
                        @endif

                        {{-- Pagination Elements --}}
                        @php
                            $currentPage = $invoices->currentPage();
                            $lastPage = $invoices->lastPage();
                            $start = max(1, $currentPage - 2);
                            $end = min($lastPage, $currentPage + 2);
                        @endphp

                        {{-- First Page --}}
                        @if($start > 1)
                            <a href="{{ $invoices->url(1) }}" class="pagination-item">1</a>
                            @if($start > 2)
                                <span class="pagination-dots">...</span>
                            @endif
                        @endif

                        {{-- Page Numbers --}}
                        @for($page = $start; $page <= $end; $page++)
                            @if($page == $currentPage)
                                <span class="pagination-item active">{{ $page }}</span>
                            @else
                                <a href="{{ $invoices->url($page) }}" class="pagination-item">{{ $page }}</a>
                            @endif
                        @endfor

                        {{-- Last Page --}}
                        @if($end < $lastPage)
                            @if($end < $lastPage - 1)
                                <span class="pagination-dots">...</span>
                            @endif
                            <a href="{{ $invoices->url($lastPage) }}" class="pagination-item">{{ $lastPage }}</a>
                        @endif

                        {{-- Next Page Link --}}
                        @if($invoices->hasMorePages())
                            <a href="{{ $invoices->nextPageUrl() }}" class="pagination-item next">
                                <i class="fas fa-chevron-left"></i>
                            </a>
                        @else
                            <span class="pagination-item disabled">
                                <i class="fas fa-chevron-left"></i>
                            </span>
                        @endif
                    </div>

                    {{-- Pagination Info --}}
                    <div class="pagination-info">
                        <i class="fas fa-info-circle"></i>
                        عرض {{ $invoices->firstItem() }} إلى {{ $invoices->lastItem() }} من أصل {{ $invoices->total() }} فاتورة
                    </div>
                @endif
            </div>
        @else
            <div class="empty-state">
                <div class="empty-icon">
                    <i class="fas fa-file-invoice"></i>
                </div>
                <h3>لا توجد فواتير</h3>
                <p>لم يتم العثور على فواتير مطابقة للمعايير المحددة</p>
                <a href="{{ route('invoices.create') }}" class="btn-primary">
                    <i class="fas fa-plus"></i>
                    إنشاء فاتورة جديدة
                </a>
            </div>
        @endif
    </div>
</div>
@endsection

<!-- ZATCA Confirmation Modal -->
<div id="zatcaModal" class="zatca-modal" style="display: none;">
    <div class="zatca-modal-overlay"></div>
    <div class="zatca-modal-content">
        <div class="zatca-modal-header">
            <div class="zatca-modal-icon">
                <i class="fas fa-paper-plane"></i>
            </div>
            <h3>تأكيد الإرسال إلى هيئة الزكاة</h3>
        </div>
        <div class="zatca-modal-body">
            <p>هل تريد إرسال هذه الفاتورة إلى هيئة الزكاة والضريبة والجمارك؟</p>
            <div class="zatca-modal-details">
                <div class="detail-item">
                    <i class="fas fa-file-invoice"></i>
                    <span>رقم الفاتورة: <strong id="modalInvoiceNumber"></strong></span>
                </div>
                <div class="detail-item">
                    <i class="fas fa-calendar"></i>
                    <span>تاريخ الإرسال: <strong id="modalSendDate"></strong></span>
                </div>
                <div class="detail-item">
                    <i class="fas fa-shield-alt"></i>
                    <span>سيتم الإرسال وفقاً لمعايير هيئة الزكاة والضريبة والجمارك</span>
                </div>
            </div>
        </div>
        <div class="zatca-modal-actions">
            <button type="button" id="confirmSendBtn" class="zatca-btn zatca-btn-primary">
                <i class="fas fa-paper-plane"></i>
                تأكيد الإرسال
            </button>
            <button type="button" id="cancelSendBtn" class="zatca-btn zatca-btn-secondary">
                <i class="fas fa-times"></i>
                إلغاء
            </button>
        </div>
    </div>
</div>

@push('styles')
<style>
/* ZATCA Modal Styles */
.zatca-modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    z-index: 10000;
    display: flex;
    align-items: center;
    justify-content: center;
    opacity: 0;
    visibility: hidden;
    transition: all 0.3s ease;
}

.zatca-modal.show {
    opacity: 1;
    visibility: visible;
}

.zatca-modal-overlay {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.5);
    backdrop-filter: blur(5px);
}

.zatca-modal-content {
    position: relative;
    background: var(--white);
    border-radius: 16px;
    box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
    max-width: 500px;
    width: 90%;
    max-height: 90vh;
    overflow: hidden;
    transform: scale(0.8) translateY(20px);
    transition: all 0.3s ease;
}

.zatca-modal.show .zatca-modal-content {
    transform: scale(1) translateY(0);
}

.zatca-modal-header {
    background: linear-gradient(135deg, var(--primary) 0%, var(--primary-light) 100%);
    color: white;
    padding: 2rem;
    text-align: center;
    position: relative;
    overflow: hidden;
}

.zatca-modal-header::before {
    content: '';
    position: absolute;
    top: 0;
    right: 0;
    width: 100px;
    height: 100px;
    background: rgba(255, 255, 255, 0.1);
    border-radius: 50%;
    transform: translate(30px, -30px);
}

.zatca-modal-icon {
    width: 80px;
    height: 80px;
    background: rgba(255, 255, 255, 0.2);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 1rem;
    font-size: 2rem;
    backdrop-filter: blur(10px);
    border: 2px solid rgba(255, 255, 255, 0.3);
}

.zatca-modal-header h3 {
    margin: 0;
    font-size: 1.5rem;
    font-weight: 700;
    text-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.zatca-modal-body {
    padding: 2rem;
}

.zatca-modal-body p {
    font-size: 1.1rem;
    color: var(--gray-700);
    text-align: center;
    margin: 0 0 1.5rem 0;
    font-weight: 500;
    line-height: 1.6;
}

.zatca-modal-details {
    background: var(--gray-50);
    border-radius: 12px;
    padding: 1.5rem;
    border: 1px solid var(--gray-200);
}

.detail-item {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    margin-bottom: 1rem;
    font-size: 0.95rem;
    color: var(--gray-600);
}

.detail-item:last-child {
    margin-bottom: 0;
}

.detail-item i {
    color: var(--primary);
    width: 20px;
    text-align: center;
}

.detail-item strong {
    color: var(--gray-800);
}

.zatca-modal-actions {
    padding: 1.5rem 2rem 2rem;
    display: flex;
    gap: 1rem;
    justify-content: center;
}

.zatca-btn {
    padding: 0.75rem 2rem;
    border-radius: 10px;
    font-weight: 600;
    font-size: 0.95rem;
    border: none;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    min-width: 140px;
    justify-content: center;
}

.zatca-btn-primary {
    background: linear-gradient(135deg, var(--primary), var(--primary-light));
    color: white;
    box-shadow: 0 4px 12px rgba(79, 70, 229, 0.3);
}

.zatca-btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 20px rgba(79, 70, 229, 0.4);
}

.zatca-btn-secondary {
    background: var(--gray-100);
    color: var(--gray-700);
    border: 1px solid var(--gray-300);
}

.zatca-btn-secondary:hover {
    background: var(--gray-200);
    border-color: var(--gray-400);
    transform: translateY(-1px);
}

.zatca-btn:active {
    transform: translateY(0);
}

.zatca-btn.loading {
    pointer-events: none;
    opacity: 0.7;
}

.zatca-btn.loading i {
    animation: spin 1s linear infinite;
}

/* Responsive */
@media (max-width: 480px) {
    .zatca-modal-content {
        width: 95%;
        margin: 1rem;
    }
    
    .zatca-modal-header,
    .zatca-modal-body {
        padding: 1.5rem;
    }
    
    .zatca-modal-actions {
        padding: 1rem 1.5rem 1.5rem;
        flex-direction: column;
    }
    
    .zatca-btn {
        width: 100%;
    }
    
    .zatca-modal-header h3 {
        font-size: 1.25rem;
    }
    
    .zatca-modal-icon {
        width: 60px;
        height: 60px;
        font-size: 1.5rem;
    }
}
</style>
@endpush

@push('scripts')
<script>
function toggleFilters() {
    const filtersSection = document.getElementById('filters-section');
    const filterBtn = document.getElementById('filterBtn');
    
    if (filtersSection.classList.contains('show')) {
        filtersSection.classList.remove('show');
        filterBtn.classList.remove('active');
    } else {
        filtersSection.classList.add('show');
        filterBtn.classList.add('active');
    }
}

function refreshPage() {
    const btn = event.target.closest('button');
    const originalHTML = btn.innerHTML;
    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري التحديث...';
    btn.disabled = true;
    
    setTimeout(() => {
        window.location.reload();
    }, 1000);
}

function exportInvoices() {
    const btn = event.target.closest('button');
    const originalHTML = btn.innerHTML;
    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري التصدير...';
    btn.disabled = true;
    
    setTimeout(() => {
        btn.innerHTML = originalHTML;
        btn.disabled = false;
        showToast('تم تصدير الفواتير بنجاح', 'success');
    }, 2000);
}

function sendToZakat(invoiceId) {
    // Get invoice details from the table row
    const btn = event.target.closest('button');
    const row = btn.closest('tr');
    const invoiceNumber = row.querySelector('.invoice-number').textContent.trim();
    
    // Show the custom modal
    showZatcaModal(invoiceId, invoiceNumber, btn);
}

function showZatcaModal(invoiceId, invoiceNumber, triggerBtn) {
    const modal = document.getElementById('zatcaModal');
    const confirmBtn = document.getElementById('confirmSendBtn');
    const cancelBtn = document.getElementById('cancelSendBtn');
    const overlay = modal.querySelector('.zatca-modal-overlay');
    
    // Set invoice details in modal
    document.getElementById('modalInvoiceNumber').textContent = invoiceNumber;
    document.getElementById('modalSendDate').textContent = new Date().toLocaleDateString('ar-SA');
    
    // Show modal
    modal.style.display = 'flex';
    setTimeout(() => modal.classList.add('show'), 10);
    
    // Handle confirm button
    function handleConfirm() {
        hideZatcaModal();
        performZatcaSend(invoiceId, triggerBtn);
    }
    
    // Handle cancel button
    function handleCancel() {
        hideZatcaModal();
    }
    
    // Handle overlay click
    function handleOverlayClick() {
        hideZatcaModal();
    }
    
    // Add event listeners
    confirmBtn.addEventListener('click', handleConfirm);
    cancelBtn.addEventListener('click', handleCancel);
    overlay.addEventListener('click', handleOverlayClick);
    
    // Handle escape key
    function handleEscape(e) {
        if (e.key === 'Escape') {
            hideZatcaModal();
        }
    }
    document.addEventListener('keydown', handleEscape);
    
    // Cleanup function
    function hideZatcaModal() {
        modal.classList.remove('show');
        setTimeout(() => {
            modal.style.display = 'none';
            // Remove event listeners
            confirmBtn.removeEventListener('click', handleConfirm);
            cancelBtn.removeEventListener('click', handleCancel);
            overlay.removeEventListener('click', handleOverlayClick);
            document.removeEventListener('keydown', handleEscape);
        }, 300);
    }
}

function performZatcaSend(invoiceId, btn) {
    const originalHTML = btn.innerHTML;
    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
    btn.disabled = true;
    
    // Show loading button in modal if it's still open
    const confirmBtn = document.getElementById('confirmSendBtn');
    if (confirmBtn) {
        confirmBtn.classList.add('loading');
        confirmBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري الإرسال...';
    }
    
    showToast('جاري إرسال الفاتورة...', 'info');
    
    // إرسال طلب POST إلى الخادم
    fetch(`{{ url('/invoices') }}/${invoiceId}/send-to-zatca`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        }
    })
    .then(response => {
        if (response.ok) {
            return response.json().catch(() => {
                // If response is not JSON, treat as success
                return { success: true };
            });
        } else {
            return response.json().catch(() => {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }).then(data => {
                throw new Error(data.message || `HTTP ${response.status}: ${response.statusText}`);
            });
        }
    })
    .then(data => {
        if (data && data.success !== false) {
            showToast('تم إرسال الفاتورة بنجاح', 'success');
            setTimeout(() => window.location.reload(), 1500);
        } else {
            throw new Error((data && data.message) || 'حدث خطأ أثناء الإرسال');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        const errorMessage = error.message || 'حدث خطأ غير متوقع';
        showToast('حدث خطأ أثناء إرسال الفاتورة: ' + errorMessage, 'danger');
        btn.innerHTML = originalHTML;
        btn.disabled = false;
        
        // Reset modal button if still open
        if (confirmBtn) {
            confirmBtn.classList.remove('loading');
            confirmBtn.innerHTML = '<i class="fas fa-paper-plane"></i> تأكيد الإرسال';
        }
    });
}

function sendWhatsApp(invoiceId, invoiceNumber, totalAmount) {
    const customerName = event.target.closest('tr').querySelector('.customer-name')?.textContent || 'العميل المحترم';
    const message = `مرحباً ${customerName}،

نرسل لكم فاتورة رقم: ${invoiceNumber}
المبلغ الإجمالي: ${new Intl.NumberFormat('ar-SA').format(totalAmount)} ريال سعودي

يمكنكم الاطلاع على تفاصيل الفاتورة والدفع من خلال الرابط المرفق.

شكراً لتعاملكم معنا
فريق المبيعات`;
    
    // ترميز الرسالة للواتساب
    const encodedMessage = encodeURIComponent(message);
    const whatsappUrl = `https://wa.me/?text=${encodedMessage}`;
    
    // فتح واتساب في نافذة جديدة
    window.open(whatsappUrl, '_blank', 'width=600,height=600');
    
    showToast('تم فتح واتساب لإرسال الفاتورة', 'success');
}

function sendEmail(invoiceId, invoiceNumber) {
    const customerName = event.target.closest('tr').querySelector('.customer-name')?.textContent || 'العميل المحترم';
    const totalAmount = event.target.closest('tr').querySelector('.amount-main')?.textContent || '';
    
    const subject = `فاتورة رقم: ${invoiceNumber}`;
    const body = `السلام عليكم ورحمة الله وبركاته ${customerName}،

نتشرف بإرسال الفاتورة رقم: ${invoiceNumber}
المبلغ الإجمالي: ${totalAmount}

تفاصيل الفاتورة:
• رقم الفاتورة: ${invoiceNumber}
• المبلغ: ${totalAmount}
• التاريخ: ${new Date().toLocaleDateString('ar-SA')}

يرجى مراجعة الفاتورة والتواصل معنا في حال وجود أي استفسار.

شكراً لتعاملكم معنا
فريق المحاسبة`;
    
    // ترميز محتوى البريد الإلكتروني
    const encodedSubject = encodeURIComponent(subject);
    const encodedBody = encodeURIComponent(body);
    
    // فتح عميل البريد الإلكتروني
    const emailUrl = `mailto:?subject=${encodedSubject}&body=${encodedBody}`;
    window.location.href = emailUrl;
    
    showToast('تم فتح عميل البريد الإلكتروني', 'success');
}

function showToast(message, type = 'info') {
    const colors = {
        success: '#10b981',
        warning: '#f59e0b',
        info: '#06b6d4',
        danger: '#ef4444'
    };
    
    const icons = {
        success: 'check-circle',
        warning: 'exclamation-triangle',
        info: 'info-circle',
        danger: 'times-circle'
    };
    
    const toast = document.createElement('div');
    toast.style.cssText = `
        position: fixed;
        top: 20px;
        left: 20px;
        background: white;
        border-radius: 8px;
        padding: 1rem 1.5rem;
        box-shadow: 0 10px 15px -3px rgb(0 0 0 / 0.1);
        border-left: 4px solid ${colors[type]};
        z-index: 9999;
        display: flex;
        align-items: center;
        gap: 0.75rem;
        min-width: 300px;
        animation: slideInLeft 0.3s ease;
    `;
    
    toast.innerHTML = `
        <i class="fas fa-${icons[type]}" style="color: ${colors[type]}; font-size: 1.2rem;"></i>
        <span style="color: var(--gray-800); font-weight: 500;">${message}</span>
    `;
    
    document.body.appendChild(toast);
    
    setTimeout(() => {
        toast.style.animation = 'slideOutLeft 0.3s ease';
        setTimeout(() => toast.remove(), 300);
    }, 3000);
}

// Add CSS animations
const style = document.createElement('style');
style.textContent = `
    @keyframes slideInLeft {
        from { transform: translateX(-100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    @keyframes slideOutLeft {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(-100%); opacity: 0; }
    }
`;
document.head.appendChild(style);

document.addEventListener('DOMContentLoaded', function() {
    // Show filters if URL has parameters
    const urlParams = new URLSearchParams(window.location.search);
    if (urlParams.has('type') || urlParams.has('status') || urlParams.has('date_from') || 
        urlParams.has('date_to') || urlParams.has('search')) {
        toggleFilters();
    }
});
</script>
@endpush
