@extends('layouts.app')

@section('title', 'تقرير متابعة الفواتير')
@section('page-title', 'تقرير متابعة الفواتير')

@push('styles')
<style>
/* ========== Modern Simplified Design ========== */
:root {
    --primary: #4f46e5;
    --primary-light: #818cf8;
    --success: #10b981;
    --warning: #f59e0b;
    --danger: #ef4444;
    --info: #06b6d4;
    --gray-50: #f9fafb;
    --gray-100: #f3f4f6;
    --gray-200: #e5e7eb;
    --gray-300: #d1d5db;
    --gray-400: #9ca3af;
    --gray-500: #6b7280;
    --gray-600: #4b5563;
    --gray-700: #374151;
    --gray-800: #1f2937;
    --gray-900: #111827;
    --white: #ffffff;
    --border-radius: 12px;
    --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
    --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
    --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
}

.page-container {
    padding: 2rem;
    max-width: 1400px;
    margin: 0 auto;
}

/* ========== Header Section ========== */
.page-header {
    background: linear-gradient(135deg, var(--primary) 0%, var(--primary-light) 100%);
    border-radius: var(--border-radius);
    padding: 2.5rem;
    margin-bottom: 2rem;
    color: white;
    position: relative;
    overflow: hidden;
}

.page-header::before {
    content: '';
    position: absolute;
    top: 0;
    right: 0;
    width: 200px;
    height: 200px;
    background: rgba(255, 255, 255, 0.1);
    border-radius: 50%;
    transform: translate(50px, -50px);
}

.header-content {
    position: relative;
    z-index: 1;
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 2rem;
}

.header-title {
    flex: 1;
}

.header-title h1 {
    font-size: 2.5rem;
    font-weight: 800;
    margin: 0 0 0.5rem 0;
    text-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.header-title p {
    font-size: 1.1rem;
    opacity: 0.9;
    margin: 0;
    font-weight: 300;
}

.header-actions {
    display: flex;
    gap: 1rem;
    flex-wrap: wrap;
}

.btn-primary {
    background: rgba(255, 255, 255, 0.2);
    border: 1px solid rgba(255, 255, 255, 0.3);
    color: white;
    padding: 0.75rem 1.5rem;
    border-radius: var(--border-radius);
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
    backdrop-filter: blur(10px);
    display: flex;
    align-items: center;
    gap: 0.5rem;
    cursor: pointer;
}

.btn-primary:hover {
    background: rgba(255, 255, 255, 0.3);
    transform: translateY(-2px);
    color: white;
    box-shadow: var(--shadow-lg);
}

/* ========== Enhanced Professional Stats Cards ========== */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(4, 1fr);
    gap: 2rem;
    margin-bottom: 2.5rem;
}

.stat-card {
    background: var(--white);
    border-radius: 16px;
    padding: 0;
    border: 1px solid var(--gray-200);
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
    overflow: hidden;
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
}

.stat-card:hover {
    transform: translateY(-8px) scale(1.02);
    box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
    border-color: var(--gray-300);
}

.stat-card.primary-gradient::before {
    background: linear-gradient(135deg, var(--primary) 0%, #6366f1 100%);
}

.stat-card.success-gradient::before {
    background: linear-gradient(135deg, var(--success) 0%, #059669 100%);
}

.stat-card.warning-gradient::before {
    background: linear-gradient(135deg, var(--warning) 0%, #d97706 100%);
}

.stat-card.danger-gradient::before {
    background: linear-gradient(135deg, var(--danger) 0%, #dc2626 100%);
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 6px;
    background: var(--primary);
}

.stat-card-header {
    padding: 1.5rem 1.5rem 0 1.5rem;
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 1rem;
}

.stat-icon-wrapper {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.stat-icon {
    width: 56px;
    height: 56px;
    border-radius: 14px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.75rem;
    color: white;
    background: var(--primary);
    box-shadow: 0 8px 16px rgba(0, 0, 0, 0.15);
    position: relative;
}

.stat-icon.primary { background: linear-gradient(135deg, var(--primary), #6366f1); }
.stat-icon.success { background: linear-gradient(135deg, var(--success), #059669); }
.stat-icon.warning { background: linear-gradient(135deg, var(--warning), #d97706); }
.stat-icon.danger { background: linear-gradient(135deg, var(--danger), #dc2626); }

.stat-trend {
    display: flex;
    align-items: center;
    gap: 0.25rem;
    padding: 0.375rem 0.75rem;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 700;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.stat-trend.positive {
    background: rgba(16, 185, 129, 0.1);
    color: var(--success);
    border: 1px solid rgba(16, 185, 129, 0.2);
}

.stat-trend.neutral {
    background: rgba(107, 114, 128, 0.1);
    color: var(--gray-600);
    border: 1px solid rgba(107, 114, 128, 0.2);
}

.stat-badge {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    background: rgba(255, 255, 255, 0.9);
    border-radius: 20px;
    font-size: 0.75rem;
    font-weight: 600;
    color: var(--gray-700);
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
    backdrop-filter: blur(10px);
}

.stat-content {
    padding: 0 1.5rem 1.5rem 1.5rem;
}

.stat-value {
    font-size: 3rem;
    font-weight: 900;
    color: var(--gray-900);
    margin-bottom: 0.5rem;
    line-height: 1;
    background: linear-gradient(135deg, var(--gray-900), var(--gray-700));
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.stat-label {
    font-size: 1.1rem;
    font-weight: 700;
    color: var(--gray-800);
    margin-bottom: 0.5rem;
    line-height: 1.2;
}

.stat-description {
    font-size: 0.9rem;
    color: var(--gray-600);
    line-height: 1.4;
}

/* Professional Hover Effects */
.stat-card::after {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.4), transparent);
    transition: left 0.6s ease;
}

.stat-card:hover::after {
    left: 100%;
}

@keyframes cardFloat {
    0%, 100% { transform: translateY(0px); }
    50% { transform: translateY(-5px); }
}

.stat-card {
    animation: cardFloat 6s ease-in-out infinite;
}

.stat-card:nth-child(2) { animation-delay: -1s; }
.stat-card:nth-child(3) { animation-delay: -2s; }
.stat-card:nth-child(4) { animation-delay: -3s; }

/* ========== Main Content Card ========== */
.content-card {
    background: var(--white);
    border-radius: var(--border-radius);
    border: 1px solid var(--gray-200);
    overflow: hidden;
    box-shadow: var(--shadow-sm);
}

.content-header {
    background: var(--gray-50);
    padding: 2rem;
    border-bottom: 1px solid var(--gray-200);
}

.content-header-main {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1.5rem;
    flex-wrap: wrap;
    gap: 1rem;
}

.content-title h2 {
    font-size: 1.5rem;
    font-weight: 700;
    color: var(--gray-900);
    margin: 0 0 0.25rem 0;
}

.content-title p {
    color: var(--gray-600);
    margin: 0;
}

.content-controls {
    display: flex;
    gap: 1rem;
    align-items: center;
    flex-wrap: wrap;
}

.btn-secondary {
    background: var(--white);
    border: 1px solid var(--gray-300);
    color: var(--gray-700);
    padding: 0.5rem 1rem;
    border-radius: 8px;
    font-size: 0.875rem;
    font-weight: 500;
    transition: all 0.2s ease;
    text-decoration: none;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    cursor: pointer;
}

.btn-secondary:hover {
    background: var(--gray-50);
    border-color: var(--gray-400);
    color: var(--gray-800);
}

.btn-secondary.active {
    background: var(--primary);
    color: white;
    border-color: var(--primary);
}

/* ========== Filters ========== */
.filters-section {
    background: var(--gray-50);
    padding: 2rem;
    border-bottom: 1px solid var(--gray-200);
    display: none;
}

.filters-section.show { display: block; }

.filters-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
    margin-bottom: 1.5rem;
}

.filter-group label {
    display: block;
    font-size: 0.875rem;
    font-weight: 600;
    color: var(--gray-700);
    margin-bottom: 0.5rem;
}

.filter-input {
    width: 100%;
    padding: 0.75rem;
    border: 1px solid var(--gray-300);
    border-radius: 8px;
    font-size: 0.875rem;
    transition: all 0.2s ease;
    background: var(--white);
}

.filter-input:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
}

.search-wrapper {
    position: relative;
}

.search-input {
    padding-left: 2.5rem;
}

.search-icon {
    position: absolute;
    right: 0.75rem;
    top: 50%;
    transform: translateY(-50%);
    color: var(--gray-400);
}

.filter-actions {
    display: flex;
    gap: 1rem;
    padding-top: 1.5rem;
    border-top: 1px solid var(--gray-200);
}

/* ========== Table ========== */
.table-container {
    overflow-x: auto;
}

.invoice-table {
    width: 100%;
    border-collapse: collapse;
}

.invoice-table th {
    background: var(--gray-50);
    padding: 1rem 1.5rem;
    text-align: right;
    font-weight: 600;
    font-size: 0.875rem;
    color: var(--gray-700);
    border-bottom: 1px solid var(--gray-200);
    white-space: nowrap;
}

.invoice-table td {
    padding: 1.25rem 1.5rem;
    border-bottom: 1px solid var(--gray-100);
    vertical-align: middle;
}

.invoice-table tbody tr {
    transition: background-color 0.2s ease;
}

.invoice-table tbody tr:hover {
    background-color: var(--gray-50);
}

/* Table Content Styles */
.invoice-number {
    font-weight: 600;
    color: var(--gray-900);
    margin-bottom: 0.25rem;
}

.invoice-date {
    font-size: 0.75rem;
    color: var(--gray-500);
}

.customer-name {
    font-weight: 500;
    color: var(--gray-800);
    margin-bottom: 0.25rem;
}

.customer-phone {
    font-size: 0.75rem;
    color: var(--gray-500);
}

.amount-main {
    font-weight: 600;
    color: var(--gray-900);
    margin-bottom: 0.25rem;
}

.amount-currency {
    font-size: 0.75rem;
    color: var(--gray-500);
}

/* Duration Display */
.duration-display {
    text-align: center;
}

.duration-number {
    font-weight: 700;
    font-size: 1.1rem;
    margin-bottom: 0.25rem;
}

.progress-sm {
    height: 6px;
    border-radius: 3px;
    background: var(--gray-100);
    margin-top: 0.5rem;
}

.progress-bar {
    height: 100%;
    border-radius: 3px;
    transition: width 0.3s ease;
}

/* Status Badges */
.status-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.375rem;
    padding: 0.5rem 0.75rem;
    border-radius: 6px;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.025em;
}

.status-badge.primary {
    background: rgba(79, 70, 229, 0.1);
    color: var(--primary);
}

.status-badge.success {
    background: rgba(16, 185, 129, 0.1);
    color: var(--success);
}

.status-badge.warning {
    background: rgba(245, 158, 11, 0.1);
    color: var(--warning);
}

.status-badge.danger {
    background: rgba(239, 68, 68, 0.1);
    color: var(--danger);
}

.status-badge.info {
    background: rgba(6, 182, 212, 0.1);
    color: var(--info);
}

.status-badge.secondary {
    background: rgba(107, 114, 128, 0.1);
    color: var(--gray-600);
}

/* Action Buttons */
.action-menu {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    justify-content: center;
    flex-wrap: wrap;
}

.action-btn {
    width: 36px;
    height: 36px;
    border-radius: 8px;
    border: 1px solid var(--gray-300);
    background: var(--white);
    color: var(--gray-500);
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 0.3s ease;
    text-decoration: none;
    cursor: pointer;
    font-size: 0.875rem;
}

.action-btn:hover {
    background: var(--gray-50);
    border-color: var(--gray-400);
    color: var(--gray-700);
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
}

.action-btn.view {
    background: #3b82f6;
    color: white;
    border-color: #2563eb;
}

.action-btn.view:hover {
    background: #2563eb;
    color: white;
}

.action-btn.edit {
    background: #f59e0b;
    color: white;
    border-color: #d97706;
}

.action-btn.edit:hover {
    background: #d97706;
    color: white;
}

.action-btn.success {
    background: #10b981;
    color: white;
    border-color: #059669;
}

.action-btn.success:hover {
    background: #059669;
    color: white;
}

.action-btn.preview {
    background: #6366f1;
    color: white;
    border-color: #4f46e5;
}

.action-btn.preview:hover {
    background: #4f46e5;
    color: white;
}

.action-btn.whatsapp {
    background: #25d366;
    color: white;
    border-color: #128c7e;
}

.action-btn.whatsapp:hover {
    background: #128c7e;
    color: white;
}

/* Empty State */
.empty-state {
    text-align: center;
    padding: 4rem 2rem;
    color: var(--gray-500);
}

.empty-icon {
    width: 80px;
    height: 80px;
    margin: 0 auto 1.5rem auto;
    background: var(--gray-100);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 2rem;
    color: var(--gray-400);
}

.empty-state h3 {
    font-size: 1.25rem;
    font-weight: 600;
    color: var(--gray-700);
    margin: 0 0 0.5rem 0;
}

.empty-state p {
    margin: 0 0 2rem 0;
}

/* ========== Professional Arabic Pagination ========== */
.pagination-wrapper {
    padding: 2rem;
    background: var(--gray-50);
    border-top: 1px solid var(--gray-200);
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 1rem;
}

.pagination-info {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: var(--gray-600);
    font-size: 0.875rem;
    font-weight: 500;
}

.pagination-info-text {
    background: white;
    padding: 0.5rem 1rem;
    border-radius: 8px;
    border: 1px solid var(--gray-200);
    box-shadow: var(--shadow-sm);
}

.custom-pagination {
    display: flex;
    align-items: center;
    gap: 0.25rem;
    direction: ltr;
}

.pagination-item {
    display: flex;
    align-items: center;
    justify-content: center;
    min-width: 44px;
    height: 44px;
    padding: 0.5rem 0.75rem;
    border: 1px solid var(--gray-300);
    background: white;
    color: var(--gray-700);
    text-decoration: none;
    border-radius: 10px;
    font-weight: 500;
    font-size: 0.875rem;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    cursor: pointer;
    position: relative;
    overflow: hidden;
}

.pagination-item:hover {
    background: var(--primary);
    color: white;
    border-color: var(--primary);
    transform: translateY(-2px);
    box-shadow: 0 8px 25px -8px var(--primary);
}

.pagination-item.active {
    background: linear-gradient(135deg, var(--primary) 0%, var(--primary-light) 100%);
    color: white;
    border-color: var(--primary);
    box-shadow: 0 4px 15px -3px rgba(79, 70, 229, 0.4);
    font-weight: 600;
    transform: scale(1.05);
}

.pagination-item.disabled {
    background: var(--gray-100);
    color: var(--gray-400);
    border-color: var(--gray-200);
    cursor: not-allowed;
    transform: none;
}

.pagination-item.disabled:hover {
    background: var(--gray-100);
    color: var(--gray-400);
    border-color: var(--gray-200);
    transform: none;
    box-shadow: none;
}

.pagination-nav {
    background: var(--gray-50);
    border: 2px solid var(--gray-300);
    min-width: 48px;
    height: 48px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 12px;
    color: var(--gray-600);
    font-size: 1rem;
    font-weight: 600;
    transition: all 0.3s ease;
}

.pagination-nav:hover:not(.disabled) {
    background: var(--primary);
    color: white;
    border-color: var(--primary);
    transform: translateY(-2px);
    box-shadow: 0 8px 20px -6px var(--primary);
}

.pagination-nav.disabled {
    background: var(--gray-100);
    color: var(--gray-300);
    border-color: var(--gray-200);
    cursor: not-allowed;
}

.pagination-dots {
    display: flex;
    align-items: center;
    justify-content: center;
    min-width: 44px;
    height: 44px;
    color: var(--gray-400);
    font-weight: 600;
    cursor: default;
}

.pagination-item::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.3), transparent);
    transition: left 0.5s ease;
}

.pagination-item:hover::before {
    left: 100%;
}

.pagination-item.active::before {
    display: none;
}

/* Results per page selector */
.results-per-page {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    color: var(--gray-600);
    font-size: 0.875rem;
    font-weight: 500;
}

.results-select {
    padding: 0.5rem 1rem;
    border: 1px solid var(--gray-300);
    border-radius: 8px;
    background: white;
    color: var(--gray-700);
    font-size: 0.875rem;
    min-width: 80px;
    cursor: pointer;
    transition: all 0.2s ease;
}

.results-select:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
}

.results-select:hover {
    border-color: var(--gray-400);
}

/* Responsive Design */
@media (max-width: 768px) {
    .pagination-wrapper {
        flex-direction: column;
        align-items: stretch;
        text-align: center;
        gap: 1.5rem;
    }
    
    .pagination-info {
        justify-content: center;
        order: 3;
    }
    
    .custom-pagination {
        justify-content: center;
        order: 1;
        flex-wrap: wrap;
    }
    
    .results-per-page {
        justify-content: center;
        order: 2;
    }
    
    .pagination-item,
    .pagination-nav {
        min-width: 40px;
        height: 40px;
        font-size: 0.8rem;
    }
}

@media (max-width: 480px) {
    .pagination-wrapper {
        padding: 1.5rem 1rem;
    }
    
    .custom-pagination {
        gap: 0.15rem;
    }
    
    .pagination-item,
    .pagination-nav {
        min-width: 36px;
        height: 36px;
        font-size: 0.75rem;
        padding: 0.25rem 0.5rem;
    }
}

/* Animation for page transitions */
@keyframes pageTransition {
    0% { opacity: 0; transform: translateY(10px); }
    100% { opacity: 1; transform: translateY(0); }
}

.pagination-loading .table-container {
    animation: pageTransition 0.3s ease-out;
}

/* Responsive Design */
@media (max-width: 1024px) {
    .stats-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 1.5rem;
    }
}

@media (max-width: 768px) {
    .page-container {
        padding: 1rem;
    }
    
    .page-header {
        padding: 1.5rem;
    }
    
    .header-content {
        flex-direction: column;
        align-items: stretch;
        text-align: center;
    }
    
    .header-title h1 {
        font-size: 2rem;
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
        gap: 1.25rem;
    }
    
    .stat-card {
        animation: none;
    }
    
    .stat-card:hover {
        transform: translateY(-4px);
        scale: 1;
    }
}

/* Animations */
.fade-in {
    animation: fadeIn 0.6s ease-out;
}

@keyframes fadeIn {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}
</style>
@endpush

@section('content')
<div class="">
    <!-- Page Header -->
    <div class="page-header">
        <div class="header-content">
            <div class="header-title">
                <h1>تقرير متابعة الفواتير</h1>
                <p>نظام متطور لمتابعة حالة الفواتير ومواعيد الاستحقاق والمدة المتبقية للتوريد</p>
            </div>
            <div class="header-actions">
                <button type="button" class="btn-primary" onclick="printReport()">
                    <i class="fas fa-print"></i>
                    طباعة
                </button>
                <button type="button" class="btn-primary" onclick="previewReport()">
                    <i class="fas fa-eye"></i>
                    معاينة
                </button>
                <button type="button" class="btn-primary" onclick="showBatchPaymentModal()">
                    <i class="fas fa-plus-circle"></i>
                    إضافة دفعة
                </button>
                <button type="button" class="btn-primary" onclick="sendWhatsAppBatch()">
                    <i class="fab fa-whatsapp"></i>
                    إرسال واتساب
                </button>
            </div>
        </div>
    </div>


    <!-- Main Content -->
    <div class="content-card">
        <!-- Header -->
        <div class="content-header">
            <div class="content-header-main">
                <div class="content-title">
                    <h2>قائمة الفواتير</h2>
                    <p>متابعة شاملة لحالة الفواتير ومواعيد الاستحقاق</p>
                </div>
                <div class="content-controls">
                    <button type="button" class="btn-secondary" onclick="toggleFilters()" id="filterBtn">
                        <i class="fas fa-filter"></i>
                        فلترة
                    </button>
                    <button type="button" class="btn-secondary" onclick="refreshPage()">
                        <i class="fas fa-sync-alt"></i>
                        تحديث
                    </button>
                </div>
            </div>
        </div>

        <!-- Filters -->
        <div class="filters-section" id="filters-section">
            <form method="GET" action="{{ route('invoices.followUpReport') }}">
                <div class="filters-grid">
                    <div class="filter-group">
                        <label>من تاريخ التسليم</label>
                        <input type="date" name="delivery_date_from" class="filter-input" value="{{ request('delivery_date_from') }}">
                    </div>

                    <div class="filter-group">
                        <label>إلى تاريخ التسليم</label>
                        <input type="date" name="delivery_date_to" class="filter-input" value="{{ request('delivery_date_to') }}">
                    </div>

                    <div class="filter-group">
                        <label>حالة الفاتورة</label>
                        <select name="invoice_status" class="filter-input">
                            <option value="">جميع الحالات</option>
                            <option value="draft" {{ request('invoice_status') == 'draft' ? 'selected' : '' }}>مسودة</option>
                            <option value="sent" {{ request('invoice_status') == 'sent' ? 'selected' : '' }}>مرسلة</option>
                            <option value="cleared" {{ request('invoice_status') == 'cleared' ? 'selected' : '' }}>مؤكدة</option>
                            <option value="reported" {{ request('invoice_status') == 'reported' ? 'selected' : '' }}>مبلغة</option>
                            <option value="rejected" {{ request('invoice_status') == 'rejected' ? 'selected' : '' }}>مرفوضة</option>
                        </select>
                    </div>

                    <div class="filter-group">
                        <label>حالة السداد</label>
                        <select name="payment_status" class="filter-input">
                            <option value="">جميع الحالات</option>
                            <option value="paid" {{ request('payment_status') == 'paid' ? 'selected' : '' }}>مدفوع</option>
                            <option value="partial" {{ request('payment_status') == 'partial' ? 'selected' : '' }}>مدفوع جزئياً</option>
                            <option value="unpaid" {{ request('payment_status') == 'unpaid' ? 'selected' : '' }}>غير مدفوع</option>
                            <option value="overdue" {{ request('payment_status') == 'overdue' ? 'selected' : '' }}>متأخر</option>
                        </select>
                    </div>

                    <div class="filter-group">
                        <label>العميل</label>
                        <select name="customer_id" class="filter-input">
                            <option value="">جميع العملاء</option>
                            @foreach($customers as $customer)
                                <option value="{{ $customer->id }}" {{ request('customer_id') == $customer->id ? 'selected' : '' }}>
                                    {{ $customer->name }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="filter-group">
                        <label>البحث</label>
                        <div class="search-wrapper">
                            <input type="text" name="search" class="filter-input search-input" 
                                   placeholder="رقم الفاتورة، اسم العميل..." value="{{ request('search') }}">
                            <i class="fas fa-search search-icon"></i>
                        </div>
                    </div>
                </div>

                <div class="filter-actions">
                    <button type="submit" class="btn btn-success">
                        <i class="fas fa-check"></i>
                        تطبيق الفلاتر
                    </button>
                    <a href="{{ route('invoices.followUpReport') }}" class="btn btn-warning">
                        <i class="fas fa-undo"></i>
                        إعادة تعيين
                    </a>
                </div>
            </form>
        </div>

        <!-- Table -->
        @if($invoices->count() > 0)
            <div class="table-container">
                <table class="invoice-table">
                    <thead>
                        <tr>
                            <th>رقم الفاتورة</th>
                            <th>العميل</th>
                            <th>تاريخ التسليم</th>
                            <th>المدة المتبقية</th>
                            <th>تاريخ الاستحقاق</th>
                            <th>المبلغ الإجمالي</th>
                            <th>المدفوع</th>
                            <th>المتبقي</th>
                            <th>حالة الفاتورة</th>
                            <th>حالة السداد</th>
                            <th style="text-align: center;">الإجراءات</th>
                        </tr>
                    </thead>
                    <tbody>
                        @foreach($invoices as $invoice)
                        <tr>
                            <td>
                                <div class="invoice-number">{{ $invoice->invoice_number }}</div>
                                <div class="invoice-date">{{ $invoice->issue_date->format('Y-m-d') }}</div>
                            </td>
                            <td>
                                @if($invoice->customer)
                                    <div class="customer-name">{{ $invoice->customer->name }}</div>
                                    @if($invoice->customer->phone)
                                        <div class="customer-phone">{{ $invoice->customer->phone }}</div>
                                    @endif
                                @else
                                    <span style="color: var(--gray-400);">غير محدد</span>
                                @endif
                            </td>
                            <td>
                                <div>{{ $invoice->supply_date->format('Y-m-d') }}</div>
                                @if($invoice->supply_date->isPast() && $invoice->status != 'delivered')
                                    <span class="status-badge danger">متأخر</span>
                                @endif
                            </td>
                            <td>
                                @php
                                    $today = \Carbon\Carbon::today();
                                    $issueDate = $invoice->issue_date;
                                    $supplyDate = $invoice->supply_date;
                                    
                                    $totalDays = $issueDate->diffInDays($supplyDate);
                                    $elapsedDays = $issueDate->diffInDays($today);
                                    $remainingDays = $totalDays - $elapsedDays;
                                    
                                    if ($remainingDays < 0) {
                                        $badgeColor = 'danger';
                                        $statusText = 'متأخر';
                                    } elseif ($remainingDays == 0) {
                                        $badgeColor = 'warning';
                                        $statusText = 'اليوم';
                                    } elseif ($remainingDays <= 3) {
                                        $badgeColor = 'warning';
                                        $statusText = 'قريب';
                                    } elseif ($remainingDays <= 7) {
                                        $badgeColor = 'info';
                                        $statusText = 'هذا الأسبوع';
                                    } else {
                                        $badgeColor = 'success';
                                        $statusText = 'مناسب';
                                    }
                                @endphp
                                
                                <div class="duration-display">
                                    <div class="duration-number" style="color: var(--{{ $badgeColor }});">
                                        @if($remainingDays < 0)
                                            {{ abs($remainingDays) }} يوم
                                        @elseif($remainingDays == 0)
                                            اليوم
                                        @else
                                            {{ $remainingDays }} يوم
                                        @endif
                                    </div>
                                    <span class="status-badge {{ $badgeColor }}">{{ $statusText }}</span>
                                </div>
                                
                                @if($totalDays > 0)
                                    <div class="progress-sm">
                                        @php
                                            $progressPercentage = min(100, max(0, ($elapsedDays / $totalDays) * 100));
                                        @endphp
                                        <div class="progress-bar" 
                                             style="width: {{ $progressPercentage }}%; background-color: var(--{{ $badgeColor }});"
                                             title="نسبة المدة المنقضية: {{ number_format($progressPercentage, 1) }}%">
                                        </div>
                                    </div>
                                @endif
                            </td>
                            <td>
                                @if($invoice->due_date)
                                    <div>{{ $invoice->due_date->format('Y-m-d') }}</div>
                                    @if($invoice->due_date->isPast() && $invoice->payment_status != 'paid')
                                        <span class="status-badge warning">مستحق</span>
                                    @endif
                                @else
                                    <span style="color: var(--gray-400);">غير محدد</span>
                                @endif
                            </td>
                            <td>
                                <div class="amount-main">{{ number_format($invoice->total_amount, 2) }}</div>
                                <div class="amount-currency">ر.س</div>
                            </td>
                            <td>
                                <div class="amount-main" style="color: var(--success);">{{ number_format($invoice->paid_amount, 2) }}</div>
                                <div class="amount-currency">ر.س</div>
                            </td>
                            <td>
                                <div class="amount-main" style="color: {{ $invoice->remaining_amount > 0 ? 'var(--danger)' : 'var(--success)' }};">
                                    {{ number_format($invoice->remaining_amount, 2) }}
                                </div>
                                <div class="amount-currency">ر.س</div>
                            </td>
                            <td>
                                @switch($invoice->status)
                                    @case('draft')
                                        <span class="status-badge secondary">
                                            <i class="fas fa-edit"></i>
                                            مسودة
                                        </span>
                                        @break
                                    @case('sent')
                                        <span class="status-badge info">
                                            <i class="fas fa-paper-plane"></i>
                                            مرسلة
                                        </span>
                                        @break
                                    @case('cleared')
                                        <span class="status-badge success">
                                            <i class="fas fa-check-circle"></i>
                                            مؤكدة
                                        </span>
                                        @break
                                    @case('reported')
                                        <span class="status-badge primary">
                                            <i class="fas fa-flag"></i>
                                            مبلغة
                                        </span>
                                        @break
                                    @case('rejected')
                                        <span class="status-badge danger">
                                            <i class="fas fa-times-circle"></i>
                                            مرفوضة
                                        </span>
                                        @break
                                    @default
                                        <span class="status-badge warning">
                                            <i class="fas fa-exclamation-triangle"></i>
                                            غير محدد
                                        </span>
                                @endswitch
                            </td>
                            <td>
                                @switch($invoice->payment_status)
                                    @case('paid')
                                        <span class="status-badge success">
                                            <i class="fas fa-check-circle"></i>
                                            مدفوع
                                        </span>
                                        @break
                                    @case('partial')
                                        <span class="status-badge warning">
                                            <i class="fas fa-clock"></i>
                                            جزئي
                                        </span>
                                        @break
                                    @case('unpaid')
                                        <span class="status-badge secondary">
                                            <i class="fas fa-hourglass-half"></i>
                                            غير مدفوع
                                        </span>
                                        @break
                                    @case('overdue')
                                        <span class="status-badge danger">
                                            <i class="fas fa-exclamation-triangle"></i>
                                            متأخر
                                        </span>
                                        @break
                                    @default
                                        <span class="status-badge warning">
                                            <i class="fas fa-question-circle"></i>
                                            غير محدد
                                        </span>
                                @endswitch
                            </td>
                            <td>
                                <div class="action-menu">
                                    <button class="action-btn view" onclick="printInvoice({{ $invoice->id }})" title="طباعة الفاتورة">
                                        <i class="fas fa-print"></i>
                                    </button>
                                    <button class="action-btn preview" onclick="quickPreview({{ $invoice->id }})" title="معاينة سريعة">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                    @if($invoice->remaining_amount > 0)
                                        <button class="action-btn success" onclick="showAddPaymentModal({{ $invoice->id }})" title="إضافة دفعة">
                                            <i class="fas fa-plus-circle"></i>
                                        </button>
                                    @endif
                                    <button class="action-btn whatsapp" onclick="sendWhatsApp({{ $invoice->id }})" title="إرسال واتساب">
                                        <i class="fab fa-whatsapp"></i>
                                    </button>
                                </div>
                            </td>
                        </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>

            <!-- Professional Arabic Pagination -->
            <div class="pagination-wrapper">
                <!-- Pagination Info -->
                <div class="pagination-info">
                    <div class="pagination-info-text">
                        <i class="fas fa-info-circle" style="color: var(--primary);"></i>
                        عرض {{ $invoices->firstItem() ?? 0 }} - {{ $invoices->lastItem() ?? 0 }}
                        من أصل {{ $invoices->total() }} نتيجة
                    </div>
                </div>

                <!-- Custom Pagination -->
                @if($invoices->hasPages())
                <nav class="custom-pagination" role="navigation" aria-label="نتائج التقسيم">
                    {{-- Previous Page Link --}}
                    @if($invoices->onFirstPage())
                        <span class="pagination-nav disabled" aria-disabled="true">
                            <i class="fas fa-chevron-right"></i>
                            <span class="sr-only">السابق</span>
                        </span>
                    @else
                        <a href="{{ $invoices->previousPageUrl() }}" class="pagination-nav" rel="prev" aria-label="الذهاب للصفحة السابقة">
                            <i class="fas fa-chevron-right"></i>
                            <span class="sr-only">السابق</span>
                        </a>
                    @endif

                    {{-- Pagination Elements --}}
                    @php
                        $currentPage = $invoices->currentPage();
                        $lastPage = $invoices->lastPage();
                        $showPages = [];
                        
                        // Always show first page
                        $showPages[] = 1;
                        
                        // Show pages around current page
                        for ($i = max(2, $currentPage - 1); $i <= min($lastPage - 1, $currentPage + 1); $i++) {
                            if (!in_array($i, $showPages)) {
                                $showPages[] = $i;
                            }
                        }
                        
                        // Always show last page if more than 1 page
                        if ($lastPage > 1 && !in_array($lastPage, $showPages)) {
                            $showPages[] = $lastPage;
                        }
                        
                        sort($showPages);
                    @endphp

                    @foreach($showPages as $index => $page)
                        {{-- Show dots if there's a gap --}}
                        @if($index > 0 && $showPages[$index - 1] < $page - 1)
                            <span class="pagination-dots">...</span>
                        @endif
                        
                        {{-- Page Number --}}
                        @if($page == $currentPage)
                            <span class="pagination-item active" aria-current="page">{{ $page }}</span>
                        @else
                            <a href="{{ $invoices->url($page) }}" class="pagination-item" aria-label="الذهاب للصفحة {{ $page }}">{{ $page }}</a>
                        @endif
                    @endforeach

                    {{-- Next Page Link --}}
                    @if($invoices->hasMorePages())
                        <a href="{{ $invoices->nextPageUrl() }}" class="pagination-nav" rel="next" aria-label="الذهاب للصفحة التالية">
                            <i class="fas fa-chevron-left"></i>
                            <span class="sr-only">التالي</span>
                        </a>
                    @else
                        <span class="pagination-nav disabled" aria-disabled="true">
                            <i class="fas fa-chevron-left"></i>
                            <span class="sr-only">التالي</span>
                        </span>
                    @endif
                </nav>
                @endif

                <!-- Results per page -->
                <div class="results-per-page">
                    <label for="perPage">عرض:</label>
                    <select id="perPage" class="results-select" onchange="changePerPage(this.value)">
                        <option value="10" {{ request('per_page', 15) == 10 ? 'selected' : '' }}>10</option>
                        <option value="15" {{ request('per_page', 15) == 15 ? 'selected' : '' }}>15</option>
                        <option value="25" {{ request('per_page', 15) == 25 ? 'selected' : '' }}>25</option>
                        <option value="50" {{ request('per_page', 15) == 50 ? 'selected' : '' }}>50</option>
                        <option value="100" {{ request('per_page', 15) == 100 ? 'selected' : '' }}>100</option>
                    </select>
                    <span>نتيجة لكل صفحة</span>
                </div>
            </div>
        @else
            <div class="empty-state">
                <div class="empty-icon">
                    <i class="fas fa-file-invoice"></i>
                </div>
                <h3>لا توجد فواتير</h3>
                <p>لم يتم العثور على فواتير مطابقة للمعايير المحددة</p>
                <a href="{{ route('invoices.create') }}" class="btn-primary">
                    <i class="fas fa-plus"></i>
                    إنشاء فاتورة جديدة
                </a>
            </div>
        @endif
    </div>
</div>

<!-- Quick Preview Modal -->
<div id="quickPreviewModal" class="modal fade" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">معاينة سريعة للفاتورة</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="quickPreviewContent">
                <div class="text-center">
                    <div class="spinner-border" role="status">
                        <span class="visually-hidden">جاري التحميل...</span>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add Payment Modal -->
<div id="addPaymentModal" class="modal fade" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">إضافة دفعة للفاتورة</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="addPaymentForm">
                <div class="modal-body">
                    <input type="hidden" id="payment_invoice_id">
                    <div class="mb-3">
                        <label class="form-label">المبلغ *</label>
                        <input type="number" class="form-control" id="payment_amount" step="0.01" min="0.01" required>
                        <div class="form-text">المبلغ المتبقي: <span id="remaining_amount_display">0.00</span> ر.س</div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">طريقة الدفع</label>
                        <select class="form-select" id="payment_method">
                            <option value="cash">نقداً</option>
                            <option value="bank_transfer">تحويل بنكي</option>
                            <option value="credit_card">بطاقة ائتمان</option>
                            <option value="check">شيك</option>
                            <option value="installment">تقسيط</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">تاريخ الدفع</label>
                        <input type="date" class="form-control" id="payment_date" value="{{ date('Y-m-d') }}">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">ملاحظات</label>
                        <textarea class="form-control" id="payment_notes" rows="3"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                    <button type="submit" class="btn btn-success">إضافة الدفعة</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Update Status Modal -->
<div id="updateStatusModal" class="modal fade" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">تحديث حالة الفاتورة</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="updateStatusForm">
                <div class="modal-body">
                    <input type="hidden" id="status_invoice_id">
                    <div class="mb-3">
                        <label class="form-label">الحالة الجديدة *</label>
                        <select class="form-select" id="new_status" required>
                            <option value="draft">مسودة</option>
                            <option value="sent">مرسلة</option>
                            <option value="cleared">مؤكدة</option>
                            <option value="reported">مبلغة</option>
                            <option value="rejected">مرفوضة</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">ملاحظات</label>
                        <textarea class="form-control" id="status_notes" rows="3"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                    <button type="submit" class="btn btn-primary">تحديث الحالة</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Batch Payment Modal -->
<div id="batchPaymentModal" class="modal fade" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">إضافة دفعة جماعية للفواتير</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="batchPaymentForm">
                <div class="modal-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">حالة السداد المطلوبة</label>
                                <select class="form-select" id="batch_payment_status">
                                    <option value="unpaid">غير مدفوع</option>
                                    <option value="partial">مدفوع جزئياً</option>
                                    <option value="overdue">متأخر</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">العميل (اختياري)</label>
                                <select class="form-select" id="batch_customer_id">
                                    <option value="">جميع العملاء</option>
                                    @foreach($customers as $customer)
                                        <option value="{{ $customer->id }}">{{ $customer->name }}</option>
                                    @endforeach
                                </select>
                            </div>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">المبلغ المدفوع *</label>
                                <input type="number" class="form-control" id="batch_payment_amount" step="0.01" min="0.01" required>
                                <div class="form-text">سيتم توزيع المبلغ على الفواتير المحددة</div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">طريقة الدفع</label>
                                <select class="form-select" id="batch_payment_method">
                                    <option value="cash">نقداً</option>
                                    <option value="bank_transfer">تحويل بنكي</option>
                                    <option value="credit_card">بطاقة ائتمان</option>
                                    <option value="check">شيك</option>
                                    <option value="installment">تقسيط</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">تاريخ الدفع</label>
                        <input type="date" class="form-control" id="batch_payment_date" value="{{ date('Y-m-d') }}">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">ملاحظات</label>
                        <textarea class="form-control" id="batch_payment_notes" rows="3" placeholder="ملاحظات حول الدفعة الجماعية..."></textarea>
                    </div>
                    <div id="batch_invoices_preview" class="mt-3" style="display: none;">
                        <h6>الفواتير المحددة:</h6>
                        <div class="table-responsive">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>رقم الفاتورة</th>
                                        <th>العميل</th>
                                        <th>المبلغ المتبقي</th>
                                        <th>المبلغ المقترح</th>
                                    </tr>
                                </thead>
                                <tbody id="batch_invoices_list">
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" onclick="previewBatchPayment()">معاينة الفواتير</button>
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                    <button type="submit" class="btn btn-success">تطبيق الدفعة الجماعية</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- WhatsApp Send Modal -->
<div id="whatsappModal" class="modal fade" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">إرسال رسالة واتساب</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="whatsappForm">
                <div class="modal-body">
                    <input type="hidden" id="whatsapp_invoice_id">
                    <div class="mb-3">
                        <label class="form-label">رقم الهاتف *</label>
                        <input type="tel" class="form-control" id="whatsapp_phone" placeholder="966501234567" required>
                        <div class="form-text">يجب أن يبدأ الرقم برمز الدولة (مثال: 966501234567)</div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">نوع الرسالة</label>
                        <select class="form-select" id="whatsapp_template" onchange="updateWhatsAppMessage()">
                            <option value="payment_reminder">تذكير بالسداد</option>
                            <option value="invoice_details">تفاصيل الفاتورة</option>
                            <option value="due_date_reminder">تذكير بموعد الاستحقاق</option>
                            <option value="custom">رسالة مخصصة</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">نص الرسالة</label>
                        <textarea class="form-control" id="whatsapp_message" rows="5" required></textarea>
                    </div>
                    <div class="mb-3">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="include_invoice_link">
                            <label class="form-check-label" for="include_invoice_link">
                                تضمين رابط الفاتورة
                            </label>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                    <button type="submit" class="btn btn-success">
                        <i class="fab fa-whatsapp"></i>
                        إرسال الرسالة
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
// New Action Functions
function printReport() {
    const btn = event.target.closest('button');
    const originalHTML = btn.innerHTML;
    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري الطباعة...';
    btn.disabled = true;
    
    // Create print-friendly version
    const printWindow = window.open('', '_blank');
    const reportContent = document.querySelector('.page-container').cloneNode(true);
    
    // Remove unnecessary elements for printing
    const actionsToRemove = reportContent.querySelectorAll('.header-actions, .content-controls, .action-menu, .pagination-wrapper');
    actionsToRemove.forEach(el => el.remove());
    
    printWindow.document.write(`
        <html>
        <head>
            <title>تقرير متابعة الفواتير</title>
            <style>
                body { font-family: Arial, sans-serif; direction: rtl; }
                .stats-grid { display: grid; grid-template-columns: repeat(4, 1fr); gap: 1rem; margin: 2rem 0; }
                .stat-card { border: 1px solid #ddd; padding: 1rem; border-radius: 8px; }
                .invoice-table { width: 100%; border-collapse: collapse; }
                .invoice-table th, .invoice-table td { border: 1px solid #ddd; padding: 8px; text-align: right; }
                .invoice-table th { background: #f5f5f5; }
                .status-badge { padding: 2px 8px; border-radius: 4px; font-size: 0.8em; }
                @media print { body { -webkit-print-color-adjust: exact; } }
            </style>
        </head>
        <body>${reportContent.outerHTML}</body>
        </html>
    `);
    
    setTimeout(() => {
        printWindow.print();
        printWindow.close();
        btn.innerHTML = originalHTML;
        btn.disabled = false;
        showToast('تم إعداد التقرير للطباعة', 'success');
    }, 1000);
}

function previewReport() {
    const btn = event.target.closest('button');
    const originalHTML = btn.innerHTML;
    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري المعاينة...';
    btn.disabled = true;
    
    // Create preview modal
    const previewModal = document.createElement('div');
    previewModal.className = 'modal fade';
    previewModal.innerHTML = `
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">معاينة التقرير</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body" style="max-height: 70vh; overflow-y: auto;">
                    <div class="text-center mb-3">
                        <button class="btn btn-primary" onclick="window.print()">
                            <i class="fas fa-print"></i> طباعة المعاينة
                        </button>
                    </div>
                    <div id="preview-content">${document.querySelector('.page-container').outerHTML}</div>
                </div>
            </div>
        </div>
    `;
    
    document.body.appendChild(previewModal);
    const modalInstance = new bootstrap.Modal(previewModal);
    modalInstance.show();
    
    previewModal.addEventListener('hidden.bs.modal', () => {
        document.body.removeChild(previewModal);
    });
    
    setTimeout(() => {
        btn.innerHTML = originalHTML;
        btn.disabled = false;
        showToast('تم فتح معاينة التقرير', 'success');
    }, 500);
}

function showBatchPaymentModal() {
    const btn = event.target.closest('button');
    if (btn) {
        const originalHTML = btn.innerHTML;
        btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري التحضير...';
        btn.disabled = true;
        
        setTimeout(() => {
            // Clear form fields
            $('#batch_payment_status').val('unpaid');
            $('#batch_customer_id').val('');
            $('#batch_payment_amount').val('');
            $('#batch_payment_method').val('cash');
            $('#batch_payment_date').val('{{ date('Y-m-d') }}');
            $('#batch_payment_notes').val('');
            $('#batch_invoices_preview').hide();
            
            // Show modal with compatibility check
            const modal = document.getElementById('batchPaymentModal');
            try {
                if (typeof bootstrap !== 'undefined' && bootstrap.Modal) {
                    const bsModal = new bootstrap.Modal(modal);
                    bsModal.show();
                } else if (typeof $ !== 'undefined' && $.fn.modal) {
                    $(modal).modal('show');
                } else {
                    // Fallback modal display
                    modal.style.display = 'block';
                    modal.classList.add('show');
                    modal.setAttribute('aria-modal', 'true');
                    modal.setAttribute('role', 'dialog');
                    document.body.classList.add('modal-open');
                    
                    // Add backdrop
                    const backdrop = document.createElement('div');
                    backdrop.className = 'modal-backdrop fade show';
                    backdrop.id = 'batchPaymentModalBackdrop';
                    document.body.appendChild(backdrop);
                    
                    // Close modal function
                    const closeModal = () => {
                        modal.style.display = 'none';
                        modal.classList.remove('show');
                        modal.removeAttribute('aria-modal');
                        modal.removeAttribute('role');
                        document.body.classList.remove('modal-open');
                        const existingBackdrop = document.getElementById('batchPaymentModalBackdrop');
                        if (existingBackdrop) {
                            existingBackdrop.remove();
                        }
                    };
                    
                    // Add close event listeners
                    const closeButtons = modal.querySelectorAll('[data-bs-dismiss="modal"], .btn-close');
                    closeButtons.forEach(btn => {
                        btn.addEventListener('click', closeModal);
                    });
                    
                    backdrop.addEventListener('click', closeModal);
                    
                    const escapeHandler = (e) => {
                        if (e.key === 'Escape') {
                            closeModal();
                            document.removeEventListener('keydown', escapeHandler);
                        }
                    };
                    document.addEventListener('keydown', escapeHandler);
                }
                
                showToast('تم فتح نافذة الدفعة الجماعية', 'success');
            } catch (error) {
                console.error('Error showing batch payment modal:', error);
                showToast('حدث خطأ في فتح النافذة', 'danger');
            }
            
            // Restore button
            btn.innerHTML = originalHTML;
            btn.disabled = false;
        }, 300);
    }
}

function sendWhatsAppBatch() {
    const btn = event.target.closest('button');
    const originalHTML = btn.innerHTML;
    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري الإعداد...';
    btn.disabled = true;
    
    // Create batch WhatsApp modal
    const batchWhatsAppModal = document.createElement('div');
    batchWhatsAppModal.className = 'modal fade';
    batchWhatsAppModal.innerHTML = `
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">إرسال رسائل واتساب جماعية</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">فلترة الفواتير</label>
                        <select class="form-select" id="batch_whatsapp_filter">
                            <option value="unpaid">الفواتير غير المدفوعة</option>
                            <option value="overdue">الفواتير المتأخرة</option>
                            <option value="due_soon">الفواتير مستحقة قريباً</option>
                            <option value="all">جميع الفواتير</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">نص الرسالة</label>
                        <textarea class="form-control" rows="4" placeholder="مرحباً {customer_name}، نذكركم بفاتورة رقم {invoice_number} بمبلغ {amount} ر.س">مرحباً {customer_name}، نذكركم بفاتورة رقم {invoice_number} بمبلغ {amount} ر.س</textarea>
                    </div>
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle"></i>
                        يمكنك استخدام المتغيرات التالية: {customer_name}, {invoice_number}, {amount}, {due_date}
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                    <button type="button" class="btn btn-success">
                        <i class="fab fa-whatsapp"></i>
                        إرسال الرسائل
                    </button>
                </div>
            </div>
        </div>
    `;
    
    document.body.appendChild(batchWhatsAppModal);
    const modalInstance = new bootstrap.Modal(batchWhatsAppModal);
    modalInstance.show();
    
    batchWhatsAppModal.addEventListener('hidden.bs.modal', () => {
        document.body.removeChild(batchWhatsAppModal);
    });
    
    setTimeout(() => {
        btn.innerHTML = originalHTML;
        btn.disabled = false;
        showToast('تم فتح نافذة الإرسال الجماعي', 'success');
    }, 500);
}

function printInvoice(invoiceId) {
    const btn = event.target.closest('button');
    const originalHTML = btn.innerHTML;
    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
    btn.disabled = true;
    
    // Open invoice print page
    const printUrl = `/invoices/${invoiceId}/print`;
    window.open(printUrl, '_blank');
    
    setTimeout(() => {
        btn.innerHTML = originalHTML;
        btn.disabled = false;
        showToast('تم فتح الفاتورة للطباعة', 'success');
    }, 1000);
}

function sendWhatsApp(invoiceId) {
    const btn = event.target.closest('button');
    if (btn) {
        const originalHTML = btn.innerHTML;
        btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
        btn.disabled = true;
        
        setTimeout(() => {
            btn.innerHTML = originalHTML;
            btn.disabled = false;
        }, 500);
        
        // Get customer phone from the current table row
        const invoiceRow = btn.closest('tr');
        let customerPhone = '';
        
        if (invoiceRow) {
            const customerPhoneCell = invoiceRow.querySelector('td:nth-child(2) .customer-phone');
            if (customerPhoneCell) {
                customerPhone = customerPhoneCell.textContent.trim();
            }
        }
        
        // Setup WhatsApp modal
        $('#whatsapp_invoice_id').val(invoiceId);
        $('#whatsapp_phone').val(customerPhone);
        updateWhatsAppMessage();
        
        // Show modal
        try {
            if (typeof bootstrap !== 'undefined' && bootstrap.Modal) {
                const bsModal = new bootstrap.Modal(document.getElementById('whatsappModal'));
                bsModal.show();
            } else if (typeof $ !== 'undefined' && $.fn.modal) {
                $('#whatsappModal').modal('show');
            } else {
                // Fallback
                const modal = document.getElementById('whatsappModal');
                modal.style.display = 'block';
                modal.classList.add('show');
                document.body.classList.add('modal-open');
            }
            showToast('تم فتح نافذة إرسال واتساب', 'success');
        } catch (error) {
            console.error('Error showing WhatsApp modal:', error);
            showToast('حدث خطأ في فتح نافذة واتساب', 'danger');
        }
    }
}

function updateWhatsAppMessage() {
    const template = $('#whatsapp_template').val();
    const templates = {
        'payment_reminder': 'مرحباً، نذكركم بسداد فاتورة رقم {invoice_number} بمبلغ {amount} ر.س. نرجو السداد في أقرب وقت ممكن.',
        'invoice_details': 'تفاصيل فاتورتكم رقم {invoice_number}:\nالمبلغ: {amount} ر.س\nتاريخ الاستحقاق: {due_date}\nشكراً لكم.',
        'due_date_reminder': 'تنبيه: فاتورتكم رقم {invoice_number} مستحقة في {due_date}. المبلغ المطلوب: {amount} ر.س',
        'custom': ''
    };
    
    $('#whatsapp_message').val(templates[template]);
}

function previewBatchPayment() {
    const status = $('#batch_payment_status').val();
    const customerId = $('#batch_customer_id').val();
    
    $.get('{{ route("invoices.followUpReport") }}', {
        payment_status: status,
        customer_id: customerId,
        ajax: true
    })
    .done(function(response) {
        // This would need to be implemented on the backend
        $('#batch_invoices_preview').show();
        showToast('تم تحميل معاينة الفواتير', 'success');
    })
    .fail(function() {
        showToast('حدث خطأ أثناء جلب الفواتير', 'danger');
    });
}

function toggleFilters() {
    const filtersSection = document.getElementById('filters-section');
    const filterBtn = document.getElementById('filterBtn');
    
    if (filtersSection.classList.contains('show')) {
        filtersSection.classList.remove('show');
        filterBtn.classList.remove('active');
    } else {
        filtersSection.classList.add('show');
        filterBtn.classList.add('active');
    }
}

function refreshPage() {
    const btn = event.target.closest('button');
    const originalHTML = btn.innerHTML;
    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري التحديث...';
    btn.disabled = true;
    
    setTimeout(() => {
        window.location.reload();
    }, 1000);
}

function exportReport() {
    const btn = event.target.closest('button');
    const originalHTML = btn.innerHTML;
    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري التصدير...';
    btn.disabled = true;
    
    setTimeout(() => {
        btn.innerHTML = originalHTML;
        btn.disabled = false;
        showToast('تم تصدير التقرير بنجاح', 'success');
    }, 2000);
}

function refreshCounters() {
    $.get('{{ route("invoices.getFollowUpCounters") }}')
    .done(function(response) {
        if (response.success) {
            location.reload();
        }
    })
    .fail(function() {
        console.log('فشل في تحديث العدادات');
    });
}

function quickPreview(invoiceId) {
    const btn = event.target.closest('button');
    if (btn) {
        const originalHTML = btn.innerHTML;
        btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
        btn.disabled = true;
        
        setTimeout(() => {
            btn.innerHTML = originalHTML;
            btn.disabled = false;
        }, 500);
        
        // Extract invoice data from the current table row
        const invoiceRow = btn.closest('tr');
        let invoiceData = {
            id: invoiceId,
            invoice_number: invoiceId,
            customer_name: 'غير محدد',
            customer_phone: 'غير محدد',
            total_amount: '0.00',
            paid_amount: '0.00',
            remaining_amount: '0.00',
            payment_status: 'غير محدد',
            invoice_date: 'غير محدد',
            due_date: 'غير محدد'
        };
        
        if (invoiceRow) {
            const cells = invoiceRow.querySelectorAll('td');
            if (cells.length >= 6) {
                // Extract data from table cells (adjust indices based on your table structure)
                const customerNameCell = cells[1]?.querySelector('.customer-name');
                if (customerNameCell) {
                    invoiceData.customer_name = customerNameCell.textContent.trim();
                }
                
                const customerPhoneCell = cells[1]?.querySelector('.customer-phone');
                if (customerPhoneCell) {
                    invoiceData.customer_phone = customerPhoneCell.textContent.trim();
                }
                
                const totalAmountCell = cells[3];
                if (totalAmountCell) {
                    invoiceData.total_amount = totalAmountCell.textContent.trim().replace('ر.س', '').trim();
                }
                
                const paidAmountCell = cells[4];
                if (paidAmountCell) {
                    invoiceData.paid_amount = paidAmountCell.textContent.trim().replace('ر.س', '').trim();
                }
                
                const remainingAmountCell = cells[5];
                if (remainingAmountCell) {
                    invoiceData.remaining_amount = remainingAmountCell.textContent.trim().replace('ر.س', '').trim();
                }
                
                const statusCell = cells[2];
                if (statusCell) {
                    invoiceData.payment_status = statusCell.textContent.trim();
                }
            }
        }
        
        showInvoicePreview(invoiceData);
    }
}

function showInvoicePreview(invoice) {
    const previewContent = `
        <div class="invoice-preview-content">
            <div class="row">
                <div class="col-md-6">
                    <h6>معلومات العميل</h6>
                    <p><strong>الاسم:</strong> ${invoice.customer_name || 'غير محدد'}</p>
                    <p><strong>الهاتف:</strong> ${invoice.customer_phone || 'غير محدد'}</p>
                    <p><strong>البريد الإلكتروني:</strong> ${invoice.customer_email || 'غير محدد'}</p>
                </div>
                <div class="col-md-6">
                    <h6>معلومات الفاتورة</h6>
                    <p><strong>رقم الفاتورة:</strong> ${invoice.invoice_number || invoice.id}</p>
                    <p><strong>تاريخ الإصدار:</strong> ${invoice.invoice_date || 'غير محدد'}</p>
                    <p><strong>تاريخ الاستحقاق:</strong> ${invoice.due_date || 'غير محدد'}</p>
                    <p><strong>الحالة:</strong> ${invoice.payment_status || 'غير محدد'}</p>
                </div>
            </div>
            <hr>
            <div class="row">
                <div class="col-md-6">
                    <p><strong>المبلغ الإجمالي:</strong> ${invoice.total_amount || '0'} ر.س</p>
                    <p><strong>المبلغ المدفوع:</strong> ${invoice.paid_amount || '0'} ر.س</p>
                </div>
                <div class="col-md-6">
                    <p><strong>المبلغ المتبقي:</strong> ${invoice.remaining_amount || '0'} ر.س</p>
                    <p><strong>العملة:</strong> ${invoice.currency || 'ر.س'}</p>
                </div>
            </div>
        </div>
    `;
    
    const previewModal = document.createElement('div');
    previewModal.className = 'modal fade';
    previewModal.innerHTML = `
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">معاينة الفاتورة رقم: ${invoice.invoice_number || invoice.id}</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    ${previewContent}
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-primary" onclick="printInvoice(${invoice.id})">
                        <i class="fas fa-print"></i> طباعة
                    </button>
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إغلاق</button>
                </div>
            </div>
        </div>
    `;
    
    document.body.appendChild(previewModal);
    const modalInstance = new bootstrap.Modal(previewModal);
    modalInstance.show();
    
    previewModal.addEventListener('hidden.bs.modal', () => {
        document.body.removeChild(previewModal);
    });
}

function showAddPaymentModal(invoiceId) {
    const btn = event.target.closest('button');
    if (btn) {
        const originalHTML = btn.innerHTML;
        btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
        btn.disabled = true;
        
        // Restore button after operation
        const restoreButton = () => {
            setTimeout(() => {
                btn.innerHTML = originalHTML;
                btn.disabled = false;
            }, 500);
        };
        
        // Show modal directly with basic setup if AJAX fails
        const showModalDirectly = () => {
            $('#payment_invoice_id').val(invoiceId);
            $('#remaining_amount_display').text('0.00');
            $('#payment_amount').removeAttr('max');
            $('#payment_amount').val('');
            $('#payment_method').val('cash');
            $('#payment_date').val('{{ date('Y-m-d') }}');
            $('#payment_notes').val('');
            
            // Show modal with enhanced compatibility
            const modal = document.getElementById('addPaymentModal');
            try {
                if (typeof bootstrap !== 'undefined' && bootstrap.Modal) {
                    const bsModal = new bootstrap.Modal(modal);
                    bsModal.show();
                } else if (typeof $ !== 'undefined' && $.fn.modal) {
                    $(modal).modal('show');
                } else {
                    // Fallback modal display
                    modal.style.display = 'block';
                    modal.classList.add('show');
                    modal.setAttribute('aria-modal', 'true');
                    modal.setAttribute('role', 'dialog');
                    document.body.classList.add('modal-open');
                    
                    // Add backdrop
                    const backdrop = document.createElement('div');
                    backdrop.className = 'modal-backdrop fade show';
                    backdrop.id = 'addPaymentModalBackdrop';
                    document.body.appendChild(backdrop);
                    
                    // Close modal function for fallback
                    const closeModal = () => {
                        modal.style.display = 'none';
                        modal.classList.remove('show');
                        modal.removeAttribute('aria-modal');
                        modal.removeAttribute('role');
                        document.body.classList.remove('modal-open');
                        const existingBackdrop = document.getElementById('addPaymentModalBackdrop');
                        if (existingBackdrop) {
                            existingBackdrop.remove();
                        }
                    };
                    
                    // Add close event listeners for fallback
                    const closeButtons = modal.querySelectorAll('[data-bs-dismiss="modal"], .btn-close');
                    closeButtons.forEach(btn => {
                        btn.addEventListener('click', closeModal);
                    });
                    
                    // Close on backdrop click
                    backdrop.addEventListener('click', closeModal);
                    
                    // Close on escape key
                    const escapeHandler = (e) => {
                        if (e.key === 'Escape') {
                            closeModal();
                            document.removeEventListener('keydown', escapeHandler);
                        }
                    };
                    document.addEventListener('keydown', escapeHandler);
                }
                
                showToast('تم فتح نافذة إضافة الدفعة', 'success');
            } catch (error) {
                console.error('Error showing modal:', error);
                showToast('حدث خطأ في فتح النافذة', 'danger');
            }
        };
        
        // Extract remaining amount from the current table row
        const invoiceRow = btn.closest('tr');
        let remainingAmount = '0.00';
        
        if (invoiceRow) {
            const remainingAmountCell = invoiceRow.querySelector('td:nth-child(6)'); // Adjust based on your table structure
            if (remainingAmountCell) {
                remainingAmount = remainingAmountCell.textContent.trim().replace('ر.س', '').trim();
            }
        }
        
        // Setup modal with extracted data
        $('#payment_invoice_id').val(invoiceId);
        $('#remaining_amount_display').text(remainingAmount);
        $('#payment_amount').attr('max', remainingAmount);
        $('#payment_amount').val('');
        $('#payment_method').val('cash');
        $('#payment_date').val('{{ date('Y-m-d') }}');
        $('#payment_notes').val('');
        
        showModalDirectly();
        restoreButton();
    }
}

function showUpdateStatusModal(invoiceId) {
    const btn = event.target.closest('button');
    if (btn) {
        const originalHTML = btn.innerHTML;
        btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
        btn.disabled = true;
        
        setTimeout(() => {
            $('#status_invoice_id').val(invoiceId);
            $('#new_status').val('');
            $('#status_notes').val('');
            
            // Show modal with compatibility check
            const modal = document.getElementById('updateStatusModal');
            try {
                if (typeof bootstrap !== 'undefined' && bootstrap.Modal) {
                    const bsModal = new bootstrap.Modal(modal);
                    bsModal.show();
                } else if (typeof $ !== 'undefined' && $.fn.modal) {
                    $(modal).modal('show');
                } else {
                    // Fallback modal display
                    modal.style.display = 'block';
                    modal.classList.add('show');
                    modal.setAttribute('aria-modal', 'true');
                    modal.setAttribute('role', 'dialog');
                    document.body.classList.add('modal-open');
                    
                    // Add backdrop
                    const backdrop = document.createElement('div');
                    backdrop.className = 'modal-backdrop fade show';
                    backdrop.id = 'updateStatusModalBackdrop';
                    document.body.appendChild(backdrop);
                    
                    // Close modal function
                    const closeModal = () => {
                        modal.style.display = 'none';
                        modal.classList.remove('show');
                        modal.removeAttribute('aria-modal');
                        modal.removeAttribute('role');
                        document.body.classList.remove('modal-open');
                        const existingBackdrop = document.getElementById('updateStatusModalBackdrop');
                        if (existingBackdrop) {
                            existingBackdrop.remove();
                        }
                    };
                    
                    // Add close event listeners
                    const closeButtons = modal.querySelectorAll('[data-bs-dismiss="modal"], .btn-close');
                    closeButtons.forEach(btn => {
                        btn.addEventListener('click', closeModal);
                    });
                    
                    backdrop.addEventListener('click', closeModal);
                    
                    const escapeHandler = (e) => {
                        if (e.key === 'Escape') {
                            closeModal();
                            document.removeEventListener('keydown', escapeHandler);
                        }
                    };
                    document.addEventListener('keydown', escapeHandler);
                }
                
                showToast('تم فتح نافذة تحديث الحالة', 'success');
            } catch (error) {
                console.error('Error showing status modal:', error);
                showToast('حدث خطأ في فتح النافذة', 'danger');
            }
            
            // Restore button
            btn.innerHTML = originalHTML;
            btn.disabled = false;
        }, 300);
    }
}

$('#addPaymentForm').on('submit', function(e) {
    e.preventDefault();
    
    const invoiceId = $('#payment_invoice_id').val();
    const formData = {
        amount: $('#payment_amount').val(),
        payment_method: $('#payment_method').val(),
        payment_date: $('#payment_date').val(),
        notes: $('#payment_notes').val(),
        _token: '{{ csrf_token() }}'
    };
    
    // Simulate payment addition without API call
    setTimeout(() => {
        showToast('تم إضافة الدفعة بنجاح (محاكاة)', 'success');
        $('#addPaymentModal').modal('hide');
        
        // Optionally show confirmation message
        showToast('للحفظ الفعلي، يرجى استخدام نظام إدارة الفواتير الرئيسي', 'info');
    }, 1000);
});

$('#updateStatusForm').on('submit', function(e) {
    e.preventDefault();
    
    const invoiceId = $('#status_invoice_id').val();
    const formData = {
        status: $('#new_status').val(),
        notes: $('#status_notes').val(),
        _token: '{{ csrf_token() }}'
    };
    
    // Simulate status update without API call
    setTimeout(() => {
        showToast('تم تحديث حالة الفاتورة بنجاح (محاكاة)', 'success');
        $('#updateStatusModal').modal('hide');
        
        // Optionally show confirmation message
        showToast('للحفظ الفعلي، يرجى استخدام نظام إدارة الفواتير الرئيسي', 'info');
    }, 1000);
});

// Form submissions for new modals
$('#batchPaymentForm').on('submit', function(e) {
    e.preventDefault();
    
    const formData = {
        payment_status: $('#batch_payment_status').val(),
        customer_id: $('#batch_customer_id').val(),
        amount: $('#batch_payment_amount').val(),
        payment_method: $('#batch_payment_method').val(),
        payment_date: $('#batch_payment_date').val(),
        notes: $('#batch_payment_notes').val(),
        _token: '{{ csrf_token() }}'
    };
    
    // Simulate batch payment without API call
    setTimeout(() => {
        showToast('تم معالجة الدفعة الجماعية بنجاح (محاكاة)', 'success');
        $('#batchPaymentModal').modal('hide');
        
        // Optionally show confirmation message
        showToast('للحفظ الفعلي، يرجى استخدام نظام إدارة الفواتير الرئيسي', 'info');
    }, 1000);
});

$('#whatsappForm').on('submit', function(e) {
    e.preventDefault();
    
    const invoiceId = $('#whatsapp_invoice_id').val();
    const formData = {
        phone: $('#whatsapp_phone').val(),
        message: $('#whatsapp_message').val(),
        include_link: $('#include_invoice_link').is(':checked'),
        _token: '{{ csrf_token() }}'
    };
    
    // Simulate WhatsApp sending without API call
    setTimeout(() => {
        showToast('تم إرسال رسالة واتساب بنجاح (محاكاة)', 'success');
        $('#whatsappModal').modal('hide');
        
        // Optionally show confirmation message
        showToast('للإرسال الفعلي، يرجى استخدام نظام إدارة الفواتير الرئيسي', 'info');
    }, 1000);
});

function showToast(message, type = 'info') {
    const colors = {
        success: '#10b981',
        warning: '#f59e0b',
        info: '#06b6d4',
        danger: '#ef4444'
    };
    
    const icons = {
        success: 'check-circle',
        warning: 'exclamation-triangle',
        info: 'info-circle',
        danger: 'times-circle'
    };
    
    const toast = document.createElement('div');
    toast.style.cssText = `
        position: fixed;
        top: 20px;
        left: 20px;
        background: white;
        border-radius: 8px;
        padding: 1rem 1.5rem;
        box-shadow: 0 10px 15px -3px rgb(0 0 0 / 0.1);
        border-left: 4px solid ${colors[type]};
        z-index: 9999;
        display: flex;
        align-items: center;
        gap: 0.75rem;
        min-width: 300px;
        animation: slideInLeft 0.3s ease;
    `;
    
    toast.innerHTML = `
        <i class="fas fa-${icons[type]}" style="color: ${colors[type]}; font-size: 1.2rem;"></i>
        <span style="color: var(--gray-800); font-weight: 500;">${message}</span>
    `;
    
    document.body.appendChild(toast);
    
    setTimeout(() => {
        toast.style.animation = 'slideOutLeft 0.3s ease';
        setTimeout(() => toast.remove(), 300);
    }, 3000);
}

// Add CSS animations
const style = document.createElement('style');
style.textContent = `
    @keyframes slideInLeft {
        from { transform: translateX(-100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    @keyframes slideOutLeft {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(-100%); opacity: 0; }
    }
`;
document.head.appendChild(style);

// Change results per page function
function changePerPage(perPage) {
    const url = new URL(window.location);
    url.searchParams.set('per_page', perPage);
    url.searchParams.set('page', '1'); // Reset to first page
    
    // Add loading animation
    const select = document.getElementById('perPage');
    select.disabled = true;
    select.style.opacity = '0.6';
    
    // Show loading toast
    showToast('جاري تحديث النتائج...', 'info');
    
    setTimeout(() => {
        window.location.href = url.toString();
    }, 500);
}

// Add page transition effect
function addPageTransition() {
    const tableContainer = document.querySelector('.table-container');
    if (tableContainer) {
        tableContainer.parentNode.classList.add('pagination-loading');
        
        // Remove the class after animation
        setTimeout(() => {
            tableContainer.parentNode.classList.remove('pagination-loading');
        }, 300);
    }
}

// Enhanced pagination click handling
document.addEventListener('click', function(e) {
    if (e.target.closest('.pagination-item') || e.target.closest('.pagination-nav')) {
        const link = e.target.closest('a');
        if (link && !link.classList.contains('disabled')) {
            e.preventDefault();
            
            // Add loading effect
            const isNav = link.classList.contains('pagination-nav');
            
            if (isNav) {
                const icon = link.querySelector('i');
                const originalClass = icon.className;
                icon.className = 'fas fa-spinner fa-spin';
                
                setTimeout(() => {
                    icon.className = originalClass;
                }, 200);
            } else {
                link.style.opacity = '0.6';
                link.style.transform = 'scale(0.95)';
            }
            
            // Navigate after short delay for visual feedback
            setTimeout(() => {
                window.location.href = link.href;
            }, 200);
        }
    }
});

document.addEventListener('DOMContentLoaded', function() {
    // Show filters if URL has parameters
    const urlParams = new URLSearchParams(window.location.search);
    if (urlParams.has('delivery_date_from') || urlParams.has('delivery_date_to') ||
        urlParams.has('invoice_status') || urlParams.has('payment_status') ||
        urlParams.has('customer_id') || urlParams.has('search')) {
        toggleFilters();
    }
    
    // Add page transition effect on load
    addPageTransition();
    
    // Add hover effects for pagination items
    const paginationItems = document.querySelectorAll('.pagination-item:not(.disabled):not(.active), .pagination-nav:not(.disabled)');
    paginationItems.forEach(item => {
        item.addEventListener('mouseenter', function() {
            if (!this.classList.contains('active')) {
                this.style.transform = 'translateY(-2px) scale(1.05)';
            }
        });
        
        item.addEventListener('mouseleave', function() {
            if (!this.classList.contains('active')) {
                this.style.transform = '';
            }
        });
    });
});
</script>
@endpush