@extends('layouts.app')

@section('title', 'تعديل الفاتورة رقم ' . $invoice->invoice_number)
@section('page-title', 'تعديل الفاتورة')

@section('page-actions')
<div class="btn-group">
    <a href="{{ route('invoices.show', $invoice) }}" class="btn btn-secondary">
        <i class="fas fa-eye me-2"></i>
        عرض الفاتورة
    </a>
    <a href="{{ route('invoices.index') }}" class="btn btn-outline-secondary">
        <i class="fas fa-arrow-left me-2"></i>
        العودة للقائمة
    </a>
</div>
@endsection

@section('content')
<form method="POST" action="{{ route('invoices.update', $invoice) }}" id="invoice-form">
    @csrf
    @method('PUT')
    
    <div class="row">
        <!-- معلومات الفاتورة الأساسية -->
        <div class="col-lg-8">
            <div class="card">
                <div class="card-header">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-file-invoice me-2"></i>
                        تعديل الفاتورة رقم: {{ $invoice->invoice_number }}
                    </h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <label class="form-label">نوع الفاتورة</label>
                            <select name="type" class="form-select" disabled>
                                <option value="standard" {{ $invoice->type == 'standard' ? 'selected' : '' }}>
                                    فاتورة ضريبية معيارية (B2B)
                                </option>
                                <option value="simplified" {{ $invoice->type == 'simplified' ? 'selected' : '' }}>
                                    فاتورة ضريبية مبسطة (B2C)
                                </option>
                            </select>
                            <small class="text-muted">لا يمكن تغيير نوع الفاتورة بعد الإنشاء</small>
                        </div>
                        
                        <div class="col-md-6">
                            <label class="form-label">العميل</label>
                            <select name="customer_id" class="form-select @error('customer_id') is-invalid @enderror" id="customer-select">
                                <option value="">اختر العميل (اختياري للفواتير المبسطة)</option>
                                @foreach($customers as $customer)
                                    <option value="{{ $customer->id }}" {{ old('customer_id', $invoice->customer_id) == $customer->id ? 'selected' : '' }}>
                                        {{ $customer->name }}
                                        @if($customer->customer_type == 'business' && $customer->vat_number)
                                            ({{ $customer->vat_number }})
                                        @endif
                                    </option>
                                @endforeach
                            </select>
                            @error('customer_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    
                    <div class="row mt-3">
                        <div class="col-md-4">
                            <label class="form-label">تاريخ الإصدار <span class="text-danger">*</span></label>
                            <input type="date" name="issue_date" class="form-control @error('issue_date') is-invalid @enderror" 
                                   value="{{ old('issue_date', $invoice->issue_date->format('Y-m-d')) }}" required>
                            @error('issue_date')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                        
                        <div class="col-md-4">
                            <label class="form-label">تاريخ الاستحقاق</label>
                            <input type="date" name="due_date" class="form-control @error('due_date') is-invalid @enderror" 
                                   value="{{ old('due_date', $invoice->due_date?->format('Y-m-d')) }}">
                            @error('due_date')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                        
                        <div class="col-md-4">
                            <label class="form-label">تاريخ التوريد</label>
                            <input type="date" name="supply_date" class="form-control @error('supply_date') is-invalid @enderror" 
                                   value="{{ old('supply_date', $invoice->supply_date->format('Y-m-d')) }}">
                            @error('supply_date')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    
                    <div class="row mt-3">
                        <div class="col-12">
                            <label class="form-label">ملاحظات</label>
                            <textarea name="notes" class="form-control" rows="3" placeholder="ملاحظات إضافية على الفاتورة">{{ old('notes', $invoice->notes) }}</textarea>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- عناصر الفاتورة -->
            <div class="card mt-4">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-list me-2"></i>
                        عناصر الفاتورة
                    </h5>
                    <button type="button" class="btn btn-primary btn-sm" onclick="addInvoiceItem()">
                        <i class="fas fa-plus me-1"></i>
                        إضافة عنصر
                    </button>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table" id="invoice-items-table">
                                <tr>
                                    <th>المنتج/الخدمة</th>
                                    <th>الكمية</th>
                                    <th>سعر الوحدة</th>
                                    <th>الخصم</th>
                                    <th>الإجمالي</th>
                                    <th>الضريبة</th>
                                    <th>الإجمالي مع الضريبة</th>
                                    <th>إجراء</th>
                                </tr>
                            <tbody id="invoice-items">
                                @foreach($invoice->items as $index => $item)
                                <tr id="item-{{ $index }}">
                                    <td>
                                        <input type="hidden" name="items[{{ $index }}][product_id]" value="{{ $item->product_id }}">
                                        <strong>{{ $item->name }}</strong>
                                        @if($item->description)
                                            <br><small class="text-muted">{{ $item->description }}</small>
                                        @endif
                                        <br><small class="text-info">رمز: {{ $item->sku }}</small>
                                    </td>
                                    <td>
                                        <input type="number" name="items[{{ $index }}][quantity]" class="form-control" 
                                               value="{{ $item->quantity }}" step="0.001" min="0.001" required onchange="calculateItemTotal({{ $index }})">
                                    </td>
                                    <td>
                                        <input type="number" name="items[{{ $index }}][unit_price]" class="form-control" 
                                               value="{{ $item->unit_price }}" step="0.01" min="0" required onchange="calculateItemTotal({{ $index }})">
                                    </td>
                                    <td>
                                        <input type="number" name="items[{{ $index }}][discount_amount]" class="form-control" 
                                               value="{{ $item->discount_amount }}" step="0.01" min="0" onchange="calculateItemTotal({{ $index }})">
                                    </td>
                                    <td>
                                        <span id="line-total-{{ $index }}">{{ number_format($item->line_total, 2) }}</span> ر.س
                                    </td>
                                    <td>
                                        <span id="tax-amount-{{ $index }}">{{ number_format($item->tax_amount, 2) }}</span> ر.س
                                    </td>
                                    <td>
                                        <strong><span id="total-with-tax-{{ $index }}">{{ number_format($item->total_with_tax, 2) }}</span> ر.س</strong>
                                    </td>
                                    <td>
                                        <button type="button" class="btn btn-danger btn-sm" onclick="removeItem({{ $index }})">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                                @endforeach
                            </tbody>
                        </table>
                    </div>
                    
                    @error('items')
                        <div class="alert alert-danger mt-2">{{ $message }}</div>
                    @enderror
                </div>
            </div>
        </div>
        
        <!-- ملخص الفاتورة -->
        <div class="col-lg-4">
            <div class="card">
                <div class="card-header">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-calculator me-2"></i>
                        ملخص الفاتورة
                    </h5>
                </div>
                <div class="card-body">
                    <!-- معلومات الشركة -->
                    <div class="mb-3">
                        <h6 class="text-muted">معلومات الشركة:</h6>
                        <p class="mb-1"><strong>{{ $invoice->company->name }}</strong></p>
                        <p class="mb-1 text-muted">الرقم الضريبي: {{ $invoice->company->vat_number }}</p>
                        <p class="mb-1 text-muted">{{ $invoice->company->getFullAddressAttribute() }}</p>
                    </div>
                    
                    <hr>
                    
                    <!-- الإجماليات -->
                    <div class="row">
                        <div class="col-6">
                            <label class="form-label">المجموع الفرعي:</label>
                        </div>
                        <div class="col-6 text-end">
                            <span id="subtotal-display">{{ number_format($invoice->subtotal, 2) }} ر.س</span>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-6">
                            <label class="form-label">الخصم:</label>
                        </div>
                        <div class="col-6">
                            <div class="input-group input-group-sm">
                                <input type="number" name="discount_amount" class="form-control text-end" 
                                       value="{{ old('discount_amount', $invoice->discount_amount) }}" step="0.01" min="0" 
                                       onchange="calculateTotals()">
                                <span class="input-group-text">ر.س</span>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row mt-2">
                        <div class="col-6">
                            <label class="form-label">إجمالي الضريبة:</label>
                        </div>
                        <div class="col-6 text-end">
                            <span id="tax-total-display">{{ number_format($invoice->tax_amount, 2) }} ر.س</span>
                        </div>
                    </div>
                    
                    <hr>
                    
                    <div class="row">
                        <div class="col-6">
                            <strong>المبلغ الإجمالي:</strong>
                        </div>
                        <div class="col-6 text-end">
                            <strong id="total-display" class="text-primary">{{ number_format($invoice->total_amount, 2) }} ر.س</strong>
                        </div>
                    </div>
                    
                    <div class="mt-4">
                        <button type="submit" class="btn btn-success w-100">
                            <i class="fas fa-save me-2"></i>
                            حفظ التعديلات
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</form>

<!-- نموذج اختيار المنتج -->
<div class="modal fade" id="productModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">اختيار منتج أو خدمة</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="mb-3">
                    <input type="text" class="form-control" id="product-search" placeholder="البحث عن منتج...">
                </div>
                <div class="table-responsive">
                    <table class="table table-hover">
                            <tr>
                                <th>الاسم</th>
                                <th>الرمز</th>
                                <th>النوع</th>
                                <th>السعر</th>
                                <th>المخزون</th>
                                <th>اختيار</th>
                            </tr>
                        <tbody id="products-list">
                            @foreach($products as $product)
                            <tr data-product='@json($product)'>
                                <td>
                                    <strong>{{ $product->name }}</strong>
                                    @if($product->description)
                                        <br><small class="text-muted">{{ Str::limit($product->description, 50) }}</small>
                                    @endif
                                </td>
                                <td>{{ $product->sku }}</td>
                                <td>
                                    @if($product->type == 'product')
                                        <span class="badge bg-primary">منتج</span>
                                    @else
                                        <span class="badge bg-info">خدمة</span>
                                    @endif
                                </td>
                                <td>{{ number_format($product->price, 2) }} ر.س</td>
                                <td>
                                    @if($product->type == 'product')
                                        {{ $product->stock_quantity }} {{ $product->unit }}
                                    @else
                                        غير محدود
                                    @endif
                                </td>
                                <td>
                                    <button type="button" class="btn btn-primary btn-sm" 
                                            onclick="selectProduct(this)">
                                        اختيار
                                    </button>
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
let itemIndex = {{ $invoice->items->count() }};

function addInvoiceItem() {
    $('#productModal').modal('show');
}

function selectProduct(button) {
    const row = button.closest('tr');
    const product = JSON.parse(row.getAttribute('data-product'));
    
    const itemHtml = `
        <tr id="item-${itemIndex}">
            <td>
                <input type="hidden" name="items[${itemIndex}][product_id]" value="${product.id}">
                <strong>${product.name}</strong>
                ${product.description ? `<br><small class="text-muted">${product.description}</small>` : ''}
                <br><small class="text-info">رمز: ${product.sku}</small>
            </td>
            <td>
                <input type="number" name="items[${itemIndex}][quantity]" class="form-control" 
                       value="1" step="0.001" min="0.001" required onchange="calculateItemTotal(${itemIndex})">
            </td>
            <td>
                <input type="number" name="items[${itemIndex}][unit_price]" class="form-control" 
                       value="${product.price}" step="0.01" min="0" required onchange="calculateItemTotal(${itemIndex})">
            </td>
            <td>
                <input type="number" name="items[${itemIndex}][discount_amount]" class="form-control" 
                       value="0" step="0.01" min="0" onchange="calculateItemTotal(${itemIndex})">
            </td>
            <td>
                <span id="line-total-${itemIndex}">${product.price}</span> ر.س
            </td>
            <td>
                <span id="tax-amount-${itemIndex}">${(product.price * product.tax_rate / 100).toFixed(2)}</span> ر.س
            </td>
            <td>
                <strong><span id="total-with-tax-${itemIndex}">${(product.price * (1 + product.tax_rate / 100)).toFixed(2)}</span> ر.س</strong>
            </td>
            <td>
                <button type="button" class="btn btn-danger btn-sm" onclick="removeItem(${itemIndex})">
                    <i class="fas fa-trash"></i>
                </button>
            </td>
        </tr>
    `;
    
    document.getElementById('invoice-items').insertAdjacentHTML('beforeend', itemHtml);
    itemIndex++;
    calculateTotals();
    $('#productModal').modal('hide');
}

function removeItem(index) {
    document.getElementById(`item-${index}`).remove();
    calculateTotals();
}

function calculateItemTotal(index) {
    const quantityInput = document.querySelector(`input[name="items[${index}][quantity]"]`);
    const priceInput = document.querySelector(`input[name="items[${index}][unit_price]"]`);
    const discountInput = document.querySelector(`input[name="items[${index}][discount_amount]"]`);
    
    const quantity = parseFloat(quantityInput.value) || 0;
    const price = parseFloat(priceInput.value) || 0;
    const discount = parseFloat(discountInput.value) || 0;
    
    const lineTotal = (quantity * price) - discount;
    const taxAmount = lineTotal * 0.15; // 15% ضريبة
    const totalWithTax = lineTotal + taxAmount;
    
    document.getElementById(`line-total-${index}`).textContent = lineTotal.toFixed(2);
    document.getElementById(`tax-amount-${index}`).textContent = taxAmount.toFixed(2);
    document.getElementById(`total-with-tax-${index}`).textContent = totalWithTax.toFixed(2);
    
    calculateTotals();
}

function calculateTotals() {
    let subtotal = 0;
    let totalTax = 0;
    
    // حساب المجاميع من كل الصفوف
    document.querySelectorAll('#invoice-items tr').forEach(row => {
        const lineTotalElement = row.querySelector('[id^="line-total-"]');
        const taxAmountElement = row.querySelector('[id^="tax-amount-"]');
        
        if (lineTotalElement && taxAmountElement) {
            subtotal += parseFloat(lineTotalElement.textContent) || 0;
            totalTax += parseFloat(taxAmountElement.textContent) || 0;
        }
    });
    
    const discount = parseFloat(document.querySelector('input[name="discount_amount"]').value) || 0;
    const finalTotal = subtotal + totalTax - discount;
    
    document.getElementById('subtotal-display').textContent = subtotal.toFixed(2) + ' ر.س';
    document.getElementById('tax-total-display').textContent = totalTax.toFixed(2) + ' ر.س';
    document.getElementById('total-display').textContent = finalTotal.toFixed(2) + ' ر.س';
}

// البحث في المنتجات
document.getElementById('product-search').addEventListener('input', function() {
    const searchTerm = this.value.toLowerCase();
    const rows = document.querySelectorAll('#products-list tr');
    
    rows.forEach(row => {
        const text = row.textContent.toLowerCase();
        if (text.includes(searchTerm)) {
            row.style.display = '';
        } else {
            row.style.display = 'none';
        }
    });
});

// حساب الإجماليات عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', function() {
    // حساب الإجماليات للعناصر الموجودة
    @foreach($invoice->items as $index => $item)
        calculateItemTotal({{ $index }});
    @endforeach
    
    calculateTotals();
});

// التحقق من وجود عناصر قبل الإرسال
document.getElementById('invoice-form').addEventListener('submit', function(e) {
    const items = document.querySelectorAll('#invoice-items tr');
    if (items.length === 0) {
        e.preventDefault();
        showValidationModal('عنصر مطلوب', 'يجب وجود عنصر واحد على الأقل في الفاتورة', 'warning');
    }
});

// دالة عرض رسائل التحقق في modal
function showValidationModal(title, message, type = 'warning') {
    const iconClass = type === 'danger' ? 'fas fa-exclamation-triangle' :
                     type === 'warning' ? 'fas fa-exclamation-circle' :
                     'fas fa-info-circle';
    
    const modalHtml = `
        <div class="modal fade" id="validationModal" tabindex="-1" aria-hidden="true">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header bg-${type} text-white">
                        <h5 class="modal-title">
                            <i class="${iconClass}"></i>
                            ${title}
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="alert alert-${type} border-0">
                            <i class="${iconClass} me-2"></i>
                            ${message}
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                            <i class="fas fa-check"></i>
                            فهمت
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    // إزالة modal سابق إن وجد
    const existingModal = document.getElementById('validationModal');
    if (existingModal) {
        existingModal.remove();
    }
    
    // إضافة modal جديد
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    
    // عرض modal
    const modal = new bootstrap.Modal(document.getElementById('validationModal'));
    modal.show();
    
    // إزالة modal بعد إغلاقه
    document.getElementById('validationModal').addEventListener('hidden.bs.modal', function() {
        this.remove();
    });
}
</script>
@endpush