@extends('layouts.app')

@section('title', 'إدارة الديون')
@section('page-title', 'إدارة الديون والمتأخرات')

@push('styles')
<style>
/* Modern Page Header Styles */
.modern-page-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    padding: 2rem 0;
    margin-bottom: 2rem;
    position: relative;
    overflow: hidden;
}

.modern-page-header::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(255, 255, 255, 0.1);
    backdrop-filter: blur(10px);
}

.header-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 0 1.5rem;
    position: relative;
    z-index: 2;
}

.header-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 1.5rem;
}

.header-info {
    display: flex;
    align-items: center;
    gap: 1.5rem;
}

.header-icon {
    width: 60px;
    height: 60px;
    background: rgba(255, 255, 255, 0.2);
    border-radius: 16px;
    display: flex;
    align-items: center;
    justify-content: center;
    backdrop-filter: blur(10px);
    border: 1px solid rgba(255, 255, 255, 0.3);
}

.header-icon i {
    font-size: 1.75rem;
    color: white;
}

.header-text h1 {
    font-size: 2rem;
    font-weight: 700;
    color: white;
    margin: 0 0 0.5rem 0;
    text-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.header-text p {
    font-size: 1rem;
    color: rgba(255, 255, 255, 0.9);
    margin: 0;
}

.header-actions {
    display: flex;
    gap: 1rem;
    flex-wrap: wrap;
}

.modern-btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    border-radius: 12px;
    font-weight: 600;
    text-decoration: none;
    transition: all 0.3s ease;
    border: none;
    cursor: pointer;
    font-size: 0.875rem;
    backdrop-filter: blur(10px);
    position: relative;
    overflow: hidden;
}

.modern-btn::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.2), transparent);
    transition: left 0.6s ease;
}

.modern-btn:hover::before {
    left: 100%;
}

.modern-btn.btn-secondary {
    background: rgba(255, 255, 255, 0.2);
    color: white;
    border: 1px solid rgba(255, 255, 255, 0.3);
}

.modern-btn.btn-secondary:hover {
    background: rgba(255, 255, 255, 0.3);
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
}

.modern-btn.btn-warning {
    background: rgba(251, 191, 36, 0.9);
    color: white;
    border: 1px solid rgba(251, 191, 36, 0.3);
}

.modern-btn.btn-warning:hover {
    background: rgba(251, 191, 36, 1);
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(251, 191, 36, 0.3);
}

.modern-btn.btn-success {
    background: rgba(34, 197, 94, 0.9);
    color: white;
    border: 1px solid rgba(34, 197, 94, 0.3);
}

.modern-btn.btn-success:hover {
    background: rgba(34, 197, 94, 1);
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(34, 197, 94, 0.3);
}

/* Enhanced Stats Grid */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.stat-card {
    background: white;
    border-radius: 20px;
    padding: 2rem;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.08);
    border: 1px solid rgba(0, 0, 0, 0.05);
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 4px;
    background: linear-gradient(90deg, var(--accent-color), var(--primary-light));
}

.stat-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 20px 40px rgba(0, 0, 0, 0.12);
}

.stat-card.total::before {
    background: linear-gradient(90deg, #ef4444, #f97316);
}

.stat-card.low-risk::before {
    background: linear-gradient(90deg, #f59e0b, #eab308);
}

.stat-card.medium-risk::before {
    background: linear-gradient(90deg, #f97316, #ef4444);
}

.stat-card.high-risk::before {
    background: linear-gradient(90deg, #dc2626, #991b1b);
}

.stat-content {
    display: flex;
    align-items: center;
    gap: 1.5rem;
}

.stat-icon {
    width: 60px;
    height: 60px;
    border-radius: 16px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    flex-shrink: 0;
}

.stat-card.total .stat-icon {
    background: linear-gradient(135deg, #ef4444, #f97316);
    color: white;
}

.stat-card.low-risk .stat-icon {
    background: linear-gradient(135deg, #f59e0b, #eab308);
    color: white;
}

.stat-card.medium-risk .stat-icon {
    background: linear-gradient(135deg, #f97316, #ef4444);
    color: white;
}

.stat-card.high-risk .stat-icon {
    background: linear-gradient(135deg, #dc2626, #991b1b);
    color: white;
}

.stat-info {
    flex: 1;
}

.stat-number {
    font-size: 2rem;
    font-weight: 700;
    color: #1f2937;
    line-height: 1;
    margin-bottom: 0.25rem;
}

.stat-currency {
    font-size: 0.875rem;
    color: #6b7280;
    margin-bottom: 0.5rem;
}

.stat-label {
    font-size: 1rem;
    color: #374151;
    font-weight: 600;
    margin-bottom: 0.25rem;
}

.stat-count {
    font-size: 0.875rem;
    color: #6b7280;
}

/* Enhanced Filter Card */
.filter-card {
    background: white;
    border-radius: 20px;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.08);
    margin-bottom: 2rem;
    overflow: hidden;
    border: 1px solid rgba(0, 0, 0, 0.05);
}

.filter-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 1.5rem 2rem;
    background: linear-gradient(135deg, #f8fafc, #e2e8f0);
    border-bottom: 1px solid rgba(0, 0, 0, 0.05);
}

.filter-title {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    font-size: 1.125rem;
    font-weight: 600;
    color: #1f2937;
}

.filter-title i {
    color: #667eea;
}

.filter-toggle {
    background: none;
    border: none;
    color: #667eea;
    font-size: 1.25rem;
    cursor: pointer;
    transition: transform 0.3s ease;
}

.filter-content {
    padding: 2rem;
    transition: all 0.3s ease;
}

.filter-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.form-group {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.form-label {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-weight: 600;
    color: #374151;
    font-size: 0.875rem;
}

.form-label i {
    color: #667eea;
    width: 16px;
}

.form-input, .form-select {
    padding: 0.75rem 1rem;
    border: 2px solid #e5e7eb;
    border-radius: 12px;
    font-size: 0.875rem;
    transition: all 0.3s ease;
    background: white;
}

.form-input:focus, .form-select:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
}

.filter-actions {
    display: flex;
    gap: 1rem;
    justify-content: flex-end;
    grid-column: 1 / -1;
}

.btn-clean {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    border-radius: 12px;
    font-weight: 600;
    text-decoration: none;
    transition: all 0.3s ease;
    border: 2px solid;
    cursor: pointer;
    font-size: 0.875rem;
}

.btn-clean.btn-primary {
    background: #667eea;
    color: white;
    border-color: #667eea;
}

.btn-clean.btn-primary:hover {
    background: #5a67d8;
    border-color: #5a67d8;
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
}

.btn-clean.btn-secondary {
    background: transparent;
    color: #6b7280;
    border-color: #d1d5db;
}

.btn-clean.btn-secondary:hover {
    background: #f3f4f6;
    border-color: #9ca3af;
}

/* Enhanced Table Card */
.table-card {
    background: white;
    border-radius: 20px;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.08);
    overflow: hidden;
    border: 1px solid rgba(0, 0, 0, 0.05);
}

.table-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 2rem;
    background: linear-gradient(135deg, #f8fafc, #e2e8f0);
    border-bottom: 1px solid rgba(0, 0, 0, 0.05);
}

.table-title {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    font-size: 1.25rem;
    font-weight: 700;
    color: #1f2937;
}

.table-title i {
    color: #667eea;
}

.table-info {
    display: flex;
    align-items: center;
    gap: 1.5rem;
}

.item-counter {
    font-size: 0.875rem;
    color: #6b7280;
    font-weight: 600;
}

.checkbox-wrapper {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-size: 0.875rem;
    color: #374151;
    cursor: pointer;
}

.table-responsive {
    overflow-x: auto;
}

.clean-table {
    width: 100%;
    border-collapse: collapse;
}

.clean-table th {
    padding: 1rem 1.5rem;
    text-align: right;
    font-weight: 600;
    color: #374151;
    background: #f8fafc;
    border-bottom: 1px solid #e5e7eb;
    font-size: 0.875rem;
}

.clean-table td {
    padding: 1rem 1.5rem;
    border-bottom: 1px solid #f3f4f6;
    font-size: 0.875rem;
    color: #374151;
}

.clean-table tr:hover {
    background: #f8fafc;
}

/* Responsive Design */
@media (max-width: 768px) {
    .header-content {
        flex-direction: column;
        text-align: center;
    }
    
    .header-actions {
        justify-content: center;
    }
    
    .modern-btn {
        padding: 0.625rem 1.25rem;
        font-size: 0.8rem;
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
    }
    
    .filter-grid {
        grid-template-columns: 1fr;
    }
    
    .table-header {
        flex-direction: column;
        gap: 1rem;
        text-align: center;
    }
}

/* Animation Classes */
.pulse-animation {
    animation: pulse 2s infinite;
}

@keyframes pulse {
    0% { transform: scale(1); }
    50% { transform: scale(1.05); }
    100% { transform: scale(1); }
}

.fade-in {
    animation: fadeIn 0.6s ease-in-out;
}

@keyframes fadeIn {
    from { opacity: 0; transform: translateY(20px); }
    to { opacity: 1; transform: translateY(0); }
}
</style>
@endpush

@section('content')

<!-- Modern Page Header -->
<div class="modern-page-header">
    <div class="header-container">
        <div class="header-content">
            <div class="header-info">
                <div class="header-icon">
                    <i class="fas fa-exclamation-triangle"></i>
                </div>
                <div class="header-text">
                    <h1>إدارة الديون والمتأخرات</h1>
                    <p>متابعة وإدارة الفواتير المتأخرة وإرسال التذكيرات</p>
                </div>
            </div>
            <div class="header-actions">
                <a href="{{ route('invoices.index') }}" class="modern-btn btn-secondary">
                    <i class="fas fa-arrow-left"></i>
                    <span>العودة للفواتير</span>
                </a>
                
                <button type="button" class="modern-btn btn-warning" onclick="sendBulkReminders()">
                    <i class="fas fa-paper-plane"></i>
                    <span>إرسال تذكيرات جماعية</span>
                </button>
                
                <a href="{{ route('invoices.export-debt-report') }}" class="modern-btn btn-success">
                    <i class="fas fa-file-excel"></i>
                    <span>تصدير تقرير الديون</span>
                </a>
            </div>
        </div>
    </div>
</div>

<!-- إحصائيات الديون -->
<div class="stats-grid">
    <div class="stat-card total">
        <div class="stat-content">
            <div class="stat-icon">
                <i class="fas fa-exclamation-triangle"></i>
            </div>
            <div class="stat-info">
                <div class="stat-number">{{ number_format($stats['total_debt'], 2) }}</div>
                <div class="stat-currency">ر.س</div>
                <div class="stat-label">إجمالي الديون</div>
                <div class="stat-count">{{ $stats['total_invoices'] }} فاتورة</div>
            </div>
        </div>
    </div>
    
    <div class="stat-card low-risk">
        <div class="stat-content">
            <div class="stat-icon">
                <i class="fas fa-clock"></i>
            </div>
            <div class="stat-info">
                <div class="stat-number">{{ number_format($stats['debt_0_30'], 2) }}</div>
                <div class="stat-currency">ر.س</div>
                <div class="stat-label">متأخر أقل من 30 يوم</div>
                <div class="stat-count">{{ $stats['count_0_30'] }} فاتورة</div>
            </div>
        </div>
    </div>
    
    <div class="stat-card medium-risk">
        <div class="stat-content">
            <div class="stat-icon">
                <i class="fas fa-calendar-times"></i>
            </div>
            <div class="stat-info">
                <div class="stat-number">{{ number_format($stats['debt_30_90'], 2) }}</div>
                <div class="stat-currency">ر.س</div>
                <div class="stat-label">متأخر 30-90 يوم</div>
                <div class="stat-count">{{ $stats['count_30_90'] }} فاتورة</div>
            </div>
        </div>
    </div>
    
    <div class="stat-card high-risk">
        <div class="stat-content">
            <div class="stat-icon">
                <i class="fas fa-ban"></i>
            </div>
            <div class="stat-info">
                <div class="stat-number">{{ number_format($stats['debt_90_plus'], 2) }}</div>
                <div class="stat-currency">ر.س</div>
                <div class="stat-label">متأخر أكثر من 90 يوم</div>
                <div class="stat-count">{{ $stats['count_90_plus'] }} فاتورة</div>
            </div>
        </div>
    </div>
</div>

<!-- فلاتر البحث -->
<div class="filter-card">
    <div class="filter-header">
        <div class="filter-title">
            <i class="fas fa-filter"></i>
            <span>فلاتر البحث</span>
        </div>
        <button type="button" class="filter-toggle" onclick="toggleFilters()">
            <i class="fas fa-chevron-down"></i>
        </button>
    </div>
    <div class="filter-content" id="filterContent">
        <form method="GET" action="{{ route('invoices.debt-management') }}">
            <div class="filter-grid">
                <div class="form-group">
                    <label class="form-label">
                        <i class="fas fa-hourglass-half"></i>
                        عمر الدين
                    </label>
                    <select name="debt_age" class="form-select">
                        <option value="">جميع الأعمار</option>
                        <option value="0-30" {{ request('debt_age') === '0-30' ? 'selected' : '' }}>0-30 يوم</option>
                        <option value="31-60" {{ request('debt_age') === '31-60' ? 'selected' : '' }}>31-60 يوم</option>
                        <option value="61-90" {{ request('debt_age') === '61-90' ? 'selected' : '' }}>61-90 يوم</option>
                        <option value="90+" {{ request('debt_age') === '90+' ? 'selected' : '' }}>أكثر من 90 يوم</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label">
                        <i class="fas fa-coins"></i>
                        المبلغ من
                    </label>
                    <input type="number" name="amount_from" class="form-input"
                           value="{{ request('amount_from') }}" placeholder="0">
                </div>
                
                <div class="form-group">
                    <label class="form-label">
                        <i class="fas fa-coins"></i>
                        المبلغ إلى
                    </label>
                    <input type="number" name="amount_to" class="form-input"
                           value="{{ request('amount_to') }}" placeholder="بدون حد">
                </div>
                
                <div class="form-group">
                    <label class="form-label">
                        <i class="fas fa-user"></i>
                        العميل
                    </label>
                    <input type="text" name="customer" class="form-input"
                           value="{{ request('customer') }}" placeholder="اسم العميل">
                </div>
                
                <div class="form-group">
                    <label class="form-label">
                        <i class="fas fa-sort"></i>
                        ترتيب حسب
                    </label>
                    <select name="sort_by" class="form-select">
                        <option value="overdue_days" {{ request('sort_by') === 'overdue_days' ? 'selected' : '' }}>أيام التأخير</option>
                        <option value="amount" {{ request('sort_by') === 'amount' ? 'selected' : '' }}>المبلغ</option>
                        <option value="customer" {{ request('sort_by') === 'customer' ? 'selected' : '' }}>اسم العميل</option>
                        <option value="due_date" {{ request('sort_by') === 'due_date' ? 'selected' : '' }}>تاريخ الاستحقاق</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label">
                        <i class="fas fa-arrows-alt-v"></i>
                        اتجاه الترتيب
                    </label>
                    <select name="sort_direction" class="form-select">
                        <option value="desc" {{ request('sort_direction') === 'desc' ? 'selected' : '' }}>تنازلي</option>
                        <option value="asc" {{ request('sort_direction') === 'asc' ? 'selected' : '' }}>تصاعدي</option>
                    </select>
                </div>
                
                <div class="filter-actions">
                    <button type="submit" class="btn-clean btn-primary">
                        <i class="fas fa-search"></i>
                        <span>بحث</span>
                    </button>
                    <a href="{{ route('invoices.debt-management') }}" class="btn-clean btn-secondary">
                        <i class="fas fa-times"></i>
                        <span>إلغاء الفلاتر</span>
                    </a>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- قائمة الديون -->
<div class="table-card">
    <div class="table-header">
        <div class="table-title">
            <i class="fas fa-list-alt"></i>
            <span>قائمة الديون المتأخرة</span>
        </div>
        <div class="table-info">
            <span class="item-counter">{{ $invoices->total() }} فاتورة</span>
            <label class="checkbox-wrapper">
                <input type="checkbox" id="selectAll" onchange="toggleSelectAll()">
                <span>تحديد الكل</span>
            </label>
        </div>
    </div>
    
    @if($invoices->count() > 0)
        <div class="table-responsive">
            <table class="clean-table">
                <thead>
                    <tr>
                        <th><i class="fas fa-check-square"></i></th>
                        <th class="sortable">رقم الفاتورة <i class="fas fa-sort"></i></th>
                        <th class="sortable">العميل <i class="fas fa-sort"></i></th>
                        <th class="sortable">تاريخ الإصدار <i class="fas fa-sort"></i></th>
                        <th class="sortable">تاريخ الاستحقاق <i class="fas fa-sort"></i></th>
                        <th class="sortable">أيام التأخير <i class="fas fa-sort"></i></th>
                        <th class="sortable">المبلغ المتبقي <i class="fas fa-sort"></i></th>
                        <th>حالة المخاطر</th>
                        <th>آخر تذكير</th>
                        <th>الإجراءات</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($invoices as $invoice)
                        @php
                            $overdueDays = now()->diffInDays($invoice->due_date);
                            $riskLevel = $overdueDays <= 30 ? 'low' : ($overdueDays <= 90 ? 'medium' : 'high');
                            $riskTexts = ['low' => 'منخفض', 'medium' => 'متوسط', 'high' => 'عالي'];
                        @endphp
                        <tr>
                            <td>
                                <input type="checkbox" class="invoice-checkbox" value="{{ $invoice->id }}">
                            </td>
                            <td>
                                <a href="{{ route('invoices.show', $invoice) }}" style="text-decoration: none; color: var(--primary);">
                                    {{ $invoice->invoice_number }}
                                </a>
                                @if($invoice->type == 'standard')
                                    <span style="background: rgba(16, 185, 129, 0.1); color: #059669; padding: 2px 6px; border-radius: 4px; font-size: 0.75rem;">معيارية</span>
                                @else
                                    <span style="background: rgba(59, 130, 246, 0.1); color: var(--primary); padding: 2px 6px; border-radius: 4px; font-size: 0.75rem;">مبسطة</span>
                                @endif
                            </td>
                            <td>
                                <div>
                                    <div style="font-weight: 600;">{{ $invoice->customer->name ?? 'غير محدد' }}</div>
                                    @if($invoice->customer && $invoice->customer->email)
                                        <div style="font-size: 0.75rem; color: var(--text-light);">
                                            <i class="fas fa-envelope"></i> {{ $invoice->customer->email }}
                                        </div>
                                    @endif
                                </div>
                            </td>
                            <td>{{ $invoice->issue_date->format('Y-m-d') }}</td>
                            <td>{{ $invoice->due_date->format('Y-m-d') }}</td>
                            <td>
                                <span style="background: rgba({{ $riskLevel === 'low' ? '245, 158, 11' : ($riskLevel === 'medium' ? '249, 115, 22' : '239, 68, 68') }}, 0.1); color: {{ $riskLevel === 'low' ? '#d97706' : ($riskLevel === 'medium' ? '#ea580c' : '#dc2626') }}; padding: 4px 8px; border-radius: 4px; font-size: 0.75rem; font-weight: 600;">
                                    {{ $overdueDays }} يوم
                                </span>
                            </td>
                            <td style="text-align: center;">
                                <div style="font-size: 1rem; font-weight: 700;">{{ number_format($invoice->remaining_amount, 2) }}</div>
                                <div style="font-size: 0.75rem; color: var(--text-light);">ر.س</div>
                            </td>
                            <td>
                                <span style="background: rgba({{ $riskLevel === 'low' ? '245, 158, 11' : ($riskLevel === 'medium' ? '249, 115, 22' : '239, 68, 68') }}, 0.1); color: {{ $riskLevel === 'low' ? '#d97706' : ($riskLevel === 'medium' ? '#ea580c' : '#dc2626') }}; padding: 4px 8px; border-radius: 4px; font-size: 0.75rem; font-weight: 500; display: inline-flex; align-items: center; gap: 4px;">
                                    <span style="width: 6px; height: 6px; border-radius: 50%; background: currentColor;"></span>
                                    {{ $riskTexts[$riskLevel] }}
                                </span>
                            </td>
                            <td>
                                @if($invoice->last_reminder_sent_at)
                                    <div style="font-size: 0.75rem; color: var(--text-light);">
                                        <i class="fas fa-paper-plane"></i>
                                        {{ $invoice->last_reminder_sent_at->diffForHumans() }}
                                    </div>
                                @else
                                    <div style="font-size: 0.75rem; color: var(--text-light);">
                                        <i class="fas fa-times-circle"></i>
                                        لم يتم إرسال
                                    </div>
                                @endif
                            </td>
                            <td>
                                <div style="display: flex; gap: 4px; justify-content: center;">
                                    <button type="button" onclick="sendReminder({{ $invoice->id }})"
                                            style="width: 32px; height: 32px; border-radius: var(--radius); border: 1px solid var(--border); background: var(--white); color: var(--text); cursor: pointer; display: flex; align-items: center; justify-content: center; transition: all 0.2s ease;"
                                            title="إرسال تذكير"
                                            onmouseover="this.style.background='var(--warning)'; this.style.color='var(--white)'"
                                            onmouseout="this.style.background='var(--white)'; this.style.color='var(--text)'">
                                        <i class="fas fa-paper-plane"></i>
                                    </button>
                                    
                                    <button type="button" onclick="addPayment({{ $invoice->id }})"
                                            style="width: 32px; height: 32px; border-radius: var(--radius); border: 1px solid var(--border); background: var(--white); color: var(--text); cursor: pointer; display: flex; align-items: center; justify-content: center; transition: all 0.2s ease;"
                                            title="إضافة دفعة"
                                            onmouseover="this.style.background='var(--success)'; this.style.color='var(--white)'"
                                            onmouseout="this.style.background='var(--white)'; this.style.color='var(--text)'">
                                        <i class="fas fa-plus"></i>
                                    </button>
                                    
                                    <a href="{{ route('invoices.show', $invoice) }}"
                                       style="width: 32px; height: 32px; border-radius: var(--radius); border: 1px solid var(--border); background: var(--white); color: var(--text); cursor: pointer; display: flex; align-items: center; justify-content: center; transition: all 0.2s ease; text-decoration: none;"
                                       title="عرض"
                                       onmouseover="this.style.background='var(--primary)'; this.style.color='var(--white)'"
                                       onmouseout="this.style.background='var(--white)'; this.style.color='var(--text)'">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                </div>
                            </td>
                        </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
        
        <div style="display: flex; justify-content: space-between; align-items: center; padding: 1rem 1.5rem; background: var(--light); border-top: 1px solid var(--border);">
            <div style="font-size: 0.875rem; color: var(--text-light);">
                عرض {{ $invoices->firstItem() }} إلى {{ $invoices->lastItem() }} من {{ $invoices->total() }} فاتورة
            </div>
            <div>{{ $invoices->links() }}</div>
        </div>
    @else
        <div style="padding: 4rem 2rem; text-align: center;">
            <i class="fas fa-check-circle" style="font-size: 4rem; color: var(--success); margin-bottom: 1rem;"></i>
            <div style="font-size: 1.5rem; font-weight: 700; color: var(--dark); margin-bottom: 0.5rem;">لا توجد ديون متأخرة</div>
            <div style="font-size: 1rem; color: var(--text-light);">جميع الفواتير مدفوعة أو في الموعد المحدد</div>
            <div style="font-size: 3rem; margin-top: 1rem;">🎉</div>
        </div>
    @endif
</div>

<!-- نافذة إدارة الديون المحسنة والاحترافية -->
<div class="modal fade professional-debt-modal" id="addPaymentModal" tabindex="-1">
    <div class="modal-dialog modal-dialog-centered modal-xl">
        <div class="modal-content professional-modal-content">
            <!-- رأس النافذة الاحترافي -->
            <div class="modal-header professional-header">
                <div class="header-content">
                    <div class="header-main">
                        <div class="modal-icon-wrapper">
                            <div class="modal-main-icon">
                                <i class="fas fa-hand-holding-usd"></i>
                            </div>
                        </div>
                        <div class="header-text">
                            <h4 class="modal-title">إدارة دفعات الديون</h4>
                            <p class="modal-description">تسديد وإدارة المدفوعات المتأخرة</p>
                        </div>
                    </div>
                    <div class="header-actions">
                        <div class="status-indicator">
                            <div class="status-badge processing">
                                <i class="fas fa-clock"></i>
                                <span>قيد المعالجة</span>
                            </div>
                        </div>
                        <button type="button" class="professional-close-btn" data-bs-dismiss="modal">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                </div>
            </div>

            <!-- محتوى النافذة الرئيسي -->
            <div class="modal-body professional-body">
                <!-- قسم معلومات الفاتورة -->
                <div class="invoice-info-section">
                    <div class="section-header">
                        <h5 class="section-title">
                            <i class="fas fa-file-invoice"></i>
                            معلومات الفاتورة
                        </h5>
                        <div class="section-actions">
                            <!-- أزرار التصدير والإرسال -->
                            <div class="export-actions">
                                <button type="button" class="export-btn pdf-btn" onclick="exportToPDF()" title="تصدير PDF">
                                    <i class="fas fa-file-pdf"></i>
                                    <span>PDF</span>
                                </button>
                                <button type="button" class="export-btn excel-btn" onclick="exportToExcel()" title="تصدير Excel">
                                    <i class="fas fa-file-excel"></i>
                                    <span>Excel</span>
                                </button>
                                <button type="button" class="export-btn whatsapp-btn" onclick="sendWhatsApp()" title="إرسال واتساب">
                                    <i class="fab fa-whatsapp"></i>
                                    <span>واتساب</span>
                                </button>
                                <button type="button" class="export-btn email-btn" onclick="sendEmail()" title="إرسال بريد إلكتروني">
                                    <i class="fas fa-envelope"></i>
                                    <span>بريد</span>
                                </button>
                            </div>
                        </div>
                    </div>

                    <div class="invoice-details-card">
                        <div class="invoice-grid">
                            <div class="invoice-detail">
                                <div class="detail-icon">
                                    <i class="fas fa-hashtag"></i>
                                </div>
                                <div class="detail-content">
                                    <label class="detail-label">رقم الفاتورة</label>
                                    <span class="detail-value" id="modal-invoice-number">-</span>
                                </div>
                            </div>
                            
                            <div class="invoice-detail">
                                <div class="detail-icon">
                                    <i class="fas fa-user"></i>
                                </div>
                                <div class="detail-content">
                                    <label class="detail-label">العميل</label>
                                    <span class="detail-value" id="modal-customer-name">-</span>
                                </div>
                            </div>
                            
                            <div class="invoice-detail">
                                <div class="detail-icon">
                                    <i class="fas fa-money-bill"></i>
                                </div>
                                <div class="detail-content">
                                    <label class="detail-label">إجمالي الفاتورة</label>
                                    <span class="detail-value" id="modal-total-amount">-</span>
                                </div>
                            </div>
                            
                            <div class="invoice-detail">
                                <div class="detail-icon">
                                    <i class="fas fa-exclamation-triangle"></i>
                                </div>
                                <div class="detail-content">
                                    <label class="detail-label">المبلغ المتبقي</label>
                                    <span class="detail-value remaining-debt" id="modal-remaining-amount">-</span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- شريط التقدم -->
                        <div class="payment-progress-section">
                            <div class="progress-header">
                                <span class="progress-title">تقدم السداد</span>
                                <span class="progress-percentage" id="modal-payment-percentage">0%</span>
                            </div>
                            <div class="progress-container">
                                <div class="progress-track">
                                    <div class="progress-fill" id="modal-progress-bar" style="width: 0%"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- قسم إدخال بيانات الدفعة -->
                <div class="payment-form-section">
                    <div class="section-header">
                        <h5 class="section-title">
                            <i class="fas fa-credit-card"></i>
                            بيانات الدفعة الجديدة
                        </h5>
                    </div>

                    <form id="addPaymentForm" class="professional-form">
                        @csrf
                        <input type="hidden" id="paymentInvoiceId">
                        
                        <div class="form-grid">
                            <!-- المبلغ -->
                            <div class="form-group">
                                <label class="form-label">
                                    <i class="fas fa-coins"></i>
                                    مبلغ الدفعة
                                    <span class="required">*</span>
                                </label>
                                <div class="input-wrapper amount-wrapper">
                                    <input type="number" step="0.01" class="form-input"
                                           id="paymentAmount" placeholder="أدخل المبلغ" required min="0.01">
                                    <div class="input-addon">ر.س</div>
                                    <div class="input-validation" id="amountValidation"></div>
                                </div>
                                <small class="input-help">الحد الأقصى: <span id="maxAmount">0.00</span> ر.س</small>
                            </div>

                            <!-- تاريخ الدفعة -->
                            <div class="form-group">
                                <label class="form-label">
                                    <i class="fas fa-calendar"></i>
                                    تاريخ الدفعة
                                </label>
                                <div class="input-wrapper">
                                    <input type="date" class="form-input"
                                           id="paymentDate" value="{{ date('Y-m-d') }}">
                                    <i class="input-icon fas fa-calendar"></i>
                                </div>
                            </div>

                            <!-- طريقة الدفع -->
                            <div class="form-group">
                                <label class="form-label">
                                    <i class="fas fa-credit-card"></i>
                                    طريقة الدفع
                                    <span class="required">*</span>
                                </label>
                                <div class="select-wrapper">
                                    <select class="form-select" id="paymentMethod" required>
                                        <option value="">اختر طريقة الدفع</option>
                                        <option value="cash">💵 نقداً</option>
                                        <option value="bank_transfer">🏦 تحويل بنكي</option>
                                        <option value="check">📝 شيك</option>
                                        <option value="credit_card">💳 بطاقة ائتمان</option>
                                        <option value="installment">📅 تقسيط</option>
                                    </select>
                                    <i class="select-arrow fas fa-chevron-down"></i>
                                </div>
                            </div>

                            <!-- رقم المرجع -->
                            <div class="form-group">
                                <label class="form-label">
                                    <i class="fas fa-barcode"></i>
                                    رقم المرجع
                                </label>
                                <div class="input-wrapper">
                                    <input type="text" class="form-input"
                                           id="paymentReference" placeholder="رقم الشيك أو المرجع">
                                    <i class="input-icon fas fa-hashtag"></i>
                                </div>
                            </div>
                        </div>

                        <!-- ملاحظات -->
                        <div class="form-group full-width">
                            <label class="form-label">
                                <i class="fas fa-comment"></i>
                                ملاحظات
                            </label>
                            <textarea class="form-textarea" id="paymentNotes" rows="3"
                                      placeholder="أضف أي ملاحظات حول عملية الدفع..."></textarea>
                        </div>

                        <!-- معاينة الدفعة -->
                        <div class="payment-preview-card" id="paymentPreview" style="display: none;">
                            <div class="preview-header">
                                <h6>
                                    <i class="fas fa-eye"></i>
                                    معاينة الدفعة
                                </h6>
                                <button type="button" class="preview-close" onclick="hidePaymentPreview()">
                                    <i class="fas fa-times"></i>
                                </button>
                            </div>
                            <div class="preview-grid">
                                <div class="preview-item">
                                    <span class="preview-label">المبلغ المدفوع:</span>
                                    <span class="preview-value" id="previewAmount">0.00 ر.س</span>
                                </div>
                                <div class="preview-item">
                                    <span class="preview-label">نسبة السداد:</span>
                                    <span class="preview-value" id="previewPercentage">0%</span>
                                </div>
                                <div class="preview-item">
                                    <span class="preview-label">المتبقي بعد الدفع:</span>
                                    <span class="preview-value" id="previewRemaining">0.00 ر.س</span>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>

            <!-- تذييل النافذة -->
            <div class="modal-footer professional-footer">
                <div class="footer-left">
                    <div class="security-info">
                        <i class="fas fa-shield-alt"></i>
                        <span>معاملة آمنة ومشفرة</span>
                    </div>
                </div>
                
                <div class="footer-right">
                    <div class="action-buttons">
                        <button type="button" class="btn btn-outline" data-bs-dismiss="modal">
                            <i class="fas fa-times"></i>
                            <span>إلغاء</span>
                        </button>
                        
                        <button type="button" class="btn btn-info" onclick="togglePaymentPreview()">
                            <i class="fas fa-eye"></i>
                            <span>معاينة</span>
                        </button>
                        
                        <button type="submit" form="addPaymentForm" class="btn btn-primary">
                            <i class="fas fa-check"></i>
                            <span>تأكيد الدفعة</span>
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- نافذة تأكيد حفظ الدفعة -->
<div class="modal fade success-modal" id="paymentSuccessModal" tabindex="-1">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content success-modal-content">
            <div class="modal-body success-modal-body">
                <div class="success-animation">
                    <div class="success-checkmark">
                        <div class="check-icon">
                            <span class="icon-line line-tip"></span>
                            <span class="icon-line line-long"></span>
                            <div class="icon-circle"></div>
                            <div class="icon-fix"></div>
                        </div>
                    </div>
                </div>
                <div class="success-content">
                    <h4 class="success-title">تم حفظ الدفعة بنجاح!</h4>
                    <p class="success-message">تم تحديث حالة الفاتورة وسيتم إعادة تحميل الصفحة</p>
                    <div class="success-details">
                        <div class="detail-row">
                            <span>المبلغ المدفوع:</span>
                            <span id="successAmount">0.00 ر.س</span>
                        </div>
                        <div class="detail-row">
                            <span>طريقة الدفع:</span>
                            <span id="successMethod">-</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

@endsection

@push('scripts')
<script>
// Enhanced JavaScript functionality for debt management
document.addEventListener('DOMContentLoaded', function() {
    initializeDebtManagement();
    initializeAnimations();
    initializeTableSorting();
    setupKeyboardShortcuts();
});

// Initialize debt management functionality
function initializeDebtManagement() {
    // Enhanced checkbox selection
    setupCheckboxInteractions();
    // Initialize risk indicators
    setupRiskIndicators();
    // Setup table interactions
    setupTableInteractions();
}

// Enhanced checkbox interactions
function setupCheckboxInteractions() {
    const selectAllCheckbox = document.getElementById('selectAll');
    const invoiceCheckboxes = document.querySelectorAll('.invoice-checkbox');
    
    // Update bulk actions visibility based on selection
    function updateBulkActions() {
        const selectedCount = document.querySelectorAll('.invoice-checkbox:checked').length;
        const bulkButton = document.querySelector('.modern-btn.btn-warning');
        
        if (selectedCount > 0) {
            bulkButton.classList.add('pulse-animation');
            bulkButton.innerHTML = `<i class="fas fa-paper-plane"></i><span>إرسال ${selectedCount} تذكيرات</span>`;
        } else {
            bulkButton.classList.remove('pulse-animation');
            bulkButton.innerHTML = `<i class="fas fa-paper-plane"></i><span>إرسال تذكيرات جماعية</span>`;
        }
    }
    
    // Enhanced select all functionality
    if (selectAllCheckbox) {
        selectAllCheckbox.addEventListener('change', function() {
            invoiceCheckboxes.forEach((checkbox, index) => {
                setTimeout(() => {
                    checkbox.checked = this.checked;
                    const row = checkbox.closest('.table-row');
                    if (this.checked) {
                        row.classList.add('selected');
                    } else {
                        row.classList.remove('selected');
                    }
                }, index * 50);
            });
            updateBulkActions();
        });
    }
    
    // Individual checkbox changes
    invoiceCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', function() {
            const row = this.closest('.table-row');
            if (this.checked) {
                row.classList.add('selected');
            } else {
                row.classList.remove('selected');
            }
            
            // Update select all checkbox state
            const allChecked = Array.from(invoiceCheckboxes).every(cb => cb.checked);
            const noneChecked = Array.from(invoiceCheckboxes).every(cb => !cb.checked);
            
            if (selectAllCheckbox) {
                selectAllCheckbox.checked = allChecked;
                selectAllCheckbox.indeterminate = !allChecked && !noneChecked;
            }
            
            updateBulkActions();
        });
    });
}

// Setup risk indicators with animations
function setupRiskIndicators() {
    document.querySelectorAll('.risk-indicator').forEach(indicator => {
        indicator.addEventListener('mouseenter', function() {
            const riskDot = this.querySelector('.risk-dot');
            if (riskDot) {
                riskDot.style.transform = 'scale(1.3)';
            }
        });
        
        indicator.addEventListener('mouseleave', function() {
            const riskDot = this.querySelector('.risk-dot');
            if (riskDot) {
                riskDot.style.transform = 'scale(1)';
            }
        });
    });
}

// Enhanced table interactions
function setupTableInteractions() {
    document.querySelectorAll('.debt-row').forEach(row => {
        row.addEventListener('mouseenter', function() {
            this.style.transform = 'translateX(-2px)';
            this.style.boxShadow = '2px 0 8px rgba(0,0,0,0.1)';
        });
        
        row.addEventListener('mouseleave', function() {
            if (!this.classList.contains('selected')) {
                this.style.transform = 'translateX(0)';
                this.style.boxShadow = 'none';
            }
        });
    });
}

// Enhanced filter toggle
function toggleDebtFilters() {
    const filterContent = document.getElementById('debtFilterContent');
    const toggleBtn = document.querySelector('.filter-toggle-btn i');
    
    if (filterContent.style.maxHeight === '0px' || !filterContent.style.maxHeight) {
        filterContent.style.maxHeight = filterContent.scrollHeight + 'px';
        filterContent.style.padding = '2rem';
        toggleBtn.style.transform = 'rotate(180deg)';
    } else {
        filterContent.style.maxHeight = '0px';
        filterContent.style.padding = '0 2rem';
        toggleBtn.style.transform = 'rotate(0deg)';
    }
}

// تحديد/إلغاء تحديد الكل
function toggleSelectAll() {
    const selectAll = document.getElementById('selectAll');
    const checkboxes = document.querySelectorAll('.invoice-checkbox');
    
    checkboxes.forEach((checkbox, index) => {
        setTimeout(() => {
            checkbox.checked = selectAll.checked;
            const row = checkbox.closest('.table-row');
            if (selectAll.checked) {
                row.classList.add('selected');
            } else {
                row.classList.remove('selected');
            }
        }, index * 30);
    });
}

// Enhanced reminder sending with loading states
function sendReminder(invoiceId) {
    if (confirm('هل تريد إرسال تذكير بالسداد لهذه الفاتورة؟')) {
        const button = event.target.closest('.reminder-btn');
        const originalContent = button.innerHTML;
        
        // Show loading state
        button.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
        button.disabled = true;
        
        fetch(`/invoices/${invoiceId}/send-reminder`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                button.innerHTML = '<i class="fas fa-check"></i>';
                button.classList.add('success-state');
                setTimeout(() => location.reload(), 1500);
            } else {
                throw new Error(data.message || 'خطأ غير معروف');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            button.innerHTML = '<i class="fas fa-exclamation-triangle"></i>';
            button.classList.add('error-state');
            
            setTimeout(() => {
                button.innerHTML = originalContent;
                button.disabled = false;
                button.classList.remove('error-state');
            }, 3000);
            
            alert('حدث خطأ: ' + error.message);
        });
    }
}

// Enhanced bulk reminders with progress indication
function sendBulkReminders() {
    const selectedInvoices = Array.from(document.querySelectorAll('.invoice-checkbox:checked'))
        .map(checkbox => checkbox.value);
    
    if (selectedInvoices.length === 0) {
        alert('يرجى اختيار فاتورة واحدة على الأقل');
        return;
    }
    
    if (confirm(`هل تريد إرسال تذكيرات لـ ${selectedInvoices.length} فاتورة؟`)) {
        const button = event.target.closest('.modern-btn');
        const originalContent = button.innerHTML;
        
        // Show progress
        button.innerHTML = '<i class="fas fa-spinner fa-spin"></i><span>جاري الإرسال...</span>';
        button.disabled = true;
        
        fetch('/invoices/send-bulk-reminders', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({ invoice_ids: selectedInvoices })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                button.innerHTML = '<i class="fas fa-check"></i><span>تم الإرسال بنجاح!</span>';
                button.classList.add('btn-success');
                setTimeout(() => location.reload(), 2000);
            } else {
                throw new Error(data.message || 'خطأ غير معروف');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            button.innerHTML = originalContent;
            button.disabled = false;
            alert('حدث خطأ: ' + error.message);
        });
    }
}

// Enhanced add payment modal with complete functionality
function addPayment(invoiceId, invoiceNumber = null, remainingAmount = null) {
    document.getElementById('paymentInvoiceId').value = invoiceId;
    
    // Reset form with animation
    const form = document.getElementById('addPaymentForm');
    form.reset();
    
    // Set default date to today
    document.getElementById('paymentDate').value = new Date().toISOString().split('T')[0];
    
    // Update modal information if provided
    if (invoiceNumber) {
        document.getElementById('modal-invoice-number').textContent = invoiceNumber;
    }
    
    if (remainingAmount) {
        const formattedAmount = parseFloat(remainingAmount).toFixed(2);
        document.getElementById('modal-remaining-amount').textContent = formattedAmount + ' ر.س';
        document.getElementById('maxAmount').textContent = formattedAmount;
        document.getElementById('paymentAmount').setAttribute('max', remainingAmount);
    }
    
    // Reset payment preview
    hidePaymentPreview();
    
    // Show modal with entrance animation
    const modal = new bootstrap.Modal(document.getElementById('addPaymentModal'));
    modal.show();
    
    // Add entrance animation
    const modalElement = document.getElementById('addPaymentModal');
    modalElement.addEventListener('shown.bs.modal', function() {
        const modalDialog = this.querySelector('.modal-dialog');
        modalDialog.style.transform = 'scale(0.9)';
        modalDialog.style.opacity = '0';
        
        setTimeout(() => {
            modalDialog.style.transition = 'all 0.3s ease-out';
            modalDialog.style.transform = 'scale(1)';
            modalDialog.style.opacity = '1';
        }, 50);
    });
    
    // Setup real-time amount validation
    setupAmountValidation();
}

// Form submission for payment
document.getElementById('addPaymentForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const invoiceId = document.getElementById('paymentInvoiceId').value;
    const amount = document.getElementById('paymentAmount').value;
    const method = document.getElementById('paymentMethod').value;
    const notes = document.getElementById('paymentNotes').value;
    
    const submitBtn = this.querySelector('button[type="submit"]');
    const originalContent = submitBtn.innerHTML;
    
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري الحفظ...';
    submitBtn.disabled = true;
    
    fetch(`/invoices/${invoiceId}/add-payment`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        },
        body: JSON.stringify({
            amount: amount,
            method: method,
            notes: notes
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            submitBtn.innerHTML = '<i class="fas fa-check"></i> تم الحفظ بنجاح!';
            setTimeout(() => {
                bootstrap.Modal.getInstance(document.getElementById('addPaymentModal')).hide();
                location.reload();
            }, 1500);
        } else {
            throw new Error(data.message || 'خطأ غير معروف');
        }
    })
    .catch(error => {
        submitBtn.innerHTML = originalContent;
        submitBtn.disabled = false;
        alert('حدث خطأ: ' + error.message);
    });
});
</script>
@endpush