@extends('layouts.app')

@section('title', 'إنشاء فاتورة جديدة')
@section('page-title', 'إنشاء فاتورة جديدة')

@section('page-actions')
<a href="{{ route('invoices.index') }}" class="btn btn-secondary btn-success">
    <i class="fas fa-arrow-left me-2"></i>
    العودة للقائمة
</a>
@endsection

@section('content')
<form method="POST" action="{{ route('invoices.store') }}" id="invoice-form">
    @csrf
    
    
    <div class="row">


        <!-- معلومات الفاتورة الأساسية -->
        <div class="col-lg-12">
            <div class="invoice-info-card">
                <div class="card-header-modern">
                    <div class="header-content">
                  
                        <div class="header-text">
                            <h5 class="card-title mb-1">معلومات الفاتورة  <i class="fas fa-file-invoice-dollar"></i></h5>
                        </div>
                    </div>
                    <div class="header-decoration">
                        <div class="decoration-circle"></div>
                        <div class="decoration-circle"></div>
                        <div class="decoration-circle"></div>
                    </div>
                </div>
                
                <div class="card-body-modern">
                    <!-- قسم معلومات الفاتورة الشامل -->
                    <div class="section-wrapper">
                  
                        <!-- صف نوع الفاتورة والعميل وتاريخ الإصدار -->
                        <div class="form-row-modern five-columns">
                            <div class="form-group-enhanced">
                                <div class="input-group mb-3">
                                    <div class="input-group-prepend">
                                        <button class="btn btn-primary" type="button"><i class="fas fa-qrcode"></i></button>
                                    </div>
                                    <select name="type" class="form-control @error('type') is-invalid @enderror" required>
                                        <option value="">اختر نوع الفاتورة</option>
                                        <option value="standard" {{ old('type') == 'standard' ? 'selected' : '' }}>
                                            فاتورة ضريبية معيارية (B2B)
                                        </option>
                                        <option value="simplified" {{ old('simplified') == 'simplified' ? 'selected' : '' }}>
                                            فاتورة ضريبية مبسطة (B2C)
                                        </option>
                                    </select>
                                </div>
                                @error('type')
                                    <div class="invalid-feedback-enhanced">
                                        <i class="fas fa-exclamation-triangle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                            </div>
                            
                            <div class="form-group-enhanced">
                                <div class="input-group mb-3 customer-input-group">
                                    <div class="input-group-prepend">
                                        <button class="btn btn-primary customer-icon-btn" type="button"><i class="fas fa-users"></i></button>
                                    </div>
                                    <select name="customer_id" class="form-control customer-select @error('customer_id') is-invalid @enderror" id="customer-select">
                                        <option value="">اختر العميل (اختياري للفواتير المبسطة)</option>
                                        @foreach($customers as $customer)
                                            <option value="{{ $customer->id }}" {{ old('customer_id') == $customer->id ? 'selected' : '' }}
                                                    data-customer-type="{{ $customer->customer_type }}"
                                                    data-vat-number="{{ $customer->vat_number }}"
                                                    data-phone="{{ $customer->phone }}"
                                                    data-email="{{ $customer->email }}">
                                                {{ $customer->name }}
                                                @if($customer->customer_type == 'business' && $customer->vat_number)
                                                    ({{ $customer->vat_number }})
                                                @endif
                                            </option>
                                        @endforeach
                                    </select>
                                    <div class="input-group-append">
                                        <a href="{{ route('customers.create') }}" target="_blank" class="btn btn-success add-customer-btn" title="إضافة عميل جديد">
                                            <i class="fas fa-plus-circle"></i>
                                        </a>
                                    </div>
                                </div>
                                @error('customer_id')
                                    <div class="invalid-feedback-enhanced">
                                        <i class="fas fa-exclamation-triangle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                            </div>
                            
                            <div class="form-group-enhanced">
                                <div class="input-group mb-3">
                                    <div class="input-group-prepend">
                                        <button class="btn btn-primary" type="button"><i class="fas fa-calendar-plus"></i></button>
                                    </div>
                                    <input type="date" name="issue_date" title="تاريخ الإصدار" class="form-control @error('issue_date') is-invalid @enderror"
                                           value="{{ old('issue_date', date('Y-m-d')) }}" required>
                                </div>
                                @error('issue_date')
                                    <div class="invalid-feedback-enhanced">
                                        <i class="fas fa-exclamation-triangle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                            </div>
                            
                            <div class="form-group-enhanced">
                        
                                <div class="input-group mb-3">
                                    <div class="input-group-prepend">
                                        <button class="btn btn-primary" type="button"><i class="fas fa-calendar-check"></i></button>
                                    </div>
                                    <input type="date" name="due_date" title="تاريخ الإستحقاق" class="form-control @error('due_date') is-invalid @enderror"
                                           value="{{ old('due_date') }}" id="due-date-input">
                                </div>
                                @error('due_date')
                                    <div class="invalid-feedback-enhanced">
                                        <i class="fas fa-exclamation-triangle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                            </div>
                            
                            <div class="form-group-enhanced">
                        
                                <div class="input-group mb-3">
                                    <div class="input-group-prepend">
                                        <button class="btn btn-primary" type="button"><i class="fas fa-truck"></i></button>
                                    </div>
                                    <input type="date" name="supply_date" title="تاريخ التوريد" class="form-control @error('supply_date') is-invalid @enderror"
                                           value="{{ old('supply_date', date('Y-m-d')) }}">
                                </div>
                                @error('supply_date')
                                    <div class="invalid-feedback-enhanced">
                                        <i class="fas fa-exclamation-triangle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                            </div>
                        </div>
          
                        <!-- صف الإشعارات مع نوع الفاتورة -->
                        <div class="form-row-modern three-columns">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    إشعار الدائن
                                </label>
                                <div class="input-group mb-3">
                                    <div class="input-group-prepend">
                                        <button class="btn btn-primary" type="button"><i class="fas fa-bell"></i></button>
                                    </div>
                                    <div class="form-control d-flex align-items-center">
                                        <input class="form-check-input me-2" type="checkbox" name="send_creditor_notification"
                                               id="send_creditor_notification" value="1" {{ old('send_creditor_notification') ? 'checked' : '' }}>
                                        <label class="form-check-label mb-0" for="send_creditor_notification">
                                            إرسال إشعار للدائن عند إنشاء الفاتورة
                                        </label>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    إشعار المدين
                                </label>
                                <div class="input-group mb-3">
                                    <div class="input-group-prepend">
                                        <button class="btn btn-primary" type="button"><i class="fas fa-user-check"></i></button>
                                    </div>
                                    <div class="form-control d-flex align-items-center">
                                        <input class="form-check-input me-2" type="checkbox" name="send_debtor_notification"
                                               id="send_debtor_notification" value="1" {{ old('send_debtor_notification') ? 'checked' : '' }}>
                                        <label class="form-check-label mb-0" for="send_debtor_notification">
                                            إرسال إشعار للعميل (المدين) بالفاتورة
                                        </label>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="form-group-enhanced full-width">
                            <div class="input-group mb-3">
                                <div class="input-group-prepend">
                                    <button class="btn btn-primary" type="button"><i class="fas fa-comment-alt"></i></button>
                                </div>
                                <textarea name="notes" class="form-control" rows="4"
                                          placeholder="أدخل أي ملاحظات إضافية أو تعليمات خاصة بهذه الفاتورة...">{{ old('notes') }}</textarea>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- عناصر الفاتورة -->
            <div class="invoice-items-card">
                <div class="items-card-header">
                    <div class="items-header-content">
          
                        <div class="items-header-text">
                            <h5 class="items-card-title">عناصر الفاتورة                             <i class="fas fa-shopping-cart"></i>
</h5>
                        </div>
                    </div>
                    <div class="items-header-actions" align="left">
                        <button type="button" class="btn-add-item warning" onclick="addInvoiceItem()">
                            <div class="btn-add-icon">
                                <i class="fas fa-plus"></i>
                            </div>
                            <span class="btn-add-text">إضافة عنصر</span>
                        </button>
                    </div>
                </div>
                
                <div class="items-card-body">
                    <div class="items-table-container">
                        <div class="table-wrapper">
                            <table class="invoice-items-table" id="invoice-items-table">
                                <thead class="table-header-modern">
                                    <tr>
                                        <th class="th-product">
                                            <div class="th-content">
                                                <i class="fas fa-box"></i>
                                                <span>المنتج/الخدمة</span>
                                            </div>
                                        </th>
                                        <th class="th-quantity">
                                            <div class="th-content">
                                                <i class="fas fa-sort-numeric-up"></i>
                                                <span>الكمية</span>
                                            </div>
                                        </th>
                                        <th class="th-price">
                                            <div class="th-content">
                                                <i class="fas fa-tag"></i>
                                                <span>سعر الوحدة</span>
                                            </div>
                                        </th>
                                        <th class="th-discount">
                                            <div class="th-content">
                                                <i class="fas fa-percentage"></i>
                                                <span>الخصم</span>
                                            </div>
                                        </th>
                                        <th class="th-subtotal">
                                            <div class="th-content">
                                                <i class="fas fa-calculator"></i>
                                                <span>الإجمالي</span>
                                            </div>
                                        </th>
                                        <th class="th-tax">
                                            <div class="th-content">
                                                <i class="fas fa-receipt"></i>
                                                <span>الضريبة</span>
                                            </div>
                                        </th>
                                        <th class="th-total">
                                            <div class="th-content">
                                                <i class="fas fa-money-bill-wave"></i>
                                                <span>الإجمالي مع الضريبة</span>
                                            </div>
                                        </th>
                                        <th class="th-actions">
                                            <div class="th-content">
                                                <i class="fas fa-cogs"></i>
                                                <span>إجراء</span>
                                            </div>
                                        </th>
                                    </tr>
                                </thead>
                                <tbody id="invoice-items" class="table-body-modern">
                                    <tr class="empty-state-row">
                                        <td colspan="8" class="empty-state">
                                            <div class="empty-state-content">
                                                <div class="empty-state-icon">
                                                    <i class="fas fa-shopping-basket"></i>
                                                </div>
                                                <h6 class="empty-state-title">لا توجد عناصر في الفاتورة</h6>
                                                <button type="button" class="btn-empty-add" onclick="addInvoiceItem()">
                                                    <i class="fas fa-plus me-2"></i>
                                                    إضافة أول عنصر
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                    
                    @error('items')
                        <div class="items-error-alert">
                            <div class="error-icon">
                                <i class="fas fa-exclamation-triangle"></i>
                            </div>
                            <div class="error-content">
                                <strong>خطأ في العناصر:</strong>
                                <span>{{ $message }}</span>
                            </div>
                        </div>
                    @enderror
                </div>
            </div>
        </div>
        
        <!-- بطاقة ملخص الفاتورة والسداد الموحدة -->
        <div class="col-lg-12">
            <div class="unified-summary-card">
                <div class="unified-card-header">
                    <div class="header-content-unified">
                
                        <div class="header-text-unified">
                            <h5 class="unified-card-title">                             <i class="fas fa-file-invoice-dollar"></i>
 ملخص الفاتورة والسداد</h5>
                        </div>
                    </div>
                    <div class="header-decoration-unified">
                        <div class="decoration-dot"></div>
                        <div class="decoration-dot"></div>
                        <div class="decoration-dot"></div>
                    </div>
                </div>
                
                <div class="unified-card-body">
                    <!-- معلومات الشركة مخفية -->
                    <div class="company-info-section" style="display: none;">
                        <div class="section-header-mini">
                            <i class="fas fa-building"></i>
                            <span>معلومات الشركة</span>
                        </div>
                        <div class="company-details">
                            <div class="company-name">{{ $company->name }}</div>
                            <div class="company-meta">
                                <span class="vat-number">
                                    <i class="fas fa-hashtag"></i>
                                    {{ $company->vat_number }}
                                </span>
                                <span class="company-address">
                                    <i class="fas fa-map-marker-alt"></i>
                                    {{ $company->getFullAddressAttribute() }}
                                </span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- قسم الإجماليات والسداد الموحد -->
                    <div class="unified-totals-payment-section">
                        <!-- الصف الأول - الإجماليات الأساسية -->
                        <div class="totals-payment-row">
                            <div class="total-item-compact">
                                <div class="total-label-compact">
                                    <i class="fas fa-list"></i>
                                    المجموع الفرعي
                                </div>
                                <div class="total-value-compact" id="subtotal-display">0.00 ر.س</div>
                            </div>
                            
                            <div class="total-item-compact editable-compact">
                                <div class="total-label-compact">
                                    الخصم
                                </div>
                                <div class="input-group">
                                    <div class="input-group-prepend">
                                        <button class="btn btn-primary btn-sm" type="button"><i class="fas fa-percentage"></i></button>
                                    </div>
                                    <input type="number" name="discount_amount" class="form-control"
                                           value="{{ old('discount_amount', 0) }}" step="0.01" min="0"
                                           onchange="calculateTotals()" placeholder="0.00">
                                    <div class="input-group-append">
                                        <span class="input-group-text">ر.س</span>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="total-item-compact">
                                <div class="total-label-compact">
                                    <i class="fas fa-receipt"></i>
                                    إجمالي الضريبة
                                </div>
                                <div class="total-value-compact" id="tax-total-display">0.00 ر.س</div>
                            </div>
                            
                            <div class="total-item-compact grand-total-compact">
                                <div class="total-label-compact">
                                    <i class="fas fa-money-bill-wave"></i>
                                    المبلغ الإجمالي
                                </div>
                                <div class="total-value-compact grand" id="total-display">0.00 ر.س</div>
                            </div>
                        </div>

                        <!-- الصف الثاني - معلومات السداد -->
                        <div class="totals-payment-row">
                            <!-- طريقة السداد -->
                            <div class="payment-field-compact">
                                <label class="payment-label-compact">طريقة السداد</label>
                                <div class="input-group mb-3">
                                    <div class="input-group-prepend">
                                        <button class="btn btn-primary" type="button"><i class="fas fa-wallet"></i></button>
                                    </div>
                                    <select name="payment_method" class="form-control @error('payment_method') is-invalid @enderror">
                                        <option value="">اختر طريقة السداد</option>
                                        <option value="cash" {{ old('payment_method') == 'cash' ? 'selected' : '' }}>نقداً</option>
                                        <option value="bank_transfer" {{ old('payment_method') == 'bank_transfer' ? 'selected' : '' }}>تحويل بنكي</option>
                                        <option value="check" {{ old('payment_method') == 'check' ? 'selected' : '' }}>شيك</option>
                                        <option value="credit_card" {{ old('payment_method') == 'credit_card' ? 'selected' : '' }}>بطاقة ائتمان</option>
                                        <option value="installment" {{ old('payment_method') == 'installment' ? 'selected' : '' }}>تقسيط</option>
                                    </select>
                                </div>
                                @error('payment_method')
                                    <div class="payment-error-compact">{{ $message }}</div>
                                @enderror
                            </div>
                            
                            <!-- المبلغ المدفوع -->
                            <div class="payment-field-compact">
                                <label class="payment-label-compact">المبلغ المدفوع</label>
                                <div class="input-group mb-3">
                                    <div class="input-group-prepend">
                                        <button class="btn btn-primary" type="button"><i class="fas fa-hand-holding-usd"></i></button>
                                    </div>
                                    <input type="number" name="paid_amount" class="form-control @error('paid_amount') is-invalid @enderror"
                                           value="{{ old('paid_amount', 0) }}" step="0.01" min="0"
                                           onchange="calculatePaymentAmounts()" placeholder="0.00">
                                    <div class="input-group-append">
                                        <span class="input-group-text">ر.س</span>
                                    </div>
                                </div>
                                @error('paid_amount')
                                    <div class="payment-error-compact">{{ $message }}</div>
                                @enderror
                            </div>
                            
                            <!-- المبلغ المتبقي -->
                            <div class="payment-field-compact">
                                <label class="payment-label-compact">المبلغ المتبقي</label>
                                <div class="input-group mb-3">
                                    <div class="input-group-prepend">
                                        <button class="btn btn-primary" type="button"><i class="fas fa-clock"></i></button>
                                    </div>
                                    <input type="number" name="remaining_amount" class="form-control"
                                           id="remaining-amount-input" readonly placeholder="0.00">
                                    <div class="input-group-append">
                                        <span class="input-group-text">ر.س</span>
                                    </div>
                                </div>
                                <div id="remaining-amount-display" class="payment-status-display-compact"></div>
                            </div>
                            
                            <!-- تاريخ السداد -->
                            <div class="payment-field-compact">
                                <label class="payment-label-compact">تاريخ السداد</label>
                                <div class="input-group mb-3">
                                    <div class="input-group-prepend">
                                        <button class="btn btn-primary" type="button"><i class="fas fa-calendar-alt"></i></button>
                                    </div>
                                    <input type="datetime-local" name="payment_date" class="form-control @error('payment_date') is-invalid @enderror"
                                           value="{{ old('payment_date', now()->format('Y-m-d\TH:i')) }}">
                                </div>
                                @error('payment_date')
                                    <div class="payment-error-compact">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <!-- ملاحظات السداد - صف منفصل -->
                        <div class="payment-notes-row">
                            <div class="payment-field-compact full-width">
                                <div class="input-group mb-3">
                                    <div class="input-group-prepend">
                                        <button class="btn btn-primary" type="button"><i class="fas fa-sticky-note"></i></button>
                                    </div>
                                    <textarea name="payment_notes" class="form-control" rows="2"
                                              placeholder="ملاحظات حول عملية السداد...">{{ old('payment_notes') }}</textarea>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- أزرار الحفظ المحسنة -->
            <div class="action-buttons-enhanced mt-4">
                <div class="buttons-grid">
                    <button type="submit" class="btn-action-enhanced btn-save-enhanced">
                        <div class="btn-icon-wrapper">
                            <i class="fas fa-save"></i>
                        </div>
                        <div class="btn-content">
                            <span class="btn-title">حفظ الفاتورة</span>
                            <span class="btn-subtitle">حفظ كمسودة</span>
                        </div>
                    </button>
                    
                    <button type="submit" name="action" value="save_and_send" class="btn-action-enhanced btn-send-enhanced">
                        <div class="btn-icon-wrapper">
                            <i class="fas fa-paper-plane"></i>
                        </div>
                        <div class="btn-content">
                            <span class="btn-title">حفظ وإرسال</span>
                            <span class="btn-subtitle">إرسال للعميل</span>
                        </div>
                    </button>
                    
                    <button type="button" onclick="window.print()" class="btn-action-enhanced btn-print-enhanced">
                        <div class="btn-icon-wrapper">
                            <i class="fas fa-print"></i>
                        </div>
                        <div class="btn-content">
                            <span class="btn-title">طباعة</span>
                            <span class="btn-subtitle">معاينة وطباعة</span>
                        </div>
                    </button>
                </div>
                
                <div class="secondary-actions">
                    <a href="{{ route('invoices.index') }}" class="btn-cancel-enhanced btn-success">
                        <i class="fas fa-arrow-right me-2"></i>
                        العودة للقائمة
                    </a>
                    
                    <button type="button" class="btn-draft-enhanced" onclick="saveDraft()">
                        <i class="fas fa-bookmark me-2"></i>
                        حفظ مسودة
                    </button>
                </div>
            </div>
        </div>
    </div>
</form>

<!-- نموذج اختيار المنتج المبسط -->
<div class="modal fade" id="productModal" tabindex="-1" aria-labelledby="productModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header bg-primary text-white">
                <h5 class="modal-title" id="productModalLabel">
                    <i class="fas fa-shopping-cart me-2"></i>
                    اختيار منتج أو خدمة
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="إغلاق"></button>
            </div>
            <div class="modal-body">
                <!-- شريط البحث البسيط -->
                <div class="mb-3">
                    <div class="input-group">
                        <span class="input-group-text">
                            <i class="fas fa-search"></i>
                        </span>
                        <input type="text" class="form-control" id="product-search" placeholder="ابحث عن منتج بالاسم أو الرمز...">
                    </div>
                </div>

                <!-- فلاتر بسيطة -->
                <div class="mb-3">
                    <div class="btn-group" role="group" aria-label="فلاتر المنتجات">
                        <input type="radio" class="btn-check" name="product-filter" id="filter-all" autocomplete="off" checked>
                        <label class="btn btn-outline-primary" for="filter-all">الكل</label>

                        <input type="radio" class="btn-check" name="product-filter" id="filter-products" autocomplete="off">
                        <label class="btn btn-outline-primary" for="filter-products">المنتجات</label>

                        <input type="radio" class="btn-check" name="product-filter" id="filter-services" autocomplete="off">
                        <label class="btn btn-outline-primary" for="filter-services">الخدمات</label>
                    </div>
                </div>

                <!-- قائمة المنتجات البسيطة -->
                <div class="product-list" style="max-height: 400px; overflow-y: auto;">
                    @foreach($products as $product)
                    <div class="product-item border rounded p-3 mb-2" data-type="{{ $product->type }}" data-name="{{ strtolower($product->name) }}" data-sku="{{ strtolower($product->sku) }}">
                        <div class="row align-items-center">
                            <div class="col-md-6">
                                <div class="d-flex align-items-center">
                                    <div class="me-3">
                                        @if($product->type == 'product')
                                            <div class="bg-success text-white rounded-circle d-flex align-items-center justify-content-center" style="width: 40px; height: 40px;">
                                                <i class="fas fa-box"></i>
                                            </div>
                                        @else
                                            <div class="bg-info text-white rounded-circle d-flex align-items-center justify-content-center" style="width: 40px; height: 40px;">
                                                <i class="fas fa-cogs"></i>
                                            </div>
                                        @endif
                                    </div>
                                    <div>
                                        <h6 class="mb-1">{{ $product->name }}</h6>
                                        <small class="text-muted">رمز: {{ $product->sku }}</small>
                                        @if($product->description)
                                            <br><small class="text-muted">{{ Str::limit($product->description, 50) }}</small>
                                        @endif
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3 text-center">
                                <div class="price-info">
                                    <strong class="text-primary">{{ number_format($product->price, 2) }} ر.س</strong>
                                    @if($product->is_taxable)
                                        <br><small class="text-muted">+ ضريبة {{ $product->tax_rate }}%</small>
                                    @endif
                                </div>
                                @if($product->type == 'product')
                                    <div class="stock-info mt-1">
                                        @if($product->stock_quantity > 0)
                                            <small class="badge bg-success">متوفر ({{ $product->stock_quantity }})</small>
                                        @else
                                            <small class="badge bg-danger">نفد المخزون</small>
                                        @endif
                                    </div>
                                @else
                                    <div class="stock-info mt-1">
                                        <small class="badge bg-info">خدمة متاحة</small>
                                    </div>
                                @endif
                            </div>
                            <div class="col-md-3 text-end">
                                <button type="button" class="btn btn-primary btn-sm"
                                        onclick="selectSimpleProduct({{ $product->id }}, '{{ addslashes($product->name) }}', '{{ $product->sku }}', {{ $product->price }}, {{ $product->tax_rate ?? 15 }}, '{{ $product->type }}', {{ $product->stock_quantity ?? 999 }})"
                                        {{ $product->type == 'product' && $product->stock_quantity <= 0 ? 'disabled' : '' }}>
                                    <i class="fas fa-plus me-1"></i>
                                    إضافة
                                </button>
                            </div>
                        </div>
                    </div>
                    @endforeach
                </div>

                <!-- رسالة عدم وجود نتائج -->
                <div id="no-results" class="text-center py-4" style="display: none;">
                    <i class="fas fa-search text-muted mb-2" style="font-size: 2rem;"></i>
                    <p class="text-muted">لم يتم العثور على منتجات مطابقة للبحث</p>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <i class="fas fa-times me-2"></i>
                    إغلاق
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@push('styles')
<!-- Select2 CSS -->
<link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
<link href="https://cdn.jsdelivr.net/npm/select2-bootstrap-5-theme@1.3.0/dist/select2-bootstrap-5-theme.rtl.min.css" rel="stylesheet" />

<style>
/* === تصميم مبسط وجميل === */
:root {
    --primary-color: #4f46e5;
    --secondary-color: #f8fafc;
    --success-color: #10b981;
    --warning-color: #f59e0b;
    --danger-color: #ef4444;
    --text-primary: #1f2937;
    --text-secondary: #6b7280;
    --border-color: #e5e7eb;
    --shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
    --border-radius: 8px;
    --font-size: 14px;
    --spacing: 1rem;
}

body {
    font-family: 'Cairo', 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    background-color: #f9fafb;
    font-size: var(--font-size);
    line-height: 1.5;
    color: var(--text-primary);
}

/* === تصميم البطاقات === */
.invoice-info-card,
.invoice-items-card,
.unified-summary-card,
.notification-card {
    background: white;
    border: 1px solid var(--border-color);
    border-radius: var(--border-radius);
    box-shadow: var(--shadow);
    margin-bottom: var(--spacing);
    overflow: hidden;
}

/* === رؤوس البطاقات === */
.card-header-modern,
.items-card-header,
.unified-card-header,
.card-header {
    background:  linear-gradient(135deg, rgb(37, 99, 235) 0%, rgb(29, 78, 216) 100%);
    color: white;
    padding: 1.5rem;
    border: none;
    position: relative;
}

.card-header-modern h5,
.items-card-title,
.unified-card-title,
.card-header h5 {
    margin: 0;
    font-size: 1.1rem;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

/* === محتوى البطاقات === */
.card-body-modern,
.items-card-body,
.unified-card-body,
.card-body {
    padding: 1.5rem;
}

/* === تصميم النماذج === */
.form-group-enhanced {
    margin-bottom: 1rem;
}

.form-label-enhanced,
.payment-label-compact {
    display: block;
    margin-bottom: 0.5rem;
    font-weight: 600;
    color: var(--text-primary);
    font-size: 0.875rem;
}

/* === تصميم Input Group === */
.input-group {
    direction: rtl;
    margin-bottom: 0.5rem;
}

.input-group .btn {
    border-radius: var(--border-radius) 0 0 var(--border-radius);
    background: var(--primary-color);
    border-color: var(--primary-color);
    color: white;
    padding: 0.75rem 1rem;
    font-size: var(--font-size);
    display: flex;
    align-items: center;
    justify-content: center;
}

.input-group .btn:hover {
    background: #4338ca;
    border-color: #4338ca;
}

.input-group .form-control {
    border-radius: 0 var(--border-radius) var(--border-radius) 0;
    border: 1px solid var(--border-color);
    padding: 0.75rem;
    font-size: var(--font-size);
    background: white;
    transition: all 0.2s ease;
}

.input-group .form-control:focus {
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
    outline: none;
}

.input-group-text {
    background: var(--secondary-color);
    border: 1px solid var(--border-color);
    color: var(--text-secondary);
    padding: 0.75rem;
    font-size: var(--font-size);
    border-radius: 0 var(--border-radius) var(--border-radius) 0;
}

.input-group-prepend .btn {
    border-radius: var(--border-radius) 0 0 var(--border-radius);
}

.input-group-append .input-group-text {
    border-radius: 0 var(--border-radius) var(--border-radius) 0;
}

/* تصميم خاص للـ Checkbox داخل Input Group */
.input-group .form-control.d-flex {
    background: var(--secondary-color);
    border: 1px solid var(--border-color);
}

/* تصميم Input Group في الجداول */
.table-body-modern .input-group {
    margin-bottom: 0;
    width: 100%;
}

.table-body-modern .input-group .btn {
    padding: 0.5rem 0.75rem;
    font-size: 0.75rem;
}

.table-body-modern .input-group .form-control {
    padding: 0.5rem;
    font-size: 0.875rem;
}

.table-body-modern .input-group-text {
    padding: 0.5rem 0.75rem;
    font-size: 0.75rem;
}

/* تحسين التصميم المتجاوب للـ Input Groups */
@media (max-width: 768px) {
    .input-group .btn {
        padding: 0.5rem 0.75rem;
        font-size: 0.75rem;
    }
    
    .input-group .form-control {
        padding: 0.5rem;
        font-size: 0.875rem;
    }
    
    .input-group-text {
        padding: 0.5rem 0.75rem;
        font-size: 0.75rem;
    }
    
    .table-body-modern .input-group .btn {
        padding: 0.375rem 0.5rem;
        font-size: 0.625rem;
    }
    
    .table-body-modern .input-group .form-control {
        padding: 0.375rem;
        font-size: 0.75rem;
    }
    
    .table-body-modern .input-group-text {
        padding: 0.375rem 0.5rem;
        font-size: 0.625rem;
    }
}

/* === تحسين محاذاة حقل العميل === */
.customer-input-group {
    display: flex;
    align-items: stretch;
    width: 100%;
}

.customer-input-group .input-group-prepend,
.customer-input-group .input-group-append {
    display: flex;
    align-items: stretch;
}

.customer-icon-btn,
.add-customer-btn {
    display: flex;
    align-items: center;
    justify-content: center;
    min-width: 45px;
    padding: 0.75rem;
    border: none;
    font-size: var(--font-size);
    white-space: nowrap;
    text-decoration: none;
}

.customer-icon-btn {
    background: var(--primary-color);
    color: white;
    border-radius: var(--border-radius) 0 0 var(--border-radius);
    border-right: 1px solid var(--primary-color);
}

.customer-icon-btn:hover {
    background: #4338ca;
    color: white;
}

.add-customer-btn {
    background: var(--success-color);
    color: white;
    border-radius: 0 var(--border-radius) var(--border-radius) 0;
    border-left: 1px solid var(--success-color);
    transition: all 0.2s ease;
}

.add-customer-btn:hover {
    background: #059669;
    color: white;
    text-decoration: none;
    transform: scale(1.02);
}

.customer-select {
    flex: 1;
    border-radius: 0;
    border-left: none;
    border-right: none;
    min-height: 48px;
    display: flex;
    align-items: center;
}

/* تحسين التصميم المتجاوب لحقل العميل */
@media (max-width: 768px) {
    .customer-icon-btn,
    .add-customer-btn {
        min-width: 40px;
        padding: 0.5rem 0.75rem;
        font-size: 0.875rem;
    }
    
    .customer-select {
        min-height: 42px;
        font-size: 0.875rem;
    }
}

.form-select-enhanced,
.form-control-enhanced,
.payment-select-compact,
.payment-input-compact,
.payment-textarea-compact {
    width: 100%;
    padding: 0.75rem;
    border: 1px solid var(--border-color);
    border-radius: var(--border-radius);
    font-size: var(--font-size);
    transition: all 0.2s ease;
    background: white;
}

.form-select-enhanced:focus,
.form-control-enhanced:focus,
.payment-select-compact:focus,
.payment-input-compact:focus,
.payment-textarea-compact:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
}

/* === تصميم الجدول === */
.invoice-items-table {
    width: 100%;
    border-collapse: collapse;
    margin: 0;
}

.table-header-modern th {
    background: var(--secondary-color);
    color: var(--text-primary);
    padding: 1rem 0.75rem;
    border: 1px solid var(--border-color);
    font-weight: 600;
    font-size: 0.875rem;
    text-align: center;
}

.table-body-modern td {
    padding: 1rem 0.75rem;
    border: 1px solid var(--border-color);
    vertical-align: middle;
    text-align: center;
}

/* === حالة الجدول الفارغ === */
.empty-state {
    text-align: center;
    padding: 3rem 1rem;
    color: var(--text-secondary);
}

.empty-state-icon i {
    font-size: 3rem;
    color: var(--border-color);
    margin-bottom: 1rem;
}

.empty-state-title {
    font-size: 1.1rem;
    font-weight: 600;
    margin-bottom: 1rem;
    color: var(--text-secondary);
}

/* === الأزرار === */
.btn-add-item,
.btn-empty-add,
.btn-action-enhanced {
    background: var(--primary-color);
    color: white;
    border: none;
    padding: 0.75rem 1.5rem;
    border-radius: var(--border-radius);
    font-size: var(--font-size);
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s ease;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
}

.btn-add-item:hover,
.btn-empty-add:hover,
.btn-action-enhanced:hover {
    background: #4338ca;
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(79, 70, 229, 0.3);
}

.btn-save-enhanced {
    background: var(--success-color);
}

.btn-save-enhanced:hover {
    background: #059669;
}

.btn-send-enhanced {
    background: var(--warning-color);
}

.btn-send-enhanced:hover {
    background: #d97706;
}

.btn-print-enhanced {
    background: var(--text-secondary);
}

.btn-print-enhanced:hover {
    background: #4b5563;
}

.btn-cancel-enhanced,
.btn-draft-enhanced {
    background: transparent;
    color: var(--text-secondary);
    border: 1px solid var(--border-color);
    padding: 0.75rem 1.5rem;
    border-radius: var(--border-radius);
    font-size: var(--font-size);
    cursor: pointer;
    transition: all 0.2s ease;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
}

.btn-cancel-enhanced:hover,
.btn-draft-enhanced:hover {
    background: var(--secondary-color);
    color: var(--text-primary);
    border-color: var(--primary-color);
}

/* === تصميم الإجماليات === */
.totals-payment-row {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
    margin-bottom: 1rem;
}

.total-item-compact {
    background: var(--secondary-color);
    padding: 1rem;
    border-radius: var(--border-radius);
    border: 1px solid var(--border-color);
}

.total-label-compact {
    font-size: 0.875rem;
    color: var(--text-secondary);
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    margin-bottom: 0.5rem;
}

.total-value-compact {
    font-size: 1.1rem;
    font-weight: 700;
    color: var(--text-primary);
}

.grand-total-compact .total-value-compact {
    color: var(--primary-color);
    font-size: 1.25rem;
}

/* === تصميم حقول الدفع === */
.payment-field-compact {
    display: flex;
    flex-direction: column;
}

.payment-input-wrapper-compact {
    position: relative;
    display: flex;
    align-items: center;
}

.payment-currency-compact {
    position: absolute;
    left: 0.75rem;
    color: var(--text-secondary);
    font-size: 0.875rem;
    pointer-events: none;
}

.payment-input-compact {
    padding-left: 2.5rem;
}

/* === تصميم أزرار العمل === */
.action-buttons-enhanced {
    background: white;
    border: 1px solid var(--border-color);
    border-radius: var(--border-radius);
    padding: 1.5rem;
    box-shadow: var(--shadow);
}

.buttons-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
    margin-bottom: 1rem;
}

.secondary-actions {
    display: flex;
    gap: 1rem;
    justify-content: center;
    padding-top: 1rem;
    border-top: 1px solid var(--border-color);
}

/* === التصميم المتجاوب === */
@media (max-width: 768px) {
    .form-row-modern {
        grid-template-columns: 1fr;
    }
    
    .totals-payment-row {
        grid-template-columns: 1fr;
    }
    
    .buttons-grid {
        grid-template-columns: 1fr;
    }
    
    .secondary-actions {
        flex-direction: column;
        align-items: center;
    }
    
    .card-body-modern,
    .items-card-body,
    .unified-card-body {
        padding: 1rem;
    }
}

/* === تحسينات إضافية === */
.form-row-modern {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1rem;
    margin-bottom: 1rem;
}

.select-wrapper,
.input-wrapper {
    position: relative;
}

.required-asterisk {
    color: var(--danger-color);
    margin-left: 0.25rem;
}

.invalid-feedback-enhanced {
    color: var(--danger-color);
    font-size: 0.875rem;
    margin-top: 0.25rem;
    display: flex;
    align-items: center;
    gap: 0.25rem;
}

/* === تحسين الجدول للهاتف === */
@media (max-width: 768px) {
    .invoice-items-table {
        font-size: 0.75rem;
    }
    
    .table-header-modern th,
    .table-body-modern td {
        padding: 0.5rem 0.25rem;
    }
}

/* === تصميم النموذج الخاص === */
.product-item {
    transition: all 0.2s ease;
}

.product-item:hover {
    background: var(--secondary-color);
    transform: translateY(-1px);
    box-shadow: var(--shadow);
}

/* === تحسين أيقونات النموذج === */
.label-icon,
.input-icon,
.select-icon {
    color: var(--primary-color);
}

/* === تصميم الـ Checkbox === */
.checkbox-wrapper {
    display: flex;
    align-items: flex-start;
    gap: 0.5rem;
    padding: 0.75rem;
    background: var(--secondary-color);
    border: 1px solid var(--border-color);
    border-radius: var(--border-radius);
    transition: all 0.2s ease;
}

.checkbox-wrapper:hover {
    background: #f1f5f9;
    border-color: var(--primary-color);
}

.form-check-input {
    margin: 0;
    width: 1.125rem;
    height: 1.125rem;
    border: 2px solid var(--border-color);
    border-radius: 4px;
    background: white;
    transition: all 0.2s ease;
}

.form-check-input:checked {
    background-color: var(--primary-color);
    border-color: var(--primary-color);
}

.form-check-input:focus {
    outline: none;
    box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
}

.checkbox-label {
    font-size: 0.875rem;
    color: var(--text-primary);
    margin: 0;
    cursor: pointer;
    line-height: 1.4;
}

.three-columns {
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
}

/* === إنهاء التصميم === */
</style>
@endpush

@push('scripts')
<!-- Select2 JS -->
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/i18n/ar.js"></script>
<script>
let itemIndex = 0;

// تهيئة Select2 للعملاء
$(document).ready(function() {
    $('#customer-select').select2({
        theme: 'bootstrap-5',
        language: 'ar',
        placeholder: 'اختر العميل أو ابحث عنه...',
        allowClear: true,
        width: '100%',
        templateResult: formatCustomer,
        templateSelection: formatCustomerSelection,
        escapeMarkup: function(markup) {
            return markup;
        }
    });
});

// تنسيق عرض العملاء في القائمة المنسدلة
function formatCustomer(customer) {
    if (!customer.id) {
        return customer.text;
    }
    
    const $customer = $(customer.element);
    const customerType = $customer.data('customer-type');
    const vatNumber = $customer.data('vat-number');
    const phone = $customer.data('phone');
    const email = $customer.data('email');
    
    let typeLabel = customerType === 'business' ? 'شركة' : 'فرد';
    let typeBadgeClass = customerType === 'business' ? 'customer-type-business' : 'customer-type-individual';
    
    let html = `
        <div>
            <strong>${customer.text}</strong>
            <span class="customer-type-badge ${typeBadgeClass}">${typeLabel}</span>
            <div class="customer-info">
    `;
    
    if (vatNumber) {
        html += `<small>الرقم الضريبي: ${vatNumber}</small><br>`;
    }
    if (phone) {
        html += `<small>الهاتف: ${phone}</small>`;
    }
    if (email) {
        html += `<small> | البريد: ${email}</small>`;
    }
    
    html += `
            </div>
        </div>
    `;
    
    return html;
}

// تنسيق عرض العميل المختار
function formatCustomerSelection(customer) {
    if (!customer.id) {
        return customer.text;
    }
    
    const $customer = $(customer.element);
    const customerType = $customer.data('customer-type');
    const vatNumber = $customer.data('vat-number');
    
    let result = customer.text;
    if (customerType === 'business' && vatNumber) {
        result += ` (${vatNumber})`;
    }
    
    return result;
}

// دالة إضافة عنصر فاتورة
function addInvoiceItem() {
    // إزالة الصف الفارغ إذا كان موجوداً
    const emptyRow = document.querySelector('.empty-state-row');
    if (emptyRow) {
        emptyRow.remove();
    }
    
    // فتح مودال اختيار المنتج
    const productModal = new bootstrap.Modal(document.getElementById('productModal'));
    productModal.show();
}

function selectProduct(button) {
    const row = button.closest('tr');
    const product = JSON.parse(row.getAttribute('data-product'));
    
    const itemHtml = `
        <tr id="item-${itemIndex}">
            <td>
                <input type="hidden" name="items[${itemIndex}][product_id]" value="${product.id}">
                <strong>${product.name}</strong>
                ${product.description ? `<br><small class="text-muted">${product.description}</small>` : ''}
                <br><small class="text-info">رمز: ${product.sku}</small>
            </td>
            <td>
                <div class="input-group">
                    <div class="input-group-prepend">
                        <button class="btn btn-primary btn-sm" type="button"><i class="fas fa-sort-numeric-up"></i></button>
                    </div>
                    <input type="number" name="items[${itemIndex}][quantity]" class="form-control"
                           value="1" step="0.001" min="0.001" required onchange="calculateItemTotal(${itemIndex})">
                    <div class="input-group-append">
                        <span class="input-group-text">قطعة</span>
                    </div>
                </div>
            </td>
            <td>
                <div class="input-group">
                    <div class="input-group-prepend">
                        <button class="btn btn-primary btn-sm" type="button"><i class="fas fa-tag"></i></button>
                    </div>
                    <input type="number" name="items[${itemIndex}][unit_price]" class="form-control"
                           value="${product.price}" step="0.01" min="0" required onchange="calculateItemTotal(${itemIndex})">
                    <div class="input-group-append">
                        <span class="input-group-text">ر.س</span>
                    </div>
                </div>
            </td>
            <td>
                <div class="input-group">
                    <div class="input-group-prepend">
                        <button class="btn btn-primary btn-sm" type="button"><i class="fas fa-percentage"></i></button>
                    </div>
                    <input type="number" name="items[${itemIndex}][discount_amount]" class="form-control"
                           value="0" step="0.01" min="0" onchange="calculateItemTotal(${itemIndex})">
                    <div class="input-group-append">
                        <span class="input-group-text">ر.س</span>
                    </div>
                </div>
            </td>
            <td>
                <span id="line-total-${itemIndex}">${product.price}</span> ر.س
            </td>
            <td>
                <span id="tax-amount-${itemIndex}">${(product.price * product.tax_rate / 100).toFixed(2)}</span> ر.س
            </td>
            <td>
                <strong><span id="total-with-tax-${itemIndex}">${(product.price * (1 + product.tax_rate / 100)).toFixed(2)}</span> ر.س</strong>
            </td>
            <td>
                <button type="button" class="btn btn-danger btn-sm" onclick="removeItem(${itemIndex})">
                    <i class="fas fa-trash"></i>
                </button>
            </td>
        </tr>
    `;
    
    document.getElementById('invoice-items').insertAdjacentHTML('beforeend', itemHtml);
    itemIndex++;
    calculateTotals();
    $('#productModal').modal('hide');
}

function removeItem(index) {
    document.getElementById(`item-${index}`).remove();
    calculateTotals();
}

function calculateItemTotal(index) {
    const quantityInput = document.querySelector(`input[name="items[${index}][quantity]"]`);
    const priceInput = document.querySelector(`input[name="items[${index}][unit_price]"]`);
    const discountInput = document.querySelector(`input[name="items[${index}][discount_amount]"]`);
    
    const quantity = parseFloat(quantityInput.value) || 0;
    const price = parseFloat(priceInput.value) || 0;
    const discount = parseFloat(discountInput.value) || 0;
    
    const lineTotal = (quantity * price) - discount;
    const taxAmount = lineTotal * 0.15; // 15% ضريبة
    const totalWithTax = lineTotal + taxAmount;
    
    document.getElementById(`line-total-${index}`).textContent = lineTotal.toFixed(2);
    document.getElementById(`tax-amount-${index}`).textContent = taxAmount.toFixed(2);
    document.getElementById(`total-with-tax-${index}`).textContent = totalWithTax.toFixed(2);
    
    calculateTotals();
}

function calculateTotals() {
    let subtotal = 0;
    let totalTax = 0;
    
    // حساب المجاميع من كل الصفوف
    document.querySelectorAll('#invoice-items tr').forEach(row => {
        const lineTotalElement = row.querySelector('[id^="line-total-"]');
        const taxAmountElement = row.querySelector('[id^="tax-amount-"]');
        
        if (lineTotalElement && taxAmountElement) {
            subtotal += parseFloat(lineTotalElement.textContent) || 0;
            totalTax += parseFloat(taxAmountElement.textContent) || 0;
        }
    });
    
    const discount = parseFloat(document.querySelector('input[name="discount_amount"]').value) || 0;
    const finalTotal = subtotal + totalTax - discount;
    
    document.getElementById('subtotal-display').textContent = subtotal.toFixed(2) + ' ر.س';
    document.getElementById('tax-total-display').textContent = totalTax.toFixed(2) + ' ر.س';
    document.getElementById('total-display').textContent = finalTotal.toFixed(2) + ' ر.س';
    
    // تحديث المبلغ المتبقي
    calculatePaymentAmounts();
}

// حساب مبالغ السداد
function calculatePaymentAmounts() {
    const totalAmountText = document.getElementById('total-display').textContent;
    const totalAmount = parseFloat(totalAmountText.replace(' ر.س', '')) || 0;
    const paidAmount = parseFloat(document.querySelector('input[name="paid_amount"]').value) || 0;
    const remainingAmount = Math.max(0, totalAmount - paidAmount);
    
    // تحديث المبلغ المتبقي
    document.getElementById('remaining-amount-input').value = remainingAmount.toFixed(2);
    
    // تحديث عرض المبلغ المتبقي مع اللون المناسب
    const remainingDisplay = document.getElementById('remaining-amount-display');
    if (remainingAmount <= 0) {
        remainingDisplay.innerHTML = '<span class="payment-status-badge payment-status-paid">مدفوعة بالكامل</span>';
    } else if (paidAmount > 0) {
        remainingDisplay.innerHTML = '<span class="payment-status-badge payment-status-partial">مدفوعة جزئياً</span>';
    } else {
        remainingDisplay.innerHTML = '<span class="payment-status-badge payment-status-unpaid">غير مدفوعة</span>';
    }
    
    // تحديث نسبة السداد
    const paymentPercentage = totalAmount > 0 ? (paidAmount / totalAmount) * 100 : 0;
    updatePaymentProgress(paymentPercentage);
}

// تحديث شريط تقدم السداد
function updatePaymentProgress(percentage) {
    let progressBar = document.getElementById('payment-progress');
    if (!progressBar) {
        // إنشاء شريط التقدم إذا لم يكن موجوداً
        const progressContainer = document.createElement('div');
        progressContainer.innerHTML = `
            <div class="mt-3">
                <div class="d-flex justify-content-between mb-2">
                    <span class="small text-bold">تقدم السداد</span>
                    <span class="small text-bold" id="payment-percentage">${percentage.toFixed(1)}%</span>
                </div>
                <div class="payment-progress">
                    <div class="progress-bar" id="payment-progress" role="progressbar"
                         style="width: ${percentage}%"
                         aria-valuenow="${percentage}" aria-valuemin="0" aria-valuemax="100">
                    </div>
                </div>
            </div>
        `;
        document.getElementById('remaining-amount-display').appendChild(progressContainer);
        progressBar = document.getElementById('payment-progress');
    }
    
    // تحديث الشريط
    progressBar.style.width = percentage + '%';
    progressBar.setAttribute('aria-valuenow', percentage);
    document.getElementById('payment-percentage').textContent = percentage.toFixed(1) + '%';
    
    // تحديث لون الشريط
    progressBar.className = 'progress-bar';
    if (percentage >= 100) {
        progressBar.classList.add('bg-success');
    } else if (percentage > 0) {
        progressBar.classList.add('bg-warning');
    } else {
        progressBar.classList.add('bg-danger');
    }
}

// البحث في المنتجات (نسخة محدثة)
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('product-search');
    if (searchInput) {
        searchInput.addEventListener('input', function() {
            filterProductsSimple();
        });
    }
    
    // ربط أحداث الفلاتر
    const filterButtons = document.querySelectorAll('input[name="product-filter"]');
    filterButtons.forEach(button => {
        button.addEventListener('change', filterProductsSimple);
    });
});

// تحديد متطلبات العميل حسب نوع الفاتورة
document.querySelector('select[name="type"]').addEventListener('change', function() {
    const customerSelect = $('#customer-select');
    const label = customerSelect.parent().find('label');
    
    if (this.value === 'standard') {
        customerSelect.prop('required', true);
        label.html('العميل <span class="text-danger">*</span>');
        // تحديث placeholder لSelect2
        customerSelect.data('select2').options.options.placeholder = 'اختر العميل (مطلوب للفواتير المعيارية)';
    } else {
        customerSelect.prop('required', false);
        label.text('العميل');
        // تحديث placeholder لSelect2
        customerSelect.data('select2').options.options.placeholder = 'اختر العميل (اختياري للفواتير المبسطة)';
    }
    
    // إعادة تهيئة Select2 لتطبيق التغييرات
    customerSelect.trigger('change.select2');
});

// تحديث تاريخ الاستحقاق بناءً على تاريخ الإصدار
document.querySelector('input[name="issue_date"]').addEventListener('change', function() {
    const issueDateInput = this;
    const dueDateInput = document.getElementById('due-date-input');
    
    if (issueDateInput.value) {
        // تعيين الحد الأدنى لتاريخ الاستحقاق ليكون بنفس تاريخ الإصدار
        dueDateInput.min = issueDateInput.value;
        
        // إذا كان تاريخ الاستحقاق أقل من تاريخ الإصدار، تحديثه لنفس تاريخ الإصدار
        if (dueDateInput.value && dueDateInput.value < issueDateInput.value) {
            dueDateInput.value = issueDateInput.value;
        }
        
        // إذا لم يكن هناك تاريخ استحقاق، تعيينه لنفس تاريخ الإصدار
        if (!dueDateInput.value) {
            dueDateInput.value = issueDateInput.value;
        }
    }
});

// التحقق من صحة تاريخ الاستحقاق عند تغييره
document.getElementById('due-date-input').addEventListener('change', function() {
    const dueDateInput = this;
    const issueDateInput = document.querySelector('input[name="issue_date"]');
    
    if (issueDateInput.value && dueDateInput.value && dueDateInput.value < issueDateInput.value) {
        // إظهار رسالة خطأ
        showValidationModal('خطأ في التاريخ', 'تاريخ الاستحقاق لا يمكن أن يكون قبل تاريخ الإصدار', 'warning');
        
        // تعيين تاريخ الاستحقاق لنفس تاريخ الإصدار
        dueDateInput.value = issueDateInput.value;
    }
});

// التحقق من وجود عناصر قبل الإرسال
document.getElementById('invoice-form').addEventListener('submit', function(e) {
    const items = document.querySelectorAll('#invoice-items tr');
    if (items.length === 0) {
        e.preventDefault();
        showValidationModal('عنصر مطلوب', 'يجب إضافة عنصر واحد على الأقل للفاتورة', 'warning');
        return;
    }
    
    // التحقق من صحة المبلغ المدفوع
    const totalAmountText = document.getElementById('total-display').textContent;
    const totalAmount = parseFloat(totalAmountText.replace(' ر.س', '')) || 0;
    const paidAmount = parseFloat(document.querySelector('input[name="paid_amount"]').value) || 0;
    
    if (paidAmount > totalAmount) {
        e.preventDefault();
        showValidationModal('خطأ في المبلغ', 'المبلغ المدفوع لا يمكن أن يكون أكبر من المبلغ الإجمالي للفاتورة', 'danger');
        return;
    }
});

// إضافة حدث تغيير لحقل المبلغ المدفوع
document.querySelector('input[name="paid_amount"]').addEventListener('input', calculatePaymentAmounts);

// تهيئة العرض عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', function() {
    calculatePaymentAmounts();
    
    // تهيئة تاريخ الاستحقاق
    const issueDateInput = document.querySelector('input[name="issue_date"]');
    const dueDateInput = document.getElementById('due-date-input');
    
    if (issueDateInput && dueDateInput) {
        // تعيين الحد الأدنى لتاريخ الاستحقاق
        if (issueDateInput.value) {
            dueDateInput.min = issueDateInput.value;
            
            // إذا لم يكن هناك تاريخ استحقاق محدد، تعيينه لنفس تاريخ الإصدار
            if (!dueDateInput.value) {
                dueDateInput.value = issueDateInput.value;
            }
        }
    }
});

// دالة عرض رسائل التحقق في modal
function showValidationModal(title, message, type = 'warning') {
    const iconClass = type === 'danger' ? 'fas fa-exclamation-triangle' :
                     type === 'warning' ? 'fas fa-exclamation-circle' :
                     'fas fa-info-circle';
    
    const modalHtml = `
        <div class="modal fade" id="validationModal" tabindex="-1" aria-hidden="true">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header bg-${type} text-white">
                        <h5 class="modal-title">
                            <i class="${iconClass}"></i>
                            ${title}
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="alert alert-${type} border-0">
                            <i class="${iconClass} me-2"></i>
                            ${message}
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                            <i class="fas fa-check"></i>
                            فهمت
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    // إزالة modal سابق إن وجد
    const existingModal = document.getElementById('validationModal');
    if (existingModal) {
        existingModal.remove();
    }
    
    // إضافة modal جديد
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    
    // عرض modal
    const modal = new bootstrap.Modal(document.getElementById('validationModal'));
    modal.show();
    
    // إزالة modal بعد إغلاقه
    document.getElementById('validationModal').addEventListener('hidden.bs.modal', function() {
        this.remove();
    });
}

// ===== وظائف المودال المحسن للمنتجات =====

// متغيرات عامة للمودال
let selectedProducts = [];
let currentView = 'grid';
let currentFilter = 'all';
let allProducts = [];

// تهيئة المودال عند تحميل الصفحة
$(document).ready(function() {
    initializeProductModal();
});

// تهيئة المودال
function initializeProductModal() {
    // جمع بيانات المنتجات
    collectProductsData();
    
    // ربط الأحداث
    bindProductModalEvents();
    
    // تهيئة البحث
    initializeSearch();
}

// جمع بيانات المنتجات من DOM
function collectProductsData() {
    allProducts = [];
    
    // جمع البيانات من Grid View
    document.querySelectorAll('.product-card').forEach(card => {
        const productData = JSON.parse(card.getAttribute('data-product'));
        const productType = card.getAttribute('data-type');
        const stockQuantity = parseInt(card.getAttribute('data-stock'));
        
        allProducts.push({
            ...productData,
            type: productType,
            stock_quantity: stockQuantity,
            element: card
        });
    });
}

// ربط الأحداث
function bindProductModalEvents() {
    // أحداث البحث
    const searchInput = document.getElementById('product-search');
    const searchClear = document.getElementById('search-clear');
    
    if (searchInput) {
        searchInput.addEventListener('input', handleSearch);
        searchInput.addEventListener('keydown', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                handleSearch();
            }
        });
    }
    
    if (searchClear) {
        searchClear.addEventListener('click', clearSearch);
    }
    
    // أحداث الفلاتر
    document.querySelectorAll('.filter-tab').forEach(tab => {
        tab.addEventListener('click', function() {
            const filter = this.getAttribute('data-filter');
            switchFilter(filter);
        });
    });
    
    // أحداث تبديل العرض
    document.querySelectorAll('.view-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const view = this.getAttribute('data-view');
            switchView(view);
        });
    });
    
    // إغلاق البحث عند الضغط على خارج المودال
    document.getElementById('productModal').addEventListener('hidden.bs.modal', function() {
        resetModalState();
    });
}

// إعادة تعيين حالة المودال
function resetModalState() {
    // مسح البحث
    const searchInput = document.getElementById('product-search');
    if (searchInput) {
        searchInput.value = '';
    }
    
    // إخفاء زر المسح
    const searchClear = document.getElementById('search-clear');
    if (searchClear) {
        searchClear.style.display = 'none';
    }
    
    // إعادة تعيين الفلتر للكل
    const allFilter = document.getElementById('filter-all');
    if (allFilter) {
        allFilter.checked = true;
    }
    
    // تحديث الفلاتر النشطة
    document.querySelectorAll('.filter-tab').forEach(tab => {
        tab.classList.remove('active');
    });
    
    const allTab = document.querySelector('[data-filter="all"]');
    if (allTab) {
        allTab.classList.add('active');
    }
    
    // إعادة عرض جميع المنتجات
    currentFilter = 'all';
    if (typeof filterProductsSimple === 'function') {
        filterProductsSimple();
    }
    
    // إزالة التركيز من العناصر
    if (document.activeElement) {
        document.activeElement.blur();
    }
}

// تهيئة البحث
function initializeSearch() {
    const searchInput = document.getElementById('product-search');
    if (searchInput) {
        // إضافة تأثيرات بصرية
        searchInput.addEventListener('focus', function() {
            this.parentElement.classList.add('search-focused');
        });
        
        searchInput.addEventListener('blur', function() {
            this.parentElement.classList.remove('search-focused');
        });
    }
}

// معالجة البحث
function handleSearch() {
    const searchInput = document.getElementById('product-search');
    const searchTerm = searchInput.value.toLowerCase().trim();
    const searchClear = document.getElementById('search-clear');
    
    // إظهار/إخفاء زر المسح
    if (searchTerm) {
        searchClear.style.display = 'flex';
    } else {
        searchClear.style.display = 'none';
    }
    
    // تطبيق البحث والفلترة
    filterProductsSimple();
}

// مسح البحث
function clearSearch() {
    const searchInput = document.getElementById('product-search');
    const searchClear = document.getElementById('search-clear');
    
    searchInput.value = '';
    searchClear.style.display = 'none';
    
    // إعادة عرض جميع المنتجات
    filterProductsSimple();
    
    // التركيز على حقل البحث
    searchInput.focus();
}

// تبديل الفلتر
function switchFilter(filter) {
    // تحديث الفلتر النشط
    document.querySelectorAll('.filter-tab').forEach(tab => {
        tab.classList.remove('active');
    });
    
    document.querySelector(`[data-filter="${filter}"]`).classList.add('active');
    
    currentFilter = filter;
    filterProductsSimple();
}

// تبديل طريقة العرض
function switchView(view) {
    // تحديث الأزرار
    document.querySelectorAll('.view-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    
    document.querySelector(`[data-view="${view}"]`).classList.add('active');
    
    // تبديل العرض
    const gridView = document.getElementById('products-grid');
    const listView = document.getElementById('products-list');
    
    if (view === 'grid') {
        gridView.classList.add('active');
        listView.classList.remove('active');
        currentView = 'grid';
    } else {
        listView.classList.add('active');
        gridView.classList.remove('active');
        currentView = 'list';
    }
    
    // إعادة تطبيق الفلترة
    filterProductsSimple();
}

// فلترة المنتجات
// دالة إظهار Toast للنجاح
function showSuccessToast(message) {
    const toastHtml = `
        <div class="toast align-items-center text-white bg-success border-0" role="alert" aria-live="assertive" aria-atomic="true">
            <div class="d-flex">
                <div class="toast-body">
                    <i class="fas fa-check-circle me-2"></i>
                    ${message}
                </div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
            </div>
        </div>
    `;
    
    // إضافة Toast container إذا لم يكن موجود
    let toastContainer = document.getElementById('toast-container');
    if (!toastContainer) {
        toastContainer = document.createElement('div');
        toastContainer.id = 'toast-container';
        toastContainer.className = 'toast-container position-fixed top-0 end-0 p-3';
        toastContainer.style.zIndex = '9999';
        document.body.appendChild(toastContainer);
    }
    
    // إضافة Toast
    toastContainer.insertAdjacentHTML('beforeend', toastHtml);
    
    // إظهار Toast
    const toastElement = toastContainer.lastElementChild;
    const toast = new bootstrap.Toast(toastElement, {
        autohide: true,
        delay: 3000
    });
    toast.show();
    
    // إزالة Toast بعد إخفائه
    toastElement.addEventListener('hidden.bs.toast', function() {
        this.remove();
    });
}

// دالة فلترة المنتجات المبسطة (محدثة)
function filterProductsSimple() {
    const searchInput = document.getElementById('product-search');
    if (!searchInput) return;
    
    const searchTerm = searchInput.value.toLowerCase().trim();
    const selectedFilter = document.querySelector('input[name="product-filter"]:checked');
    const filterValue = selectedFilter ? selectedFilter.value : 'all';
    
    const productItems = document.querySelectorAll('.product-item');
    const noResults = document.getElementById('no-results');
    let visibleCount = 0;
    
    productItems.forEach(item => {
        const productName = item.getAttribute('data-name') || '';
        const productSku = item.getAttribute('data-sku') || '';
        const productType = item.getAttribute('data-type') || '';
        
        // فلترة حسب النوع
        let typeMatch = false;
        if (filterValue === 'all') {
            typeMatch = true;
        } else if (filterValue === 'products' && productType === 'product') {
            typeMatch = true;
        } else if (filterValue === 'services' && productType === 'service') {
            typeMatch = true;
        }
        
        // فلترة حسب البحث
        let searchMatch = true;
        if (searchTerm) {
            const searchableText = (productName + ' ' + productSku).toLowerCase();
            searchMatch = searchableText.includes(searchTerm);
        }
        
        // إظهار أو إخفاء العنصر
        if (typeMatch && searchMatch) {
            item.style.display = 'block';
            visibleCount++;
        } else {
            item.style.display = 'none';
        }
    });
    
    // إظهار رسالة عدم وجود نتائج
    if (noResults) {
        if (visibleCount === 0) {
            noResults.style.display = 'block';
        } else {
            noResults.style.display = 'none';
        }
    }
}

// دالة اختيار منتج مبسطة
function selectSimpleProduct(productId, productName, productSku, productPrice, taxRate, productType, stockQuantity) {
    const productData = {
        id: productId,
        name: productName,
        sku: productSku,
        price: productPrice,
        tax_rate: taxRate || 15,
        type: productType,
        stock_quantity: stockQuantity
    };
    
    // التحقق من وجود المنتج مسبقاً في الفاتورة
    try {
        const invoiceItems = document.getElementById('invoice-items');
        if (!invoiceItems) {
            console.error('عنصر invoice-items غير موجود');
            return;
        }
        
        const existingItem = invoiceItems.querySelector(`input[value="${productData.id}"]`);
        if (existingItem) {
            // زيادة الكمية إذا كان موجود
            const row = existingItem.closest('tr');
            if (row) {
                const quantityInput = row.querySelector('input[name*="[quantity]"]');
                if (quantityInput) {
                    const currentQuantity = parseFloat(quantityInput.value) || 0;
                    quantityInput.value = currentQuantity + 1;
                    
                    // إعادة حساب الإجماليات
                    const itemIndexMatch = quantityInput.name.match(/\[(\d+)\]/);
                    if (itemIndexMatch) {
                        const itemIndex = itemIndexMatch[1];
                        calculateItemTotal(itemIndex);
                    }
                    
                    if (typeof showSuccessToast === 'function') {
                        showSuccessToast(`تم زيادة كمية "${productData.name}" في الفاتورة`);
                    }
                }
            }
        } else {
            // إضافة منتج جديد
            if (typeof addProductToInvoice === 'function') {
                addProductToInvoice(productData);
                if (typeof showSuccessToast === 'function') {
                    showSuccessToast(`تم إضافة "${productData.name}" إلى الفاتورة`);
                }
            }
        }
        
        // إغلاق المودال
        const productModal = document.getElementById('productModal');
        if (productModal) {
            const modal = bootstrap.Modal.getInstance(productModal);
            if (modal) {
                modal.hide();
            }
        }
        
    } catch (error) {
        console.error('خطأ في اختيار المنتج:', error);
        if (typeof showValidationModal === 'function') {
            showValidationModal('خطأ', 'حدث خطأ أثناء إضافة المنتج. يرجى المحاولة مرة أخرى.', 'danger');
        }
    }
}

// دالة إضافة منتج للفاتورة المبسطة
function addProductToInvoice(product) {
    // إزالة الصف الفارغ إذا كان موجود
    const emptyRow = document.querySelector('.empty-state-row');
    if (emptyRow) {
        emptyRow.remove();
    }
    
    // إنشاء صف جديد
    const itemHtml = `
        <tr id="item-${itemIndex}" class="invoice-item-row">
            <td class="product-cell">
                <input type="hidden" name="items[${itemIndex}][product_id]" value="${product.id}">
                <div class="product-info">
                    <div class="product-name">${product.name}</div>
                    ${product.description ? `<div class="product-desc">${product.description.substring(0, 60)}${product.description.length > 60 ? '...' : ''}</div>` : ''}
                    <div class="product-meta">
                        <span class="product-sku"><i class="fas fa-barcode"></i> ${product.sku}</span>
                        <span class="product-type ${product.type}">
                            <i class="fas fa-${product.type === 'product' ? 'box' : 'cogs'}"></i>
                            ${product.type === 'product' ? 'منتج' : 'خدمة'}
                        </span>
                    </div>
                </div>
            </td>
            <td class="quantity-cell">
                <div class="input-group">
                    <div class="input-group-prepend">
                        <button class="btn btn-primary btn-sm" type="button"><i class="fas fa-sort-numeric-up"></i></button>
                    </div>
                    <input type="number" name="items[${itemIndex}][quantity]" class="form-control"
                           value="1" step="0.001" min="0.001" required onchange="calculateItemTotal(${itemIndex})"
                           ${product.type === 'product' && product.stock_quantity <= 1 ? 'max="' + product.stock_quantity + '"' : ''}>
                    <div class="input-group-append">
                        <span class="input-group-text">${product.unit || 'قطعة'}</span>
                    </div>
                </div>
            </td>
            <td class="price-cell">
                <div class="input-group">
                    <div class="input-group-prepend">
                        <button class="btn btn-primary btn-sm" type="button"><i class="fas fa-tag"></i></button>
                    </div>
                    <input type="number" name="items[${itemIndex}][unit_price]" class="form-control"
                           value="${product.price}" step="0.01" min="0" required onchange="calculateItemTotal(${itemIndex})">
                    <div class="input-group-append">
                        <span class="input-group-text">ر.س</span>
                    </div>
                </div>
            </td>
            <td class="discount-cell">
                <div class="input-group">
                    <div class="input-group-prepend">
                        <button class="btn btn-primary btn-sm" type="button"><i class="fas fa-percentage"></i></button>
                    </div>
                    <input type="number" name="items[${itemIndex}][discount_amount]" class="form-control"
                           value="0" step="0.01" min="0" onchange="calculateItemTotal(${itemIndex})">
                    <div class="input-group-append">
                        <span class="input-group-text">ر.س</span>
                    </div>
                </div>
            </td>
            <td class="subtotal-cell">
                <div class="amount-display">
                    <span id="line-total-${itemIndex}" class="amount">${parseFloat(product.price).toFixed(2)}</span>
                    <span class="currency">ر.س</span>
                </div>
            </td>
            <td class="tax-cell">
                <div class="amount-display">
                    <span id="tax-amount-${itemIndex}" class="amount">${(parseFloat(product.price) * (product.tax_rate || 15) / 100).toFixed(2)}</span>
                    <span class="currency">ر.س</span>
                </div>
            </td>
            <td class="total-cell">
                <div class="amount-display total">
                    <span id="total-with-tax-${itemIndex}" class="amount">${(parseFloat(product.price) * (1 + (product.tax_rate || 15) / 100)).toFixed(2)}</span>
                    <span class="currency">ر.س</span>
                </div>
            </td>
            <td class="actions-cell">
                <button type="button" class="btn-remove-item" onclick="removeItem(${itemIndex})" title="حذف العنصر">
                    <i class="fas fa-trash-alt"></i>
                </button>
            </td>
        </tr>
    `;
    
    // إضافة الصف للجدول
    document.getElementById('invoice-items').insertAdjacentHTML('beforeend', itemHtml);
    
    // تحديث العداد وحساب الإجماليات
    itemIndex++;
    calculateTotals();
}

// دالة حفظ المسودة
function saveDraft() {
    // إضافة حقل مخفي لتحديد أن هذا حفظ مسودة
    const form = document.getElementById('invoice-form');
    const draftInput = document.createElement('input');
    draftInput.type = 'hidden';
    draftInput.name = 'save_as_draft';
    draftInput.value = '1';
    form.appendChild(draftInput);
    
    // إرسال النموذج
    form.submit();
    
    // إظهار رسالة نجاح
    showSuccessToast('تم حفظ الفاتورة كمسودة');
}

</script>
@endpush