@extends('layouts.app')

@section('title', 'تقرير أداء التحصيل')
@section('page-title', 'تقرير أداء التحصيل')

@push('styles')
<style>
    /* Modern CSS Variables */
    :root {
        --primary-color: #3b82f6;
        --secondary-color: #6b7280;
        --success-color: #10b981;
        --warning-color: #f59e0b;
        --danger-color: #ef4444;
        --info-color: #06b6d4;
        --bg-gradient: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        --card-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
        --card-shadow-hover: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
        --border-radius: 16px;
        --transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    }

    /* Global Styles */
    body {
        background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
        font-family: 'Cairo', sans-serif;
    }

    /* Modern Header */
    .modern-header {
        background: var(--bg-gradient);
        padding: 2rem 0;
        margin-bottom: 2rem;
        border-radius: 0 0 var(--border-radius) var(--border-radius);
        box-shadow: var(--card-shadow);
    }

    .header-content {
        display: flex;
        justify-content: space-between;
        align-items: center;
        max-width: 1200px;
        margin: 0 auto;
        padding: 0 1.5rem;
    }

    .header-title {
        color: white;
        margin: 0;
        font-size: 2rem;
        font-weight: 700;
        display: flex;
        align-items: center;
        gap: 0.75rem;
    }

    .header-actions {
        display: flex;
        gap: 1rem;
        align-items: center;
    }

    .btn-modern {
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        padding: 0.75rem 1.5rem;
        border-radius: 12px;
        font-weight: 600;
        text-decoration: none;
        transition: var(--transition);
        border: none;
        cursor: pointer;
        font-size: 0.9rem;
    }

    .btn-primary {
        background: white;
        color: var(--primary-color);
        box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
    }

    .btn-primary:hover {
        transform: translateY(-1px);
        box-shadow: 0 4px 8px rgba(0, 0, 0, 0.15);
    }

    .btn-secondary {
        background: rgba(255, 255, 255, 0.2);
        color: white;
        border: 1px solid rgba(255, 255, 255, 0.3);
    }

    .btn-secondary:hover {
        background: rgba(255, 255, 255, 0.3);
    }

    /* Modern Statistics Grid */
    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
        gap: 1.5rem;
        margin-bottom: 2rem;
        max-width: 1690px;
        margin-left: auto;
        margin-right: auto;
        padding: 0 1.5rem;
    }

    .stat-card {
        background: white;
        border-radius: var(--border-radius);
        padding: 2rem;
        box-shadow: var(--card-shadow);
        transition: var(--transition);
        position: relative;
        overflow: hidden;
    }

    .stat-card:hover {
        transform: translateY(-2px);
        box-shadow: var(--card-shadow-hover);
    }

    .stat-card::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 4px;
        background: var(--accent-color);
    }

    .stat-card.primary::before { background: var(--primary-color); }
    .stat-card.success::before { background: var(--success-color); }
    .stat-card.warning::before { background: var(--warning-color); }
    .stat-card.info::before { background: var(--info-color); }

    .stat-header {
        display: flex;
        justify-content: space-between;
        align-items: flex-start;
        margin-bottom: 1rem;
    }

    .stat-icon {
        width: 3rem;
        height: 3rem;
        border-radius: 12px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 1.25rem;
        color: white;
    }

    .stat-icon.primary { background: var(--primary-color); }
    .stat-icon.success { background: var(--success-color); }
    .stat-icon.warning { background: var(--warning-color); }
    .stat-icon.info { background: var(--info-color); }

    .stat-value {
        font-size: 2.5rem;
        font-weight: 800;
        color: #1f2937;
        margin-bottom: 0.5rem;
        line-height: 1;
    }

    .stat-label {
        color: var(--secondary-color);
        font-size: 0.9rem;
        font-weight: 500;
        margin-bottom: 1rem;
    }

    .stat-badge {
        display: inline-flex;
        align-items: center;
        gap: 0.25rem;
        padding: 0.5rem 1rem;
        border-radius: 50px;
        font-size: 0.8rem;
        font-weight: 600;
    }

    .stat-badge.success {
        background: rgba(16, 185, 129, 0.1);
        color: var(--success-color);
    }

    .stat-badge.warning {
        background: rgba(245, 158, 11, 0.1);
        color: var(--warning-color);
    }

    .stat-badge.danger {
        background: rgba(239, 68, 68, 0.1);
        color: var(--danger-color);
    }

    /* Charts Section */
    .charts-section {
        max-width: 1200px;
        margin: 0 auto 2rem;
        padding: 0 1.5rem;
    }

    .charts-grid {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 1.5rem;
        margin-bottom: 1.5rem;
    }

    .chart-card {
        background: white;
        border-radius: var(--border-radius);
        padding: 1.5rem;
        box-shadow: var(--card-shadow);
        transition: var(--transition);
    }

    .chart-card:hover {
        box-shadow: var(--card-shadow-hover);
    }

    .chart-header {
        margin-bottom: 1.5rem;
    }

    .chart-title {
        font-size: 1.1rem;
        font-weight: 700;
        color: #1f2937;
        margin-bottom: 0.25rem;
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }

    .chart-subtitle {
        color: var(--secondary-color);
        font-size: 0.85rem;
    }

    .chart-container {
        height: 250px;
        position: relative;
    }

    .revenue-chart {
        grid-column: 1 / -1;
    }

    .revenue-chart .chart-container {
        height: 300px;
    }

    /* Modern Table */
    .table-section {
        max-width: 1200px;
        margin: 0 auto 2rem;
        padding: 0 1.5rem;
    }

    .table-card {
        background: white;
        border-radius: var(--border-radius);
        box-shadow: var(--card-shadow);
        overflow: hidden;
    }

    .table-header {
        padding: 1.5rem;
        background: linear-gradient(90deg, #f8fafc 0%, #f1f5f9 100%);
        border-bottom: 1px solid #e2e8f0;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }

    .table-title {
        font-size: 1.2rem;
        font-weight: 700;
        color: #1f2937;
        margin: 0;
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }

    .table-actions {
        display: flex;
        gap: 0.75rem;
    }

    .modern-table {
        width: 100%;
        border-collapse: collapse;
    }

    .modern-table th {
        background: #f8fafc;
        padding: 1rem 1.5rem;
        text-align: right;
        font-weight: 600;
        color: #374151;
        border-bottom: 2px solid #e5e7eb;
        font-size: 0.9rem;
    }

    .modern-table td {
        padding: 1rem 1.5rem;
        border-bottom: 1px solid #f3f4f6;
        vertical-align: middle;
    }

    .modern-table tbody tr {
        transition: var(--transition);
    }

    .modern-table tbody tr:hover {
        background: #f9fafb;
    }

    .table-cell {
        display: flex;
        flex-direction: column;
        gap: 0.25rem;
    }

    .primary-value {
        font-weight: 600;
        color: #1f2937;
    }

    .secondary-value {
        font-size: 0.8rem;
        color: var(--secondary-color);
    }

    .progress-bar {
        width: 100%;
        height: 6px;
        background: #e5e7eb;
        border-radius: 3px;
        overflow: hidden;
        margin-bottom: 0.25rem;
    }

    .progress-fill {
        height: 100%;
        border-radius: 3px;
        transition: width 1s ease-in-out;
    }

    .progress-fill.success { background: var(--success-color); }
    .progress-fill.warning { background: var(--warning-color); }
    .progress-fill.danger { background: var(--danger-color); }

    .table-badge {
        display: inline-flex;
        align-items: center;
        gap: 0.25rem;
        padding: 0.25rem 0.75rem;
        border-radius: 50px;
        font-size: 0.8rem;
        font-weight: 600;
    }

    .table-badge.success {
        background: rgba(16, 185, 129, 0.1);
        color: var(--success-color);
    }

    .table-badge.warning {
        background: rgba(245, 158, 11, 0.1);
        color: var(--warning-color);
    }

    .table-badge.danger {
        background: rgba(239, 68, 68, 0.1);
        color: var(--danger-color);
    }

    .table-badge.primary {
        background: rgba(59, 130, 246, 0.1);
        color: var(--primary-color);
    }

    /* KPI Section */
    .kpi-section {
        max-width: 1200px;
        margin: 0 auto;
        padding: 0 1.5rem;
    }

    .kpi-grid {
        display: grid;
        grid-template-columns: 2fr 1fr;
        gap: 1.5rem;
    }

    .kpi-card {
        background: white;
        border-radius: var(--border-radius);
        padding: 1.5rem;
        box-shadow: var(--card-shadow);
    }

    .kpi-header {
        margin-bottom: 1.5rem;
    }

    .kpi-title {
        font-size: 1.1rem;
        font-weight: 700;
        color: #1f2937;
        margin-bottom: 0.25rem;
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }

    .recommendations-list {
        display: flex;
        flex-direction: column;
        gap: 1rem;
    }

    .recommendation-item {
        display: flex;
        gap: 1rem;
        padding: 1rem;
        background: #f9fafb;
        border-radius: 12px;
        border-left: 4px solid var(--primary-color);
    }

    .recommendation-icon {
        width: 2rem;
        height: 2rem;
        background: var(--primary-color);
        color: white;
        border-radius: 8px;
        display: flex;
        align-items: center;
        justify-content: center;
        flex-shrink: 0;
    }

    .recommendation-content h6 {
        margin: 0 0 0.25rem 0;
        font-weight: 600;
        color: #1f2937;
    }

    .recommendation-content p {
        margin: 0;
        color: var(--secondary-color);
        font-size: 0.9rem;
    }

    .metrics-list {
        display: flex;
        flex-direction: column;
        gap: 1rem;
    }

    .metric-item {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 1rem;
        background: #f9fafb;
        border-radius: 12px;
    }

    .metric-info h6 {
        margin: 0 0 0.25rem 0;
        font-weight: 600;
        color: #1f2937;
    }

    .metric-value {
        font-size: 1.5rem;
        font-weight: 800;
        margin: 0;
    }

    /* Animations */
    .fade-in {
        opacity: 0;
        transform: translateY(20px);
        animation: fadeIn 0.6s ease-out forwards;
    }

    @keyframes fadeIn {
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    .fade-in:nth-child(1) { animation-delay: 0.1s; }
    .fade-in:nth-child(2) { animation-delay: 0.2s; }
    .fade-in:nth-child(3) { animation-delay: 0.3s; }
    .fade-in:nth-child(4) { animation-delay: 0.4s; }

    /* Responsive Design */
    @media (max-width: 768px) {
        .header-content {
            flex-direction: column;
            gap: 1rem;
            text-align: center;
        }

        .header-actions {
            flex-wrap: wrap;
            justify-content: center;
        }

        .charts-grid {
            grid-template-columns: 1fr;
        }

        .kpi-grid {
            grid-template-columns: 1fr;
        }

        .stats-grid {
            grid-template-columns: 1fr;
        }

        .modern-table {
            font-size: 0.8rem;
        }

        .modern-table th,
        .modern-table td {
            padding: 0.75rem;
        }
    }

    /* Dropdown Styles */
    .dropdown-menu {
        border: none;
        border-radius: 12px;
        box-shadow: var(--card-shadow);
        padding: 0.5rem;
    }

    .dropdown-item {
        border-radius: 8px;
        padding: 0.75rem 1rem;
        transition: var(--transition);
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }

    .dropdown-item:hover {
        background: #f3f4f6;
    }
</style>
@endpush

@section('content')
<!-- Modern Header -->
<div class="modern-header">
    <div class="header-content">
        <h1 class="header-title">
            <i class="fas fa-chart-line"></i>
            تقرير أداء التحصيل
        </h1>
        <div class="header-actions">
            <a href="{{ route('invoices.payments-report') }}" class="btn-modern btn-secondary">
                <i class="fas fa-arrow-left"></i>
                العودة
            </a>
            
            <div class="dropdown">
                <button class="btn-modern btn-secondary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                    <i class="fas fa-calendar"></i>
                    الفترة الزمنية
                </button>
                <ul class="dropdown-menu">
                    <li><a class="dropdown-item" href="?period=6">
                        <i class="fas fa-clock"></i>آخر 6 أشهر
                    </a></li>
                    <li><a class="dropdown-item" href="?period=12">
                        <i class="fas fa-calendar-alt"></i>آخر 12 شهر
                    </a></li>
                    <li><a class="dropdown-item" href="?period=24">
                        <i class="fas fa-calendar"></i>آخر سنتين
                    </a></li>
                </ul>
            </div>
            
            <a href="{{ route('invoices.export-collection-performance', ['period' => request('period', 12)]) }}"
               class="btn-modern btn-primary">
                <i class="fas fa-download"></i>
                تصدير التقرير
            </a>
        </div>
    </div>
</div>

<!-- Statistics Cards -->
<div class="stats-grid">
    <div class="stat-card primary fade-in">
        <div class="stat-header">
            <div>
                <div class="stat-value" data-counter="{{ number_format(collect($performanceData)->avg('collection_rate'), 1) }}">
                    {{ number_format(collect($performanceData)->avg('collection_rate'), 1) }}%
                </div>
                <div class="stat-label">متوسط معدل التحصيل</div>
            </div>
            <div class="stat-icon primary">
                <i class="fas fa-percentage"></i>
            </div>
        </div>
        <div class="stat-badge @if(collect($performanceData)->avg('collection_rate') >= 90) success @elseif(collect($performanceData)->avg('collection_rate') >= 75) warning @else danger @endif">
            <i class="fas fa-@if(collect($performanceData)->avg('collection_rate') >= 90)trophy @elseif(collect($performanceData)->avg('collection_rate') >= 75)star @else exclamation @endif"></i>
            @if(collect($performanceData)->avg('collection_rate') >= 90) ممتاز @elseif(collect($performanceData)->avg('collection_rate') >= 75) جيد @else يحتاج تحسين @endif
        </div>
    </div>
    
    <div class="stat-card info fade-in">
        <div class="stat-header">
            <div>
                <div class="stat-value" data-counter="{{ number_format(collect($performanceData)->avg('avg_collection_time'), 1) }}">
                    {{ number_format(collect($performanceData)->avg('avg_collection_time'), 1) }}
                </div>
                <div class="stat-label">متوسط وقت التحصيل (يوم)</div>
            </div>
            <div class="stat-icon info">
                <i class="fas fa-clock"></i>
            </div>
        </div>
        <div class="stat-badge @if(collect($performanceData)->avg('avg_collection_time') <= 15) success @elseif(collect($performanceData)->avg('avg_collection_time') <= 30) warning @else danger @endif">
            <i class="fas fa-@if(collect($performanceData)->avg('avg_collection_time') <= 15)rocket @elseif(collect($performanceData)->avg('avg_collection_time') <= 30)walking @else turtle @endif"></i>
            @if(collect($performanceData)->avg('avg_collection_time') <= 15) سريع @elseif(collect($performanceData)->avg('avg_collection_time') <= 30) متوسط @else بطيء @endif
        </div>
    </div>
    
    <div class="stat-card success fade-in">
        <div class="stat-header">
            <div>
                <div class="stat-value" data-counter="{{ collect($performanceData)->sum('total_invoices') }}">
                    {{ number_format(collect($performanceData)->sum('total_invoices')) }}
                </div>
                <div class="stat-label">إجمالي الفواتير</div>
            </div>
            <div class="stat-icon success">
                <i class="fas fa-file-invoice-dollar"></i>
            </div>
        </div>
        <div class="stat-badge primary">
            <i class="fas fa-chart-line"></i>
            {{ number_format(collect($performanceData)->sum('total_amount'), 0) }} ر.س
        </div>
    </div>
    
    <div class="stat-card warning fade-in">
        <div class="stat-header">
            <div>
                <div class="stat-value" data-counter="{{ collect($performanceData)->avg('overdue_invoices') }}">
                    {{ number_format(collect($performanceData)->avg('overdue_invoices'), 0) }}
                </div>
                <div class="stat-label">متوسط المتأخرات الشهرية</div>
            </div>
            <div class="stat-icon warning">
                <i class="fas fa-exclamation-triangle"></i>
            </div>
        </div>
        <div class="stat-badge @if(collect($performanceData)->avg('overdue_invoices') == 0) success @elseif(collect($performanceData)->avg('overdue_invoices') < 5) warning @else danger @endif">
            <i class="fas fa-@if(collect($performanceData)->avg('overdue_invoices') == 0)check @elseif(collect($performanceData)->avg('overdue_invoices') < 5)info @else exclamation @endif"></i>
            @if(collect($performanceData)->avg('overdue_invoices') == 0) ممتاز @elseif(collect($performanceData)->avg('overdue_invoices') < 5) مقبول @else يحتاج متابعة @endif
        </div>
    </div>
</div>

<!-- Charts Section -->
<div class="">
    <div class="charts-grid">
        <div class="chart-card fade-in">
            <div class="chart-header">
                <h3 class="chart-title">
                    <i class="fas fa-chart-line"></i>
                    تطور معدل التحصيل
                </h3>
                <p class="chart-subtitle">نسبة التحصيل الشهرية</p>
            </div>
            <div class="chart-container">
                <canvas id="collectionRateChart"></canvas>
            </div>
        </div>
        
        <div class="chart-card fade-in">
            <div class="chart-header">
                <h3 class="chart-title">
                    <i class="fas fa-clock"></i>
                    متوسط وقت التحصيل
                </h3>
                <p class="chart-subtitle">الأيام من تاريخ الإصدار</p>
            </div>
            <div class="chart-container">
                <canvas id="collectionTimeChart"></canvas>
            </div>
        </div>
    </div>
    
    <div class="chart-card revenue-chart fade-in">
        <div class="chart-header">
            <h3 class="chart-title">
                <i class="fas fa-chart-bar"></i>
                الإيرادات والمحصلات الشهرية
            </h3>
            <p class="chart-subtitle">مقارنة المبالغ المستحقة والمحصلة</p>
        </div>
        <div class="chart-container">
            <canvas id="revenueChart"></canvas>
        </div>
    </div>
</div>
<br>
<!-- Data Table -->
<div class="">
    <div class="table-card fade-in">
        <div class="table-header">
            <h3 class="table-title">
                <i class="fas fa-table"></i>
                البيانات التفصيلية
            </h3>
            <div class="table-actions">
                <div class="dropdown">
                    <button class="btn-modern btn-secondary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                        <i class="fas fa-filter"></i>
                        فلتر
                    </button>
                    <ul class="dropdown-menu">
                        <li><a class="dropdown-item" href="#" onclick="filterByPerformance('all')">جميع البيانات</a></li>
                        <li><a class="dropdown-item" href="#" onclick="filterByPerformance('excellent')">الأداء الممتاز</a></li>
                        <li><a class="dropdown-item" href="#" onclick="filterByPerformance('poor')">الأداء الضعيف</a></li>
                    </ul>
                </div>
                <button class="btn-modern btn-primary" onclick="exportTableData()">
                    <i class="fas fa-download"></i>
                    تصدير
                </button>
            </div>
        </div>
        <div style="overflow-x: auto;">
            <table class="modern-table" id="performanceTable">
                <thead>
                    <tr>
                        <th>الشهر</th>
                        <th>عدد الفواتير</th>
                        <th>إجمالي المبلغ</th>
                        <th>المحصل</th>
                        <th>معدل التحصيل</th>
                        <th>المتأخرات</th>
                        <th>متوسط الوقت</th>
                        <th>التقييم</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($performanceData as $data)
                        @php
                            $performance = 'ممتاز';
                            $performanceClass = 'success';
                            
                            if ($data['collection_rate'] < 70) {
                                $performance = 'ضعيف';
                                $performanceClass = 'danger';
                            } elseif ($data['collection_rate'] < 85) {
                                $performance = 'متوسط';
                                $performanceClass = 'warning';
                            } elseif ($data['collection_rate'] < 95) {
                                $performance = 'جيد';
                                $performanceClass = 'primary';
                            }
                        @endphp
                        <tr data-performance="{{ $performanceClass }}">
                            <td>
                                <div class="table-cell">
                                    <span class="primary-value">{{ $data['month'] }}</span>
                                    <span class="secondary-value">{{ date('Y', strtotime($data['month'])) }}</span>
                                </div>
                            </td>
                            <td>
                                <div class="table-cell">
                                    <span class="table-badge primary">{{ number_format($data['total_invoices']) }}</span>
                                </div>
                            </td>
                            <td>
                                <div class="table-cell">
                                    <span class="primary-value">{{ number_format($data['total_amount'], 2) }}</span>
                                    <span class="secondary-value">ريال سعودي</span>
                                </div>
                            </td>
                            <td>
                                <div class="table-cell">
                                    <span class="primary-value" style="color: var(--success-color);">{{ number_format($data['collected_amount'], 2) }}</span>
                                    <span class="secondary-value">ريال سعودي</span>
                                </div>
                            </td>
                            <td>
                                <div class="table-cell">
                                    <div class="progress-bar">
                                        <div class="progress-fill {{ $performanceClass }}" style="width: {{ $data['collection_rate'] }}%"></div>
                                    </div>
                                    <span class="primary-value">{{ number_format($data['collection_rate'], 1) }}%</span>
                                </div>
                            </td>
                            <td>
                                <div class="table-cell">
                                    @if($data['overdue_invoices'] > 0)
                                        <span class="table-badge warning">{{ $data['overdue_invoices'] }}</span>
                                    @else
                                        <span class="table-badge success">0</span>
                                    @endif
                                </div>
                            </td>
                            <td>
                                <div class="table-cell">
                                    <span class="table-badge {{ $data['avg_collection_time'] > 30 ? 'danger' : ($data['avg_collection_time'] > 15 ? 'warning' : 'success') }}">
                                        {{ number_format($data['avg_collection_time'], 1) }} يوم
                                    </span>
                                </div>
                            </td>
                            <td>
                                <div class="table-cell">
                                    <span class="table-badge {{ $performanceClass }}">
                                        @if($performanceClass == 'success')
                                            <i class="fas fa-trophy"></i>
                                        @elseif($performanceClass == 'primary')
                                            <i class="fas fa-thumbs-up"></i>
                                        @elseif($performanceClass == 'warning')
                                            <i class="fas fa-meh"></i>
                                        @else
                                            <i class="fas fa-exclamation"></i>
                                        @endif
                                        {{ $performance }}
                                    </span>
                                </div>
                            </td>
                        </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- KPI and Recommendations -->
<div class="kpi-section">
    <div class="kpi-grid">
        <div class="kpi-card fade-in">
            <div class="kpi-header">
                <h3 class="kpi-title">
                    <i class="fas fa-lightbulb"></i>
                    توصيات لتحسين الأداء
                </h3>
                <p class="chart-subtitle">نصائح مخصصة لتطوير الأداء</p>
            </div>
            <div class="recommendations-list">
                @if(collect($performanceData)->avg('collection_rate') < 85)
                    <div class="recommendation-item">
                        <div class="recommendation-icon">
                            <i class="fas fa-bell"></i>
                        </div>
                        <div class="recommendation-content">
                            <h6>تفعيل التذكيرات التلقائية</h6>
                            <p>إرسال تذكيرات دورية للعملاء قبل وبعد تاريخ الاستحقاق</p>
                        </div>
                    </div>
                @endif
                
                @if(collect($performanceData)->avg('avg_collection_time') > 20)
                    <div class="recommendation-item">
                        <div class="recommendation-icon">
                            <i class="fas fa-calendar-minus"></i>
                        </div>
                        <div class="recommendation-content">
                            <h6>تقليل مدة الائتمان</h6>
                            <p>مراجعة شروط السداد وتقليل المدة الممنوحة للعملاء</p>
                        </div>
                    </div>
                @endif
                
                <div class="recommendation-item">
                    <div class="recommendation-icon">
                        <i class="fas fa-paper-plane"></i>
                    </div>
                    <div class="recommendation-content">
                        <h6>الفوترة الإلكترونية الفورية</h6>
                        <p>إرسال الفواتير فور إنجاز الخدمة أو تسليم المنتج</p>
                    </div>
                </div>
                
                <div class="recommendation-item">
                    <div class="recommendation-icon">
                        <i class="fas fa-percentage"></i>
                    </div>
                    <div class="recommendation-content">
                        <h6>حوافز السداد المبكر</h6>
                        <p>تقديم خصومات مجزية للعملاء الذين يسددون مبكراً</p>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="kpi-card fade-in">
            <div class="kpi-header">
                <h3 class="kpi-title">
                    <i class="fas fa-tachometer-alt"></i>
                    مؤشرات الأداء الرئيسية
                </h3>
            </div>
            <div class="metrics-list">
                @php
                    $avgRate = collect($performanceData)->avg('collection_rate');
                    $avgTime = collect($performanceData)->avg('avg_collection_time');
                    $totalOverdue = collect($performanceData)->sum('overdue_invoices');
                @endphp
                
                <div class="metric-item">
                    <div class="metric-info">
                        <h6>معدل التحصيل العام</h6>
                        <p class="metric-value" style="color: {{ $avgRate >= 90 ? 'var(--success-color)' : ($avgRate >= 75 ? 'var(--warning-color)' : 'var(--danger-color)') }};">
                            {{ number_format($avgRate, 1) }}%
                        </p>
                    </div>
                    <i class="fas fa-chart-pie" style="color: {{ $avgRate >= 90 ? 'var(--success-color)' : ($avgRate >= 75 ? 'var(--warning-color)' : 'var(--danger-color)') }};"></i>
                </div>
                
                <div class="metric-item">
                    <div class="metric-info">
                        <h6>متوسط سرعة التحصيل</h6>
                        <p class="metric-value" style="color: {{ $avgTime <= 15 ? 'var(--success-color)' : ($avgTime <= 30 ? 'var(--warning-color)' : 'var(--danger-color)') }};">
                            {{ number_format($avgTime, 1) }} يوم
                        </p>
                    </div>
                    <i class="fas fa-stopwatch" style="color: {{ $avgTime <= 15 ? 'var(--success-color)' : ($avgTime <= 30 ? 'var(--warning-color)' : 'var(--danger-color)') }};"></i>
                </div>
                
                <div class="metric-item">
                    <div class="metric-info">
                        <h6>إجمالي المتأخرات</h6>
                        <p class="metric-value" style="color: {{ $totalOverdue == 0 ? 'var(--success-color)' : ($totalOverdue < 10 ? 'var(--warning-color)' : 'var(--danger-color)') }};">
                            {{ $totalOverdue }}
                        </p>
                    </div>
                    <i class="fas fa-exclamation-triangle" style="color: {{ $totalOverdue == 0 ? 'var(--success-color)' : ($totalOverdue < 10 ? 'var(--warning-color)' : 'var(--danger-color)') }};"></i>
                </div>
            </div>
        </div>
    </div>
</div>

@endsection

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Counter animation
    const animateCounter = (element) => {
        const target = parseFloat(element.dataset.counter);
        const duration = 2000;
        const increment = target / (duration / 16);
        let current = 0;
        
        const timer = setInterval(() => {
            current += increment;
            if (current >= target) {
                current = target;
                clearInterval(timer);
            }
            
            if (element.textContent.includes('%')) {
                element.textContent = current.toFixed(1) + '%';
            } else {
                element.textContent = Math.floor(current).toLocaleString('ar-SA');
            }
        }, 16);
    };
    
    // Animate progress bars
    setTimeout(() => {
        document.querySelectorAll('.progress-fill').forEach(bar => {
            const width = bar.style.width;
            bar.style.width = '0%';
            setTimeout(() => {
                bar.style.width = width;
            }, 100);
        });
    }, 500);
    
    // Charts configuration
    const performanceData = @json($performanceData);
    
    const commonOptions = {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: { display: false },
            tooltip: {
                backgroundColor: 'rgba(0, 0, 0, 0.9)',
                titleColor: 'white',
                bodyColor: 'white',
                cornerRadius: 8,
                padding: 12
            }
        },
        scales: {
            x: {
                grid: { display: false },
                ticks: { font: { family: 'Cairo', size: 11 }, color: '#6b7280' }
            },
            y: {
                grid: { color: 'rgba(0, 0, 0, 0.05)' },
                ticks: { font: { family: 'Cairo', size: 11 }, color: '#6b7280' }
            }
        },
        animation: { duration: 1500, easing: 'easeInOutCubic' }
    };
    
    // Collection Rate Chart
    setTimeout(() => {
        const ctx1 = document.getElementById('collectionRateChart').getContext('2d');
        new Chart(ctx1, {
            type: 'line',
            data: {
                labels: performanceData.map(d => d.month),
                datasets: [{
                    data: performanceData.map(d => d.collection_rate),
                    borderColor: '#3b82f6',
                    backgroundColor: 'rgba(59, 130, 246, 0.1)',
                    borderWidth: 3,
                    tension: 0.4,
                    pointBackgroundColor: '#3b82f6',
                    pointBorderColor: '#ffffff',
                    pointBorderWidth: 2,
                    pointRadius: 4,
                    fill: true
                }]
            },
            options: {
                ...commonOptions,
                scales: {
                    ...commonOptions.scales,
                    y: {
                        ...commonOptions.scales.y,
                        max: 100,
                        ticks: {
                            ...commonOptions.scales.y.ticks,
                            callback: value => value + '%'
                        }
                    }
                }
            }
        });
    }, 200);
    
    // Collection Time Chart
    setTimeout(() => {
        const ctx2 = document.getElementById('collectionTimeChart').getContext('2d');
        new Chart(ctx2, {
            type: 'bar',
            data: {
                labels: performanceData.map(d => d.month),
                datasets: [{
                    data: performanceData.map(d => d.avg_collection_time),
                    backgroundColor: performanceData.map(d => {
                        if (d.avg_collection_time <= 15) return '#10b981';
                        if (d.avg_collection_time <= 30) return '#f59e0b';
                        return '#ef4444';
                    }),
                    borderRadius: 6
                }]
            },
            options: commonOptions
        });
    }, 400);
    
    // Revenue Chart
    setTimeout(() => {
        const ctx3 = document.getElementById('revenueChart').getContext('2d');
        new Chart(ctx3, {
            type: 'bar',
            data: {
                labels: performanceData.map(d => d.month),
                datasets: [{
                    label: 'إجمالي المبلغ',
                    data: performanceData.map(d => d.total_amount),
                    backgroundColor: '#3b82f6',
                    borderRadius: 6
                }, {
                    label: 'المحصل فعلياً',
                    data: performanceData.map(d => d.collected_amount),
                    backgroundColor: '#10b981',
                    borderRadius: 6
                }]
            },
            options: {
                ...commonOptions,
                plugins: {
                    ...commonOptions.plugins,
                    legend: {
                        display: true,
                        position: 'top',
                        align: 'end',
                        labels: {
                            font: { family: 'Cairo', size: 12 },
                            color: '#6b7280',
                            usePointStyle: true
                        }
                    }
                }
            }
        });
    }, 600);
    
    // Initialize counters
    document.querySelectorAll('[data-counter]').forEach(animateCounter);
    
    // Table functions
    window.filterByPerformance = function(filterType) {
        const rows = document.querySelectorAll('#performanceTable tbody tr');
        rows.forEach(row => {
            const performance = row.dataset.performance;
            let shouldShow = true;
            
            switch(filterType) {
                case 'excellent':
                    shouldShow = performance === 'success';
                    break;
                case 'poor':
                    shouldShow = performance === 'danger';
                    break;
                default:
                    shouldShow = true;
            }
            
            row.style.display = shouldShow ? '' : 'none';
        });
    };

    window.exportTableData = function() {
        const btn = event.target.closest('button');
        const originalText = btn.innerHTML;
        btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري التصدير...';
        btn.disabled = true;
        
        setTimeout(() => {
            btn.innerHTML = '<i class="fas fa-check"></i> تم التصدير';
            setTimeout(() => {
                btn.innerHTML = originalText;
                btn.disabled = false;
            }, 2000);
        }, 1500);
    };
});
</script>
@endpush