@extends('layouts.app')

@section('title', 'تقرير حركة الصندوق للفواتير')
@section('page-title', 'حركة الصندوق')

@push('styles')
<style>
:root {
    --primary: #4f46e5;
    --success: #10b981;
    --warning: #f59e0b;
    --danger: #ef4444;
    --info: #06b6d4;
    --gray-50: #f9fafb;
    --gray-100: #f3f4f6;
    --gray-200: #e5e7eb;
    --gray-300: #d1d5db;
    --gray-500: #6b7280;
    --gray-600: #4b5563;
    --gray-700: #374151;
    --gray-800: #1f2937;
    --gray-900: #111827;
    --border-radius: 12px;
    --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
    --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
    --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
}

.page-container {
    padding: 2rem;
    max-width: 1400px;
    margin: 0 auto;
}

.page-header {
    background: linear-gradient(135deg, var(--success) 0%, #34d399 100%);
    border-radius: var(--border-radius);
    padding: 2.5rem;
    margin-bottom: 2rem;
    color: white;
    position: relative;
    overflow: hidden;
}

.page-header::before {
    content: '';
    position: absolute;
    top: 0;
    right: 0;
    width: 200px;
    height: 200px;
    background: rgba(255, 255, 255, 0.1);
    border-radius: 50%;
    transform: translate(50px, -50px);
}

.header-content {
    position: relative;
    z-index: 1;
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 2rem;
}

.header-title h1 {
    font-size: 2.5rem;
    font-weight: 800;
    margin: 0 0 0.5rem 0;
    text-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.header-title p {
    font-size: 1.1rem;
    opacity: 0.9;
    margin: 0;
    font-weight: 300;
}

.header-actions {
    display: flex;
    gap: 0.75rem;
    flex-wrap: wrap;
}

.action-btn {
    background: rgba(255, 255, 255, 0.15);
    color: white;
    padding: 0.75rem 1.25rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    border: 1px solid rgba(255, 255, 255, 0.2);
    backdrop-filter: blur(10px);
    font-size: 0.875rem;
}

.action-btn:hover {
    background: rgba(255, 255, 255, 0.25);
    transform: translateY(-1px);
    color: white;
    text-decoration: none;
}

.action-btn.print-btn {
    background: rgba(59, 130, 246, 0.2);
    border-color: rgba(59, 130, 246, 0.3);
}

.action-btn.excel-btn {
    background: rgba(34, 197, 94, 0.2);
    border-color: rgba(34, 197, 94, 0.3);
}

.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.stat-card {
    background: white;
    border-radius: var(--border-radius);
    padding: 2rem;
    border: 1px solid var(--gray-200);
    transition: all 0.3s ease;
    position: relative;
}

.stat-card:hover {
    transform: translateY(-4px);
    box-shadow: var(--shadow-lg);
    border-color: var(--gray-300);
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    border-radius: var(--border-radius) var(--border-radius) 0 0;
    background: var(--success);
}

.stat-card.warning::before { background: var(--warning); }
.stat-card.info::before { background: var(--info); }
.stat-card.primary::before { background: var(--primary); }

.stat-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
}

.stat-icon {
    width: 48px;
    height: 48px;
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    background: var(--success);
    color: white;
}

.stat-card.warning .stat-icon { background: var(--warning); }
.stat-card.info .stat-icon { background: var(--info); }
.stat-card.primary .stat-icon { background: var(--primary); }

.stat-value {
    font-size: 2rem;
    font-weight: 800;
    color: var(--gray-900);
    margin-bottom: 0.5rem;
}

.stat-label {
    font-size: 1rem;
    font-weight: 600;
    color: var(--gray-600);
    margin-bottom: 0.25rem;
}

.content-card {
    background: white;
    border-radius: var(--border-radius);
    border: 1px solid var(--gray-200);
    overflow: hidden;
    box-shadow: var(--shadow-sm);
    margin-bottom: 2rem;
}

.content-header {
    background: var(--gray-50);
    padding: 2rem;
    border-bottom: 1px solid var(--gray-200);
}

.content-header-main {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1.5rem;
    flex-wrap: wrap;
    gap: 1rem;
}

.content-title h2 {
    font-size: 1.5rem;
    font-weight: 700;
    color: var(--gray-900);
    margin: 0 0 0.25rem 0;
}

.btn-primary {
    background: var(--primary);
    color: white;
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    border: none;
    cursor: pointer;
}

.btn-primary:hover {
    background: #3730a3;
    transform: translateY(-1px);
    color: white;
}

.btn-secondary {
    background: white;
    border: 1px solid var(--gray-300);
    color: var(--gray-700);
    padding: 0.75rem 1rem;
    border-radius: 8px;
    font-size: 0.875rem;
    font-weight: 500;
    transition: all 0.2s ease;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
}

.btn-secondary:hover {
    background: var(--gray-50);
    border-color: var(--gray-400);
    color: var(--gray-800);
}

.filters-section {
    padding: 2rem;
    background: var(--gray-50);
    border-bottom: 1px solid var(--gray-200);
}

.filters-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
    margin-bottom: 1rem;
}

.filter-group label {
    display: block;
    font-size: 0.875rem;
    font-weight: 600;
    color: var(--gray-700);
    margin-bottom: 0.5rem;
}

.filter-input {
    width: 100%;
    padding: 0.75rem;
    border: 1px solid var(--gray-300);
    border-radius: 8px;
    font-size: 0.875rem;
    transition: all 0.2s ease;
    background: white;
}

.filter-input:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
}

.table-container {
    overflow-x: auto;
}

.data-table {
    width: 100%;
    border-collapse: collapse;
}

.data-table th {
    background: var(--gray-50);
    padding: 1rem 1.5rem;
    text-align: right;
    font-weight: 600;
    font-size: 0.875rem;
    color: #000000;
    border-bottom: 1px solid var(--gray-200);
    white-space: nowrap;
}

.data-table td {
    padding: 1.25rem 1.5rem;
    border-bottom: 1px solid var(--gray-100);
    vertical-align: middle;
    color: #000000;
    font-family: 'Arial', sans-serif;
}

.data-table tbody tr:hover {
    background-color: var(--gray-50);
}

.amount-positive {
    color: var(--success);
    font-weight: 600;
    font-family: 'Arial', sans-serif;
    direction: ltr;
    text-align: left;
}

.amount-negative {
    color: var(--danger);
    font-weight: 600;
    font-family: 'Arial', sans-serif;
    direction: ltr;
    text-align: left;
}

.amount-neutral {
    color: #000000;
    font-weight: 600;
    font-family: 'Arial', sans-serif;
    direction: ltr;
    text-align: left;
}

/* تنسيق الأرقام باللغة الإنجليزية */
.number, .stat-value {
    font-family: 'Arial', sans-serif !important;
    direction: ltr !important;
    color: #000000;
    unicode-bidi: bidi-override;
}

/* تأكيد استخدام الأرقام الإنجليزية */
.amount-positive, .amount-negative, .amount-neutral {
    font-family: 'Arial', sans-serif !important;
    direction: ltr !important;
    unicode-bidi: bidi-override;
}

/* تنسيق جميع الأرقام في الجداول */
.data-table td.number,
.data-table td .number,
.data-table .amount-positive,
.data-table .amount-negative,
.data-table .amount-neutral {
    font-family: 'Arial', sans-serif !important;
    direction: ltr !important;
    unicode-bidi: bidi-override;
    font-variant-numeric: lining-nums;
}

.stat-label {
    color: #000000;
}

/* تحسين نصوص الجداول */
.data-table tbody tr td {
    color: #000000;
}

.data-table tfoot tr td {
    color: #000000;
    font-weight: 600;
}

/* Print Styles - Professional & Simplified */
@media print {
    /* إعدادات الصفحة */
    @page {
        margin: 1.5cm;
        size: A4 portrait;
    }

    /* إخفاء عناصر التفاعل */
    nav, .navbar, .sidebar, .header-actions, .filters-section,
    .btn-primary, .btn-secondary, .action-btn, .filter-input,
    form, button, .breadcrumb, .main-header, .main-sidebar,
    .control-sidebar, .content-header-main .btn-primary {
        display: none !important;
    }

    /* تنسيق أساسي للطباعة */
    body {
        font-family: 'Arial', sans-serif !important;
        font-size: 11px !important;
        line-height: 1.3 !important;
        color: #000 !important;
        background: white !important;
        margin: 0 !important;
        padding: 0 !important;
        direction: rtl !important;
    }
    
    .page-container {
        max-width: 100% !important;
        margin: 0 !important;
        padding: 0 !important;
        direction: rtl !important;
    }

    /* معلومات الشركة - تنسيق مبسط */
    .company-info.print-only {
        display: block !important;
        margin-bottom: 25px !important;
        padding: 15px !important;
        border: 2px solid #000 !important;
        background: white !important;
        text-align: center !important;
        page-break-inside: avoid !important;
    }

    .company-info.print-only h1 {
        font-size: 20px !important;
        font-weight: bold !important;
        margin: 8px 0 !important;
        color: #000 !important;
    }

    .company-info.print-only h2 {
        font-size: 14px !important;
        color: #666 !important;
        margin: 5px 0 !important;
    }

    /* رأس التقرير - مبسط */
    .page-header {
        background: none !important;
        color: #000 !important;
        border: 1px solid #000 !important;
        margin-bottom: 20px !important;
        padding: 15px !important;
        text-align: center !important;
        page-break-inside: avoid !important;
    }
    
    .page-header::before {
        display: none !important;
    }
    
    .header-title h1 {
        font-size: 18px !important;
        color: #000 !important;
        margin: 0 0 8px 0 !important;
        font-weight: bold !important;
    }
    
    .header-title p {
        font-size: 12px !important;
        color: #666 !important;
        margin: 0 !important;
    }

    .print-period {
        display: block !important;
        font-size: 11px !important;
        color: #000 !important;
        margin-top: 10px !important;
        padding-top: 8px !important;
        border-top: 1px solid #ddd !important;
    }

    /* بطاقات الإحصائيات - تخطيط مبسط */
    .stats-grid {
        display: table !important;
        width: 100% !important;
        margin-bottom: 20px !important;
        border-collapse: collapse !important;
        page-break-inside: avoid !important;
    }
    
    .stat-card {
        display: table-cell !important;
        width: 25% !important;
        border: 1px solid #333 !important;
        padding: 10px !important;
        text-align: center !important;
        vertical-align: middle !important;
        background: #f8f8f8 !important;
    }
    
    .stat-card::before {
        display: none !important;
    }
    
    .stat-icon {
        display: none !important;
    }
    
    .stat-value {
        font-size: 14px !important;
        font-weight: bold !important;
        color: #000 !important;
        margin-bottom: 5px !important;
        font-family: 'Arial', sans-serif !important;
        direction: ltr !important;
    }
    
    .stat-label {
        font-size: 10px !important;
        color: #000 !important;
        font-weight: normal !important;
    }

    /* بطاقات المحتوى - تنسيق مبسط */
    .content-card {
        border: 1px solid #333 !important;
        margin-bottom: 20px !important;
        page-break-inside: avoid !important;
        background: white !important;
    }
    
    .content-header {
        background: #f0f0f0 !important;
        border-bottom: 1px solid #333 !important;
        padding: 10px !important;
    }
    
    .content-title h2 {
        font-size: 14px !important;
        color: #000 !important;
        margin: 0 !important;
        font-weight: bold !important;
        text-align: center !important;
    }

    /* الجداول - تنسيق احترافي مبسط */
    .data-table {
        width: 100% !important;
        border-collapse: collapse !important;
        font-size: 10px !important;
        border: 1px solid #333 !important;
    }
    
    .data-table th {
        background: #e8e8e8 !important;
        border: 1px solid #333 !important;
        padding: 6px 4px !important;
        font-weight: bold !important;
        color: #000 !important;
        text-align: center !important;
        font-size: 10px !important;
    }
    
    .data-table td {
        border: 1px solid #333 !important;
        padding: 5px 4px !important;
        text-align: center !important;
        color: #000 !important;
        font-size: 10px !important;
    }

    .data-table tfoot td {
        background: #f0f0f0 !important;
        font-weight: bold !important;
    }

    /* الأرقام والمبالغ - تنسيق موحد */
    .amount-positive, .amount-negative, .amount-neutral, .number {
        font-family: 'Arial', sans-serif !important;
        direction: ltr !important;
        font-weight: bold !important;
        color: #000 !important;
    }

    /* تاريخ الطباعة */
    .print-date {
        display: block !important;
        text-align: center !important;
        font-size: 9px !important;
        color: #666 !important;
        margin-top: 15px !important;
        padding-top: 8px !important;
        border-top: 1px solid #ddd !important;
    }

    /* إزالة جميع التأثيرات الإضافية */
    * {
        box-shadow: none !important;
        text-shadow: none !important;
        transform: none !important;
        transition: none !important;
        -webkit-print-color-adjust: exact !important;
        color-adjust: exact !important;
    }

    /* تجنب تقسيم العناصر المهمة */
    .stat-card, .content-card, .stats-grid {
        page-break-inside: avoid !important;
    }

    /* تحسين المسافات */
    .table-container {
        margin: 0 !important;
        padding: 0 !important;
    }

    /* إخفاء عناصر الفلترة */
    .content-card:has(.filters-section) {
        display: none !important;
    }
}

/* تنسيق عام للأرقام الإنجليزية */
* {
    font-variant-numeric: lining-nums;
}

/* تأكيد استخدام الأرقام الإنجليزية في جميع العناصر */
body * {
    font-feature-settings: "lnum" 1;
}

/* تنسيق خاص للنصوص التي تحتوي على أرقام */
[class*="amount"], [class*="number"], .stat-value {
    font-family: 'Arial', sans-serif !important;
    direction: ltr !important;
    unicode-bidi: bidi-override !important;
    font-variant-numeric: lining-nums !important;
}

/* تحسينات إضافية للتفاعل */
.action-btn:active {
    transform: translateY(0);
    box-shadow: inset 0 2px 4px rgba(0, 0, 0, 0.1);
}

.stat-card:active {
    transform: translateY(-2px);
}

/* تأثيرات التحميل */
.loading {
    pointer-events: none;
    opacity: 0.6;
}

.loading::after {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    width: 20px;
    height: 20px;
    margin: -10px 0 0 -10px;
    border: 2px solid transparent;
    border-top: 2px solid #4f46e5;
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

/* تحسينات للوصولية */
.action-btn:focus {
    outline: 2px solid #4f46e5;
    outline-offset: 2px;
}

.filter-input:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
}

/* رسائل التأكيد */
.success-message {
    background: #d1fae5;
    border: 1px solid #a7f3d0;
    color: #065f46;
    padding: 12px 16px;
    border-radius: 8px;
    margin: 16px 0;
    font-size: 14px;
}

.error-message {
    background: #fee2e2;
    border: 1px solid #fecaca;
    color: #991b1b;
    padding: 12px 16px;
    border-radius: 8px;
    margin: 16px 0;
    font-size: 14px;
}

@media (max-width: 768px) {
    .page-container { padding: 1rem; }
    .page-header { padding: 1.5rem; }
    .header-title h1 { font-size: 2rem; }
    .stats-grid { grid-template-columns: 1fr; }
    .filters-grid { grid-template-columns: 1fr; }
    .header-actions {
        width: 100%;
        justify-content: center;
    }
    .action-btn {
        flex: 1;
        justify-content: center;
        min-width: 120px;
    }
}
</style>
@endpush

@section('content')
<div class="">
    <!-- معلومات الشركة للطباعة -->
    <div class="company-info print-only" style="display: none;">
        <div style="margin: 0 auto; padding: 20px; border: 2px solid #000; background: white; max-width: 800px; text-align: center;">
            @php
                $company = \App\Models\Company::first();
            @endphp
            
            @if($company && $company->hasLogo())
                <div style="margin-bottom: 15px;">
                    <img src="{{ $company->logo_url }}"
                         alt="شعار الشركة"
                         style="max-height: 80px;">
                </div>
            @endif
            
            <h1 style="margin: 10px 0; font-size: 24px; color: #000; font-weight: bold; text-align: center;">
                {{ $company->name ?? 'اسم الشركة' }}
            </h1>
            
            @if($company && $company->name_en)
                <h2 style="margin: 5px 0; font-size: 16px; color: #555; font-style: italic; text-align: center;">
                    {{ $company->name_en }}
                </h2>
            @endif
            
            @if($company)
                <!-- صف واحد للمعلومات الأساسية -->
                <div style="margin: 15px 0; display: flex; justify-content: center; flex-wrap: wrap; gap: 20px; text-align: center;">
                    @if($company->formatted_vat_numbers && $company->formatted_vat_numbers !== 'غير محدد')
                        <span style="font-size: 12px; color: #333;">
                            <strong>الرقم الضريبي:</strong> {{ $company->formatted_vat_numbers }}
                        </span>
                    @endif
                    
                    @if($company->cr_number)
                        <span style="font-size: 12px; color: #333;">
                            <strong>السجل التجاري:</strong> {{ $company->cr_number }}
                        </span>
                    @endif
                    
                    @if($company->phone)
                        <span style="font-size: 12px; color: #333;">
                            <strong>هاتف:</strong> {{ $company->phone }}
                        </span>
                    @endif
                </div>
                
                <!-- العنوان في سطر منفصل -->
                @if($company->full_address)
                    <div style="margin: 10px 0; text-align: center;">
                        <p style="margin: 0; font-size: 12px; color: #333; line-height: 1.4;">
                            <strong>العنوان:</strong> {{ $company->full_address }}
                        </p>
                    </div>
                @endif
                
                @if($company->email || $company->website)
                    <div style="margin: 10px 0; text-align: center;">
                        @if($company->email)
                            <span style="font-size: 12px; color: #333; margin-left: 15px;">
                                <strong>البريد:</strong> {{ $company->email }}
                            </span>
                        @endif
                        @if($company->website)
                            <span style="font-size: 12px; color: #333;">
                                <strong>الموقع:</strong> {{ $company->website }}
                            </span>
                        @endif
                    </div>
                @endif
            @else
                <!-- رسالة في حالة عدم وجود بيانات الشركة -->
                <div style="margin: 15px 0; padding: 15px; background: #fff3cd; border: 1px solid #ffeaa7; border-radius: 5px;">
                    <p style="margin: 0; font-size: 14px; color: #856404; text-align: center;">
                        <strong>تنبيه:</strong> لم يتم العثور على بيانات الشركة. يرجى إضافة بيانات الشركة من الإعدادات.
                    </p>
                </div>
            @endif
            
            <!-- خط فاصل -->
            <hr style="margin: 20px 0; border: 1px solid #ddd;">
        </div>
    </div>

    <!-- Page Header -->
    <div class="page-header">
        <div class="header-content">
            <div class="header-title">
                <h1>تقرير حركة الصندوق</h1>
                <p>تتبع وتحليل المقبوضات النقدية من الفواتير</p>
                <div class="print-period" style="margin-top: 15px; font-size: 14px; color: #666;">
                    <strong>فترة التقرير:</strong> من {{ $startDate }} إلى {{ $endDate }}
                    @if($paymentMethod)
                        <br><strong>طريقة الدفع:</strong> {{ $paymentMethod }}
                    @endif
                </div>
            </div>
            <div class="header-actions">
                <button onclick="printReport()" class="action-btn print-btn">
                    <i class="fas fa-print"></i>
                    طباعة التقرير
                </button>
                <a href="{{ route('invoices.export-cash-flow-excel', request()->all()) }}" class="action-btn excel-btn">
                    <i class="fas fa-file-excel"></i>
                    تصدير Excel
                </a>
                <a href="{{ route('invoices.export-cash-flow-report', request()->all()) }}" class="action-btn">
                    <i class="fas fa-file-csv"></i>
                    تصدير CSV
                </a>
                <a href="{{ route('invoices.index') }}" class="action-btn">
                    <i class="fas fa-arrow-left"></i>
                    العودة للفواتير
                </a>
            </div>
        </div>
    </div>

    <!-- Statistics Cards -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-header">
                <div class="stat-icon">
                    <i class="fas fa-file-invoice"></i>
                </div>
            </div>
            <div class="stat-value number">{{ number_format($cashFlowData['total_invoices'], 0, '.', ',') }}</div>
            <div class="stat-label">إجمالي الفواتير</div>
        </div>

        <div class="stat-card warning">
            <div class="stat-header">
                <div class="stat-icon">
                    <i class="fas fa-coins"></i>
                </div>
            </div>
            <div class="stat-value number">{{ number_format($cashFlowData['total_amount'], 2, '.', ',') }}</div>
            <div class="stat-label">إجمالي المبلغ (ر.س)</div>
        </div>

        <div class="stat-card">
            <div class="stat-header">
                <div class="stat-icon">
                    <i class="fas fa-money-bill-wave"></i>
                </div>
            </div>
            <div class="stat-value number">{{ number_format($cashFlowData['cash_received'], 2, '.', ',') }}</div>
            <div class="stat-label">المقبوضات النقدية (ر.س)</div>
        </div>

        <div class="stat-card info">
            <div class="stat-header">
                <div class="stat-icon">
                    <i class="fas fa-clock"></i>
                </div>
            </div>
            <div class="stat-value number">{{ number_format($cashFlowData['pending_amount'], 2, '.', ',') }}</div>
            <div class="stat-label">المبالغ المعلقة (ر.س)</div>
        </div>
    </div>

    <!-- Filters -->
    <div class="content-card">
        <div class="content-header">
            <div class="content-header-main">
                <div class="content-title">
                    <h2>فلترة البيانات</h2>
                </div>
            </div>
        </div>

        <div class="filters-section">
            <form method="GET" action="{{ route('invoices.cash-flow-report') }}">
                <div class="filters-grid">
                    <div class="filter-group">
                        <label>من تاريخ</label>
                        <input type="date" name="start_date" class="filter-input" value="{{ $startDate }}">
                    </div>

                    <div class="filter-group">
                        <label>إلى تاريخ</label>
                        <input type="date" name="end_date" class="filter-input" value="{{ $endDate }}">
                    </div>

                    <div class="filter-group">
                        <label>طريقة الدفع</label>
                        <select name="payment_method" class="filter-input">
                            <option value="">جميع الطرق</option>
                            <option value="cash" {{ $paymentMethod == 'cash' ? 'selected' : '' }}>نقداً</option>
                            <option value="bank_transfer" {{ $paymentMethod == 'bank_transfer' ? 'selected' : '' }}>تحويل بنكي</option>
                            <option value="credit_card" {{ $paymentMethod == 'credit_card' ? 'selected' : '' }}>بطاقة ائتمان</option>
                            <option value="check" {{ $paymentMethod == 'check' ? 'selected' : '' }}>شيك</option>
                        </select>
                    </div>

                    <div class="filter-group" style="display: flex; align-items: end;">
                        <button type="submit" class="btn-primary" style="width: 100%;">
                            <i class="fas fa-filter"></i>
                            تطبيق الفلاتر
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Payment Methods Breakdown -->
    <div class="content-card">
        <div class="content-header">
            <div class="content-header-main">
                <div class="content-title">
                    <h2>توزيع طرق الدفع</h2>
                </div>
            </div>
        </div>

        <div class="table-container">
            <table class="data-table">
                <thead>
                    <tr>
                        <th>طريقة الدفع</th>
                        <th>المبلغ المحصل</th>
                        <th>النسبة المئوية</th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td><i class="fas fa-money-bill-wave text-success"></i> نقداً</td>
                        <td class="amount-positive">{{ number_format($cashFlowData['cash_received'], 2, '.', ',') }} ر.س</td>
                        <td class="number">{{ $cashFlowData['total_received'] > 0 ? number_format(($cashFlowData['cash_received'] / $cashFlowData['total_received']) * 100, 1, '.', ',') : '0' }}%</td>
                    </tr>
                    <tr>
                        <td><i class="fas fa-university text-primary"></i> تحويل بنكي</td>
                        <td class="amount-positive">{{ number_format($cashFlowData['bank_transfers'], 2, '.', ',') }} ر.س</td>
                        <td class="number">{{ $cashFlowData['total_received'] > 0 ? number_format(($cashFlowData['bank_transfers'] / $cashFlowData['total_received']) * 100, 1, '.', ',') : '0' }}%</td>
                    </tr>
                    <tr>
                        <td><i class="fas fa-credit-card text-info"></i> بطاقة ائتمان</td>
                        <td class="amount-positive">{{ number_format($cashFlowData['card_payments'], 2, '.', ',') }} ر.س</td>
                        <td class="number">{{ $cashFlowData['total_received'] > 0 ? number_format(($cashFlowData['card_payments'] / $cashFlowData['total_received']) * 100, 1, '.', ',') : '0' }}%</td>
                    </tr>
                    <tr>
                        <td><i class="fas fa-money-check text-warning"></i> شيك</td>
                        <td class="amount-positive">{{ number_format($cashFlowData['check_payments'], 2, '.', ',') }} ر.س</td>
                        <td class="number">{{ $cashFlowData['total_received'] > 0 ? number_format(($cashFlowData['check_payments'] / $cashFlowData['total_received']) * 100, 1, '.', ',') : '0' }}%</td>
                    </tr>
                </tbody>
                <tfoot style="background: var(--gray-50); font-weight: 600;">
                    <tr>
                        <td>الإجمالي</td>
                        <td class="amount-positive">{{ number_format($cashFlowData['total_received'], 2, '.', ',') }} ر.س</td>
                        <td class="number">100%</td>
                    </tr>
                </tfoot>
            </table>
        </div>
    </div>

    <!-- Daily Cash Flow -->
    @if($dailyCashFlow->count() > 0)
    <div class="content-card">
        <div class="content-header">
            <div class="content-header-main">
                <div class="content-title">
                    <h2>الحركة النقدية اليومية</h2>
                </div>
            </div>
        </div>

        <div class="table-container">
            <table class="data-table">
                <thead>
                    <tr>
                        <th>التاريخ</th>
                        <th>عدد الفواتير</th>
                        <th>إجمالي المبلغ</th>
                        <th>المقبوضات النقدية</th>
                        <th>مقبوضات أخرى</th>
                        <th>المبلغ المعلق</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($dailyCashFlow as $day)
                    <tr>
                        <td>{{ \Carbon\Carbon::parse($day['date'])->format('Y-m-d') }}</td>
                        <td class="number">{{ number_format($day['invoices_count'], 0, '.', ',') }}</td>
                        <td class="amount-neutral">{{ number_format($day['total_amount'], 2, '.', ',') }} ر.س</td>
                        <td class="amount-positive">{{ number_format($day['cash_received'], 2, '.', ',') }} ر.س</td>
                        <td class="amount-positive">{{ number_format($day['other_received'], 2, '.', ',') }} ر.س</td>
                        <td class="amount-negative">{{ number_format($day['pending'], 2, '.', ',') }} ر.س</td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
    </div>
    @endif

    <!-- Top Cash Customers -->
    @if($topCashCustomers->count() > 0)
    <div class="content-card">
        <div class="content-header">
            <div class="content-header-main">
                <div class="content-title">
                    <h2>أفضل العملاء نقدياً</h2>
                </div>
            </div>
        </div>

        <div class="table-container">
            <table class="data-table">
                <thead>
                    <tr>
                        <th>العميل</th>
                        <th>عدد الفواتير</th>
                        <th>إجمالي المدفوع نقداً</th>
                        <th>متوسط الدفعة</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($topCashCustomers as $customer)
                    <tr>
                        <td>{{ $customer['customer']->name }}</td>
                        <td class="number">{{ number_format($customer['invoices_count'], 0, '.', ',') }}</td>
                        <td class="amount-positive">{{ number_format($customer['total_paid'], 2, '.', ',') }} ر.س</td>
                        <td class="amount-neutral">{{ number_format($customer['avg_payment'], 2, '.', ',') }} ر.س</td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
    </div>
    @endif

    <!-- Print Date Footer (hidden in normal view, shown in print) -->
    <div class="print-date number" style="display: none;">
        تاريخ الطباعة: {{ now()->format('Y-m-d H:i:s') }} |
        فترة التقرير: من {{ $startDate }} إلى {{ $endDate }}
        @if($paymentMethod)
            | طريقة الدفع: {{ $paymentMethod }}
        @endif
    </div>
</div>
@endsection

@push('scripts')
<script>
/**
 * طباعة التقرير بتنسيق احترافي مبسط
 */
function printReport() {
    // تحضير الصفحة للطباعة
    preparePrintLayout();
    
    // إظهار معلومات الشركة
    showCompanyInfoForPrint();
    
    // تحسين عرض الأرقام
    optimizeNumbersForPrint();

    // إضافة تاريخ الطباعة
    updatePrintDate();

    // حفظ العنوان الأصلي وتحديثه
    const originalTitle = document.title;
    const currentDate = new Date().toLocaleString('ar-SA', {
        year: 'numeric',
        month: '2-digit',
        day: '2-digit',
        hour: '2-digit',
        minute: '2-digit'
    });
    document.title = `تقرير حركة الصندوق - ${currentDate}`;

    // تنفيذ الطباعة
    window.print();

    // استعادة الحالة الأصلية
    setTimeout(() => {
        document.title = originalTitle;
        restoreOriginalLayout();
    }, 1000);
}

/**
 * تحضير التخطيط للطباعة
 */
function preparePrintLayout() {
    // إخفاء عناصر التفاعل
    const elementsToHide = [
        '.header-actions', '.filters-section', '.btn-primary',
        '.btn-secondary', '.action-btn', '.filter-input',
        'form', 'button', '.breadcrumb', '.navbar', 'nav'
    ];
    
    elementsToHide.forEach(selector => {
        const elements = document.querySelectorAll(selector);
        elements.forEach(element => {
            element.style.display = 'none';
            element.setAttribute('data-hidden-for-print', 'true');
        });
    });

    // إضافة فئة الطباعة
    document.body.classList.add('printing');
}

/**
 * استعادة التخطيط الأصلي
 */
function restoreOriginalLayout() {
    // إظهار العناصر المخفية
    const hiddenElements = document.querySelectorAll('[data-hidden-for-print="true"]');
    hiddenElements.forEach(element => {
        element.style.display = '';
        element.removeAttribute('data-hidden-for-print');
    });

    // إزالة فئة الطباعة
    document.body.classList.remove('printing');
    
    // إخفاء معلومات الشركة
    const companyInfo = document.querySelector('.company-info.print-only');
    if (companyInfo) {
        companyInfo.style.display = 'none';
    }

    const printDate = document.querySelector('.print-date');
    if (printDate) {
        printDate.style.display = 'none';
    }
}

/**
 * إظهار معلومات الشركة للطباعة
 */
function showCompanyInfoForPrint() {
    const companyInfo = document.querySelector('.company-info.print-only');
    if (companyInfo) {
        companyInfo.style.display = 'block';
    }
    
    const printPeriod = document.querySelector('.print-period');
    if (printPeriod) {
        printPeriod.style.display = 'block';
    }
}

/**
 * تحديث تاريخ الطباعة
 */
function updatePrintDate() {
    const printDateElement = document.querySelector('.print-date');
    if (printDateElement) {
        const now = new Date();
        const dateString = now.toLocaleString('ar-SA', {
            year: 'numeric',
            month: '2-digit',
            day: '2-digit',
            hour: '2-digit',
            minute: '2-digit'
        });
        printDateElement.style.display = 'block';
        
        // تحديث النص إذا لم يكن محدثاً
        if (!printDateElement.textContent.includes(dateString.split(' ')[0])) {
            const periodInfo = printDateElement.textContent.split('|').slice(1).join('|');
            printDateElement.textContent = `تاريخ الطباعة: ${dateString}${periodInfo}`;
        }
    }
}

/**
 * تحسين عرض الأرقام للطباعة
 */
function optimizeNumbersForPrint() {
    // تحويل الأرقام العربية إلى إنجليزية
    const numberElements = document.querySelectorAll('.number, .stat-value, .amount-positive, .amount-negative, .amount-neutral');
    
    numberElements.forEach(element => {
        let text = element.textContent;
        
        // تحويل الأرقام العربية
        const arabicNumbers = ['٠', '١', '٢', '٣', '٤', '٥', '٦', '٧', '٨', '٩'];
        const englishNumbers = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'];
        
        for (let i = 0; i < arabicNumbers.length; i++) {
            text = text.replace(new RegExp(arabicNumbers[i], 'g'), englishNumbers[i]);
        }
        
        element.textContent = text;
        
        // ضمان الاتجاه الصحيح للأرقام
        element.style.direction = 'ltr';
        element.style.fontFamily = 'Arial, sans-serif';
    });
}

/**
 * تصدير البيانات كـ Excel
 */
function exportToExcel() {
    // إضافة مؤشر التحميل
    const excelBtn = document.querySelector('.excel-btn');
    const originalText = excelBtn.innerHTML;
    
    excelBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري التصدير...';
    excelBtn.style.pointerEvents = 'none';

    // استعادة النص الأصلي بعد ثانيتين
    setTimeout(() => {
        excelBtn.innerHTML = originalText;
        excelBtn.style.pointerEvents = 'auto';
    }, 2000);
}

/**
 * تصدير البيانات كـ Excel مع مؤشر تحميل
 */
function exportToExcel() {
    const excelBtn = document.querySelector('.excel-btn');
    if (!excelBtn) return;
    
    const originalText = excelBtn.innerHTML;
    excelBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري التصدير...';
    excelBtn.style.pointerEvents = 'none';

    setTimeout(() => {
        excelBtn.innerHTML = originalText;
        excelBtn.style.pointerEvents = 'auto';
    }, 2000);
}

/**
 * معالجة أحداث الطباعة من المتصفح
 */
window.addEventListener('beforeprint', function() {
    preparePrintLayout();
    showCompanyInfoForPrint();
    optimizeNumbersForPrint();
    updatePrintDate();
});

window.addEventListener('afterprint', function() {
    restoreOriginalLayout();
});

/**
 * تحويل الأرقام العربية إلى إنجليزية
 */
function convertArabicToEnglishNumbers(text) {
    const arabicNumbers = ['٠', '١', '٢', '٣', '٤', '٥', '٦', '٧', '٨', '٩'];
    const englishNumbers = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'];
    
    let result = text;
    for (let i = 0; i < arabicNumbers.length; i++) {
        result = result.replace(new RegExp(arabicNumbers[i], 'g'), englishNumbers[i]);
    }
    return result;
}

/**
 * تحسين عرض الأرقام للشاشة والطباعة
 */
function formatNumbers() {
    const numberElements = document.querySelectorAll('.number, .stat-value, .amount-positive, .amount-negative, .amount-neutral');
    
    numberElements.forEach(element => {
        let text = element.textContent;
        text = convertArabicToEnglishNumbers(text);
        element.textContent = text;
        
        // ضمان الاتجاه الصحيح للأرقام
        element.style.direction = 'ltr';
        element.style.fontFamily = 'Arial, sans-serif';
    });
}

/**
 * إضافة التفاعلات للجداول
 */
function addTableInteractions() {
    const tableRows = document.querySelectorAll('.data-table tbody tr');
    tableRows.forEach(row => {
        row.addEventListener('mouseenter', function() {
            this.style.backgroundColor = 'rgba(79, 70, 229, 0.05)';
        });
        
        row.addEventListener('mouseleave', function() {
            this.style.backgroundColor = '';
        });
    });
}

/**
 * إعداد اختصارات لوحة المفاتيح
 */
function setupKeyboardShortcuts() {
    document.addEventListener('keydown', function(e) {
        // Ctrl+P للطباعة
        if (e.ctrlKey && e.key === 'p') {
            e.preventDefault();
            printReport();
        }
        
        // Ctrl+E لتصدير Excel
        if (e.ctrlKey && e.key === 'e') {
            e.preventDefault();
            const excelBtn = document.querySelector('.excel-btn');
            if (excelBtn) excelBtn.click();
        }
    });
}

/**
 * تهيئة التقرير عند تحميل الصفحة
 */
document.addEventListener('DOMContentLoaded', function() {
    // تحسين عرض الأرقام
    formatNumbers();
    
    // إضافة التفاعلات
    addTableInteractions();
    
    // إعداد اختصارات لوحة المفاتيح
    setupKeyboardShortcuts();
    
    // معالج زر Excel
    const excelBtn = document.querySelector('.excel-btn');
    if (excelBtn) {
        excelBtn.addEventListener('click', exportToExcel);
    }

    // تحديث الأرقام بشكل دوري
    setInterval(formatNumbers, 2000);

    console.log('تقرير حركة الصندوق - تم التحميل بنجاح');
});
</script>
@endpush