@extends('layouts.app')

@section('title', 'البحث في الفواتير')
@section('page-title', 'البحث في الفواتير')

@push('styles')
<style>
:root {
    --primary: #4f46e5;
    --success: #10b981;
    --warning: #f59e0b;
    --danger: #ef4444;
    --info: #06b6d4;
    --gray-50: #f8fafc;
    --gray-100: #f1f5f9;
    --gray-200: #e2e8f0;
    --gray-300: #cbd5e1;
    --gray-500: #64748b;
    --gray-600: #475569;
    --gray-700: #334155;
    --gray-800: #1e293b;
    --gray-900: #0f172a;
    --border-radius: 16px;
    --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
    --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
    --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
}

body {
    background: linear-gradient(135deg, var(--gray-50) 0%, #f0f4f8 100%);
    min-height: 100vh;
}

.page-container {
    padding: 1.5rem;
    max-width: 1200px;
    margin: 0 auto;
}

.page-header {
    background: linear-gradient(135deg, var(--primary) 0%, #6366f1 50%, #8b5cf6 100%);
    border-radius: var(--border-radius);
    padding: 2rem;
    margin-bottom: 2rem;
    color: white;
    position: relative;
    overflow: hidden;
    box-shadow: var(--shadow-lg);
}

.page-header::before {
    content: '';
    position: absolute;
    top: -50%;
    right: -20%;
    width: 300px;
    height: 300px;
    background: radial-gradient(circle, rgba(255, 255, 255, 0.2) 0%, transparent 70%);
    border-radius: 50%;
}

.header-content {
    position: relative;
    z-index: 2;
    text-align: center;
}

.header-title h1 {
    font-size: 2.25rem;
    font-weight: 700;
    margin: 0 0 0.5rem 0;
    text-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
}

.header-title p {
    font-size: 1rem;
    opacity: 0.9;
    margin: 0;
    font-weight: 300;
}

.back-btn {
    position: absolute;
    top: 1rem;
    left: 1rem;
    background: rgba(255, 255, 255, 0.2);
    color: white;
    padding: 0.5rem 1rem;
    border-radius: 8px;
    text-decoration: none;
    font-size: 0.875rem;
    font-weight: 500;
    transition: all 0.3s ease;
    backdrop-filter: blur(10px);
}

.back-btn:hover {
    background: rgba(255, 255, 255, 0.3);
    color: white;
    transform: translateX(-2px);
}

.search-card {
    background: white;
    border-radius: var(--border-radius);
    box-shadow: var(--shadow-md);
    margin-bottom: 2rem;
    overflow: hidden;
    border: 1px solid var(--gray-200);
}

.search-header {
    background: linear-gradient(135deg, var(--gray-50) 0%, white 100%);
    padding: 1.5rem;
    border-bottom: 1px solid var(--gray-200);
    text-align: center;
}

.search-header h2 {
    margin: 0 0 0.5rem 0;
    color: var(--gray-800);
    font-weight: 600;
    font-size: 1.25rem;
}

.search-header p {
    margin: 0;
    color: var(--gray-600);
    font-size: 0.875rem;
}

.search-body {
    padding: 2rem;
}

.simple-search-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.search-field {
    position: relative;
}

.search-field label {
    display: block;
    font-size: 0.875rem;
    font-weight: 600;
    color: var(--gray-700);
    margin-bottom: 0.5rem;
}

.search-input, .search-select {
    width: 100%;
    padding: 0.875rem 1rem;
    border: 2px solid var(--gray-200);
    border-radius: 12px;
    font-size: 0.875rem;
    transition: all 0.3s ease;
    background: white;
    color: var(--gray-800);
}

.search-input:focus, .search-select:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 4px rgba(79, 70, 229, 0.1);
    transform: translateY(-1px);
}

.search-input::placeholder {
    color: var(--gray-400);
}

.search-actions {
    display: flex;
    gap: 1rem;
    justify-content: center;
    align-items: center;
    flex-wrap: wrap;
    padding-top: 1.5rem;
    border-top: 1px solid var(--gray-200);
}

.btn {
    padding: 0.875rem 2rem;
    border-radius: 12px;
    font-weight: 600;
    font-size: 0.875rem;
    transition: all 0.3s ease;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    border: 2px solid transparent;
    cursor: pointer;
    text-decoration: none;
    min-width: 140px;
    justify-content: center;
}

.btn-primary {
    background: linear-gradient(135deg, var(--primary) 0%, #6366f1 100%);
    color: white;
    border-color: var(--primary);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(79, 70, 229, 0.3);
    color: white;
}

.btn-secondary {
    background: white;
    color: var(--gray-700);
    border-color: var(--gray-300);
}

.btn-secondary:hover {
    background: var(--gray-50);
    border-color: var(--gray-400);
    color: var(--gray-800);
    transform: translateY(-1px);
}

.results-card {
    background: white;
    border-radius: var(--border-radius);
    border: 1px solid var(--gray-200);
    overflow: hidden;
    box-shadow: var(--shadow-md);
}

.results-header {
    background: linear-gradient(135deg, var(--gray-50) 0%, white 100%);
    padding: 1.5rem;
    border-bottom: 1px solid var(--gray-200);
}

.results-stats {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
}

.result-stat {
    text-align: center;
    padding: 1.5rem;
    background: linear-gradient(135deg, white 0%, var(--gray-50) 100%);
    border-radius: 12px;
    border: 1px solid var(--gray-200);
    transition: all 0.3s ease;
}

.result-stat:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.result-stat-value {
    font-size: 1.75rem;
    font-weight: 700;
    color: var(--gray-900);
    margin-bottom: 0.5rem;
}

.result-stat-label {
    font-size: 0.875rem;
    color: var(--gray-600);
    font-weight: 500;
}

.table-container {
    overflow-x: auto;
}

.results-table {
    width: 100%;
    border-collapse: collapse;
}

.results-table th {
    background: linear-gradient(135deg, var(--gray-50) 0%, white 100%);
    padding: 1rem 1.5rem;
    text-align: right;
    font-weight: 600;
    font-size: 0.875rem;
    color: var(--gray-700);
    border-bottom: 2px solid var(--gray-200);
    white-space: nowrap;
}

.results-table td {
    padding: 1.25rem 1.5rem;
    border-bottom: 1px solid var(--gray-100);
    vertical-align: middle;
}

.results-table tbody tr {
    transition: all 0.2s ease;
}

.results-table tbody tr:hover {
    background-color: var(--gray-50);
    transform: scale(1.001);
}

.invoice-number {
    font-weight: 600;
    color: var(--gray-900);
    margin-bottom: 0.25rem;
}

.invoice-uuid {
    font-size: 0.75rem;
    color: var(--gray-500);
    font-family: 'Courier New', monospace;
}

.customer-name {
    font-weight: 500;
    color: var(--gray-800);
    margin-bottom: 0.25rem;
}

.customer-vat {
    font-size: 0.75rem;
    color: var(--gray-500);
    font-family: 'Courier New', monospace;
}

.status-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.375rem;
    padding: 0.5rem 0.75rem;
    border-radius: 6px;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.025em;
}

.status-badge.primary {
    background: rgba(79, 70, 229, 0.1);
    color: var(--primary);
}

.status-badge.success {
    background: rgba(16, 185, 129, 0.1);
    color: var(--success);
}

.status-badge.warning {
    background: rgba(245, 158, 11, 0.1);
    color: var(--warning);
}

.status-badge.danger {
    background: rgba(239, 68, 68, 0.1);
    color: var(--danger);
}

.status-badge.info {
    background: rgba(6, 182, 212, 0.1);
    color: var(--info);
}

.status-badge.secondary {
    background: rgba(107, 114, 128, 0.1);
    color: var(--gray-600);
}

.action-menu {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    justify-content: center;
}

.action-btn {
    width: 36px;
    height: 36px;
    border-radius: 8px;
    border: 1px solid var(--gray-300);
    background: white;
    color: var(--gray-500);
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 0.2s ease;
    text-decoration: none;
    cursor: pointer;
    font-size: 0.875rem;
}

.action-btn:hover {
    background: var(--gray-50);
    border-color: var(--gray-400);
    color: var(--gray-700);
    transform: translateY(-1px);
}

.action-btn.view:hover {
    background: rgba(59, 130, 246, 0.1);
    color: #3b82f6;
    border-color: #93c5fd;
}

.empty-state {
    text-align: center;
    padding: 4rem 2rem;
    color: var(--gray-500);
}

.empty-icon {
    width: 100px;
    height: 100px;
    margin: 0 auto 1.5rem auto;
    background: linear-gradient(135deg, var(--gray-100) 0%, var(--gray-200) 100%);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 2.5rem;
    color: var(--gray-400);
}

/* Custom Pagination Styles */
.pagination-wrapper {
    padding: 1.5rem;
    background: linear-gradient(135deg, var(--gray-50) 0%, white 100%);
    border-top: 1px solid var(--gray-200);
    display: flex;
    justify-content: center;
    align-items: center;
}

.pagination {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    list-style: none;
    margin: 0;
    padding: 0;
}

.pagination .page-item {
    margin: 0;
}

.pagination .page-link {
    display: flex;
    align-items: center;
    justify-content: center;
    min-width: 40px;
    height: 40px;
    padding: 0.5rem 0.75rem;
    color: var(--gray-700);
    background: white;
    border: 1px solid var(--gray-300);
    border-radius: 8px;
    text-decoration: none;
    font-size: 0.875rem;
    font-weight: 500;
    transition: all 0.2s ease;
}

.pagination .page-link:hover {
    background: var(--gray-50);
    border-color: var(--gray-400);
    color: var(--gray-900);
    transform: translateY(-1px);
}

.pagination .page-item.active .page-link {
    background: var(--primary);
    border-color: var(--primary);
    color: white;
    box-shadow: 0 4px 12px rgba(79, 70, 229, 0.3);
}

.pagination .page-item.disabled .page-link {
    color: var(--gray-400);
    background: var(--gray-100);
    border-color: var(--gray-200);
    cursor: not-allowed;
}

.pagination .page-item.disabled .page-link:hover {
    transform: none;
}

@media (max-width: 768px) {
    .page-container { padding: 1rem; }
    .page-header { padding: 1.5rem; position: relative; }
    .header-title h1 { font-size: 1.75rem; }
    .simple-search-grid { grid-template-columns: 1fr; }
    .results-stats { grid-template-columns: repeat(2, 1fr); }
    .search-actions { justify-content: stretch; }
    .search-actions .btn { flex: 1; }
    .back-btn { position: static; margin-bottom: 1rem; display: inline-flex; }
}
</style>
@endpush

@section('content')
<div class="">
    <!-- Page Header -->
    <div class="page-header">
        <a href="{{ route('invoices.index') }}" class="back-btn">
            <i class="fas fa-arrow-left"></i>
            العودة للفواتير
        </a>
        <div class="header-content">
            <div class="header-title">
                <h1>البحث في الفواتير</h1>
                <p>بحث سريع ومبسط في جميع الفواتير</p>
            </div>
        </div>
    </div>

    <!-- Simple Search Form -->
    <div class="search-card">
        <div class="search-header">
            <h2>البحث السريع</h2>
            <p>استخدم الحقول أدناه للبحث في الفواتير</p>
        </div>

        <div class="search-body">
            <form method="GET" action="{{ route('invoices.advanced-search') }}">
                <div class="simple-search-grid">
                    <!-- رقم الفاتورة -->
                    <div class="search-field">
                        <label>رقم الفاتورة</label>
                        <input type="text" name="invoice_number" class="search-input"
                               placeholder="ادخل رقم الفاتورة..." value="{{ request('invoice_number') }}">
                    </div>

                    <!-- اسم العميل أو الرقم الضريبي -->
                    <div class="search-field">
                        <label>اسم العميل أو الرقم الضريبي</label>
                        <input type="text" name="customer_search" class="search-input"
                               placeholder="ادخل اسم العميل أو الرقم الضريبي..." value="{{ request('customer_search') }}">
                    </div>

                    <!-- التاريخ من -->
                    <div class="search-field">
                        <label>من تاريخ</label>
                        <input type="date" name="date_from" class="search-input" value="{{ request('date_from') }}">
                    </div>

                    <!-- التاريخ إلى -->
                    <div class="search-field">
                        <label>إلى تاريخ</label>
                        <input type="date" name="date_to" class="search-input" value="{{ request('date_to') }}">
                    </div>
                </div>

                <div class="search-actions">
                    <a href="{{ route('invoices.advanced-search') }}" class="btn btn-secondary">
                        <i class="fas fa-undo"></i>
                        مسح الفلاتر
                    </a>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-search"></i>
                        بحث
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Search Results -->
    @if(isset($invoices))
    <div class="results-card">
        <div class="results-header">
            <div class="results-stats">
                <div class="result-stat">
                    <div class="result-stat-value">{{ $stats['total_invoices'] }}</div>
                    <div class="result-stat-label">إجمالي النتائج</div>
                </div>
                <div class="result-stat">
                    <div class="result-stat-value">{{ number_format($stats['total_amount'], 0) }}</div>
                    <div class="result-stat-label">إجمالي المبلغ (ر.س)</div>
                </div>
                <div class="result-stat">
                    <div class="result-stat-value">{{ number_format($stats['total_paid'], 0) }}</div>
                    <div class="result-stat-label">إجمالي المدفوع (ر.س)</div>
                </div>
                <div class="result-stat">
                    <div class="result-stat-value">{{ number_format($stats['total_pending'], 0) }}</div>
                    <div class="result-stat-label">المبلغ المعلق (ر.س)</div>
                </div>
            </div>
        </div>

        @if($invoices->count() > 0)
            <div class="table-container">
                <table class="results-table">
                    <thead>
                        <tr>
                            <th>رقم الفاتورة</th>
                            <th>النوع</th>
                            <th>العميل</th>
                            <th>التاريخ</th>
                            <th>المبلغ</th>
                            <th>حالة الفاتورة</th>
                            <th>حالة السداد</th>
                            <th style="text-align: center;">الإجراءات</th>
                        </tr>
                    </thead>
                    <tbody>
                        @foreach($invoices as $invoice)
                        <tr>
                            <td>
                                <div class="invoice-number">{{ $invoice->invoice_number }}</div>
                                @if($invoice->uuid)
                                    <div class="invoice-uuid">{{ Str::limit($invoice->uuid, 8) }}...</div>
                                @endif
                            </td>
                            <td>
                                @if($invoice->type == 'standard')
                                    <span class="status-badge primary">
                                        <i class="fas fa-file-invoice-dollar"></i>
                                        معيارية
                                    </span>
                                @else
                                    <span class="status-badge info">
                                        <i class="fas fa-receipt"></i>
                                        مبسطة
                                    </span>
                                @endif
                            </td>
                            <td>
                                @if($invoice->customer)
                                    <div class="customer-name">{{ $invoice->customer->name }}</div>
                                    @if($invoice->customer->customer_type == 'business' && $invoice->customer->vat_number)
                                        <div class="customer-vat">{{ $invoice->customer->vat_number }}</div>
                                    @endif
                                @else
                                    <span style="color: var(--gray-400);">غير محدد</span>
                                @endif
                            </td>
                            <td>
                                <div>{{ $invoice->issue_date->format('Y-m-d') }}</div>
                                <div style="font-size: 0.75rem; color: var(--gray-500);">{{ $invoice->issue_date->format('l') }}</div>
                            </td>
                            <td>
                                <div style="font-weight: 600; color: var(--gray-900); margin-bottom: 0.25rem;">{{ number_format($invoice->total_amount, 2) }} ر.س</div>
                                <div style="font-size: 0.75rem; color: var(--gray-500);">ضريبة: {{ number_format($invoice->tax_amount, 2) }}</div>
                            </td>
                            <td>
                                @switch($invoice->status)
                                    @case('draft')
                                        <span class="status-badge secondary">
                                            <i class="fas fa-edit"></i>
                                            مسودة
                                        </span>
                                        @break
                                    @case('sent')
                                        <span class="status-badge info">
                                            <i class="fas fa-paper-plane"></i>
                                            مرسلة
                                        </span>
                                        @break
                                    @case('cleared')
                                        <span class="status-badge success">
                                            <i class="fas fa-check-circle"></i>
                                            مؤكدة
                                        </span>
                                        @break
                                    @case('reported')
                                        <span class="status-badge primary">
                                            <i class="fas fa-flag"></i>
                                            مبلغة
                                        </span>
                                        @break
                                    @case('rejected')
                                        <span class="status-badge danger">
                                            <i class="fas fa-times-circle"></i>
                                            مرفوضة
                                        </span>
                                        @break
                                    @default
                                        <span class="status-badge warning">
                                            <i class="fas fa-exclamation-triangle"></i>
                                            غير محدد
                                        </span>
                                @endswitch
                            </td>
                            <td>
                                @switch($invoice->payment_status)
                                    @case('paid')
                                        <span class="status-badge success">
                                            <i class="fas fa-check-circle"></i>
                                            مدفوع
                                        </span>
                                        @break
                                    @case('partial')
                                        <span class="status-badge warning">
                                            <i class="fas fa-clock"></i>
                                            جزئي
                                        </span>
                                        @break
                                    @case('unpaid')
                                        <span class="status-badge danger">
                                            <i class="fas fa-times-circle"></i>
                                            غير مدفوع
                                        </span>
                                        @break
                                    @case('overdue')
                                        <span class="status-badge danger">
                                            <i class="fas fa-exclamation-triangle"></i>
                                            متأخر
                                        </span>
                                        @break
                                    @default
                                        <span class="status-badge secondary">
                                            <i class="fas fa-question"></i>
                                            غير محدد
                                        </span>
                                @endswitch
                            </td>
                            <td>
                                <div class="action-menu">
                                    <a href="{{ route('invoices.show', $invoice) }}" class="action-btn view" title="عرض">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                </div>
                            </td>
                        </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>

            <!-- Custom Pagination -->
            <div class="pagination-wrapper">
                {{ $invoices->appends(request()->query())->links('pagination::bootstrap-4') }}
            </div>
        @else
            <div class="empty-state">
                <div class="empty-icon">
                    <i class="fas fa-search"></i>
                </div>
                <h3>لا توجد نتائج</h3>
                <p>لم يتم العثور على فواتير تطابق معايير البحث المحددة</p>
                <a href="{{ route('invoices.advanced-search') }}" class="btn-primary">
                    <i class="fas fa-undo"></i>
                    مسح الفلاتر والبدء من جديد
                </a>
            </div>
        @endif
    </div>
    @endif
</div>
@endsection

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Add smooth animations
    const searchInputs = document.querySelectorAll('.search-input, .search-select');
    
    searchInputs.forEach(input => {
        input.addEventListener('focus', function() {
            this.parentElement.style.transform = 'scale(1.02)';
        });
        
        input.addEventListener('blur', function() {
            this.parentElement.style.transform = 'scale(1)';
        });
    });

    // Auto-clear form functionality
    const clearBtn = document.querySelector('.btn-secondary');
    if (clearBtn) {
        clearBtn.addEventListener('click', function(e) {
            e.preventDefault();
            
            // Clear all inputs
            searchInputs.forEach(input => {
                input.value = '';
            });
            
            // Submit form to clear results
            window.location.href = '{{ route("invoices.advanced-search") }}';
        });
    }

    console.log('البحث المبسط جاهز للاستخدام');
});
</script>
@endpush