@extends('layouts.app')

@section('title', 'إنشاء تسوية مخزون')
@section('page-title', 'إنشاء تسوية مخزون')

@section('page-actions')
    <a href="{{ route('inventory.adjustments.index') }}" class="btn btn-secondary">
        <i class="fas fa-arrow-right me-2"></i>
        العودة للقائمة
    </a>
@endsection

@push('styles')
<style>
    .product-row {
        border: 1px solid #dee2e6;
        border-radius: 8px;
        padding: 15px;
        margin-bottom: 15px;
        background-color: #f8f9fa;
    }
    .remove-item {
        position: absolute;
        top: 10px;
        left: 10px;
    }
    .quantity-info {
        background-color: #e9ecef;
        padding: 8px;
        border-radius: 4px;
        margin-bottom: 10px;
    }
</style>
@endpush

@section('content')
<form method="POST" action="{{ route('inventory.adjustments.store') }}" id="adjustmentForm">
    @csrf
    
    <div class="row">
        <!-- بيانات التسوية -->
        <div class="col-md-8">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">
                        <i class="fas fa-info-circle me-2"></i>
                        بيانات التسوية
                    </h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="adjustment_date" class="form-label">تاريخ التسوية <span class="text-danger">*</span></label>
                                <input type="date" class="form-control @error('adjustment_date') is-invalid @enderror" 
                                       id="adjustment_date" name="adjustment_date" value="{{ old('adjustment_date', date('Y-m-d')) }}" required>
                                @error('adjustment_date')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="type" class="form-label">نوع التسوية <span class="text-danger">*</span></label>
                                <select class="form-control @error('type') is-invalid @enderror" 
                                        id="type" name="type" required>
                                    <option value="">اختر نوع التسوية</option>
                                    <option value="increase" {{ old('type') == 'increase' ? 'selected' : '' }}>زيادة</option>
                                    <option value="decrease" {{ old('type') == 'decrease' ? 'selected' : '' }}>نقص</option>
                                    <option value="recount" {{ old('type') == 'recount' ? 'selected' : '' }}>إعادة عد</option>
                                </select>
                                @error('type')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="reason" class="form-label">السبب <span class="text-danger">*</span></label>
                        <input type="text" class="form-control @error('reason') is-invalid @enderror" 
                               id="reason" name="reason" value="{{ old('reason') }}" required 
                               placeholder="مثال: تلف، فقدان، خطأ في العد، إلخ">
                        @error('reason')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    
                    <div class="mb-3">
                        <label for="notes" class="form-label">ملاحظات</label>
                        <textarea class="form-control @error('notes') is-invalid @enderror" 
                                  id="notes" name="notes" rows="3">{{ old('notes') }}</textarea>
                        @error('notes')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                </div>
            </div>
            
            <!-- عناصر التسوية -->
            <div class="card mt-4">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">
                        <i class="fas fa-list me-2"></i>
                        المنتجات المراد تسويتها
                    </h5>
                    <button type="button" class="btn btn-primary btn-sm" id="addItem">
                        <i class="fas fa-plus me-2"></i>
                        إضافة منتج
                    </button>
                </div>
                <div class="card-body">
                    <div id="itemsContainer">
                        <!-- سيتم إضافة العناصر هنا -->
                    </div>
                    
                    @error('items')
                        <div class="alert alert-danger">{{ $message }}</div>
                    @enderror
                </div>
            </div>
        </div>
        
        <!-- ملخص التسوية -->
        <div class="col-md-4">
            <div class="card sticky-top">
                <div class="card-header">
                    <h5 class="mb-0">
                        <i class="fas fa-calculator me-2"></i>
                        ملخص التسوية
                    </h5>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <strong>إجمالي العناصر:</strong>
                        <span id="totalItems" class="float-end">0</span>
                    </div>
                    
                    <div class="mb-3">
                        <strong>إجمالي التأثير على القيمة:</strong>
                        <span id="totalValueImpact" class="float-end">0.00 ر.س</span>
                    </div>
                    
                    <hr>
                    
                    <div class="alert alert-info">
                        <small>
                            <i class="fas fa-info-circle me-2"></i>
                            ستحتاج هذه التسوية لموافقة المدير قبل تطبيقها على المخزون
                        </small>
                    </div>
                    
                    <div class="d-grid gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i>
                            حفظ التسوية
                        </button>
                        
                        <a href="{{ route('inventory.adjustments.index') }}" class="btn btn-secondary">
                            <i class="fas fa-times me-2"></i>
                            إلغاء
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</form>

<!-- قالب عنصر المنتج -->
<template id="itemTemplate">
    <div class="product-row position-relative" data-index="__INDEX__">
        <button type="button" class="btn btn-danger btn-sm remove-item">
            <i class="fas fa-times"></i>
        </button>
        
        <div class="row">
            <div class="col-md-6">
                <div class="mb-3">
                    <label class="form-label">المنتج <span class="text-danger">*</span></label>
                    <select class="form-control product-select" name="items[__INDEX__][product_id]" required>
                        <option value="">اختر المنتج</option>
                        @foreach($products as $product)
                            <option value="{{ $product->id }}" 
                                    data-current-stock="{{ $product->stock_quantity }}"
                                    data-average-cost="{{ $product->getAverageCost() }}">
                                {{ $product->name }} ({{ $product->sku }})
                            </option>
                        @endforeach
                    </select>
                </div>
                
                <div class="quantity-info" style="display: none;">
                    <small>
                        <strong>الكمية الحالية:</strong> <span class="current-stock">0</span><br>
                        <strong>متوسط التكلفة:</strong> <span class="average-cost">0.00</span> ر.س
                    </small>
                </div>
            </div>
            
            <div class="col-md-6">
                <div class="row">
                    <div class="col-6">
                        <div class="mb-3">
                            <label class="form-label">الكمية الحالية</label>
                            <input type="number" class="form-control current-quantity-input" 
                                   name="items[__INDEX__][current_quantity]" readonly>
                        </div>
                    </div>
                    
                    <div class="col-6">
                        <div class="mb-3">
                            <label class="form-label">الكمية الجديدة <span class="text-danger">*</span></label>
                            <input type="number" class="form-control adjusted-quantity-input" 
                                   name="items[__INDEX__][adjusted_quantity]" min="0" required>
                        </div>
                    </div>
                </div>
                
                <div class="row">
                    <div class="col-6">
                        <div class="mb-3">
                            <label class="form-label">الفرق</label>
                            <input type="text" class="form-control quantity-difference" readonly>
                        </div>
                    </div>
                    
                    <div class="col-6">
                        <div class="mb-3">
                            <label class="form-label">تكلفة الوحدة <span class="text-danger">*</span></label>
                            <input type="number" step="0.01" class="form-control unit-cost-input" 
                                   name="items[__INDEX__][unit_cost]" min="0" required>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="mb-3">
            <label class="form-label">سبب التسوية</label>
            <input type="text" class="form-control" name="items[__INDEX__][reason]" 
                   placeholder="سبب خاص بهذا المنتج (اختياري)">
        </div>
    </div>
</template>
@endsection

@push('scripts')
<script>
let itemIndex = 0;

$(document).ready(function() {
    // إضافة عنصر جديد
    $('#addItem').click(function() {
        addNewItem();
    });
    
    // حذف عنصر
    $(document).on('click', '.remove-item', function() {
        $(this).closest('.product-row').remove();
        updateSummary();
    });
    
    // تحديث البيانات عند تغيير المنتج
    $(document).on('change', '.product-select', function() {
        const option = $(this).find('option:selected');
        const row = $(this).closest('.product-row');
        
        if (option.val()) {
            const currentStock = parseInt(option.data('current-stock')) || 0;
            const averageCost = parseFloat(option.data('average-cost')) || 0;
            
            row.find('.current-stock').text(currentStock);
            row.find('.average-cost').text(averageCost.toFixed(2));
            row.find('.current-quantity-input').val(currentStock);
            row.find('.unit-cost-input').val(averageCost.toFixed(2));
            row.find('.quantity-info').show();
        } else {
            row.find('.quantity-info').hide();
            row.find('.current-quantity-input').val('');
            row.find('.unit-cost-input').val('');
        }
        
        calculateDifference(row);
    });
    
    // حساب الفرق عند تغيير الكمية
    $(document).on('input', '.adjusted-quantity-input', function() {
        const row = $(this).closest('.product-row');
        calculateDifference(row);
    });
    
    // تحديث الملخص عند تغيير التكلفة
    $(document).on('input', '.unit-cost-input', function() {
        updateSummary();
    });
});

function addNewItem() {
    const template = $('#itemTemplate').html();
    const newItem = template.replace(/__INDEX__/g, itemIndex);
    $('#itemsContainer').append(newItem);
    itemIndex++;
}

function calculateDifference(row) {
    const currentQty = parseInt(row.find('.current-quantity-input').val()) || 0;
    const adjustedQty = parseInt(row.find('.adjusted-quantity-input').val()) || 0;
    const difference = adjustedQty - currentQty;
    
    let differenceText = difference.toString();
    if (difference > 0) {
        differenceText = '+' + difference;
        row.find('.quantity-difference').removeClass('text-danger').addClass('text-success');
    } else if (difference < 0) {
        row.find('.quantity-difference').removeClass('text-success').addClass('text-danger');
    } else {
        row.find('.quantity-difference').removeClass('text-success text-danger');
    }
    
    row.find('.quantity-difference').val(differenceText);
    updateSummary();
}

function updateSummary() {
    let totalItems = 0;
    let totalValueImpact = 0;
    
    $('.product-row').each(function() {
        const currentQty = parseInt($(this).find('.current-quantity-input').val()) || 0;
        const adjustedQty = parseInt($(this).find('.adjusted-quantity-input').val()) || 0;
        const unitCost = parseFloat($(this).find('.unit-cost-input').val()) || 0;
        
        if ($(this).find('.product-select').val()) {
            totalItems++;
            const difference = adjustedQty - currentQty;
            totalValueImpact += difference * unitCost;
        }
    });
    
    $('#totalItems').text(totalItems);
    $('#totalValueImpact').text(totalValueImpact.toFixed(2) + ' ر.س');
    
    // تغيير لون القيمة حسب التأثير
    if (totalValueImpact > 0) {
        $('#totalValueImpact').removeClass('text-danger').addClass('text-success');
    } else if (totalValueImpact < 0) {
        $('#totalValueImpact').removeClass('text-success').addClass('text-danger');
    } else {
        $('#totalValueImpact').removeClass('text-success text-danger');
    }
}
</script>
@endpush