@extends('layouts.app')

@section('title', 'تسجيل استلام بضائع')
@section('page-title', 'تسجيل استلام بضائع جديد')

@section('page-actions')
    <a href="{{ route('goods-receipts.index') }}" class="btn btn-secondary">
        <i class="fas fa-arrow-right me-2"></i>
        العودة للقائمة
    </a>
@endsection

@push('styles')
<style>
    .product-row {
        border: 1px solid #dee2e6;
        border-radius: 8px;
        padding: 15px;
        margin-bottom: 15px;
        background-color: #f8f9fa;
    }
    .quantity-info {
        background-color: #e9ecef;
        padding: 8px;
        border-radius: 4px;
        margin-bottom: 10px;
    }
</style>
@endpush

@section('content')
<form method="POST" action="{{ route('goods-receipts.store') }}" id="goodsReceiptForm">
    @csrf
    
    <div class="row">
        <!-- بيانات الاستلام -->
        <div class="col-md-8">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">
                        <i class="fas fa-info-circle me-2"></i>
                        بيانات الاستلام
                    </h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="purchase_order_id" class="form-label">أمر الشراء <span class="text-danger">*</span></label>
                                <select class="form-control @error('purchase_order_id') is-invalid @enderror" 
                                        id="purchase_order_id" name="purchase_order_id" required>
                                    <option value="">اختر أمر الشراء</option>
                                    @foreach($purchaseOrders as $order)
                                        <option value="{{ $order->id }}" 
                                                {{ (old('purchase_order_id') ?? request('purchase_order_id')) == $order->id ? 'selected' : '' }}
                                                data-supplier="{{ $order->supplier->name }}"
                                                data-items="{{ json_encode($order->items->map(function($item) {
                                                    return [
                                                        'id' => $item->id,
                                                        'product_id' => $item->product_id,
                                                        'product_name' => $item->product->name,
                                                        'product_sku' => $item->product->sku,
                                                        'ordered_quantity' => $item->quantity,
                                                        'unit_price' => $item->unit_price,
                                                        'received_quantity' => $item->receivedQuantity()
                                                    ];
                                                })) }}">
                                            {{ $order->po_number }} - {{ $order->supplier->name }}
                                            <small>({{ $order->order_date->format('Y/m/d') }})</small>
                                        </option>
                                    @endforeach
                                </select>
                                @error('purchase_order_id')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="receipt_date" class="form-label">تاريخ الاستلام <span class="text-danger">*</span></label>
                                <input type="date" class="form-control @error('receipt_date') is-invalid @enderror" 
                                       id="receipt_date" name="receipt_date" value="{{ old('receipt_date', date('Y-m-d')) }}" required>
                                @error('receipt_date')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="notes" class="form-label">ملاحظات الاستلام</label>
                        <textarea class="form-control @error('notes') is-invalid @enderror" 
                                  id="notes" name="notes" rows="3">{{ old('notes') }}</textarea>
                        @error('notes')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    
                    <!-- معلومات أمر الشراء -->
                    <div id="orderInfo" style="display: none;" class="alert alert-info">
                        <h6><i class="fas fa-info-circle me-2"></i>معلومات أمر الشراء</h6>
                        <div class="row">
                            <div class="col-md-6">
                                <strong>المورد:</strong> <span id="supplierName"></span>
                            </div>
                            <div class="col-md-6">
                                <strong>عدد العناصر:</strong> <span id="itemsCount"></span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- عناصر الاستلام -->
            <div class="card mt-4">
                <div class="card-header">
                    <h5 class="mb-0">
                        <i class="fas fa-list me-2"></i>
                        عناصر الاستلام
                    </h5>
                </div>
                <div class="card-body">
                    <div id="itemsContainer">
                        <div class="text-center text-muted py-4">
                            <i class="fas fa-shopping-cart fa-3x mb-3"></i>
                            <p>اختر أمر شراء أولاً لعرض العناصر</p>
                        </div>
                    </div>
                    
                    @error('items')
                        <div class="alert alert-danger">{{ $message }}</div>
                    @enderror
                </div>
            </div>
        </div>
        
        <!-- ملخص الاستلام -->
        <div class="col-md-4">
            <div class="card sticky-top">
                <div class="card-header">
                    <h5 class="mb-0">
                        <i class="fas fa-calculator me-2"></i>
                        ملخص الاستلام
                    </h5>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <strong>إجمالي العناصر:</strong>
                        <span id="totalItems" class="float-end">0</span>
                    </div>
                    
                    <div class="mb-3">
                        <strong>الكمية المطلوبة:</strong>
                        <span id="totalOrdered" class="float-end">0</span>
                    </div>
                    
                    <div class="mb-3">
                        <strong>الكمية المستلمة:</strong>
                        <span id="totalReceived" class="float-end">0</span>
                    </div>
                    
                    <div class="mb-3">
                        <strong>نسبة الاستلام:</strong>
                        <div class="progress mt-2">
                            <div id="progressBar" class="progress-bar bg-info" style="width: 0%">0%</div>
                        </div>
                    </div>
                    
                    <hr>
                    
                    <div class="alert alert-warning">
                        <small>
                            <i class="fas fa-info-circle me-2"></i>
                            يمكنك استلام كمية أقل من المطلوب (استلام جزئي)
                        </small>
                    </div>
                    
                    <div class="d-grid gap-2">
                        <button type="submit" class="btn btn-primary" id="submitBtn" disabled>
                            <i class="fas fa-save me-2"></i>
                            حفظ الاستلام
                        </button>
                        
                        <a href="{{ route('goods-receipts.index') }}" class="btn btn-secondary">
                            <i class="fas fa-times me-2"></i>
                            إلغاء
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</form>
@endsection

@push('scripts')
<script>
let orderItems = [];

$(document).ready(function() {
    // تحديث العناصر عند تغيير أمر الشراء
    $('#purchase_order_id').change(function() {
        loadOrderItems();
    });
    
    // تحديث الملخص عند تغيير الكميات
    $(document).on('input', '.received-quantity', function() {
        updateSummary();
        validateForm();
    });
    
    // تحميل العناصر إذا كان هناك أمر محدد مسبقاً
    if ($('#purchase_order_id').val()) {
        loadOrderItems();
    }
});

function loadOrderItems() {
    const selectedOption = $('#purchase_order_id option:selected');
    
    if (!selectedOption.val()) {
        $('#orderInfo').hide();
        $('#itemsContainer').html(`
            <div class="text-center text-muted py-4">
                <i class="fas fa-shopping-cart fa-3x mb-3"></i>
                <p>اختر أمر شراء أولاً لعرض العناصر</p>
            </div>
        `);
        $('#submitBtn').prop('disabled', true);
        return;
    }
    
    // عرض معلومات الأمر
    const supplierName = selectedOption.data('supplier');
    orderItems = selectedOption.data('items') || [];
    
    $('#supplierName').text(supplierName);
    $('#itemsCount').text(orderItems.length);
    $('#orderInfo').show();
    
    // إنشاء عناصر الاستلام
    let itemsHtml = '';
    
    if (orderItems.length === 0) {
        itemsHtml = `
            <div class="text-center text-muted py-4">
                <i class="fas fa-exclamation-triangle fa-3x mb-3"></i>
                <p>لا توجد عناصر في أمر الشراء المحدد</p>
            </div>
        `;
    } else {
        orderItems.forEach((item, index) => {
            const remainingQuantity = item.ordered_quantity - item.received_quantity;
            const isFullyReceived = remainingQuantity <= 0;
            
            itemsHtml += `
                <div class="product-row ${isFullyReceived ? 'bg-light' : ''}" data-index="${index}">
                    <input type="hidden" name="items[${index}][purchase_order_item_id]" value="${item.id}">
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label"><strong>${item.product_name}</strong></label>
                                <div class="quantity-info">
                                    <small>
                                        <strong>رمز المنتج:</strong> ${item.product_sku}<br>
                                        <strong>الكمية المطلوبة:</strong> ${item.ordered_quantity}<br>
                                        <strong>مستلم سابقاً:</strong> ${item.received_quantity}<br>
                                        <strong>المتبقي:</strong> <span class="text-${isFullyReceived ? 'success' : 'warning'}">${remainingQuantity}</span>
                                    </small>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="row">
                                <div class="col-6">
                                    <div class="mb-3">
                                        <label class="form-label">الكمية المستلمة</label>
                                        <input type="number" 
                                               class="form-control received-quantity" 
                                               name="items[${index}][received_quantity]" 
                                               min="0" 
                                               max="${remainingQuantity}" 
                                               value="${isFullyReceived ? 0 : remainingQuantity}"
                                               ${isFullyReceived ? 'readonly' : 'required'}>
                                    </div>
                                </div>
                                
                                <div class="col-6">
                                    <div class="mb-3">
                                        <label class="form-label">تكلفة الوحدة</label>
                                        <input type="number" 
                                               step="0.01" 
                                               class="form-control unit-cost" 
                                               name="items[${index}][unit_cost]" 
                                               value="${item.unit_price}" 
                                               min="0" 
                                               required>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">ملاحظات (اختياري)</label>
                                <input type="text" 
                                       class="form-control" 
                                       name="items[${index}][notes]" 
                                       placeholder="ملاحظات خاصة بهذا المنتج">
                            </div>
                        </div>
                    </div>
                    
                    ${isFullyReceived ? '<div class="alert alert-success">تم استلام هذا المنتج بالكامل</div>' : ''}
                </div>
            `;
        });
    }
    
    $('#itemsContainer').html(itemsHtml);
    updateSummary();
    validateForm();
}

function updateSummary() {
    let totalItems = 0;
    let totalOrdered = 0;
    let totalReceived = 0;
    
    $('.product-row').each(function(index) {
        const orderedQty = orderItems[index]?.ordered_quantity || 0;
        const receivedQty = parseFloat($(this).find('.received-quantity').val()) || 0;
        
        if (orderedQty > 0) {
            totalItems++;
            totalOrdered += orderedQty;
            totalReceived += receivedQty;
        }
    });
    
    const percentage = totalOrdered > 0 ? (totalReceived / totalOrdered) * 100 : 0;
    
    $('#totalItems').text(totalItems);
    $('#totalOrdered').text(totalOrdered);
    $('#totalReceived').text(totalReceived);
    $('#progressBar').css('width', percentage + '%').text(Math.round(percentage) + '%');
    
    // تغيير لون شريط التقدم
    $('#progressBar').removeClass('bg-danger bg-warning bg-success bg-info');
    if (percentage === 100) {
        $('#progressBar').addClass('bg-success');
    } else if (percentage > 50) {
        $('#progressBar').addClass('bg-warning');
    } else if (percentage > 0) {
        $('#progressBar').addClass('bg-info');
    } else {
        $('#progressBar').addClass('bg-danger');
    }
}

function validateForm() {
    let hasValidItems = false;
    let isValid = true;
    
    $('.product-row').each(function() {
        const receivedQty = parseFloat($(this).find('.received-quantity').val()) || 0;
        const unitCost = parseFloat($(this).find('.unit-cost').val()) || 0;
        
        if (receivedQty > 0) {
            hasValidItems = true;
            if (unitCost <= 0) {
                isValid = false;
            }
        }
    });
    
    const purchaseOrderSelected = $('#purchase_order_id').val();
    const receiptDateFilled = $('#receipt_date').val();
    
    $('#submitBtn').prop('disabled', !(purchaseOrderSelected && receiptDateFilled && hasValidItems && isValid));
}
</script>
@endpush