@extends('layouts.app')

@section('title', 'تفاصيل العميل: ' . $customer->name)
@section('page-title', 'تفاصيل العميل')

@push('styles')
<style>
    .customer-profile-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        border-radius: 20px;
        padding: 2.5rem;
        margin-bottom: 2rem;
        color: white;
        position: relative;
        overflow: hidden;
        box-shadow: 0 15px 35px rgba(102, 126, 234, 0.3);
    }
    
    .customer-profile-header::before {
        content: '';
        position: absolute;
        top: 0;
        right: 0;
        width: 100%;
        height: 100%;
        background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="customer-grain" width="100" height="100" patternUnits="userSpaceOnUse"><circle cx="25" cy="25" r="1.5" fill="white" fill-opacity="0.15"/><circle cx="75" cy="75" r="1" fill="white" fill-opacity="0.1"/><circle cx="50" cy="10" r="1.2" fill="white" fill-opacity="0.12"/><circle cx="10" cy="60" r="0.8" fill="white" fill-opacity="0.08"/></pattern></defs><rect width="100" height="100" fill="url(%23customer-grain)"/></svg>');
        pointer-events: none;
    }
    
    .customer-avatar-large {
        width: 120px;
        height: 120px;
        border-radius: 50%;
        background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
        display: flex;
        align-items: center;
        justify-content: center;
        color: #667eea;
        font-weight: bold;
        font-size: 3rem;
        margin-bottom: 1rem;
        border: 4px solid rgba(255, 255, 255, 0.3);
        box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
    }
    
    .professional-card {
        border: none;
        border-radius: 20px;
        box-shadow: 0 15px 35px rgba(0, 0, 0, 0.08);
        transition: all 0.4s ease;
        background: white;
        position: relative;
        overflow: hidden;
        margin-bottom: 2rem;
    }
    
    .professional-card::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 5px;
        background: linear-gradient(90deg, #3498db, #9b59b6, #e74c3c, #f39c12);
    }
    
    .professional-card:hover {
        transform: translateY(-8px);
        box-shadow: 0 25px 50px rgba(0, 0, 0, 0.15);
    }
    
    .info-section {
        padding: 2rem;
    }
    
    .section-header {
        display: flex;
        align-items: center;
        margin-bottom: 1.5rem;
        padding-bottom: 1rem;
        border-bottom: 2px solid #f8f9fa;
    }
    
    .section-icon {
        width: 50px;
        height: 50px;
        border-radius: 50%;
        background: linear-gradient(135deg, #3498db, #2980b9);
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-size: 1.5rem;
        margin-left: 1rem;
        box-shadow: 0 8px 16px rgba(52, 152, 219, 0.3);
    }
    
    .section-title {
        font-size: 1.4rem;
        font-weight: 700;
        color: #2c3e50;
        margin: 0;
    }
    
    .info-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 1.5rem;
        margin-top: 1rem;
    }
    
    .info-item {
        background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%);
        border-radius: 15px;
        padding: 1.5rem;
        border: 1px solid #e9ecef;
        transition: all 0.3s ease;
        position: relative;
        overflow: hidden;
    }
    
    .info-item::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        width: 4px;
        height: 100%;
        background: linear-gradient(180deg, #3498db, #2980b9);
    }
    
    .info-item:hover {
        transform: translateY(-3px);
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
        background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
    }
    
    .info-label {
        font-size: 0.9rem;
        font-weight: 600;
        color: #6c757d;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        margin-bottom: 0.5rem;
        display: flex;
        align-items: center;
    }
    
    .info-label i {
        margin-left: 0.5rem;
        color: #3498db;
    }
    
    .info-value {
        font-size: 1.1rem;
        font-weight: 600;
        color: #2c3e50;
        line-height: 1.4;
    }
    
    .status-badge-large {
        padding: 12px 24px;
        border-radius: 30px;
        font-size: 1rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 1px;
        display: inline-flex;
        align-items: center;
        box-shadow: 0 5px 15px rgba(0, 0, 0, 0.2);
    }
    
    .type-badge-large {
        padding: 10px 20px;
        border-radius: 25px;
        font-size: 0.95rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        display: inline-flex;
        align-items: center;
        box-shadow: 0 5px 15px rgba(0, 0, 0, 0.15);
    }
    
    .stats-container {
        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
        border-radius: 20px;
        padding: 2rem;
        margin-bottom: 2rem;
    }
    
    .stat-item {
        text-align: center;
        padding: 1.5rem;
        background: white;
        border-radius: 15px;
        box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
        transition: all 0.3s ease;
        position: relative;
        overflow: hidden;
    }
    
    .stat-item::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 4px;
    }
    
    .stat-item.primary::before {
        background: linear-gradient(90deg, #3498db, #2980b9);
    }
    
    .stat-item.success::before {
        background: linear-gradient(90deg, #27ae60, #229954);
    }
    
    .stat-item.info::before {
        background: linear-gradient(90deg, #17a2b8, #138496);
    }
    
    .stat-item.warning::before {
        background: linear-gradient(90deg, #f39c12, #e67e22);
    }
    
    .stat-item:hover {
        transform: translateY(-5px);
        box-shadow: 0 15px 30px rgba(0, 0, 0, 0.15);
    }
    
    .stat-number {
        font-size: 2.5rem;
        font-weight: 700;
        margin-bottom: 0.5rem;
        display: block;
    }
    
    .stat-label {
        color: #6c757d;
        font-size: 0.9rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .modern-table {
        border-radius: 15px;
        overflow: hidden;
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.08);
        border: none;
    }
    
    .modern-table thead {
        background: linear-gradient(135deg, #2c3e50 0%, #34495e 100%);
        color: white;
    }
    
    .modern-table thead th {
        border: none;
        padding: 18px 15px;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        font-size: 0.9rem;
    }
    
    .modern-table tbody tr {
        transition: all 0.3s ease;
        border-bottom: 1px solid #f8f9fa;
    }
    
    .modern-table tbody tr:hover {
        background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%);
        transform: scale(1.02);
        box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
    }
    
    .modern-table tbody td {
        padding: 15px;
        vertical-align: middle;
        border: none;
    }
    
    .action-buttons-enhanced {
        display: flex;
        gap: 8px;
        flex-wrap: wrap;
    }
    
    .action-btn-enhanced {
        padding: 12px 20px;
        border-radius: 12px;
        border: none;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        transition: all 0.3s ease;
        display: flex;
        align-items: center;
        justify-content: center;
        min-width: 140px;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    }
    
    .action-btn-enhanced:hover {
        transform: translateY(-3px);
        box-shadow: 0 8px 20px rgba(0, 0, 0, 0.2);
    }
    
    .action-btn-enhanced i {
        margin-left: 8px;
        font-size: 1.1rem;
    }
    
    .address-display {
        background: linear-gradient(135deg, #3498db 0%, #2980b9 100%);
        color: white;
        padding: 1.5rem;
        border-radius: 15px;
        margin-top: 1rem;
        position: relative;
        overflow: hidden;
    }
    
    .address-display::before {
        content: '';
        position: absolute;
        top: 0;
        right: 0;
        width: 100%;
        height: 100%;
        background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="address-pattern" width="50" height="50" patternUnits="userSpaceOnUse"><path d="M0,50 Q25,25 50,50 T100,50" stroke="white" stroke-opacity="0.1" fill="none"/></pattern></defs><rect width="100" height="100" fill="url(%23address-pattern)"/></svg>');
        pointer-events: none;
    }
    
    .quick-actions-sidebar {
        position: sticky;
        top: 2rem;
    }
    
    .sidebar-card {
        background: white;
        border-radius: 20px;
        padding: 1.5rem;
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.08);
        margin-bottom: 1.5rem;
        border: none;
        transition: all 0.3s ease;
    }
    
    .sidebar-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 15px 35px rgba(0, 0, 0, 0.12);
    }
    
    .quick-action-btn {
        width: 100%;
        padding: 12px 16px;
        border-radius: 12px;
        border: none;
        margin-bottom: 10px;
        font-weight: 600;
        transition: all 0.3s ease;
        display: flex;
        align-items: center;
        justify-content: flex-start;
        text-align: right;
    }
    
    .quick-action-btn:hover {
        transform: translateX(-5px);
        box-shadow: 0 5px 15px rgba(0, 0, 0, 0.15);
    }
    
    .quick-action-btn i {
        margin-left: 10px;
        width: 20px;
        text-align: center;
    }
    
    @keyframes fadeInUp {
        from {
            opacity: 0;
            transform: translateY(30px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }
    
    @keyframes slideInRight {
        from {
            opacity: 0;
            transform: translateX(30px);
        }
        to {
            opacity: 1;
            transform: translateX(0);
        }
    }
    
    .fade-in-up {
        animation: fadeInUp 0.8s ease-out;
    }
    
    .slide-in-right {
        animation: slideInRight 0.8s ease-out;
    }
    
    @media (max-width: 768px) {
        .customer-profile-header {
            padding: 1.5rem;
            text-align: center;
        }
        
        .customer-avatar-large {
            width: 80px;
            height: 80px;
            font-size: 2rem;
            margin: 0 auto 1rem;
        }
        
        .info-grid {
            grid-template-columns: 1fr;
        }
        
        .action-buttons-enhanced {
            flex-direction: column;
        }
        
        .action-btn-enhanced {
            min-width: auto;
            width: 100%;
        }
        
        .stats-container {
            padding: 1rem;
        }
        
        .stat-item {
            padding: 1rem;
        }
        
        .stat-number {
            font-size: 2rem;
        }
    }
    
    @media print {
        .action-buttons-enhanced,
        .quick-actions-sidebar,
        .no-print {
            display: none !important;
        }
        
        .professional-card {
            border: 1px solid #ddd !important;
            box-shadow: none !important;
            break-inside: avoid;
        }
        
        .customer-profile-header {
            background: #f8f9fa !important;
            color: #333 !important;
        }
    }
</style>
@endpush

@section('page-actions')
<div class="action-buttons-enhanced no-print">
    <a href="{{ route('customers.edit', $customer) }}" class="action-btn-enhanced btn btn-warning">
        <i class="fas fa-edit"></i>
        تعديل البيانات
    </a>
    <a href="{{ route('invoices.create', ['customer_id' => $customer->id]) }}" class="action-btn-enhanced btn btn-primary">
        <i class="fas fa-file-invoice"></i>
        إنشاء فاتورة
    </a>
    <a href="{{ route('customers.index') }}" class="action-btn-enhanced btn btn-secondary">
        <i class="fas fa-arrow-left"></i>
        العودة للقائمة
    </a>
    <div class="dropdown">
        <button type="button" class="action-btn-enhanced btn btn-outline-secondary dropdown-toggle" data-bs-toggle="dropdown">
            <i class="fas fa-ellipsis-v"></i>
            المزيد
        </button>
        <ul class="dropdown-menu shadow">
            <li>
                <button type="button" class="dropdown-item" onclick="exportCustomer()">
                    <i class="fas fa-download me-2"></i>
                    تصدير البيانات
                </button>
            </li>
            <li>
                <button type="button" class="dropdown-item" onclick="printCustomerInfo()">
                    <i class="fas fa-print me-2"></i>
                    طباعة البيانات
                </button>
            </li>
            <li><hr class="dropdown-divider"></li>
            <li>
                <button type="button" class="dropdown-item text-danger"
                        data-bs-toggle="modal"
                        data-bs-target="#deleteCustomerModal"
                        data-customer-id="{{ $customer->id }}"
                        data-customer-name="{{ $customer->name }}"
                        data-customer-type="{{ $customer->customer_type == 'individual' ? 'فرد' : 'شركة' }}">
                    <i class="fas fa-trash me-2"></i>
                    حذف العميل
                </button>
            </li>
        </ul>
    </div>
</div>
@endsection

@section('content')
<!-- رأس الملف الشخصي الاحترافي -->
<div class="customer-profile-header fade-in-up">
    <div class="row align-items-center">
        <div class="col-md-8">
            <div class="d-flex align-items-center">
                <div class="customer-avatar-large">
                    {{ strtoupper(substr($customer->name, 0, 1)) }}
                </div>
                <div>
                    <h1 class="mb-2">{{ $customer->name }}</h1>
                    <div class="mb-3">
                        @if($customer->customer_type == 'individual')
                            <span class="type-badge-large bg-info text-white">
                                <i class="fas fa-user me-2"></i>
                                عميل فرد
                            </span>
                        @else
                            <span class="type-badge-large bg-primary text-white">
                                <i class="fas fa-building me-2"></i>
                                عميل شركة
                            </span>
                        @endif
                        
                        @if($customer->is_active)
                            <span class="status-badge-large bg-success text-white ms-2">
                                <i class="fas fa-check-circle me-2"></i>
                                نشط
                            </span>
                        @else
                            <span class="status-badge-large bg-secondary text-white ms-2">
                                <i class="fas fa-pause-circle me-2"></i>
                                غير نشط
                            </span>
                        @endif
                    </div>
                    <p class="mb-0 opacity-75">
                        <i class="fas fa-map-marker-alt me-2"></i>
                        {{ $customer->getFullAddressAttribute() }}
                    </p>
                </div>
            </div>
        </div>
        <div class="col-md-4 text-md-end mt-3 mt-md-0">
            <div class="stats-container">
                <div class="row text-center">
                    <div class="col-6">
                        <div class="stat-item primary">
                            <span class="stat-number text-primary">{{ $customer->invoices->count() }}</span>
                            <div class="stat-label">عدد الفواتير</div>
                        </div>
                    </div>
                    <div class="col-6">
                        <div class="stat-item success">
                            <span class="stat-number text-success">{{ number_format($customer->invoices->sum('total_amount'), 0) }}</span>
                            <div class="stat-label">إجمالي المشتريات</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="row">
    <!-- المحتوى الرئيسي -->
    <div class="col-lg-8">
        <!-- المعلومات الأساسية -->
        <div class="professional-card fade-in-up">
            <div class="info-section">
                <div class="section-header">
                    <div class="section-icon">
                        <i class="fas fa-user"></i>
                    </div>
                    <h3 class="section-title">المعلومات الأساسية</h3>
                </div>
                
                <div class="info-grid">
                    <div class="info-item">
                        <div class="info-label">
                            <i class="fas fa-signature"></i>
                            اسم العميل
                        </div>
                        <div class="info-value">{{ $customer->name }}</div>
                    </div>
                    
                    <div class="info-item">
                        <div class="info-label">
                            <i class="fas fa-id-card"></i>
                            رقم التعريف
                        </div>
                        <div class="info-value">
                            @if($customer->customer_type == 'individual')
                                {{ $customer->national_id }}
                                <br><small class="text-muted">هوية وطنية</small>
                            @else
                                {{ $customer->vat_number }}
                                <br><small class="text-muted">رقم ضريبي</small>
                                @if($customer->cr_number)
                                    <br>{{ $customer->cr_number }}
                                    <br><small class="text-muted">سجل تجاري</small>
                                @endif
                            @endif
                        </div>
                    </div>
                    
                    <div class="info-item">
                        <div class="info-label">
                            <i class="fas fa-calendar-alt"></i>
                            تاريخ التسجيل
                        </div>
                        <div class="info-value">{{ $customer->created_at->format('Y-m-d') }}</div>
                    </div>
                    
                    <div class="info-item">
                        <div class="info-label">
                            <i class="fas fa-hashtag"></i>
                            معرف العميل
                        </div>
                        <div class="info-value">#{{ $customer->id }}</div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- معلومات الاتصال -->
        <div class="professional-card fade-in-up">
            <div class="info-section">
                <div class="section-header">
                    <div class="section-icon" style="background: linear-gradient(135deg, #27ae60, #229954);">
                        <i class="fas fa-phone"></i>
                    </div>
                    <h3 class="section-title">معلومات الاتصال</h3>
                </div>
                
                @if($customer->email || $customer->phone)
                    <div class="info-grid">
                        @if($customer->email)
                            <div class="info-item">
                                <div class="info-label">
                                    <i class="fas fa-envelope"></i>
                                    البريد الإلكتروني
                                </div>
                                <div class="info-value">
                                    <a href="mailto:{{ $customer->email }}" class="text-decoration-none">{{ $customer->email }}</a>
                                </div>
                            </div>
                        @endif
                        
                        @if($customer->phone)
                            <div class="info-item">
                                <div class="info-label">
                                    <i class="fas fa-phone"></i>
                                    رقم الهاتف
                                </div>
                                <div class="info-value">
                                    <a href="tel:{{ $customer->phone }}" class="text-decoration-none">{{ $customer->phone }}</a>
                                </div>
                            </div>
                        @endif
                    </div>
                @else
                    <div class="text-center text-muted py-5">
                        <i class="fas fa-info-circle fa-3x mb-3 opacity-50"></i>
                        <h5>لا توجد معلومات اتصال مسجلة</h5>
                        <p>يمكنك إضافة معلومات الاتصال عبر تعديل بيانات العميل</p>
                    </div>
                @endif
            </div>
        </div>
        
        <!-- العنوان الوطني -->
        <div class="professional-card fade-in-up">
            <div class="info-section">
                <div class="section-header">
                    <div class="section-icon" style="background: linear-gradient(135deg, #e74c3c, #c0392b);">
                        <i class="fas fa-map-marker-alt"></i>
                    </div>
                    <h3 class="section-title">العنوان الوطني</h3>
                </div>
                
                <div class="info-grid">
                    <div class="info-item">
                        <div class="info-label">
                            <i class="fas fa-city"></i>
                            المدينة والحي
                        </div>
                        <div class="info-value">{{ $customer->city }}، {{ $customer->district }}</div>
                    </div>
                    
                    <div class="info-item">
                        <div class="info-label">
                            <i class="fas fa-road"></i>
                            الشارع
                        </div>
                        <div class="info-value">{{ $customer->street }}</div>
                    </div>
                    
                    <div class="info-item">
                        <div class="info-label">
                            <i class="fas fa-building"></i>
                            رقم المبنى
                        </div>
                        <div class="info-value">{{ $customer->building_number }}</div>
                    </div>
                    
                    <div class="info-item">
                        <div class="info-label">
                            <i class="fas fa-plus"></i>
                            الرقم الإضافي
                        </div>
                        <div class="info-value">{{ $customer->additional_number }}</div>
                    </div>
                    
                    <div class="info-item">
                        <div class="info-label">
                            <i class="fas fa-mail-bulk"></i>
                            الرمز البريدي
                        </div>
                        <div class="info-value">{{ $customer->postal_code }}</div>
                    </div>
                    
                    @if($customer->unit_number)
                        <div class="info-item">
                            <div class="info-label">
                                <i class="fas fa-door-open"></i>
                                رقم الوحدة
                            </div>
                            <div class="info-value">{{ $customer->unit_number }}</div>
                        </div>
                    @endif
                </div>
                
                <div class="address-display">
                    <h6 class="mb-2">
                        <i class="fas fa-map me-2"></i>
                        العنوان الكامل
                    </h6>
                    <p class="mb-0 fs-5">{{ $customer->getFullAddressAttribute() }}</p>
                </div>
            </div>
        </div>
        
        <!-- تاريخ الفواتير -->
        @if($customer->invoices->count() > 0)
        <div class="professional-card fade-in-up">
            <div class="info-section">
                <div class="section-header">
                    <div class="section-icon" style="background: linear-gradient(135deg, #f39c12, #e67e22);">
                        <i class="fas fa-file-invoice"></i>
                    </div>
                    <h3 class="section-title">تاريخ الفواتير ({{ $customer->invoices->count() }})</h3>
                    <div class="ms-auto">
                        <a href="{{ route('invoices.index', ['customer_id' => $customer->id]) }}" class="btn btn-outline-primary rounded-pill">
                            <i class="fas fa-external-link-alt me-1"></i>
                            عرض الكل
                        </a>
                    </div>
                </div>
                
                <div class="table-responsive">
                    <table class="modern-table table">
                            <tr>
                                <th><i class="fas fa-hashtag me-2"></i>رقم الفاتورة</th>
                                <th><i class="fas fa-tag me-2"></i>النوع</th>
                                <th><i class="fas fa-calendar me-2"></i>التاريخ</th>
                                <th><i class="fas fa-money-bill me-2"></i>المبلغ</th>
                                <th><i class="fas fa-info-circle me-2"></i>الحالة</th>
                                <th><i class="fas fa-cogs me-2"></i>إجراءات</th>
                            </tr>
                        <tbody>
                            @foreach($customer->invoices->take(5) as $invoice)
                            <tr>
                                <td>
                                    <a href="{{ route('invoices.show', $invoice) }}" class="text-decoration-none fw-bold">
                                        {{ $invoice->invoice_number }}
                                    </a>
                                </td>
                                <td>
                                    @if($invoice->type == 'standard')
                                        <span class="badge bg-primary rounded-pill">
                                            <i class="fas fa-file-invoice me-1"></i>
                                            معيارية
                                        </span>
                                    @else
                                        <span class="badge bg-success rounded-pill">
                                            <i class="fas fa-file-invoice-dollar me-1"></i>
                                            مبسطة
                                        </span>
                                    @endif
                                </td>
                                <td>{{ $invoice->issue_date->format('Y-m-d') }}</td>
                                <td>
                                    <span class="fw-bold text-success">{{ number_format($invoice->total_amount, 2) }}</span>
                                    <small class="text-muted">ر.س</small>
                                </td>
                                <td>
                                    @switch($invoice->zatca_status)
                                        @case('sent')
                                            <span class="badge bg-success rounded-pill">
                                                <i class="fas fa-check-circle me-1"></i>
                                                مُرسلة
                                            </span>
                                            @break
                                        @case('failed')
                                            <span class="badge bg-danger rounded-pill">
                                                <i class="fas fa-times-circle me-1"></i>
                                                فشل
                                            </span>
                                            @break
                                        @default
                                            <span class="badge bg-warning rounded-pill">
                                                <i class="fas fa-clock me-1"></i>
                                                في الانتظار
                                            </span>
                                    @endswitch
                                </td>
                                <td>
                                    <div class="btn-group btn-group-sm">
                                        <a href="{{ route('invoices.show', $invoice) }}" class="btn btn-outline-primary" title="عرض">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                        <a href="{{ route('invoices.pdf', $invoice) }}" class="btn btn-outline-danger" title="PDF">
                                            <i class="fas fa-file-pdf"></i>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
                
                @if($customer->invoices->count() > 5)
                    <div class="text-center mt-3 p-3 bg-light rounded">
                        <i class="fas fa-info-circle text-primary me-2"></i>
                        <small class="text-muted">عرض 5 من أصل {{ $customer->invoices->count() }} فاتورة</small>
                    </div>
                @endif
            </div>
        </div>
        @endif
    </div>
    
    <!-- الشريط الجانبي -->
    <div class="col-lg-4 quick-actions-sidebar">
        <!-- إحصائيات متقدمة -->
        <div class="sidebar-card slide-in-right">
            <div class="section-header">
                <div class="section-icon" style="background: linear-gradient(135deg, #9b59b6, #8e44ad); width: 40px; height: 40px; font-size: 1.2rem;">
                    <i class="fas fa-chart-line"></i>
                </div>
                <h5 class="section-title" style="font-size: 1.2rem;">إحصائيات العميل</h5>
            </div>
            
            <div class="row text-center g-2">
                <div class="col-6">
                    <div class="stat-item primary">
                        <span class="stat-number text-primary">{{ $customer->invoices->count() }}</span>
                        <div class="stat-label">عدد الفواتير</div>
                    </div>
                </div>
                <div class="col-6">
                    <div class="stat-item success">
                        <span class="stat-number text-success">{{ number_format($customer->invoices->sum('total_amount'), 0) }}</span>
                        <div class="stat-label">إجمالي المشتريات</div>
                    </div>
                </div>
                <div class="col-6">
                    <div class="stat-item info">
                        <span class="stat-number text-info">{{ $customer->invoices->where('type', 'standard')->count() }}</span>
                        <div class="stat-label">فواتير معيارية</div>
                    </div>
                </div>
                <div class="col-6">
                    <div class="stat-item warning">
                        <span class="stat-number text-warning">{{ $customer->invoices->where('type', 'simplified')->count() }}</span>
                        <div class="stat-label">فواتير مبسطة</div>
                    </div>
                </div>
            </div>
            
            @if($customer->invoices->count() > 0)
                <hr class="my-3">
                <div class="text-center">
                    <div class="row">
                        <div class="col-6">
                            <small class="text-muted d-block">أول فاتورة</small>
                            <strong class="text-primary">{{ $customer->invoices->first()->created_at->format('Y-m-d') }}</strong>
                        </div>
                        <div class="col-6">
                            <small class="text-muted d-block">آخر فاتورة</small>
                            <strong class="text-success">{{ $customer->invoices->last()->created_at->diffForHumans() }}</strong>
                        </div>
                    </div>
                </div>
            @endif
        </div>
        
        <!-- إجراءات سريعة -->
        <div class="sidebar-card slide-in-right">
            <div class="section-header">
                <div class="section-icon" style="background: linear-gradient(135deg, #1abc9c, #16a085); width: 40px; height: 40px; font-size: 1.2rem;">
                    <i class="fas fa-bolt"></i>
                </div>
                <h5 class="section-title" style="font-size: 1.2rem;">إجراءات سريعة</h5>
            </div>
            
            <div class="d-grid gap-2">
                <a href="{{ route('invoices.create', ['customer_id' => $customer->id]) }}" class="quick-action-btn btn btn-primary">
                    <i class="fas fa-file-invoice"></i>
                    إنشاء فاتورة جديدة
                </a>
                
                @if($customer->email)
                    <a href="mailto:{{ $customer->email }}" class="quick-action-btn btn btn-info">
                        <i class="fas fa-envelope"></i>
                        إرسال بريد إلكتروني
                    </a>
                @endif
                
                @if($customer->phone)
                    <a href="tel:{{ $customer->phone }}" class="quick-action-btn btn btn-success">
                        <i class="fas fa-phone"></i>
                        اتصال هاتفي
                    </a>
                @endif
                
                <button type="button" class="quick-action-btn btn btn-warning" onclick="generateCustomerReport()">
                    <i class="fas fa-chart-bar"></i>
                    تقرير العميل
                </button>
            </div>
        </div>
        
        <!-- معلومات إضافية -->
        <div class="sidebar-card slide-in-right">
            <div class="section-header">
                <div class="section-icon" style="background: linear-gradient(135deg, #34495e, #2c3e50); width: 40px; height: 40px; font-size: 1.2rem;">
                    <i class="fas fa-info"></i>
                </div>
                <h5 class="section-title" style="font-size: 1.2rem;">معلومات إضافية</h5>
            </div>
            
            <div class="info-grid">
                <div class="info-item">
                    <div class="info-label">
                        <i class="fas fa-calendar-plus"></i>
                        تاريخ التسجيل
                    </div>
                    <div class="info-value">{{ $customer->created_at->format('Y-m-d H:i') }}</div>
                </div>
                
                @if($customer->updated_at != $customer->created_at)
                    <div class="info-item">
                        <div class="info-label">
                            <i class="fas fa-edit"></i>
                            آخر تحديث
                        </div>
                        <div class="info-value">{{ $customer->updated_at->diffForHumans() }}</div>
                    </div>
                @endif
                
                <div class="info-item">
                    <div class="info-label">
                        <i class="fas fa-calculator"></i>
                        متوسط قيمة الفاتورة
                    </div>
                    <div class="info-value">
                        @if($customer->invoices->count() > 0)
                            {{ number_format($customer->invoices->avg('total_amount'), 2) }} ر.س
                        @else
                            لا توجد فواتير
                        @endif
                    </div>
                </div>
            </div>
        </div>
        
        @if($customer->invoices->count() > 0)
        <!-- تحذير الحذف -->
        <div class="sidebar-card slide-in-right">
            <div class="section-header">
                <div class="section-icon" style="background: linear-gradient(135deg, #f39c12, #e67e22); width: 40px; height: 40px; font-size: 1.2rem;">
                    <i class="fas fa-exclamation-triangle"></i>
                </div>
                <h5 class="section-title text-warning" style="font-size: 1.2rem;">تنبيه هام</h5>
            </div>
            
            <div class="alert alert-warning border-0" style="background: linear-gradient(135deg, #fff3cd 0%, #ffeaa7 100%);">
                <div class="d-flex align-items-start">
                    <i class="fas fa-shield-alt text-warning me-2 mt-1"></i>
                    <div>
                        <small class="fw-bold d-block">حماية البيانات المالية</small>
                        <small class="text-muted">
                            هذا العميل لديه {{ $customer->invoices->count() }} فاتورة.
                            لا يمكن حذفه لحماية سلامة البيانات المالية.
                        </small>
                    </div>
                </div>
            </div>
        </div>
        @endif
    </div>
</div>
@endsection

@push('scripts')
<script>
// تأثيرات التحريك عند التحميل
document.addEventListener('DOMContentLoaded', function() {
    // تأثير التحريك للبطاقات
    const cards = document.querySelectorAll('.professional-card');
    cards.forEach((card, index) => {
        card.style.animationDelay = `${index * 0.2}s`;
    });
    
    // تأثيرات التحريك للشريط الجانبي
    const sidebarCards = document.querySelectorAll('.sidebar-card');
    sidebarCards.forEach((card, index) => {
        card.style.animationDelay = `${index * 0.3}s`;
    });
    
    // تأثيرات hover للإحصائيات
    const statItems = document.querySelectorAll('.stat-item');
    statItems.forEach(item => {
        item.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-10px) scale(1.05)';
        });
        
        item.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(-5px) scale(1)';
        });
    });
    
    // تحسين تجربة الأزرار
    const actionButtons = document.querySelectorAll('.quick-action-btn, .action-btn-enhanced');
    actionButtons.forEach(button => {
        button.addEventListener('mouseenter', function() {
            this.style.transform = 'translateX(-8px) scale(1.02)';
        });
        
        button.addEventListener('mouseleave', function() {
            this.style.transform = 'translateX(-5px) scale(1)';
        });
    });
    
    // تأثيرات للجدول
    const tableRows = document.querySelectorAll('.modern-table tbody tr');
    tableRows.forEach((row, index) => {
        row.style.animationDelay = `${index * 0.1}s`;
        row.classList.add('fade-in-up');
    });
});

function exportCustomer() {
    // إظهار نافذة تصدير محسنة
    const exportModal = `
        <div class="modal fade" id="exportCustomerModal" tabindex="-1">
            <div class="modal-dialog modal-dialog-centered">
                <div class="modal-content border-0 shadow-lg" style="border-radius: 20px;">
                    <div class="modal-header" style="background: linear-gradient(135deg, #3498db, #2980b9); border-radius: 20px 20px 0 0;">
                        <h5 class="modal-title text-white">
                            <i class="fas fa-download me-2"></i>
                            تصدير بيانات العميل
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body text-center py-4">
                        <div class="mb-4">
                            <div class="customer-avatar-large mx-auto" style="width: 80px; height: 80px; font-size: 2rem;">
                                {{ strtoupper(substr($customer->name, 0, 1)) }}
                            </div>
                            <h6 class="mt-3 mb-1">{{ $customer->name }}</h6>
                            <small class="text-muted">{{ $customer->getFullAddressAttribute() }}</small>
                        </div>
                        
                        <p class="text-muted mb-4">اختر صيغة التصدير المرغوبة:</p>
                        
                        <div class="d-grid gap-3">
                            <button class="btn btn-outline-success d-flex align-items-center justify-content-center py-3" onclick="exportToFormat('excel')" style="border-radius: 15px;">
                                <i class="fas fa-file-excel fa-2x me-3 text-success"></i>
                                <div class="text-start">
                                    <div class="fw-bold">تصدير Excel</div>
                                    <small class="text-muted">ملف جداول بيانات كامل</small>
                                </div>
                            </button>
                            
                            <button class="btn btn-outline-danger d-flex align-items-center justify-content-center py-3" onclick="exportToFormat('pdf')" style="border-radius: 15px;">
                                <i class="fas fa-file-pdf fa-2x me-3 text-danger"></i>
                                <div class="text-start">
                                    <div class="fw-bold">تصدير PDF</div>
                                    <small class="text-muted">ملف قابل للطباعة</small>
                                </div>
                            </button>
                            
                            <button class="btn btn-outline-info d-flex align-items-center justify-content-center py-3" onclick="exportToFormat('json')" style="border-radius: 15px;">
                                <i class="fas fa-file-code fa-2x me-3 text-info"></i>
                                <div class="text-start">
                                    <div class="fw-bold">تصدير JSON</div>
                                    <small class="text-muted">بيانات منظمة للتطبيقات</small>
                                </div>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    // إضافة النافذة وإظهارها
    if (!document.getElementById('exportCustomerModal')) {
        document.body.insertAdjacentHTML('beforeend', exportModal);
    }
    
    const modal = new bootstrap.Modal(document.getElementById('exportCustomerModal'));
    modal.show();
}

function exportToFormat(format) {
    const customerData = {
        basic_info: {
            name: '{{ $customer->name }}',
            type: '{{ $customer->customer_type }}',
            identification: '{{ $customer->customer_type == "individual" ? $customer->national_id : $customer->vat_number }}',
            status: '{{ $customer->is_active ? "نشط" : "غير نشط" }}',
            registration_date: '{{ $customer->created_at->format('Y-m-d H:i') }}'
        },
        contact_info: {
            email: '{{ $customer->email }}',
            phone: '{{ $customer->phone }}'
        },
        address: {
            full_address: '{{ $customer->getFullAddressAttribute() }}',
            city: '{{ $customer->city }}',
            district: '{{ $customer->district }}',
            street: '{{ $customer->street }}',
            building_number: '{{ $customer->building_number }}',
            additional_number: '{{ $customer->additional_number }}',
            postal_code: '{{ $customer->postal_code }}'
        },
        statistics: {
            invoices_count: {{ $customer->invoices->count() }},
            total_purchases: {{ $customer->invoices->sum('total_amount') }},
            standard_invoices: {{ $customer->invoices->where('type', 'standard')->count() }},
            simplified_invoices: {{ $customer->invoices->where('type', 'simplified')->count() }},
            average_invoice_value: {{ $customer->invoices->count() > 0 ? $customer->invoices->avg('total_amount') : 0 }}
        }
    };
    
    switch(format) {
        case 'json':
            const dataStr = JSON.stringify(customerData, null, 2);
            const dataBlob = new Blob([dataStr], {type: 'application/json'});
            const url = URL.createObjectURL(dataBlob);
            const link = document.createElement('a');
            link.href = url;
            link.download = 'customer_{{ $customer->id }}_data.json';
            link.click();
            URL.revokeObjectURL(url);
            break;
            
        case 'excel':
            showNotification('سيتم تطوير تصدير Excel قريباً', 'info');
            break;
            
        case 'pdf':
            showNotification('سيتم تطوير تصدير PDF قريباً', 'info');
            break;
    }
    
    bootstrap.Modal.getInstance(document.getElementById('exportCustomerModal')).hide();
}

function printCustomerInfo() {
    // تحسين تجربة الطباعة
    showNotification('جاري تحضير الصفحة للطباعة...', 'info');
    
    setTimeout(() => {
        window.print();
    }, 500);
}

function generateCustomerReport() {
    // نافذة تقرير محسنة
    const reportModal = `
        <div class="modal fade" id="reportModal" tabindex="-1">
            <div class="modal-dialog modal-lg modal-dialog-centered">
                <div class="modal-content border-0 shadow-lg" style="border-radius: 20px;">
                    <div class="modal-header" style="background: linear-gradient(135deg, #f39c12, #e67e22); border-radius: 20px 20px 0 0;">
                        <h5 class="modal-title text-white">
                            <i class="fas fa-chart-bar me-2"></i>
                            تقرير العميل المفصل
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body p-4">
                        <div class="text-center mb-4">
                            <i class="fas fa-chart-line fa-4x text-primary mb-3"></i>
                            <h5>قريباً - ميزة التقارير المتقدمة</h5>
                            <p class="text-muted">سيتم إطلاق نظام التقارير المتقدم الذي سيشمل:</p>
                        </div>
                        
                        <div class="row g-3">
                            <div class="col-md-6">
                                <div class="d-flex align-items-center p-3 bg-light rounded">
                                    <i class="fas fa-chart-pie text-primary me-3"></i>
                                    <div>
                                        <div class="fw-bold">تحليل المبيعات</div>
                                        <small class="text-muted">رسوم بيانية تفاعلية</small>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="d-flex align-items-center p-3 bg-light rounded">
                                    <i class="fas fa-calendar-alt text-success me-3"></i>
                                    <div>
                                        <div class="fw-bold">التقارير الزمنية</div>
                                        <small class="text-muted">تحليل بالفترات الزمنية</small>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="d-flex align-items-center p-3 bg-light rounded">
                                    <i class="fas fa-file-export text-warning me-3"></i>
                                    <div>
                                        <div class="fw-bold">تصدير متقدم</div>
                                        <small class="text-muted">تقارير بصيغ متعددة</small>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="d-flex align-items-center p-3 bg-light rounded">
                                    <i class="fas fa-bell text-info me-3"></i>
                                    <div>
                                        <div class="fw-bold">التنبيهات الذكية</div>
                                        <small class="text-muted">إشعارات تلقائية</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    if (!document.getElementById('reportModal')) {
        document.body.insertAdjacentHTML('beforeend', reportModal);
    }
    
    const modal = new bootstrap.Modal(document.getElementById('reportModal'));
    modal.show();
}

function showNotification(message, type = 'success') {
    const notification = `
        <div class="toast align-items-center text-white bg-${type} border-0 position-fixed"
             style="top: 20px; left: 20px; z-index: 9999; border-radius: 10px;" role="alert">
            <div class="d-flex">
                <div class="toast-body">
                    <i class="fas fa-info-circle me-2"></i>
                    ${message}
                </div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
            </div>
        </div>
    `;
    
    document.body.insertAdjacentHTML('beforeend', notification);
    const toast = new bootstrap.Toast(document.querySelector('.toast:last-child'));
    toast.show();
    
    // إزالة العنصر بعد إخفائه
    setTimeout(() => {
        const toastElement = document.querySelector('.toast:last-child');
        if (toastElement) {
            toastElement.remove();
        }
    }, 4000);
}

// تحسين الأداء - Lazy Loading للصور (إن وجدت)
document.addEventListener('DOMContentLoaded', function() {
    const images = document.querySelectorAll('img[data-src]');
    const imageObserver = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const img = entry.target;
                img.src = img.dataset.src;
                img.classList.remove('lazy');
                imageObserver.unobserve(img);
            }
        });
    });
    
    images.forEach(img => imageObserver.observe(img));
});
</script>

<!-- Modal تأكيد حذف العميل -->
<div class="modal fade" id="deleteCustomerModal" tabindex="-1" aria-labelledby="deleteCustomerModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title" id="deleteCustomerModalLabel">
                    <i class="fas fa-exclamation-triangle"></i>
                    تأكيد حذف العميل
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="إغلاق"></button>
            </div>
            <div class="modal-body">
                <div class="alert alert-warning">
                    <i class="fas fa-exclamation-triangle"></i>
                    <strong>تحذير:</strong> هذا الإجراء لا يمكن التراجع عنه!
                </div>
                <p class="mb-3">هل تريد حذف هذا العميل؟</p>
                <div class="card">
                    <div class="card-body">
                        <h6 class="card-title">تفاصيل العميل:</h6>
                        <p class="mb-1"><strong>الاسم:</strong> <span id="customerNameToDelete"></span></p>
                        <p class="mb-0"><strong>النوع:</strong> <span id="customerTypeToDelete"></span></p>
                    </div>
                </div>
                <div class="alert alert-info mt-3">
                    <i class="fas fa-info-circle"></i>
                    <strong>ملاحظة:</strong> سيتم حذف جميع البيانات المرتبطة بهذا العميل بما في ذلك الفواتير والمعاملات المالية.
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <i class="fas fa-times"></i>
                    إلغاء
                </button>
                <form id="deleteCustomerForm" method="POST" class="d-inline">
                    @csrf
                    @method('DELETE')
                    <button type="submit" class="btn btn-danger">
                        <i class="fas fa-trash"></i>
                        تأكيد الحذف
                    </button>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const deleteModal = document.getElementById('deleteCustomerModal');
    const deleteForm = document.getElementById('deleteCustomerForm');
    const customerName = document.getElementById('customerNameToDelete');
    const customerType = document.getElementById('customerTypeToDelete');

    deleteModal.addEventListener('show.bs.modal', function (event) {
        const button = event.relatedTarget;
        const customerId = button.getAttribute('data-customer-id');
        const customerNameValue = button.getAttribute('data-customer-name');
        const customerTypeValue = button.getAttribute('data-customer-type');

        // تحديث النموذج
        deleteForm.action = `{{ url('customers') }}/${customerId}`;
        
        // تحديث محتوى المودال
        customerName.textContent = customerNameValue;
        customerType.textContent = customerTypeValue;
    });
});
</script>
@endpush

@push('styles')
<style>
@media print {
    .btn, .dropdown, .no-print {
        display: none !important;
    }
    
    .card {
        border: 1px solid #ddd !important;
        box-shadow: none !important;
    }
}
</style>
@endpush