@extends('layouts.app')

@section('title', 'العملاء')
@section('page-title', 'إدارة العملاء')

@push('styles')
<style>
    /* ========== Enhanced Professional Styling ========== */
    :root {
        --customers-primary: #3b82f6;
        --customers-success: #10b981;
        --customers-warning: #f59e0b;
        --customers-danger: #ef4444;
        --customers-info: #06b6d4;
        --customers-light: #f8fafc;
        --customers-dark: #1e293b;
        --customers-gray-100: #f1f5f9;
        --customers-gray-200: #e2e8f0;
        --customers-gray-300: #cbd5e1;
        --customers-gray-500: #64748b;
        --customers-gray-600: #475569;
        --customers-gray-700: #334155;
        
        --customers-shadow-sm: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
        --customers-shadow-md: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        --customers-shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
        --customers-radius-md: 12px;
        --customers-radius-lg: 16px;
        --customers-radius-xl: 20px;
    }

    /* Professional Page Header */
    .customers-header {
        background: linear-gradient(135deg, var(--customers-primary) 0%, #1d4ed8 100%);
        border-radius: var(--customers-radius-xl);
        padding: 2rem;
        margin-bottom: 2rem;
        color: white;
        box-shadow: var(--customers-shadow-lg);
        position: relative;
        overflow: hidden;
    }

    .customers-header::before {
        content: '';
        position: absolute;
        top: -50%;
        right: -50%;
        width: 200%;
        height: 200%;
        background: radial-gradient(circle, rgba(255,255,255,0.1) 0%, transparent 70%);
        transform: rotate(45deg);
        pointer-events: none;
    }

    .customers-header .row {
        position: relative;
        z-index: 1;
    }

    .customers-header h2 {
        font-size: 2rem;
        font-weight: 700;
        margin-bottom: 0.5rem;
        text-shadow: 0 2px 4px rgba(0,0,0,0.1);
    }

    .customers-header p {
        font-size: 1.1rem;
        opacity: 0.9;
        margin-bottom: 0;
    }

    .filter-toggle-btn {
        background: rgba(255, 255, 255, 0.15);
        border: 1px solid rgba(255, 255, 255, 0.2);
        color: white;
        backdrop-filter: blur(10px);
        border-radius: var(--customers-radius-md);
        padding: 0.5rem 1rem;
        font-weight: 500;
        transition: all 0.3s ease;
    }

    .filter-toggle-btn:hover {
        background: rgba(255, 255, 255, 0.25);
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        color: white;
    }

    /* Professional Card Container */
    .professional-card {
        background: white;
        border-radius: var(--customers-radius-xl);
        box-shadow: var(--customers-shadow-md);
        overflow: hidden;
        border: 1px solid var(--customers-gray-200);
        margin-bottom: 2rem;
    }

    /* Enhanced Filter Section */
    .filter-section {
        background: linear-gradient(135deg, var(--customers-gray-100) 0%, white 100%);
        padding: 2rem;
        border-bottom: 2px solid var(--customers-gray-200);
        margin: 0;
    }

    .filter-section .form-label {
        color: var(--customers-gray-700);
        font-weight: 600;
        margin-bottom: 0.5rem;
        font-size: 0.9rem;
    }

    .filter-section .form-select,
    .filter-section .form-control {
        border: 2px solid var(--customers-gray-300);
        border-radius: var(--customers-radius-md);
        padding: 0.75rem 1rem;
        font-size: 0.95rem;
        transition: all 0.3s ease;
        background: white;
    }

    .filter-section .form-select:focus,
    .filter-section .form-control:focus {
        border-color: var(--customers-primary);
        box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        outline: none;
    }

    .search-box {
        position: relative;
    }

    .search-box .search-icon {
        position: absolute;
        left: 15px;
        top: 50%;
        transform: translateY(-50%);
        color: var(--customers-gray-500);
        z-index: 2;
    }

    .search-box .form-control {
        padding-left: 3rem;
    }

    /* Enhanced Table Styling */
    .advanced-table {
        margin: 0;
        font-size: 0.95rem;
    }

    .advanced-table th {
        background: linear-gradient(135deg, var(--customers-gray-100) 0%, var(--customers-gray-200) 100%);
        color: var(--customers-gray-700);
        font-weight: 600;
        padding: 1.25rem 1rem;
        border: none;
        font-size: 0.9rem;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        position: sticky;
        top: 0;
        z-index: 10;
    }

    .advanced-table td {
        padding: 1.25rem 1rem;
        border-bottom: 1px solid var(--customers-gray-200);
        vertical-align: middle;
        background: white;
        transition: all 0.3s ease;
    }

    .advanced-table tbody tr:hover {
        background: linear-gradient(135deg, rgba(59, 130, 246, 0.02) 0%, rgba(59, 130, 246, 0.05) 100%);
        transform: translateY(-1px);
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
    }

    /* Professional Customer Info */
    .customer-info {
        display: flex;
        align-items: center;
        gap: 1rem;
    }

    .customer-avatar {
        width: 48px;
        height: 48px;
        background: linear-gradient(135deg, var(--customers-primary) 0%, #1d4ed8 100%);
        color: white;
        border-radius: 12px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: 700;
        font-size: 1.2rem;
        box-shadow: var(--customers-shadow-md);
        flex-shrink: 0;
    }

    /* Enhanced Badges */
    .type-badge,
    .status-badge {
        padding: 0.4rem 0.8rem;
        border-radius: 8px;
        font-size: 0.8rem;
        font-weight: 600;
        display: inline-flex;
        align-items: center;
        gap: 0.3rem;
        box-shadow: var(--customers-shadow-sm);
        text-transform: uppercase;
        letter-spacing: 0.3px;
    }

    .type-badge.bg-info {
        background: linear-gradient(135deg, var(--customers-info) 0%, #0891b2 100%) !important;
    }

    .type-badge.bg-primary {
        background: linear-gradient(135deg, var(--customers-primary) 0%, #1d4ed8 100%) !important;
    }

    .status-badge.bg-success {
        background: linear-gradient(135deg, var(--customers-success) 0%, #059669 100%) !important;
    }

    .status-badge.bg-secondary {
        background: linear-gradient(135deg, var(--customers-gray-500) 0%, var(--customers-gray-600) 100%) !important;
    }

    /* Enhanced Action Buttons */
    .action-buttons {
        display: flex;
        gap: 0.5rem;
        align-items: center;
    }

    .action-btn {
        width: 36px;
        height: 36px;
        border-radius: 10px;
        display: flex;
        align-items: center;
        justify-content: center;
        border: 2px solid;
        font-size: 0.85rem;
        transition: all 0.3s ease;
        position: relative;
        overflow: hidden;
    }

    .action-btn::before {
        content: '';
        position: absolute;
        top: 0;
        left: -100%;
        width: 100%;
        height: 100%;
        background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
        transition: all 0.5s ease;
    }

    .action-btn:hover::before {
        left: 100%;
    }

    .action-btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    }

    /* Enhanced Statistics Cards */
    .stats-cards {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 1.5rem;
        margin-top: 2rem;
    }

    .stat-card {
        background: white;
        padding: 2rem;
        border-radius: var(--customers-radius-lg);
        box-shadow: var(--customers-shadow-md);
        border: 1px solid var(--customers-gray-200);
        text-align: center;
        position: relative;
        overflow: hidden;
        transition: all 0.3s ease;
    }

    .stat-card::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 4px;
        background: linear-gradient(90deg, var(--customers-primary) 0%, #1d4ed8 100%);
    }

    .stat-card.individuals::before {
        background: linear-gradient(90deg, var(--customers-info) 0%, #0891b2 100%);
    }

    .stat-card.businesses::before {
        background: linear-gradient(90deg, var(--customers-primary) 0%, #1d4ed8 100%);
    }

    .stat-card.active::before {
        background: linear-gradient(90deg, var(--customers-success) 0%, #059669 100%);
    }

    .stat-card.total::before {
        background: linear-gradient(90deg, var(--customers-warning) 0%, #d97706 100%);
    }

    .stat-card:hover {
        transform: translateY(-4px);
        box-shadow: var(--customers-shadow-lg);
    }

    .stat-number {
        font-size: 2.5rem;
        font-weight: 800;
        display: block;
        margin-bottom: 0.5rem;
    }

    .stat-label {
        font-size: 1.1rem;
        font-weight: 600;
        color: var(--customers-gray-700);
        margin-bottom: 0.5rem;
    }

    /* Professional Empty State */
    .empty-state {
        text-align: center;
        padding: 4rem 2rem;
        color: var(--customers-gray-600);
    }

    .empty-state-icon {
        font-size: 4rem;
        color: var(--customers-gray-400);
        margin-bottom: 1.5rem;
    }

    .empty-state h5 {
        color: var(--customers-gray-700);
        margin-bottom: 1rem;
    }

    /* ========== Ultra Professional Pagination Design ========== */
    .modern-pagination {
        background: linear-gradient(145deg,
            rgba(255, 255, 255, 0.95) 0%,
            rgba(248, 250, 252, 0.98) 50%,
            rgba(255, 255, 255, 0.95) 100%
        );
        backdrop-filter: blur(20px);
        border-radius: 24px;
        padding: 3rem 2.5rem;
        margin: 3rem 0;
        box-shadow:
            0 25px 50px -12px rgba(0, 0, 0, 0.08),
            0 0 0 1px rgba(255, 255, 255, 0.05),
            inset 0 1px 0 rgba(255, 255, 255, 0.9),
            inset 0 -1px 0 rgba(0, 0, 0, 0.02);
        border: 1px solid rgba(226, 232, 240, 0.6);
        position: relative;
        overflow: hidden;
    }

    .modern-pagination::before {
        content: '';
        position: absolute;
        top: 0;
        left: -100%;
        width: 100%;
        height: 2px;
        background: linear-gradient(
            90deg,
            transparent 0%,
            #3b82f6 20%,
            #10b981 40%,
            #f59e0b 60%,
            #ef4444 80%,
            transparent 100%
        );
        animation: rainbowShimmer 4s ease-in-out infinite;
    }

    @keyframes rainbowShimmer {
        0%, 100% { left: -100%; opacity: 0; }
        50% { left: 100%; opacity: 1; }
    }

    .modern-pagination::after {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background:
            radial-gradient(circle at 25% 25%, rgba(59, 130, 246, 0.05) 0%, transparent 50%),
            radial-gradient(circle at 75% 75%, rgba(16, 185, 129, 0.05) 0%, transparent 50%);
        pointer-events: none;
        z-index: 1;
    }

    .pagination-info {
        position: relative;
        z-index: 2;
        background: linear-gradient(135deg,
            rgba(59, 130, 246, 0.12) 0%,
            rgba(16, 185, 129, 0.12) 50%,
            rgba(245, 158, 11, 0.12) 100%
        );
        color: var(--customers-gray-700);
        font-size: 1rem;
        font-weight: 600;
        padding: 1rem 2rem;
        border-radius: 60px;
        display: inline-block;
        margin-bottom: 2rem;
        border: 2px solid transparent;
        background-clip: padding-box;
        box-shadow:
            0 8px 32px rgba(0, 0, 0, 0.08),
            inset 0 1px 0 rgba(255, 255, 255, 0.7);
        position: relative;
        overflow: hidden;
    }

    .pagination-info::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: linear-gradient(135deg,
            rgba(59, 130, 246, 0.15) 0%,
            rgba(16, 185, 129, 0.15) 50%,
            rgba(245, 158, 11, 0.15) 100%
        );
        border-radius: inherit;
        z-index: -1;
    }

    .pagination-info::after {
        content: '';
        position: absolute;
        top: 1px;
        left: 1px;
        right: 1px;
        bottom: 1px;
        background: inherit;
        border-radius: calc(60px - 1px);
        z-index: -1;
    }

    .pagination-controls {
        position: relative;
        z-index: 2;
        display: flex;
        align-items: center;
        justify-content: center;
        margin-bottom: 2rem;
    }

    .modern-pagination .pagination {
        margin: 0;
        display: flex;
        list-style: none;
        padding: 0;
        gap: 0.75rem;
        align-items: center;
        justify-content: center;
        flex-wrap: wrap;
    }

    .modern-pagination .page-item {
        margin: 0;
    }

    .modern-pagination .page-link {
        display: inline-flex;
        align-items: center;
        justify-content: center;
        min-width: 56px;
        height: 56px;
        padding: 0.875rem 1.125rem;
        color: var(--customers-gray-700);
        text-decoration: none;
        font-weight: 700;
        font-size: 1rem;
        border-radius: 18px;
        transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
        position: relative;
        overflow: hidden;
        background:
            linear-gradient(145deg, #ffffff 0%, #f8fafc 100%);
        border: 2px solid transparent;
        box-shadow:
            0 8px 25px rgba(0, 0, 0, 0.08),
            0 3px 10px rgba(0, 0, 0, 0.04),
            inset 0 1px 0 rgba(255, 255, 255, 0.9),
            inset 0 -1px 0 rgba(0, 0, 0, 0.02);
    }

    .modern-pagination .page-link::before {
        content: '';
        position: absolute;
        top: 0;
        left: -120%;
        width: 120%;
        height: 100%;
        background: linear-gradient(
            120deg,
            transparent 0%,
            rgba(59, 130, 246, 0.15) 40%,
            rgba(16, 185, 129, 0.15) 60%,
            transparent 100%
        );
        transition: all 0.8s cubic-bezier(0.19, 1, 0.22, 1);
        z-index: 1;
        transform: skewX(-15deg);
    }

    .modern-pagination .page-link::after {
        content: '';
        position: absolute;
        top: 50%;
        left: 50%;
        width: 0;
        height: 0;
        background: radial-gradient(circle, rgba(59, 130, 246, 0.25) 0%, transparent 70%);
        border-radius: 50%;
        transform: translate(-50%, -50%);
        transition: all 0.5s cubic-bezier(0.68, -0.55, 0.265, 1.55);
        z-index: 1;
    }

    .modern-pagination .page-link:hover {
        color: white;
        background: linear-gradient(145deg,
            var(--customers-primary) 0%,
            #1d4ed8 50%,
            var(--customers-info) 100%
        );
        border-color: rgba(59, 130, 246, 0.4);
        transform: translateY(-4px) scale(1.08) rotate(1deg);
        box-shadow:
            0 20px 40px rgba(59, 130, 246, 0.3),
            0 15px 25px rgba(16, 185, 129, 0.1),
            0 5px 15px rgba(0, 0, 0, 0.1),
            inset 0 1px 0 rgba(255, 255, 255, 0.3);
    }

    .modern-pagination .page-link:hover::before {
        left: 120%;
    }

    .modern-pagination .page-link:hover::after {
        width: 120%;
        height: 120%;
    }

    .modern-pagination .page-link:active {
        transform: translateY(-2px) scale(1.05);
    }

    .modern-pagination .page-item.active .page-link {
        background: linear-gradient(145deg,
            var(--customers-primary) 0%,
            #1d4ed8 30%,
            var(--customers-info) 70%,
            var(--customers-success) 100%
        );
        color: white;
        border-color: rgba(59, 130, 246, 0.6);
        transform: translateY(-2px) scale(1.1);
        box-shadow:
            0 25px 50px rgba(59, 130, 246, 0.4),
            0 15px 30px rgba(16, 185, 129, 0.2),
            0 5px 15px rgba(0, 0, 0, 0.15),
            inset 0 2px 0 rgba(255, 255, 255, 0.4),
            inset 0 -2px 0 rgba(0, 0, 0, 0.1);
        position: relative;
        z-index: 3;
    }

    .modern-pagination .page-item.active .page-link::before {
        background: linear-gradient(
            120deg,
            transparent 0%,
            rgba(255, 255, 255, 0.3) 40%,
            rgba(255, 255, 255, 0.4) 60%,
            transparent 100%
        );
        animation: activePulse 3s ease-in-out infinite;
    }

    @keyframes activePulse {
        0%, 100% { left: -120%; }
        50% { left: 120%; }
    }

    .modern-pagination .page-item.active .page-link::after {
        width: 100%;
        height: 100%;
        background: radial-gradient(circle, rgba(255, 255, 255, 0.2) 0%, transparent 70%);
        animation: activeGlow 2s ease-in-out infinite alternate;
    }

    @keyframes activeGlow {
        0% { opacity: 0.5; transform: translate(-50%, -50%) scale(0.9); }
        100% { opacity: 1; transform: translate(-50%, -50%) scale(1.1); }
    }

    .modern-pagination .page-item.disabled .page-link {
        color: var(--customers-gray-400);
        background: linear-gradient(145deg, #f8fafc 0%, #f1f5f9 100%);
        border-color: var(--customers-gray-200);
        cursor: not-allowed;
        box-shadow:
            0 2px 8px rgba(0, 0, 0, 0.04),
            inset 0 1px 0 rgba(255, 255, 255, 0.7);
        opacity: 0.5;
    }

    .modern-pagination .page-item.disabled .page-link:hover {
        color: var(--customers-gray-400);
        background: linear-gradient(145deg, #f8fafc 0%, #f1f5f9 100%);
        border-color: var(--customers-gray-200);
        transform: none;
        box-shadow:
            0 2px 8px rgba(0, 0, 0, 0.04),
            inset 0 1px 0 rgba(255, 255, 255, 0.7);
    }

    .modern-pagination .page-item.disabled .page-link::before,
    .modern-pagination .page-item.disabled .page-link::after {
        display: none;
    }

    /* Enhanced Navigation Arrows */
    .modern-pagination .page-link[rel="prev"],
    .modern-pagination .page-link[rel="next"] {
        background: linear-gradient(145deg,
            var(--customers-gray-100) 0%,
            white 50%,
            var(--customers-gray-100) 100%
        );
        border-color: var(--customers-gray-300);
        font-size: 1.2rem;
        min-width: 64px;
        height: 64px;
        border-radius: 20px;
    }

    .modern-pagination .page-link[rel="prev"]:hover,
    .modern-pagination .page-link[rel="next"]:hover {
        background: linear-gradient(145deg,
            var(--customers-primary) 0%,
            #1d4ed8 50%,
            var(--customers-info) 100%
        );
        color: white;
        border-color: var(--customers-primary);
        transform: translateY(-4px) scale(1.1);
        box-shadow:
            0 20px 40px rgba(59, 130, 246, 0.35),
            0 15px 25px rgba(0, 0, 0, 0.1);
    }

    .pagination-summary {
        position: relative;
        z-index: 2;
        font-size: 0.9rem;
        color: var(--customers-gray-600);
        font-weight: 600;
        padding: 0.75rem 1.5rem;
        background: linear-gradient(135deg,
            rgba(241, 245, 249, 0.9) 0%,
            rgba(255, 255, 255, 0.9) 100%
        );
        border-radius: 30px;
        border: 1px solid rgba(226, 232, 240, 0.8);
        backdrop-filter: blur(10px);
        box-shadow: 0 4px 15px rgba(0, 0, 0, 0.06);
        display: inline-block;
    }

    /* ========== Ultra Responsive Mobile Design ========== */
    @media (max-width: 768px) {
        .modern-pagination {
            padding: 2rem 1.5rem;
            margin: 2rem 0;
            border-radius: 20px;
        }

        .pagination-info {
            font-size: 0.9rem;
            padding: 0.75rem 1.5rem;
            margin-bottom: 1.5rem;
            border-radius: 50px;
        }

        .modern-pagination .pagination {
            gap: 0.5rem;
        }

        .modern-pagination .page-link {
            min-width: 48px;
            height: 48px;
            padding: 0.75rem 1rem;
            font-size: 0.9rem;
            border-radius: 14px;
        }

        .modern-pagination .page-link[rel="prev"],
        .modern-pagination .page-link[rel="next"] {
            min-width: 56px;
            height: 56px;
            border-radius: 16px;
            font-size: 1.1rem;
        }

        .pagination-summary {
            font-size: 0.85rem;
            padding: 0.6rem 1.2rem;
        }
    }

    @media (max-width: 640px) {
        .modern-pagination {
            padding: 1.5rem 1rem;
            margin: 1.5rem 0;
        }

        .pagination-info {
            font-size: 0.85rem;
            padding: 0.6rem 1.2rem;
            margin-bottom: 1.2rem;
        }

        .modern-pagination .pagination {
            gap: 0.35rem;
        }

        .modern-pagination .page-link {
            min-width: 42px;
            height: 42px;
            padding: 0.6rem 0.8rem;
            font-size: 0.85rem;
            border-radius: 12px;
        }

        .modern-pagination .page-link[rel="prev"],
        .modern-pagination .page-link[rel="next"] {
            min-width: 48px;
            height: 48px;
            font-size: 1rem;
            border-radius: 14px;
        }

        .pagination-summary {
            font-size: 0.8rem;
            padding: 0.5rem 1rem;
        }
    }

    @media (max-width: 480px) {
        .modern-pagination {
            padding: 1.2rem 0.8rem;
        }

        .pagination-info {
            font-size: 0.8rem;
            padding: 0.5rem 1rem;
            margin-bottom: 1rem;
        }

        .modern-pagination .pagination {
            gap: 0.25rem;
        }

        .modern-pagination .page-link {
            min-width: 38px;
            height: 38px;
            padding: 0.5rem 0.7rem;
            font-size: 0.8rem;
            border-radius: 10px;
        }

        .modern-pagination .page-link[rel="prev"],
        .modern-pagination .page-link[rel="next"] {
            min-width: 42px;
            height: 42px;
            font-size: 0.9rem;
            border-radius: 12px;
        }

        .pagination-summary {
            font-size: 0.75rem;
            padding: 0.4rem 0.8rem;
        }
    }

    /* Enhanced Responsive Design */
    @media (max-width: 768px) {
        .customers-header {
            padding: 1.5rem;
            margin-bottom: 1.5rem;
            text-align: center;
        }
        
        .customers-header h2 {
            font-size: 1.5rem;
        }
        
        .customers-header .col-md-4 {
            text-align: center !important;
        }
        
        .filter-section {
            padding: 1.5rem;
        }
        
        .stats-cards {
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-top: 1.5rem;
        }
        
        .stat-card {
            padding: 1.5rem;
        }
        
        .stat-number {
            font-size: 2rem;
        }
        
        .advanced-table {
            font-size: 0.85rem;
        }
        
        .advanced-table th,
        .advanced-table td {
            padding: 1rem 0.75rem;
        }
        
        .customer-info {
            flex-direction: row;
            align-items: center;
            gap: 0.75rem;
        }
        
        .customer-avatar {
            width: 40px;
            height: 40px;
            font-size: 1rem;
        }
        
        .modern-pagination {
            padding: 15px;
        }
        
        .modern-pagination .page-link {
            padding: 6px 10px;
            font-size: 13px;
        }
        
        .pagination-info {
            font-size: 13px;
            margin-bottom: 10px;
        }
        
        .action-buttons {
            flex-direction: row;
            flex-wrap: wrap;
            gap: 0.25rem;
        }
        
        .action-btn {
            width: 32px;
            height: 32px;
            font-size: 0.8rem;
        }
        
        .type-badge,
        .status-badge {
            font-size: 0.7rem;
            padding: 0.3rem 0.6rem;
        }
    }
    
    @media (max-width: 576px) {
        .customers-header {
            padding: 1rem;
        }
        
        .customers-header h2 {
            font-size: 1.3rem;
        }
        
        .filter-toggle-btn {
            padding: 0.4rem 0.8rem;
            font-size: 0.85rem;
        }
        
        .professional-card {
            margin-bottom: 1.5rem;
        }
        
        .filter-section {
            padding: 1rem;
        }
        
        .stats-cards {
            grid-template-columns: 1fr;
            gap: 1rem;
        }
        
        .stat-card {
            padding: 1.25rem;
        }
        
        .modern-pagination {
            padding: 10px;
        }
        
        .modern-pagination .page-link {
            padding: 5px 8px;
            font-size: 12px;
        }
        
        .pagination-info {
            font-size: 12px;
            margin-bottom: 8px;
        }
        
        .empty-state {
            padding: 3rem 1rem;
        }
        
        .empty-state-icon {
            font-size: 3rem;
        }
    }

    /* Animation Classes */
    .fade-in-up {
        animation: fadeInUp 0.6s ease-out;
    }

    @keyframes fadeInUp {
        from {
            opacity: 0;
            transform: translateY(30px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    /* Loading Animation for Table Rows */
    .advanced-table tbody tr {
        animation: slideInLeft 0.5s ease-out backwards;
    }

    .advanced-table tbody tr:nth-child(1) { animation-delay: 0.1s; }
    .advanced-table tbody tr:nth-child(2) { animation-delay: 0.2s; }
    .advanced-table tbody tr:nth-child(3) { animation-delay: 0.3s; }
    .advanced-table tbody tr:nth-child(4) { animation-delay: 0.4s; }
    .advanced-table tbody tr:nth-child(5) { animation-delay: 0.5s; }

    @keyframes slideInLeft {
        from {
            opacity: 0;
            transform: translateX(-30px);
        }
        to {
            opacity: 1;
            transform: translateX(0);
        }
    }
</style>

@section('page-actions')
<a href="{{ route('customers.create') }}" class="btn btn-primary rounded-pill px-4">
    <i class="fas fa-user-plus me-2"></i>
    إضافة عميل جديد
</a>
@endsection

@section('content')
<!-- رأس الصفحة الاحترافي -->
<div class="customers-header fade-in-up">
    <div class="row align-items-center">
        <div class="col-md-8">
            <h2 class="mb-2">
                <i class="fas fa-users me-3"></i>
                إدارة العملاء
            </h2>
            <p class="mb-0 opacity-75">
                إدارة شاملة لجميع بيانات العملاء والتواصل معهم
            </p>
        </div>
        <div class="col-md-4 text-md-end mt-3 mt-md-0">
            <div class="d-flex justify-content-md-end gap-2">
                <button type="button" class="btn btn-light btn-sm filter-toggle-btn" onclick="toggleFilters()">
                    <i class="fas fa-filter me-1"></i>
                    الفلاتر
                </button>
                <button type="button" class="btn btn-info btn-sm filter-toggle-btn" onclick="exportCustomers()">
                    <i class="fas fa-download me-1"></i>
                    تصدير
                </button>
            </div>
        </div>
    </div>
</div>

<!-- البطاقة الرئيسية -->
<div class="professional-card fade-in-up">
    <!-- قسم الفلترة المتقدم -->
    <div class="filter-section" id="filters-section" style="display: none;">
        <form method="GET" action="{{ route('customers.index') }}" class="row g-3">
            <div class="col-md-3">
                <label class="form-label fw-bold">
                    <i class="fas fa-user-tag me-1 text-primary"></i>
                    نوع العميل
                </label>
                <select name="type" class="form-select">
                    <option value="">جميع الأنواع</option>
                    <option value="individual" {{ request('type') == 'individual' ? 'selected' : '' }}>فرد</option>
                    <option value="business" {{ request('type') == 'business' ? 'selected' : '' }}>شركة</option>
                </select>
            </div>
            
            <div class="col-md-3">
                <label class="form-label fw-bold">
                    <i class="fas fa-toggle-on me-1 text-success"></i>
                    الحالة
                </label>
                <select name="status" class="form-select">
                    <option value="">جميع الحالات</option>
                    <option value="active" {{ request('status') == 'active' ? 'selected' : '' }}>نشط</option>
                    <option value="inactive" {{ request('status') == 'inactive' ? 'selected' : '' }}>غير نشط</option>
                </select>
            </div>
            
            <div class="col-md-6">
                <label class="form-label fw-bold">
                    <i class="fas fa-search me-1 text-info"></i>
                    البحث السريع
                </label>
                <div class="search-box">
                    <input type="text" name="search" class="form-control" 
                           placeholder="الاسم، البريد، الهاتف، أو الرقم الضريبي" value="{{ request('search') }}">
                    <i class="fas fa-search search-icon"></i>
                </div>
            </div>
            
            <div class="col-12">
                <div class="d-flex gap-2">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-filter me-1"></i>
                        تطبيق الفلاتر
                    </button>
                    <a href="{{ route('customers.index') }}" class="btn btn-outline-secondary">
                        <i class="fas fa-times me-1"></i>
                        إلغاء الفلاتر
                    </a>
                </div>
            </div>
        </form>
    </div>
    
    <!-- محتوى الجدول -->
    <div class="p-3">
        @if($customers->count() > 0)
            <div class="table-responsive">
                <table class="table advanced-table">
                    <thead>
                        <tr>
                            <th>
                                <i class="fas fa-user me-2"></i>
                                بيانات العميل
                            </th>
                            <th>
                                <i class="fas fa-tag me-2"></i>
                                النوع
                            </th>
                            <th>
                                <i class="fas fa-id-card me-2"></i>
                                رقم التعريف
                            </th>
                            <th>
                                <i class="fas fa-map-marker-alt me-2"></i>
                                الموقع
                            </th>
                            <th>
                                <i class="fas fa-phone me-2"></i>
                                التواصل
                            </th>
                            <th>
                                <i class="fas fa-toggle-on me-2"></i>
                                الحالة
                            </th>
                            <th>
                                <i class="fas fa-cogs me-2"></i>
                                الإجراءات
                            </th>
                        </tr>
                    </thead>
                    <tbody>
                        @foreach($customers as $customer)
                        <tr>
                            <td>
                                <div class="customer-info">
                                    <div class="customer-avatar">
                                        {{ strtoupper(substr($customer->name, 0, 1)) }}
                                    </div>
                                    <div>
                                        <div class="fw-bold">{{ $customer->name }}</div>
                                        <small class="text-muted">{{ $customer->full_address }}</small>
                                    </div>
                                </div>
                            </td>
                            <td>
                                @if($customer->customer_type == 'individual')
                                    <span class="type-badge bg-info text-white">
                                        <i class="fas fa-user me-1"></i>
                                        فرد
                                    </span>
                                @else
                                    <span class="type-badge bg-primary text-white">
                                        <i class="fas fa-building me-1"></i>
                                        شركة
                                    </span>
                                @endif
                            </td>
                            <td>
                                @if($customer->customer_type == 'individual')
                                    <div class="fw-bold">{{ $customer->national_id }}</div>
                                    <small class="text-muted">هوية وطنية</small>
                                @else
                                    <div class="fw-bold">{{ $customer->vat_number }}</div>
                                    <small class="text-muted">رقم ضريبي</small>
                                    @if($customer->cr_number)
                                        <br><small class="text-muted">س.ت: {{ $customer->cr_number }}</small>
                                    @endif
                                @endif
                            </td>
                            <td>
                                <div class="fw-bold">{{ $customer->city }}</div>
                                <small class="text-muted">{{ $customer->district }}</small>
                            </td>
                            <td>
                                @if($customer->email)
                                    <div class="mb-1">
                                        <i class="fas fa-envelope text-primary me-1"></i>
                                        <small>{{ $customer->email }}</small>
                                    </div>
                                @endif
                                @if($customer->phone)
                                    <div>
                                        <i class="fas fa-phone text-success me-1"></i>
                                        <small>{{ $customer->phone }}</small>
                                    </div>
                                @endif
                                @if(!$customer->email && !$customer->phone)
                                    <span class="text-muted">غير متوفر</span>
                                @endif
                            </td>
                            <td>
                                @if($customer->is_active)
                                    <span class="status-badge bg-success text-white">
                                        <i class="fas fa-check-circle me-1"></i>
                                        نشط
                                    </span>
                                @else
                                    <span class="status-badge bg-secondary text-white">
                                        <i class="fas fa-pause-circle me-1"></i>
                                        غير نشط
                                    </span>
                                @endif
                            </td>
                            <td>
                                <div class="action-buttons">
                                    <a href="{{ route('customers.show', $customer) }}" 
                                       class="action-btn btn btn-outline-primary" title="عرض التفاصيل">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                    
                                    <a href="{{ route('customers.edit', $customer) }}" 
                                       class="action-btn btn btn-outline-warning" title="تعديل">
                                        <i class="fas fa-edit"></i>
                                    </a>
                                    
                                    <div class="dropdown">
                                        <button class="action-btn btn btn-outline-secondary dropdown-toggle" 
                                                type="button" data-bs-toggle="dropdown" title="المزيد">
                                            <i class="fas fa-ellipsis-v"></i>
                                        </button>
                                        <ul class="dropdown-menu shadow">
                                            <li>
                                                <a class="dropdown-item" href="{{ route('invoices.create', ['customer_id' => $customer->id]) }}">
                                                    <i class="fas fa-file-invoice text-success me-2"></i>
                                                    إنشاء فاتورة
                                                </a>
                                            </li>
                                            <li><hr class="dropdown-divider"></li>
                                            <li>
                                                <button type="button" class="dropdown-item text-danger w-100 text-start"
                                                        data-bs-toggle="modal"
                                                        data-bs-target="#deleteCustomerModal"
                                                        data-customer-id="{{ $customer->id }}"
                                                        data-customer-name="{{ $customer->name }}"
                                                        data-customer-type="{{ $customer->customer_type == 'individual' ? 'فرد' : 'شركة' }}">
                                                    <i class="fas fa-trash me-2"></i>
                                                    حذف
                                                </button>
                                            </li>
                                        </ul>
                                    </div>
                                </div>
                            </td>
                        </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>
            
            <!-- Professional Modern Pagination -->
            @if($customers->hasPages())
                <div class="modern-pagination">
                    <div class="pagination-info">
                        @php
                            $from = ($customers->currentPage() - 1) * $customers->perPage() + 1;
                            $to = min($customers->currentPage() * $customers->perPage(), $customers->total());
                        @endphp
                        عرض {{ $from }} - {{ $to }} من إجمالي {{ $customers->total() }} عميل
                    </div>
                    
                    <div class="pagination-controls">
                        {{ $customers->appends(request()->query())->links() }}
                    </div>
                    
                    <div class="pagination-summary">
                        <i class="fas fa-info-circle me-1"></i>
                        الصفحة {{ $customers->currentPage() }} من {{ $customers->lastPage() }}
                        @if($customers->hasMorePages())
                            • {{ $customers->lastPage() - $customers->currentPage() }} صفحة متبقية
                        @endif
                    </div>
                </div>
            @endif
        @else
            <div class="empty-state">
                <div class="empty-state-icon">
                    <i class="fas fa-users"></i>
                </div>
                <h5 class="fw-bold">لا يوجد عملاء</h5>
                <p class="mb-3">لم يتم العثور على عملاء مطابقين للمعايير المحددة</p>
                <a href="{{ route('customers.create') }}" class="btn btn-primary rounded-pill px-4">
                    <i class="fas fa-user-plus me-2"></i>
                    إضافة عميل جديد
                </a>
            </div>
        @endif
    </div>
</div>

<!-- إحصائيات متقدمة -->
<div class="stats-cards fade-in-up">
    <div class="stat-card individuals">
        <span class="stat-number text-info">{{ $customers->where('customer_type', 'individual')->count() }}</span>
        <div class="stat-label">عملاء أفراد</div>
        <small class="text-muted d-block mt-2">
            <i class="fas fa-user me-1"></i>
            عملاء فرديين نشطين
        </small>
    </div>
    
    <div class="stat-card businesses">
        <span class="stat-number text-primary">{{ $customers->where('customer_type', 'business')->count() }}</span>
        <div class="stat-label">عملاء شركات</div>
        <small class="text-muted d-block mt-2">
            <i class="fas fa-building me-1"></i>
            عملاء مؤسسيين
        </small>
    </div>
    
    <div class="stat-card active">
        <span class="stat-number text-success">{{ $customers->where('is_active', true)->count() }}</span>
        <div class="stat-label">عملاء نشطين</div>
        <small class="text-muted d-block mt-2">
            <i class="fas fa-check-circle me-1"></i>
            عملاء يتفاعلون بانتظام
        </small>
    </div>
    
    <div class="stat-card total">
        <span class="stat-number text-warning">{{ $customers->total() }}</span>
        <div class="stat-label">إجمالي العملاء</div>
        <small class="text-muted d-block mt-2">
            <i class="fas fa-chart-line me-1"></i>
            العدد الكلي للعملاء
        </small>
    </div>
</div>
@endsection

@push('scripts')
<script>
function toggleFilters() {
    const filtersSection = document.getElementById('filters-section');
    const isVisible = filtersSection.style.display !== 'none';
    
    if (isVisible) {
        filtersSection.style.display = 'none';
    } else {
        filtersSection.style.display = 'block';
        filtersSection.classList.add('fade-in-up');
    }
}

function exportCustomers() {
    // إظهار نافذة تأكيد مع خيارات التصدير
    const exportModal = `
        <div class="modal fade" id="exportModal" tabindex="-1">
            <div class="modal-dialog modal-dialog-centered">
                <div class="modal-content">
                    <div class="modal-header bg-info text-white">
                        <h5 class="modal-title">
                            <i class="fas fa-download me-2"></i>
                            تصدير بيانات العملاء
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body text-center py-4">
                        <i class="fas fa-file-export text-info fa-3x mb-3"></i>
                        <h6>اختر صيغة التصدير المرغوبة:</h6>
                        <div class="d-grid gap-2 mt-3">
                            <button class="btn btn-outline-success" onclick="exportToExcel()">
                                <i class="fas fa-file-excel me-2"></i>
                                تصدير Excel
                            </button>
                            <button class="btn btn-outline-danger" onclick="exportToPDF()">
                                <i class="fas fa-file-pdf me-2"></i>
                                تصدير PDF
                            </button>
                            <button class="btn btn-outline-primary" onclick="exportToCSV()">
                                <i class="fas fa-file-csv me-2"></i>
                                تصدير CSV
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    // إضافة النافذة إلى الصفحة وإظهارها
    if (!document.getElementById('exportModal')) {
        document.body.insertAdjacentHTML('beforeend', exportModal);
    }
    
    const modal = new bootstrap.Modal(document.getElementById('exportModal'));
    modal.show();
}

function exportToExcel() {
    alert('سيتم تطوير وظيفة تصدير Excel قريباً');
    bootstrap.Modal.getInstance(document.getElementById('exportModal')).hide();
}

function exportToPDF() {
    alert('سيتم تطوير وظيفة تصدير PDF قريباً');
    bootstrap.Modal.getInstance(document.getElementById('exportModal')).hide();
}

function exportToCSV() {
    alert('سيتم تطوير وظيفة تصدير CSV قريباً');
    bootstrap.Modal.getInstance(document.getElementById('exportModal')).hide();
}

// إظهار الفلاتر إذا كان هناك فلاتر مطبقة
$(document).ready(function() {
    const urlParams = new URLSearchParams(window.location.search);
    if (urlParams.has('type') || urlParams.has('status') || urlParams.has('search')) {
        document.getElementById('filters-section').style.display = 'block';
    }
    
    // إضافة تأثيرات التحريك للجدول
    $('.advanced-table tbody tr').each(function(index) {
        $(this).css('animation-delay', (index * 0.1) + 's');
        $(this).addClass('fade-in-up');
    });
    
    // تحسين تجربة البحث
    $('input[name="search"]').on('input', function() {
        const searchValue = $(this).val();
        if (searchValue.length > 2) {
            // يمكن إضافة بحث مباشر هنا
        }
    });
});

// Enhanced Interactive Features
document.querySelectorAll('.action-btn').forEach(button => {
    button.addEventListener('mouseenter', function() {
        this.style.transform = 'translateY(-2px) scale(1.05)';
    });
    
    button.addEventListener('mouseleave', function() {
        this.style.transform = 'translateY(0) scale(1)';
    });
});

// Enhanced Search Experience with Real-time Feedback
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.querySelector('input[name="search"]');
    const tableRows = document.querySelectorAll('.advanced-table tbody tr');
    
    if (searchInput) {
        let searchTimer;
        searchInput.addEventListener('input', function() {
            const searchValue = this.value.toLowerCase().trim();
            
            // Clear previous timer
            clearTimeout(searchTimer);
            
            // Add loading state
            this.classList.add('search-loading');
            
            // Simulate search delay for better UX
            searchTimer = setTimeout(() => {
                this.classList.remove('search-loading');
                
                if (searchValue.length >= 2) {
                    // Add visual feedback for active search
                    this.style.borderColor = '#3b82f6';
                    this.style.boxShadow = '0 0 0 3px rgba(59, 130, 246, 0.1)';
                } else {
                    // Reset search styling
                    this.style.borderColor = '';
                    this.style.boxShadow = '';
                }
            }, 500);
        });
    }
    
    // Enhanced hover effects for stat cards
    document.querySelectorAll('.stat-card').forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-8px) scale(1.02)';
            this.style.boxShadow = '0 20px 40px rgba(0, 0, 0, 0.1)';
        });
        
        card.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(-4px) scale(1)';
            this.style.boxShadow = '';
        });
    });
    
    // Professional loading animation for pagination clicks
    document.querySelectorAll('.pagination a').forEach(link => {
        link.addEventListener('click', function(e) {
            // Add loading state to clicked pagination link
            this.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
            this.style.pointerEvents = 'none';
            
            // Show loading overlay
            const overlay = document.createElement('div');
            overlay.className = 'loading-overlay';
            overlay.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: rgba(255, 255, 255, 0.8);
                backdrop-filter: blur(2px);
                z-index: 9999;
                display: flex;
                align-items: center;
                justify-content: center;
                font-size: 1.2rem;
                color: #3b82f6;
            `;
            overlay.innerHTML = '<div><i class="fas fa-spinner fa-spin me-2"></i>جاري التحديث...</div>';
            document.body.appendChild(overlay);
        });
    });
    
    // Enhanced filter toggle animation
    const filterToggleBtn = document.querySelector('.filter-toggle-btn');
    if (filterToggleBtn) {
        filterToggleBtn.addEventListener('click', function() {
            const icon = this.querySelector('i');
            const isActive = document.getElementById('filters-section').style.display !== 'none';
            
            if (isActive) {
                icon.className = 'fas fa-filter me-1';
                this.classList.remove('active-filter');
            } else {
                icon.className = 'fas fa-filter-circle-dollar me-1';
                this.classList.add('active-filter');
            }
        });
    }
});

// CSS for search loading state
const searchLoadingCSS = `
    .search-loading {
        position: relative;
    }
    .search-loading::after {
        content: '';
        position: absolute;
        right: 40px;
        top: 50%;
        transform: translateY(-50%);
        width: 16px;
        height: 16px;
        border: 2px solid #3b82f6;
        border-top: 2px solid transparent;
        border-radius: 50%;
        animation: spin 1s linear infinite;
    }
    
    .active-filter {
        background: rgba(255, 255, 255, 0.3) !important;
        transform: translateY(-1px) !important;
    }
    
    @keyframes spin {
        0% { transform: translateY(-50%) rotate(0deg); }
        100% { transform: translateY(-50%) rotate(360deg); }
    }
`;

// Inject search loading CSS
const styleSheet = document.createElement('style');
styleSheet.textContent = searchLoadingCSS;
document.head.appendChild(styleSheet);

function showDeleteCustomerModal(customerId, customerName, customerType) {
    const modal = document.getElementById('deleteCustomerModal');
    const form = document.getElementById('deleteCustomerForm');
    const nameSpan = document.getElementById('customerNameToDelete');
    const typeSpan = document.getElementById('customerTypeToDelete');
    
    form.action = `{{ url('customers') }}/${customerId}`;
    nameSpan.textContent = customerName;
    typeSpan.textContent = customerType;
    
    const bootstrapModal = new bootstrap.Modal(modal);
    bootstrapModal.show();
}
</script>

<!-- Modal تأكيد حذف العميل -->
<div class="modal fade" id="deleteCustomerModal" tabindex="-1" aria-labelledby="deleteCustomerModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title" id="deleteCustomerModalLabel">
                    <i class="fas fa-exclamation-triangle"></i>
                    تأكيد حذف العميل
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="إغلاق"></button>
            </div>
            <div class="modal-body">
                <div class="alert alert-warning">
                    <i class="fas fa-exclamation-triangle"></i>
                    <strong>تحذير:</strong> هذا الإجراء لا يمكن التراجع عنه!
                </div>
                <p class="mb-3">هل تريد حذف هذا العميل؟</p>
                <div class="card">
                    <div class="card-body">
                        <h6 class="card-title">تفاصيل العميل:</h6>
                        <p class="mb-1"><strong>الاسم:</strong> <span id="customerNameToDelete"></span></p>
                        <p class="mb-0"><strong>النوع:</strong> <span id="customerTypeToDelete"></span></p>
                    </div>
                </div>
                <div class="alert alert-info mt-3">
                    <i class="fas fa-info-circle"></i>
                    <strong>ملاحظة:</strong> سيتم حذف جميع البيانات المرتبطة بهذا العميل بما في ذلك الفواتير والمعاملات المالية.
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <i class="fas fa-times"></i>
                    إلغاء
                </button>
                <form id="deleteCustomerForm" method="POST" class="d-inline">
                    @csrf
                    @method('DELETE')
                    <button type="submit" class="btn btn-danger">
                        <i class="fas fa-trash"></i>
                        تأكيد الحذف
                    </button>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const deleteModal = document.getElementById('deleteCustomerModal');
    const deleteForm = document.getElementById('deleteCustomerForm');
    const customerName = document.getElementById('customerNameToDelete');
    const customerType = document.getElementById('customerTypeToDelete');

    deleteModal.addEventListener('show.bs.modal', function (event) {
        const button = event.relatedTarget;
        const customerId = button.getAttribute('data-customer-id');
        const customerNameValue = button.getAttribute('data-customer-name');
        const customerTypeValue = button.getAttribute('data-customer-type');

        // تحديث النموذج
        deleteForm.action = `{{ url('customers') }}/${customerId}`;
        
        // تحديث محتوى المودال
        customerName.textContent = customerNameValue;
        customerType.textContent = customerTypeValue;
    });
});
</script>
@endpush