@extends('layouts.app')

@section('title', 'تعديل مركز التكلفة')

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="card-title mb-0">تعديل مركز التكلفة: {{ $costCenter->name }}</h5>
                    <div>
                        <a href="{{ route('accounting.cost-centers.show', $costCenter) }}" class="btn btn-info">
                            <i class="fas fa-eye"></i> عرض المركز
                        </a>
                        <a href="{{ route('accounting.cost-centers.index') }}" class="btn btn-secondary">
                            <i class="fas fa-arrow-left"></i> العودة للقائمة
                        </a>
                    </div>
                </div>

                <div class="card-body">
                    <form action="{{ route('accounting.cost-centers.update', $costCenter) }}" method="POST">
                        @csrf
                        @method('PUT')
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="code" class="form-label">كود مركز التكلفة <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control @error('code') is-invalid @enderror" 
                                           id="code" name="code" value="{{ old('code', $costCenter->code) }}" 
                                           placeholder="مثال: CC001" required>
                                    @error('code')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                    @if($hasTransactions)
                                        <div class="form-text text-warning">
                                            <i class="fas fa-exclamation-triangle"></i>
                                            تحذير: يوجد {{ $transactionCount }} حركة مرتبطة بهذا المركز. تغيير الكود قد يؤثر على التقارير.
                                        </div>
                                    @endif
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="name" class="form-label">اسم مركز التكلفة <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control searchable-input @error('name') is-invalid @enderror"
                                           id="name" name="name" value="{{ old('name', $costCenter->name) }}"
                                           placeholder="مثال: الإدارة العامة"
                                           list="cost-center-names-list" autocomplete="off" required>
                                    <datalist id="cost-center-names-list">
                                        @if(isset($existingCostCenters))
                                            @foreach($existingCostCenters as $existingCostCenter)
                                                @if($existingCostCenter->id !== $costCenter->id)
                                                    <option value="{{ $existingCostCenter->name }}">{{ $existingCostCenter->code }} - {{ $existingCostCenter->name }}</option>
                                                @endif
                                            @endforeach
                                        @endif
                                    </datalist>
                                    @error('name')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                    <div class="form-text">ابدأ بالكتابة للبحث في أسماء مراكز التكلفة الموجودة</div>
                                </div>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label for="description" class="form-label">الوصف</label>
                            <textarea class="form-control @error('description') is-invalid @enderror" 
                                      id="description" name="description" rows="3" 
                                      placeholder="وصف تفصيلي لمركز التكلفة ونشاطاته">{{ old('description', $costCenter->description) }}</textarea>
                            @error('description')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="budget_amount" class="form-label">الموازنة السنوية</label>
                                    <div class="input-group">
                                        <input type="number" step="0.01" min="0" class="form-control @error('budget_amount') is-invalid @enderror" 
                                               id="budget_amount" name="budget_amount" value="{{ old('budget_amount', $costCenter->budget_amount) }}" 
                                               placeholder="0.00">
                                        <span class="input-group-text">ريال</span>
                                    </div>
                                    @error('budget_amount')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                    <div class="form-text">
                                        الموازنة المخططة لهذا المركز للسنة الحالية
                                    </div>
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="manager" class="form-label">مدير المركز</label>
                                    <input type="text" class="form-control @error('manager') is-invalid @enderror" 
                                           id="manager" name="manager" value="{{ old('manager', $costCenter->manager) }}" 
                                           placeholder="اسم مدير مركز التكلفة">
                                    @error('manager')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="mb-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="is_active" name="is_active" value="1"
                                       {{ old('is_active', $costCenter->is_active) ? 'checked' : '' }}>
                                <label class="form-check-label" for="is_active">
                                    <strong>مركز تكلفة نشط</strong>
                                </label>
                                <div class="form-text">
                                    يمكن استخدام مركز التكلفة في القيود المحاسبية والتقارير
                                </div>
                                @if($hasTransactions && old('is_active', $costCenter->is_active))
                                    <div class="form-text text-warning">
                                        <i class="fas fa-exclamation-triangle"></i>
                                        تحذير: إلغاء تفعيل هذا المركز قد يؤثر على التقارير الحالية
                                    </div>
                                @endif
                            </div>
                        </div>

                        <!-- معلومات إضافية -->
                        <div class="row">
                            <div class="col-md-6">
                                <div class="card border-info">
                                    <div class="card-header bg-info text-white">
                                        <h6 class="mb-0">معلومات المركز</h6>
                                    </div>
                                    <div class="card-body">
                                        <p><strong>تاريخ الإنشاء:</strong> {{ $costCenter->created_at->format('Y-m-d H:i') }}</p>
                                        <p><strong>آخر تحديث:</strong> {{ $costCenter->updated_at->format('Y-m-d H:i') }}</p>
                                        <p><strong>عدد الحركات:</strong> {{ $transactionCount }} حركة</p>
                                        @if($costCenter->budget_amount)
                                            @php
                                                $currentYear = date('Y');
                                                $actualAmount = \App\Models\JournalEntryLine::where('cost_center_id', $costCenter->id)
                                                    ->whereHas('journalEntry', function($query) use ($currentYear) {
                                                        $query->whereYear('entry_date', $currentYear)
                                                              ->where('status', 'posted');
                                                    })
                                                    ->sum('debit_amount') - \App\Models\JournalEntryLine::where('cost_center_id', $costCenter->id)
                                                    ->whereHas('journalEntry', function($query) use ($currentYear) {
                                                        $query->whereYear('entry_date', $currentYear)
                                                              ->where('status', 'posted');
                                                    })
                                                    ->sum('credit_amount');
                                                
                                                $utilization = $costCenter->budget_amount > 0 ? ($actualAmount / $costCenter->budget_amount) * 100 : 0;
                                            @endphp
                                            <p><strong>استغلال الموازنة:</strong> 
                                                <span class="{{ $utilization <= 80 ? 'text-success' : ($utilization <= 100 ? 'text-warning' : 'text-danger') }}">
                                                    {{ number_format($utilization, 1) }}%
                                                </span>
                                            </p>
                                        @endif
                                    </div>
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="card border-warning">
                                    <div class="card-header bg-warning text-dark">
                                        <h6 class="mb-0">تنبيهات</h6>
                                    </div>
                                    <div class="card-body">
                                        @if($hasTransactions)
                                            <div class="alert alert-warning">
                                                <i class="fas fa-exclamation-triangle"></i>
                                                يوجد {{ $transactionCount }} حركة مرتبطة بهذا المركز. 
                                                كن حذراً عند تعديل البيانات الأساسية.
                                            </div>
                                        @else
                                            <div class="alert alert-success">
                                                <i class="fas fa-check-circle"></i>
                                                لا توجد حركات مرتبطة بهذا المركز. يمكن تعديل جميع البيانات بأمان.
                                            </div>
                                        @endif

                                        @if($costCenter->budget_amount && $actualAmount > $costCenter->budget_amount)
                                            <div class="alert alert-danger">
                                                <i class="fas fa-exclamation-circle"></i>
                                                تم تجاوز الموازنة المحددة!
                                            </div>
                                        @endif
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Current Budget Analysis -->
                        @if($costCenter->budget_amount)
                        <div class="row mt-3">
                            <div class="col-12">
                                <div class="card bg-light">
                                    <div class="card-header">
                                        <h6 class="mb-0">تحليل الموازنة الحالية</h6>
                                    </div>
                                    <div class="card-body">
                                        <div class="row">
                                            <div class="col-md-3">
                                                <strong>الموازنة المخططة:</strong><br>
                                                <span class="text-primary">{{ number_format($costCenter->budget_amount, 2) }} ريال</span>
                                            </div>
                                            <div class="col-md-3">
                                                <strong>المبلغ الفعلي:</strong><br>
                                                <span class="{{ $actualAmount >= 0 ? 'text-success' : 'text-danger' }}">
                                                    {{ number_format($actualAmount, 2) }} ريال
                                                </span>
                                            </div>
                                            <div class="col-md-3">
                                                <strong>المتبقي:</strong><br>
                                                @php $remaining = $costCenter->budget_amount - $actualAmount; @endphp
                                                <span class="{{ $remaining >= 0 ? 'text-success' : 'text-danger' }}">
                                                    {{ number_format($remaining, 2) }} ريال
                                                </span>
                                            </div>
                                            <div class="col-md-3">
                                                <strong>نسبة الاستغلال:</strong><br>
                                                <span class="{{ $utilization <= 80 ? 'text-success' : ($utilization <= 100 ? 'text-warning' : 'text-danger') }}">
                                                    {{ number_format($utilization, 1) }}%
                                                </span>
                                            </div>
                                        </div>

                                        <div class="mt-3">
                                            <div class="progress" style="height: 20px;">
                                                @php
                                                    $progressPercent = min(100, $utilization);
                                                    $progressClass = $utilization <= 80 ? 'bg-success' : ($utilization <= 100 ? 'bg-warning' : 'bg-danger');
                                                @endphp
                                                <div class="progress-bar {{ $progressClass }}" role="progressbar" 
                                                     style="width: {{ $progressPercent }}%">
                                                    {{ number_format($utilization, 1) }}%
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        @endif

                        <div class="d-flex justify-content-between mt-4">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save"></i> حفظ التغييرات
                            </button>
                            <div>
                                <a href="{{ route('accounting.cost-centers.show', $costCenter) }}" class="btn btn-secondary">
                                    <i class="fas fa-times"></i> إلغاء
                                </a>
                                @if(!$hasTransactions)
                                    <button type="button" class="btn btn-danger" data-bs-toggle="modal" data-bs-target="#deleteModal">
                                        <i class="fas fa-trash"></i> حذف المركز
                                    </button>
                                @endif
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
@if(!$hasTransactions)
<div class="modal fade" id="deleteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">تأكيد الحذف</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                هل أنت متأكد من حذف مركز التكلفة "{{ $costCenter->name }}"؟
                <br><small class="text-muted">لا يمكن التراجع عن هذا الإجراء.</small>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                <form action="{{ route('accounting.cost-centers.destroy', $costCenter) }}" method="POST" class="d-inline">
                    @csrf
                    @method('DELETE')
                    <button type="submit" class="btn btn-danger">حذف نهائي</button>
                </form>
            </div>
        </div>
    </div>
</div>
@endif
@endsection

@push('styles')
<!-- Select2 CSS -->
<link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
<link href="https://cdn.jsdelivr.net/npm/select2-bootstrap-5-theme@1.3.0/dist/select2-bootstrap-5-theme.min.css" rel="stylesheet" />

<style>
.searchable-input {
    position: relative;
}

.searchable-input:focus {
    border-color: #0d6efd;
    box-shadow: 0 0 0 0.25rem rgba(13, 110, 253, 0.25);
}

.searchable-select {
    min-height: 38px;
}

/* Select2 custom styling */
.select2-container--bootstrap-5 .select2-selection {
    min-height: 38px;
    border: 1px solid #ced4da;
}

.select2-container--bootstrap-5 .select2-selection--single .select2-selection__rendered {
    padding-left: 12px;
    padding-right: 20px;
    color: #495057;
    line-height: 36px;
}

.select2-container--bootstrap-5 .select2-selection--single .select2-selection__arrow {
    height: 36px;
    right: 3px;
}

.select2-container--bootstrap-5.select2-container--focus .select2-selection {
    border-color: #0d6efd;
    box-shadow: 0 0 0 0.25rem rgba(13, 110, 253, 0.25);
}

.select2-dropdown {
    border: 1px solid #ced4da;
    border-radius: 0.375rem;
}

.select2-search--dropdown .select2-search__field {
    border: 1px solid #ced4da;
    border-radius: 0.25rem;
    padding: 6px 12px;
}

.select2-results__option {
    padding: 8px 12px;
}

.select2-results__option--highlighted {
    background-color: #0d6efd;
    color: white;
}

/* Datalist styling */
datalist {
    position: absolute;
    background-color: white;
    border: 1px solid #ced4da;
    border-radius: 0.375rem;
    max-height: 200px;
    overflow-y: auto;
    z-index: 1000;
}
</style>
@endpush

@push('scripts')
<!-- Select2 JS -->
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>

<script>
$(document).ready(function() {
    // Enhanced search functionality for cost center name field
    const nameInput = document.getElementById('name');
    const costCenterNamesList = document.getElementById('cost-center-names-list');
    
    if (nameInput && costCenterNamesList) {
        nameInput.addEventListener('input', function() {
            const query = this.value.toLowerCase();
            if (query.length >= 2) {
                // Filter datalist options based on input
                const options = costCenterNamesList.querySelectorAll('option');
                options.forEach(option => {
                    const text = option.textContent.toLowerCase();
                    option.style.display = text.includes(query) ? 'block' : 'none';
                });
            }
        });

        // Add autocomplete styling and behavior
        nameInput.addEventListener('focus', function() {
            this.setAttribute('placeholder', 'ابدأ بالكتابة للبحث في أسماء مراكز التكلفة...');
        });

        nameInput.addEventListener('blur', function() {
            this.setAttribute('placeholder', 'مثال: الإدارة العامة');
        });
    }

    // تنسيق الكود تلقائياً
    $('#code').on('input', function() {
        this.value = this.value.toUpperCase().replace(/[^A-Z0-9]/g, '');
    });
    
    // تنسيق المبلغ
    $('#budget_amount').on('input', function() {
        let value = parseFloat(this.value);
        if (!isNaN(value) && value >= 0) {
            // إضافة فواصل للآلاف عند فقدان التركيز
            $(this).on('blur', function() {
                if (this.value) {
                    const formatted = parseFloat(this.value).toLocaleString('en-US', {
                        minimumFractionDigits: 2,
                        maximumFractionDigits: 2
                    });
                    console.log('Formatted budget:', formatted);
                }
            });
        }
    });
    
    // التحذير عند تغيير البيانات الحساسة
    $('#code, #is_active').on('change', function() {
        const hasTransactions = {{ $hasTransactions ? 'true' : 'false' }};
        if (hasTransactions) {
            const fieldName = this.id === 'code' ? 'الكود' : 'حالة التفعيل';
            if (confirm(`تحذير: تغيير ${fieldName} قد يؤثر على التقارير المالية الموجودة. هل تريد المتابعة؟`)) {
                return true;
            } else {
                // إعادة القيمة الأصلية
                if (this.id === 'code') {
                    this.value = '{{ $costCenter->code }}';
                } else {
                    this.checked = {{ $costCenter->is_active ? 'true' : 'false' }};
                }
            }
        }
    });
    
    // التحقق من صحة البيانات قبل الإرسال
    $('form').on('submit', function(e) {
        const code = $('#code').val().trim();
        const name = $('#name').val().trim();
        
        if (!code || !name) {
            e.preventDefault();
            showValidationModal('حقول مطلوبة', 'يرجى ملء جميع الحقول المطلوبة', 'warning');
            return false;
        }
        
        if (code.length < 3) {
            e.preventDefault();
            showValidationModal('رمز غير صحيح', 'كود مركز التكلفة يجب أن يكون 3 أحرف على الأقل', 'danger');
            $('#code').focus();
            return false;
        }
    });
});

// دالة عرض رسائل التحقق في modal
function showValidationModal(title, message, type = 'warning') {
    const iconClass = type === 'danger' ? 'fas fa-exclamation-triangle' :
                     type === 'warning' ? 'fas fa-exclamation-circle' :
                     'fas fa-info-circle';
    
    const modalHtml = `
        <div class="modal fade" id="validationModal" tabindex="-1" aria-hidden="true">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header bg-${type} text-white">
                        <h5 class="modal-title">
                            <i class="${iconClass}"></i>
                            ${title}
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="alert alert-${type} border-0">
                            <i class="${iconClass} me-2"></i>
                            ${message}
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                            <i class="fas fa-check"></i>
                            فهمت
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    // إزالة modal سابق إن وجد
    const existingModal = document.getElementById('validationModal');
    if (existingModal) {
        existingModal.remove();
    }
    
    // إضافة modal جديد
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    
    // عرض modal
    const modal = new bootstrap.Modal(document.getElementById('validationModal'));
    modal.show();
    
    // إزالة modal بعد إغلاقه
    document.getElementById('validationModal').addEventListener('hidden.bs.modal', function() {
        this.remove();
    });
}
</script>
@endpush