@extends('layouts.app')

@section('title', 'تفاصيل الشركة: ' . $company->name)
@section('page-title', $company->name)

@section('page-actions')
<div class="d-flex gap-2 flex-wrap">
    <a href="{{ route('companies.edit', $company) }}" class="btn btn-warning">
        <i class="fas fa-edit me-1"></i> تعديل
    </a>
    <a href="{{ route('invoices.create', ['company_id' => $company->id]) }}" class="btn btn-primary">
        <i class="fas fa-file-invoice me-1"></i> إنشاء فاتورة
    </a>
    <a href="{{ route('companies.index') }}" class="btn btn-secondary">
        <i class="fas fa-arrow-left me-1"></i> العودة
    </a>
    
    <div class="dropdown">
        <button class="btn btn-success dropdown-toggle" data-bs-toggle="dropdown">
            <i class="fas fa-ellipsis-v"></i>
        </button>
        <ul class="dropdown-menu">
            @if(!$company->zatcaSettings)
                <li><a class="dropdown-item" href="{{ route('zatca-settings.create', ['company_id' => $company->id]) }}">
                    <i class="fas fa-plus me-2"></i>إعداد ZATCA
                </a></li>
            @else
                <li><a class="dropdown-item" href="{{ route('zatca-settings.show', $company->zatcaSettings) }}">
                    <i class="fas fa-cog me-2"></i>إدارة ZATCA
                </a></li>
            @endif
            <li><a class="dropdown-item" href="#" onclick="exportCompany()">
                <i class="fas fa-download me-2"></i>تصدير البيانات
            </a></li>
            <li><hr class="dropdown-divider"></li>
            <li><a class="dropdown-item text-warning" href="#" onclick="toggleCompanyStatus()">
                <i class="fas fa-{{ $company->is_active ? 'pause' : 'play' }} me-2"></i>
                {{ $company->is_active ? 'تعطيل' : 'تفعيل' }} الشركة
            </a></li>
            @if($company->invoices->count() == 0)
            <li><a class="dropdown-item text-danger" href="#" onclick="deleteCompany()">
                <i class="fas fa-trash me-2"></i>حذف الشركة
            </a></li>
            @endif
        </ul>
    </div>
</div>
@endsection

@push('styles')
<style>
.company-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 12px;
    padding: 2rem;
    margin-bottom: 2rem;
    box-shadow: 0 8px 25px rgba(0,0,0,0.1);
}

.company-avatar {
    width: 80px;
    height: 80px;
    background: rgba(255,255,255,0.2);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 2rem;
    margin-bottom: 1rem;
    overflow: hidden;
}

.company-logo {
    width: 100%;
    height: 100%;
    object-fit: cover;
    border-radius: 50%;
}

.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.stat-card {
    background: white;
    border-radius: 12px;
    padding: 1.5rem;
    text-align: center;
    box-shadow: 0 4px 20px rgba(0,0,0,0.08);
    border: 1px solid #f0f0f0;
    transition: transform 0.2s ease;
}

.stat-card:hover {
    transform: translateY(-4px);
    box-shadow: 0 8px 30px rgba(0,0,0,0.12);
}

.stat-number {
    font-size: 2rem;
    font-weight: 700;
    color: #2d3748;
    margin-bottom: 0.5rem;
}

.stat-label {
    color: #718096;
    font-size: 0.9rem;
    font-weight: 500;
}

.info-card {
    background: white;
    border-radius: 12px;
    margin-bottom: 1.5rem;
    box-shadow: 0 4px 20px rgba(0,0,0,0.08);
    border: 1px solid #f0f0f0;
    overflow: hidden;
}

.info-card-header {
    padding: 1rem 1.5rem;
    border-bottom: 1px solid #e2e8f0;
    background: #f8fafc;
}

.info-card-header h5 {
    margin: 0;
    color: #2d3748;
    font-weight: 600;
    font-size: 1.1rem;
}

.info-card-body {
    padding: 1.5rem;
}

.info-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1rem;
}

.info-item {
    display: flex;
    flex-direction: column;
    padding: 0.75rem;
    background: #f8fafc;
    border-radius: 8px;
    border-left: 3px solid #e2e8f0;
}

.info-label {
    font-size: 0.85rem;
    color: #718096;
    font-weight: 500;
    margin-bottom: 0.25rem;
}

.info-value {
    font-weight: 600;
    color: #2d3748;
}

.status-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    border-radius: 20px;
    font-size: 0.85rem;
    font-weight: 600;
}

.status-active {
    background: #48bb7833;
    color: #38a169;
}

.status-inactive {
    background: #f5656533;
    color: #e53e3e;
}

.status-zatca {
    background: #4299e133;
    color: #3182ce;
}

.table-responsive {
    border-radius: 8px;
    overflow: hidden;
}

.table {
    margin: 0;
}

.table th {
    background: #f8fafc;
    border: none;
    font-weight: 600;
    color: #2d3748;
    padding: 1rem;
}

.table td {
    padding: 1rem;
    border-color: #e2e8f0;
}

.badge {
    font-size: 0.75rem;
    padding: 0.4rem 0.8rem;
    border-radius: 12px;
}

.quick-actions {
    background: white;
    border-radius: 12px;
    padding: 1.5rem;
    box-shadow: 0 4px 20px rgba(0,0,0,0.08);
    border: 1px solid #f0f0f0;
    margin-bottom: 1.5rem;
}

.quick-actions .btn {
    width: 100%;
    margin-bottom: 0.75rem;
    justify-content: flex-start;
}

.quick-actions .btn:last-child {
    margin-bottom: 0;
}

@media (max-width: 768px) {
    .company-header {
        padding: 1.5rem;
        text-align: center;
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
    }
    
    .info-grid {
        grid-template-columns: 1fr;
    }
}
</style>
@endpush

@section('content')
<div class="container-fluid">
    <!-- Company Header -->
    <div class="company-header">
        <div class="row align-items-center">
            <div class="col-md-2 text-center">
                <div class="company-avatar">
                    @if($company->logo)
                        <img src="{{ asset('zatca/public/' . $company->logo) }}" alt="{{ $company->name }}" class="company-logo">
                    @else
                        <i class="fas fa-building"></i>
                    @endif
                </div>
            </div>
            <div class="col-md-7">
                <h1 class="h2 mb-2">{{ $company->name }}</h1>
                <p class="mb-3 opacity-75">{{ $company->getFullAddressAttribute() }}</p>
                <div class="d-flex gap-2 flex-wrap">
                    <span class="status-badge {{ $company->is_active ? 'status-active' : 'status-inactive' }}">
                        <i class="fas fa-{{ $company->is_active ? 'check' : 'pause' }}-circle"></i>
                        {{ $company->is_active ? 'نشطة' : 'غير نشطة' }}
                    </span>
                    @if($company->zatcaSettings)
                        <span class="status-badge status-zatca">
                            <i class="fas fa-cog"></i>
                            ZATCA مُعد
                        </span>
                    @endif
                </div>
            </div>
            <div class="col-md-3 text-center">
                <div class="h2 mb-0">{{ $company->invoices->count() }}</div>
                <div class="opacity-75">إجمالي الفواتير</div>
            </div>
        </div>
    </div>

    <!-- Statistics -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-number text-primary">{{ $company->invoices->count() }}</div>
            <div class="stat-label">إجمالي الفواتير</div>
        </div>
        
        <div class="stat-card">
            <div class="stat-number text-success">{{ number_format($company->invoices->sum('total_amount'), 0) }}</div>
            <div class="stat-label">إجمالي المبيعات (ر.س)</div>
        </div>
        
        <div class="stat-card">
            <div class="stat-number text-warning">{{ $company->invoices->count() > 0 ? number_format($company->invoices->avg('total_amount'), 0) : '0' }}</div>
            <div class="stat-label">متوسط قيمة الفاتورة (ر.س)</div>
        </div>
        
        @if($company->zatcaSettings)
        <div class="stat-card">
            <div class="stat-number text-info">{{ $company->invoices->where('zatca_status', 'sent')->count() }}</div>
            <div class="stat-label">مُرسلة لـ ZATCA</div>
        </div>
        @endif
    </div>

    <div class="row">
        <!-- Main Content -->
        <div class="col-lg-8">
            <!-- Basic Information -->
            <div class="info-card">
                <div class="info-card-header">
                    <h5><i class="fas fa-info-circle me-2"></i>المعلومات الأساسية</h5>
                </div>
                <div class="info-card-body">
                    <div class="info-grid">
                        <div class="info-item">
                            <div class="info-label">اسم الشركة</div>
                            <div class="info-value">{{ $company->name }}</div>
                        </div>
                        <div class="info-item">
                            <div class="info-label">الرقم الضريبي</div>
                            <div class="info-value">{{ $company->vat_number }}</div>
                        </div>
                        <div class="info-item">
                            <div class="info-label">السجل التجاري</div>
                            <div class="info-value">{{ $company->cr_number }}</div>
                        </div>
                        <div class="info-item">
                            <div class="info-label">معرف الشركة</div>
                            <div class="info-value">#{{ $company->id }}</div>
                        </div>
                        <div class="info-item">
                            <div class="info-label">تاريخ التسجيل</div>
                            <div class="info-value">{{ $company->created_at->format('Y-m-d') }}</div>
                        </div>
                        <div class="info-item">
                            <div class="info-label">آخر تحديث</div>
                            <div class="info-value">{{ $company->updated_at->diffForHumans() }}</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Contact Information -->
            @if($company->email || $company->phone || $company->website)
            <div class="info-card">
                <div class="info-card-header">
                    <h5><i class="fas fa-phone me-2"></i>معلومات الاتصال</h5>
                </div>
                <div class="info-card-body">
                    <div class="info-grid">
                        @if($company->email)
                        <div class="info-item">
                            <div class="info-label">البريد الإلكتروني</div>
                            <div class="info-value">
                                <a href="mailto:{{ $company->email }}" class="text-decoration-none">{{ $company->email }}</a>
                            </div>
                        </div>
                        @endif
                        
                        @if($company->phone)
                        <div class="info-item">
                            <div class="info-label">رقم الهاتف</div>
                            <div class="info-value">
                                <a href="tel:{{ $company->phone }}" class="text-decoration-none">{{ $company->phone }}</a>
                            </div>
                        </div>
                        @endif
                        
                        @if($company->website)
                        <div class="info-item">
                            <div class="info-label">الموقع الإلكتروني</div>
                            <div class="info-value">
                                <a href="{{ $company->website }}" target="_blank" class="text-decoration-none">{{ $company->website }}</a>
                            </div>
                        </div>
                        @endif
                    </div>
                </div>
            </div>
            @endif

            <!-- Address Information -->
            <div class="info-card">
                <div class="info-card-header">
                    <h5><i class="fas fa-map-marker-alt me-2"></i>العنوان الوطني</h5>
                </div>
                <div class="info-card-body">
                    <div class="info-grid">
                        <div class="info-item">
                            <div class="info-label">المدينة</div>
                            <div class="info-value">{{ $company->city }}</div>
                        </div>
                        <div class="info-item">
                            <div class="info-label">الحي</div>
                            <div class="info-value">{{ $company->district }}</div>
                        </div>
                        <div class="info-item">
                            <div class="info-label">الشارع</div>
                            <div class="info-value">{{ $company->street }}</div>
                        </div>
                        <div class="info-item">
                            <div class="info-label">رقم المبنى</div>
                            <div class="info-value">{{ $company->building_number }}</div>
                        </div>
                        <div class="info-item">
                            <div class="info-label">الرقم الإضافي</div>
                            <div class="info-value">{{ $company->additional_number }}</div>
                        </div>
                        <div class="info-item">
                            <div class="info-label">الرمز البريدي</div>
                            <div class="info-value">{{ $company->postal_code }}</div>
                        </div>
                    </div>
                    <div class="mt-3 p-3 bg-light rounded">
                        <strong>العنوان الكامل:</strong> {{ $company->getFullAddressAttribute() }}
                    </div>
                </div>
            </div>

            <!-- ZATCA Settings -->
            <div class="info-card">
                <div class="info-card-header">
                    <h5><i class="fas fa-cog me-2"></i>حالة الربط الإلكتروني مع ZATCA</h5>
                </div>
                <div class="info-card-body">
                    @if($company->zatcaSettings)
                        <div class="info-grid">
                            <div class="info-item">
                                <div class="info-label">نوع الربط</div>
                                <div class="info-value">
                                    <span class="badge bg-{{ $company->zatcaSettings->connection_type == 'clearance' ? 'primary' : 'info' }}">
                                        {{ $company->zatcaSettings->connection_type == 'clearance' ? 'Clearance (B2B)' : 'Reporting (B2C)' }}
                                    </span>
                                </div>
                            </div>
                            <div class="info-item">
                                <div class="info-label">البيئة</div>
                                <div class="info-value">
                                    <span class="badge bg-{{ $company->zatcaSettings->environment == 'sandbox' ? 'warning' : 'success' }}">
                                        {{ $company->zatcaSettings->environment == 'sandbox' ? 'بيئة تجريبية' : 'بيئة إنتاج' }}
                                    </span>
                                </div>
                            </div>
                            <div class="info-item">
                                <div class="info-label">حالة الاتصال</div>
                                <div class="info-value">
                                    @switch($company->zatcaSettings->connection_status)
                                        @case('connected')
                                            <span class="badge bg-success"><i class="fas fa-check me-1"></i>متصل</span>
                                            @break
                                        @case('failed')
                                            <span class="badge bg-danger"><i class="fas fa-times me-1"></i>فشل الاتصال</span>
                                            @break
                                        @default
                                            <span class="badge bg-secondary"><i class="fas fa-plug me-1"></i>غير متصل</span>
                                    @endswitch
                                </div>
                            </div>
                            <div class="info-item">
                                <div class="info-label">عداد الفواتير</div>
                                <div class="info-value">{{ number_format($company->zatcaSettings->invoice_counter) }}</div>
                            </div>
                        </div>
                        
                        @if($company->zatcaSettings->last_error)
                            <div class="alert alert-danger mt-3">
                                <h6><i class="fas fa-exclamation-triangle me-2"></i>آخر خطأ:</h6>
                                <p class="mb-0">{{ $company->zatcaSettings->last_error }}</p>
                            </div>
                        @endif
                    @else
                        <div class="alert alert-warning">
                            <h6><i class="fas fa-exclamation-triangle me-2"></i>لم يتم إعداد ZATCA</h6>
                            <p class="mb-3">يجب إعداد الربط الإلكتروني مع ZATCA لإصدار الفواتير الإلكترونية المعتمدة.</p>
                            <a href="{{ route('zatca-settings.create', ['company_id' => $company->id]) }}" class="btn btn-warning">
                                <i class="fas fa-cog me-2"></i>إعداد ZATCA الآن
                            </a>
                        </div>
                    @endif
                </div>
            </div>

            <!-- Recent Invoices -->
            @if($company->invoices->count() > 0)
            <div class="info-card">
                <div class="info-card-header d-flex justify-content-between align-items-center">
                    <h5><i class="fas fa-file-invoice me-2"></i>آخر الفواتير ({{ $company->invoices->count() }})</h5>
                    <a href="{{ route('invoices.index', ['company_id' => $company->id]) }}" class="btn btn-sm btn-outline-primary">
                        <i class="fas fa-eye me-1"></i>عرض الكل
                    </a>
                </div>
                <div class="table-responsive">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>رقم الفاتورة</th>
                                <th>العميل</th>
                                <th>النوع</th>
                                <th>التاريخ</th>
                                <th>المبلغ</th>
                                <th>حالة ZATCA</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($company->invoices->take(5) as $invoice)
                            <tr>
                                <td>
                                    <a href="{{ route('invoices.show', $invoice) }}" class="text-decoration-none fw-bold">
                                        {{ $invoice->invoice_number }}
                                    </a>
                                </td>
                                <td>{{ $invoice->customer ? Str::limit($invoice->customer->name, 20) : 'غير محدد' }}</td>
                                <td>
                                    <span class="badge bg-{{ $invoice->type == 'standard' ? 'primary' : 'success' }}">
                                        {{ $invoice->type == 'standard' ? 'معيارية' : 'مبسطة' }}
                                    </span>
                                </td>
                                <td>{{ $invoice->issue_date->format('Y-m-d') }}</td>
                                <td class="fw-bold">{{ number_format($invoice->total_amount, 2) }} ر.س</td>
                                <td>
                                    @switch($invoice->zatca_status)
                                        @case('sent')
                                            <span class="badge bg-success"><i class="fas fa-check me-1"></i>مُرسلة</span>
                                            @break
                                        @case('failed')
                                            <span class="badge bg-danger"><i class="fas fa-times me-1"></i>فشل</span>
                                            @break
                                        @default
                                            <span class="badge bg-warning"><i class="fas fa-clock me-1"></i>في الانتظار</span>
                                    @endswitch
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>
            @endif
        </div>

        <!-- Sidebar -->
        <div class="col-lg-4">
            <!-- Quick Actions -->
            <div class="quick-actions">
                <h5 class="mb-3"><i class="fas fa-bolt me-2"></i>إجراءات سريعة</h5>
                <a href="{{ route('invoices.create', ['company_id' => $company->id]) }}" class="btn btn-primary">
                    <i class="fas fa-file-invoice me-2"></i>إنشاء فاتورة جديدة
                </a>
                
                @if($company->zatcaSettings)
                    <a href="{{ route('zatca-settings.show', $company->zatcaSettings) }}" class="btn btn-info">
                        <i class="fas fa-cog me-2"></i>إدارة إعدادات ZATCA
                    </a>
                @else
                    <a href="{{ route('zatca-settings.create', ['company_id' => $company->id]) }}" class="btn btn-warning">
                        <i class="fas fa-plus me-2"></i>إعداد ZATCA
                    </a>
                @endif
                
                <button class="btn btn-success" onclick="exportCompany()">
                    <i class="fas fa-download me-2"></i>تصدير البيانات
                </button>
            </div>

            <!-- Additional Stats -->
            @if($company->invoices->count() > 0)
            <div class="info-card">
                <div class="info-card-header">
                    <h5><i class="fas fa-chart-bar me-2"></i>إحصائيات إضافية</h5>
                </div>
                <div class="info-card-body">
                    <div class="mb-3">
                        <small class="text-muted">متوسط قيمة الفاتورة</small>
                        <div class="h5">{{ number_format($company->invoices->avg('total_amount'), 2) }} ر.س</div>
                    </div>
                    <div class="mb-3">
                        <small class="text-muted">أعلى فاتورة</small>
                        <div class="h5">{{ number_format($company->invoices->max('total_amount'), 2) }} ر.س</div>
                    </div>
                    <div class="mb-3">
                        <small class="text-muted">فواتير معيارية</small>
                        <div class="h5">{{ $company->invoices->where('type', 'standard')->count() }}</div>
                    </div>
                    <div class="mb-0">
                        <small class="text-muted">فواتير مبسطة</small>
                        <div class="h5">{{ $company->invoices->where('type', 'simplified')->count() }}</div>
                    </div>
                </div>
            </div>
            @endif

            @if($company->invoices->count() > 0)
            <div class="alert alert-info">
                <i class="fas fa-info-circle me-2"></i>
                <small>هذه الشركة لديها {{ $company->invoices->count() }} فاتورة. لا يمكن حذفها لحماية سلامة البيانات المالية.</small>
            </div>
            @endif
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
function exportCompany() {
    const companyData = {
        name: '{{ $company->name }}',
        vat_number: '{{ $company->vat_number }}',
        cr_number: '{{ $company->cr_number }}',
        email: '{{ $company->email }}',
        phone: '{{ $company->phone }}',
        website: '{{ $company->website }}',
        address: '{{ $company->getFullAddressAttribute() }}',
        invoices_count: {{ $company->invoices->count() }},
        total_sales: {{ $company->invoices->sum('total_amount') }},
        zatca_configured: {{ $company->zatcaSettings ? 'true' : 'false' }},
        registration_date: '{{ $company->created_at->format('Y-m-d') }}'
    };
    
    const dataStr = JSON.stringify(companyData, null, 2);
    const dataBlob = new Blob([dataStr], {type: 'application/json'});
    const url = URL.createObjectURL(dataBlob);
    const link = document.createElement('a');
    link.href = url;
    link.download = 'company_{{ $company->id }}_data.json';
    link.click();
    
    alert('تم تصدير بيانات الشركة بنجاح!');
}

function toggleCompanyStatus() {
    const isActive = {{ $company->is_active ? 'true' : 'false' }};
    const action = isActive ? 'تعطيل' : 'تفعيل';
    
    if (confirm(`هل تريد ${action} هذه الشركة؟`)) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = '{{ route("companies.update", $company) }}';
        
        const csrfInput = document.createElement('input');
        csrfInput.type = 'hidden';
        csrfInput.name = '_token';
        csrfInput.value = '{{ csrf_token() }}';
        form.appendChild(csrfInput);
        
        const methodInput = document.createElement('input');
        methodInput.type = 'hidden';
        methodInput.name = '_method';
        methodInput.value = 'PUT';
        form.appendChild(methodInput);
        
        const statusInput = document.createElement('input');
        statusInput.type = 'hidden';
        statusInput.name = 'is_active';
        statusInput.value = isActive ? '0' : '1';
        form.appendChild(statusInput);
        
        document.body.appendChild(form);
        form.submit();
    }
}

function deleteCompany() {
    const hasInvoices = {{ $company->invoices->count() }};
    
    if (hasInvoices > 0) {
        alert(`لا يمكن حذف هذه الشركة لأنها تحتوي على ${hasInvoices} فاتورة. يجب حذف جميع الفواتير أولاً.`);
        return;
    }
    
    const userInput = prompt('اكتب "حذف" للتأكيد من حذف الشركة:');
    
    if (userInput === 'حذف') {
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = '{{ route("companies.destroy", $company) }}';
        
        const csrfInput = document.createElement('input');
        csrfInput.type = 'hidden';
        csrfInput.name = '_token';
        csrfInput.value = '{{ csrf_token() }}';
        form.appendChild(csrfInput);
        
        const methodInput = document.createElement('input');
        methodInput.type = 'hidden';
        methodInput.name = '_method';
        methodInput.value = 'DELETE';
        form.appendChild(methodInput);
        
        document.body.appendChild(form);
        form.submit();
    } else if (userInput !== null) {
        alert('لم يتم حذف الشركة. يجب كتابة "حذف" بالضبط للتأكيد.');
    }
}
</script>
@endpush