@extends('layouts.app')

@section('title', 'الشركات')
@section('page-title', 'إدارة الشركات')

@section('page-actions')
<div class="d-flex gap-2">
    <a href="{{ route('companies.create') }}" class="btn btn-primary btn-lg">
        <i class="fas fa-plus me-2"></i>
        إضافة شركة جديدة
    </a>
    <button type="button" class="btn btn-success btn-lg" onclick="exportData()">
        <i class="fas fa-download me-2"></i>
        تصدير البيانات
    </button>
</div>
@endsection

@push('styles')
<style>
    .companies-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        border-radius: 15px;
        padding: 30px;
        margin-bottom: 30px;
        position: relative;
        overflow: hidden;
    }
    
    .companies-header::before {
        content: '';
        position: absolute;
        top: -50%;
        right: -10%;
        width: 200px;
        height: 200px;
        background: rgba(255,255,255,0.1);
        border-radius: 50%;
        transform: rotate(45deg);
    }
    
    .company-card {
        border: none;
        border-radius: 15px;
        box-shadow: 0 8px 25px rgba(0,0,0,0.1);
        transition: all 0.3s ease;
        overflow: hidden;
        margin-bottom: 20px;
    }
    
    .company-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 15px 35px rgba(0,0,0,0.15);
    }
    
    .company-avatar {
        width: 60px;
        height: 60px;
        background: linear-gradient(135deg, #3498db, #2980b9);
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-size: 24px;
        margin-left: 15px;
    }
    
    .status-badge {
        padding: 8px 15px;
        border-radius: 20px;
        font-size: 0.85rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .zatca-connected {
        background: linear-gradient(135deg, #27ae60, #2ecc71);
        color: white;
        box-shadow: 0 3px 10px rgba(39, 174, 96, 0.3);
    }
    
    .zatca-failed {
        background: linear-gradient(135deg, #e74c3c, #c0392b);
        color: white;
        box-shadow: 0 3px 10px rgba(231, 76, 60, 0.3);
    }
    
    .zatca-pending {
        background: linear-gradient(135deg, #f39c12, #e67e22);
        color: white;
        box-shadow: 0 3px 10px rgba(243, 156, 18, 0.3);
    }
    
    .zatca-inactive {
        background: linear-gradient(135deg, #95a5a6, #7f8c8d);
        color: white;
        box-shadow: 0 3px 10px rgba(149, 165, 166, 0.3);
    }
    
    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 20px;
        margin-bottom: 30px;
    }
    
    .stat-card {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        border-radius: 15px;
        padding: 25px;
        text-align: center;
        position: relative;
        overflow: hidden;
    }
    
    .stat-card::before {
        content: '';
        position: absolute;
        top: -50%;
        left: -50%;
        width: 200%;
        height: 200%;
        background: radial-gradient(circle, rgba(255,255,255,0.1) 0%, transparent 70%);
        transform: scale(0);
        transition: transform 0.6s ease;
    }
    
    .stat-card:hover::before {
        transform: scale(1);
    }
    
    .stat-number {
        font-size: 2.5rem;
        font-weight: 700;
        margin-bottom: 10px;
        position: relative;
        z-index: 1;
    }
    
    .stat-label {
        font-size: 1rem;
        opacity: 0.9;
        position: relative;
        z-index: 1;
    }
    
    .action-buttons {
        display: flex;
        gap: 8px;
        flex-wrap: wrap;
    }
    
    .btn-action {
        padding: 8px 12px;
        border-radius: 8px;
        border: none;
        font-size: 0.9rem;
        transition: all 0.3s ease;
        position: relative;
        overflow: hidden;
    }
    
    .btn-action::before {
        content: '';
        position: absolute;
        top: 50%;
        left: 50%;
        width: 0;
        height: 0;
        background: rgba(255,255,255,0.2);
        border-radius: 50%;
        transform: translate(-50%, -50%);
        transition: all 0.3s ease;
    }
    
    .btn-action:hover::before {
        width: 100%;
        height: 100%;
    }
    
    .btn-view {
        background: linear-gradient(135deg, #3498db, #2980b9);
        color: white;
    }
    
    .btn-edit {
        background: linear-gradient(135deg, #f39c12, #e67e22);
        color: white;
    }
    
    .btn-more {
        background: linear-gradient(135deg, #95a5a6, #7f8c8d);
        color: white;
    }
    
    .empty-state {
        text-align: center;
        padding: 60px 20px;
        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
        border-radius: 15px;
        margin: 20px 0;
    }
    
    .empty-state-icon {
        font-size: 4rem;
        color: #6c757d;
        margin-bottom: 20px;
    }
    
    .company-grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
        gap: 20px;
        margin-top: 20px;
    }
    
    .contact-info {
        display: flex;
        flex-direction: column;
        gap: 5px;
    }
    
    .contact-item {
        display: flex;
        align-items: center;
        font-size: 0.9rem;
        color: #6c757d;
    }
    
    .contact-item i {
        width: 16px;
        margin-left: 8px;
    }
    
    .pagination-wrapper {
        margin-top: 30px;
        display: flex;
        justify-content: center;
    }
    
    .info-panel {
        background: linear-gradient(135deg, #e3f2fd 0%, #bbdefb 100%);
        border: none;
        border-radius: 15px;
        margin-top: 30px;
    }
    
    .info-panel .card-header {
        background: linear-gradient(135deg, #2196f3, #1976d2);
        border-radius: 15px 15px 0 0;
    }
    
    .requirements-list {
        list-style: none;
        padding: 0;
    }
    
    .requirements-list li {
        padding: 10px 0;
        border-bottom: 1px solid rgba(0,0,0,0.1);
        position: relative;
        padding-right: 30px;
    }
    
    .requirements-list li:last-child {
        border-bottom: none;
    }
    
    .requirements-list li::before {
        content: '✓';
        position: absolute;
        right: 0;
        top: 10px;
        color: #27ae60;
        font-weight: bold;
    }
</style>
@endpush

@section('content')
<!-- Header Section -->
<div class="companies-header">
    <div class="row align-items-center">
        <div class="col-md-8">
            <h2 class="mb-2">
                <i class="fas fa-building me-3"></i>
                نظام إدارة الشركات
            </h2>
            <p class="mb-0 opacity-75">إدارة شاملة لبيانات الشركات ومتطلبات ZATCA</p>
        </div>
        <div class="col-md-4 text-end">
            <div class="d-flex align-items-center justify-content-end">
                <div class="text-end">
                    <div class="h4 mb-0">{{ $companies->total() }}</div>
                    <small class="opacity-75">إجمالي الشركات</small>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Statistics Cards -->
@if($companies->count() > 0)
<div class="stats-grid">
    <div class="stat-card">
        <div class="stat-number">{{ $companies->where('is_active', true)->count() }}</div>
        <div class="stat-label text-white">
            <i class="fas fa-check-circle me-2"></i>
            شركات نشطة
        </div>
    </div>
    <div class="stat-card">
        <div class="stat-number">{{ $companies->filter(function($c) { return $c->zatcaSettings && $c->zatcaSettings->is_active; })->count() }}</div>
        <div class="stat-label text-white">
            <i class="fas fa-link me-2"></i>
            مرتبطة بـ ZATCA
        </div>
    </div>
    <div class="stat-card">
        <div class="stat-number">{{ $companies->sum(function($c) { return $c->invoices->count(); }) }}</div>
        <div class="stat-label text-white">
            <i class="fas fa-file-invoice me-2"></i>
            إجمالي الفواتير
        </div>
    </div>
    <div class="stat-card">
        <div class="stat-number">{{ number_format($companies->sum(function($c) { return $c->invoices->sum('total_amount'); }), 0) }}</div>
        <div class="stat-label text-white">
            <i class="fas fa-money-bill-wave me-2"></i>
            إجمالي المبيعات (ر.س)
        </div>
    </div>
</div>
@endif

<!-- Companies Grid -->
@if($companies->count() > 0)
    <div class="company-grid">
        @foreach($companies as $company)
        <div class="company-card card">
            <div class="card-body p-4">
                <!-- Company Header -->
                <div class="d-flex align-items-start mb-3">
                    <div class="company-avatar">
                        <i class="fas fa-building"></i>
                    </div>
                    <div class="flex-grow-1">
                        <h5 class="card-title mb-1">
                            <a href="{{ route('companies.show', $company) }}" class="text-decoration-none text-dark">
                                {{ $company->name }}
                            </a>
                        </h5>
                        <p class="text-muted mb-2 small">{{ Str::limit($company->getFullAddressAttribute(), 50) }}</p>
                        
                        <!-- Status Badges -->
                        <div class="d-flex gap-2 mb-2">
                            @if($company->is_active)
                                <span class="badge bg-success">نشطة</span>
                            @else
                                <span class="badge bg-secondary">غير نشطة</span>
                            @endif
                            
                            @php $zatcaSettings = $company->zatcaSettings; @endphp
                            @if($zatcaSettings)
                                @if($zatcaSettings->is_active)
                                    @switch($zatcaSettings->connection_status)
                                        @case('connected')
                                            <span class="status-badge zatca-connected">
                                                <i class="fas fa-check-circle me-1"></i>
                                                ZATCA متصل
                                            </span>
                                            @break
                                        @case('failed')
                                            <span class="status-badge zatca-failed">
                                                <i class="fas fa-times-circle me-1"></i>
                                                ZATCA فشل
                                            </span>
                                            @break
                                        @default
                                            <span class="status-badge zatca-pending">
                                                <i class="fas fa-clock me-1"></i>
                                                ZATCA انتظار
                                            </span>
                                    @endswitch
                                @else
                                    <span class="status-badge zatca-inactive">
                                        <i class="fas fa-pause-circle me-1"></i>
                                        ZATCA غير نشط
                                    </span>
                                @endif
                            @else
                                <span class="status-badge zatca-inactive">
                                    <i class="fas fa-exclamation-circle me-1"></i>
                                    ZATCA غير مُعد
                                </span>
                            @endif
                        </div>
                    </div>
                </div>
                
                <!-- Company Details -->
                <div class="row mb-3">
                    <div class="col-6">
                        <small class="text-muted d-block">الرقم الضريبي</small>
                        <strong class="text-primary">{{ $company->vat_number }}</strong>
                    </div>
                    <div class="col-6">
                        <small class="text-muted d-block">السجل التجاري</small>
                        <strong class="text-info">{{ $company->cr_number }}</strong>
                    </div>
                </div>
                
                <!-- Contact Information -->
                <div class="contact-info mb-3">
                    @if($company->email)
                        <div class="contact-item">
                            <i class="fas fa-envelope text-primary"></i>
                            <a href="mailto:{{ $company->email }}" class="text-decoration-none">{{ $company->email }}</a>
                        </div>
                    @endif
                    @if($company->phone)
                        <div class="contact-item">
                            <i class="fas fa-phone text-success"></i>
                            <a href="tel:{{ $company->phone }}" class="text-decoration-none">{{ $company->phone }}</a>
                        </div>
                    @endif
                    @if($company->website)
                        <div class="contact-item">
                            <i class="fas fa-globe text-info"></i>
                            <a href="{{ $company->website }}" target="_blank" class="text-decoration-none">الموقع الإلكتروني</a>
                        </div>
                    @endif
                </div>
                
                <!-- Statistics -->
                <div class="row text-center mb-3">
                    <div class="col-6">
                        <div class="small text-muted">الفواتير</div>
                        <div class="fw-bold text-primary">{{ $company->invoices->count() }}</div>
                    </div>
                    <div class="col-6">
                        <div class="small text-muted">المبيعات</div>
                        <div class="fw-bold text-success">{{ number_format($company->invoices->sum('total_amount'), 0) }}</div>
                    </div>
                </div>
                
                <!-- Action Buttons -->
                <div class="action-buttons">
                    <a href="{{ route('companies.show', $company) }}" class="btn btn-action btn-view" title="عرض التفاصيل">
                        <i class="fas fa-eye"></i>
                    </a>
                    <a href="{{ route('companies.edit', $company) }}" class="btn btn-action btn-edit" title="تعديل">
                        <i class="fas fa-edit"></i>
                    </a>
                    <div class="dropdown">
                        <button class="btn btn-action btn-more dropdown-toggle" type="button" data-bs-toggle="dropdown" title="المزيد">
                            <i class="fas fa-ellipsis-v"></i>
                        </button>
                        <ul class="dropdown-menu">
                            @if(!$company->zatcaSettings)
                                <li>
                                    <a class="dropdown-item" href="{{ route('zatca-settings.create', ['company_id' => $company->id]) }}">
                                        <i class="fas fa-cog me-2 text-warning"></i>
                                        إعداد ZATCA
                                    </a>
                                </li>
                            @else
                                <li>
                                    <a class="dropdown-item" href="{{ route('zatca-settings.show', $company->zatcaSettings) }}">
                                        <i class="fas fa-cog me-2 text-info"></i>
                                        إدارة ZATCA
                                    </a>
                                </li>
                            @endif
                            <li>
                                <a class="dropdown-item" href="{{ route('invoices.create', ['company_id' => $company->id]) }}">
                                    <i class="fas fa-file-invoice me-2 text-primary"></i>
                                    إنشاء فاتورة
                                </a>
                            </li>
                            <li><hr class="dropdown-divider"></li>
                            <li>
                                <form method="POST" action="{{ route('companies.destroy', $company) }}" style="display: inline;">
                                    @csrf
                                    @method('DELETE')
                                    <button type="submit" class="dropdown-item text-danger"
                                            onclick="return confirm('هل تريد حذف هذه الشركة؟\nسيتم حذف جميع البيانات المرتبطة بها.')">
                                        <i class="fas fa-trash me-2"></i>
                                        حذف الشركة
                                    </button>
                                </form>
                            </li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
        @endforeach
    </div>
    
    <!-- Pagination -->
    <div class="pagination-wrapper">
        {{ $companies->links() }}
    </div>
@else
    <!-- Empty State -->
    <div class="empty-state">
        <div class="empty-state-icon">
            <i class="fas fa-building"></i>
        </div>
        <h4 class="text-muted mb-3">لا توجد شركات مسجلة</h4>
        <p class="text-muted mb-4">ابدأ بإنشاء شركتك الأولى لتتمكن من إصدار الفواتير الإلكترونية</p>
        <a href="{{ route('companies.create') }}" class="btn btn-primary btn-lg">
            <i class="fas fa-plus me-2"></i>
            إنشاء شركة جديدة
        </a>
    </div>
@endif

<!-- Information Panel -->
<div class="card info-panel">
    <div class="card-header">
        <h6 class="card-title mb-0">
            <i class="fas fa-info-circle me-2"></i>
            متطلبات ZATCA للشركات
        </h6>
    </div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-6">
                <h6 class="text-primary mb-3">
                    <i class="fas fa-check-circle me-2"></i>
                    المتطلبات الأساسية
                </h6>
                <ul class="requirements-list">
                    <li>الرقم الضريبي 15 رقم يبدأ بـ 3</li>
                    <li>رقم السجل التجاري صحيح ومُسجل</li>
                    <li>العنوان الوطني السعودي كامل</li>
                    <li>معلومات الاتصال محدثة</li>
                </ul>
            </div>
            <div class="col-md-6">
                <h6 class="text-success mb-3">
                    <i class="fas fa-cog me-2"></i>
                    إعدادات ZATCA
                </h6>
                <ul class="requirements-list">
                    <li>إعداد شهادة رقمية من ZATCA</li>
                    <li>اختبار الاتصال مع خوادم ZATCA</li>
                    <li>تفعيل الإرسال التلقائي للفواتير</li>
                    <li>مراقبة حالة الربط باستمرار</li>
                </ul>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
function exportData() {
    // تجميع بيانات الشركات للتصدير
    const companies = @json($companies->items());
    
    const exportData = companies.map(company => ({
        'اسم الشركة': company.name,
        'الرقم الضريبي': company.vat_number,
        'السجل التجاري': company.cr_number,
        'المدينة': company.city,
        'الحي': company.district,
        'البريد الإلكتروني': company.email || 'غير متوفر',
        'الهاتف': company.phone || 'غير متوفر',
        'الموقع الإلكتروني': company.website || 'غير متوفر',
        'الحالة': company.is_active ? 'نشطة' : 'غير نشطة',
        'حالة ZATCA': getZatcaStatus(company),
        'عدد الفواتير': company.invoices ? company.invoices.length : 0,
        'تاريخ التسجيل': company.created_at
    }));
    
    // تحويل إلى CSV
    const csvContent = convertToCSV(exportData);
    
    // تنزيل الملف
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.download = 'companies_export_' + new Date().toISOString().split('T')[0] + '.csv';
    link.click();
}

function getZatcaStatus(company) {
    if (!company.zatca_settings) return 'غير مُعد';
    if (!company.zatca_settings.is_active) return 'غير نشط';
    
    switch(company.zatca_settings.connection_status) {
        case 'connected': return 'متصل';
        case 'failed': return 'فشل';
        default: return 'غير مختبر';
    }
}

function convertToCSV(data) {
    if (!data || !data.length) return '';
    
    const headers = Object.keys(data[0]);
    const csvHeaders = headers.join(',');
    
    const csvRows = data.map(row =>
        headers.map(header => {
            const value = row[header];
            return typeof value === 'string' && value.includes(',') ? `"${value}"` : value;
        }).join(',')
    );
    
    return [csvHeaders, ...csvRows].join('\n');
}

// تحديث الصفحة كل 30 ثانية لمراقبة حالة ZATCA
setInterval(() => {
    // تحديث حالة ZATCA للشركات المعروضة
    const statusElements = document.querySelectorAll('[data-company-id]');
    statusElements.forEach(element => {
        const companyId = element.dataset.companyId;
        // يمكن إضافة AJAX call هنا لتحديث الحالة
    });
}, 30000);
</script>
@endpush