@extends('layouts.app')

@section('title', 'تعديل الشركة: ' . $company->name)
@section('page-title', 'تعديل الشركة')

@section('page-actions')
<div class="d-flex gap-2">
    <a href="{{ route('companies.show', $company) }}" class="btn btn-info">
        <i class="fas fa-eye me-2"></i>
        عرض الشركة
    </a>
    <a href="{{ route('companies.index') }}" class="btn btn-secondary">
        <i class="fas fa-arrow-left me-2"></i>
        العودة للقائمة
    </a>
    <button type="button" class="btn btn-warning" onclick="resetForm()">
        <i class="fas fa-undo me-2"></i>
        استعادة القيم الأصلية
    </button>
</div>
@endsection

@push('styles')
<style>
    /* Professional Design Variables */
    :root {
        --primary-color: #2563eb;
        --primary-dark: #1d4ed8;
        --primary-light: #3b82f6;
        --secondary-color: #64748b;
        --success-color: #059669;
        --warning-color: #d97706;
        --danger-color: #dc2626;
        --gray-50: #f8fafc;
        --gray-100: #f1f5f9;
        --gray-200: #e2e8f0;
        --gray-300: #cbd5e1;
        --gray-500: #64748b;
        --gray-700: #334155;
        --gray-900: #0f172a;
        --border-radius: 12px;
        --border-radius-lg: 16px;
        --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
        --shadow: 0 1px 3px 0 rgb(0 0 0 / 0.1), 0 1px 2px -1px rgb(0 0 0 / 0.1);
        --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
        --shadow-xl: 0 20px 25px -5px rgb(0 0 0 / 0.1), 0 8px 10px -6px rgb(0 0 0 / 0.1);
    }

    /* Professional Header */
    .edit-header {
        background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
        color: white;
        border-radius: var(--border-radius-lg);
        padding: 2rem;
        margin-bottom: 2rem;
        position: relative;
        overflow: hidden;
        box-shadow: var(--shadow-xl);
        border: 1px solid rgba(255, 255, 255, 0.1);
    }
    
    .edit-header::before {
        content: '';
        position: absolute;
        top: 0;
        right: 0;
        width: 100%;
        height: 100%;
        background: url("data:image/svg+xml,%3Csvg width='60' height='60' viewBox='0 0 60 60' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='none' fill-rule='evenodd'%3E%3Cg fill='%23ffffff' fill-opacity='0.05'%3E%3Ccircle cx='30' cy='30' r='4'/%3E%3C/g%3E%3C/g%3E%3C/svg%3E") repeat;
        animation: none;
    }
    
    /* Professional Form Sections */
    .form-section {
        background: white;
        border-radius: var(--border-radius-lg);
        box-shadow: var(--shadow);
        margin-bottom: 1.5rem;
        overflow: hidden;
        transition: all 0.2s ease-in-out;
        border: 1px solid var(--gray-200);
    }
    
    .form-section:hover {
        box-shadow: var(--shadow-lg);
        transform: translateY(-1px);
        border-color: var(--primary-light);
    }
    
    .form-section.changed {
        border-left: 4px solid var(--warning-color);
        box-shadow: var(--shadow-lg);
    }
    
    .section-header {
        background: linear-gradient(135deg, var(--gray-900) 0%, var(--gray-700) 100%);
        color: white;
        padding: 1.25rem 1.5rem;
        border: none;
        position: relative;
        overflow: hidden;
    }
    
    .section-header::before {
        content: '';
        position: absolute;
        top: 0;
        left: -100%;
        width: 100%;
        height: 100%;
        background: linear-gradient(90deg, transparent, rgba(255,255,255,0.1), transparent);
        transition: left 0.3s ease;
    }
    
    .section-header:hover::before {
        left: 100%;
    }
    
    .section-title {
        font-size: 1rem;
        font-weight: 600;
        margin: 0;
        display: flex;
        align-items: center;
        letter-spacing: -0.025em;
    }
    
    .section-title i {
        margin-left: 0.5rem;
        font-size: 1.125rem;
        opacity: 0.9;
    }
    
    /* Professional Form Controls */
    .form-group-enhanced {
        margin-bottom: 1.5rem;
        position: relative;
    }
    
    .form-label-enhanced {
        font-weight: 500;
        color: var(--gray-700);
        margin-bottom: 0.5rem;
        display: flex;
        align-items: center;
        font-size: 0.875rem;
        letter-spacing: -0.025em;
    }
    
    .form-label-enhanced .required {
        color: var(--danger-color);
        margin-right: 0.25rem;
        font-size: 1rem;
    }
    
    .form-control-enhanced {
        border: 1px solid var(--gray-300);
        border-radius: var(--border-radius);
        padding: 0.75rem 1rem;
        font-size: 0.875rem;
        transition: all 0.15s ease-in-out;
        background: white;
        width: 100%;
        line-height: 1.5;
    }
    
    .form-control-enhanced:focus {
        border-color: var(--primary-color);
        box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        background: white;
        outline: none;
    }
    
    .form-control-enhanced:hover:not(:focus) {
        border-color: var(--gray-400);
    }
    
    .form-control-enhanced.is-valid {
        border-color: var(--success-color);
        background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 20 20' fill='%23059669'%3e%3cpath fill-rule='evenodd' d='M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z' clip-rule='evenodd'/%3e%3c/svg%3e");
        background-repeat: no-repeat;
        background-position: right 0.75rem center;
        background-size: 1rem;
        padding-right: 2.5rem;
    }
    
    .form-control-enhanced.is-invalid {
        border-color: var(--danger-color);
        background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 20 20' fill='%23dc2626'%3e%3cpath fill-rule='evenodd' d='M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z' clip-rule='evenodd'/%3e%3c/svg%3e");
        background-repeat: no-repeat;
        background-position: right 0.75rem center;
        background-size: 1rem;
        padding-right: 2.5rem;
    }
    
    .form-control-enhanced.changed {
        border-color: var(--warning-color);
        background-color: #fef3c7;
    }
    
    .help-text {
        font-size: 0.75rem;
        color: var(--gray-500);
        margin-top: 0.25rem;
        display: flex;
        align-items: center;
        line-height: 1.4;
    }
    
    .help-text i {
        margin-left: 0.25rem;
        color: var(--primary-color);
        opacity: 0.7;
    }
    
    /* Professional Preview Card */
    .preview-card {
        background: white;
        border: 1px solid var(--gray-200);
        border-radius: var(--border-radius-lg);
        overflow: hidden;
        position: sticky;
        top: 1.5rem;
        box-shadow: var(--shadow);
        transition: all 0.2s ease-in-out;
    }
    
    .preview-card:hover {
        box-shadow: var(--shadow-lg);
        transform: translateY(-1px);
    }
    
    .preview-header {
        background: linear-gradient(135deg, var(--gray-900) 0%, var(--gray-700) 100%);
        color: white;
        padding: 1.25rem 1.5rem;
        position: relative;
        overflow: hidden;
    }
    
    .preview-header::before {
        content: '';
        position: absolute;
        top: 0;
        left: -100%;
        width: 100%;
        height: 100%;
        background: linear-gradient(90deg, transparent, rgba(255,255,255,0.1), transparent);
        transition: left 0.3s ease;
    }
    
    .preview-header:hover::before {
        left: 100%;
    }
    
    .company-preview {
        background: white;
        padding: 1.5rem;
        text-align: center;
    }
    
    .preview-avatar {
        width: 4rem;
        height: 4rem;
        background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        margin: 0 auto 1rem;
        color: white;
        font-size: 1.5rem;
        box-shadow: var(--shadow);
        transition: all 0.2s ease-in-out;
    }
    
    .preview-avatar:hover {
        transform: scale(1.05);
        box-shadow: var(--shadow-lg);
    }
    
    .preview-name {
        font-size: 1.125rem;
        font-weight: 600;
        color: var(--gray-900);
        margin-bottom: 0.25rem;
        letter-spacing: -0.025em;
    }
    
    .preview-vat {
        font-size: 0.75rem;
        color: var(--gray-500);
        font-family: ui-monospace, SFMono-Regular, "SF Mono", Monaco, Inconsolata, "Liberation Mono", "Consolas", monospace;
        background: var(--gray-100);
        padding: 0.25rem 0.5rem;
        border-radius: 0.375rem;
        display: inline-block;
        margin-bottom: 1rem;
        letter-spacing: 0.05em;
    }
    
    .preview-section {
        text-align: right;
        margin-bottom: 1rem;
        padding: 0.75rem;
        background: var(--gray-50);
        border-radius: var(--border-radius);
        border: 1px solid var(--gray-200);
    }
    
    .preview-section-title {
        font-weight: 500;
        color: var(--gray-700);
        margin-bottom: 0.25rem;
        font-size: 0.75rem;
        text-transform: uppercase;
        letter-spacing: 0.05em;
    }
    
    .preview-section-content {
        color: var(--gray-600);
        font-size: 0.875rem;
        line-height: 1.4;
    }
    
    /* Professional Save Button */
    .save-btn {
        background: var(--primary-color);
        border: none;
        border-radius: var(--border-radius);
        padding: 0.75rem 1.5rem;
        color: white;
        font-weight: 500;
        font-size: 0.875rem;
        transition: all 0.15s ease-in-out;
        width: 100%;
        box-shadow: var(--shadow-sm);
        letter-spacing: -0.025em;
    }
    
    .save-btn:hover:not(:disabled) {
        background: var(--primary-dark);
        box-shadow: var(--shadow);
        transform: translateY(-1px);
    }
    
    .save-btn:active {
        transform: translateY(0);
        box-shadow: var(--shadow-sm);
    }
    
    .save-btn:disabled {
        opacity: 0.5;
        cursor: not-allowed;
        transform: none;
        box-shadow: var(--shadow-sm);
    }
    
    /* Professional Stats Cards */
    .stats-card {
        background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
        color: white;
        border-radius: var(--border-radius-lg);
        padding: 1.5rem;
        margin-bottom: 1rem;
        text-align: center;
        transition: all 0.2s ease-in-out;
        box-shadow: var(--shadow);
    }
    
    .stats-card:hover {
        transform: translateY(-2px);
        box-shadow: var(--shadow-lg);
    }
    
    .stats-number {
        font-size: 1.875rem;
        font-weight: 700;
        margin-bottom: 0.25rem;
        letter-spacing: -0.025em;
    }
    
    .stats-label {
        font-size: 0.875rem;
        opacity: 0.9;
        font-weight: 400;
    }
    
    /* Professional ZATCA Status Card */
    .zatca-status-card {
        background: linear-gradient(135deg, var(--gray-900) 0%, var(--gray-700) 100%);
        color: white;
        border-radius: var(--border-radius-lg);
        padding: 1.5rem;
        margin-bottom: 1rem;
        transition: all 0.2s ease-in-out;
        box-shadow: var(--shadow);
    }
    
    .zatca-status-card:hover {
        transform: translateY(-2px);
        box-shadow: var(--shadow-lg);
    }
    
    .zatca-status-header {
        display: flex;
        align-items: center;
        margin-bottom: 1rem;
    }
    
    .zatca-status-icon {
        width: 2.5rem;
        height: 2.5rem;
        background: rgba(255,255,255,0.1);
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        margin-left: 0.75rem;
    }
    
    /* Professional Status Badges */
    .status-badge {
        padding: 0.25rem 0.75rem;
        border-radius: 9999px;
        font-size: 0.75rem;
        font-weight: 500;
        display: inline-block;
        margin: 0.125rem;
        letter-spacing: 0.025em;
    }
    
    .status-connected {
        background: var(--success-color);
        color: white;
    }
    
    .status-failed {
        background: var(--danger-color);
        color: white;
    }
    
    .status-pending {
        background: var(--warning-color);
        color: white;
    }
    
    .status-inactive {
        background: var(--gray-500);
        color: white;
    }
    
    /* Professional Changes Indicator */
    .changes-indicator {
        position: fixed;
        top: 50%;
        left: 1.5rem;
        transform: translateY(-50%);
        background: var(--warning-color);
        color: white;
        padding: 0.75rem 1rem;
        border-radius: 9999px;
        box-shadow: var(--shadow-lg);
        z-index: 1000;
        opacity: 0;
        visibility: hidden;
        transition: all 0.2s ease-in-out;
        font-size: 0.875rem;
        font-weight: 500;
        backdrop-filter: blur(8px);
        border: 1px solid rgba(255,255,255,0.1);
    }
    
    .changes-indicator.show {
        opacity: 1;
        visibility: visible;
    }
    
    /* Professional Warning Card */
    .warning-card {
        background: #fef3c7;
        border: 1px solid #f59e0b;
        border-radius: var(--border-radius-lg);
        margin-top: 1rem;
        box-shadow: var(--shadow-sm);
    }
    
    .warning-header {
        background: var(--warning-color);
        color: white;
        padding: 1rem 1.25rem;
        border-radius: var(--border-radius-lg) var(--border-radius-lg) 0 0;
        font-weight: 500;
    }
    
    /* Professional Scrollbar */
    ::-webkit-scrollbar {
        width: 6px;
    }
    
    ::-webkit-scrollbar-track {
        background: var(--gray-100);
        border-radius: 3px;
    }
    
    ::-webkit-scrollbar-thumb {
        background: var(--gray-300);
        border-radius: 3px;
    }
    
    ::-webkit-scrollbar-thumb:hover {
        background: var(--gray-400);
    }
    
    /* Professional Loading Overlay */
    .loading-overlay {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(255,255,255,0.95);
        display: flex;
        justify-content: center;
        align-items: center;
        z-index: 9999;
        opacity: 0;
        visibility: hidden;
        transition: all 0.2s ease;
        backdrop-filter: blur(4px);
    }
    
    .loading-overlay.show {
        opacity: 1;
        visibility: visible;
    }
    
    .loading-spinner {
        width: 2rem;
        height: 2rem;
        border: 2px solid var(--gray-200);
        border-top: 2px solid var(--primary-color);
        border-radius: 50%;
        animation: spin 1s linear infinite;
    }
    
    @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }
    
    /* Professional Input Groups */
    .input-group .input-group-text {
        border-radius: var(--border-radius) 0 0 var(--border-radius);
        background: var(--gray-50);
        border: 1px solid var(--gray-300);
        border-right: none;
        color: var(--gray-500);
        transition: all 0.15s ease-in-out;
    }
    
    .input-group .form-control-enhanced {
        border-radius: 0 var(--border-radius) var(--border-radius) 0;
        border-left: none;
    }
    
    .input-group .form-control-enhanced:focus ~ .input-group-text,
    .input-group .input-group-text:has(~ .form-control-enhanced:focus) {
        border-color: var(--primary-color);
        background: white;
        color: var(--primary-color);
    }
    
    /* Professional Badges */
    .badge {
        font-size: 0.75rem !important;
        padding: 0.25rem 0.5rem;
        border-radius: 0.375rem;
        font-weight: 500;
        letter-spacing: 0.025em;
    }
    
    /* Professional Cards */
    .card {
        border: 1px solid var(--gray-200);
        border-radius: var(--border-radius-lg);
        box-shadow: var(--shadow-sm);
        transition: all 0.15s ease-in-out;
    }
    
    .card:hover {
        box-shadow: var(--shadow);
    }
    
    .card-header {
        background: var(--gray-50);
        border-bottom: 1px solid var(--gray-200);
        padding: 1rem 1.25rem;
        font-weight: 500;
    }
    
    .card-body {
        padding: 1.25rem;
    }
    
    /* Professional Alert */
    .alert {
        border-radius: var(--border-radius);
        padding: 0.75rem 1rem;
        margin-bottom: 1rem;
        border: 1px solid transparent;
    }
    
    .alert-warning {
        background-color: #fef3c7;
        border-color: #f59e0b;
        color: #92400e;
    }
    
    /* Logo Upload Styles */
    .logo-upload-container {
        position: relative;
    }
    
    .logo-preview {
        width: 150px;
        height: 150px;
        border: 2px dashed var(--gray-300);
        border-radius: var(--border-radius-lg);
        display: flex;
        align-items: center;
        justify-content: center;
        margin-bottom: 15px;
        transition: all 0.3s ease;
        cursor: pointer;
        background: var(--gray-50);
        overflow: hidden;
    }
    
    .logo-preview:hover {
        border-color: var(--primary-color);
        background: #f0f8ff;
        transform: translateY(-2px);
        box-shadow: var(--shadow-lg);
    }
    
    .logo-preview.has-image {
        border: 2px solid var(--success-color);
        background: white;
    }
    
    .logo-placeholder {
        text-align: center;
        color: var(--gray-500);
        transition: all 0.3s ease;
    }
    
    .logo-placeholder i {
        font-size: 3rem;
        margin-bottom: 10px;
        display: block;
        color: var(--primary-color);
    }
    
    .logo-placeholder p {
        margin: 0;
        font-size: 0.9rem;
        font-weight: 500;
    }
    
    .logo-preview:hover .logo-placeholder {
        color: var(--primary-color);
    }
    
    .logo-preview img {
        width: 100%;
        height: 100%;
        object-fit: contain;
        border-radius: calc(var(--border-radius-lg) - 2px);
    }
    
    .preview-company-logo {
        width: 60px;
        height: 60px;
        border-radius: var(--border-radius);
        object-fit: contain;
        border: 2px solid var(--gray-200);
        margin: 0 auto 15px;
        display: block;
    }
    
    .preview-company-name-en {
        font-size: 1rem;
        color: var(--gray-500);
        font-style: italic;
        margin-bottom: 10px;
    }
    
    .remove-logo-btn {
        margin-top: 10px;
        border-radius: var(--border-radius);
        transition: all 0.3s ease;
    }
    
    .remove-logo-btn:hover {
        transform: translateY(-1px);
        box-shadow: 0 4px 12px rgba(220, 38, 38, 0.3);
    }
    
    .logo-error {
        background: #fef2f2;
        border: 1px solid #fecaca;
        color: #991b1b;
        padding: 10px 15px;
        border-radius: var(--border-radius);
        margin-top: 10px;
        font-size: 0.9rem;
        display: flex;
        align-items: center;
    }
    
    .logo-error i {
        margin-left: 8px;
        color: var(--danger-color);
    }
    
    .logo-preview.drag-over {
        border-color: var(--success-color);
        background: #d1fae5;
        transform: scale(1.02);
    }
    
    .logo-preview.drag-over .logo-placeholder {
        color: var(--success-color);
    }
    
    .logo-placeholder small {
        display: block;
        margin-top: 5px;
        color: var(--gray-400);
        font-size: 0.8rem;
    }
    
    /* Progress Bar Styles */
    .progress-bar-custom {
        height: 6px;
        background: var(--gray-200);
        border-radius: 3px;
        margin-bottom: 10px;
        overflow: hidden;
        box-shadow: inset 0 1px 2px rgba(0,0,0,0.1);
    }
    
    .progress-fill {
        height: 100%;
        background: linear-gradient(90deg, var(--primary-color), var(--primary-light));
        border-radius: 3px;
        transition: width 0.3s ease;
        box-shadow: 0 1px 2px rgba(37, 99, 235, 0.3);
    }
    
    /* VAT Numbers Styles */
    .vat-number-group {
        position: relative;
    }
    
    .vat-number-group .input-group {
        margin-bottom: 5px;
    }
    
    .add-vat-number {
        border-left: 0;
        border-top-left-radius: 0;
        border-bottom-left-radius: 0;
    }
    
    .remove-vat-number {
        border-left: 0;
        border-top-left-radius: 0;
        border-bottom-left-radius: 0;
    }
    
    .vat-number-input.is-valid {
        background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 20 20' fill='%23059669'%3e%3cpath fill-rule='evenodd' d='M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z' clip-rule='evenodd'/%3e%3c/svg%3e");
        padding-right: 2.5rem;
    }
    
    .vat-number-input.is-invalid {
        background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 20 20' fill='%23dc2626'%3e%3cpath fill-rule='evenodd' d='M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z' clip-rule='evenodd'/%3e%3c/svg%3e");
        padding-right: 2.5rem;
    }
</style>
@endpush

@section('content')
<!-- Header Section -->
<div class="edit-header">
    <div class="row align-items-center">
        <div class="col-md-8">
            <h1 class="mb-2" style="font-size: 1.875rem; font-weight: 700; letter-spacing: -0.025em;">
                <i class="fas fa-edit me-2" style="opacity: 0.9;"></i>
                تعديل {{ $company->name }}
            </h1>
            <p class="mb-0 opacity-90" style="font-size: 1rem; line-height: 1.5;">تحديث معلومات الشركة مع مراعاة متطلبات ZATCA</p>
            <div class="mt-3 d-flex flex-wrap gap-2">
                <span class="badge bg-white bg-opacity-20 text-white">
                    <i class="fas fa-clock me-1"></i>
                    آخر تحديث: {{ $company->updated_at->diffForHumans() }}
                </span>
                @if($company->is_active)
                    <span class="badge bg-white bg-opacity-20 text-white">
                        <i class="fas fa-check-circle me-1"></i>
                        شركة نشطة
                    </span>
                @else
                    <span class="badge bg-warning">
                        <i class="fas fa-pause-circle me-1"></i>
                        شركة معطلة
                    </span>
                @endif
            </div>
        </div>
        <div class="col-md-4 text-end">
            <div class="text-white">
                <div class="h2 mb-1" style="font-weight: 700; font-size: 2.25rem;">{{ $company->invoices->count() }}</div>
                <div class="opacity-90 mb-2" style="font-size: 0.875rem;">فاتورة مُصدرة</div>
                <div class="opacity-80" style="font-size: 0.75rem;">
                    إجمالي المبيعات: {{ number_format($company->invoices->sum('total_amount'), 2) }} ر.س
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Loading Overlay -->
<div class="loading-overlay" id="loadingOverlay">
    <div class="text-center">
        <div class="loading-spinner mb-3"></div>
        <h5 class="text-gray-900 mb-1" style="font-weight: 600;">جاري حفظ التعديلات...</h5>
        <p class="text-gray-500 mb-0" style="font-size: 0.875rem;">يرجى الانتظار</p>
    </div>
</div>

<!-- Changes Indicator -->
<div class="changes-indicator" id="changesIndicator">
    <i class="fas fa-exclamation-triangle me-2"></i>
    يوجد تعديلات غير محفوظة
</div>

<!-- Progress Indicator -->
<div class="row mb-4">
    <div class="col-12">
        <div class="progress-bar-custom">
            <div class="progress-fill" id="formProgress" style="width: 100%;"></div>
        </div>
        <div class="text-center">
            <small class="text-muted">تقدم التحديث: <span id="progressText">مكتمل</span></small>
        </div>
    </div>
</div>

<form method="POST" action="{{ route('companies.update', $company) }}" id="company-form" enctype="multipart/form-data">
    @csrf
    @method('PUT')
    
    <div class="row g-4">
        <div class="col-lg-8">
            <!-- المعلومات الأساسية -->
            <div class="form-section" id="basicInfo">
                <div class="section-header">
                    <h5 class="section-title">
                        <i class="fas fa-info-circle"></i>
                        المعلومات الأساسية
                    </h5>
                </div>
                <div class="card-body p-4">
                    <div class="form-group-enhanced">
                        <label class="form-label-enhanced">
                            <span class="required">*</span>
                            اسم الشركة
                            <span class="badge bg-primary ms-2" style="font-size: 0.7rem;">مطلوب</span>
                        </label>
                        <div class="input-group">
                            <span class="input-group-text bg-light border-0">
                                <i class="fas fa-building text-primary"></i>
                            </span>
                            <input type="text" name="name" class="form-control form-control-enhanced @error('name') is-invalid @enderror"
                                   value="{{ old('name', $company->name) }}" required placeholder="أدخل الاسم الكامل للشركة"
                                   data-original="{{ $company->name }}" style="border-right: none;">
                        </div>
                        <div class="help-text">
                            <i class="fas fa-info-circle"></i>
                            يجب أن يكون الاسم مطابقاً لاسم الشركة في السجل التجاري
                        </div>
                        @error('name')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    
                    <div class="form-group-enhanced">
                        <label class="form-label-enhanced">
                            اسم الشركة باللغة الإنجليزية
                        </label>
                        <div class="input-group">
                            <span class="input-group-text bg-light border-0">
                                <i class="fas fa-globe text-info"></i>
                            </span>
                            <input type="text" name="name_en" class="form-control form-control-enhanced @error('name_en') is-invalid @enderror"
                                   value="{{ old('name_en', $company->name_en ?? '') }}" placeholder="Enter company name in English"
                                   data-original="{{ $company->name_en ?? '' }}" style="border-right: none;">
                        </div>
                        <div class="help-text">
                            <i class="fas fa-info-circle"></i>
                            اسم الشركة باللغة الإنجليزية (اختياري)
                        </div>
                        @error('name_en')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    
                    <div class="form-group-enhanced">
                        <label class="form-label-enhanced">
                            شعار الشركة
                        </label>
                        <div class="logo-upload-container">
                            <div class="logo-preview @if($company->hasLogo()) has-image @endif" id="logoPreview">
                                @if($company->hasLogo())
                                    <img src="{{ $company->logo_url }}" alt="شعار {{ $company->name }}">
                                @else
                                    <div class="logo-placeholder">
                                        <i class="fas fa-cloud-upload-alt"></i>
                                        <p>اسحب الشعار هنا أو اضغط للاختيار</p>
                                        <small>PNG, JPG - أقل من 2 ميجابايت</small>
                                    </div>
                                @endif
                            </div>
                            <button type="button" class="btn btn-danger btn-sm remove-logo-btn" id="removeLogo"
                                    style="display: @if($company->hasLogo()) inline-block @else none @endif;">
                                <i class="fas fa-trash"></i>
                                إزالة الشعار
                            </button>
                            <input type="file" name="logo" class="form-control form-control-enhanced @error('logo') is-invalid @enderror"
                                   id="logoInput" accept="image/*" style="display: none;">
                            <div class="help-text">
                                <i class="fas fa-info-circle"></i>
                                يُفضل أن يكون الشعار بصيغة PNG أو JPG وبحجم أقصى 2 ميجابايت
                            </div>
                            <div class="logo-error" id="logoError" style="display: none;">
                                <i class="fas fa-exclamation-triangle"></i>
                                <span></span>
                            </div>
                            @error('logo')
                                <div class="invalid-feedback d-block">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    <span class="required">*</span>
                                    الأرقام الضريبية
                                    <span class="badge bg-warning ms-2" style="font-size: 0.7rem;">15 رقم</span>
                                </label>
                                
                                <div id="vatNumbersContainer">
                                    @php
                                        $vatNumbers = old('vat_numbers', $company->vat_numbers ?? []);
                                        if (empty($vatNumbers)) {
                                            $vatNumbers = [''];
                                        }
                                    @endphp
                                    
                                    @foreach($vatNumbers as $index => $vatNumber)
                                    <div class="vat-number-group mb-2" data-index="{{ $index }}">
                                        <div class="input-group">
                                            <span class="input-group-text bg-light border-0">
                                                <i class="fas fa-shield-alt text-warning"></i>
                                            </span>
                                            <input type="text" name="vat_numbers[{{ $index }}]" class="form-control form-control-enhanced vat-number-input @error('vat_numbers.' . $index) is-invalid @enderror"
                                                   value="{{ $vatNumber }}" required placeholder="300000000000003" maxlength="15"
                                                   data-original="{{ $vatNumber }}" style="border-right: none; border-left: none;">
                                            @if($index == 0)
                                                <button type="button" class="btn btn-outline-success add-vat-number" title="إضافة رقم ضريبي">
                                                    <i class="fas fa-plus"></i>
                                                </button>
                                            @else
                                                <button type="button" class="btn btn-outline-danger remove-vat-number" title="إزالة رقم ضريبي">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            @endif
                                            <span class="input-group-text bg-light border-0" id="vatValidationIcon{{ $index }}">
                                                <i class="fas fa-question-circle text-muted"></i>
                                            </span>
                                        </div>
                                        @error('vat_numbers.' . $index)
                                            <div class="invalid-feedback d-block">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    @endforeach
                                </div>
                                
                                @error('vat_numbers')
                                    <div class="invalid-feedback d-block">{{ $message }}</div>
                                @enderror
                                
                                <div class="help-text">
                                    <i class="fas fa-shield-alt"></i>
                                    كل رقم يجب أن يكون 15 رقم ويبدأ بـ 3 (مطلوب من هيئة الزكاة والضريبة)
                                </div>
                                
                                @if($company->zatcaSettings && $company->zatcaSettings->is_active)
                                    <div class="alert alert-warning mt-2">
                                        <i class="fas fa-exclamation-triangle me-2"></i>
                                        <small>تعديل الأرقام الضريبية يتطلب تحديث إعدادات ZATCA</small>
                                    </div>
                                @endif
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    <span class="required">*</span>
                                    رقم السجل التجاري
                                    <span class="badge bg-info ms-2" style="font-size: 0.7rem;">رسمي</span>
                                </label>
                                <div class="input-group">
                                    <span class="input-group-text bg-light border-0">
                                        <i class="fas fa-certificate text-info"></i>
                                    </span>
                                    <input type="text" name="cr_number" class="form-control form-control-enhanced @error('cr_number') is-invalid @enderror"
                                           value="{{ old('cr_number', $company->cr_number) }}" required placeholder="1010000000"
                                           data-original="{{ $company->cr_number }}" id="crNumberInput" style="border-right: none;">
                                </div>
                                <div class="help-text">
                                    <i class="fas fa-certificate"></i>
                                    رقم السجل التجاري الرسمي من وزارة التجارة
                                </div>
                                @error('cr_number')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- العنوان الوطني -->
            <div class="form-section" id="addressInfo">
                <div class="section-header">
                    <h5 class="section-title">
                        <i class="fas fa-map-marker-alt"></i>
                        العنوان الوطني السعودي
                    </h5>
                </div>
                <div class="p-4">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    <span class="required">*</span>
                                    المدينة
                                </label>
                                <input type="text" name="city" class="form-control form-control-enhanced @error('city') is-invalid @enderror"
                                       value="{{ old('city', $company->city) }}" required placeholder="الرياض"
                                       data-original="{{ $company->city }}">
                                <div class="help-text">
                                    <i class="fas fa-city"></i>
                                    اختر المدينة الرئيسية للمملكة العربية السعودية
                                </div>
                                @error('city')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    <span class="required">*</span>
                                    الحي
                                </label>
                                <input type="text" name="district" class="form-control form-control-enhanced @error('district') is-invalid @enderror"
                                       value="{{ old('district', $company->district) }}" required placeholder="الملز"
                                       data-original="{{ $company->district }}">
                                <div class="help-text">
                                    <i class="fas fa-map"></i>
                                    اسم الحي أو المنطقة داخل المدينة
                                </div>
                                @error('district')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="form-group-enhanced">
                        <label class="form-label-enhanced">
                            <span class="required">*</span>
                            اسم الشارع
                        </label>
                        <input type="text" name="street" class="form-control form-control-enhanced @error('street') is-invalid @enderror"
                               value="{{ old('street', $company->street) }}" required placeholder="شارع الملك فهد"
                               data-original="{{ $company->street }}">
                        <div class="help-text">
                            <i class="fas fa-road"></i>
                            الاسم الكامل للشارع أو الطريق
                        </div>
                        @error('street')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    
                    <div class="row">
                        <div class="col-md-3">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    <span class="required">*</span>
                                    رقم المبنى
                                </label>
                                <input type="text" name="building_number" class="form-control form-control-enhanced @error('building_number') is-invalid @enderror"
                                       value="{{ old('building_number', $company->building_number) }}" required placeholder="1234" maxlength="4"
                                       data-original="{{ $company->building_number }}">
                                <div class="help-text">
                                    <i class="fas fa-building"></i>
                                    4 أرقام كحد أقصى
                                </div>
                                @error('building_number')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="col-md-3">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    <span class="required">*</span>
                                    الرقم الإضافي
                                </label>
                                <input type="text" name="additional_number" class="form-control form-control-enhanced @error('additional_number') is-invalid @enderror"
                                       value="{{ old('additional_number', $company->additional_number) }}" required placeholder="5678" maxlength="4"
                                       data-original="{{ $company->additional_number }}">
                                <div class="help-text">
                                    <i class="fas fa-plus"></i>
                                    رقم إضافي من العنوان الوطني
                                </div>
                                @error('additional_number')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="col-md-3">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    <span class="required">*</span>
                                    الرمز البريدي
                                </label>
                                <input type="text" name="postal_code" class="form-control form-control-enhanced @error('postal_code') is-invalid @enderror"
                                       value="{{ old('postal_code', $company->postal_code) }}" required placeholder="12345" maxlength="5"
                                       data-original="{{ $company->postal_code }}">
                                <div class="help-text">
                                    <i class="fas fa-mailbox"></i>
                                    5 أرقام للرمز البريدي
                                </div>
                                @error('postal_code')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="col-md-3">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    رقم الوحدة
                                </label>
                                <input type="text" name="unit_number" class="form-control form-control-enhanced @error('unit_number') is-invalid @enderror"
                                       value="{{ old('unit_number', $company->unit_number) }}" placeholder="101"
                                       data-original="{{ $company->unit_number }}">
                                <div class="help-text">
                                    <i class="fas fa-door-open"></i>
                                    اختياري - رقم الشقة أو المكتب
                                </div>
                                @error('unit_number')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- معلومات الاتصال -->
            <div class="form-section" id="contactInfo">
                <div class="section-header">
                    <h5 class="section-title">
                        <i class="fas fa-phone"></i>
                        معلومات الاتصال
                    </h5>
                </div>
                <div class="p-4">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    البريد الإلكتروني
                                </label>
                                <input type="email" name="email" class="form-control form-control-enhanced @error('email') is-invalid @enderror"
                                       value="{{ old('email', $company->email) }}" placeholder="info@company.com"
                                       data-original="{{ $company->email }}">
                                <div class="help-text">
                                    <i class="fas fa-envelope"></i>
                                    البريد الإلكتروني الرسمي للشركة
                                </div>
                                @error('email')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    رقم الهاتف
                                </label>
                                <input type="tel" name="phone" class="form-control form-control-enhanced @error('phone') is-invalid @enderror"
                                       value="{{ old('phone', $company->phone) }}" placeholder="+966112345678"
                                       data-original="{{ $company->phone }}">
                                <div class="help-text">
                                    <i class="fas fa-phone"></i>
                                    رقم الهاتف مع رمز البلد (+966)
                                </div>
                                @error('phone')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="form-group-enhanced">
                        <label class="form-label-enhanced">
                            الموقع الإلكتروني
                        </label>
                        <input type="url" name="website" class="form-control form-control-enhanced @error('website') is-invalid @enderror"
                               value="{{ old('website', $company->website) }}" placeholder="https://www.company.com"
                               data-original="{{ $company->website }}">
                        <div class="help-text">
                            <i class="fas fa-globe"></i>
                            الموقع الإلكتروني الرسمي للشركة (اختياري)
                        </div>
                        @error('website')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                </div>
            </div>
        </div>
        
        <!-- معاينة الشركة والإحصائيات -->
        <div class="col-lg-4">
            <!-- معاينة الشركة -->
            <div class="preview-card">
                <div class="preview-header">
                    <h5 class="mb-0">
                        <i class="fas fa-eye me-2"></i>
                        معاينة الشركة
                    </h5>
                </div>
                
                <div class="company-preview">
                    <div class="preview-avatar" id="company-preview-avatar">
                        <i class="fas fa-building"></i>
                    </div>
                    <img class="preview-company-logo" id="company-preview-logo" style="display: @if($company->hasLogo()) block @else none @endif;"
                         src="{{ $company->hasLogo() ? $company->logo_url : '' }}" alt="شعار الشركة">
                    <div class="preview-name" id="company-preview-name">{{ $company->name }}</div>
                    <div class="preview-company-name-en" id="company-preview-name-en" style="display: {{ $company->name_en ? 'block' : 'none' }};">{{ $company->name_en ?? '' }}</div>
                    <div class="preview-vat" id="company-preview-vat">{{ $company->formatted_vat_numbers }}</div>
                    
                    <div class="preview-section">
                        <div class="preview-section-title">السجل التجاري</div>
                        <div class="preview-section-content" id="company-preview-cr">{{ $company->cr_number }}</div>
                    </div>
                    
                    <div class="preview-section">
                        <div class="preview-section-title">العنوان الكامل</div>
                        <div class="preview-section-content" id="company-preview-address">{{ $company->getFullAddressAttribute() }}</div>
                    </div>
                    
                    <div class="preview-section">
                        <div class="preview-section-title">معلومات الاتصال</div>
                        <div class="preview-section-content" id="company-preview-contact">
                            @if($company->email || $company->phone || $company->website)
                                @if($company->email)📧 {{ $company->email }}@endif
                                @if($company->email && ($company->phone || $company->website))<br>@endif
                                @if($company->phone)📞 {{ $company->phone }}@endif
                                @if($company->phone && $company->website)<br>@endif
                                @if($company->website)🌐 {{ $company->website }}@endif
                            @else
                                غير متوفرة
                            @endif
                        </div>
                    </div>
                    
                    <div class="form-check mb-3">
                        <input type="checkbox" name="is_active" class="form-check-input"
                               value="1" {{ old('is_active', $company->is_active) ? 'checked' : '' }} id="activeStatus">
                        <label class="form-check-label fw-bold" for="activeStatus">
                            <i class="fas fa-toggle-on text-success me-2"></i>
                            شركة نشطة
                        </label>
                    </div>
                    
                    <button type="submit" class="save-btn" id="saveBtn">
                        <i class="fas fa-save me-2"></i>
                        حفظ التعديلات
                    </button>
                </div>
            </div>
            
            <!-- إحصائيات الشركة -->
            <div class="stats-card">
                <div class="stats-number">{{ $company->invoices->count() }}</div>
                <div class="stats-label">
                    <i class="fas fa-file-invoice me-2"></i>
                    إجمالي الفواتير
                </div>
            </div>
            
            <div class="stats-card" style="background: linear-gradient(135deg, #27ae60 0%, #2ecc71 100%);">
                <div class="stats-number">{{ number_format($company->invoices->sum('total_amount'), 0) }}</div>
                <div class="stats-label">
                    <i class="fas fa-money-bill-wave me-2"></i>
                    إجمالي المبيعات (ر.س)
                </div>
            </div>
            
            <!-- حالة ZATCA -->
            @if($company->zatcaSettings)
            <div class="zatca-status-card">
                <div class="zatca-status-header">
                    <div class="zatca-status-icon">
                        <i class="fas fa-cog"></i>
                    </div>
                    <div>
                        <h6 class="mb-1">إعدادات ZATCA</h6>
                        <small class="opacity-75">مُعدة ونشطة</small>
                    </div>
                </div>
                
                <div class="mb-2">
                    @if($company->zatcaSettings->connection_type == 'clearance')
                        <span class="status-badge status-connected">Clearance</span>
                    @else
                        <span class="status-badge status-pending">Reporting</span>
                    @endif
                    
                    @if($company->zatcaSettings->environment == 'sandbox')
                        <span class="status-badge status-pending">تجريبية</span>
                    @else
                        <span class="status-badge status-connected">إنتاج</span>
                    @endif
                </div>
                
                <div class="mb-3">
                    @switch($company->zatcaSettings->connection_status)
                        @case('connected')
                            <span class="status-badge status-connected">متصل</span>
                            @break
                        @case('failed')
                            <span class="status-badge status-failed">فشل</span>
                            @break
                        @default
                            <span class="status-badge status-inactive">غير متصل</span>
                    @endswitch
                </div>
                
                <a href="{{ route('zatca-settings.show', $company->zatcaSettings) }}" class="btn btn-sm btn-outline-light w-100">
                    <i class="fas fa-cog me-1"></i>
                    إدارة إعدادات ZATCA
                </a>
            </div>
            @else
            <div class="warning-card">
                <div class="warning-header">
                    <h6 class="mb-0">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        تنبيه ZATCA
                    </h6>
                </div>
                <div class="card-body p-3">
                    <div class="alert alert-warning border-0 mb-3">
                        <small>
                            لم يتم إعداد ZATCA لهذه الشركة بعد.
                            يجب إعداد ZATCA لإصدار الفواتير الإلكترونية.
                        </small>
                    </div>
                    <a href="{{ route('zatca-settings.create', ['company_id' => $company->id]) }}" class="btn btn-warning w-100">
                        <i class="fas fa-plus me-1"></i>
                        إعداد ZATCA الآن
                    </a>
                </div>
            </div>
            @endif
            
            <!-- إحصائيات الشركة -->
            <div class="card mt-3">
                <div class="card-header">
                    <h6 class="card-title mb-0">
                        <i class="fas fa-chart-bar me-2"></i>
                        إحصائيات الشركة
                    </h6>
                </div>
                <div class="card-body">
                    <div class="row text-center">
                        <div class="col-6">
                            <div class="mb-2">
                                <h5 class="text-primary mb-0">{{ $company->invoices->count() }}</h5>
                                <small class="text-muted">عدد الفواتير</small>
                            </div>
                        </div>
                        <div class="col-6">
                            <div class="mb-2">
                                <h5 class="text-success mb-0">{{ number_format($company->invoices->sum('total_amount'), 2) }}</h5>
                                <small class="text-muted">إجمالي المبيعات</small>
                            </div>
                        </div>
                    </div>
                    
                    @if($company->invoices->count() > 0)
                        <hr>
                        <div class="text-center">
                            <small class="text-muted">
                                آخر فاتورة: {{ $company->invoices->last()->created_at->diffForHumans() }}
                            </small>
                        </div>
                    @endif
                </div>
            </div>
            
            <!-- حالة ZATCA -->
            @if($company->zatcaSettings)
            <div class="card mt-3">
                <div class="card-header">
                    <h6 class="card-title mb-0">
                        <i class="fas fa-cog me-2"></i>
                        حالة ZATCA
                    </h6>
                </div>
                <div class="card-body">
                    <div class="mb-2">
                        <strong>نوع الربط:</strong>
                        @if($company->zatcaSettings->connection_type == 'clearance')
                            <span class="badge bg-primary">Clearance</span>
                        @else
                            <span class="badge bg-info">Reporting</span>
                        @endif
                    </div>
                    
                    <div class="mb-2">
                        <strong>البيئة:</strong>
                        @if($company->zatcaSettings->environment == 'sandbox')
                            <span class="badge bg-warning">تجريبية</span>
                        @else
                            <span class="badge bg-success">إنتاج</span>
                        @endif
                    </div>
                    
                    <div class="mb-2">
                        <strong>حالة الاتصال:</strong>
                        @switch($company->zatcaSettings->connection_status)
                            @case('connected')
                                <span class="badge bg-success">متصل</span>
                                @break
                            @case('failed')
                                <span class="badge bg-danger">فشل</span>
                                @break
                            @default
                                <span class="badge bg-secondary">غير متصل</span>
                        @endswitch
                    </div>
                    
                    <div class="mt-2">
                        <a href="{{ route('zatca-settings.show', $company->zatcaSettings) }}" class="btn btn-sm btn-outline-primary w-100">
                            <i class="fas fa-cog me-1"></i>
                            إدارة إعدادات ZATCA
                        </a>
                    </div>
                </div>
            </div>
            @else
            <div class="card mt-3">
                <div class="card-header">
                    <h6 class="card-title mb-0 text-warning">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        تنبيه ZATCA
                    </h6>
                </div>
                <div class="card-body">
                    <div class="alert alert-warning">
                        <small>
                            لم يتم إعداد ZATCA لهذه الشركة بعد.
                            يجب إعداد ZATCA لإصدار الفواتير الإلكترونية.
                        </small>
                    </div>
                    <a href="{{ route('zatca-settings.create', ['company_id' => $company->id]) }}" class="btn btn-sm btn-warning w-100">
                        <i class="fas fa-plus me-1"></i>
                        إعداد ZATCA
                    </a>
                </div>
            </div>
            @endif
            
            <!-- تحذيرات -->
            @if($company->invoices->count() > 0)
            <div class="card mt-3">
                <div class="card-header">
                    <h6 class="card-title mb-0 text-warning">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        تنبيه
                    </h6>
                </div>
                <div class="card-body">
                    <div class="alert alert-warning">
                        <small>
                            هذه الشركة لديها {{ $company->invoices->count() }} فاتورة.
                            تعديل الرقم الضريبي أو السجل التجاري قد يؤثر على التقارير والامتثال.
                        </small>
                    </div>
                </div>
            </div>
            @endif
        </div>
    </div>
</form>
@endsection

@push('scripts')
<script>
// متغيرات عامة
let hasChanges = false;
const originalValues = {};

// تحديث المعاينة
function updatePreview() {
    // اسم الشركة
    const name = document.querySelector('input[name="name"]').value || 'اسم الشركة';
    document.getElementById('company-preview-name').textContent = name;
    
    // اسم الشركة بالإنجليزية
    const nameEn = document.querySelector('input[name="name_en"]').value;
    const previewNameEn = document.getElementById('company-preview-name-en');
    if (nameEn.trim()) {
        previewNameEn.textContent = nameEn;
        previewNameEn.style.display = 'block';
    } else {
        previewNameEn.style.display = 'none';
    }
    
    // الأرقام الضريبية
    const vatInputs = document.querySelectorAll('.vat-number-input');
    let vatNumbers = [];
    vatInputs.forEach(input => {
        if (input.value.trim() !== '') {
            vatNumbers.push(input.value.trim());
        }
    });
    
    const vatDisplay = vatNumbers.length > 0 ? vatNumbers.join(' - ') : 'غير محدد';
    document.getElementById('company-preview-vat').textContent = vatDisplay;
    
    // السجل التجاري
    const crNumber = document.querySelector('input[name="cr_number"]').value || 'غير محدد';
    document.getElementById('company-preview-cr').textContent = crNumber;
    
    // العنوان
    updateAddressPreview();
    
    // معلومات الاتصال
    updateContactPreview();
    
    // فحص التغييرات
    checkForChanges();
    
    // تحديث شريط التقدم
    updateProgressBar();
}

// تحديث شريط التقدم
function updateProgressBar() {
    const requiredFields = document.querySelectorAll('input[required]');
    const filledFields = Array.from(requiredFields).filter(field => field.value.trim() !== '');
    const progress = requiredFields.length > 0 ? (filledFields.length / requiredFields.length) * 100 : 100;
    
    document.getElementById('formProgress').style.width = progress + '%';
    
    if (progress === 100) {
        document.getElementById('progressText').textContent = 'مكتمل';
    } else {
        document.getElementById('progressText').textContent = Math.round(progress) + '%';
    }
}

function updateAddressPreview() {
    const city = document.querySelector('input[name="city"]').value;
    const district = document.querySelector('input[name="district"]').value;
    const street = document.querySelector('input[name="street"]').value;
    const building = document.querySelector('input[name="building_number"]').value;
    const postal = document.querySelector('input[name="postal_code"]').value;
    const additional = document.querySelector('input[name="additional_number"]').value;
    const unit = document.querySelector('input[name="unit_number"]').value;
    
    let address = '';
    if (street) address += street;
    if (building) address += (address ? '، مبنى ' : 'مبنى ') + building;
    if (additional) address += (address ? '، إضافي ' : 'إضافي ') + additional;
    if (unit) address += (address ? '، وحدة ' : 'وحدة ') + unit;
    if (district) address += (address ? '، ' : '') + district;
    if (city) address += (address ? '، ' : '') + city;
    if (postal) address += (address ? ' ' : '') + postal;
    
    document.getElementById('company-preview-address').textContent = address || 'لم يتم إدخال العنوان';
}

function updateContactPreview() {
    const email = document.querySelector('input[name="email"]').value;
    const phone = document.querySelector('input[name="phone"]').value;
    const website = document.querySelector('input[name="website"]').value;
    
    let contact = '';
    if (email) contact += '📧 ' + email;
    if (phone) contact += (contact ? '<br>📞 ' : '📞 ') + phone;
    if (website) contact += (contact ? '<br>🌐 ' : '🌐 ') + website;
    
    document.getElementById('company-preview-contact').innerHTML = contact || 'غير متوفرة';
}

// فحص التغييرات
function checkForChanges() {
    hasChanges = false;
    const inputs = document.querySelectorAll('input[data-original]');
    
    inputs.forEach(input => {
        const original = input.getAttribute('data-original') || '';
        const current = input.value || '';
        const section = input.closest('.form-section');
        
        if (original !== current) {
            hasChanges = true;
            input.classList.add('changed');
            if (section) section.classList.add('changed');
        } else {
            input.classList.remove('changed');
            if (section && !section.querySelector('.form-control-enhanced.changed')) {
                section.classList.remove('changed');
            }
        }
    });
    
    // تحديث مؤشر التغييرات
    const indicator = document.getElementById('changesIndicator');
    const saveBtn = document.getElementById('saveBtn');
    
    if (hasChanges) {
        indicator.classList.add('show');
        saveBtn.disabled = false;
        saveBtn.innerHTML = '<i class="fas fa-save me-2"></i>حفظ التعديلات';
    } else {
        indicator.classList.remove('show');
        saveBtn.disabled = true;
        saveBtn.innerHTML = '<i class="fas fa-check me-2"></i>لا توجد تغييرات';
    }
}

// التحقق من صحة الملف
function validateLogoFile(file) {
    const maxSize = 2 * 1024 * 1024; // 2MB
    const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
    
    if (!allowedTypes.includes(file.type)) {
        return 'نوع الملف غير مدعوم. يرجى اختيار صورة بصيغة PNG, JPG, أو GIF';
    }
    
    if (file.size > maxSize) {
        return 'حجم الملف كبير جداً. يجب أن يكون أقل من 2 ميجابايت';
    }
    
    return null;
}

// عرض رسالة خطأ
function showLogoError(message) {
    const errorDiv = document.getElementById('logoError');
    const errorSpan = errorDiv.querySelector('span');
    
    errorSpan.textContent = message;
    errorDiv.style.display = 'flex';
    
    // إخفاء الرسالة بعد 5 ثوان
    setTimeout(() => {
        errorDiv.style.display = 'none';
    }, 5000);
}

// إخفاء رسالة الخطأ
function hideLogoError() {
    const errorDiv = document.getElementById('logoError');
    errorDiv.style.display = 'none';
}

// تحديث معاينة الشعار
function updateLogoPreview(file) {
    const logoPreview = document.getElementById('logoPreview');
    const companyPreviewLogo = document.getElementById('company-preview-logo');
    const companyPreviewAvatar = document.getElementById('company-preview-avatar');
    const removeBtn = document.getElementById('removeLogo');
    
    if (file && file.type.startsWith('image/')) {
        // التحقق من صحة الملف
        const validationError = validateLogoFile(file);
        if (validationError) {
            showLogoError(validationError);
            return;
        }
        
        hideLogoError();
        
        const reader = new FileReader();
        reader.onload = function(e) {
            logoPreview.innerHTML = `<img src="${e.target.result}" alt="شعار الشركة">`;
            logoPreview.classList.add('has-image');
            
            // تحديث معاينة الشعار في البطاقة
            companyPreviewLogo.src = e.target.result;
            companyPreviewLogo.style.display = 'block';
            companyPreviewAvatar.style.display = 'none';
            
            // إظهار زر الإزالة
            removeBtn.style.display = 'inline-block';
        };
        reader.readAsDataURL(file);
    } else {
        clearLogoPreview();
    }
}

// مسح معاينة الشعار
function clearLogoPreview() {
    const logoPreview = document.getElementById('logoPreview');
    const companyPreviewLogo = document.getElementById('company-preview-logo');
    const companyPreviewAvatar = document.getElementById('company-preview-avatar');
    const removeBtn = document.getElementById('removeLogo');
    const logoInput = document.getElementById('logoInput');
    
    // إعادة تعيين المعاينة
    logoPreview.innerHTML = `
        <div class="logo-placeholder">
            <i class="fas fa-cloud-upload-alt"></i>
            <p>اسحب الشعار هنا أو اضغط للاختيار</p>
            <small>PNG, JPG - أقل من 2 ميجابايت</small>
        </div>
    `;
    logoPreview.classList.remove('has-image');
    companyPreviewLogo.style.display = 'none';
    companyPreviewAvatar.style.display = 'flex';
    removeBtn.style.display = 'none';
    logoInput.value = '';
    hideLogoError();
}

// التحقق من صحة الأرقام الضريبية
function validateVatNumbers() {
    const vatInputs = document.querySelectorAll('.vat-number-input');
    let allValid = true;
    let hasValues = false;
    
    vatInputs.forEach((input, index) => {
        const vatNumber = input.value;
        const validationIcon = document.getElementById('vatValidationIcon' + index);
        
        if (vatNumber.length > 0) {
            hasValues = true;
            if (vatNumber.length === 15 && vatNumber.startsWith('3')) {
                input.classList.remove('is-invalid');
                input.classList.add('is-valid');
                if (validationIcon) {
                    validationIcon.innerHTML = '<i class="fas fa-check-circle text-success"></i>';
                    validationIcon.title = 'رقم ضريبي صحيح';
                }
            } else {
                input.classList.remove('is-valid');
                input.classList.add('is-invalid');
                if (validationIcon) {
                    validationIcon.innerHTML = '<i class="fas fa-exclamation-circle text-danger"></i>';
                    validationIcon.title = 'رقم ضريبي غير صحيح - يجب أن يكون 15 رقم ويبدأ بـ 3';
                }
                allValid = false;
            }
        } else {
            input.classList.remove('is-valid', 'is-invalid');
            if (validationIcon) {
                validationIcon.innerHTML = '<i class="fas fa-question-circle text-muted"></i>';
                validationIcon.title = 'أدخل الرقم الضريبي';
            }
            if (input.hasAttribute('required')) {
                allValid = false;
            }
        }
    });
    
    return allValid && hasValues;
}

// إضافة حقل رقم ضريبي جديد
function addVatNumberField() {
    const container = document.getElementById('vatNumbersContainer');
    const groups = container.querySelectorAll('.vat-number-group');
    const newIndex = groups.length;
    
    const newGroup = document.createElement('div');
    newGroup.className = 'vat-number-group mb-2';
    newGroup.setAttribute('data-index', newIndex);
    
    newGroup.innerHTML = `
        <div class="input-group">
            <span class="input-group-text bg-light border-0">
                <i class="fas fa-shield-alt text-warning"></i>
            </span>
            <input type="text" name="vat_numbers[${newIndex}]" class="form-control form-control-enhanced vat-number-input"
                   placeholder="300000000000003" maxlength="15" required data-original=""
                   style="border-right: none; border-left: none;">
            <button type="button" class="btn btn-outline-danger remove-vat-number" title="إزالة رقم ضريبي">
                <i class="fas fa-trash"></i>
            </button>
            <span class="input-group-text bg-light border-0" id="vatValidationIcon${newIndex}">
                <i class="fas fa-question-circle text-muted"></i>
            </span>
        </div>
    `;
    
    container.appendChild(newGroup);
    
    // ربط الأحداث للحقل الجديد
    const newInput = newGroup.querySelector('.vat-number-input');
    newInput.addEventListener('input', function() {
        validateVatNumbers();
        updatePreview();
        autoSave();
    });
    
    const removeBtn = newGroup.querySelector('.remove-vat-number');
    removeBtn.addEventListener('click', function() {
        removeVatNumberField(newGroup);
    });
    
    // التركيز على الحقل الجديد
    newInput.focus();
    
    validateVatNumbers();
    updatePreview();
}

// إزالة حقل رقم ضريبي
function removeVatNumberField(group) {
    const container = document.getElementById('vatNumbersContainer');
    const groups = container.querySelectorAll('.vat-number-group');
    
    // لا يمكن إزالة الحقل الأول إذا كان الوحيد
    if (groups.length > 1) {
        group.remove();
        validateVatNumbers();
        updatePreview();
        autoSave();
    }
}

// استعادة القيم الأصلية
function resetForm() {
    if (confirm('هل تريد استعادة جميع القيم الأصلية؟ سيتم فقدان جميع التعديلات غير المحفوظة.')) {
        const inputs = document.querySelectorAll('input[data-original]');
        
        inputs.forEach(input => {
            const original = input.getAttribute('data-original') || '';
            input.value = original;
            input.classList.remove('changed', 'is-valid', 'is-invalid');
        });
        
        // استعادة حالة checkbox
        const activeCheckbox = document.querySelector('input[name="is_active"]');
        const originalActive = {{ $company->is_active ? 'true' : 'false' }};
        activeCheckbox.checked = originalActive;
        
        // استعادة الشعار الأصلي
        const logoInput = document.getElementById('logoInput');
        if (logoInput) {
            logoInput.value = '';
        }
        
        // استعادة معاينة الشعار الأصلية
        @if($company->hasLogo())
            const logoPreview = document.getElementById('logoPreview');
            const companyPreviewLogo = document.getElementById('company-preview-logo');
            const companyPreviewAvatar = document.getElementById('company-preview-avatar');
            const removeBtn = document.getElementById('removeLogo');
            
            logoPreview.innerHTML = `<img src="{{ $company->logo_url }}" alt="شعار الشركة">`;
            logoPreview.classList.add('has-image');
            companyPreviewLogo.src = "{{ $company->logo_url }}";
            companyPreviewLogo.style.display = 'block';
            companyPreviewAvatar.style.display = 'none';
            removeBtn.style.display = 'inline-block';
        @else
            clearLogoPreview();
        @endif
        
        // إزالة فئات التغيير من الأقسام
        document.querySelectorAll('.form-section').forEach(section => {
            section.classList.remove('changed');
        });
        
        // تحديث المعاينة
        updatePreview();
        validateVatNumbers();
    }
}

// حفظ تلقائي للنموذج
function autoSave() {
    const formData = new FormData(document.getElementById('company-form'));
    const data = {};
    
    for (let [key, value] of formData.entries()) {
        // تجاهل ملفات الشعار في الحفظ التلقائي
        if (key !== 'logo') {
            data[key] = value;
        }
    }
    
    localStorage.setItem('companyEditData_{{ $company->id }}', JSON.stringify(data));
}

// استرداد البيانات المحفوظة
function loadAutoSavedData() {
    const savedData = localStorage.getItem('companyEditData_{{ $company->id }}');
    
    if (savedData && confirm('تم العثور على بيانات محفوظة مؤقتاً. هل تريد استردادها؟')) {
        const data = JSON.parse(savedData);
        
        Object.keys(data).forEach(key => {
            const input = document.querySelector(`input[name="${key}"]`);
            if (input && input.type !== 'checkbox') {
                input.value = data[key];
            } else if (input && input.type === 'checkbox') {
                input.checked = data[key] === '1';
            }
        });
        
        updatePreview();
        validateVatNumber();
    }
}

// تهيئة الصفحة
document.addEventListener('DOMContentLoaded', function() {
    // حفظ القيم الأصلية
    document.querySelectorAll('input[data-original]').forEach(input => {
        originalValues[input.name] = input.getAttribute('data-original') || '';
    });
    
    // ربط أحداث التحديث
    document.querySelectorAll('input').forEach(input => {
        input.addEventListener('input', function() {
            updatePreview();
            autoSave();
        });
        
        // تأثيرات التركيز
        input.addEventListener('focus', function() {
            const section = this.closest('.form-section');
            if (section) {
                section.style.transform = 'translateY(-2px)';
                section.style.boxShadow = '0 15px 40px rgba(0,0,0,0.2)';
            }
        });
        
        input.addEventListener('blur', function() {
            const section = this.closest('.form-section');
            if (section) {
                section.style.transform = 'translateY(0)';
                section.style.boxShadow = '';
            }
        });
    });
    
    // التحقق من الأرقام الضريبية
    document.querySelectorAll('.vat-number-input').forEach(input => {
        input.addEventListener('input', function() {
            validateVatNumbers();
            updatePreview();
            autoSave();
        });
    });
    
    // ربط أحداث أزرار إضافة/إزالة الأرقام الضريبية
    document.querySelectorAll('.add-vat-number').forEach(btn => {
        btn.addEventListener('click', addVatNumberField);
    });
    
    document.querySelectorAll('.remove-vat-number').forEach(btn => {
        btn.addEventListener('click', function() {
            removeVatNumberField(this.closest('.vat-number-group'));
        });
    });
    
    // ربط حدث تحميل الشعار
    const logoInput = document.getElementById('logoInput');
    const logoPreview = document.getElementById('logoPreview');
    const removeLogo = document.getElementById('removeLogo');
    
    if (logoInput && logoPreview) {
        logoInput.addEventListener('change', function(e) {
            const file = e.target.files[0];
            updateLogoPreview(file);
        });
        
        // جعل منطقة المعاينة قابلة للنقر
        logoPreview.addEventListener('click', function() {
            if (!logoPreview.classList.contains('has-image')) {
                logoInput.click();
            }
        });
        
        // وظائف السحب والإفلات
        logoPreview.addEventListener('dragover', function(e) {
            e.preventDefault();
            logoPreview.classList.add('drag-over');
        });
        
        logoPreview.addEventListener('dragleave', function(e) {
            e.preventDefault();
            logoPreview.classList.remove('drag-over');
        });
        
        logoPreview.addEventListener('drop', function(e) {
            e.preventDefault();
            logoPreview.classList.remove('drag-over');
            
            const files = e.dataTransfer.files;
            if (files.length > 0) {
                const file = files[0];
                if (file.type.startsWith('image/')) {
                    logoInput.files = files;
                    updateLogoPreview(file);
                } else {
                    showLogoError('يرجى سحب ملف صورة فقط');
                }
            }
        });
    }
    
    // ربط زر إزالة الشعار
    if (removeLogo) {
        removeLogo.addEventListener('click', function() {
            clearLogoPreview();
        });
    }
    
    // منع السلوك الافتراضي للسحب والإفلات في النافذة
    ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
        document.addEventListener(eventName, function(e) {
            e.preventDefault();
        });
    });
    
    // تحديث المعاينة الأولية
    updatePreview();
    validateVatNumbers();
    
    // تحميل البيانات المحفوظة تلقائياً
    loadAutoSavedData();
    
    // منع إرسال النموذج إذا كانت البيانات غير صحيحة
    document.getElementById('company-form').addEventListener('submit', function(e) {
        const vatInputs = document.querySelectorAll('.vat-number-input');
        const logoInput = document.getElementById('logoInput');
        const nameEn = document.querySelector('input[name="name_en"]').value.trim();
        let hasValidVat = false;
        
        // التحقق من الأرقام الضريبية
        for (let input of vatInputs) {
            const vatNumber = input.value.trim();
            if (vatNumber.length > 0) {
                if (vatNumber.length !== 15 || !vatNumber.startsWith('3')) {
                    e.preventDefault();
                    alert('جميع الأرقام الضريبية يجب أن تكون 15 رقم وتبدأ بـ 3');
                    input.focus();
                    return false;
                }
                hasValidVat = true;
            }
        }
        
        if (!hasValidVat) {
            e.preventDefault();
            alert('يجب إدخال رقم ضريبي واحد على الأقل');
            document.querySelector('.vat-number-input').focus();
            return false;
        }
        
        // التحقق من الشعار إذا كان محدداً
        if (logoInput && logoInput.files.length > 0) {
            const file = logoInput.files[0];
            const validationError = validateLogoFile(file);
            if (validationError) {
                e.preventDefault();
                alert('خطأ في الشعار: ' + validationError);
                return false;
            }
        }
        
        // التحقق من اسم الشركة بالإنجليزية (إذا كان مدخلاً)
        if (nameEn && !/^[a-zA-Z0-9\s\-&.,()]+$/.test(nameEn)) {
            e.preventDefault();
            alert('اسم الشركة بالإنجليزية يجب أن يحتوي على أحرف إنجليزية فقط');
            document.querySelector('input[name="name_en"]').focus();
            return false;
        }
        
        if (!hasChanges) {
            e.preventDefault();
            alert('لا توجد تغييرات للحفظ');
            return false;
        }
        
        // إظهار مؤشر التحميل
        const submitBtn = document.getElementById('saveBtn');
        const loadingOverlay = document.getElementById('loadingOverlay');
        
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>جاري الحفظ...';
        submitBtn.disabled = true;
        loadingOverlay.classList.add('show');
        
        // مسح البيانات المحفوظة مؤقتاً
        localStorage.removeItem('companyEditData_{{ $company->id }}');
        
        // إضافة تأخير لإظهار التأثير
        setTimeout(() => {
            // السماح للنموذج بالإرسال
        }, 500);
    });
    
    // تحذير عند مغادرة الصفحة مع وجود تغييرات
    window.addEventListener('beforeunload', function(e) {
        if (hasChanges) {
            e.preventDefault();
            e.returnValue = 'يوجد تعديلات غير محفوظة. هل تريد المغادرة؟';
            return e.returnValue;
        }
    });
});

// دالة التحقق من اتصال ZATCA (إذا كانت متوفرة)
function checkZatcaConnection() {
    @if($company->zatcaSettings)
    // التحقق من وجود الراوت قبل الاستدعاء
    try {
        // استخدام راوت بديل آمن أو تجاهل الفحص إذا لم يكن متوفر
        console.log('إعدادات ZATCA موجودة للشركة');
        // يمكن إضافة فحص الاتصال هنا لاحقاً عند توفر الراوت المناسب
    } catch (error) {
        console.log('فحص اتصال ZATCA غير متاح حالياً');
    }
    @else
    console.log('لا توجد إعدادات ZATCA للشركة');
    @endif
}

// فحص اتصال ZATCA عند تحميل الصفحة (اختياري)
setTimeout(checkZatcaConnection, 3000);
</script>
@endpush