@extends('layouts.app')

@section('title', 'إضافة شركة جديدة')
@section('page-title', 'إضافة شركة جديدة')

@section('page-actions')
<div class="d-flex gap-2">
    <a href="{{ route('companies.index') }}" class="btn btn-secondary">
        <i class="fas fa-arrow-left me-2"></i>
        العودة للقائمة
    </a>
    <button type="button" class="btn btn-info" onclick="clearForm()">
        <i class="fas fa-eraser me-2"></i>
        مسح النموذج
    </button>
</div>
@endsection

@push('styles')
<style>
    .create-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        border-radius: 15px;
        padding: 30px;
        margin-bottom: 30px;
        position: relative;
        overflow: hidden;
    }
    
    .create-header::before {
        content: '';
        position: absolute;
        top: -50%;
        right: -20%;
        width: 300px;
        height: 300px;
        background: rgba(255,255,255,0.1);
        border-radius: 50%;
        transform: rotate(45deg);
    }
    
    .form-section {
        background: white;
        border-radius: 15px;
        box-shadow: 0 8px 25px rgba(0,0,0,0.1);
        margin-bottom: 25px;
        overflow: hidden;
        transition: all 0.3s ease;
    }
    
    .form-section:hover {
        box-shadow: 0 12px 35px rgba(0,0,0,0.15);
    }
    
    .section-header {
        background: linear-gradient(135deg, #3498db, #2980b9);
        color: white;
        padding: 20px;
        border: none;
        position: relative;
    }
    
    .section-header::after {
        content: '';
        position: absolute;
        bottom: 0;
        left: 0;
        right: 0;
        height: 3px;
        background: linear-gradient(90deg, rgba(255,255,255,0.3), transparent);
    }
    
    .section-title {
        font-size: 1.1rem;
        font-weight: 600;
        margin: 0;
        display: flex;
        align-items: center;
    }
    
    .section-title i {
        margin-left: 10px;
        font-size: 1.2rem;
    }
    
    .form-group-enhanced {
        margin-bottom: 25px;
        position: relative;
    }
    
    .form-label-enhanced {
        font-weight: 600;
        color: #2c3e50;
        margin-bottom: 8px;
        display: flex;
        align-items: center;
        font-size: 0.95rem;
    }
    
    .form-label-enhanced .required {
        color: #e74c3c;
        margin-right: 5px;
        font-size: 1.1rem;
    }
    
    .form-control-enhanced {
        border: 2px solid #e9ecef;
        border-radius: 10px;
        padding: 12px 16px;
        font-size: 1rem;
        transition: all 0.3s ease;
        background: #f8f9fa;
    }
    
    .form-control-enhanced:focus {
        border-color: #3498db;
        box-shadow: 0 0 0 0.2rem rgba(52, 152, 219, 0.25);
        background: white;
        transform: translateY(-1px);
    }
    
    .form-control-enhanced.is-valid {
        border-color: #27ae60;
        background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 8 8'%3e%3cpath fill='%2327ae60' d='m2.3 6.73.94-.94 2.94 2.94-2.94-2.94L5.66 3.84l-.94-.94Z'/%3e%3c/svg%3e");
        padding-left: 40px;
    }
    
    .form-control-enhanced.is-invalid {
        border-color: #e74c3c;
        background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 8 8'%3e%3cpath fill='%23e74c3c' d='M2.646 1.646a.5.5 0 0 1 .708 0L4 2.293l.646-.647a.5.5 0 0 1 .708.708L4.707 3l.647.646a.5.5 0 0 1-.708.708L4 3.707l-.646.647a.5.5 0 0 1-.708-.708L3.293 3l-.647-.646a.5.5 0 0 1 0-.708z'/%3e%3c/svg%3e");
        padding-left: 40px;
    }
    
    .help-text {
        font-size: 0.85rem;
        color: #6c757d;
        margin-top: 5px;
        display: flex;
        align-items: center;
    }
    
    .help-text i {
        margin-left: 5px;
        color: #3498db;
    }
    
    .preview-card {
        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
        border: none;
        border-radius: 15px;
        overflow: hidden;
        position: sticky;
        top: 20px;
    }
    
    .preview-header {
        background: linear-gradient(135deg, #27ae60, #2ecc71);
        color: white;
        padding: 20px;
    }
    
    .company-preview {
        background: white;
        border-radius: 12px;
        padding: 20px;
        margin: 20px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        text-align: center;
        transition: all 0.3s ease;
    }
    
    .preview-avatar {
        width: 80px;
        height: 80px;
        background: linear-gradient(135deg, #3498db, #2980b9);
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        margin: 0 auto 15px;
        color: white;
        font-size: 2rem;
        box-shadow: 0 5px 15px rgba(52, 152, 219, 0.3);
    }
    
    .preview-name {
        font-size: 1.2rem;
        font-weight: 700;
        color: #2c3e50;
        margin-bottom: 5px;
    }
    
    .preview-vat {
        font-size: 0.9rem;
        color: #7f8c8d;
        font-family: monospace;
        background: #ecf0f1;
        padding: 5px 10px;
        border-radius: 20px;
        display: inline-block;
        margin-bottom: 15px;
    }
    
    .preview-section {
        text-align: right;
        margin-bottom: 15px;
        padding: 15px;
        background: #f8f9fa;
        border-radius: 10px;
    }
    
    .preview-section-title {
        font-weight: 600;
        color: #2c3e50;
        margin-bottom: 8px;
        font-size: 0.9rem;
    }
    
    .preview-section-content {
        color: #7f8c8d;
        font-size: 0.85rem;
        line-height: 1.4;
    }
    
    .status-toggle {
        background: linear-gradient(135deg, #27ae60, #2ecc71);
        border: none;
        border-radius: 25px;
        padding: 10px 20px;
        color: white;
        font-weight: 600;
        transition: all 0.3s ease;
        display: flex;
        align-items: center;
        justify-content: center;
        width: 100%;
        margin-bottom: 20px;
    }
    
    .status-toggle:hover {
        transform: translateY(-2px);
        box-shadow: 0 8px 20px rgba(39, 174, 96, 0.3);
    }
    
    .save-btn {
        background: linear-gradient(135deg, #27ae60, #2ecc71);
        border: none;
        border-radius: 12px;
        padding: 15px 30px;
        color: white;
        font-weight: 600;
        font-size: 1.1rem;
        transition: all 0.3s ease;
        width: 100%;
        box-shadow: 0 5px 15px rgba(39, 174, 96, 0.3);
    }
    
    .save-btn:hover {
        transform: translateY(-3px);
        box-shadow: 0 10px 25px rgba(39, 174, 96, 0.4);
        background: linear-gradient(135deg, #2ecc71, #27ae60);
    }
    
    .guidelines-card {
        background: linear-gradient(135deg, #fff3cd 0%, #ffeaa7 100%);
        border: none;
        border-radius: 15px;
        margin-top: 20px;
    }
    
    .guidelines-header {
        background: linear-gradient(135deg, #f39c12, #e67e22);
        color: white;
        padding: 15px 20px;
        border-radius: 15px 15px 0 0;
    }
    
    .step-indicator {
        display: flex;
        justify-content: space-between;
        margin-bottom: 30px;
        padding: 0 20px;
    }
    
    .step {
        flex: 1;
        text-align: center;
        position: relative;
    }
    
    .step::after {
        content: '';
        position: absolute;
        top: 15px;
        left: 60%;
        right: -40%;
        height: 2px;
        background: #e9ecef;
        z-index: 1;
    }
    
    .step:last-child::after {
        display: none;
    }
    
    .step.active::after {
        background: #3498db;
    }
    
    .step-circle {
        width: 30px;
        height: 30px;
        border-radius: 50%;
        background: #e9ecef;
        color: #6c757d;
        display: flex;
        align-items: center;
        justify-content: center;
        margin: 0 auto 8px;
        font-weight: 600;
        font-size: 0.9rem;
        position: relative;
        z-index: 2;
        transition: all 0.3s ease;
    }
    
    .step.active .step-circle {
        background: #3498db;
        color: white;
    }
    
    .step.completed .step-circle {
        background: #27ae60;
        color: white;
    }
    
    .step-title {
        font-size: 0.8rem;
        color: #6c757d;
        font-weight: 500;
    }
    
    .step.active .step-title {
        color: #3498db;
        font-weight: 600;
    }
    
    .validation-message {
        background: linear-gradient(135deg, #d4edda, #c3e6cb);
        border: 1px solid #27ae60;
        border-radius: 10px;
        padding: 15px;
        margin-top: 10px;
        display: none;
    }
    
    .validation-message.show {
        display: block;
        animation: slideDown 0.3s ease;
    }
    
    @keyframes slideDown {
        from {
            opacity: 0;
            transform: translateY(-10px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }
    
    .progress-bar-custom {
        height: 6px;
        background: #e9ecef;
        border-radius: 3px;
        margin-bottom: 20px;
        overflow: hidden;
    }
    
    .progress-fill {
        height: 100%;
        background: linear-gradient(90deg, #3498db, #2980b9);
        border-radius: 3px;
        transition: width 0.3s ease;
        width: 0%;
    }
    
    .vat-number-group {
        position: relative;
    }
    
    .vat-number-group .input-group {
        margin-bottom: 5px;
    }
    
    .add-vat-number {
        border-left: 0;
        border-top-left-radius: 0;
        border-bottom-left-radius: 0;
    }
    
    .remove-vat-number {
        border-left: 0;
        border-top-left-radius: 0;
        border-bottom-left-radius: 0;
    }
    
    .vat-number-input.is-valid {
        background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 8 8'%3e%3cpath fill='%2327ae60' d='m2.3 6.73.94-.94 2.94 2.94-2.94-2.94L5.66 3.84l-.94-.94Z'/%3e%3c/svg%3e");
        padding-left: 35px;
    }
    
    .vat-number-input.is-invalid {
        background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 8 8'%3e%3cpath fill='%23e74c3c' d='M2.646 1.646a.5.5 0 0 1 .708 0L4 2.293l.646-.647a.5.5 0 0 1 .708.708L4.707 3l.647.646a.5.5 0 0 1-.708.708L4 3.707l-.646.647a.5.5 0 0 1-.708-.708L3.293 3l-.647-.646a.5.5 0 0 1 0-.708z'/%3e%3c/svg%3e");
        padding-left: 35px;
    }
    
    .logo-upload-container {
        position: relative;
    }
    
    .logo-preview {
        width: 150px;
        height: 150px;
        border: 2px dashed #e9ecef;
        border-radius: 15px;
        display: flex;
        align-items: center;
        justify-content: center;
        margin-bottom: 15px;
        transition: all 0.3s ease;
        cursor: pointer;
        background: #f8f9fa;
        overflow: hidden;
    }
    
    .logo-preview:hover {
        border-color: #3498db;
        background: #f0f8ff;
        transform: translateY(-2px);
        box-shadow: 0 8px 25px rgba(52, 152, 219, 0.15);
    }
    
    .logo-preview.has-image {
        border: 2px solid #27ae60;
        background: white;
    }
    
    .logo-placeholder {
        text-align: center;
        color: #6c757d;
        transition: all 0.3s ease;
    }
    
    .logo-placeholder i {
        font-size: 3rem;
        margin-bottom: 10px;
        display: block;
        color: #3498db;
    }
    
    .logo-placeholder p {
        margin: 0;
        font-size: 0.9rem;
        font-weight: 500;
    }
    
    .logo-preview:hover .logo-placeholder {
        color: #3498db;
    }
    
    .logo-preview img {
        width: 100%;
        height: 100%;
        object-fit: contain;
        border-radius: 12px;
    }
    
    .preview-company-logo {
        width: 60px;
        height: 60px;
        border-radius: 12px;
        object-fit: contain;
        border: 2px solid #e9ecef;
        margin: 0 auto 15px;
        display: block;
    }
    
    .preview-company-name-en {
        font-size: 1rem;
        color: #7f8c8d;
        font-style: italic;
        margin-bottom: 10px;
    }
    
    .remove-logo-btn {
        margin-top: 10px;
        border-radius: 8px;
        transition: all 0.3s ease;
    }
    
    .remove-logo-btn:hover {
        transform: translateY(-1px);
        box-shadow: 0 4px 12px rgba(231, 76, 60, 0.3);
    }
    
    .logo-error {
        background: #f8d7da;
        border: 1px solid #f5c6cb;
        color: #721c24;
        padding: 10px 15px;
        border-radius: 8px;
        margin-top: 10px;
        font-size: 0.9rem;
        display: flex;
        align-items: center;
    }
    
    .logo-error i {
        margin-left: 8px;
        color: #e74c3c;
    }
    
    .logo-preview.drag-over {
        border-color: #27ae60;
        background: #d5f4e6;
        transform: scale(1.02);
    }
    
    .logo-preview.drag-over .logo-placeholder {
        color: #27ae60;
    }
    
    .logo-placeholder small {
        display: block;
        margin-top: 5px;
        color: #95a5a6;
        font-size: 0.8rem;
    }
</style>
@endpush

@section('content')
<!-- Header Section -->
<div class="create-header">
    <div class="row align-items-center">
        <div class="col-md-8">
            <h2 class="mb-2">
                <i class="fas fa-plus-circle me-3"></i>
                إنشاء شركة جديدة
            </h2>
            <p class="mb-0 opacity-75">أدخل معلومات الشركة الأساسية وتأكد من صحة البيانات للامتثال لمتطلبات ZATCA</p>
        </div>
        <div class="col-md-4 text-end">
            <div class="progress-bar-custom">
                <div class="progress-fill" id="formProgress"></div>
            </div>
            <small class="opacity-75">تقدم النموذج: <span id="progressText">0%</span></small>
        </div>
    </div>
</div>

<!-- Step Indicator -->
<div class="step-indicator">
    <div class="step active" id="step1">
        <div class="step-circle">1</div>
        <div class="step-title">المعلومات الأساسية</div>
    </div>
    <div class="step" id="step2">
        <div class="step-circle">2</div>
        <div class="step-title">العنوان الوطني</div>
    </div>
    <div class="step" id="step3">
        <div class="step-circle">3</div>
        <div class="step-title">معلومات الاتصال</div>
    </div>
    <div class="step" id="step4">
        <div class="step-circle">4</div>
        <div class="step-title">مراجعة وحفظ</div>
    </div>
</div>

<form method="POST" action="{{ route('companies.store') }}" id="company-form" enctype="multipart/form-data">
    @csrf
    
    <div class="row">
        <div class="col-lg-8">
            <!-- المعلومات الأساسية -->
            <div class="form-section" id="basicInfo">
                <div class="section-header">
                    <h5 class="section-title">
                        <i class="fas fa-info-circle"></i>
                        المعلومات الأساسية
                    </h5>
                </div>
                <div class="card-body p-4">
                    <div class="form-group-enhanced">
                        <label class="form-label-enhanced">
                            <span class="required">*</span>
                            اسم الشركة
                        </label>
                        <input type="text" name="name" class="form-control form-control-enhanced @error('name') is-invalid @enderror"
                               value="{{ old('name') }}" required placeholder="أدخل الاسم الكامل للشركة">
                        <div class="help-text">
                            <i class="fas fa-info-circle"></i>
                            يجب أن يكون الاسم مطابقاً لاسم الشركة في السجل التجاري
                        </div>
                        @error('name')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    
                    <div class="form-group-enhanced">
                        <label class="form-label-enhanced">
                            اسم الشركة باللغة الإنجليزية
                        </label>
                        <input type="text" name="name_en" class="form-control form-control-enhanced @error('name_en') is-invalid @enderror"
                               value="{{ old('name_en') }}" placeholder="Enter company name in English">
                        <div class="help-text">
                            <i class="fas fa-info-circle"></i>
                            اسم الشركة باللغة الإنجليزية (اختياري)
                        </div>
                        @error('name_en')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    
                    <div class="form-group-enhanced">
                        <label class="form-label-enhanced">
                            شعار الشركة
                        </label>
                        <div class="logo-upload-container">
                            <div class="logo-preview" id="logoPreview">
                                <div class="logo-placeholder">
                                    <i class="fas fa-cloud-upload-alt"></i>
                                    <p>اسحب الشعار هنا أو اضغط للاختيار</p>
                                    <small>PNG, JPG - أقل من 2 ميجابايت</small>
                                </div>
                            </div>
                            <button type="button" class="btn btn-danger btn-sm remove-logo-btn" id="removeLogo" style="display: none;">
                                <i class="fas fa-trash"></i>
                                إزالة الشعار
                            </button>
                            <input type="file" name="logo" class="form-control form-control-enhanced @error('logo') is-invalid @enderror"
                                   id="logoInput" accept="image/*" style="display: none;">
                            <div class="help-text">
                                <i class="fas fa-info-circle"></i>
                                يُفضل أن يكون الشعار بصيغة PNG أو JPG وبحجم أقصى 2 ميجابايت
                            </div>
                            <div class="logo-error" id="logoError" style="display: none;">
                                <i class="fas fa-exclamation-triangle"></i>
                                <span></span>
                            </div>
                            @error('logo')
                                <div class="invalid-feedback d-block">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    <span class="required">*</span>
                                    الأرقام الضريبية
                                </label>
                                
                                <div id="vatNumbersContainer">
                                    <div class="vat-number-group mb-2" data-index="0">
                                        <div class="input-group">
                                            <input type="text" name="vat_numbers[0]" class="form-control form-control-enhanced vat-number-input @error('vat_numbers.0') is-invalid @enderror"
                                                   value="{{ old('vat_numbers.0') }}" required placeholder="300000000000003" maxlength="15">
                                            <button type="button" class="btn btn-outline-success add-vat-number" title="إضافة رقم ضريبي">
                                                <i class="fas fa-plus"></i>
                                            </button>
                                        </div>
                                        @error('vat_numbers.0')
                                            <div class="invalid-feedback d-block">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                                
                                @error('vat_numbers')
                                    <div class="invalid-feedback d-block">{{ $message }}</div>
                                @enderror
                                
                                <div class="help-text">
                                    <i class="fas fa-shield-alt"></i>
                                    كل رقم يجب أن يكون 15 رقم ويبدأ بـ 3 (مطلوب من هيئة الزكاة والضريبة)
                                </div>
                                
                                <div class="validation-message" id="vatValidation">
                                    <i class="fas fa-check-circle text-success me-2"></i>
                                    الأرقام الضريبية صحيحة ومطابقة للمعايير
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    <span class="required">*</span>
                                    رقم السجل التجاري
                                </label>
                                <input type="text" name="cr_number" class="form-control form-control-enhanced @error('cr_number') is-invalid @enderror"
                                       value="{{ old('cr_number') }}" required placeholder="1010000000" id="crNumberInput">
                                <div class="help-text">
                                    <i class="fas fa-certificate"></i>
                                    رقم السجل التجاري الرسمي من وزارة التجارة
                                </div>
                                @error('cr_number')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- العنوان الوطني -->
            <div class="form-section" id="addressInfo">
                <div class="section-header">
                    <h5 class="section-title">
                        <i class="fas fa-map-marker-alt"></i>
                        العنوان الوطني السعودي
                    </h5>
                </div>
                <div class="card-body p-4">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    <span class="required">*</span>
                                    المدينة
                                </label>
                                <input type="text" name="city" class="form-control form-control-enhanced @error('city') is-invalid @enderror"
                                       value="{{ old('city') }}" required placeholder="الرياض">
                                <div class="help-text">
                                    <i class="fas fa-city"></i>
                                    اختر المدينة الرئيسية للمملكة العربية السعودية
                                </div>
                                @error('city')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    <span class="required">*</span>
                                    الحي
                                </label>
                                <input type="text" name="district" class="form-control form-control-enhanced @error('district') is-invalid @enderror"
                                       value="{{ old('district') }}" required placeholder="الملز">
                                <div class="help-text">
                                    <i class="fas fa-map"></i>
                                    اسم الحي أو المنطقة داخل المدينة
                                </div>
                                @error('district')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="form-group-enhanced">
                        <label class="form-label-enhanced">
                            <span class="required">*</span>
                            اسم الشارع
                        </label>
                        <input type="text" name="street" class="form-control form-control-enhanced @error('street') is-invalid @enderror"
                               value="{{ old('street') }}" required placeholder="شارع الملك فهد">
                        <div class="help-text">
                            <i class="fas fa-road"></i>
                            الاسم الكامل للشارع أو الطريق
                        </div>
                        @error('street')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    
                    <div class="row">
                        <div class="col-md-3">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    <span class="required">*</span>
                                    رقم المبنى
                                </label>
                                <input type="text" name="building_number" class="form-control form-control-enhanced @error('building_number') is-invalid @enderror"
                                       value="{{ old('building_number') }}" required placeholder="1234" maxlength="4">
                                <div class="help-text">
                                    <i class="fas fa-building"></i>
                                    4 أرقام كحد أقصى
                                </div>
                                @error('building_number')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="col-md-3">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    <span class="required">*</span>
                                    الرقم الإضافي
                                </label>
                                <input type="text" name="additional_number" class="form-control form-control-enhanced @error('additional_number') is-invalid @enderror"
                                       value="{{ old('additional_number') }}" required placeholder="5678" maxlength="4">
                                <div class="help-text">
                                    <i class="fas fa-plus"></i>
                                    رقم إضافي من العنوان الوطني
                                </div>
                                @error('additional_number')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="col-md-3">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    <span class="required">*</span>
                                    الرمز البريدي
                                </label>
                                <input type="text" name="postal_code" class="form-control form-control-enhanced @error('postal_code') is-invalid @enderror"
                                       value="{{ old('postal_code') }}" required placeholder="12345" maxlength="5">
                                <div class="help-text">
                                    <i class="fas fa-mailbox"></i>
                                    5 أرقام للرمز البريدي
                                </div>
                                @error('postal_code')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="col-md-3">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    رقم الوحدة
                                </label>
                                <input type="text" name="unit_number" class="form-control form-control-enhanced @error('unit_number') is-invalid @enderror"
                                       value="{{ old('unit_number') }}" placeholder="101">
                                <div class="help-text">
                                    <i class="fas fa-door-open"></i>
                                    اختياري - رقم الشقة أو المكتب
                                </div>
                                @error('unit_number')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- معلومات الاتصال -->
            <div class="form-section" id="contactInfo">
                <div class="section-header">
                    <h5 class="section-title">
                        <i class="fas fa-phone"></i>
                        معلومات الاتصال
                    </h5>
                </div>
                <div class="card-body p-4">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    البريد الإلكتروني
                                </label>
                                <input type="email" name="email" class="form-control form-control-enhanced @error('email') is-invalid @enderror"
                                       value="{{ old('email') }}" placeholder="info@company.com">
                                <div class="help-text">
                                    <i class="fas fa-envelope"></i>
                                    البريد الإلكتروني الرسمي للشركة
                                </div>
                                @error('email')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="form-group-enhanced">
                                <label class="form-label-enhanced">
                                    رقم الهاتف
                                </label>
                                <input type="tel" name="phone" class="form-control form-control-enhanced @error('phone') is-invalid @enderror"
                                       value="{{ old('phone') }}" placeholder="+966112345678">
                                <div class="help-text">
                                    <i class="fas fa-phone"></i>
                                    رقم الهاتف مع رمز البلد (+966)
                                </div>
                                @error('phone')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="form-group-enhanced">
                        <label class="form-label-enhanced">
                            الموقع الإلكتروني
                        </label>
                        <input type="url" name="website" class="form-control form-control-enhanced @error('website') is-invalid @enderror"
                               value="{{ old('website') }}" placeholder="https://www.company.com">
                        <div class="help-text">
                            <i class="fas fa-globe"></i>
                            الموقع الإلكتروني الرسمي للشركة (اختياري)
                        </div>
                        @error('website')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                </div>
            </div>
        </div>
        
        <!-- معاينة الشركة -->
        <div class="col-lg-4">
            <div class="preview-card">
                <div class="preview-header">
                    <h5 class="mb-0">
                        <i class="fas fa-eye me-2"></i>
                        معاينة الشركة
                    </h5>
                </div>
                
                <div class="company-preview">
                    <div class="preview-avatar" id="company-preview-avatar">
                        <i class="fas fa-building"></i>
                    </div>
                    <img class="preview-company-logo" id="company-preview-logo" style="display: none;">
                    <div class="preview-name" id="company-preview-name">اسم الشركة</div>
                    <div class="preview-company-name-en" id="company-preview-name-en" style="display: none;"></div>
                    <div class="preview-vat" id="company-preview-vat">الرقم الضريبي</div>
                    
                    <div class="preview-section">
                        <div class="preview-section-title">السجل التجاري</div>
                        <div class="preview-section-content" id="company-preview-cr">غير محدد</div>
                    </div>
                    
                    <div class="preview-section">
                        <div class="preview-section-title">العنوان الكامل</div>
                        <div class="preview-section-content" id="company-preview-address">لم يتم إدخال العنوان</div>
                    </div>
                    
                    <div class="preview-section">
                        <div class="preview-section-title">معلومات الاتصال</div>
                        <div class="preview-section-content" id="company-preview-contact">غير متوفرة</div>
                    </div>
                    
                    <div class="form-check mb-3">
                        <input type="checkbox" name="is_active" class="form-check-input"
                               value="1" {{ old('is_active', true) ? 'checked' : '' }} id="activeStatus">
                        <label class="form-check-label fw-bold" for="activeStatus">
                            <i class="fas fa-toggle-on text-success me-2"></i>
                            شركة نشطة
                        </label>
                    </div>
                    
                    <button type="submit" class="save-btn">
                        <i class="fas fa-save me-2"></i>
                        حفظ الشركة
                    </button>
                </div>
            </div>
            
            <!-- إرشادات -->
            <div class="guidelines-card">
                <div class="guidelines-header">
                    <h6 class="mb-0">
                        <i class="fas fa-lightbulb me-2"></i>
                        إرشادات مهمة
                    </h6>
                </div>
                <div class="card-body p-3">
                    <div class="alert alert-info border-0 mb-3">
                        <h6 class="text-info">
                            <i class="fas fa-check-circle me-2"></i>
                            متطلبات ZATCA
                        </h6>
                        <ul class="mb-0 small list-unstyled">
                            <li class="mb-1"><i class="fas fa-dot-circle me-2 text-primary"></i>الرقم الضريبي صحيح ومُسجل</li>
                            <li class="mb-1"><i class="fas fa-dot-circle me-2 text-primary"></i>العنوان الوطني كامل ودقيق</li>
                            <li class="mb-1"><i class="fas fa-dot-circle me-2 text-primary"></i>السجل التجاري صالح</li>
                            <li class="mb-1"><i class="fas fa-dot-circle me-2 text-primary"></i>بيانات الاتصال محدثة</li>
                        </ul>
                    </div>
                    
                    <div class="alert alert-warning border-0 mb-0">
                        <h6 class="text-warning">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            تنبيه مهم
                        </h6>
                        <p class="mb-0 small">
                            تأكد من صحة البيانات قبل الحفظ. بعض الحقول قد تتطلب تحديثاً في ZATCA عند التعديل لاحقاً.
                        </p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</form>
@endsection

@push('scripts')
<script>
// متغيرات عامة
let currentStep = 1;
const totalSteps = 4;

// تحديث شريط التقدم
function updateProgress() {
    const requiredFields = document.querySelectorAll('input[required]');
    const filledFields = Array.from(requiredFields).filter(field => field.value.trim() !== '');
    const progress = (filledFields.length / requiredFields.length) * 100;
    
    document.getElementById('formProgress').style.width = progress + '%';
    document.getElementById('progressText').textContent = Math.round(progress) + '%';
    
    // تحديث خطوات النموذج
    updateSteps();
}

// تحديث حالة الخطوات
function updateSteps() {
    const basicFields = ['name', 'cr_number'];
    const addressFields = ['city', 'district', 'street', 'building_number', 'additional_number', 'postal_code'];
    const contactFields = ['email', 'phone', 'website'];
    
    // فحص الخطوة الأولى (بما في ذلك الأرقام الضريبية)
    const basicFieldsCompleted = basicFields.every(field =>
        document.querySelector(`input[name="${field}"]`).value.trim() !== ''
    );
    
    const vatNumbers = document.querySelectorAll('.vat-number-input');
    const vatCompleted = Array.from(vatNumbers).some(input => input.value.trim() !== '');
    
    const basicCompleted = basicFieldsCompleted && vatCompleted;
    
    // فحص الخطوة الثانية
    const addressCompleted = addressFields.slice(0, -1).every(field =>
        document.querySelector(`input[name="${field}"]`).value.trim() !== ''
    ); // عدم تضمين postal_code لأنه اختياري
    
    // فحص الخطوة الثالثة
    const contactCompleted = contactFields.some(field =>
        document.querySelector(`input[name="${field}"]`).value.trim() !== ''
    );
    
    // تحديث العرض
    updateStepDisplay(1, basicCompleted);
    updateStepDisplay(2, addressCompleted);
    updateStepDisplay(3, contactCompleted);
    updateStepDisplay(4, basicCompleted && addressCompleted);
}

function updateStepDisplay(stepNumber, completed) {
    const step = document.getElementById(`step${stepNumber}`);
    
    if (completed) {
        step.classList.add('completed');
        step.classList.remove('active');
    } else if (stepNumber === currentStep) {
        step.classList.add('active');
        step.classList.remove('completed');
    } else {
        step.classList.remove('active', 'completed');
    }
}

// تحديث المعاينة
function updatePreview() {
    // اسم الشركة
    const name = document.querySelector('input[name="name"]').value || 'اسم الشركة';
    document.getElementById('company-preview-name').textContent = name;
    
    // اسم الشركة بالإنجليزية
    const nameEn = document.querySelector('input[name="name_en"]').value;
    const previewNameEn = document.getElementById('company-preview-name-en');
    if (nameEn.trim()) {
        previewNameEn.textContent = nameEn;
        previewNameEn.style.display = 'block';
    } else {
        previewNameEn.style.display = 'none';
    }
    
    // الأرقام الضريبية
    const vatInputs = document.querySelectorAll('.vat-number-input');
    let vatNumbers = [];
    vatInputs.forEach(input => {
        if (input.value.trim() !== '') {
            vatNumbers.push(input.value.trim());
        }
    });
    
    const vatDisplay = vatNumbers.length > 0 ? vatNumbers.join(' - ') : 'غير محدد';
    document.getElementById('company-preview-vat').textContent = vatDisplay;
    
    // السجل التجاري
    const crNumber = document.querySelector('input[name="cr_number"]').value || 'غير محدد';
    document.getElementById('company-preview-cr').textContent = crNumber;
    
    // العنوان
    updateAddressPreview();
    
    // معلومات الاتصال
    updateContactPreview();
    
    // تحديث التقدم
    updateProgress();
}

function updateAddressPreview() {
    const city = document.querySelector('input[name="city"]').value;
    const district = document.querySelector('input[name="district"]').value;
    const street = document.querySelector('input[name="street"]').value;
    const building = document.querySelector('input[name="building_number"]').value;
    const postal = document.querySelector('input[name="postal_code"]').value;
    const additional = document.querySelector('input[name="additional_number"]').value;
    const unit = document.querySelector('input[name="unit_number"]').value;
    
    let address = '';
    if (street) address += street;
    if (building) address += (address ? '، مبنى ' : 'مبنى ') + building;
    if (additional) address += (address ? '، إضافي ' : 'إضافي ') + additional;
    if (unit) address += (address ? '، وحدة ' : 'وحدة ') + unit;
    if (district) address += (address ? '، ' : '') + district;
    if (city) address += (address ? '، ' : '') + city;
    if (postal) address += (address ? ' ' : '') + postal;
    
    document.getElementById('company-preview-address').textContent = address || 'لم يتم إدخال العنوان';
}

function updateContactPreview() {
    const email = document.querySelector('input[name="email"]').value;
    const phone = document.querySelector('input[name="phone"]').value;
    const website = document.querySelector('input[name="website"]').value;
    
    let contact = '';
    if (email) contact += '📧 ' + email;
    if (phone) contact += (contact ? '\n📞 ' : '📞 ') + phone;
    if (website) contact += (contact ? '\n🌐 ' : '🌐 ') + website;
    
    document.getElementById('company-preview-contact').innerHTML = contact.replace(/\n/g, '<br>') || 'غير متوفرة';
}

// التحقق من صحة الأرقام الضريبية
function validateVatNumbers() {
    const vatInputs = document.querySelectorAll('.vat-number-input');
    const validationDiv = document.getElementById('vatValidation');
    let allValid = true;
    let hasValues = false;
    
    vatInputs.forEach(input => {
        const vatNumber = input.value;
        
        if (vatNumber.length > 0) {
            hasValues = true;
            if (vatNumber.length === 15 && vatNumber.startsWith('3')) {
                input.classList.remove('is-invalid');
                input.classList.add('is-valid');
            } else {
                input.classList.remove('is-valid');
                input.classList.add('is-invalid');
                allValid = false;
            }
        } else {
            input.classList.remove('is-valid', 'is-invalid');
            if (input.hasAttribute('required')) {
                allValid = false;
            }
        }
    });
    
    if (allValid && hasValues) {
        validationDiv.classList.add('show');
    } else {
        validationDiv.classList.remove('show');
    }
}

// إضافة حقل رقم ضريبي جديد
function addVatNumberField() {
    const container = document.getElementById('vatNumbersContainer');
    const groups = container.querySelectorAll('.vat-number-group');
    const newIndex = groups.length;
    
    const newGroup = document.createElement('div');
    newGroup.className = 'vat-number-group mb-2';
    newGroup.setAttribute('data-index', newIndex);
    
    newGroup.innerHTML = `
        <div class="input-group">
            <input type="text" name="vat_numbers[${newIndex}]" class="form-control form-control-enhanced vat-number-input"
                   placeholder="300000000000003" maxlength="15" required>
            <button type="button" class="btn btn-outline-success add-vat-number" title="إضافة رقم ضريبي">
                <i class="fas fa-plus"></i>
            </button>
            <button type="button" class="btn btn-outline-danger remove-vat-number" title="إزالة رقم ضريبي">
                <i class="fas fa-trash"></i>
            </button>
        </div>
    `;
    
    container.appendChild(newGroup);
    
    // ربط الأحداث للحقل الجديد
    const newInput = newGroup.querySelector('.vat-number-input');
    newInput.addEventListener('input', function() {
        validateVatNumbers();
        updatePreview();
    });
    
    const addBtn = newGroup.querySelector('.add-vat-number');
    addBtn.addEventListener('click', addVatNumberField);
    
    const removeBtn = newGroup.querySelector('.remove-vat-number');
    removeBtn.addEventListener('click', function() {
        removeVatNumberField(newGroup);
    });
    
    // التركيز على الحقل الجديد
    newInput.focus();
    
    validateVatNumbers();
    updatePreview();
}

// إزالة حقل رقم ضريبي
function removeVatNumberField(group) {
    const container = document.getElementById('vatNumbersContainer');
    const groups = container.querySelectorAll('.vat-number-group');
    
    // لا يمكن إزالة الحقل الأول إذا كان الوحيد
    if (groups.length > 1) {
        group.remove();
        validateVatNumbers();
        updatePreview();
    }
}

// مسح النموذج
function clearForm() {
    if (confirm('هل تريد مسح جميع البيانات المدخلة؟')) {
        document.getElementById('company-form').reset();
        
        // إزالة فئات التحقق
        document.querySelectorAll('.form-control-enhanced').forEach(input => {
            input.classList.remove('is-valid', 'is-invalid');
        });
        
        // تحديث المعاينة
        updatePreview();
        
        // إعادة تعيين التقدم
        currentStep = 1;
        updateProgress();
    }
}

// التحقق من صحة الملف
function validateLogoFile(file) {
    const maxSize = 2 * 1024 * 1024; // 2MB
    const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
    
    if (!allowedTypes.includes(file.type)) {
        return 'نوع الملف غير مدعوم. يرجى اختيار صورة بصيغة PNG, JPG, أو GIF';
    }
    
    if (file.size > maxSize) {
        return 'حجم الملف كبير جداً. يجب أن يكون أقل من 2 ميجابايت';
    }
    
    return null;
}

// عرض رسالة خطأ
function showLogoError(message) {
    const errorDiv = document.getElementById('logoError');
    const errorSpan = errorDiv.querySelector('span');
    
    errorSpan.textContent = message;
    errorDiv.style.display = 'flex';
    
    // إخفاء الرسالة بعد 5 ثوان
    setTimeout(() => {
        errorDiv.style.display = 'none';
    }, 5000);
}

// إخفاء رسالة الخطأ
function hideLogoError() {
    const errorDiv = document.getElementById('logoError');
    errorDiv.style.display = 'none';
}

// تحديث معاينة الشعار
function updateLogoPreview(file) {
    const logoPreview = document.getElementById('logoPreview');
    const companyPreviewLogo = document.getElementById('company-preview-logo');
    const companyPreviewAvatar = document.getElementById('company-preview-avatar');
    const removeBtn = document.getElementById('removeLogo');
    
    if (file && file.type.startsWith('image/')) {
        // التحقق من صحة الملف
        const validationError = validateLogoFile(file);
        if (validationError) {
            showLogoError(validationError);
            return;
        }
        
        hideLogoError();
        
        const reader = new FileReader();
        reader.onload = function(e) {
            logoPreview.innerHTML = `<img src="${e.target.result}" alt="شعار الشركة">`;
            logoPreview.classList.add('has-image');
            
            // تحديث معاينة الشعار في البطاقة
            companyPreviewLogo.src = e.target.result;
            companyPreviewLogo.style.display = 'block';
            companyPreviewAvatar.style.display = 'none';
            
            // إظهار زر الإزالة
            removeBtn.style.display = 'inline-block';
        };
        reader.readAsDataURL(file);
    } else {
        clearLogoPreview();
    }
}

// مسح معاينة الشعار
function clearLogoPreview() {
    const logoPreview = document.getElementById('logoPreview');
    const companyPreviewLogo = document.getElementById('company-preview-logo');
    const companyPreviewAvatar = document.getElementById('company-preview-avatar');
    const removeBtn = document.getElementById('removeLogo');
    const logoInput = document.getElementById('logoInput');
    
    // إعادة تعيين المعاينة
    logoPreview.innerHTML = `
        <div class="logo-placeholder">
            <i class="fas fa-cloud-upload-alt"></i>
            <p>اسحب الشعار هنا أو اضغط للاختيار</p>
            <small>PNG, JPG - أقل من 2 ميجابايت</small>
        </div>
    `;
    logoPreview.classList.remove('has-image');
    companyPreviewLogo.style.display = 'none';
    companyPreviewAvatar.style.display = 'flex';
    removeBtn.style.display = 'none';
    logoInput.value = '';
    hideLogoError();
}

// تهيئة الصفحة
document.addEventListener('DOMContentLoaded', function() {
    // ربط أحداث التحديث للحقول العادية
    document.querySelectorAll('input:not(.vat-number-input):not(#logoInput)').forEach(input => {
        input.addEventListener('input', updatePreview);
        
        // تأثيرات التركيز
        input.addEventListener('focus', function() {
            this.closest('.form-section').style.transform = 'translateY(-2px)';
        });
        
        input.addEventListener('blur', function() {
            this.closest('.form-section').style.transform = 'translateY(0)';
        });
    });
    
    // ربط حدث تحميل الشعار
    const logoInput = document.getElementById('logoInput');
    const logoPreview = document.getElementById('logoPreview');
    const removeLogo = document.getElementById('removeLogo');
    
    logoInput.addEventListener('change', function(e) {
        const file = e.target.files[0];
        updateLogoPreview(file);
    });
    
    // جعل منطقة المعاينة قابلة للنقر
    logoPreview.addEventListener('click', function() {
        if (!logoPreview.classList.contains('has-image')) {
            logoInput.click();
        }
    });
    
    // ربط زر إزالة الشعار
    removeLogo.addEventListener('click', function() {
        clearLogoPreview();
    });
    
    // وظائف السحب والإفلات
    logoPreview.addEventListener('dragover', function(e) {
        e.preventDefault();
        logoPreview.classList.add('drag-over');
    });
    
    logoPreview.addEventListener('dragleave', function(e) {
        e.preventDefault();
        logoPreview.classList.remove('drag-over');
    });
    
    logoPreview.addEventListener('drop', function(e) {
        e.preventDefault();
        logoPreview.classList.remove('drag-over');
        
        const files = e.dataTransfer.files;
        if (files.length > 0) {
            const file = files[0];
            if (file.type.startsWith('image/')) {
                logoInput.files = files;
                updateLogoPreview(file);
            } else {
                showLogoError('يرجى سحب ملف صورة فقط');
            }
        }
    });
    
    // منع السلوك الافتراضي للسحب والإفلات في النافذة
    ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
        document.addEventListener(eventName, function(e) {
            e.preventDefault();
        });
    });
    
    // ربط أحداث الأرقام الضريبية
    document.querySelectorAll('.vat-number-input').forEach(input => {
        input.addEventListener('input', function() {
            validateVatNumbers();
            updatePreview();
        });
    });
    
    // ربط أحداث أزرار إضافة/إزالة الأرقام الضريبية
    document.querySelectorAll('.add-vat-number').forEach(btn => {
        btn.addEventListener('click', addVatNumberField);
    });
    
    // تحديث المعاينة الأولية
    updatePreview();
    
    // منع إرسال النموذج إذا كانت البيانات غير صحيحة
    document.getElementById('company-form').addEventListener('submit', function(e) {
        const vatInputs = document.querySelectorAll('.vat-number-input');
        const logoInput = document.getElementById('logoInput');
        let hasValidVat = false;
        
        // التحقق من الأرقام الضريبية
        for (let input of vatInputs) {
            const vatNumber = input.value.trim();
            if (vatNumber.length > 0) {
                if (vatNumber.length !== 15 || !vatNumber.startsWith('3')) {
                    e.preventDefault();
                    alert('جميع الأرقام الضريبية يجب أن تكون 15 رقم وتبدأ بـ 3');
                    input.focus();
                    return false;
                }
                hasValidVat = true;
            }
        }
        
        if (!hasValidVat) {
            e.preventDefault();
            alert('يجب إدخال رقم ضريبي واحد على الأقل');
            document.querySelector('.vat-number-input').focus();
            return false;
        }
        
        // التحقق من الشعار إذا كان محدداً
        if (logoInput.files.length > 0) {
            const file = logoInput.files[0];
            const validationError = validateLogoFile(file);
            if (validationError) {
                e.preventDefault();
                alert('خطأ في الشعار: ' + validationError);
                return false;
            }
        }
        
        // التحقق من اسم الشركة بالإنجليزية (إذا كان مدخلاً)
        const nameEn = document.querySelector('input[name="name_en"]').value.trim();
        if (nameEn && !/^[a-zA-Z0-9\s\-&.,()]+$/.test(nameEn)) {
            e.preventDefault();
            alert('اسم الشركة بالإنجليزية يجب أن يحتوي على أحرف إنجليزية فقط');
            document.querySelector('input[name="name_en"]').focus();
            return false;
        }
        
        // إظهار مؤشر التحميل
        const submitBtn = document.querySelector('.save-btn');
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>جاري الحفظ...';
        submitBtn.disabled = true;
    });
});

// حفظ البيانات محلياً أثناء الكتابة
function saveFormData() {
    const formData = new FormData(document.getElementById('company-form'));
    const data = {};
    
    for (let [key, value] of formData.entries()) {
        data[key] = value;
    }
    
    localStorage.setItem('companyFormData', JSON.stringify(data));
}

// استرداد البيانات المحفوظة
function loadFormData() {
    const savedData = localStorage.getItem('companyFormData');
    
    if (savedData) {
        const data = JSON.parse(savedData);
        
        Object.keys(data).forEach(key => {
            const input = document.querySelector(`input[name="${key}"]`);
            if (input && input.type !== 'checkbox') {
                input.value = data[key];
            } else if (input && input.type === 'checkbox') {
                input.checked = data[key] === '1';
            }
        });
        
        updatePreview();
    }
}

// مسح البيانات المحفوظة عند نجاح الحفظ
window.addEventListener('beforeunload', function() {
    if (document.getElementById('company-form').checkValidity()) {
        localStorage.removeItem('companyFormData');
    } else {
        saveFormData();
    }
});
</script>
@endpush