@extends('layouts.app')

@section('title', 'تعديل الحركة البنكية')

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="card-title mb-0">تعديل الحركة البنكية</h5>
                    <div>
                        <a href="{{ route('accounting.bank-transactions.show', $bankTransaction) }}" class="btn btn-info">
                            <i class="fas fa-eye"></i> عرض الحركة
                        </a>
                        <a href="{{ route('accounting.bank-transactions.index') }}" class="btn btn-secondary">
                            <i class="fas fa-arrow-left"></i> العودة للقائمة
                        </a>
                    </div>
                </div>

                <div class="card-body">
                    @if($bankTransaction->status != 'pending')
                        <div class="alert alert-warning">
                            <i class="fas fa-exclamation-triangle"></i>
                            <strong>تنبيه:</strong> هذه الحركة {{ $bankTransaction->status == 'cleared' ? 'مصفاة' : 'ملغية' }} ولا يمكن تعديلها.
                        </div>
                    @else
                        <form action="{{ route('accounting.bank-transactions.update', $bankTransaction) }}" method="POST">
                            @csrf
                            @method('PUT')
                            
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="bank_account_id" class="form-label">الحساب البنكي <span class="text-danger">*</span></label>
                                        <select class="form-select @error('bank_account_id') is-invalid @enderror" 
                                                id="bank_account_id" name="bank_account_id" required onchange="updateAccountInfo()">
                                            <option value="">اختر الحساب البنكي</option>
                                            @foreach($bankAccounts as $bankAccount)
                                                <option value="{{ $bankAccount->id }}" 
                                                        data-currency="{{ $bankAccount->currency }}"
                                                        data-balance="{{ $bankAccount->current_balance }}"
                                                        {{ old('bank_account_id', $bankTransaction->bank_account_id) == $bankAccount->id ? 'selected' : '' }}>
                                                    {{ $bankAccount->bank_name }} - {{ $bankAccount->account_number }}
                                                    ({{ number_format($bankAccount->current_balance, 2) }} {{ $bankAccount->currency }})
                                                </option>
                                            @endforeach
                                        </select>
                                        @error('bank_account_id')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        @if($bankTransaction->journal_entry_id)
                                            <div class="form-text text-warning">
                                                <i class="fas fa-exclamation-triangle"></i>
                                                تحذير: يوجد قيد محاسبي مرتبط بهذه الحركة. تغيير الحساب قد يؤثر على القيد.
                                            </div>
                                        @endif
                                    </div>
                                </div>

                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="type" class="form-label">نوع الحركة <span class="text-danger">*</span></label>
                                        <select class="form-select @error('type') is-invalid @enderror"
                                                id="type" name="type" required onchange="updateTransactionType()">
                                            <option value="">اختر نوع الحركة</option>
                                            <option value="deposit" {{ old('type', $bankTransaction->type) == 'deposit' ? 'selected' : '' }}>
                                                إيداع
                                            </option>
                                            <option value="withdrawal" {{ old('type', $bankTransaction->type) == 'withdrawal' ? 'selected' : '' }}>
                                                سحب
                                            </option>
                                            <option value="transfer" {{ old('type', $bankTransaction->type) == 'transfer' ? 'selected' : '' }}>
                                                تحويل
                                            </option>
                                        </select>
                                        @error('type')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>

                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="transaction_date" class="form-label">تاريخ الحركة <span class="text-danger">*</span></label>
                                        <input type="date" class="form-control @error('transaction_date') is-invalid @enderror" 
                                               id="transaction_date" name="transaction_date" 
                                               value="{{ old('transaction_date', $bankTransaction->transaction_date->format('Y-m-d')) }}" required>
                                        @error('transaction_date')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>

                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="amount" class="form-label">المبلغ <span class="text-danger">*</span></label>
                                        <div class="input-group">
                                            <input type="number" step="0.01" min="0.01" 
                                                   class="form-control @error('amount') is-invalid @enderror" 
                                                   id="amount" name="amount" value="{{ old('amount', $bankTransaction->amount) }}" 
                                                   placeholder="0.00" required>
                                            <span class="input-group-text" id="currencySymbol">ريال</span>
                                        </div>
                                        @error('amount')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>

                            <div class="mb-3">
                                <label for="description" class="form-label">البيان <span class="text-danger">*</span></label>
                                <textarea class="form-control @error('description') is-invalid @enderror" 
                                          id="description" name="description" rows="3" 
                                          placeholder="وصف تفصيلي للحركة البنكية" required>{{ old('description', $bankTransaction->description) }}</textarea>
                                @error('description')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <!-- Transfer specific fields -->
                            <div id="transferFields" style="display: {{ $bankTransaction->type == 'transfer' ? 'block' : 'none' }};">
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="to_bank_account_id" class="form-label">تحويل إلى حساب</label>
                                            <select class="form-select @error('to_bank_account_id') is-invalid @enderror" 
                                                    id="to_bank_account_id" name="to_bank_account_id">
                                                <option value="">اختر الحساب المستلم</option>
                                                @foreach($bankAccounts as $bankAccount)
                                                    <option value="{{ $bankAccount->id }}" 
                                                            {{ old('to_bank_account_id', $bankTransaction->to_bank_account_id) == $bankAccount->id ? 'selected' : '' }}>
                                                        {{ $bankAccount->bank_name }} - {{ $bankAccount->account_number }}
                                                    </option>
                                                @endforeach
                                            </select>
                                            @error('to_bank_account_id')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                    </div>

                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="transfer_fees" class="form-label">رسوم التحويل</label>
                                            <div class="input-group">
                                                <input type="number" step="0.01" min="0" 
                                                       class="form-control @error('transfer_fees') is-invalid @enderror" 
                                                       id="transfer_fees" name="transfer_fees" 
                                                       value="{{ old('transfer_fees', $bankTransaction->transfer_fees ?? 0) }}" 
                                                       placeholder="0.00">
                                                <span class="input-group-text">ريال</span>
                                            </div>
                                            @error('transfer_fees')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="reference_number" class="form-label">رقم المرجع</label>
                                        <input type="text" class="form-control @error('reference_number') is-invalid @enderror" 
                                               id="reference_number" name="reference_number" 
                                               value="{{ old('reference_number', $bankTransaction->reference_number) }}" 
                                               placeholder="رقم الشيك أو رقم التحويل">
                                        @error('reference_number')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>

                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="status" class="form-label">حالة الحركة <span class="text-danger">*</span></label>
                                        <select class="form-select @error('status') is-invalid @enderror" 
                                                id="status" name="status" required>
                                            <option value="pending" {{ old('status', $bankTransaction->status) == 'pending' ? 'selected' : '' }}>معلق</option>
                                            <option value="cleared" {{ old('status', $bankTransaction->status) == 'cleared' ? 'selected' : '' }}>مصفى</option>
                                        </select>
                                        @error('status')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <div class="form-text">
                                            تغيير الحالة إلى "مصفى" سيؤثر على رصيد الحساب
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Transaction Info -->
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="card border-info">
                                        <div class="card-header bg-info text-white">
                                            <h6 class="mb-0">معلومات الحركة</h6>
                                        </div>
                                        <div class="card-body">
                                            <p><strong>تاريخ الإنشاء:</strong> {{ $bankTransaction->created_at->format('Y-m-d H:i') }}</p>
                                            <p><strong>آخر تحديث:</strong> {{ $bankTransaction->updated_at->format('Y-m-d H:i') }}</p>
                                            <p><strong>الحالة الحالية:</strong> 
                                                @switch($bankTransaction->status)
                                                    @case('pending')
                                                        <span class="badge bg-warning">معلق</span>
                                                        @break
                                                    @case('cleared')
                                                        <span class="badge bg-success">مصفى</span>
                                                        @break
                                                    @case('cancelled')
                                                        <span class="badge bg-danger">ملغي</span>
                                                        @break
                                                @endswitch
                                            </p>
                                        </div>
                                    </div>
                                </div>

                                <div class="col-md-6">
                                    <div class="card border-warning">
                                        <div class="card-header bg-warning text-dark">
                                            <h6 class="mb-0">تنبيهات</h6>
                                        </div>
                                        <div class="card-body">
                                            @if($bankTransaction->journal_entry_id)
                                                <div class="alert alert-warning">
                                                    <i class="fas fa-exclamation-triangle"></i>
                                                    يوجد قيد محاسبي مرتبط بهذه الحركة. التعديل قد يؤثر على القيد.
                                                    <br><a href="{{ route('accounting.journal-entries.show', $bankTransaction->journal_entry_id) }}">عرض القيد</a>
                                                </div>
                                            @else
                                                <div class="alert alert-success">
                                                    <i class="fas fa-check-circle"></i>
                                                    لا يوجد قيد محاسبي مرتبط. يمكن التعديل بأمان.
                                                </div>
                                            @endif
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Account Balance Impact -->
                            <div id="accountInfo" class="card bg-light mb-3">
                                <div class="card-header">
                                    <h6 class="mb-0">تأثير التعديل على الرصيد</h6>
                                </div>
                                <div class="card-body">
                                    <div class="row">
                                        <div class="col-md-4">
                                            <strong>الرصيد الحالي:</strong>
                                            <span id="currentBalance" class="text-success">0.00 ريال</span>
                                        </div>
                                        <div class="col-md-4">
                                            <strong>الرصيد بعد التعديل:</strong>
                                            <span id="newBalance" class="text-info">0.00 ريال</span>
                                        </div>
                                        <div class="col-md-4">
                                            <strong>التغيير:</strong>
                                            <span id="balanceChange" class="text-primary">0.00 ريال</span>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="d-flex justify-content-between mt-4">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save"></i> حفظ التغييرات
                                </button>
                                <div>
                                    <a href="{{ route('accounting.bank-transactions.show', $bankTransaction) }}" class="btn btn-secondary">
                                        <i class="fas fa-times"></i> إلغاء
                                    </a>
                                    <button type="button" class="btn btn-danger" data-bs-toggle="modal" data-bs-target="#deleteModal">
                                        <i class="fas fa-trash"></i> حذف الحركة
                                    </button>
                                </div>
                            </div>
                        </form>
                    @endif
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
@if($bankTransaction->status == 'pending')
<div class="modal fade" id="deleteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">تأكيد الحذف</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                هل أنت متأكد من حذف هذه الحركة البنكية؟
                <br><small class="text-muted">لا يمكن التراجع عن هذا الإجراء.</small>
                @if($bankTransaction->journal_entry_id)
                    <div class="alert alert-warning mt-2">
                        <i class="fas fa-exclamation-triangle"></i>
                        تحذير: سيتم حذف القيد المحاسبي المرتبط أيضاً!
                    </div>
                @endif
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                <form action="{{ route('accounting.bank-transactions.destroy', $bankTransaction) }}" method="POST" class="d-inline">
                    @csrf
                    @method('DELETE')
                    <button type="submit" class="btn btn-danger">حذف نهائي</button>
                </form>
            </div>
        </div>
    </div>
</div>
@endif
@endsection

@push('scripts')
<script>
// Store original values for comparison
const originalAmount = {{ $bankTransaction->amount }};
const originalType = '{{ $bankTransaction->type }}';
const originalFees = {{ $bankTransaction->transfer_fees ?? 0 }};

function updateAccountInfo() {
    const select = document.getElementById('bank_account_id');
    const selectedOption = select.options[select.selectedIndex];
    
    if (selectedOption.value) {
        const currency = selectedOption.getAttribute('data-currency');
        const balance = parseFloat(selectedOption.getAttribute('data-balance'));
        
        document.getElementById('currencySymbol').textContent = currency;
        document.getElementById('currentBalance').textContent = `${balance.toFixed(2)} ${currency}`;
        
        calculateNewBalance();
    }
}

function updateTransactionType() {
    const transactionType = document.getElementById('type').value;
    const transferFields = document.getElementById('transferFields');
    
    if (transactionType === 'transfer') {
        transferFields.style.display = 'block';
        document.getElementById('to_bank_account_id').required = true;
    } else {
        transferFields.style.display = 'none';
        document.getElementById('to_bank_account_id').required = false;
    }
    
    calculateNewBalance();
}

function calculateNewBalance() {
    const select = document.getElementById('bank_account_id');
    const selectedOption = select.options[select.selectedIndex];
    
    if (!selectedOption.value) return;
    
    const currentBalance = parseFloat(selectedOption.getAttribute('data-balance'));
    const newAmount = parseFloat(document.getElementById('amount').value) || 0;
    const newType = document.getElementById('type').value;
    const newFees = parseFloat(document.getElementById('transfer_fees')?.value) || 0;
    const currency = selectedOption.getAttribute('data-currency');
    
    // حساب الرصيد بعد إلغاء تأثير الحركة الأصلية
    let adjustedBalance = currentBalance;
    if (originalType === 'deposit') {
        adjustedBalance -= originalAmount;
    } else if (originalType === 'withdrawal' || originalType === 'transfer') {
        adjustedBalance += originalAmount + originalFees;
    }
    
    // حساب الرصيد بعد تطبيق الحركة المعدلة
    let newBalance = adjustedBalance;
    let change = 0;
    
    if (newType === 'deposit') {
        newBalance = adjustedBalance + newAmount;
        change = newAmount;
    } else if (newType === 'withdrawal' || newType === 'transfer') {
        newBalance = adjustedBalance - newAmount - newFees;
        change = -(newAmount + newFees);
    }
    
    document.getElementById('newBalance').textContent = `${newBalance.toFixed(2)} ${currency}`;
    document.getElementById('newBalance').className = newBalance >= 0 ? 'text-success' : 'text-danger';
    
    document.getElementById('balanceChange').textContent = `${change >= 0 ? '+' : ''}${change.toFixed(2)} ${currency}`;
    document.getElementById('balanceChange').className = change >= 0 ? 'text-success' : 'text-danger';
    
    // تحذير عند السحب على المكشوف
    if (newBalance < 0 && (newType === 'withdrawal' || newType === 'transfer')) {
        if (!document.getElementById('overdraftWarning')) {
            const warning = document.createElement('div');
            warning.id = 'overdraftWarning';
            warning.className = 'alert alert-warning mt-2';
            warning.innerHTML = '<i class="fas fa-exclamation-triangle"></i> تحذير: هذا التعديل سيؤدي إلى سحب على المكشوف!';
            document.getElementById('accountInfo').appendChild(warning);
        }
    } else {
        const existingWarning = document.getElementById('overdraftWarning');
        if (existingWarning) {
            existingWarning.remove();
        }
    }
}

// Event listeners
document.getElementById('amount').addEventListener('input', calculateNewBalance);
document.getElementById('transfer_fees').addEventListener('input', calculateNewBalance);

// تجنب اختيار نفس الحساب للتحويل
document.getElementById('bank_account_id').addEventListener('change', function() {
    const fromAccount = this.value;
    const toAccountSelect = document.getElementById('to_bank_account_id');
    
    Array.from(toAccountSelect.options).forEach(option => {
        option.disabled = option.value === fromAccount;
    });
    
    updateAccountInfo();
});

// تحديث معلومات الحساب عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', function() {
    updateAccountInfo();
    updateTransactionType();
    
    // تحذير عند تغيير البيانات المهمة
    const criticalFields = ['bank_account_id', 'type', 'amount'];
    criticalFields.forEach(fieldId => {
        document.getElementById(fieldId).addEventListener('change', function() {
            const hasJournalEntry = {{ $bankTransaction->journal_entry_id ? 'true' : 'false' }};
            if (hasJournalEntry) {
                const fieldName = {
                    'bank_account_id': 'الحساب البنكي',
                    'type': 'نوع الحركة',
                    'amount': 'المبلغ'
                }[fieldId];
                
                if (!confirm(`تحذير: تغيير ${fieldName} قد يؤثر على القيد المحاسبي المرتبط. هل تريد المتابعة؟`)) {
                    // إعادة القيمة الأصلية
                    this.value = this.getAttribute('data-original') || '';
                    updateAccountInfo();
                    updateTransactionType();
                }
            }
        });
    });
});
</script>
@endpush