@extends('layouts.app')

@section('title', 'إضافة حركة بنكية جديدة')

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="card-title mb-0">إضافة حركة بنكية جديدة</h5>
                    <a href="{{ route('accounting.bank-transactions.index') }}" class="btn btn-secondary">
                        <i class="fas fa-arrow-left"></i> العودة للقائمة
                    </a>
                </div>

                <div class="card-body">
                    <form action="{{ route('accounting.bank-transactions.store') }}" method="POST">
                        @csrf
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="bank_account_id" class="form-label">الحساب البنكي <span class="text-danger">*</span></label>
                                    <select class="form-select @error('bank_account_id') is-invalid @enderror" 
                                            id="bank_account_id" name="bank_account_id" required onchange="updateAccountInfo()">
                                        <option value="">اختر الحساب البنكي</option>
                                        @foreach($bankAccounts as $bankAccount)
                                            <option value="{{ $bankAccount->id }}" 
                                                    data-currency="{{ $bankAccount->currency }}"
                                                    data-balance="{{ $bankAccount->current_balance }}"
                                                    {{ old('bank_account_id', request('bank_account_id')) == $bankAccount->id ? 'selected' : '' }}>
                                                {{ $bankAccount->bank_name }} - {{ $bankAccount->account_number }}
                                                ({{ number_format($bankAccount->current_balance, 2) }} {{ $bankAccount->currency }})
                                            </option>
                                        @endforeach
                                    </select>
                                    @error('bank_account_id')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="type" class="form-label">نوع الحركة <span class="text-danger">*</span></label>
                                    <select class="form-select @error('type') is-invalid @enderror"
                                            id="type" name="type" required onchange="updateTransactionType()">
                                        <option value="">اختر نوع الحركة</option>
                                        <option value="deposit" {{ old('type') == 'deposit' ? 'selected' : '' }}>
                                            <i class="fas fa-arrow-down"></i> إيداع
                                        </option>
                                        <option value="withdrawal" {{ old('type') == 'withdrawal' ? 'selected' : '' }}>
                                            <i class="fas fa-arrow-up"></i> سحب
                                        </option>
                                        <option value="transfer" {{ old('type') == 'transfer' ? 'selected' : '' }}>
                                            <i class="fas fa-exchange-alt"></i> تحويل
                                        </option>
                                    </select>
                                    @error('type')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="transaction_date" class="form-label">تاريخ الحركة <span class="text-danger">*</span></label>
                                    <input type="date" class="form-control @error('transaction_date') is-invalid @enderror" 
                                           id="transaction_date" name="transaction_date" 
                                           value="{{ old('transaction_date', date('Y-m-d')) }}" required>
                                    @error('transaction_date')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="amount" class="form-label">المبلغ <span class="text-danger">*</span></label>
                                    <div class="input-group">
                                        <input type="number" step="0.01" min="0.01" 
                                               class="form-control @error('amount') is-invalid @enderror" 
                                               id="amount" name="amount" value="{{ old('amount') }}" 
                                               placeholder="0.00" required>
                                        <span class="input-group-text" id="currencySymbol">ريال</span>
                                    </div>
                                    @error('amount')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label for="description" class="form-label">البيان <span class="text-danger">*</span></label>
                            <textarea class="form-control @error('description') is-invalid @enderror" 
                                      id="description" name="description" rows="3" 
                                      placeholder="وصف تفصيلي للحركة البنكية" required>{{ old('description') }}</textarea>
                            @error('description')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Transfer specific fields -->
                        <div id="transferFields" style="display: none;">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="to_bank_account_id" class="form-label">تحويل إلى حساب</label>
                                        <select class="form-select @error('to_bank_account_id') is-invalid @enderror" 
                                                id="to_bank_account_id" name="to_bank_account_id">
                                            <option value="">اختر الحساب المستلم</option>
                                            @foreach($bankAccounts as $bankAccount)
                                                <option value="{{ $bankAccount->id }}" {{ old('to_bank_account_id') == $bankAccount->id ? 'selected' : '' }}>
                                                    {{ $bankAccount->bank_name }} - {{ $bankAccount->account_number }}
                                                </option>
                                            @endforeach
                                        </select>
                                        @error('to_bank_account_id')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>

                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="transfer_fees" class="form-label">رسوم التحويل</label>
                                        <div class="input-group">
                                            <input type="number" step="0.01" min="0" 
                                                   class="form-control @error('transfer_fees') is-invalid @enderror" 
                                                   id="transfer_fees" name="transfer_fees" value="{{ old('transfer_fees', 0) }}" 
                                                   placeholder="0.00">
                                            <span class="input-group-text">ريال</span>
                                        </div>
                                        @error('transfer_fees')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="reference_number" class="form-label">رقم المرجع</label>
                                    <input type="text" class="form-control @error('reference_number') is-invalid @enderror" 
                                           id="reference_number" name="reference_number" value="{{ old('reference_number') }}" 
                                           placeholder="رقم الشيك أو رقم التحويل">
                                    @error('reference_number')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="status" class="form-label">حالة الحركة <span class="text-danger">*</span></label>
                                    <select class="form-select @error('status') is-invalid @enderror" 
                                            id="status" name="status" required>
                                        <option value="pending" {{ old('status', 'pending') == 'pending' ? 'selected' : '' }}>معلق</option>
                                        <option value="cleared" {{ old('status') == 'cleared' ? 'selected' : '' }}>مصفى</option>
                                    </select>
                                    @error('status')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                    <div class="form-text">
                                        الحركات المعلقة لا تؤثر على الرصيد حتى يتم تصفيتها
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Account Balance Info -->
                        <div id="accountInfo" class="card bg-light mb-3" style="display: none;">
                            <div class="card-body">
                                <div class="row">
                                    <div class="col-md-4">
                                        <strong>الرصيد الحالي:</strong>
                                        <span id="currentBalance" class="text-success">0.00 ريال</span>
                                    </div>
                                    <div class="col-md-4">
                                        <strong>الرصيد بعد الحركة:</strong>
                                        <span id="newBalance" class="text-info">0.00 ريال</span>
                                    </div>
                                    <div class="col-md-4">
                                        <strong>التغيير:</strong>
                                        <span id="balanceChange" class="text-primary">0.00 ريال</span>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Auto Journal Entry -->
                        <div class="mb-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="auto_journal_entry" name="auto_journal_entry" value="1"
                                       {{ old('auto_journal_entry', true) ? 'checked' : '' }}>
                                <label class="form-check-label" for="auto_journal_entry">
                                    <strong>إنشاء قيد محاسبي تلقائي</strong>
                                </label>
                                <div class="form-text">
                                    سيتم إنشاء قيد محاسبي تلقائياً للحركة البنكية (مستحسن)
                                </div>
                            </div>
                        </div>

                        <!-- Information Cards -->
                        <div class="row">
                            <div class="col-md-6">
                                <div class="card border-info">
                                    <div class="card-header bg-info text-white">
                                        <h6 class="mb-0">أنواع الحركات البنكية</h6>
                                    </div>
                                    <div class="card-body">
                                        <ul class="list-unstyled mb-0">
                                            <li><i class="fas fa-arrow-down text-success"></i> <strong>إيداع:</strong> إضافة أموال للحساب</li>
                                            <li><i class="fas fa-arrow-up text-danger"></i> <strong>سحب:</strong> سحب أموال من الحساب</li>
                                            <li><i class="fas fa-exchange-alt text-info"></i> <strong>تحويل:</strong> نقل أموال بين الحسابات</li>
                                        </ul>
                                    </div>
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="card border-warning">
                                    <div class="card-header bg-warning text-dark">
                                        <h6 class="mb-0">حالات الحركة</h6>
                                    </div>
                                    <div class="card-body">
                                        <ul class="list-unstyled mb-0">
                                            <li><i class="fas fa-clock text-warning"></i> <strong>معلق:</strong> في انتظار التصفية</li>
                                            <li><i class="fas fa-check text-success"></i> <strong>مصفى:</strong> تم تنفيذ الحركة</li>
                                            <li><i class="fas fa-times text-danger"></i> <strong>ملغي:</strong> تم إلغاء الحركة</li>
                                        </ul>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="d-flex justify-content-between mt-4">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save"></i> حفظ الحركة البنكية
                            </button>
                            <a href="{{ route('accounting.bank-transactions.index') }}" class="btn btn-secondary">
                                <i class="fas fa-times"></i> إلغاء
                            </a>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
function updateAccountInfo() {
    const select = document.getElementById('bank_account_id');
    const selectedOption = select.options[select.selectedIndex];
    const accountInfo = document.getElementById('accountInfo');
    
    if (selectedOption.value) {
        const currency = selectedOption.getAttribute('data-currency');
        const balance = parseFloat(selectedOption.getAttribute('data-balance'));
        
        document.getElementById('currencySymbol').textContent = currency;
        document.getElementById('currentBalance').textContent = `${balance.toFixed(2)} ${currency}`;
        
        accountInfo.style.display = 'block';
        calculateNewBalance();
    } else {
        accountInfo.style.display = 'none';
    }
}

function updateTransactionType() {
    const transactionType = document.getElementById('type').value;
    const transferFields = document.getElementById('transferFields');
    
    if (transactionType === 'transfer') {
        transferFields.style.display = 'block';
        document.getElementById('to_bank_account_id').required = true;
    } else {
        transferFields.style.display = 'none';
        document.getElementById('to_bank_account_id').required = false;
    }
    
    calculateNewBalance();
}

function calculateNewBalance() {
    const select = document.getElementById('bank_account_id');
    const selectedOption = select.options[select.selectedIndex];
    
    if (!selectedOption.value) return;
    
    const currentBalance = parseFloat(selectedOption.getAttribute('data-balance'));
    const amount = parseFloat(document.getElementById('amount').value) || 0;
    const transactionType = document.getElementById('type').value;
    const currency = selectedOption.getAttribute('data-currency');
    
    let newBalance = currentBalance;
    let change = 0;
    
    if (transactionType === 'deposit') {
        newBalance = currentBalance + amount;
        change = amount;
    } else if (transactionType === 'withdrawal' || transactionType === 'transfer') {
        const fees = parseFloat(document.getElementById('transfer_fees')?.value) || 0;
        newBalance = currentBalance - amount - fees;
        change = -(amount + fees);
    }
    
    document.getElementById('newBalance').textContent = `${newBalance.toFixed(2)} ${currency}`;
    document.getElementById('newBalance').className = newBalance >= 0 ? 'text-success' : 'text-danger';
    
    document.getElementById('balanceChange').textContent = `${change >= 0 ? '+' : ''}${change.toFixed(2)} ${currency}`;
    document.getElementById('balanceChange').className = change >= 0 ? 'text-success' : 'text-danger';
    
    // تحذير عند السحب على المكشوف
    if (newBalance < 0 && (transactionType === 'withdrawal' || transactionType === 'transfer')) {
        if (!document.getElementById('overdraftWarning')) {
            const warning = document.createElement('div');
            warning.id = 'overdraftWarning';
            warning.className = 'alert alert-warning mt-2';
            warning.innerHTML = '<i class="fas fa-exclamation-triangle"></i> تحذير: هذه الحركة ستؤدي إلى سحب على المكشوف!';
            document.getElementById('accountInfo').appendChild(warning);
        }
    } else {
        const existingWarning = document.getElementById('overdraftWarning');
        if (existingWarning) {
            existingWarning.remove();
        }
    }
}

// Event listeners
document.getElementById('amount').addEventListener('input', calculateNewBalance);
document.getElementById('transfer_fees').addEventListener('input', calculateNewBalance);

// تجنب اختيار نفس الحساب للتحويل
document.getElementById('bank_account_id').addEventListener('change', function() {
    const fromAccount = this.value;
    const toAccountSelect = document.getElementById('to_bank_account_id');
    
    Array.from(toAccountSelect.options).forEach(option => {
        option.disabled = option.value === fromAccount;
    });
});

// تحديث معلومات الحساب عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', function() {
    updateAccountInfo();
    updateTransactionType();
});
</script>
@endpush