@extends('layouts.app')

@section('title', 'الحسابات البنكية')

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <!-- Bank Accounts Card -->
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-university"></i> الحسابات البنكية
                    </h5>
                    <div>
                        <a href="{{ route('accounting.bank-accounts.create') }}" class="btn btn-primary">
                            <i class="fas fa-plus"></i> إضافة حساب بنكي جديد
                        </a>
                    </div>
                </div>

                <div class="card-body">
                    @if($bankAccounts->count() > 0)
                        <!-- Summary Cards -->
                        <div class="row mb-4">
                            @php
                                $totalAccounts = $bankAccounts->count();
                                $activeAccounts = $bankAccounts->where('is_active', true)->count();
                                $totalBalance = $bankAccounts->where('is_active', true)->sum('current_balance');
                                
                                // حساب العملات
                                $currencies = $bankAccounts->groupBy('currency')->map(function($accounts, $currency) {
                                    return [
                                        'count' => $accounts->count(),
                                        'balance' => $accounts->where('is_active', true)->sum('current_balance')
                                    ];
                                });
                                
                                // حساب إجمالي الحركات البنكية
                                $totalTransactions = \App\Models\BankTransaction::whereIn('bank_account_id', $bankAccounts->pluck('id'))->count();
                                $pendingTransactions = \App\Models\BankTransaction::whereIn('bank_account_id', $bankAccounts->pluck('id'))
                                    ->where('status', 'pending')->count();
                            @endphp
                            
                            <div class="col-md-3">
                                <div class="card text-center border-primary">
                                    <div class="card-body">
                                        <h4 class="card-title text-primary">{{ $totalAccounts }}</h4>
                                        <p class="card-text">إجمالي الحسابات</p>
                                        <small class="text-muted">{{ $activeAccounts }} حساب نشط</small>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-md-3">
                                <div class="card text-center border-success">
                                    <div class="card-body">
                                        <h4 class="card-title text-success">{{ number_format($totalBalance, 2) }}</h4>
                                        <p class="card-text">إجمالي الأرصدة</p>
                                        <small class="text-muted">للحسابات النشطة</small>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-md-3">
                                <div class="card text-center border-info">
                                    <div class="card-body">
                                        <h4 class="card-title text-info">{{ $totalTransactions }}</h4>
                                        <p class="card-text">إجمالي الحركات</p>
                                        <small class="text-muted">{{ $pendingTransactions }} حركة معلقة</small>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-md-3">
                                <div class="card text-center border-warning">
                                    <div class="card-body">
                                        <h4 class="card-title text-warning">{{ $currencies->count() }}</h4>
                                        <p class="card-text">عدد العملات</p>
                                        <small class="text-muted">{{ $currencies->keys()->implode(', ') }}</small>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Currency Breakdown -->
                        @if($currencies->count() > 1)
                        <div class="card mb-4 bg-light">
                            <div class="card-header">
                                <h6 class="mb-0">توزيع الحسابات حسب العملة</h6>
                            </div>
                            <div class="card-body">
                                <div class="row">
                                    @foreach($currencies as $currency => $data)
                                    <div class="col-md-3">
                                        <div class="text-center">
                                            <h5 class="text-primary">{{ $data['count'] }} حساب</h5>
                                            <p class="mb-1"><strong>{{ $currency }}</strong></p>
                                            <p class="text-muted">الرصيد: {{ number_format($data['balance'], 2) }}</p>
                                        </div>
                                    </div>
                                    @endforeach
                                </div>
                            </div>
                        </div>
                        @endif

                        <!-- Bank Accounts Table -->
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                    <tr>
                                        <th>اسم البنك</th>
                                        <th>رقم الحساب</th>
                                        <th>الآيبان</th>
                                        <th>الحساب المحاسبي</th>
                                        <th>الرصيد الحالي</th>
                                        <th>آخر حركة</th>
                                        <th>الحالة</th>
                                        <th>الإجراءات</th>
                                    </tr>
                                <tbody>
                                    @foreach($bankAccounts as $bankAccount)
                                        @php
                                            $lastTransaction = \App\Models\BankTransaction::where('bank_account_id', $bankAccount->id)
                                                ->orderBy('transaction_date', 'desc')
                                                ->first();
                                            
                                            $transactionCount = \App\Models\BankTransaction::where('bank_account_id', $bankAccount->id)->count();
                                            $pendingCount = \App\Models\BankTransaction::where('bank_account_id', $bankAccount->id)
                                                ->where('status', 'pending')->count();
                                        @endphp
                                        <tr class="{{ !$bankAccount->is_active ? 'table-secondary' : '' }}">
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <i class="fas fa-university text-primary me-2"></i>
                                                    <div>
                                                        <a href="{{ route('accounting.bank-accounts.show', $bankAccount) }}">
                                                            <strong>{{ $bankAccount->bank_name }}</strong>
                                                        </a>
                                                        @if($bankAccount->branch)
                                                            <br><small class="text-muted">فرع: {{ $bankAccount->branch }}</small>
                                                        @endif
                                                    </div>
                                                </div>
                                            </td>
                                            <td>
                                                <span class="font-monospace">{{ $bankAccount->account_number }}</span>
                                                @if($bankAccount->swift_code)
                                                    <br><small class="text-muted">Swift: {{ $bankAccount->swift_code }}</small>
                                                @endif
                                            </td>
                                            <td>
                                                @if($bankAccount->iban)
                                                    <span class="font-monospace text-info">{{ $bankAccount->iban }}</span>
                                                @else
                                                    <span class="text-muted">غير محدد</span>
                                                @endif
                                            </td>
                                            <td>
                                                <a href="{{ route('accounting.accounts.show', $bankAccount->account) }}">
                                                    {{ $bankAccount->account->code }} - {{ $bankAccount->account->name }}
                                                </a>
                                            </td>
                                            <td>
                                                <span class="fs-6 {{ $bankAccount->current_balance >= 0 ? 'text-success' : 'text-danger' }}">
                                                    {{ number_format($bankAccount->current_balance, 2) }} {{ $bankAccount->currency }}
                                                </span>
                                                @if($transactionCount > 0)
                                                    <br><small class="text-muted">{{ $transactionCount }} حركة</small>
                                                    @if($pendingCount > 0)
                                                        <span class="badge bg-warning">{{ $pendingCount }} معلق</span>
                                                    @endif
                                                @endif
                                            </td>
                                            <td>
                                                @if($lastTransaction)
                                                    <span class="text-muted">{{ $lastTransaction->transaction_date->format('Y-m-d') }}</span>
                                                    <br>
                                                    @switch($lastTransaction->transaction_type)
                                                        @case('deposit')
                                                            <span class="badge bg-success">إيداع</span>
                                                            @break
                                                        @case('withdrawal')
                                                            <span class="badge bg-danger">سحب</span>
                                                            @break
                                                        @case('transfer')
                                                            <span class="badge bg-info">تحويل</span>
                                                            @break
                                                    @endswitch
                                                    <span class="small">{{ number_format($lastTransaction->amount, 2) }}</span>
                                                @else
                                                    <span class="text-muted">لا توجد حركات</span>
                                                @endif
                                            </td>
                                            <td>
                                                @if($bankAccount->is_active)
                                                    <span class="badge bg-success">نشط</span>
                                                @else
                                                    <span class="badge bg-danger">غير نشط</span>
                                                @endif
                                            </td>
                                            <td>
                                                <div class="btn-group" role="group">
                                                    <a href="{{ route('accounting.bank-accounts.show', $bankAccount) }}" 
                                                       class="btn btn-sm btn-outline-primary" title="عرض">
                                                        <i class="fas fa-eye"></i>
                                                    </a>
                                                    <a href="{{ route('accounting.bank-accounts.edit', $bankAccount) }}" 
                                                       class="btn btn-sm btn-outline-warning" title="تعديل">
                                                        <i class="fas fa-edit"></i>
                                                    </a>
                                                    @if($bankAccount->is_active)
                                                        <a href="{{ route('accounting.bank-accounts.statement', $bankAccount) }}" 
                                                           class="btn btn-sm btn-outline-info" title="كشف الحساب">
                                                            <i class="fas fa-file-alt"></i>
                                                        </a>
                                                        <a href="{{ route('accounting.bank-transactions.create', ['bank_account_id' => $bankAccount->id]) }}" 
                                                           class="btn btn-sm btn-outline-success" title="حركة جديدة">
                                                            <i class="fas fa-plus"></i>
                                                        </a>
                                                    @endif
                                                    @if($transactionCount == 0)
                                                        <button type="button" class="btn btn-sm btn-outline-danger" 
                                                                onclick="deleteBankAccount({{ $bankAccount->id }})" title="حذف">
                                                            <i class="fas fa-trash"></i>
                                                        </button>
                                                    @endif
                                                </div>
                                            </td>
                                        </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>

                        <!-- Quick Actions -->
                        <div class="row mt-4">
                            <div class="col-md-6">
                                <div class="card border-success">
                                    <div class="card-header bg-success text-white">
                                        <h6 class="mb-0">الحسابات الأكثر نشاطاً</h6>
                                    </div>
                                    <div class="card-body">
                                        @php
                                            $mostActive = $bankAccounts->sortByDesc(function($account) {
                                                return \App\Models\BankTransaction::where('bank_account_id', $account->id)->count();
                                            })->take(3);
                                        @endphp
                                        @foreach($mostActive as $account)
                                            @php
                                                $count = \App\Models\BankTransaction::where('bank_account_id', $account->id)->count();
                                            @endphp
                                            @if($count > 0)
                                                <div class="d-flex justify-content-between mb-2">
                                                    <span>{{ $account->bank_name }}</span>
                                                    <span class="badge bg-success">{{ $count }} حركة</span>
                                                </div>
                                            @endif
                                        @endforeach
                                        @if($mostActive->filter(function($account) { 
                                            return \App\Models\BankTransaction::where('bank_account_id', $account->id)->count() > 0; 
                                        })->count() == 0)
                                            <p class="text-muted">لا توجد حركات بنكية بعد</p>
                                        @endif
                                    </div>
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="card border-info">
                                    <div class="card-header bg-info text-white">
                                        <h6 class="mb-0">أعلى الأرصدة</h6>
                                    </div>
                                    <div class="card-body">
                                        @php
                                            $highestBalances = $bankAccounts->where('is_active', true)
                                                ->sortByDesc('current_balance')->take(3);
                                        @endphp
                                        @foreach($highestBalances as $account)
                                            @if($account->current_balance > 0)
                                                <div class="d-flex justify-content-between mb-2">
                                                    <span>{{ $account->bank_name }}</span>
                                                    <span class="badge bg-info">{{ number_format($account->current_balance, 2) }} {{ $account->currency }}</span>
                                                </div>
                                            @endif
                                        @endforeach
                                        @if($highestBalances->filter(function($account) { return $account->current_balance > 0; })->count() == 0)
                                            <p class="text-muted">لا توجد أرصدة موجبة</p>
                                        @endif
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Recent Transactions Summary -->
                        @php
                            $recentTransactions = \App\Models\BankTransaction::whereIn('bank_account_id', $bankAccounts->pluck('id'))
                                ->with(['bankAccount'])
                                ->orderBy('transaction_date', 'desc')
                                ->take(5)
                                ->get();
                        @endphp
                        @if($recentTransactions->count() > 0)
                        <div class="card mt-4">
                            <div class="card-header">
                                <h6 class="mb-0">آخر الحركات البنكية</h6>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-sm">
                                            <tr>
                                                <th>التاريخ</th>
                                                <th>البنك</th>
                                                <th>النوع</th>
                                                <th>المبلغ</th>
                                                <th>البيان</th>
                                                <th>الحالة</th>
                                            </tr>
                                        <tbody>
                                            @foreach($recentTransactions as $transaction)
                                            <tr>
                                                <td>{{ $transaction->transaction_date->format('Y-m-d') }}</td>
                                                <td>{{ $transaction->bankAccount->bank_name }}</td>
                                                <td>
                                                    @switch($transaction->transaction_type)
                                                        @case('deposit')
                                                            <span class="badge bg-success">إيداع</span>
                                                            @break
                                                        @case('withdrawal')
                                                            <span class="badge bg-danger">سحب</span>
                                                            @break
                                                        @case('transfer')
                                                            <span class="badge bg-info">تحويل</span>
                                                            @break
                                                    @endswitch
                                                </td>
                                                <td>{{ number_format($transaction->amount, 2) }} {{ $transaction->bankAccount->currency }}</td>
                                                <td>{{ Str::limit($transaction->description, 30) }}</td>
                                                <td>
                                                    @switch($transaction->status)
                                                        @case('pending')
                                                            <span class="badge bg-warning">معلق</span>
                                                            @break
                                                        @case('cleared')
                                                            <span class="badge bg-success">مصفى</span>
                                                            @break
                                                        @case('cancelled')
                                                            <span class="badge bg-danger">ملغي</span>
                                                            @break
                                                    @endswitch
                                                </td>
                                            </tr>
                                            @endforeach
                                        </tbody>
                                    </table>
                                </div>
                                <div class="text-center mt-3">
                                    <a href="{{ route('accounting.bank-transactions.index') }}" class="btn btn-outline-primary">
                                        عرض جميع الحركات البنكية
                                    </a>
                                </div>
                            </div>
                        </div>
                        @endif

                    @else
                        <div class="text-center py-5">
                            <i class="fas fa-university fa-5x text-muted mb-3"></i>
                            <h4 class="text-muted">لا توجد حسابات بنكية</h4>
                            <p class="text-muted">ابدأ بإضافة أول حساب بنكي للنظام</p>
                            <a href="{{ route('accounting.bank-accounts.create') }}" class="btn btn-primary">
                                <i class="fas fa-plus"></i> إضافة حساب بنكي
                            </a>
                        </div>
                    @endif
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deleteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">تأكيد الحذف</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                هل أنت متأكد من حذف الحساب البنكي؟ لا يمكن التراجع عن هذا الإجراء.
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                <form id="deleteForm" method="POST" class="d-inline">
                    @csrf
                    @method('DELETE')
                    <button type="submit" class="btn btn-danger">حذف نهائي</button>
                </form>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
function deleteBankAccount(bankAccountId) {
    const deleteForm = document.getElementById('deleteForm');
    deleteForm.action = `/accounting/bank-accounts/${bankAccountId}`;
    new bootstrap.Modal(document.getElementById('deleteModal')).show();
}
</script>
@endpush

@push('styles')
<style>
.font-monospace {
    font-family: 'Courier New', monospace;
}

.table td, .table th {
    vertical-align: middle;
}
</style>
@endpush