@extends('layouts.app')

@section('title', 'تعديل الحساب')

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="card-title mb-0">تعديل الحساب: {{ $account->name }}</h5>
                    <div>
                        <a href="{{ route('accounting.accounts.show', $account) }}" class="btn btn-info">
                            <i class="fas fa-eye"></i> عرض الحساب
                        </a>
                        <a href="{{ route('accounting.accounts.index') }}" class="btn btn-secondary">
                            <i class="fas fa-arrow-left"></i> العودة للقائمة
                        </a>
                    </div>
                </div>

                <div class="card-body">
                    <form action="{{ route('accounting.accounts.update', $account) }}" method="POST">
                        @csrf
                        @method('PUT')
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="code" class="form-label">كود الحساب <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control @error('code') is-invalid @enderror" 
                                           id="code" name="code" value="{{ old('code', $account->code) }}" required>
                                    @error('code')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="name" class="form-label">اسم الحساب <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control searchable-input @error('name') is-invalid @enderror"
                                           id="name" name="name" value="{{ old('name', $account->name) }}"
                                           list="account-names-list" autocomplete="off" required>
                                    <datalist id="account-names-list">
                                        @if(isset($existingAccounts))
                                            @foreach($existingAccounts as $existingAccount)
                                                @if($existingAccount->id !== $account->id)
                                                    <option value="{{ $existingAccount->name }}">{{ $existingAccount->code }} - {{ $existingAccount->name }}</option>
                                                @endif
                                            @endforeach
                                        @endif
                                    </datalist>
                                    @error('name')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                    <div class="form-text">ابدأ بالكتابة للبحث في أسماء الحسابات الموجودة</div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="type" class="form-label">نوع الحساب <span class="text-danger">*</span></label>
                                    <select class="form-select @error('type') is-invalid @enderror" 
                                            id="type" name="type" required>
                                        <option value="">اختر نوع الحساب</option>
                                        <option value="asset" {{ old('type', $account->type) == 'asset' ? 'selected' : '' }}>أصول</option>
                                        <option value="liability" {{ old('type', $account->type) == 'liability' ? 'selected' : '' }}>خصوم</option>
                                        <option value="equity" {{ old('type', $account->type) == 'equity' ? 'selected' : '' }}>حقوق ملكية</option>
                                        <option value="revenue" {{ old('type', $account->type) == 'revenue' ? 'selected' : '' }}>إيرادات</option>
                                        <option value="expense" {{ old('type', $account->type) == 'expense' ? 'selected' : '' }}>مصروفات</option>
                                    </select>
                                    @error('type')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="parent_id" class="form-label">الحساب الأب</label>
                                    <select class="form-select @error('parent_id') is-invalid @enderror"
                                            id="parent_id" name="parent_id" style="width: 100%;">
                                        <option value="">اختر الحساب الأب (اختياري)</option>
                                        @foreach($parentAccounts as $parentAccount)
                                            @if($parentAccount->id !== $account->id)
                                                <option value="{{ $parentAccount->id }}"
                                                        {{ old('parent_id', $account->parent_id) == $parentAccount->id ? 'selected' : '' }}>
                                                    {{ $parentAccount->code }} - {{ $parentAccount->name }}
                                                </option>
                                            @endif
                                        @endforeach
                                    </select>
                                    @error('parent_id')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <!-- مركز التكلفة -->
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="cost_center_id" class="form-label">مركز التكلفة</label>
                                    <select name="cost_center_id" id="cost_center_id"
                                            class="form-select searchable-select @error('cost_center_id') is-invalid @enderror"
                                            style="width: 100%;">
                                        <option value="">بدون مركز تكلفة</option>
                                        @if(isset($costCenters))
                                            @foreach($costCenters as $costCenter)
                                                <option value="{{ $costCenter->id }}" {{ old('cost_center_id', $account->cost_center_id ?? '') == $costCenter->id ? 'selected' : '' }}>
                                                    {{ $costCenter->code }} - {{ $costCenter->name }}
                                                </option>
                                            @endforeach
                                        @endif
                                    </select>
                                    @error('cost_center_id')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                    <div class="form-text">اختر مركز التكلفة المرتبط بهذا الحساب (اختياري)</div>
                                </div>
                            </div>

                            <!-- رقم المرجع -->
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="reference_number" class="form-label">رقم المرجع</label>
                                    <input type="text" name="reference_number" id="reference_number"
                                           class="form-control @error('reference_number') is-invalid @enderror"
                                           value="{{ old('reference_number', $account->reference_number ?? '') }}"
                                           placeholder="رقم مرجعي للحساب">
                                    @error('reference_number')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                    <div class="form-text">رقم مرجعي اختياري للحساب</div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="is_cash" name="is_cash" value="1"
                                               {{ old('is_cash', $account->is_cash) ? 'checked' : '' }}>
                                        <label class="form-check-label" for="is_cash">
                                            حساب نقدي
                                        </label>
                                    </div>
                                </div>
                            </div>

                            <div class="col-md-4">
                                <div class="mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="is_banking" name="is_banking" value="1"
                                               {{ old('is_banking', $account->is_banking) ? 'checked' : '' }}>
                                        <label class="form-check-label" for="is_banking">
                                            حساب بنكي
                                        </label>
                                    </div>
                                </div>
                            </div>

                            <div class="col-md-4">
                                <div class="mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="is_control" name="is_control" value="1"
                                               {{ old('is_control', $account->is_control) ? 'checked' : '' }}>
                                        <label class="form-check-label" for="is_control">
                                            حساب تحكم
                                        </label>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label for="description" class="form-label">الوصف</label>
                            <textarea class="form-control @error('description') is-invalid @enderror" 
                                      id="description" name="description" rows="3">{{ old('description', $account->description) }}</textarea>
                            @error('description')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="mb-3">
                            <label for="opening_balance" class="form-label">الرصيد الافتتاحي</label>
                            <input type="number" step="0.01" class="form-control @error('opening_balance') is-invalid @enderror" 
                                   id="opening_balance" name="opening_balance" value="{{ old('opening_balance', $account->opening_balance) }}">
                            @error('opening_balance')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                            <div class="form-text">
                                الرصيد الحالي: {{ number_format($account->current_balance, 2) }} ريال
                            </div>
                        </div>

                        <!-- إعدادات الضريبة -->
                        <div class="card mt-4">
                            <div class="card-header">
                                <h5 class="card-title mb-0">
                                    <i class="fas fa-percentage"></i>
                                    إعدادات الضريبة
                                </h5>
                            </div>
                            <div class="card-body">
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-check mb-3">
                                            <input type="checkbox" name="applies_tax" id="applies_tax"
                                                   class="form-check-input" value="1"
                                                   {{ old('applies_tax', $account->applies_tax ?? true) ? 'checked' : '' }}>
                                            <label class="form-check-label" for="applies_tax">
                                                تطبيق الضريبة على هذا الحساب
                                            </label>
                                        </div>
                                        
                                        <div class="mb-3" id="tax_settings_section">
                                            <label for="tax_calculation_method" class="form-label">طريقة حساب الضريبة</label>
                                            <select name="tax_calculation_method" id="tax_calculation_method"
                                                    class="form-select @error('tax_calculation_method') is-invalid @enderror">
                                                <option value="exclusive" {{ old('tax_calculation_method', $account->tax_calculation_method ?? 'exclusive') == 'exclusive' ? 'selected' : '' }}>
                                                    غير شامل الضريبة (Exclusive)
                                                </option>
                                                <option value="inclusive" {{ old('tax_calculation_method', $account->tax_calculation_method ?? 'exclusive') == 'inclusive' ? 'selected' : '' }}>
                                                    شامل الضريبة (Inclusive)
                                                </option>
                                            </select>
                                            @error('tax_calculation_method')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                            <div class="form-text">
                                                <strong>غير شامل الضريبة:</strong> يتم إضافة الضريبة إلى المبلغ الأساسي<br>
                                                <strong>شامل الضريبة:</strong> المبلغ المدخل يتضمن الضريبة بالفعل
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="col-md-6">
                                        <div class="mb-3" id="default_tax_rate_section">
                                            <label for="default_tax_rate" class="form-label">معدل الضريبة الافتراضي (%)</label>
                                            <div class="input-group">
                                                <input type="number" name="default_tax_rate" id="default_tax_rate"
                                                       step="0.01" min="0" max="100"
                                                       class="form-control @error('default_tax_rate') is-invalid @enderror"
                                                       value="{{ old('default_tax_rate', $account->default_tax_rate ?? '15') }}">
                                                <span class="input-group-text">%</span>
                                            </div>
                                            @error('default_tax_rate')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                            <div class="form-text">معدل الضريبة الافتراضي المطبق على هذا الحساب (15% للقيمة المضافة في السعودية)</div>
                                        </div>
                                        
                                        <!-- أمثلة توضيحية -->
                                        <div class="alert alert-info" id="tax_examples">
                                            <h6 class="alert-heading"><i class="fas fa-info-circle"></i> مثال توضيحي:</h6>
                                            <div id="tax_example_content">
                                                <p class="mb-1"><strong>مبلغ 100 ر.س مع ضريبة 15%:</strong></p>
                                                <ul class="mb-0 small">
                                                    <li id="exclusive_example">غير شامل: 100 + 15 = 115 ر.س</li>
                                                    <li id="inclusive_example" style="display: none;">شامل: 86.96 + 13.04 = 100 ر.س</li>
                                                </ul>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                @if($account->tax_calculation_method && $account->tax_calculation_method != 'exclusive')
                                    <div class="alert alert-warning">
                                        <i class="fas fa-exclamation-triangle"></i>
                                        <strong>تنبيه:</strong> تغيير طريقة حساب الضريبة قد يؤثر على القيود والحسابات المرتبطة.
                                    </div>
                                @endif
                            </div>
                        </div>

                        @if($account->children->count() > 0)
                            <div class="alert alert-warning">
                                <i class="fas fa-exclamation-triangle"></i>
                                <strong>تنبيه:</strong> هذا الحساب يحتوي على {{ $account->children->count() }} حساب فرعي.
                                تغيير النوع قد يؤثر على الحسابات الفرعية.
                            </div>
                        @endif

                        @if($account->journalEntryLines->count() > 0)
                            <div class="alert alert-info">
                                <i class="fas fa-info-circle"></i>
                                <strong>معلومة:</strong> يوجد {{ $account->journalEntryLines->count() }} حركة مرتبطة بهذا الحساب.
                            </div>
                        @endif

                        <div class="d-flex justify-content-between">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save"></i> حفظ التغييرات
                            </button>
                            <div>
                                <a href="{{ route('accounting.accounts.show', $account) }}" class="btn btn-secondary">
                                    <i class="fas fa-times"></i> إلغاء
                                </a>
                                @if($account->journalEntryLines->count() == 0 && $account->children->count() == 0)
                                    <button type="button" class="btn btn-danger" data-bs-toggle="modal" data-bs-target="#deleteModal">
                                        <i class="fas fa-trash"></i> حذف الحساب
                                    </button>
                                @endif
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
@if($account->journalEntryLines->count() == 0 && $account->children->count() == 0)
<div class="modal fade" id="deleteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">تأكيد الحذف</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                هل أنت متأكد من حذف الحساب "{{ $account->name }}"؟
                <br><small class="text-muted">لا يمكن التراجع عن هذا الإجراء.</small>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                <form action="{{ route('accounting.accounts.destroy', $account) }}" method="POST" class="d-inline">
                    @csrf
                    @method('DELETE')
                    <button type="submit" class="btn btn-danger">حذف نهائي</button>
                </form>
            </div>
        </div>
    </div>
</div>
@endif
@endsection

@push('styles')
<!-- Select2 CSS -->
<link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
<link href="https://cdn.jsdelivr.net/npm/select2-bootstrap-5-theme@1.3.0/dist/select2-bootstrap-5-theme.min.css" rel="stylesheet" />

<style>
.searchable-input {
    position: relative;
}

.searchable-input:focus {
    border-color: #0d6efd;
    box-shadow: 0 0 0 0.25rem rgba(13, 110, 253, 0.25);
}

.searchable-select {
    min-height: 38px;
}

/* Select2 custom styling */
.select2-container--bootstrap-5 .select2-selection {
    min-height: 38px;
    border: 1px solid #ced4da;
}

.select2-container--bootstrap-5 .select2-selection--single .select2-selection__rendered {
    padding-left: 12px;
    padding-right: 20px;
    color: #495057;
    line-height: 36px;
}

.select2-container--bootstrap-5 .select2-selection--single .select2-selection__arrow {
    height: 36px;
    right: 3px;
}

.select2-container--bootstrap-5.select2-container--focus .select2-selection {
    border-color: #0d6efd;
    box-shadow: 0 0 0 0.25rem rgba(13, 110, 253, 0.25);
}

.select2-dropdown {
    border: 1px solid #ced4da;
    border-radius: 0.375rem;
}

.select2-search--dropdown .select2-search__field {
    border: 1px solid #ced4da;
    border-radius: 0.25rem;
    padding: 6px 12px;
}

.select2-results__option {
    padding: 8px 12px;
}

.select2-results__option--highlighted {
    background-color: #0d6efd;
    color: white;
}

/* Datalist styling */
datalist {
    position: absolute;
    background-color: white;
    border: 1px solid #ced4da;
    border-radius: 0.375rem;
    max-height: 200px;
    overflow-y: auto;
    z-index: 1000;
}
</style>
@endpush

@push('scripts')
<!-- Select2 JS -->
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>

<script>
$(document).ready(function() {
    // Initialize Select2 for parent account field
    $('#parent_id').select2({
        theme: 'bootstrap-5',
        placeholder: 'ابحث عن الحساب الأب...',
        allowClear: true,
        language: {
            noResults: function() {
                return "لا توجد نتائج";
            },
            searching: function() {
                return "جاري البحث...";
            },
            inputTooShort: function(args) {
                return "أدخل " + (args.minimum - args.input.length) + " أحرف أو أكثر";
            }
        },
        escapeMarkup: function(markup) {
            return markup;
        }
    });

    // Initialize Select2 for cost center field
    $('#cost_center_id').select2({
        theme: 'bootstrap-5',
        placeholder: 'ابحث عن مركز التكلفة...',
        allowClear: true,
        language: {
            noResults: function() {
                return "لا توجد نتائج";
            },
            searching: function() {
                return "جاري البحث...";
            },
            inputTooShort: function(args) {
                return "أدخل " + (args.minimum - args.input.length) + " أحرف أو أكثر";
            }
        },
        escapeMarkup: function(markup) {
            return markup;
        }
    });

    // Enhanced search functionality for account name field
    const nameInput = document.getElementById('name');
    const accountNamesList = document.getElementById('account-names-list');
    
    nameInput.addEventListener('input', function() {
        const query = this.value.toLowerCase();
        if (query.length >= 2) {
            // Filter datalist options based on input
            const options = accountNamesList.querySelectorAll('option');
            options.forEach(option => {
                const text = option.textContent.toLowerCase();
                option.style.display = text.includes(query) ? 'block' : 'none';
            });
        }
    });

    // Add autocomplete styling and behavior
    nameInput.addEventListener('focus', function() {
        this.setAttribute('placeholder', 'ابدأ بالكتابة للبحث في أسماء الحسابات...');
    });

    nameInput.addEventListener('blur', function() {
        this.setAttribute('placeholder', 'اسم الحساب');
    });

    // إدارة إعدادات الضريبة
    const appliesTaxCheckbox = document.getElementById('applies_tax');
    const taxSettingsSection = document.getElementById('tax_settings_section');
    const defaultTaxRateSection = document.getElementById('default_tax_rate_section');
    const taxExamples = document.getElementById('tax_examples');
    const taxCalculationMethod = document.getElementById('tax_calculation_method');
    const defaultTaxRate = document.getElementById('default_tax_rate');

    // إظهار/إخفاء إعدادات الضريبة
    appliesTaxCheckbox.addEventListener('change', function() {
        if (this.checked) {
            taxSettingsSection.style.display = 'block';
            defaultTaxRateSection.style.display = 'block';
            taxExamples.style.display = 'block';
            taxCalculationMethod.required = true;
        } else {
            taxSettingsSection.style.display = 'none';
            defaultTaxRateSection.style.display = 'none';
            taxExamples.style.display = 'none';
            taxCalculationMethod.required = false;
        }
    });

    // تحديث المثال التوضيحي
    function updateTaxExample() {
        const method = taxCalculationMethod.value;
        const rate = parseFloat(defaultTaxRate.value) || 15;
        const exclusiveExample = document.getElementById('exclusive_example');
        const inclusiveExample = document.getElementById('inclusive_example');
        
        if (method === 'exclusive') {
            const taxAmount = (100 * rate) / 100;
            const total = 100 + taxAmount;
            exclusiveExample.innerHTML = `غير شامل: 100 + ${taxAmount.toFixed(2)} = ${total.toFixed(2)} ر.س`;
            exclusiveExample.style.display = 'list-item';
            inclusiveExample.style.display = 'none';
        } else {
            const exclusiveAmount = 100 / (1 + (rate / 100));
            const taxAmount = 100 - exclusiveAmount;
            inclusiveExample.innerHTML = `شامل: ${exclusiveAmount.toFixed(2)} + ${taxAmount.toFixed(2)} = 100 ر.س`;
            inclusiveExample.style.display = 'list-item';
            exclusiveExample.style.display = 'none';
        }
    }

    // تحديث المثال عند تغيير الطريقة أو المعدل
    taxCalculationMethod.addEventListener('change', updateTaxExample);
    defaultTaxRate.addEventListener('input', updateTaxExample);

    // تهيئة أولية للضريبة
    appliesTaxCheckbox.dispatchEvent(new Event('change'));
    updateTaxExample();
});
</script>
@endpush