/**
 * نظام نقطة البيع المتطور - JavaScript المتقدم
 * =====================================================
 */

// متغيرات النظام الأساسية
let orderItems = [];
let currentOrderId = null;
let subtotal = 0;
let taxRate = 0.15; // 15% ضريبة القيمة المضافة
let discount = 0;
let selectedPaymentMethod = 'cash';

// إعدادات النظام
const POS_CONFIG = {
    currency: 'ر.س',
    taxRate: 0.15,
    defaultPaymentMethod: 'cash',
    autoSaveInterval: 30000, // حفظ تلقائي كل 30 ثانية
    animationDuration: 300,
    searchDelay: 500,
    maxRecentProducts: 10
};

// حالة النظام
const POS_STATE = {
    isProcessing: false,
    isOffline: false,
    lastSaveTime: null,
    searchTimeout: null,
    activeCategory: 'all'
};

/**
 * =====================================================
 * تهيئة النظام عند تحميل الصفحة
 * =====================================================
 */
document.addEventListener('DOMContentLoaded', function() {
    initializePOS();
    setupEventListeners();
    loadPersistedData();
    startAutoSave();
    initializeKeyboardShortcuts();
    
    console.log('🚀 نظام نقطة البيع المتطور جاهز للاستخدام');
});

/**
 * تهيئة النظام الأساسي
 */
function initializePOS() {
    // تهيئة عداد العناصر
    updateItemsCount();
    
    // تهيئة حالة الطلب الفارغ
    showEmptyOrderState();
    
    // تهيئة طريقة الدفع الافتراضية
    selectPaymentMethod(POS_CONFIG.defaultPaymentMethod);
    
    // تهيئة البحث المتقدم
    initializeAdvancedSearch();
    
    // تهيئة الفئات
    initializeCategories();
    
    // تحديث العرض
    calculateTotals();
}

/**
 * =====================================================
 * إدارة المنتجات والفئات
 * =====================================================
 */

/**
 * تهيئة البحث المتقدم
 */
function initializeAdvancedSearch() {
    const searchInput = document.querySelector('.search-input');
    if (!searchInput) return;
    
    searchInput.addEventListener('input', function(e) {
        clearTimeout(POS_STATE.searchTimeout);
        const query = e.target.value.trim();
        
        POS_STATE.searchTimeout = setTimeout(() => {
            performAdvancedSearch(query);
        }, POS_CONFIG.searchDelay);
    });
}

/**
 * تنفيذ البحث المتقدم
 */
function performAdvancedSearch(query) {
    const productCards = document.querySelectorAll('.product-card-modern');
    
    if (!query) {
        // إظهار جميع المنتجات
        productCards.forEach(card => {
            card.style.display = 'block';
            animateCardIn(card);
        });
        return;
    }
    
    let foundCount = 0;
    
    productCards.forEach(card => {
        const productName = card.querySelector('.product-title')?.textContent.toLowerCase();
        const productCategory = card.querySelector('.product-category')?.textContent.toLowerCase();
        const productPrice = card.querySelector('.product-price')?.textContent.toLowerCase();
        
        const searchTerms = query.toLowerCase();
        const isMatch = productName?.includes(searchTerms) ||
                       productCategory?.includes(searchTerms) ||
                       productPrice?.includes(searchTerms);
        
        if (isMatch) {
            card.style.display = 'block';
            animateCardIn(card);
            foundCount++;
        } else {
            animateCardOut(card);
        }
    });
    
    // عرض رسالة عدم وجود نتائج
    showSearchResults(foundCount, query);
}

/**
 * مسح البحث
 */
function clearSearch() {
    const searchInput = document.querySelector('.search-input');
    if (searchInput) {
        searchInput.value = '';
        performAdvancedSearch('');
    }
}

/**
 * تهيئة الفئات
 */
function initializeCategories() {
    const categoryTabs = document.querySelectorAll('.category-tab');
    
    categoryTabs.forEach(tab => {
        tab.addEventListener('click', function() {
            const category = this.dataset.category;
            selectCategory(category);
        });
    });
}

/**
 * اختيار فئة
 */
function selectCategory(category) {
    // تحديث الفئة النشطة
    POS_STATE.activeCategory = category;
    
    // تحديث مظهر التبويبات
    document.querySelectorAll('.category-tab').forEach(tab => {
        tab.classList.remove('active');
    });
    
    const activeTab = document.querySelector(`[data-category="${category}"]`);
    if (activeTab) {
        activeTab.classList.add('active');
    }
    
    // تصفية المنتجات
    filterProductsByCategory(category);
}

/**
 * تصفية المنتجات حسب الفئة
 */
function filterProductsByCategory(category) {
    const productCards = document.querySelectorAll('.product-card-modern');
    
    productCards.forEach(card => {
        const productCategory = card.dataset.category;
        
        if (category === 'all' || productCategory === category) {
            card.style.display = 'block';
            animateCardIn(card);
        } else {
            animateCardOut(card);
        }
    });
}

/**
 * =====================================================
 * إدارة الطلبات
 * =====================================================
 */

/**
 * إضافة منتج إلى الطلب
 */
function addProductToOrder(productData) {
    if (POS_STATE.isProcessing) return;
    
    const existingItem = orderItems.find(item => item.id === productData.id);
    
    if (existingItem) {
        existingItem.quantity += 1;
        existingItem.total = existingItem.quantity * existingItem.price;
    } else {
        orderItems.push({
            id: productData.id,
            name: productData.name,
            price: parseFloat(productData.price),
            quantity: 1,
            total: parseFloat(productData.price),
            image: productData.image,
            category: productData.category
        });
    }
    
    // تحديث العرض
    updateOrderDisplay();
    calculateTotals();
    showSuccessAnimation(productData.name);
    
    // حفظ في التخزين المحلي
    saveOrderToLocalStorage();
}

/**
 * إزالة منتج من الطلب
 */
function removeProductFromOrder(productId) {
    const itemIndex = orderItems.findIndex(item => item.id === productId);
    
    if (itemIndex > -1) {
        const item = orderItems[itemIndex];
        
        // تأثير بصري للإزالة
        animateItemRemoval(productId);
        
        setTimeout(() => {
            orderItems.splice(itemIndex, 1);
            updateOrderDisplay();
            calculateTotals();
            saveOrderToLocalStorage();
        }, POS_CONFIG.animationDuration);
    }
}

/**
 * تحديث كمية منتج
 */
function updateProductQuantity(productId, newQuantity) {
    const item = orderItems.find(item => item.id === productId);
    
    if (item && newQuantity > 0) {
        item.quantity = parseInt(newQuantity);
        item.total = item.quantity * item.price;
        
        updateOrderDisplay();
        calculateTotals();
        saveOrderToLocalStorage();
    } else if (newQuantity <= 0) {
        removeProductFromOrder(productId);
    }
}

/**
 * تحديث عرض الطلب
 */
function updateOrderDisplay() {
    const orderContainer = document.getElementById('order-items');
    
    if (orderItems.length === 0) {
        showEmptyOrderState();
        return;
    }
    
    let orderHTML = '<div class="order-items-list">';
    
    orderItems.forEach(item => {
        orderHTML += `
            <div class="order-item-modern" data-id="${item.id}">
                <div class="item-image">
                    <img src="${item.image || '/images/products/default.jpg'}" alt="${item.name}">
                </div>
                <div class="item-details">
                    <h6 class="item-name">${item.name}</h6>
                    <div class="item-meta">
                        <span class="item-price">${item.price.toFixed(2)} ${POS_CONFIG.currency}</span>
                        <span class="item-category">${item.category}</span>
                    </div>
                </div>
                <div class="item-controls">
                    <div class="quantity-controls">
                        <button type="button" class="qty-btn minus" onclick="updateProductQuantity('${item.id}', ${item.quantity - 1})">
                            <i class="fas fa-minus"></i>
                        </button>
                        <input type="number" class="qty-input" value="${item.quantity}" 
                               onchange="updateProductQuantity('${item.id}', this.value)" min="1">
                        <button type="button" class="qty-btn plus" onclick="updateProductQuantity('${item.id}', ${item.quantity + 1})">
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                    <div class="item-total">${item.total.toFixed(2)} ${POS_CONFIG.currency}</div>
                    <button type="button" class="remove-item-btn" onclick="removeProductFromOrder('${item.id}')" title="إزالة العنصر">
                        <i class="fas fa-trash-alt"></i>
                    </button>
                </div>
            </div>
        `;
    });
    
    orderHTML += '</div>';
    orderContainer.innerHTML = orderHTML;
    
    // تحديث عداد العناصر
    updateItemsCount();
}

/**
 * إظهار حالة الطلب الفارغ
 */
function showEmptyOrderState() {
    const orderContainer = document.getElementById('order-items');
    const emptyStateHTML = `
        <div class="empty-order-state">
            <div class="empty-order-icon">
                <i class="fas fa-shopping-cart"></i>
            </div>
            <h4 class="empty-order-title">الطلب فارغ</h4>
            <p class="empty-order-text">اختر المنتجات من القائمة لإضافتها إلى الطلب</p>
            <div class="empty-order-tips">
                <div class="tip-item">
                    <i class="fas fa-mouse-pointer me-2"></i>
                    <span>انقر على أي منتج لإضافته</span>
                </div>
                <div class="tip-item">
                    <i class="fas fa-search me-2"></i>
                    <span>استخدم البحث للعثور على المنتجات</span>
                </div>
            </div>
        </div>
    `;
    
    orderContainer.innerHTML = emptyStateHTML;
    updateItemsCount();
}

/**
 * تحديث عداد العناصر
 */
function updateItemsCount() {
    const countElement = document.getElementById('items-count');
    const totalItems = orderItems.reduce((sum, item) => sum + item.quantity, 0);
    
    if (countElement) {
        countElement.textContent = `${totalItems} عنصر`;
    }
}

/**
 * مسح الطلب
 */
function clearOrder() {
    if (orderItems.length === 0) return;
    
    // تأكيد من المستخدم
    if (confirm('هل أنت متأكد من رغبتك في مسح جميع عناصر الطلب؟')) {
        orderItems = [];
        updateOrderDisplay();
        calculateTotals();
        saveOrderToLocalStorage();
        
        showNotification('تم مسح الطلب بنجاح', 'success');
    }
}

/**
 * =====================================================
 * حسابات المبالغ
 * =====================================================
 */

/**
 * حساب المجاميع
 */
function calculateTotals() {
    // حساب المجموع الفرعي
    subtotal = orderItems.reduce((sum, item) => sum + item.total, 0);
    
    // حساب الخصم
    const discountInput = document.querySelector('input[name="discount_amount"]');
    discount = discountInput ? parseFloat(discountInput.value) || 0 : 0;
    
    // حساب المجموع بعد الخصم
    const afterDiscount = subtotal - discount;
    
    // حساب الضريبة
    const tax = afterDiscount * POS_CONFIG.taxRate;
    
    // حساب المجموع الإجمالي
    const total = afterDiscount + tax;
    
    // تحديث العرض
    updateTotalsDisplay(subtotal, discount, tax, total);
}

/**
 * تحديث عرض المجاميع
 */
function updateTotalsDisplay(subtotalAmount, discountAmount, taxAmount, totalAmount) {
    const elements = {
        subtotal: document.getElementById('subtotal-display'),
        tax: document.getElementById('tax-total-display'),
        total: document.getElementById('total-display')
    };
    
    if (elements.subtotal) {
        elements.subtotal.textContent = `${subtotalAmount.toFixed(2)} ${POS_CONFIG.currency}`;
    }
    
    if (elements.tax) {
        elements.tax.textContent = `${taxAmount.toFixed(2)} ${POS_CONFIG.currency}`;
    }
    
    if (elements.total) {
        elements.total.textContent = `${totalAmount.toFixed(2)} ${POS_CONFIG.currency}`;
    }
    
    // تحديث حقول النموذج المخفية
    updateHiddenFormFields(totalAmount);
}

/**
 * تحديث حقول النموذج المخفية
 */
function updateHiddenFormFields(totalAmount) {
    const paidAmountField = document.querySelector('input[name="paid_amount"]');
    if (paidAmountField) {
        paidAmountField.value = totalAmount.toFixed(2);
    }
}

/**
 * =====================================================
 * طرق الدفع
 * =====================================================
 */

/**
 * اختيار طريقة الدفع
 */
function selectPaymentMethod(method) {
    selectedPaymentMethod = method;
    
    // تحديث المظهر
    document.querySelectorAll('.payment-option').forEach(option => {
        option.classList.remove('active');
    });
    
    const selectedOption = document.querySelector(`[data-method="${method}"]`);
    if (selectedOption) {
        selectedOption.classList.add('active');
    }
    
    // تحديث حقل النموذج المخفي
    const paymentMethodField = document.querySelector('input[name="payment_method"]');
    if (paymentMethodField) {
        paymentMethodField.value = method;
    }
    
    showNotification(`تم اختيار طريقة الدفع: ${getPaymentMethodName(method)}`, 'info');
}

/**
 * الحصول على اسم طريقة الدفع
 */
function getPaymentMethodName(method) {
    const names = {
        'cash': 'نقداً',
        'credit_card': 'البطاقة الائتمانية',
        'bank_transfer': 'تحويل بنكي'
    };
    return names[method] || method;
}

/**
 * =====================================================
 * معالجة الطلبات
 * =====================================================
 */

/**
 * دفع سريع
 */
function quickPay(type = 'full') {
    if (orderItems.length === 0) {
        showNotification('لا يوجد عناصر في الطلب', 'warning');
        return;
    }
    
    if (POS_STATE.isProcessing) return;
    
    POS_STATE.isProcessing = true;
    
    const button = document.getElementById(`${type === 'full' ? 'full' : 'partial'}-pay-btn`);
    showButtonLoading(button, true);
    
    // محاكاة معالجة الدفع
    setTimeout(() => {
        if (type === 'full') {
            processFullPayment();
        } else {
            processPartialPayment();
        }
        
        showButtonLoading(button, false);
        POS_STATE.isProcessing = false;
    }, 2000);
}

/**
 * معالجة الدفع الكامل
 */
function processFullPayment() {
    // إرسال البيانات إلى الخادم
    const formData = prepareOrderData();
    
    // محاكاة إرسال البيانات
    console.log('إرسال بيانات الطلب:', formData);
    
    // إظهار رسالة نجاح
    showNotification('تم إنجاز الطلب بنجاح! جاري التحضير للطباعة...', 'success');
    
    // مسح الطلب
    setTimeout(() => {
        clearOrderAfterSuccess();
        // هنا يمكن إضافة كود الطباعة
        printInvoice();
    }, 1500);
}

/**
 * معالجة الدفع الجزئي
 */
function processPartialPayment() {
    // فتح نافذة للدفع الجزئي
    openPartialPaymentModal();
}

/**
 * حفظ الطلب فقط
 */
function saveOrder() {
    if (orderItems.length === 0) {
        showNotification('لا يوجد عناصر في الطلب للحفظ', 'warning');
        return;
    }
    
    if (POS_STATE.isProcessing) return;
    
    POS_STATE.isProcessing = true;
    
    const button = document.getElementById('save-only-btn');
    showButtonLoading(button, true);
    
    // محاكاة حفظ البيانات
    setTimeout(() => {
        const formData = prepareOrderData();
        console.log('حفظ بيانات الطلب:', formData);
        
        showNotification('تم حفظ الطلب بنجاح', 'success');
        POS_STATE.lastSaveTime = new Date();
        
        showButtonLoading(button, false);
        POS_STATE.isProcessing = false;
    }, 1000);
}

/**
 * تحضير بيانات الطلب
 */
function prepareOrderData() {
    return {
        items: orderItems,
        subtotal: subtotal,
        discount: discount,
        tax: subtotal * POS_CONFIG.taxRate,
        total: subtotal - discount + (subtotal * POS_CONFIG.taxRate),
        payment_method: selectedPaymentMethod,
        timestamp: new Date().toISOString()
    };
}

/**
 * مسح الطلب بعد النجاح
 */
function clearOrderAfterSuccess() {
    orderItems = [];
    discount = 0;
    updateOrderDisplay();
    calculateTotals();
    clearLocalStorage();
}

/**
 * =====================================================
 * تأثيرات بصرية وانيميشن
 * =====================================================
 */

/**
 * تأثير ظهور البطاقة
 */
function animateCardIn(card) {
    card.style.opacity = '0';
    card.style.transform = 'translateY(20px)';
    
    setTimeout(() => {
        card.style.transition = 'all 0.3s ease';
        card.style.opacity = '1';
        card.style.transform = 'translateY(0)';
    }, 50);
}

/**
 * تأثير اختفاء البطاقة
 */
function animateCardOut(card) {
    card.style.transition = 'all 0.3s ease';
    card.style.opacity = '0';
    card.style.transform = 'translateY(-20px)';
    
    setTimeout(() => {
        card.style.display = 'none';
    }, POS_CONFIG.animationDuration);
}

/**
 * تأثير إزالة العنصر
 */
function animateItemRemoval(itemId) {
    const itemElement = document.querySelector(`[data-id="${itemId}"]`);
    if (itemElement) {
        itemElement.style.transition = 'all 0.3s ease';
        itemElement.style.opacity = '0';
        itemElement.style.transform = 'translateX(100%)';
    }
}

/**
 * إظهار تأثير النجاح
 */
function showSuccessAnimation(productName) {
    const notification = document.createElement('div');
    notification.className = 'success-animation';
    notification.innerHTML = `
        <i class="fas fa-check-circle"></i>
        <span>تم إضافة ${productName}</span>
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.classList.add('show');
    }, 100);
    
    setTimeout(() => {
        notification.classList.remove('show');
        setTimeout(() => {
            document.body.removeChild(notification);
        }, 300);
    }, 2000);
}

/**
 * إظهار حالة التحميل للأزرار
 */
function showButtonLoading(button, isLoading) {
    if (!button) return;
    
    const textElement = button.querySelector('.btn-text');
    const loadingElement = button.querySelector('.btn-loading');
    
    if (isLoading) {
        if (textElement) textElement.style.display = 'none';
        if (loadingElement) loadingElement.style.display = 'block';
        button.disabled = true;
    } else {
        if (textElement) textElement.style.display = 'block';
        if (loadingElement) loadingElement.style.display = 'none';
        button.disabled = false;
    }
}

/**
 * =====================================================
 * إشعارات النظام
 * =====================================================
 */

/**
 * إظهار إشعار
 */
function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `pos-notification ${type}`;
    
    const icons = {
        'success': 'fas fa-check-circle',
        'error': 'fas fa-exclamation-circle',
        'warning': 'fas fa-exclamation-triangle',
        'info': 'fas fa-info-circle'
    };
    
    notification.innerHTML = `
        <i class="${icons[type] || icons.info}"></i>
        <span>${message}</span>
        <button type="button" class="close-notification" onclick="this.parentElement.remove()">
            <i class="fas fa-times"></i>
        </button>
    `;
    
    document.body.appendChild(notification);
    
    // إظهار الإشعار
    setTimeout(() => {
        notification.classList.add('show');
    }, 100);
    
    // إخفاء الإشعار تلقائياً
    setTimeout(() => {
        if (notification.parentElement) {
            notification.classList.remove('show');
            setTimeout(() => {
                if (notification.parentElement) {
                    notification.remove();
                }
            }, 300);
        }
    }, 5000);
}

/**
 * =====================================================
 * التخزين المحلي والحفظ التلقائي
 * =====================================================
 */

/**
 * حفظ الطلب في التخزين المحلي
 */
function saveOrderToLocalStorage() {
    const orderData = {
        items: orderItems,
        discount: discount,
        paymentMethod: selectedPaymentMethod,
        timestamp: new Date().toISOString()
    };
    
    localStorage.setItem('pos_current_order', JSON.stringify(orderData));
}

/**
 * تحميل البيانات المحفوظة
 */
function loadPersistedData() {
    const savedOrder = localStorage.getItem('pos_current_order');
    
    if (savedOrder) {
        try {
            const orderData = JSON.parse(savedOrder);
            
            // التحقق من عمر البيانات (لا تحمل إذا كانت أقدم من يوم)
            const savedTime = new Date(orderData.timestamp);
            const hoursDiff = (new Date() - savedTime) / (1000 * 60 * 60);
            
            if (hoursDiff < 24) {
                orderItems = orderData.items || [];
                discount = orderData.discount || 0;
                selectedPaymentMethod = orderData.paymentMethod || 'cash';
                
                updateOrderDisplay();
                selectPaymentMethod(selectedPaymentMethod);
                calculateTotals();
                
                showNotification('تم استرداد طلب محفوظ مسبقاً', 'info');
            }
        } catch (error) {
            console.error('خطأ في تحميل البيانات المحفوظة:', error);
        }
    }
}

/**
 * مسح التخزين المحلي
 */
function clearLocalStorage() {
    localStorage.removeItem('pos_current_order');
}

/**
 * بدء الحفظ التلقائي
 */
function startAutoSave() {
    setInterval(() => {
        if (orderItems.length > 0) {
            saveOrderToLocalStorage();
        }
    }, POS_CONFIG.autoSaveInterval);
}

/**
 * =====================================================
 * اختصارات لوحة المفاتيح
 * =====================================================
 */

/**
 * تهيئة اختصارات لوحة المفاتيح
 */
function initializeKeyboardShortcuts() {
    document.addEventListener('keydown', function(e) {
        // منع التنفيذ إذا كان المستخدم يكتب في حقل إدخال
        if (e.target.tagName === 'INPUT' || e.target.tagName === 'TEXTAREA') {
            return;
        }
        
        // اختصارات لوحة المفاتيح
        switch(e.key) {
            case 'F1':
                e.preventDefault();
                openCalculator();
                break;
            case 'F2':
                e.preventDefault();
                clearOrder();
                break;
            case 'F3':
                e.preventDefault();
                document.querySelector('.search-input')?.focus();
                break;
            case 'F4':
                e.preventDefault();
                if (orderItems.length > 0) {
                    quickPay('full');
                }
                break;
            case 'Escape':
                clearSearch();
                break;
        }
        
        // اختصارات مع Ctrl
        if (e.ctrlKey) {
            switch(e.key) {
                case 's':
                    e.preventDefault();
                    saveOrder();
                    break;
                case 'p':
                    e.preventDefault();
                    if (orderItems.length > 0) {
                        quickPay('full');
                    }
                    break;
            }
        }
    });
}

/**
 * =====================================================
 * وظائف إضافية
 * =====================================================
 */

/**
 * فتح الآلة الحاسبة
 */
function openCalculator() {
    // يمكن تطوير آلة حاسبة مدمجة أو فتح آلة النظام
    showNotification('فتح الآلة الحاسبة...', 'info');
}

/**
 * طباعة الفاتورة
 */
function printInvoice() {
    showNotification('جاري تحضير الفاتورة للطباعة...', 'info');
    // هنا يمكن إضافة كود الطباعة الفعلي
}

/**
 * تعليق الطلب
 */
function holdOrder() {
    if (orderItems.length === 0) {
        showNotification('لا يوجد طلب لتعليقه', 'warning');
        return;
    }
    
    saveOrderToLocalStorage();
    clearOrderAfterSuccess();
    showNotification('تم تعليق الطلب وحفظه', 'success');
}

/**
 * استدعاء طلب معلق
 */
function loadOrder() {
    loadPersistedData();
}

/**
 * إضافة خصم
 */
function addDiscount() {
    if (orderItems.length === 0) {
        showNotification('يجب إضافة منتجات أولاً', 'warning');
        return;
    }
    
    const discountAmount = prompt('أدخل مبلغ الخصم:');
    if (discountAmount !== null && !isNaN(discountAmount)) {
        const discountInput = document.querySelector('input[name="discount_amount"]');
        if (discountInput) {
            discountInput.value = parseFloat(discountAmount);
            calculateTotals();
            saveOrderToLocalStorage();
        }
    }
}

/**
 * إضافة ملاحظة
 */
function addNote() {
    const note = prompt('أدخل ملاحظة للطلب:');
    if (note) {
        showNotification(`تم إضافة الملاحظة: ${note}`, 'info');
        // يمكن حفظ الملاحظة مع بيانات الطلب
    }
}

/**
 * عرض نتائج البحث
 */
function showSearchResults(count, query) {
    const resultsContainer = document.querySelector('.search-results');
    if (resultsContainer) {
        if (count === 0) {
            resultsContainer.innerHTML = `
                <div class="no-results">
                    <i class="fas fa-search"></i>
                    <p>لم يتم العثور على نتائج لـ "${query}"</p>
                </div>
            `;
        } else {
            resultsContainer.innerHTML = `
                <div class="search-summary">
                    <span>تم العثور على ${count} منتج</span>
                </div>
            `;
        }
    }
}

/**
 * فتح نافذة الدفع الجزئي
 */
function openPartialPaymentModal() {
    showNotification('فتح نافذة الدفع الجزئي...', 'info');
    // يمكن تطوير نافذة منبثقة للدفع الجزئي
}

/**
 * إعداد مستمعي الأحداث
 */
function setupEventListeners() {
    // مستمع للنقر على المنتجات
    document.addEventListener('click', function(e) {
        // النقر على بطاقة المنتج
        if (e.target.closest('.product-card-modern')) {
            const card = e.target.closest('.product-card-modern');
            const productData = {
                id: card.dataset.id,
                name: card.querySelector('.product-title')?.textContent,
                price: card.dataset.price,
                image: card.querySelector('.product-image')?.src,
                category: card.dataset.category
            };
            
            addProductToOrder(productData);
        }
        
        // النقر على زر الإضافة السريعة
        if (e.target.closest('.quick-add-btn')) {
            e.stopPropagation();
            const card = e.target.closest('.product-card-modern');
            const productData = {
                id: card.dataset.id,
                name: card.querySelector('.product-title')?.textContent,
                price: card.dataset.price,
                image: card.querySelector('.product-image')?.src,
                category: card.dataset.category
            };
            
            addProductToOrder(productData);
        }
    });
    
    // مستمع لتغيير قيمة الخصم
    const discountInput = document.querySelector('input[name="discount_amount"]');
    if (discountInput) {
        discountInput.addEventListener('input', function() {
            calculateTotals();
            saveOrderToLocalStorage();
        });
    }
}

// تصدير الوظائف للاستخدام العام
window.POS = {
    addProductToOrder,
    removeProductFromOrder,
    updateProductQuantity,
    selectPaymentMethod,
    quickPay,
    saveOrder,
    clearOrder,
    calculateTotals,
    selectCategory,
    clearSearch,
    addDiscount,
    addNote,
    holdOrder,
    loadOrder,
    openCalculator,
    printInvoice
};