/**
 * تحليلات المدفوعات المتقدمة - JavaScript محسّن
 * Enhanced Payment Analytics Dashboard
 */

$(document).ready(function() {
    // إعداد أساسي للرسوم البيانية
    Chart.defaults.font.family = 'Cairo, -apple-system, BlinkMacSystemFont';
    Chart.defaults.font.size = 12;
    Chart.defaults.plugins.legend.labels.usePointStyle = true;
    Chart.defaults.plugins.legend.labels.pointStyle = 'circle';
    
    // ألوان محسّنة ومتدرجة
    const colorPalette = {
        primary: ['#667eea', '#764ba2'],
        success: ['#11998e', '#38ef7d'],
        warning: ['#f7931e', '#feb47b'],
        danger: ['#ff416c', '#ff4b2b'],
        info: ['#4facfe', '#00f2fe'],
        purple: ['#a8edea', '#fed6e3']
    };
    
    // دالة لإنشاء تدرج لوني
    function createGradient(ctx, color1, color2) {
        const gradient = ctx.createLinearGradient(0, 0, 0, 400);
        gradient.addColorStop(0, color1);
        gradient.addColorStop(1, color2);
        return gradient;
    }
    
    // تحسين الرسوم البيانية
    function enhanceChart(chartId, data, type = 'line') {
        const ctx = document.getElementById(chartId);
        if (!ctx) return;
        
        const chartCtx = ctx.getContext('2d');
        
        // إعدادات مشتركة محسّنة
        const commonOptions = {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                tooltip: {
                    backgroundColor: 'rgba(255, 255, 255, 0.95)',
                    titleColor: '#333',
                    bodyColor: '#666',
                    borderColor: '#e2e8f0',
                    borderWidth: 1,
                    cornerRadius: 12,
                    titleFont: { size: 14, weight: '600' },
                    bodyFont: { size: 13 }
                },
                legend: {
                    position: 'top',
                    labels: {
                        padding: 20,
                        font: { size: 13, weight: '600' }
                    }
                }
            },
            animation: {
                duration: 2000,
                easing: 'easeInOutQuart'
            }
        };
        
        return new Chart(chartCtx, {
            type: type,
            data: data,
            options: commonOptions
        });
    }
    
    // تأثيرات تفاعلية للبطاقات
    function initCardEffects() {
        $('.stat-card').each(function(index) {
            $(this).css('animation-delay', (index * 0.1) + 's');
            
            $(this).hover(
                function() {
                    $(this).css({
                        'transform': 'translateY(-8px) scale(1.02)',
                        'box-shadow': '0 20px 40px rgba(0,0,0,0.15)'
                    });
                },
                function() {
                    $(this).css({
                        'transform': 'translateY(0) scale(1)',
                        'box-shadow': '0 4px 20px rgba(0,0,0,0.1)'
                    });
                }
            );
        });
    }
    
    // تأثيرات الجداول
    function initTableEffects() {
        $('.customer-row').hover(
            function() {
                $(this).find('.customer-avatar').css('transform', 'scale(1.1)');
                $(this).css('background', 'linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%)');
            },
            function() {
                $(this).find('.customer-avatar').css('transform', 'scale(1)');
                $(this).css('background', 'transparent');
            }
        );
    }
    
    // تأثيرات أشرطة التقدم
    function initProgressBars() {
        $('.enhanced-progress .progress-bar').each(function() {
            const width = $(this).attr('style').match(/width:\s*(\d+)%/);
            if (width) {
                const value = parseInt(width[1]);
                $(this).css('width', '0%');
                
                setTimeout(() => {
                    $(this).css({
                        'width': value + '%',
                        'transition': 'width 2s ease-in-out'
                    });
                }, 500);
            }
        });
    }
    
    // مراقب الرؤية للرسوم المتحركة
    function initIntersectionObserver() {
        const observerOptions = {
            threshold: 0.1,
            rootMargin: '0px 0px -50px 0px'
        };
        
        const observer = new IntersectionObserver((entries) => {
            entries.forEach((entry) => {
                if (entry.isIntersecting) {
                    entry.target.classList.add('animate-in');
                    observer.unobserve(entry.target);
                }
            });
        }, observerOptions);
        
        document.querySelectorAll('.fade-in-up, .fade-in-left, .fade-in-right, .stat-card').forEach((el) => {
            observer.observe(el);
        });
    }
    
    // تأثيرات التمرير
    function initScrollEffects() {
        let lastScrollTop = 0;
        const header = $('.top-header');
        
        $(window).scroll(function() {
            const scrollTop = $(this).scrollTop();
            
            if (scrollTop > lastScrollTop && scrollTop > 100) {
                // التمرير لأسفل
                header.css('transform', 'translateY(-100%)');
            } else {
                // التمرير لأعلى
                header.css('transform', 'translateY(0)');
            }
            
            lastScrollTop = scrollTop;
            
            // تأثير Parallax خفيف للبطاقات
            $('.stat-card').each(function() {
                const rate = scrollTop * -0.1;
                $(this).css('background-position', `center ${rate}px`);
            });
        });
    }
    
    // تأثيرات الماوس العامة
    function initMouseEffects() {
        // تأثير المتابعة للماوس على البطاقات الكبيرة
        $('.card').mousemove(function(e) {
            const rect = this.getBoundingClientRect();
            const x = e.clientX - rect.left;
            const y = e.clientY - rect.top;
            
            const centerX = rect.width / 2;
            const centerY = rect.height / 2;
            
            const rotateX = (y - centerY) / 10;
            const rotateY = (centerX - x) / 10;
            
            $(this).css({
                'transform': `perspective(1000px) rotateX(${rotateX}deg) rotateY(${rotateY}deg)`,
                'transition': 'transform 0.1s ease'
            });
        });
        
        $('.card').mouseleave(function() {
            $(this).css({
                'transform': 'perspective(1000px) rotateX(0deg) rotateY(0deg)',
                'transition': 'transform 0.3s ease'
            });
        });
    }
    
    // تحسينات الأداء
    function initPerformanceOptimizations() {
        // تحميل مؤجل للصور
        $('img[data-src]').each(function() {
            const img = $(this);
            const observer = new IntersectionObserver((entries) => {
                if (entries[0].isIntersecting) {
                    img.attr('src', img.attr('data-src'));
                    observer.unobserve(this);
                }
            });
            observer.observe(this);
        });
        
        // تحسين الرسوم البيانية للشاشات الصغيرة
        function optimizeChartsForMobile() {
            if (window.innerWidth < 768) {
                Chart.defaults.font.size = 10;
                Chart.defaults.plugins.legend.display = false;
            }
        }
        
        $(window).resize(optimizeChartsForMobile);
        optimizeChartsForMobile();
    }
    
    // تفعيل الملاحة بلوحة المفاتيح
    function initKeyboardNavigation() {
        $(document).keydown(function(e) {
            // ESC لإغلاق القوائم المنسدلة
            if (e.key === 'Escape') {
                $('.dropdown.show').removeClass('show');
            }
            
            // مفاتيح الأسهم للتنقل بين البطاقات
            if (e.altKey) {
                const cards = $('.stat-card');
                const currentFocus = $('.stat-card:focus');
                let index = cards.index(currentFocus);
                
                switch(e.key) {
                    case 'ArrowRight':
                        index = (index - 1 + cards.length) % cards.length;
                        break;
                    case 'ArrowLeft':
                        index = (index + 1) % cards.length;
                        break;
                }
                
                cards.eq(index).focus();
                e.preventDefault();
            }
        });
    }
    
    // إضافة إمكانية التحديث التلقائي
    function initAutoRefresh() {
        const refreshInterval = 5 * 60 * 1000; // 5 دقائق
        
        setInterval(() => {
            // تحديث البيانات في الخلفية
            fetch(window.location.href + '?ajax=1')
                .then(response => response.json())
                .then(data => {
                    // تحديث البطاقات
                    updateStatsCards(data);
                })
                .catch(console.error);
        }, refreshInterval);
    }
    
    // دالة تحديث بطاقات الإحصائيات
    function updateStatsCards(data) {
        $('.stat-value').each(function(index) {
            const newValue = Object.values(data.totals)[index];
            if (newValue !== undefined) {
                const $this = $(this);
                const currentValue = parseInt($this.text().replace(/[^\d]/g, ''));
                
                // تأثير العد التصاعدي
                $({ counter: currentValue }).animate({ counter: newValue }, {
                    duration: 1000,
                    step: function() {
                        $this.text(Math.floor(this.counter).toLocaleString());
                    },
                    complete: function() {
                        $this.text(newValue.toLocaleString());
                    }
                });
            }
        });
    }
    
    // تهيئة جميع التحسينات
    function initialize() {
        initCardEffects();
        initTableEffects();
        initProgressBars();
        initIntersectionObserver();
        initScrollEffects();
        initMouseEffects();
        initPerformanceOptimizations();
        initKeyboardNavigation();
        
        // تفعيل التحديث التلقائي في بيئة الإنتاج
        if (window.location.hostname !== 'localhost') {
            initAutoRefresh();
        }
        
        console.log('🎨 Payment Analytics Dashboard Enhanced Successfully!');
    }
    
    // بدء التهيئة
    initialize();
    
    // إضافة الأنماط الديناميكية
    const dynamicStyles = `
        .animate-in {
            animation-play-state: running !important;
        }
        
        .stat-card:focus {
            outline: 2px solid #667eea;
            outline-offset: 4px;
        }
        
        .enhanced-progress {
            overflow: hidden;
            position: relative;
        }
        
        .enhanced-progress::after {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.5), transparent);
            animation: progressShine 3s infinite;
        }
        
        @keyframes progressShine {
            0% { left: -100%; }
            100% { left: 100%; }
        }
        
        @media (prefers-reduced-motion: reduce) {
            *, *::before, *::after {
                animation-duration: 0.01ms !important;
                animation-iteration-count: 1 !important;
                transition-duration: 0.01ms !important;
            }
        }
    `;
    
    $('<style>').text(dynamicStyles).appendTo('head');
});