/**
 * Enhanced Dashboard Analytics
 * Modern interactive dashboard with real-time updates and advanced visualizations
 */

class DashboardAnalytics {
    constructor() {
        this.charts = {};
        this.updateInterval = null;
        this.isVisible = true;
        this.init();
    }

    init() {
        this.setupEventListeners();
        this.initializeCharts();
        this.startRealTimeUpdates();
        this.setupAnimations();
        this.handleVisibilityChange();
    }

    setupEventListeners() {
        // Chart period selector
        const periodSelector = document.getElementById('chartPeriod');
        if (periodSelector) {
            periodSelector.addEventListener('change', (e) => {
                this.updateChartPeriod(e.target.value);
            });
        }

        // Chart control buttons
        const controlButtons = document.querySelectorAll('.btn-chart-control');
        controlButtons.forEach(button => {
            button.addEventListener('click', (e) => {
                this.switchChartMetric(e.target.dataset.metric);
                this.updateActiveButton(e.target);
            });
        });

        // Invoice row clicks
        const invoiceRows = document.querySelectorAll('.invoice-row');
        invoiceRows.forEach(row => {
            row.addEventListener('click', (e) => {
                if (!e.target.closest('.action-btn')) {
                    const invoiceId = row.dataset.invoiceId;
                    window.location.href = `/invoices/${invoiceId}`;
                }
            });
        });

        // Download buttons
        const downloadButtons = document.querySelectorAll('.download-btn');
        downloadButtons.forEach(button => {
            button.addEventListener('click', (e) => {
                e.preventDefault();
                e.stopPropagation();
                this.downloadInvoice(e);
            });
        });
    }

    initializeCharts() {
        this.createSalesChart();
        this.createZakatChart();
        this.createInvoiceTypesChart();
        this.createPerformanceChart();
        this.createHeroMiniChart();
    }

    createSalesChart() {
        const ctx = document.getElementById('salesPurchasesChart');
        if (!ctx) return;

        const chartConfig = {
            type: 'line',
            data: {
                labels: window.dashboardData?.monthlyLabels || ['يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو'],
                datasets: [{
                    label: 'المبيعات',
                    data: window.dashboardData?.monthlyRevenue || [50000, 75000, 60000, 90000, 80000, 110000],
                    borderColor: '#6366f1',
                    backgroundColor: 'rgba(99, 102, 241, 0.1)',
                    borderWidth: 4,
                    fill: true,
                    tension: 0.4,
                    pointBackgroundColor: '#6366f1',
                    pointBorderColor: '#ffffff',
                    pointBorderWidth: 3,
                    pointRadius: 8,
                    pointHoverRadius: 12
                }, {
                    label: 'المشتريات',
                    data: window.dashboardData?.monthlyPurchases || [30000, 45000, 40000, 55000, 50000, 65000],
                    borderColor: '#06b6d4',
                    backgroundColor: 'rgba(6, 182, 212, 0.1)',
                    borderWidth: 4,
                    fill: true,
                    tension: 0.4,
                    pointBackgroundColor: '#06b6d4',
                    pointBorderColor: '#ffffff',
                    pointBorderWidth: 3,
                    pointRadius: 8,
                    pointHoverRadius: 12
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { display: false },
                    tooltip: {
                        backgroundColor: 'rgba(255, 255, 255, 0.98)',
                        titleColor: '#1f2937',
                        bodyColor: '#1f2937',
                        borderColor: 'rgba(99, 102, 241, 0.2)',
                        borderWidth: 1,
                        cornerRadius: 12,
                        padding: 16,
                        titleFont: { weight: 600, size: 14 },
                        bodyFont: { weight: 500, size: 13 },
                        callbacks: {
                            label: (context) => {
                                const formatter = new Intl.NumberFormat('ar-SA', {
                                    style: 'currency',
                                    currency: 'SAR',
                                    minimumFractionDigits: 0
                                });
                                return `${context.dataset.label}: ${formatter.format(context.parsed.y)}`;
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(0, 0, 0, 0.05)',
                            drawBorder: false
                        },
                        ticks: {
                            color: '#6b7280',
                            font: { weight: 500 },
                            padding: 12,
                            callback: (value) => {
                                const formatter = new Intl.NumberFormat('ar-SA', {
                                    style: 'currency',
                                    currency: 'SAR',
                                    minimumFractionDigits: 0,
                                    maximumFractionDigits: 0
                                });
                                return formatter.format(value);
                            }
                        }
                    },
                    x: {
                        grid: { display: false, drawBorder: false },
                        ticks: {
                            color: '#6b7280',
                            font: { weight: 500 },
                            padding: 12
                        }
                    }
                },
                interaction: { intersect: false, mode: 'index' },
                animation: {
                    duration: 2000,
                    easing: 'easeInOutQuart',
                    onComplete: () => {
                        this.addChartInteractions('salesPurchasesChart');
                    }
                }
            }
        };

        this.charts.salesChart = new Chart(ctx, chartConfig);
    }

    createZakatChart() {
        const ctx = document.getElementById('zakatChart');
        if (!ctx) return;

        const paidCount = window.dashboardData?.paidZakatInvoices || 15;
        const pendingCount = window.dashboardData?.pendingZakatInvoices || 5;

        this.charts.zakatChart = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: ['مدفوعة', 'معلقة'],
                datasets: [{
                    data: [paidCount, pendingCount],
                    backgroundColor: ['#10b981', '#f59e0b'],
                    borderWidth: 0,
                    cutout: '70%'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { display: false },
                    tooltip: {
                        backgroundColor: 'rgba(255, 255, 255, 0.98)',
                        titleColor: '#1f2937',
                        bodyColor: '#1f2937',
                        borderColor: 'rgba(0, 0, 0, 0.1)',
                        borderWidth: 1,
                        cornerRadius: 8,
                        padding: 12,
                        callbacks: {
                            label: (context) => {
                                const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                const percentage = ((context.parsed * 100) / total).toFixed(1);
                                return `${context.label}: ${context.parsed} (${percentage}%)`;
                            }
                        }
                    }
                },
                animation: { duration: 1500, easing: 'easeInOutQuart' }
            }
        });
    }

    createInvoiceTypesChart() {
        const ctx = document.getElementById('invoiceTypesChart');
        if (!ctx) return;

        const standardCount = window.dashboardData?.standardInvoicesCount || 25;
        const simplifiedCount = window.dashboardData?.simplifiedInvoicesCount || 15;

        this.charts.invoiceTypesChart = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: ['فواتير قياسية', 'فواتير مبسطة'],
                datasets: [{
                    data: [standardCount, simplifiedCount],
                    backgroundColor: ['#6366f1', '#06b6d4'],
                    borderWidth: 0,
                    cutout: '60%'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { display: false },
                    tooltip: {
                        backgroundColor: 'rgba(255, 255, 255, 0.98)',
                        titleColor: '#1f2937',
                        bodyColor: '#1f2937',
                        borderColor: 'rgba(0, 0, 0, 0.1)',
                        borderWidth: 1,
                        cornerRadius: 8,
                        padding: 12
                    }
                },
                animation: { duration: 1500, easing: 'easeInOutQuart' }
            }
        });
    }

    createPerformanceChart() {
        const ctx = document.getElementById('performanceChart');
        if (!ctx) return;

        this.charts.performanceChart = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: window.dashboardData?.monthlyLabels || ['يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو'],
                datasets: [{
                    label: 'الإيرادات',
                    data: window.dashboardData?.monthlyRevenue || [50000, 75000, 60000, 90000, 80000, 110000],
                    backgroundColor: 'rgba(99, 102, 241, 0.1)',
                    borderColor: '#6366f1',
                    borderWidth: 2,
                    borderRadius: 8,
                    borderSkipped: false
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { display: false },
                    tooltip: {
                        backgroundColor: 'rgba(255, 255, 255, 0.98)',
                        titleColor: '#1f2937',
                        bodyColor: '#1f2937',
                        borderColor: 'rgba(99, 102, 241, 0.2)',
                        borderWidth: 1,
                        cornerRadius: 12,
                        padding: 12,
                        callbacks: {
                            label: (context) => {
                                const formatter = new Intl.NumberFormat('ar-SA', {
                                    style: 'currency',
                                    currency: 'SAR',
                                    minimumFractionDigits: 0
                                });
                                return `الإيرادات: ${formatter.format(context.parsed.y)}`;
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(0, 0, 0, 0.05)',
                            drawBorder: false
                        },
                        ticks: {
                            color: '#6b7280',
                            font: { weight: 500 }
                        }
                    },
                    x: {
                        grid: { display: false, drawBorder: false },
                        ticks: {
                            color: '#6b7280',
                            font: { weight: 500 }
                        }
                    }
                },
                animation: { duration: 1500, easing: 'easeInOutQuart' }
            }
        });
    }

    createHeroMiniChart() {
        const ctx = document.getElementById('heroMiniChart');
        if (!ctx) return;

        const revenueData = window.dashboardData?.monthlyRevenue || [50000, 75000, 60000, 90000, 80000, 110000];
        const labels = window.dashboardData?.monthlyLabels || ['يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو'];

        this.charts.heroChart = new Chart(ctx, {
            type: 'line',
            data: {
                labels: labels.slice(-4),
                datasets: [{
                    data: revenueData.slice(-4),
                    borderColor: 'rgba(255, 255, 255, 0.8)',
                    backgroundColor: 'rgba(255, 255, 255, 0.1)',
                    borderWidth: 2,
                    fill: true,
                    tension: 0.4,
                    pointRadius: 0,
                    pointHoverRadius: 0
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: { legend: { display: false } },
                scales: {
                    x: { display: false },
                    y: { display: false }
                },
                interaction: { intersect: false },
                animation: { duration: 2000, easing: 'easeInOutQuart' }
            }
        });
    }

    addChartInteractions(chartId) {
        const chart = this.charts[Object.keys(this.charts).find(key => 
            this.charts[key].canvas.id === chartId
        )];

        if (!chart) return;

        chart.canvas.addEventListener('mousemove', (e) => {
            const canvasPosition = Chart.helpers.getRelativePosition(e, chart);
            const dataX = chart.scales.x.getValueForPixel(canvasPosition.x);
            const dataY = chart.scales.y.getValueForPixel(canvasPosition.y);
            
            // Custom tooltip logic here
            this.showCustomTooltip(e, chart, dataX, dataY);
        });

        chart.canvas.addEventListener('mouseleave', () => {
            this.hideCustomTooltip();
        });
    }

    showCustomTooltip(event, chart, dataX, dataY) {
        // Implementation for custom tooltips
        const tooltip = document.querySelector('.custom-tooltip') || this.createTooltipElement();
        tooltip.classList.add('show');
        tooltip.style.left = event.pageX + 10 + 'px';
        tooltip.style.top = event.pageY - 10 + 'px';
    }

    hideCustomTooltip() {
        const tooltip = document.querySelector('.custom-tooltip');
        if (tooltip) {
            tooltip.classList.remove('show');
        }
    }

    createTooltipElement() {
        const tooltip = document.createElement('div');
        tooltip.className = 'custom-tooltip';
        document.body.appendChild(tooltip);
        return tooltip;
    }

    updateChartPeriod(period) {
        // Logic to update chart data based on selected period
        console.log('Updating chart period to:', period);
        
        // Simulate API call
        this.fetchChartData(period).then(data => {
            this.updateChartsData(data);
        });
    }

    async fetchChartData(period) {
        // Simulate API call
        return new Promise(resolve => {
            setTimeout(() => {
                resolve({
                    labels: ['يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو'],
                    revenue: [Math.random() * 100000, Math.random() * 100000, Math.random() * 100000, 
                             Math.random() * 100000, Math.random() * 100000, Math.random() * 100000],
                    purchases: [Math.random() * 60000, Math.random() * 60000, Math.random() * 60000,
                               Math.random() * 60000, Math.random() * 60000, Math.random() * 60000]
                });
            }, 500);
        });
    }

    updateChartsData(data) {
        if (this.charts.salesChart) {
            this.charts.salesChart.data.labels = data.labels;
            this.charts.salesChart.data.datasets[0].data = data.revenue;
            this.charts.salesChart.data.datasets[1].data = data.purchases;
            this.charts.salesChart.update('active');
        }
    }

    switchChartMetric(metric) {
        console.log('Switching to metric:', metric);
        
        if (this.charts.performanceChart) {
            const newData = metric === 'revenue' ? 
                window.dashboardData?.monthlyRevenue || [50000, 75000, 60000, 90000, 80000, 110000] :
                [20, 25, 30, 35, 40, 45]; // Orders data
                
            this.charts.performanceChart.data.datasets[0].data = newData;
            this.charts.performanceChart.data.datasets[0].label = metric === 'revenue' ? 'الإيرادات' : 'الطلبات';
            this.charts.performanceChart.update('active');
        }
    }

    updateActiveButton(activeButton) {
        document.querySelectorAll('.btn-chart-control').forEach(btn => {
            btn.classList.remove('active');
        });
        activeButton.classList.add('active');
    }

    downloadInvoice(event) {
        const button = event.target.closest('.download-btn');
        const row = button.closest('.invoice-row');
        const invoiceId = row.dataset.invoiceId;
        
        // Simulate download
        button.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
        
        setTimeout(() => {
            button.innerHTML = '<i class="fas fa-check"></i>';
            setTimeout(() => {
                button.innerHTML = '<i class="fas fa-download"></i>';
            }, 2000);
        }, 1000);
        
        console.log('Downloading invoice:', invoiceId);
    }

    startRealTimeUpdates() {
        this.updateInterval = setInterval(() => {
            if (this.isVisible) {
                this.updateDashboardData();
            }
        }, 30000); // Update every 30 seconds
    }

    updateDashboardData() {
        // Simulate real-time data updates
        const updates = {
            salesGrowth: Math.random() * 10 + 5,
            newOrders: Math.floor(Math.random() * 5),
            onlineUsers: Math.floor(Math.random() * 50) + 10
        };

        this.animateNumberChanges(updates);
    }

    animateNumberChanges(updates) {
        // Animate number changes with count-up effect
        Object.keys(updates).forEach(key => {
            const element = document.querySelector(`[data-metric="${key}"]`);
            if (element) {
                this.countUp(element, parseFloat(element.textContent), updates[key]);
            }
        });
    }

    countUp(element, start, end, duration = 1000) {
        const range = end - start;
        const increment = range / (duration / 16);
        let current = start;
        
        const timer = setInterval(() => {
            current += increment;
            if ((increment > 0 && current >= end) || (increment < 0 && current <= end)) {
                current = end;
                clearInterval(timer);
            }
            element.textContent = Math.round(current);
        }, 16);
    }

    setupAnimations() {
        // Intersection Observer for scroll animations
        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    entry.target.classList.add('fade-in-up');
                    observer.unobserve(entry.target);
                }
            });
        }, { threshold: 0.1 });

        document.querySelectorAll('.chart-card, .zakat-status-card, .zatca-status-card').forEach(el => {
            observer.observe(el);
        });

        // Animate metrics on load
        setTimeout(() => {
            this.animateMetrics();
        }, 500);
    }

    animateMetrics() {
        const metricElements = document.querySelectorAll('.metric-number, .stat-number, .center-number');
        metricElements.forEach((element, index) => {
            setTimeout(() => {
                const finalValue = parseFloat(element.textContent.replace(/[^0-9.-]/g, ''));
                element.textContent = '0';
                this.countUp(element, 0, finalValue, 2000);
                element.style.animation = 'countUp 0.6s ease-out';
            }, index * 200);
        });
    }

    handleVisibilityChange() {
        document.addEventListener('visibilitychange', () => {
            this.isVisible = !document.hidden;
            
            if (this.isVisible) {
                this.refreshAllCharts();
            }
        });
    }

    refreshAllCharts() {
        Object.keys(this.charts).forEach(key => {
            if (this.charts[key]) {
                this.charts[key].resize();
                this.charts[key].update('none');
            }
        });
    }

    destroy() {
        if (this.updateInterval) {
            clearInterval(this.updateInterval);
        }
        
        Object.keys(this.charts).forEach(key => {
            if (this.charts[key]) {
                this.charts[key].destroy();
            }
        });
    }
}

// Initialize dashboard when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    window.dashboardAnalytics = new DashboardAnalytics();
});

// Clean up on page unload
window.addEventListener('beforeunload', () => {
    if (window.dashboardAnalytics) {
        window.dashboardAnalytics.destroy();
    }
});