<?php

/**
 * أمثلة عملية لكيفية الحصول على محتوى الشهادة والمفتاح الخاص
 * Examples for getting Certificate and Private Key content
 */

require_once __DIR__ . '/../vendor/autoload.php';

// Bootstrap Laravel application
$app = require_once __DIR__ . '/../bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Services\CertificateHelper;

// ===== مثال 1: الحصول على الشهادة والمفتاح من قاعدة البيانات =====
echo "=== مثال 1: الحصول على الشهادة والمفتاح من قاعدة البيانات ===\n";

$companyId = 1; // معرف الشركة

try {
    // الحصول على محتوى الشهادة
    $certificate = CertificateHelper::getCertificateContent($companyId);
    
    if ($certificate) {
        echo "✅ تم العثور على الشهادة\n";
        echo "طول محتوى الشهادة: " . strlen($certificate) . " حرف\n";
        echo "أول 100 حرف من الشهادة:\n";
        echo substr($certificate, 0, 100) . "...\n\n";
    } else {
        echo "❌ لم يتم العثور على الشهادة\n\n";
    }
    
    // الحصول على محتوى المفتاح الخاص
    $privateKey = CertificateHelper::getPrivateKeyContent($companyId);
    
    if ($privateKey) {
        echo "✅ تم العثور على المفتاح الخاص\n";
        echo "طول محتوى المفتاح الخاص: " . strlen($privateKey) . " حرف\n";
        echo "أول 100 حرف من المفتاح الخاص:\n";
        echo substr($privateKey, 0, 100) . "...\n\n";
    } else {
        echo "❌ لم يتم العثور على المفتاح الخاص\n\n";
    }
    
} catch (Exception $e) {
    echo "خطأ: " . $e->getMessage() . "\n\n";
}

// ===== مثال 2: قراءة الشهادة والمفتاح من ملفات =====
echo "=== مثال 2: قراءة الشهادة والمفتاح من ملفات ===\n";

$certificateFilePath = 'certificates/zatca_certificate.pem';
$privateKeyFilePath = 'certificates/zatca_private_key.pem';

try {
    // قراءة الشهادة من ملف
    if (file_exists($certificateFilePath)) {
        $certificateFromFile = CertificateHelper::readCertificateFromFile($certificateFilePath);
        echo "✅ تم قراءة الشهادة من الملف بنجاح\n";
        echo "حجم الملف: " . strlen($certificateFromFile) . " بايت\n";
    } else {
        echo "❌ ملف الشهادة غير موجود: $certificateFilePath\n";
    }
    
    // قراءة المفتاح الخاص من ملف
    if (file_exists($privateKeyFilePath)) {
        $privateKeyFromFile = CertificateHelper::readPrivateKeyFromFile($privateKeyFilePath);
        echo "✅ تم قراءة المفتاح الخاص من الملف بنجاح\n";
        echo "حجم الملف: " . strlen($privateKeyFromFile) . " بايت\n\n";
    } else {
        echo "❌ ملف المفتاح الخاص غير موجود: $privateKeyFilePath\n\n";
    }
    
} catch (Exception $e) {
    echo "خطأ في قراءة الملفات: " . $e->getMessage() . "\n\n";
}

// ===== مثال 3: تحليل معلومات الشهادة =====
echo "=== مثال 3: تحليل معلومات الشهادة ===\n";

$certificate = CertificateHelper::getCertificateContent($companyId);

if ($certificate) {
    try {
        $certInfo = CertificateHelper::parseCertificateInfo($certificate);
        
        echo "معلومات الشهادة:\n";
        echo "- الرقم التسلسلي: " . ($certInfo['serial_number'] ?? 'غير محدد') . "\n";
        echo "- صالحة من: " . $certInfo['valid_from'] . "\n";
        echo "- صالحة حتى: " . $certInfo['valid_to'] . "\n";
        echo "- البصمة: " . $certInfo['fingerprint'] . "\n";
        
        if (isset($certInfo['subject'])) {
            echo "- صاحب الشهادة:\n";
            foreach ($certInfo['subject'] as $key => $value) {
                echo "  * $key: $value\n";
            }
        }
        
        if (isset($certInfo['issuer'])) {
            echo "- مُصدر الشهادة:\n";
            foreach ($certInfo['issuer'] as $key => $value) {
                echo "  * $key: $value\n";
            }
        }
        echo "\n";
        
    } catch (Exception $e) {
        echo "خطأ في تحليل الشهادة: " . $e->getMessage() . "\n\n";
    }
}

// ===== مثال 4: التحقق من صحة المفتاح الخاص =====
echo "=== مثال 4: التحقق من صحة المفتاح الخاص ===\n";

$privateKey = CertificateHelper::getPrivateKeyContent($companyId);

if ($privateKey) {
    try {
        $isValid = CertificateHelper::validatePrivateKey($privateKey);
        
        if ($isValid) {
            echo "✅ المفتاح الخاص صحيح وصالح للاستخدام\n\n";
        } else {
            echo "❌ المفتاح الخاص غير صحيح أو تالف\n\n";
        }
        
    } catch (Exception $e) {
        echo "خطأ في التحقق من المفتاح الخاص: " . $e->getMessage() . "\n\n";
    }
}

// ===== مثال 5: تحويل إلى تنسيق Base64 =====
echo "=== مثال 5: تحويل إلى تنسيق Base64 ===\n";

$certificate = CertificateHelper::getCertificateContent($companyId);
$privateKey = CertificateHelper::getPrivateKeyContent($companyId);

if ($certificate && $privateKey) {
    try {
        $certBase64 = CertificateHelper::certificateToBase64($certificate);
        $keyBase64 = CertificateHelper::privateKeyToBase64($privateKey);
        
        echo "الشهادة بتنسيق Base64 (أول 50 حرف):\n";
        echo substr($certBase64, 0, 50) . "...\n";
        
        echo "المفتاح الخاص بتنسيق Base64 (أول 50 حرف):\n";
        echo substr($keyBase64, 0, 50) . "...\n\n";
        
    } catch (Exception $e) {
        echo "خطأ في التحويل إلى Base64: " . $e->getMessage() . "\n\n";
    }
}

// ===== مثال 6: استخراج المفتاح العام من الشهادة =====
echo "=== مثال 6: استخراج المفتاح العام من الشهادة ===\n";

$certificate = CertificateHelper::getCertificateContent($companyId);

if ($certificate) {
    try {
        $publicKey = CertificateHelper::extractPublicKeyFromCertificate($certificate);
        
        if ($publicKey) {
            echo "✅ تم استخراج المفتاح العام بنجاح\n";
            echo "طول المفتاح العام: " . strlen($publicKey) . " حرف\n";
            echo "أول 100 حرف من المفتاح العام:\n";
            echo substr($publicKey, 0, 100) . "...\n\n";
        } else {
            echo "❌ فشل في استخراج المفتاح العام\n\n";
        }
        
    } catch (Exception $e) {
        echo "خطأ في استخراج المفتاح العام: " . $e->getMessage() . "\n\n";
    }
}

// ===== مثال 7: تصدير إلى ملفات =====
echo "=== مثال 7: تصدير الشهادة والمفتاح إلى ملفات ===\n";

$exportCertPath = 'exported_certificate.pem';
$exportKeyPath = 'exported_private_key.pem';

try {
    $success = CertificateHelper::exportToFiles($companyId, $exportCertPath, $exportKeyPath);
    
    if ($success) {
        echo "✅ تم تصدير الشهادة والمفتاح إلى الملفات بنجاح\n";
        echo "- ملف الشهادة: $exportCertPath\n";
        echo "- ملف المفتاح الخاص: $exportKeyPath\n\n";
    }
    
} catch (Exception $e) {
    echo "خطأ في التصدير: " . $e->getMessage() . "\n\n";
}

// ===== مثال 8: استخدام مع ZatcaService =====
echo "=== مثال 8: استخدام مع ZatcaService ===\n";

try {
    // إنشاء إعدادات ZATCA مع الشهادة والمفتاح
    $zatcaSettings = new \App\Models\ZatcaSettings();
    $zatcaSettings->company_id = $companyId;
    $zatcaSettings->certificate = CertificateHelper::getCertificateContent($companyId);
    $zatcaSettings->private_key = CertificateHelper::getPrivateKeyContent($companyId);
    $zatcaSettings->public_key = CertificateHelper::getPublicKeyContent($companyId);
    $zatcaSettings->environment = 'sandbox';
    
    // إنشاء خدمة ZATCA
    $zatcaService = new \App\Services\ZatcaService($zatcaSettings);
    
    echo "✅ تم إنشاء خدمة ZATCA بنجاح مع الشهادة والمفتاح\n";
    echo "البيئة: " . $zatcaSettings->environment . "\n";
    echo "URL الأساسي: " . $zatcaSettings->getBaseUrl() . "\n\n";
    
} catch (Exception $e) {
    echo "خطأ في إنشاء خدمة ZATCA: " . $e->getMessage() . "\n\n";
}

echo "=== انتهت الأمثلة ===\n";

/**
 * ملاحظات مهمة:
 * 
 * 1. الشهادة (Certificate):
 *    - تحتوي على المفتاح العام والمعلومات المرتبطة به
 *    - تنسيق PEM عادة (-----BEGIN CERTIFICATE-----...)
 *    - يجب التأكد من عدم انتهاء صلاحيتها
 * 
 * 2. المفتاح الخاص (Private Key):
 *    - يستخدم للتوقيع الرقمي
 *    - يجب حمايته وعدم مشاركته
 *    - تنسيق PEM عادة (-----BEGIN PRIVATE KEY-----...)
 * 
 * 3. المفتاح العام (Public Key):
 *    - يستخرج من الشهادة
 *    - يستخدم للتحقق من التوقيع
 * 
 * 4. أمان البيانات:
 *    - احرص على تشفير المفاتيح الخاصة في قاعدة البيانات
 *    - لا تعرض المفاتيح الخاصة في السجلات
 *    - استخدم HTTPS دائماً عند نقل هذه البيانات
 */