<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\Company;
use App\Models\Customer;
use App\Models\Product;
use App\Models\User;
use Illuminate\Support\Str;
use Carbon\Carbon;

class InvoiceSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $company = Company::first();
        $customers = Customer::where('is_active', true)->get();
        $products = Product::where('is_active', true)->get();
        $users = User::all();

        if (!$company || $customers->isEmpty() || $products->isEmpty() || $users->isEmpty()) {
            return;
        }

        // فاتورة معيارية مكتملة
        $invoice1 = Invoice::create([
            'invoice_number' => 'INV-202505-000001',
            'type' => 'standard',
            'company_id' => $company->id,
            'customer_id' => $customers->where('customer_type', 'business')->first()->id,
            'user_id' => $users->first()->id,
            'issue_date' => now()->subDays(5),
            'due_date' => now()->addDays(25),
            'supply_date' => now()->subDays(5),
            'currency' => 'SAR',
            'discount_amount' => 100.00,
            'subtotal' => 0, // سيتم حسابه
            'tax_amount' => 0, // سيتم حسابه
            'total_amount' => 0, // سيتم حسابه
            'uuid' => Str::uuid(),
            'invoice_counter' => 1,
            'status' => 'cleared',
            'notes' => 'فاتورة تجريبية للاختبار - مجموعة منتجات متنوعة',
            'sent_to_zatca_at' => now()->subDays(4),
            'irn' => 'ZATCA-INV-' . str_pad(1, 10, '0', STR_PAD_LEFT),
        ]);

        // إضافة عناصر للفاتورة الأولى
        $this->addInvoiceItems($invoice1, [
            ['product' => $products->where('sku', 'DELL-INS-001')->first(), 'quantity' => 2, 'unit_price' => 2500.00],
            ['product' => $products->where('sku', 'HP-LJ-002')->first(), 'quantity' => 1, 'unit_price' => 800.00],
            ['product' => $products->where('sku', 'TECH-SUP-004')->first(), 'quantity' => 4, 'unit_price' => 150.00],
        ]);

        // فاتورة مبسطة
        $invoice2 = Invoice::create([
            'invoice_number' => 'SIM-202505-000001',
            'type' => 'simplified',
            'company_id' => $company->id,
            'customer_id' => $customers->where('customer_type', 'individual')->first()->id,
            'user_id' => $users->skip(1)->first()->id,
            'issue_date' => now()->subDays(3),
            'due_date' => null,
            'supply_date' => now()->subDays(3),
            'currency' => 'SAR',
            'discount_amount' => 0,
            'subtotal' => 0,
            'tax_amount' => 0,
            'total_amount' => 0,
            'uuid' => Str::uuid(),
            'invoice_counter' => 2,
            'status' => 'reported',
            'notes' => 'فاتورة مبسطة للعميل الفردي',
            'sent_to_zatca_at' => now()->subDays(2),
            'irn' => 'ZATCA-SIM-' . str_pad(2, 10, '0', STR_PAD_LEFT),
        ]);

        $this->addInvoiceItems($invoice2, [
            ['product' => $products->where('sku', 'PAPER-A4-006')->first(), 'quantity' => 10, 'unit_price' => 25.00],
            ['product' => $products->where('sku', 'PEN-BLUE-007')->first(), 'quantity' => 50, 'unit_price' => 3.00],
            ['product' => $products->where('sku', 'HDMI-CAB-003')->first(), 'quantity' => 3, 'unit_price' => 45.00],
        ]);

        // مسودة فاتورة
        $invoice3 = Invoice::create([
            'invoice_number' => 'INV-202505-000002',
            'type' => 'standard',
            'company_id' => $company->id,
            'customer_id' => $customers->where('customer_type', 'business')->skip(1)->first()->id,
            'user_id' => $users->skip(2)->first()->id,
            'issue_date' => now()->subDays(1),
            'due_date' => now()->addDays(29),
            'supply_date' => now()->subDays(1),
            'currency' => 'SAR',
            'discount_amount' => 50.00,
            'subtotal' => 0,
            'tax_amount' => 0,
            'total_amount' => 0,
            'uuid' => Str::uuid(),
            'invoice_counter' => 3,
            'status' => 'draft',
            'notes' => 'مسودة فاتورة - قيد المراجعة',
        ]);

        $this->addInvoiceItems($invoice3, [
            ['product' => $products->where('sku', 'CONSULT-010')->first(), 'quantity' => 8, 'unit_price' => 500.00],
            ['product' => $products->where('sku', 'TRAIN-SW-005')->first(), 'quantity' => 12, 'unit_price' => 300.00],
        ]);

        // فاتورة مع منتجات معفاة من الضريبة
        $invoice4 = Invoice::create([
            'invoice_number' => 'INV-202505-000003',
            'type' => 'standard',
            'company_id' => $company->id,
            'customer_id' => $customers->where('customer_type', 'individual')->skip(1)->first()->id,
            'user_id' => $users->skip(3)->first()->id,
            'issue_date' => now(),
            'due_date' => now()->addDays(30),
            'supply_date' => now(),
            'currency' => 'SAR',
            'discount_amount' => 0,
            'subtotal' => 0,
            'tax_amount' => 0,
            'total_amount' => 0,
            'uuid' => Str::uuid(),
            'invoice_counter' => 4,
            'status' => 'draft',
            'notes' => 'فاتورة تحتوي على منتجات معفاة من الضريبة',
        ]);

        $this->addInvoiceItems($invoice4, [
            ['product' => $products->where('sku', 'BOOK-EDU-008')->first(), 'quantity' => 5, 'unit_price' => 80.00],
            ['product' => $products->where('sku', 'PAPER-A4-006')->first(), 'quantity' => 2, 'unit_price' => 25.00],
        ]);

        // فاتورة كبيرة الحجم
        $invoice5 = Invoice::create([
            'invoice_number' => 'INV-202505-000004',
            'type' => 'standard',
            'company_id' => $company->id,
            'customer_id' => $customers->where('customer_type', 'business')->skip(2)->first()->id,
            'user_id' => $users->first()->id,
            'issue_date' => now()->subDays(7),
            'due_date' => now()->addDays(23),
            'supply_date' => now()->subDays(7),
            'currency' => 'SAR',
            'discount_amount' => 500.00,
            'subtotal' => 0,
            'tax_amount' => 0,
            'total_amount' => 0,
            'uuid' => Str::uuid(),
            'invoice_counter' => 5,
            'status' => 'sent',
            'notes' => 'طلبية كبيرة - مزيج من المنتجات والخدمات',
            'sent_to_zatca_at' => now()->subDays(6),
        ]);

        // إضافة عناصر متعددة للفاتورة الكبيرة
        $this->addInvoiceItems($invoice5, [
            ['product' => $products->where('sku', 'DELL-INS-001')->first(), 'quantity' => 10, 'unit_price' => 2400.00],
            ['product' => $products->where('sku', 'HP-LJ-002')->first(), 'quantity' => 5, 'unit_price' => 780.00],
            ['product' => $products->where('sku', 'HDMI-CAB-003')->first(), 'quantity' => 20, 'unit_price' => 42.00],
            ['product' => $products->where('sku', 'PAPER-A4-006')->first(), 'quantity' => 100, 'unit_price' => 23.00],
            ['product' => $products->where('sku', 'PEN-BLUE-007')->first(), 'quantity' => 200, 'unit_price' => 2.80],
            ['product' => $products->where('sku', 'TECH-SUP-004')->first(), 'quantity' => 20, 'unit_price' => 145.00],
            ['product' => $products->where('sku', 'TRAIN-SW-005')->first(), 'quantity' => 6, 'unit_price' => 290.00],
        ]);

        // فاتورة بدون عميل (للاختبار)
        $invoice6 = Invoice::create([
            'invoice_number' => 'SIM-202505-000002',
            'type' => 'simplified',
            'company_id' => $company->id,
            'customer_id' => null,
            'user_id' => $users->skip(1)->first()->id,
            'issue_date' => now()->subHours(2),
            'due_date' => null,
            'supply_date' => now()->subHours(2),
            'currency' => 'SAR',
            'discount_amount' => 0,
            'subtotal' => 0,
            'tax_amount' => 0,
            'total_amount' => 0,
            'uuid' => Str::uuid(),
            'invoice_counter' => 6,
            'status' => 'draft',
            'notes' => 'فاتورة مبسطة بدون عميل محدد',
        ]);

        $this->addInvoiceItems($invoice6, [
            ['product' => $products->where('sku', 'PAPER-A4-006')->first(), 'quantity' => 1, 'unit_price' => 25.00],
            ['product' => $products->where('sku', 'PEN-BLUE-007')->first(), 'quantity' => 5, 'unit_price' => 3.00],
        ]);
    }

    /**
     * إضافة عناصر للفاتورة وحساب الإجماليات
     */
    private function addInvoiceItems(Invoice $invoice, array $itemsData): void
    {
        $subtotal = 0;
        $totalTax = 0;

        foreach ($itemsData as $itemData) {
            $product = $itemData['product'];
            if (!$product) continue;

            $quantity = $itemData['quantity'];
            $unitPrice = $itemData['unit_price'];
            $lineTotal = $quantity * $unitPrice;
            $taxAmount = $product->is_taxable ? ($lineTotal * $product->tax_rate / 100) : 0;
            $totalWithTax = $lineTotal + $taxAmount;

            InvoiceItem::create([
                'invoice_id' => $invoice->id,
                'product_id' => $product->id,
                'name' => $product->name,
                'name_en' => $product->name_en,
                'description' => $product->description,
                'sku' => $product->sku,
                'quantity' => $quantity,
                'unit' => $product->unit,
                'unit_price' => $unitPrice,
                'line_total' => $lineTotal,
                'discount_amount' => 0,
                'discount_percentage' => 0,
                'tax_rate' => $product->tax_rate,
                'tax_amount' => $taxAmount,
                'total_with_tax' => $totalWithTax,
                'zatca_category_code' => $product->zatca_category_code,
                'is_taxable' => $product->is_taxable,
            ]);

            $subtotal += $lineTotal;
            $totalTax += $taxAmount;

            // تقليل المخزون للمنتجات
            if ($product->type === 'product' && $product->stock_quantity !== null) {
                $product->decrement('stock_quantity', $quantity);
            }
        }

        // تحديث إجماليات الفاتورة
        $invoice->update([
            'subtotal' => $subtotal,
            'tax_amount' => $totalTax,
            'total_amount' => $subtotal + $totalTax - $invoice->discount_amount,
        ]);

        // توليد QR Code للفواتير المكتملة
        if (in_array($invoice->status, ['sent', 'cleared', 'reported'])) {
            try {
                $invoice->saveQrCode();
            } catch (\Exception $e) {
                \Log::warning('فشل في توليد QR Code للفاتورة: ' . $invoice->id);
            }
        }
    }
}
