<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Folder;
use App\Models\File;
use App\Models\User;
use Illuminate\Support\Facades\Storage;

class FileManagerSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $admin = User::where('email', 'admin@brima.com')->first();
        if (!$admin) {
            $admin = User::first();
        }

        if (!$admin) {
            $this->command->info('لا يوجد مستخدمين في النظام. يرجى إنشاء مستخدم أولاً.');
            return;
        }

        // المجلدات الرئيسية
        $folders = [
            [
                'name' => 'المستندات',
                'description' => 'مجلد للمستندات والملفات الرسمية',
                'color' => '#3498db',
                'icon' => 'fas fa-file-alt',
                'children' => [
                    ['name' => 'الفواتير', 'color' => '#e74c3c', 'icon' => 'fas fa-file-invoice'],
                    ['name' => 'العقود', 'color' => '#2ecc71', 'icon' => 'fas fa-file-contract'],
                    ['name' => 'التقارير', 'color' => '#f39c12', 'icon' => 'fas fa-chart-line'],
                ]
            ],
            [
                'name' => 'الصور',
                'description' => 'مجلد للصور والرسوم',
                'color' => '#e74c3c',
                'icon' => 'fas fa-images',
                'children' => [
                    ['name' => 'شعارات الشركة', 'color' => '#9b59b6', 'icon' => 'fas fa-palette'],
                    ['name' => 'صور المنتجات', 'color' => '#1abc9c', 'icon' => 'fas fa-shopping-bag'],
                ]
            ],
            [
                'name' => 'الأرشيف',
                'description' => 'مجلد للملفات المؤرشفة',
                'color' => '#95a5a6',
                'icon' => 'fas fa-archive',
                'children' => [
                    ['name' => 'أرشيف 2024', 'color' => '#34495e', 'icon' => 'fas fa-calendar'],
                    ['name' => 'ملفات قديمة', 'color' => '#7f8c8d', 'icon' => 'fas fa-history'],
                ]
            ],
            [
                'name' => 'المشاع',
                'description' => 'مجلد مشترك لجميع المستخدمين',
                'color' => '#f39c12',
                'icon' => 'fas fa-users',
                'is_public' => true,
                'children' => [
                    ['name' => 'النماذج', 'color' => '#e67e22', 'icon' => 'fas fa-file-pdf', 'is_public' => true],
                    ['name' => 'الدلائل', 'color' => '#d35400', 'icon' => 'fas fa-book', 'is_public' => true],
                ]
            ],
        ];

        foreach ($folders as $folderData) {
            $folder = $this->createFolder($folderData, null, $admin->id);
            
            if (isset($folderData['children'])) {
                foreach ($folderData['children'] as $childData) {
                    $this->createFolder($childData, $folder->id, $admin->id, $folder->level + 1);
                }
            }
        }

        // إنشاء ملفات تجريبية
        $this->createSampleFiles($admin->id);

        $this->command->info('تم إنشاء مجلدات وملفات إدارة الملفات بنجاح!');
    }

    /**
     * إنشاء مجلد
     */
    private function createFolder(array $data, ?int $parentId, int $userId, int $level = 0): Folder
    {
        $slug = \Str::slug($data['name']);
        $parent = $parentId ? Folder::find($parentId) : null;
        
        // التأكد من عدم وجود مجلد بنفس الاسم
        $existingFolder = Folder::where('name', $data['name'])
            ->where('parent_id', $parentId)
            ->first();
            
        if ($existingFolder) {
            return $existingFolder;
        }

        return Folder::create([
            'name' => $data['name'],
            'slug' => Folder::generateUniqueSlug($data['name'], $parentId),
            'description' => $data['description'] ?? null,
            'color' => $data['color'] ?? '#3498db',
            'icon' => $data['icon'] ?? 'fas fa-folder',
            'parent_id' => $parentId,
            'path' => $parent ? $parent->path . '/' . $slug : $slug,
            'level' => $level,
            'is_public' => $data['is_public'] ?? false,
            'created_by' => $userId,
            'is_system' => true, // مجلدات النظام
        ]);
    }

    /**
     * إنشاء ملفات تجريبية
     */
    private function createSampleFiles(int $userId): void
    {
        // الحصول على المجلدات المنشأة
        $documentsFolder = Folder::where('name', 'المستندات')->first();
        $invoicesFolder = Folder::where('name', 'الفواتير')->first();
        $contractsFolder = Folder::where('name', 'العقود')->first();
        $reportsFolder = Folder::where('name', 'التقارير')->first();
        
        $imagesFolder = Folder::where('name', 'الصور')->first();
        $logosFolder = Folder::where('name', 'شعارات الشركة')->first();
        $productsImagesFolder = Folder::where('name', 'صور المنتجات')->first();
        
        $archiveFolder = Folder::where('name', 'الأرشيف')->first();
        $archive2024Folder = Folder::where('name', 'أرشيف 2024')->first();
        
        $sharedFolder = Folder::where('name', 'المشاع')->first();
        $templatesFolder = Folder::where('name', 'النماذج')->first();
        $guidesFolder = Folder::where('name', 'الدلائل')->first();

        // ملفات تجريبية للفواتير
        if ($invoicesFolder) {
            $this->createSampleFile([
                'name' => 'فاتورة رقم 001',
                'filename' => 'invoice_001.pdf',
                'extension' => 'pdf',
                'mime_type' => 'application/pdf',
                'size' => 245760, // 240 KB
                'description' => 'فاتورة مبيعات للعميل الأول',
                'category' => 'invoices',
                'folder_id' => $invoicesFolder->id,
                'uploaded_by' => $userId,
                'tags' => ['فواتير', 'مبيعات', '2025'],
            ]);

            $this->createSampleFile([
                'name' => 'فاتورة رقم 002',
                'filename' => 'invoice_002.pdf',
                'extension' => 'pdf',
                'mime_type' => 'application/pdf',
                'size' => 189440, // 185 KB
                'description' => 'فاتورة خدمات استشارية',
                'category' => 'invoices',
                'folder_id' => $invoicesFolder->id,
                'uploaded_by' => $userId,
                'tags' => ['فواتير', 'خدمات', 'استشارات'],
            ]);
        }

        // ملفات تجريبية للعقود
        if ($contractsFolder) {
            $this->createSampleFile([
                'name' => 'عقد توريد مواد خام',
                'filename' => 'supply_contract.docx',
                'extension' => 'docx',
                'mime_type' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                'size' => 512000, // 500 KB
                'description' => 'عقد توريد مواد خام مع المورد الرئيسي',
                'category' => 'contracts',
                'folder_id' => $contractsFolder->id,
                'uploaded_by' => $userId,
                'tags' => ['عقود', 'توريد', 'مواد خام'],
            ]);

            $this->createSampleFile([
                'name' => 'عقد عمل موظف جديد',
                'filename' => 'employment_contract.pdf',
                'extension' => 'pdf',
                'mime_type' => 'application/pdf',
                'size' => 327680, // 320 KB
                'description' => 'عقد عمل للموظف الجديد في قسم المبيعات',
                'category' => 'contracts',
                'folder_id' => $contractsFolder->id,
                'uploaded_by' => $userId,
                'tags' => ['عقود', 'موظفين', 'مبيعات'],
            ]);
        }

        // ملفات تجريبية للتقارير
        if ($reportsFolder) {
            $this->createSampleFile([
                'name' => 'تقرير المبيعات الشهري',
                'filename' => 'monthly_sales_report.xlsx',
                'extension' => 'xlsx',
                'mime_type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                'size' => 1048576, // 1 MB
                'description' => 'تقرير مبيعات شهر مايو 2025',
                'category' => 'reports',
                'folder_id' => $reportsFolder->id,
                'uploaded_by' => $userId,
                'tags' => ['تقارير', 'مبيعات', 'شهري', 'مايو'],
            ]);

            $this->createSampleFile([
                'name' => 'تقرير المخزون',
                'filename' => 'inventory_report.pdf',
                'extension' => 'pdf',
                'mime_type' => 'application/pdf',
                'size' => 756432, // 738 KB
                'description' => 'تقرير حالة المخزون الحالية',
                'category' => 'reports',
                'folder_id' => $reportsFolder->id,
                'uploaded_by' => $userId,
                'tags' => ['تقارير', 'مخزون', 'حالة'],
            ]);
        }

        // ملفات تجريبية للشعارات
        if ($logosFolder) {
            $this->createSampleFile([
                'name' => 'شعار الشركة الرئيسي',
                'filename' => 'company_logo.png',
                'extension' => 'png',
                'mime_type' => 'image/png',
                'size' => 204800, // 200 KB
                'description' => 'الشعار الرسمي للشركة بدقة عالية',
                'category' => 'images',
                'folder_id' => $logosFolder->id,
                'uploaded_by' => $userId,
                'tags' => ['شعار', 'شركة', 'رسمي'],
                'is_public' => true,
            ]);

            $this->createSampleFile([
                'name' => 'شعار مبسط للمراسلات',
                'filename' => 'logo_simple.svg',
                'extension' => 'svg',
                'mime_type' => 'image/svg+xml',
                'size' => 51200, // 50 KB
                'description' => 'شعار مبسط للاستخدام في المراسلات',
                'category' => 'images',
                'folder_id' => $logosFolder->id,
                'uploaded_by' => $userId,
                'tags' => ['شعار', 'مبسط', 'مراسلات'],
                'is_public' => true,
            ]);
        }

        // ملفات تجريبية لصور المنتجات
        if ($productsImagesFolder) {
            $this->createSampleFile([
                'name' => 'منتج A - صورة رئيسية',
                'filename' => 'product_a_main.jpg',
                'extension' => 'jpg',
                'mime_type' => 'image/jpeg',
                'size' => 512000, // 500 KB
                'description' => 'الصورة الرئيسية للمنتج A',
                'category' => 'products',
                'folder_id' => $productsImagesFolder->id,
                'uploaded_by' => $userId,
                'tags' => ['منتج', 'صورة', 'رئيسية'],
                'is_public' => true,
            ]);

            $this->createSampleFile([
                'name' => 'منتج B - كتالوج',
                'filename' => 'product_b_catalog.pdf',
                'extension' => 'pdf',
                'mime_type' => 'application/pdf',
                'size' => 2097152, // 2 MB
                'description' => 'كتالوج مفصل للمنتج B',
                'category' => 'products',
                'folder_id' => $productsImagesFolder->id,
                'uploaded_by' => $userId,
                'tags' => ['منتج', 'كتالوج', 'مفصل'],
            ]);
        }

        // ملفات تجريبية للنماذج (عامة)
        if ($templatesFolder) {
            $this->createSampleFile([
                'name' => 'نموذج طلب إجازة',
                'filename' => 'leave_request_template.docx',
                'extension' => 'docx',
                'mime_type' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                'size' => 81920, // 80 KB
                'description' => 'نموذج قياسي لطلب الإجازات',
                'category' => 'templates',
                'folder_id' => $templatesFolder->id,
                'uploaded_by' => $userId,
                'tags' => ['نموذج', 'إجازة', 'طلب'],
                'is_public' => true,
            ]);

            $this->createSampleFile([
                'name' => 'نموذج تقرير مشكلة',
                'filename' => 'bug_report_template.pdf',
                'extension' => 'pdf',
                'mime_type' => 'application/pdf',
                'size' => 163840, // 160 KB
                'description' => 'نموذج لتقرير المشاكل التقنية',
                'category' => 'templates',
                'folder_id' => $templatesFolder->id,
                'uploaded_by' => $userId,
                'tags' => ['نموذج', 'مشكلة', 'تقرير'],
                'is_public' => true,
            ]);
        }

        // ملفات تجريبية للدلائل (عامة)
        if ($guidesFolder) {
            $this->createSampleFile([
                'name' => 'دليل المستخدم',
                'filename' => 'user_guide.pdf',
                'extension' => 'pdf',
                'mime_type' => 'application/pdf',
                'size' => 3145728, // 3 MB
                'description' => 'الدليل الشامل لاستخدام النظام',
                'category' => 'guides',
                'folder_id' => $guidesFolder->id,
                'uploaded_by' => $userId,
                'tags' => ['دليل', 'مستخدم', 'نظام'],
                'is_public' => true,
            ]);

            $this->createSampleFile([
                'name' => 'سياسات الشركة',
                'filename' => 'company_policies.docx',
                'extension' => 'docx',
                'mime_type' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                'size' => 409600, // 400 KB
                'description' => 'سياسات وإجراءات الشركة',
                'category' => 'guides',
                'folder_id' => $guidesFolder->id,
                'uploaded_by' => $userId,
                'tags' => ['سياسات', 'شركة', 'إجراءات'],
                'is_public' => true,
            ]);
        }

        // تحديث إحصائيات المجلدات
        Folder::all()->each(function ($folder) {
            $folder->updateStats();
        });
    }

    /**
     * إنشاء ملف تجريبي
     */
    private function createSampleFile(array $data): File
    {
        // إنشاء مسار وهمي للملف
        $path = 'files/' . $data['folder_id'] . '/' . $data['filename'];
        
        // إنشاء hash وهمي
        $hash = hash('sha256', $data['filename'] . time());
        
        return File::create(array_merge($data, [
            'path' => $path,
            'hash' => $hash,
            'download_count' => rand(0, 10),
            'last_downloaded_at' => rand(0, 1) ? now()->subDays(rand(1, 30)) : null,
        ]));
    }
}
