<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Account;
use App\Models\CostCenter;
use App\Models\BankAccount;
use App\Models\JournalEntry;
use App\Models\JournalEntryLine;
use App\Models\CashTransaction;
use Illuminate\Support\Facades\DB;

class CompleteAccountingSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        DB::beginTransaction();
        
        try {
            // إنشاء دليل الحسابات الأساسي
            $this->createChartOfAccounts();
            
            // إنشاء مراكز التكلفة
            $this->createCostCenters();
            
            // إنشاء الحسابات البنكية
            $this->createBankAccounts();
            
            // إنشاء قيود افتتاحية تجريبية
            $this->createOpeningEntries();
            
            // إنشاء معاملات نقدية تجريبية
            $this->createSampleCashTransactions();
            
            DB::commit();
            
            $this->command->info('تم إنشاء النظام المحاسبي الكامل بنجاح!');
            
        } catch (\Exception $e) {
            DB::rollback();
            $this->command->error('خطأ في إنشاء النظام المحاسبي: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * إنشاء دليل الحسابات الأساسي
     */
    private function createChartOfAccounts()
    {
        $accounts = [
            // الأصول الرئيسية
            [
                'code' => '1000',
                'name' => 'الأصول',
                'type' => 'asset',
                'is_control_account' => true,
                'allow_manual_entries' => false,
                'children' => [
                    // الأصول المتداولة
                    [
                        'code' => '1100',
                        'name' => 'الأصول المتداولة',
                        'type' => 'asset',
                        'is_control_account' => true,
                        'allow_manual_entries' => false,
                        'children' => [
                            [
                                'code' => '1101',
                                'name' => 'النقدية في الصندوق',
                                'type' => 'asset',
                                'is_cash_account' => true,
                                'opening_balance' => 50000,
                                'description' => 'النقدية الموجودة في الصندوق الرئيسي'
                            ],
                            [
                                'code' => '1102',
                                'name' => 'البنك الأهلي - الحساب الجاري',
                                'type' => 'asset',
                                'is_bank_account' => true,
                                'opening_balance' => 200000,
                                'description' => 'الحساب الجاري في البنك الأهلي'
                            ],
                            [
                                'code' => '1103',
                                'name' => 'بنك الرياض - حساب التوفير',
                                'type' => 'asset',
                                'is_bank_account' => true,
                                'opening_balance' => 150000,
                                'description' => 'حساب التوفير في بنك الرياض'
                            ],
                            [
                                'code' => '1110',
                                'name' => 'العملاء',
                                'type' => 'asset',
                                'opening_balance' => 75000,
                                'description' => 'المبالغ المستحقة من العملاء'
                            ],
                            [
                                'code' => '1120',
                                'name' => 'المخزون',
                                'type' => 'asset',
                                'opening_balance' => 120000,
                                'description' => 'قيمة البضاعة في المخزن'
                            ],
                            [
                                'code' => '1130',
                                'name' => 'المصروفات المدفوعة مقدماً',
                                'type' => 'asset',
                                'opening_balance' => 15000,
                                'description' => 'المصروفات المدفوعة مقدماً'
                            ]
                        ]
                    ],
                    // الأصول الثابتة
                    [
                        'code' => '1200',
                        'name' => 'الأصول الثابتة',
                        'type' => 'asset',
                        'is_control_account' => true,
                        'allow_manual_entries' => false,
                        'children' => [
                            [
                                'code' => '1201',
                                'name' => 'الأراضي',
                                'type' => 'asset',
                                'opening_balance' => 500000,
                                'description' => 'قيمة الأراضي المملوكة'
                            ],
                            [
                                'code' => '1202',
                                'name' => 'المباني',
                                'type' => 'asset',
                                'opening_balance' => 800000,
                                'description' => 'قيمة المباني والمنشآت'
                            ],
                            [
                                'code' => '1203',
                                'name' => 'الأثاث والمعدات',
                                'type' => 'asset',
                                'opening_balance' => 80000,
                                'description' => 'الأثاث والمعدات المكتبية'
                            ],
                            [
                                'code' => '1204',
                                'name' => 'أجهزة الحاسوب',
                                'type' => 'asset',
                                'opening_balance' => 45000,
                                'description' => 'أجهزة الحاسوب والمعدات التقنية'
                            ],
                            [
                                'code' => '1205',
                                'name' => 'السيارات',
                                'type' => 'asset',
                                'opening_balance' => 120000,
                                'description' => 'المركبات والسيارات'
                            ]
                        ]
                    ]
                ]
            ],

            // الخصوم الرئيسية
            [
                'code' => '2000',
                'name' => 'الخصوم',
                'type' => 'liability',
                'is_control_account' => true,
                'allow_manual_entries' => false,
                'children' => [
                    // الخصوم المتداولة
                    [
                        'code' => '2100',
                        'name' => 'الخصوم المتداولة',
                        'type' => 'liability',
                        'is_control_account' => true,
                        'allow_manual_entries' => false,
                        'children' => [
                            [
                                'code' => '2101',
                                'name' => 'الموردون',
                                'type' => 'liability',
                                'opening_balance' => -60000,
                                'description' => 'المبالغ المستحقة للموردين'
                            ],
                            [
                                'code' => '2102',
                                'name' => 'المصروفات المستحقة',
                                'type' => 'liability',
                                'opening_balance' => -25000,
                                'description' => 'المصروفات المستحقة غير المدفوعة'
                            ],
                            [
                                'code' => '2103',
                                'name' => 'ضريبة القيمة المضافة',
                                'type' => 'liability',
                                'opening_balance' => -18000,
                                'description' => 'ضريبة القيمة المضافة المستحقة'
                            ]
                        ]
                    ],
                    // الخصوم طويلة الأجل
                    [
                        'code' => '2200',
                        'name' => 'الخصوم طويلة الأجل',
                        'type' => 'liability',
                        'is_control_account' => true,
                        'allow_manual_entries' => false,
                        'children' => [
                            [
                                'code' => '2201',
                                'name' => 'قرض بنكي طويل الأجل',
                                'type' => 'liability',
                                'opening_balance' => -300000,
                                'description' => 'القرض البنكي طويل الأجل'
                            ]
                        ]
                    ]
                ]
            ],

            // حقوق الملكية
            [
                'code' => '3000',
                'name' => 'حقوق الملكية',
                'type' => 'equity',
                'is_control_account' => true,
                'allow_manual_entries' => false,
                'children' => [
                    [
                        'code' => '3001',
                        'name' => 'رأس المال',
                        'type' => 'equity',
                        'opening_balance' => -1000000,
                        'description' => 'رأس المال المدفوع'
                    ],
                    [
                        'code' => '3002',
                        'name' => 'الأرباح المحتجزة',
                        'type' => 'equity',
                        'opening_balance' => -150000,
                        'description' => 'الأرباح المحتجزة من السنوات السابقة'
                    ]
                ]
            ],

            // الإيرادات
            [
                'code' => '4000',
                'name' => 'الإيرادات',
                'type' => 'revenue',
                'is_control_account' => true,
                'allow_manual_entries' => false,
                'children' => [
                    [
                        'code' => '4001',
                        'name' => 'إيرادات المبيعات',
                        'type' => 'revenue',
                        'description' => 'إيرادات بيع البضاعة والخدمات'
                    ],
                    [
                        'code' => '4002',
                        'name' => 'إيرادات الخدمات',
                        'type' => 'revenue',
                        'description' => 'إيرادات تقديم الخدمات'
                    ],
                    [
                        'code' => '4003',
                        'name' => 'إيرادات أخرى',
                        'type' => 'revenue',
                        'description' => 'الإيرادات المتنوعة الأخرى'
                    ]
                ]
            ],

            // المصروفات
            [
                'code' => '5000',
                'name' => 'المصروفات',
                'type' => 'expense',
                'is_control_account' => true,
                'allow_manual_entries' => false,
                'children' => [
                    [
                        'code' => '5001',
                        'name' => 'تكلفة البضاعة المباعة',
                        'type' => 'expense',
                        'description' => 'تكلفة البضاعة المباعة',
                        'requires_cost_center' => true
                    ],
                    [
                        'code' => '5100',
                        'name' => 'مصروفات إدارية',
                        'type' => 'expense',
                        'is_control_account' => true,
                        'allow_manual_entries' => false,
                        'children' => [
                            [
                                'code' => '5101',
                                'name' => 'الرواتب والأجور',
                                'type' => 'expense',
                                'description' => 'رواتب وأجور الموظفين',
                                'requires_cost_center' => true
                            ],
                            [
                                'code' => '5102',
                                'name' => 'الإيجار',
                                'type' => 'expense',
                                'description' => 'إيجار المكاتب والمباني',
                                'requires_cost_center' => true
                            ],
                            [
                                'code' => '5103',
                                'name' => 'الكهرباء والماء',
                                'type' => 'expense',
                                'description' => 'فواتير الكهرباء والماء',
                                'requires_cost_center' => true
                            ],
                            [
                                'code' => '5104',
                                'name' => 'الاتصالات',
                                'type' => 'expense',
                                'description' => 'فواتير الهاتف والإنترنت',
                                'requires_cost_center' => true
                            ],
                            [
                                'code' => '5105',
                                'name' => 'القرطاسية والمكتبية',
                                'type' => 'expense',
                                'description' => 'مصروفات القرطاسية والمكتبية',
                                'requires_cost_center' => true
                            ]
                        ]
                    ],
                    [
                        'code' => '5200',
                        'name' => 'مصروفات التسويق',
                        'type' => 'expense',
                        'is_control_account' => true,
                        'allow_manual_entries' => false,
                        'children' => [
                            [
                                'code' => '5201',
                                'name' => 'الإعلان والدعاية',
                                'type' => 'expense',
                                'description' => 'مصروفات الإعلان والتسويق',
                                'requires_cost_center' => true
                            ],
                            [
                                'code' => '5202',
                                'name' => 'العمولات',
                                'type' => 'expense',
                                'description' => 'عمولات المبيعات',
                                'requires_cost_center' => true
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $this->createAccountsRecursive($accounts);
    }

    /**
     * إنشاء الحسابات بشكل هرمي
     */
    private function createAccountsRecursive($accounts, $parentId = null)
    {
        foreach ($accounts as $accountData) {
            $children = $accountData['children'] ?? [];
            unset($accountData['children']);

            $account = Account::create(array_merge([
                'parent_id' => $parentId,
                'is_active' => true,
                'is_cash_account' => false,
                'is_bank_account' => false,
                'is_control_account' => false,
                'requires_cost_center' => false,
                'allow_manual_entries' => true,
                'opening_balance' => 0,
                'current_balance' => 0,
                'created_by' => 1,
            ], $accountData));

            // تحديث الرصيد الحالي ليساوي الرصيد الافتتاحي
            if (isset($accountData['opening_balance'])) {
                $account->update(['current_balance' => $accountData['opening_balance']]);
            }

            if (!empty($children)) {
                $this->createAccountsRecursive($children, $account->id);
            }
        }
    }

    /**
     * إنشاء مراكز التكلفة
     */
    private function createCostCenters()
    {
        $costCenters = [
            [
                'code' => 'CC001',
                'name' => 'الإدارة العامة',
                'type' => 'department',
                'description' => 'مركز تكلفة الإدارة العامة',
                'manager_name' => 'أحمد محمد',
                'budget_amount' => 500000,
                'budget_period' => 'annual',
                'is_active' => true,
                'track_performance' => true,
            ],
            [
                'code' => 'CC002',
                'name' => 'قسم المبيعات',
                'type' => 'department',
                'description' => 'مركز تكلفة قسم المبيعات',
                'manager_name' => 'سارة أحمد',
                'budget_amount' => 300000,
                'budget_period' => 'annual',
                'is_active' => true,
                'track_performance' => true,
            ],
            [
                'code' => 'CC003',
                'name' => 'قسم التسويق',
                'type' => 'department',
                'description' => 'مركز تكلفة قسم التسويق',
                'manager_name' => 'محمد علي',
                'budget_amount' => 200000,
                'budget_period' => 'annual',
                'is_active' => true,
                'track_performance' => true,
            ],
            [
                'code' => 'PJ001',
                'name' => 'مشروع التوسع الجديد',
                'type' => 'project',
                'description' => 'مشروع التوسع في الفرع الجديد',
                'manager_name' => 'خالد يوسف',
                'budget_amount' => 1000000,
                'budget_period' => 'annual',
                'is_active' => true,
                'track_performance' => true,
            ]
        ];

        foreach ($costCenters as $costCenterData) {
            CostCenter::create($costCenterData);
        }
    }

    /**
     * إنشاء الحسابات البنكية
     */
    private function createBankAccounts()
    {
        $bankAccounts = [
            [
                'account_id' => Account::where('code', '1102')->first()->id,
                'bank_name' => 'البنك الأهلي السعودي',
                'account_number' => '123456789',
                'iban' => 'SA0310000001234567891234',
                'swift_code' => 'NCBKSAJE',
                'branch_name' => 'فرع الرياض الرئيسي',
                'account_type' => 'current',
                'currency' => 'SAR',
                'is_active' => true,
            ],
            [
                'account_id' => Account::where('code', '1103')->first()->id,
                'bank_name' => 'بنك الرياض',
                'account_number' => '987654321',
                'iban' => 'SA0320000009876543211234',
                'swift_code' => 'RIBLSARI',
                'branch_name' => 'فرع العليا',
                'account_type' => 'savings',
                'currency' => 'SAR',
                'is_active' => true,
            ]
        ];

        foreach ($bankAccounts as $bankAccountData) {
            BankAccount::create($bankAccountData);
        }
    }

    /**
     * إنشاء قيود افتتاحية تجريبية
     */
    private function createOpeningEntries()
    {
        // قيد افتتاحي لتسجيل الأرصدة الافتتاحية
        $openingEntry = JournalEntry::create([
            'reference' => 'OP-001',
            'date' => now()->startOfYear(),
            'description' => 'قيد افتتاحي - تسجيل الأرصدة الافتتاحية',
            'total_amount' => 1540000, // إجمالي الأرصدة المدينة
            'is_approved' => true,
            'approved_by' => 1,
            'approved_at' => now(),
            'created_by' => 1,
        ]);

        // خطوط القيد الافتتاحي (الأصول - مدينة)
        $debitAccounts = [
            ['1101', 50000],   // النقدية في الصندوق
            ['1102', 200000],  // البنك الأهلي
            ['1103', 150000],  // بنك الرياض
            ['1110', 75000],   // العملاء
            ['1120', 120000],  // المخزون
            ['1130', 15000],   // المصروفات المدفوعة مقدماً
            ['1201', 500000],  // الأراضي
            ['1202', 800000],  // المباني
            ['1203', 80000],   // الأثاث والمعدات
            ['1204', 45000],   // أجهزة الحاسوب
            ['1205', 120000],  // السيارات
        ];

        foreach ($debitAccounts as [$code, $amount]) {
            $account = Account::where('code', $code)->first();
            if ($account) {
                JournalEntryLine::create([
                    'journal_entry_id' => $openingEntry->id,
                    'account_id' => $account->id,
                    'debit' => $amount,
                    'credit' => 0,
                    'description' => 'رصيد افتتاحي - ' . $account->name,
                ]);
            }
        }

        // خطوط القيد الافتتاحي (الخصوم وحقوق الملكية - دائنة)
        $creditAccounts = [
            ['2101', 60000],   // الموردون
            ['2102', 25000],   // المصروفات المستحقة
            ['2103', 18000],   // ضريبة القيمة المضافة
            ['2201', 300000],  // قرض بنكي طويل الأجل
            ['3001', 1000000], // رأس المال
            ['3002', 150000],  // الأرباح المحتجزة
        ];

        foreach ($creditAccounts as [$code, $amount]) {
            $account = Account::where('code', $code)->first();
            if ($account) {
                JournalEntryLine::create([
                    'journal_entry_id' => $openingEntry->id,
                    'account_id' => $account->id,
                    'debit' => 0,
                    'credit' => $amount,
                    'description' => 'رصيد افتتاحي - ' . $account->name,
                ]);
            }
        }
    }

    /**
     * إنشاء معاملات نقدية تجريبية
     */
    private function createSampleCashTransactions()
    {
        $cashAccount = Account::where('code', '1101')->first(); // النقدية في الصندوق
        $salesAccount = Account::where('code', '4001')->first(); // إيرادات المبيعات
        $expenseAccount = Account::where('code', '5102')->first(); // الإيجار
        $costCenter = CostCenter::where('code', 'CC001')->first(); // الإدارة العامة

        if ($cashAccount && $salesAccount) {
            // معاملة استلام نقدي من مبيعات
            CashTransaction::create([
                'reference' => 'CR-001',
                'type' => 'receipt',
                'date' => now()->subDays(5),
                'account_id' => $cashAccount->id,
                'contra_account_id' => $salesAccount->id,
                'amount' => 15000,
                'description' => 'استلام نقدي من مبيعات نقدية',
                'cost_center_id' => $costCenter ? $costCenter->id : null,
                'is_reconciled' => true,
                'created_by' => 1,
            ]);
        }

        if ($cashAccount && $expenseAccount) {
            // معاملة دفع نقدي للإيجار
            CashTransaction::create([
                'reference' => 'CP-001',
                'type' => 'payment',
                'date' => now()->subDays(3),
                'account_id' => $cashAccount->id,
                'contra_account_id' => $expenseAccount->id,
                'amount' => 8000,
                'description' => 'دفع إيجار شهر ' . now()->format('m/Y'),
                'cost_center_id' => $costCenter ? $costCenter->id : null,
                'is_reconciled' => true,
                'created_by' => 1,
            ]);
        }
    }
}