<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('categories', function (Blueprint $table) {
            $table->id();
            $table->string('name')->comment('اسم الفئة');
            $table->string('name_en')->nullable()->comment('اسم الفئة بالإنجليزية');
            $table->text('description')->nullable()->comment('وصف الفئة');
            $table->string('code', 50)->unique()->comment('رمز الفئة');
            $table->string('color', 7)->default('#6c757d')->comment('لون الفئة');
            $table->string('icon', 100)->default('fas fa-folder')->comment('أيقونة الفئة');
            $table->string('image')->nullable()->comment('صورة الفئة');
            
            // Hierarchical structure
            $table->unsignedBigInteger('parent_id')->nullable()->comment('الفئة الأساسية');
            $table->integer('sort_order')->default(0)->comment('ترتيب العرض');
            $table->integer('level')->default(0)->comment('مستوى الفئة في التسلسل الهرمي');
            $table->text('path')->nullable()->comment('مسار الفئة الهرمي');
            
            // Status and visibility
            $table->boolean('is_active')->default(true)->comment('حالة النشاط');
            $table->boolean('is_featured')->default(false)->comment('فئة مميزة');
            $table->boolean('show_on_menu')->default(true)->comment('إظهار في القائمة');
            
            // SEO and metadata
            $table->string('slug')->unique()->comment('الرابط الصديق للمحركات');
            $table->string('meta_title')->nullable()->comment('عنوان SEO');
            $table->text('meta_description')->nullable()->comment('وصف SEO');
            $table->json('meta_keywords')->nullable()->comment('كلمات مفتاحية');
            
            // Business logic
            $table->enum('type', ['product', 'service', 'both'])->default('both')->comment('نوع الفئة');
            $table->boolean('requires_inventory')->default(false)->comment('تتطلب إدارة مخزون');
            $table->decimal('default_tax_rate', 5, 2)->nullable()->comment('معدل الضريبة الافتراضي');
            $table->string('default_unit')->nullable()->comment('الوحدة الافتراضية');
            
            // Statistics
            $table->integer('products_count')->default(0)->comment('عدد المنتجات');
            $table->integer('views_count')->default(0)->comment('عدد المشاهدات');
            $table->decimal('total_sales', 15, 2)->default(0)->comment('إجمالي المبيعات');
            
            // Audit fields
            $table->unsignedBigInteger('created_by')->nullable()->comment('أنشئ بواسطة');
            $table->unsignedBigInteger('updated_by')->nullable()->comment('حُدث بواسطة');
            $table->timestamp('last_activity_at')->nullable()->comment('آخر نشاط');
            
            $table->timestamps();
            
            // Indexes
            $table->index('parent_id');
            $table->index('is_active');
            $table->index('type');
            $table->index('sort_order');
            $table->index('level');
            $table->index(['parent_id', 'sort_order']);
            $table->index('slug');
            
            // Foreign keys
            $table->foreign('parent_id')->references('id')->on('categories')->onDelete('cascade');
            $table->foreign('created_by')->references('id')->on('users')->onDelete('set null');
            $table->foreign('updated_by')->references('id')->on('users')->onDelete('set null');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('categories');
    }
};
